/*************************************************************************************************
 *                      Modul: dvishrink.c                                                       *
 *                      Autor: Ulrich Lechner 9/1994                                             *
 *                                                                                               *
 *************************************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#include <X11/Xlib.h>
#include "dvishrink.h"

#define EPS 0.0001
//#define PI 3.141592654                                          
#define DIFF_BYTE (SRC/8)                /* SRC in Bytes                                  */
#define IL 4                             /* Intervall fuer normfilter(); 3 | 4            */
#define SET_ONE ((1L<< (8*sizeof(UTYPE)))-1)    /* Eine Moegl. alle Bits einer Variable
                                                 * vom Typ "UTYPE" auf Eins zu setzen     */
#define TGL
typedef unsigned char   byte;
struct Filter filter[NUMFILTER];         /* Filter fuer Convolution                       */

/*
	Truecolor Support
*/
extern int	colortype; // 1 = Truecolor, sonst Palette
extern int	num_r , num_g , num_b;
extern int 	shift_r , shift_g , shift_b;


/*
 * Aktueller Filter 
 */
UTYPE   g_dst;                           /* Zielgroesse <-> Quellgroesse                  */
UTYPE   g_size_y;                        /* Groesse einer Filtereinheit in y-Richtung     */
UTYPE   g_size_x;                        /* Groesse einer Filtereinheit in x-Richtung     */
UTYPE  *g_cyss;                          /* Zeiger auf Verschiebung in y-Richtung         */
UTYPE  *g_cxss;                          /* Zeiger auf Verschiebung in x-Richtung         */
UTYPE  *g_data;                          /* Zeiger auf Filterdaten                        */
/*
 * Convolutionsvariablen 
 */
UTYPE   ps[NUMFILTER][MAXPS];            /* "NUMFILTER" Verschiebungsvektoren */
UTYPE   sum[NUMFILTER][NUMFILTER];       /* Convolutionssummen                */
UTYPE   c[SRC][SRC];                     /* Master; Do not change UTYPE       */
UTYPE   ony[SRC];                        /* Zeilenflags                       */
/*
 * Variable fuer fastexpand bzw. fastshrink 
 */
int     numcode_x[MAXWIDTHHEIGHT];       /* Zahlen-Code Breite      */
int     numcode_y[MAXWIDTHHEIGHT];       /* Zahlen-Code Hoehe       */
UTYPE   sr[MAXWIDTHHEIGHT];              /* Zum Offset zu addieren  */
/*
 * Sonstige 
 */
int     coltab[MAXVALUE + 1];            /* Farbuebersetzungstabelle */
double  stretch;                         /* Stretch-Faktor fuer 
                                          * normalverteilte Filter   */
/*
 * Skalierungs-Prozedur fuer Convolutionsmethode
 * Parameter:  Quelle, Ziel, Ausgabe-Prozedur, Skalierungswert dst (Skalierungsfaktor dst/SRC)
 *             Zeiger auf Farbvektor, Zahl der Farben
 */
void    do_shrink(struct SrcBmp, XImage *, void (*outfkt) (XImage *, XColor*, short, short),
                                           UTYPE, XColor *, unsigned long);
void    ConvSum(void);                   /* Convolutions-Prozedur   */
void    setfilter(int);                  /* Aktuellen Filter setzen */

/* 
 * Ausgabe-Prozeduren fuer Convolution
 * Parameter:  Ziel, Zeiger auf Farbvektor, Zeile und Spalte 
 *             ab der dst x dst Punkte gesetzt werden sollen.
 */ 
void    OutGerade(XImage *, XColor *, short, short);   /* dst gerade,   XImage bits_per_pixel=4 */
void    OutUngerade(XImage *, XColor *, short, short); /* dst ungerade, XImage bits_per_pixel=4 */
void    Out8bpp(XImage *, XColor *, short, short);     /* XImage bits_per_pixel=8               */
void    Out16bpp(XImage *, XColor *, short, short);     /* XImage bits_per_pixel=8               */
void    Out24bpp(XImage *, XColor *, short, short);     /* XImage bits_per_pixel=8               */
void    Out32bpp(XImage *, XColor *, short, short);     /* XImage bits_per_pixel=8               */
void    OutXput(XImage *, XColor *, short, short);     /* beliebiges XImage                     */

/*
 * Skalierungs-Prozeduren fuer Scanline-Konvertierungsmethode: 
 * fastshrink(): Verkleinerung  aller Typen (IMG_MSB_4, IMG_MSB_8, IMG_OTHER)
 * fastexpand(): Vergroesserung aller Typen (IMG_MSB_4, IMG_MSB_8, IMG_OTHER) 
 * img4expand(): 4 bits_per_pixel XImage (IMG_MSB_4) 
 * Parameter:  Quelle, Ziel, Ausgabe-Prozedur, Skalierungswert dst (Skalierungsfaktor dst/SRC)
 */
void    fastshrink(struct SrcBmp, XImage *, void (*outfkt) (XImage *, short, short),UTYPE);
void    sh_numcode(int *bc, int x0, int y0, int x1, int y1);
void    fastexpand(struct SrcBmp, XImage *,void (*outfkt) (XImage *, short, short),UTYPE);
void    ex_numcode(int *bc, int x0, int y0, int x1, int y1);
void    img4expand(struct SrcBmp srcbmp, XImage *,
                   unsigned long BPxl ,UTYPE); /* Parameter BPxl= Pixelwert von Vordergrundfarbe */
void    makeline(XImage *,unsigned long BPxl, unsigned long lineoffset, long ex, int np);

/*
 * Initialisierungen und Sonstige Prototypen
 */
int     init_filter(unsigned long);                    /* Convolutions-Filter            */
void    init_tab(unsigned long nc);                    /* Interne Faruebersetungstabelle */
void    init_sr_x();
int     make_ss_xy(UTYPE * y, UTYPE * x, UTYPE dst, UTYPE size_y);
int     ex_ss_xy(UTYPE * y, UTYPE * x, UTYPE dst, UTYPE size_y);
double  schnitt(double le, double re, double a, double b);
void    norm_data(UTYPE * y, UTYPE * x, UTYPE * data,UTYPE my,UTYPE mx,UTYPE size_y,UTYPE size_x);
double  simpson(double a, double b, double c, double d, double old);
double  f(double, double);
double  makemax(void);
void    make_data(UTYPE * y, UTYPE * x, UTYPE * data, UTYPE my, UTYPE mx);
void    out_filter(struct Filter f);
void    outglobals(void);
void    kill_filter(void);                             /* Speicher der Filter freigeben */


/*
 * Skalierungs-Prozedur fuer Convolutionsmethode. Die Quell-Bitmap wird mit einem Netz der
 * Maschenweite SCR x SRC Punkte ueberdeckt. Eine Masche dient zum Belegen des Masters.
 * Der Master wird mit dem aktuellen Filter verknuepft. Start der als Parameter uebergebenen
 * Ausgabe-Prozedur, um dst x dst = factor x factor Punkte im Ziel zu setzen.
 * Parameter:  Quelle, Ziel, Ausgabe-Prozedur, Skalierungswert dst (Skalierungsfaktor dst/SRC)
 *             Zeiger auf Farbvektor, Zahl der Farben
 */
void do_shrink(struct SrcBmp srcbmp, XImage *theImage, 
               void (*outfkt) (XImage * theImage, XColor *, short lpy, short lpx),
               UTYPE factor, XColor *xcolor, unsigned long num_col)
{
    char    on_flag = 1, do_flag = 1;
    register short ix, iy;
    short   px, py, srclines, oneline, u;
    byte   *p_neu = NULL, *pl = NULL, *pr = NULL, *p_in = NULL; /* Zeiger im Netz */
    unsigned short msk;                              
    unsigned long max_size; 
    short   restline, restbyte;
       
    setfilter(factor);                               /* Setzen des aktuellen Filters             */
    srclines = srcbmp.frame_width * SRC;             /* Offset: SRC-Zeilen weiter                */
    oneline  = srcbmp.frame_width;                   /* Offset: eine Zeile weiter                */

    /* letzte moegliche Wert fuer pl-Zeiger */
    max_size = srcbmp.frame_width * (srcbmp.frame_height - SRC - srcbmp.frame_height % SRC);    
    restline = (srcbmp.frame_height - SRC) % SRC;    /* Zeilen ausserhalb des maximalen Netzes   */
    restbyte = srcbmp.frame_width % DIFF_BYTE;       /* Spalten ausserhalb des maximalen Netzes; */
                                          

    /** "frame_buffer" durchlaufen mit der horizontalen bzw. vertikalen Schrittweite
      * "SRC" = Maschenweite des Netzes */
    for (pl = srcbmp.frame_buffer, py = 0; pl <= srcbmp.frame_buffer + max_size; 
         py += g_dst, pl += srclines)
    {
        /* srclines Zeilen weiter */
        for (pr = pl, px = 0; pr <= pl + srcbmp.frame_width - DIFF_BYTE; 
             px += g_dst, pr += DIFF_BYTE)
        {
            /** "DIFF_BYTE * 8" Spalten weiter; 1 Byte = 8 Spalten!!!
             *  Jetzt innerhalb einer Masche den "frame_buffer" durchlaufen */
            for (p_neu = pr, do_flag=0, on_flag = 0, iy = 0; iy < SRC; iy++, p_neu += oneline)
            {
                /* Eine Zeile weiter */
                for (p_in = p_neu, u = 0; u < DIFF_BYTE; u++, p_in++)       /* Ein Byte weiter   */
                {                   
                    if (*p_in)
                    {                    
                        on_flag = 1;  /* mindestens ein Bit gesetzt */
                        for (msk = 0x80, ix = u << 3; ix < (u + 1) << 3; ix++, msk >>= 1)
                            c[iy][ix] = ((*p_in & msk) ? SET_ONE : 0);      /* Master fuellen    */
                    }
                    else
                    {
                        for (ix = u << 3; ix < (u + 1) << 3; ix++)
                            c[iy][ix] = 0;  /* Master loeschen */
                    }
                }
                if (on_flag == 1)
                {
                    do_flag=1;     /* Zeilenflag setzen falls mindestens ein Bit in Zeile gesetzt*/
                    ony[iy] = 1;         
                }                        
                else
                    ony[iy] = 0;
                on_flag=0;
            }
            /* Status: Eine Masche durchlaufen */
            if (do_flag == 1)
            {
                /* Mindestens ein Bit in Masche gesetzt ist 
                 */         
                ConvSum();                                  /* Convolution Master mit Filter     */                   
                (*outfkt) (theImage, xcolor, py, px);       /* Ausgabe-Prozedur starten          */
            }
        }
    }

    /**
     * Falls noch Zeilen uebrig sind, werden diese jetzt behandelt. Zeilen bleiben
     * uebrig, falls sich der "frame_buffer" nicht genau mit dem Netz ueberdecken
     * laesst 
     */    
    if (restline != 0)
    {
        for (iy = restline; iy < SRC; iy++)          /* Unbenutzte Zeilenflags loeschen */
            ony[iy] = 0;                 

        /** pl zeigt bereits auf die richtige Zeile; py hat schon richtigen Wert; 
         *  Rest siehe weiter oben. 
         */
        for (pr = pl, px = 0; pr <= pl + srcbmp.frame_width - DIFF_BYTE; 
             px += g_dst, pr += DIFF_BYTE)
        {
            for (p_neu = pr, on_flag = 0, iy = 0; iy < restline; iy++, p_neu += oneline)
            {
                for (p_in = p_neu, u = 0; u < DIFF_BYTE; u++, p_in++)
                {
                    if (*p_in)
                    {
                        on_flag = 1;
                        for (msk = 0x80, ix = u << 3; ix < (u + 1) << 3; ix++, msk >>= 1)
                            c[iy][ix] = ((*p_in & msk) ? SET_ONE : 0);
                    }
                    else
                    {
                        for (ix = u << 3; ix < (u + 1) << 3; ix++)
                            c[iy][ix] = 0;
                    }
                }
                if (on_flag == 1)
                    ony[iy] = 1;
                else
                    ony[iy] = 0;
            }
            if (on_flag == 1)
            {
                ConvSum();                                  /* Convolution Master mit Filter     */
                (*outfkt) (theImage, xcolor, py, px);       /* Ausgabe-Prozedur starten          */
            }

        }
    }
    /* end of restline */

    /**
     * Falls noch Spalten uebrig sind, werden diese jetzt behandelt. Spalten bleiben
     * uebrig, falls sich der "frame_buffer" nicht genau mit dem Netz ueberdecken
     * laesst. 
     */
    if (restbyte != 0)
    {
        for (iy = 0; iy < SRC; iy++)
            for (u = restbyte; u < DIFF_BYTE; u++)
                for (ix = u << 3; ix < (u + 1) << 3; ix++)
                    c[iy][ix] = 0;                       /* Unbenutzte Teile des Masters loeschen */

        /** pr auf richtige Spalte setzen; px hat schon richtigen Wert; 
         *  Rest siehe weiter oben. 
         */
        for (pr = srcbmp.frame_buffer + srcbmp.frame_width - restbyte, 
             pl = srcbmp.frame_buffer, py = 0; pl <= srcbmp.frame_buffer + max_size;
             py += g_dst, pl += srclines, pr += srclines)
        {
            for (p_neu = pr, on_flag = 0, iy = 0; iy < SRC; iy++, p_neu += oneline)
            {
                for (p_in = p_neu, u = 0; u < restbyte; u++, p_in++)
                {
                    if (*p_in)
                    {
                        on_flag = 1;
                        for (msk = 0x80, ix = u << 3; ix < (u + 1) << 3; ix++, msk >>= 1)
                            c[iy][ix] = ((*p_in & msk) ? SET_ONE : 0);
                    }
                    else
                    {
                        for (ix = u << 3; ix < (u + 1) << 3; ix++)
                            c[iy][ix] = 0;
                    }
                }
                if (on_flag == 1)
                    ony[iy] = 1;
                else
                    ony[iy] = 0;
            }
            if (on_flag == 1)
            {
                ConvSum();                                  /* Convolution Master mit Filter     */
                (*outfkt) (theImage, xcolor, py, px);       /* Ausgabe-Prozedur starten          */
            }
        }
    }
    /* end if restbyte */

    /**
     * Falls noch Bytes am rechten unteren Ende des "frame_buffers" uebrig sind (vgl.
     * "restline" und "restbyte"). 
     */
    if (restbyte != 0 && restline != 0)
    {
        for (iy = restline; iy < SRC; iy++)
            ony[iy] = 0;                 /* Unbenuzte Zeilenflags loeschen */

        for (iy = 0; iy < SRC; iy++)
            for (u = restbyte; u < DIFF_BYTE; u++)
                for (ix = u << 3; ix < (u + 1) << 3; ix++)
                    c[iy][ix] = 0;                      /* Unbenutzte Teile des Masters loeschen */

        /** pr  zeigt bereits auf die richtige Spalte; px,py haben schon richtige Werte;
         *  Rest siehe weiter oben. 
         */
        for (p_neu = pr, on_flag = 0, iy = 0; iy < restline; iy++, p_neu += oneline)
        {
            for (p_in = p_neu, u = 0; u < restbyte; u++, p_in++)
            {
                if (*p_in)
                {
                    on_flag = 1;
                    for (msk = 0x80, ix = u << 3; ix < (u + 1) << 3; ix++, msk >>= 1)
                        c[iy][ix] = ((*p_in & msk) ? SET_ONE : 0);
                }
                else
                {
                    for (ix = u << 3; ix < (u + 1) << 3; ix++)
                        c[iy][ix] = 0;
                }
            }
            if (on_flag == 1)
                ony[iy] = 1;
            else
                ony[iy] = 0;
        }
        if (on_flag == 1)
        {
            ConvSum();                                      /* Convolution Master mit Filter     */
            (*outfkt) (theImage, xcolor, py, px);           /* Ausgabe-Prozedur starten          */
        }
    }
}

/* 
 * Ausgabe-Prozeduren fuer Convolution
 * Es werden ab dem Punkt (lpy,lpx) dst-Punkte in der Breite und dst-Punkte Hoehe im
 * XImage in der resultierenden Farbe gesetzt. Wichtig: bits_per_pixel==4
 * und dst==gerade. Die Farbnummer ergibt sich aus dem mittels Convolution berechnetem
 * Wert "sum[][]", der als Index in die Farbuebersetzungtabelle dient "coltab[]". Diese
 * Index identifiziert den Pixel-Werte "col[].pixel" der gewuenschten Farbe 
 * Parameter:  Ziel, Zeiger auf Farbvektor, Zeile und Spalte 
 *             ab der dst x dst Punkte gesetzt werden sollen.
 */
void    OutGerade(XImage *theImage,XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix, ax, lpx_h, dst_h;
    unsigned long of, offset;            /* Offsets in XImage-Daten                    */
    unsigned long fl, fr, fb;            /* Pixel-Farbwerte                            */
    int    img_bpl=theImage->bytes_per_line;
    byte   *ptr;
    byte    m;                           /* Zur Rettung des aktuellen Bytes in XImage  */
    UTYPE   pxl, pxr;                    /* Convolutionswerte                          */
    
    ptr = (byte *)theImage->data;        /* Zeiger auf XImage-Daten                    */
    lpx_h = lpx >> 1;                    /* 4-bits-per-pixel => vertikalen Offset
                                          * halbieren */
    dst_h = g_dst >> 1;                  /* 4-bits-per-pixel => zwei Punkte werden
                                          * gleichzeitig behandelt */
    for (iy = 0, of = lpy * img_bpl + lpx_h; iy < g_dst; iy++, of += img_bpl)  /* g_dst Zeilen  */
        for (ax = 0, ix = 0; ix < dst_h; ix++, ax += 2)                        /* dst_h Spalten */
        {
            pxl = sum[iy][ax];           /* Convolutionswert des linken Punktes  */
            pxr = sum[iy][ax + 1];       /* Convolutionswert des rechten Punktes */
            if (pxl | pxr)
            {
                offset = of + ix;        /* neuer Offset */
                if ((pxl != 0) && (pxr != 0))
                {                                       /* beide Punkte bekommen Farbe  */
                    fl = (col[coltab[pxl]].pixel) << 4; /* Pixel-Farbwert linker Punkt  */
                    fr = (col[coltab[pxr]].pixel);      /* Pixel-Farbwert rechter Punkt */
                    fb = fl | fr;        /* Pixel-Farbwerte kombinieren  */
                    ptr[offset] = fb;    /* Resultat in XImage schreiben */
                }
                else
                {
                    if (pxl != 0)
                    {                    /* nur linker Punkt bekommt Farbe */
                        m = ptr[offset]; /* retten des aktuellen Bytes     */
                        fl = (col[coltab[pxl]].pixel) << 4;     /* Pixel-Farbwert linker
                                                                 * Punkt */
                        fb = fl | (m & 0x0f);   /* Pixel-Farbwerte kombinieren     */
                        ptr[offset] = fb;       /* Resultat in Ximage schreiben    */
                    }
                    else
                    {
                        if (pxr != 0)
                        {                       /* nur rechter Punkt bekommt Farbe */
                            m = ptr[offset];    /* retten des aktuellen Bytes      */
                            fr = (col[coltab[pxr]].pixel);      /* Pixel-Farbwert
                                                                 * rechter Punkt   */
                            fb = (m & 0xf0) | fr;  /* Pixel-Farbwerte kombinieren  */
                            ptr[offset] = fb;      /* Resultat in Ximage schreiben */
                        }
                    }
                }
            }

        }
}
 
/* 
 * Ausgabe-Prozeduren fuer Convolution
 * Es werden ab dem Punkt (lpy,lpx) dst-Punkte in der Breite und dst-Punkte Hoehe im
 * XImage in der resultierenden Farbe gesetzt. Wichtig: bits_per_pixel==4
 * und dst==ungerade. Rest vgl. OutGerade 
 * Parameter:  Ziel, Zeiger auf Farbvektor, Zeile und Spalte 
 *             ab der dst x dst Punkte gesetzt werden sollen. 
 */
void    OutUngerade(XImage *theImage, XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix, ax, dst_h, lpx_h;
    unsigned long of, offset;            /* Offsets in XImage-Daten */
    unsigned long fl, fr, fb;            /* Pixel-Farbwerte         */
    byte   *ptr;
    int img_bpl=theImage->bytes_per_line;
    byte    m;                           /* Zur Rettung des aktuellen Bytes in XImage */
    UTYPE   pxl, pxr;                    /* Convolutionswerte                         */

    ptr = (byte *)theImage->data;        /* Zeiger auf XImage-Daten */
    lpx_h = lpx >> 1;                    /* 4-bits-per-pixel => vertikalen Offset
                                          * halbieren */
    dst_h = g_dst >> 1;                  /* 4-bits-per-pixel => zwei Punkte werden
                                          * gleichzeitig behandelt */

    if ((lpx & 0x0001) != 0)
    {                                    /* lpx ungerade d.h. zuerst linken Punkt
                                          * retten und nur rechten Punkt bearbeiten   */
                                          
        for (iy = 0, of = lpy * img_bpl + lpx_h; iy < g_dst; iy++, of += img_bpl) 
        {                                /* "g_dst" Zeilen                       */
            pxr = sum[iy][0];            /* Convolutionswert des rechten Punktes */
            if (pxr != 0)
            {                            /* nur rechter Punkt bekommt Farbe      */
                offset = of;             /* Offset */
                fr = (col[coltab[pxr]].pixel);  /* Pixel-Farbwert rechter Punkt  */
                m = ptr[offset];         /* retten des aktuellen Bytes           */
                fb = (m & 0xf0) | fr;    /* Pixel-Farbwerte kombinieren          */
                ptr[offset] = fb;        /* Resultat in Ximage schreiben         */

            }
            /*
             * jetzt sind wir auf gleichen Stand wie bei OutGerade, denn der erste
             * Punkt in jeder der "g_dst" Zeilen ist bereits bearbeitet und damit ist
             * "g_dst" quasi gerade  
             */
            for (ax = 1, ix = 1; ix <= dst_h; ix++, ax += 2)    /* dst_h Spalten */
            {

                pxl = sum[iy][ax];
                pxr = sum[iy][ax + 1];
                if (pxl | pxr)
                {
                    offset = of + ix;
                    if ((pxl != 0) && (pxr != 0))
                    {
                        fl = (col[coltab[pxl]].pixel) << 4;
                        fr = (col[coltab[pxr]].pixel);
                        fb = fl | fr;
                        ptr[offset] = fb;

                    }
                    else
                    {
                        if (pxl != 0)
                        {
                            m = ptr[offset];
                            fl = (col[coltab[pxl]].pixel) << 4;
                            fb = fl | (m & 0x0f);
                            ptr[offset] = fb;
                        }
                        else
                        {
                            if (pxr != 0)
                            {
                                m = ptr[offset];
                                fr = (col[coltab[pxr]].pixel);
                                fb = (m & 0xf0) | fr;
                                ptr[offset] = fb;
                            }
                        }
                    }
                }
            }                            /* end for(ax...) */
        }                                /* end for(iy...) */
    }                                    /* end if(lpx...) */
    else
    {
        /*
         * wie bei OutGerade, nur der uebrige linke Punkt muss noch bearbeitet werden 
         */
        for (iy = 0, of = lpy * img_bpl + lpx_h; iy < g_dst; iy++, of += img_bpl)
        {
            for (ax = 0, ix = 0; ix < dst_h; ix++, ax += 2)
            {

                pxl = sum[iy][ax];
                pxr = sum[iy][ax + 1];
                if (pxl | pxr)
                {
                    offset = of + ix;
                    if ((pxl != 0) && (pxr != 0))
                    {
                        fl = (col[coltab[pxl]].pixel) << 4;
                        fr = (col[coltab[pxr]].pixel);
                        fb = fl | fr;
                        ptr[offset] = fb;

                    }
                    else
                    {
                        if (pxl != 0)
                        {
                            m = ptr[offset];
                            fl = (col[coltab[pxl]].pixel) << 4;
                            fb = fl | (m & 0x0f);
                            ptr[offset] = fb;
                        }
                        else
                        {
                            if (pxr != 0)
                            {
                                m = ptr[offset];
                                fr = (col[coltab[pxr]].pixel);
                                fb = (m & 0xf0) | fr;
                                ptr[offset] = fb;
                            }
                        }
                    }
                }
            }  /* end for(ax...)  */
                                                   /* uebrigen linken Punkt bearbeiten    */
            pxl = sum[iy][g_dst - 1];              /* Convolutionswert des linken Punktes */
            if (pxl != 0)
            {                                      /* nur linker Punkt bekommt Farbe */
                offset = of + ((g_dst - 1) >> 1);  /* neuer Offset                   */
                m = ptr[offset];                   /* retten des aktuellen Bytes     */
                fl = (col[coltab[pxl]].pixel) << 4;/* Pixel-Farbwert linker Punkt    */
                fb = fl | (m & 0x0f);              /* Pixel-Farbwerte kombinieren    */
                ptr[offset] = fb;                  /* Resultat in Ximage schreiben   */
            }
        }  /* end for(iy...) */
    }  /* end else */
}

/* 
 * Ausgabe-Prozeduren fuer Convolution 
 * Es werden ab dem Punkt (lpy,lpx) dst-Punkte in der Breite und dst-Punkte Hoehe im
 * XImage in der resultierenden Farbe gesetzt. Wichtig: bits_per_pixel==8 und dst==beliebig. 
 * Rest vgl. OutGerade 
 * Parameter:  Ziel, Zeiger auf Farbvektor, Zeile und Spalte 
 *             ab der dst x dst Punkte gesetzt werden sollen. 
 */
void    Out8bpp(XImage *theImage, XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix;
    unsigned long of;                              /* Offsets in XImage-Daten        */
    byte   *ptr, *p = NULL;
    int img_bpl=theImage->bytes_per_line;          /* XImage Bytes pro Zeile         */                      
    UTYPE   pxl;                                   /* Convolutionswert               */

    ptr = (byte *)theImage->data;                  /* Zeiger auf XImage-Daten ...    */
    for (iy = 0, of = lpy * img_bpl + lpx; iy < g_dst; iy++, of += img_bpl)
        for (p = ptr + of, ix = 0; ix < g_dst; ix++, p++)
        {
            pxl = sum[iy][ix];
            if (pxl)
                *p = col[coltab[pxl]].pixel;
        }
}

/*
	Ausgabe in 16bpp Bitmap

*/
void    Out16bpp(XImage *theImage, XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix;
    unsigned long of;                              /* Offsets in XImage-Daten        */
    short unsigned int   *ptr, *p = NULL;
    int img_bpl=theImage->bytes_per_line/2;          /* XImage Bytes pro Zeile         */                      
    int red,green,blue;
    UTYPE   pxl;                                   /* Convolutionswert               */

    ptr = (short unsigned int *) theImage->data;                  /* Zeiger auf XImage-Daten ...    */
    for (iy = 0, of = lpy * img_bpl + lpx; iy < g_dst; iy++, of += img_bpl)
        for (p = ptr + of, ix = 0; ix < g_dst; ix++, p++)
        {
            pxl = sum[iy][ix];
            if (pxl)
	    {
	      red = col[coltab[pxl]].red * num_r / 65536;
	      green = col[coltab[pxl]].green* num_g / 65536;
	      blue = col[coltab[pxl]].blue* num_b / 65536;
	      *p = (red << shift_r) + (green << shift_g) + (blue << shift_b) ;
	    }
        }
}

/*
	Ausgabe in 24bpp Bitmap

*/
void    Out24bpp(XImage *theImage, XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix;
    unsigned long of;                              /* Offsets in XImage-Daten        */
    char *ptr, *p = NULL;
    int img_bpl=theImage->bytes_per_line;          /* XImage Bytes pro Zeile         */                      
    int red,green,blue;
    UTYPE   pxl;                                   /* Convolutionswert               */

    ptr = (char *) theImage->data;                  /* Zeiger auf XImage-Daten ...    */
    for (iy = 0, of = lpy * img_bpl + lpx; iy < g_dst; iy++, of += img_bpl)
        for (p = ptr + of, ix = 0; ix < g_dst; ix++, p+=3)
        {
            pxl = sum[iy][ix];
            if (pxl)
	    {
	      red = col[coltab[pxl]].red * num_r / 65536;
	      green = col[coltab[pxl]].green* num_g / 65536;
	      blue = col[coltab[pxl]].blue* num_b / 65536;
	      *p = red ; 
	      *(p+1) = green ; 
	      *(p+2)  = blue ;
	    }
        }
}

/*
	Ausgabe in 32bpp Bitmap

*/
void    Out32bpp(XImage *theImage, XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix;
    unsigned long of;                              /* Offsets in XImage-Daten        */
    long unsigned int   *ptr, *p = NULL;
    int img_bpl=theImage->bytes_per_line/4;          /* XImage Bytes pro Zeile         */                      
    int red,green,blue;
    UTYPE pxl;                                   /* Convolutionswert               */

    ptr = (long unsigned int *) theImage->data;                  /* Zeiger auf XImage-Daten ...    */
    for (iy = 0, of = lpy * img_bpl + lpx; iy < g_dst; iy++, of += img_bpl)
        for (p = ptr + of, ix = 0; ix < g_dst; ix++, p++)
        {
            pxl = sum[iy][ix];
            if (pxl)
	    {
	      red = col[coltab[pxl]].red * num_r / 65536;
	      green = col[coltab[pxl]].green* num_g / 65536;
	      blue = col[coltab[pxl]].blue* num_b / 65536;
	      *p = (red << shift_r) + (green << shift_g) + (blue << shift_b) ;
	    }
        }
}


/*
 * Ausgabe-Prozeduren fuer Convolution (beliebige XImages )
 * Es werden ab dem Punkt (lpy,lpx) dst-Punkte in der Breite und dst-Punkte Hoehe im
 * XImage in der resultierenden Farbe gesetzt. Hier wird die Xlib-Funktion "XPutPixel" 
 * verwendet, die fuer alle XImage-Typen verwendet werden kann!!! Rest vgl. OutGerade 
 * Parameter:  Ziel, Zeiger auf Farbvektor, Zeile und Spalte 
 *             ab der dst x dst Punkte gesetzt werden sollen.  
 */
void    OutXput(XImage *theImage, XColor *col, short lpy, short lpx)
{
    register unsigned short iy, ix, epx, epy;
    UTYPE   pxl;
    for (iy = 0, epy = lpy; iy < g_dst; iy++, epy++)
        for (ix = 0, epx = lpx; ix < g_dst; ix++, epx++)
            if ((pxl = sum[iy][ix]) != 0L)
                XPutPixel(theImage, epx, epy, col[coltab[pxl]].pixel);
}



/*
 * Setzen des aktuellen Filters
 */
void    setfilter(int factor)
{
    if(!(factor > 0 && factor <= (SRC * EXFACTOR)))
                g_dst= factor = SRC;
    g_dst = filter[factor - 1].dst;             /* Zielgroesse <-> Quellgroesse              */
    g_size_y = filter[factor - 1].size_y;       /* Groesse einer Filtereinheit in y-Richtung */
    g_size_x = filter[factor - 1].size_x;       /* Groesse einer Filtereinheit in x-Richtung */
    g_cyss = filter[factor - 1].cyss;           /* Zeiger auf Verschiebung in y-Richtung     */
    g_cxss = filter[factor - 1].cxss;           /* Zeiger auf Verschiebung in x-Richtung     */
    g_data = filter[factor - 1].data;           /* Zeiger auf Filterdaten                    */
}

/*
 * Hier findet die Convolution des aktuellen Filters mit dem Master c[][] statt. Der
 * Master wird gemaess der Eintraege der Verschiebungsvektoren des Filters auf
 * diesem plaziert. Falls im Master ein Punkt gesetzt ist, wird der darunterliegende
 * Wert des Filters zur Convolutionssumme addiert. Jeweils "g_size_y x g_size_x" -
 * Filterwerte verknuepft mit dem Master ergeben eine Summe. Insgesamt entstehen
 * "g_dst * g_dst" Summen. Durch diese wird anschliessend die Ausgabe ins Zielbild
 * bestimmt. 
 */
void    ConvSum(void)
{
    UTYPE   summe;                              /* Convolutionssumme */
    /* Zur Steuerung der Plazierungen von "master" bzw. des Filters  
     */
    register UTYPE fy_start, fx_start, cy_start, cx_start, fy, fx, cy, cx, sy, sx, ufx, ufy;
    /* Zeiger auf Verschiebungsvektoren bzw. Filterdaten 
     */
    register UTYPE *cyss_ptr = NULL, *cxss_ptr = NULL, *data = NULL;    
    int     oy, ox, ooy, oox;                   /* Offsets in Filterdaten     */
    data = g_data;                              /* Zeiger auf die Filterdaten */

    for (oy = 0, fy_start = 0, cy_start = 0, sy = 0, cyss_ptr = g_cyss; 
         fy_start < g_dst * g_size_y;fy_start += g_size_y, sy++, oy += g_size_x * g_size_y*g_dst) 
         /* g_dst Summen in y-Richtung berechnen */
    {
        ufy = g_size_y + fy_start;          /* Steuerung Anzahl Filterwerte (y-Rtg) fuer Summe */
        for (ox = 0, fx_start = 0, cx_start = 0, sx = 0, cxss_ptr = g_cxss; 
             fx_start < g_dst * g_size_x;fx_start += g_size_x, sx++, ox += g_size_x)
            /* g_dst Summen in x-Richtung berechnen */
        {

            /* Eine Convolutionssumme aus "g_size_y x g_size_x" - Filterwerten berechnen       */
            summe = 0;
            ufx = g_size_x + fx_start;   /* Steuerung Anzahl Filterwerte (x-Rtg) fuer Summe    */
            for (ooy = oy + ox, fy = fy_start, cy = cy_start; fy < ufy; 
                fy++, cy++, ooy += g_size_x * g_dst)
            {
                if (ony[cy]==1)  /* Zeilenflag */
                {
                    /* Mindestens ein Punkt in Zeile des Masters gesetzt */
                    for (oox = ooy, fx = fx_start, cx = cx_start; fx < ufx; fx++, cx++, oox++)
                    {
                        /* Zeile bearbeiten */
                        /* Alternative:if(c[cy][cx]) summe += *(data + oox);*/
                        summe += *(data + oox) & c[cy][cx];   /* Filter mit Master verknuepfen */
                    }
                }
            }
            sum[sy][sx] = summe;         /* Summe in Feld schreiben. Dient spaeter zur Ausgabe */
            /* Convolutionssumme berechnet*/
            cx_start += *cxss_ptr++;     /* Neuer x-Startwert fuer Plazierung des Master       */
        }
        cy_start += *cyss_ptr++;         /* Neuer y-Startwert fuer Plazierung des Master       */
    }
}

/* Mit dieser Prozedur kann man den aktuellen Filter (vgl. setfilter) ausgegeben */
void    outglobals(void)
{
    unsigned int m, ix, iy;
    int     wert;
    printf("PIXELSTART\n");
    for (m = 0; m < MAXPS; m++)
        printf("%2d ", *(g_cxss + m));
    printf("\nFILTER:\n");
    for (iy = 0; iy < g_size_y * g_dst; iy++)
    {
        for (ix = 0; ix < g_size_x * g_dst; ix++)
        {
            wert = *g_data++;
            printf("%02d  ", wert);
        }
        printf("\n");
    }
}

/*
 * Skalierungs-Prozeduren fuer Scanline-Konvertierungsmethode: 
 * fastshrink(): Verkleinerung  aller Typen (IMG_MSB_4, IMG_MSB_8, IMG_OTHER); Mittels 
 * sh_numcode() wird ein Bit-Code berechnet. Dieser dient zur Entscheidung welche Zeile bzw.
 * Spalte der Quell-Bitmap im resultierenden XImage erscheint (=1) bzw. weggelassen wird (=0). 
 * Parameter:  Quelle, Ziel, Ausgabe-Prozedur, Skalierungswert dst (Skalierungsfaktor dst/SRC)
 */
void fastshrink(struct SrcBmp srcbmp, XImage *theImage, 
                void (*outfkt) (XImage *, short lpy, short lpx), UTYPE srk_factor)
{
    byte   *p, *p_line;                  /* Zeiger innerhalb des frame_buffer */
    int     rx, ix, iy, i, k;
    UTYPE   msk, src, dst;               /* Maske fuer Bit-Test ...           */
    short   targetline, ex;              /* Offsets in XImage-Daten           */
    int dst_height;
    
    if (srcbmp.frame_height % SRC != 0)
        dst_height = (srcbmp.frame_height/ SRC + 1) * SRC;
    else
        dst_height = srcbmp.frame_height;

#ifdef TGL   
    dst_height = srcbmp.frame_height;   
#endif
       
    src = srcbmp.frame_width * 8;
    dst = (src * srk_factor) / SRC;        
    sh_numcode(numcode_x, 0, 0, src, dst);      /* Bit-Code fuer Breite */
    src = srcbmp.frame_height;
    dst = (src * srk_factor) / SRC;
    sh_numcode(numcode_y, 0, 0, src, dst);      /* Bit-Code fuer Hoehe  */

    init_sr_x(srcbmp.frame_width);              /* Initalisieren zu addierender Summen */

    targetline = ((dst_height) * srk_factor) / SRC;     /* letzte Zeile */
#ifdef TGL
    targetline--;
#endif
    
    /* Auswertung des Bit-Codes */
    
    for (p_line = srcbmp.frame_buffer + (srcbmp.frame_height - 1) * srcbmp.frame_width, 
             iy = srcbmp.frame_height - 1; iy >= 0; iy--, p_line -= srcbmp.frame_width) 
        if (numcode_y[iy] > 0)         /* Zeile erscheint, bearbeiten nach Bit-Code fuer Breite */
            for(targetline--,p=p_line,ex=0, rx = 0, ix = 0;ix < srcbmp.frame_width;rx+=8,ix++,p++)
                if (*p)                /* mind. 1 Bit gesetzt */
                    for (msk = 0x80, i = 0; i < 8; i++, msk >>= 1)
                        if ((*p & msk) && (numcode_x[rx + i] > 0))
                        {              
                                       /* Punkt in frame_buffer gesetzt und soll erscheinen     */
                            (*outfkt) (theImage, targetline, ex); /* Ausgape-Prozedur aufrufen  */
                            ex++;
                        }
                        else
                            ex += numcode_x[rx + i];  /* Offset falls Test negativ      */
                else
                    ex += sr[ix];                     /* Offset falls Byte-Test negativ */
}



/*
 * Modifizierter Bresenham-Linien-Algorithmus (DDA, 8-way-stepping) fuer den ersten
 * Oktanten d.h. 0 <= dy/dx <= 1 . Entscheidungsvariable "d" gibt an, ob - ausgehend
 * von Punkt (x,y) - als naechster Punkt (x+1,y) oder (x+1,y+1) gesetzt wird (d<=0 bzw.
 * d>0), und hier ob bit=0 oder bit=1 gesetzt wird. Damit wird der Vektor bc[]
 * gefuellt, der dann zur Entscheidung dient, welche Zeile bzw. Spalte im Quell-Bild
 * auch im verkleinerten Ziel-Bild erscheinen wird. 
 */
void    sh_numcode(int *bc, int x0, int y0, int x1, int y1)
{
    int     dx, dy, d, x, iE, iNE, bit, i;
    dx = x1 - x0;                        /* Steigung 0<=dy/dx<=1 d.h 1.Oktant ! */
    dy = y1 - y0;
    d = 2 * dy - dx;                     /* Entscheidungsvariable: Anfangsfehler */
    iE = 2 * dy;                         /* Addiere, falls Punkt (x+1,y) */
    iNE = 2 * (dy - dx);                 /* Addiere, falls Punkt (x+1,y+1) */

    bit = -1;
    for (i = 0, x = x0; x < x1; x++)
    {
        if (d <= 0)
        {                                /* Punkt (x+1,y) */
            d += iE;
            bit = 0;
        }
        else
        {                                /* Punkt (x+1,y+1) */
            d += iNE;
            bit = 1;
        }
        bc[i++] = bit;                   /* Vektor fuellen */
    }
}

/*
 * Skalierungs-Prozeduren fuer Scanline-Konvertierungsmethode: 
 * fastexpand: Vergroesserung  aller Typen (IMG_MSB_4, IMG_MSB_8, IMG_OTHER)
 * Mittels ex_numcode() wird ein Zahlen-Code berechnet. Dieser dient zur Entscheidung 
 * wie oft welche Zeile bzw. Spalte der Quell-Bitmap im resultierenden XImage erscheinen wird. 
 * Parameter:  Quelle, Ziel, Ausgabe-Prozedur, Skalierungswert dst (Skalierungsfaktor dst/SRC)
 */
void    fastexpand(struct SrcBmp srcbmp, XImage *theImage,
                   void (*outfkt) (XImage *, short lpy, short lpx), UTYPE srk_factor)
{
    byte   *p = NULL, *p_line = NULL;    /* Zeiger innerhalb des frame_buffer */
    int     rx, ix, iy, i, k, on_flag;
    UTYPE   msk, src, dst;               /* Maske fuer Bit-Test ...  */
    short   targetline, ex;              /* Offsets in XImage-Daten  */
    unsigned long target, source;        /* Offsets in XImage-Daten  */
    int dst_height;
     
    if (srcbmp.frame_height % SRC != 0)
        dst_height = (srcbmp.frame_height/ SRC + 1) * SRC;
    else
        dst_height = srcbmp.frame_height;   

#ifdef TGL
    dst_height = srcbmp.frame_height;
#endif 
 
    src = srcbmp.frame_width * 8;
    dst = (src * srk_factor) / SRC;
    ex_numcode(numcode_x, 0, 0, dst, src);      /* Zahlen-Code fuer Breite */
    src = srcbmp.frame_height;
    dst = (src * srk_factor) / SRC;
    ex_numcode(numcode_y, 0, 0, dst, src);      /* Zahlen-Code fuer Hoehe  */

    init_sr_x(srcbmp.frame_width);              /* Initalisieren zu addierender Summen    */

    targetline = ((dst_height) * srk_factor) / SRC; /* Eine Zeile nach letzter Ziel-Linie */
#ifdef TGL
    targetline--;
#endif

    /* Auswertung des Zahlen-Codes */    

    for (p_line = srcbmp.frame_buffer + (srcbmp.frame_height - 1) * srcbmp.frame_width, 
         iy = srcbmp.frame_height - 1; iy >= 0; iy--, p_line -= srcbmp.frame_width)
    {
        targetline--;                    /* 1.Linie direkt zeichnen */
        for (on_flag = 0, p = p_line, ex = 0, rx = 0, ix = 0; ix < srcbmp.frame_width; 
             rx += 8, ix++, p++)
        {
            if (*p)
            {                            /* mind. 1 Bit gesetzt */
                on_flag = 1;
                for (msk = 0x80, i = 0; i < 8; i++, msk >>= 1)
                    if (*p & msk)        /* Bits testen; Punkt in frame_buffer gesetzt => TRUE */
                    {
                        for (k = 0; k < numcode_x[rx + i]; k++)
                        {                /* Soviele Punkte im Ziel setzen wie numcode   angibt */
                            (*outfkt) (theImage, targetline, ex);
                            ex++;
                        }
                    }
                    else
                        ex += numcode_x[rx + i];        /* Offset falls Bit-Test negativ      */
            }
            else
                ex += sr[ix];                           /* Offset falls Byte-Test negativ     */
        } /* end for ix */
        /*
         * Falls mind. ein Bit in einer Zeile gesetzt ist, wird diese Zeile je nach
         * Zahlencode X-mal in die XImage-Daten  kopiert 
         */
        if (on_flag == 1)
        {
            source = targetline * theImage->bytes_per_line;
            for (target = source - theImage->bytes_per_line, k = 1; k < numcode_y[iy]; 
                        k++, target -= theImage->bytes_per_line)
                memcpy(theImage->data + target, theImage->data + source, 
                                       (size_t) theImage->bytes_per_line);
        }
        targetline -= numcode_y[iy] - 1;
    }  /* end for iy */

}
/*
 * Modifizierter Bresenham-Linien-Algorithmus (DDA, 8-way-stepping) fuer den zweiten
 * Oktanten d.h. dy/dx >= 1 . Entscheidungsvariable "d" gibt an, ob - ausgehend von
 * Punkt (x,y) - als naechster Punkt (x,y+1) oder (x+1,y+1) gesetzt wird. Die Summe
 * der Punkte einer Spalte wird damit gebildet. Daraus wird der Vektor "bc[]" gefuellt, 
 * der z.B. so aussieht "112112..." . Dieser dient dann zur Entscheidung , wie oft 
 * (hier z.B. 1mal oder 2mal) welche Zeile bzw. Spalte der Quell-Bitmap im 
 * vergroesserten Ziel-XImage erscheinen wird. 
 */
void    ex_numcode(int *bc, int x0, int y0, int x1, int y1)
{
    int     i, dx, dy, d, y, iE, iNE, t;
    int     summe;
    t = x0;                              /* Werte tauschen */
    x0 = y0;
    y0 = t;
    t = x1;
    x1 = y1;
    y1 = t;
    dx = x1 - x0;                        /* Steigung dy/dx>=1 d.h 2.Oktant ! */
    dy = y1 - y0;
    d = 2 * dx - dy;                     /* Entscheidungsvariable: Anfangsfehler */
    iE = 2 * dx;                         /* Addiere, falls Punkt (x,y+1)         */
    iNE = 2 * (dx - dy);                 /* Addiere, falls Punkt (x+1,y+1)       */

    for (summe = 0, i = 0, y = y0; y < y1; y++)
    {
        if (d <= 0)
        {                                /* Punkt (x,y+1) */
            d += iE;
            summe++;                     
        }
        else
        {                                /* Punkt (x+1,y+1)                  */
            d += iNE;                    /* Schritt in die naechste Spalte   */
            summe++;                     /* Punkte einer Spalte aufaddieren  */
            bc[i++] = summe;             
            summe = 0;
        }
    }
    if (summe != 0)
        bc[0] += summe;                  /* Startwert korrigieren */
}


/*
 * Skalierungs-Prozeduren fuer Scanline-Konvertierungsmethode: 
 * img4fastexpand: Vergroesserung  XImage-Typ IMG_MSB_4 . Mittels ex_numcode() wird ein 
 * Zahlen-Code berechnet. Dieser dient zur Entscheidung wie oft welche Zeile bzw. Spalte 
 * der Quell-Bitmap im resultierenden XImage erscheinen wird. 
 * Die Routine "makeline" wird zum Setzen der Punkte einer Linie verwendet 
 * Parameter:  Quelle, Ziel,Pixelwert der Vordergrundfarbe , Skalierungswert dst 
 *             (Skalierungsfaktor dst/SRC)
 */
void    img4expand(struct SrcBmp srcbmp, XImage *theImage,unsigned long BPxl, UTYPE srk_factor)
{
    byte   *p = NULL, *p_line = NULL;    /* Zeiger innerhalb des frame_buffer */
    int     rx, ix, iy, i, k, on_flag;
    UTYPE   msk, src, dst;                         /* Maske fuer Bit-Test ... */
    long    ex;                                    /* Offset in XImage-Daten  */
    unsigned long target, source, lineoffset;      /* Offsets in XImage-Daten */
    int dst_height; 
     
    if (srcbmp.frame_height % SRC != 0)
        dst_height = (srcbmp.frame_height/ SRC + 1) * SRC;
    else
        dst_height = srcbmp.frame_height;

    src = srcbmp.frame_width * 8;                  /* Zahlen-Code fuer Breite */
    dst = (src * srk_factor) / SRC;
    ex_numcode(numcode_x, 0, 0, dst, src);
    src = srcbmp.frame_height;
    dst = (src * srk_factor) / SRC;
    ex_numcode(numcode_y, 0, 0, dst, src);         /* Zahlen-Code fuer Hoehe */

    init_sr_x(srcbmp.frame_width);                 /* Initalisieren zu addierender Summen */
    
    /* Offset eine Zeile nach letzter Ziellinie 
     */
    lineoffset = theImage->bytes_per_line * ((dst_height) * srk_factor) / SRC;  
#ifdef TGL
 lineoffset-=theImage->bytes_per_line;
#endif

    /* Auswertung des Zahlen-Codes */
 
    for (p_line = srcbmp.frame_buffer + (srcbmp.frame_height - 1) * srcbmp.frame_width, 
         iy = srcbmp.frame_height - 1; iy >= 0; iy--, p_line -= srcbmp.frame_width)
    {
        lineoffset -= theImage->bytes_per_line;          /* 1.Linie direkt zeichnen */
        for (on_flag = 0, p = p_line, ex = 0, rx = 0, ix = 0; ix < srcbmp.frame_width; 
             rx += 8, ix++, p++)
        {
            if (*p)
            {                            /* mind. 1 Bit gesetzt */
                on_flag = 1;
                for (msk = 0x80, i = 0; i < 8; i++, msk >>= 1)
                {
                    /** Bits testen; Punkt in frame_buffer gesetzt => TRUE ; so viele
                     *  Punkte im Ziel setzen wie numcode angibt.*/
                    if (*p & msk)
                        makeline(theImage,BPxl,lineoffset, ex, numcode_x[rx + i]);
                    ex += numcode_x[rx + i];    /* Offset neu berechnen */
                }
            }
            else
                ex += sr[ix];            /* Offset falls Byte-Test negativ */
        }
        /*
         * Falls mind. ein Bit in einer Zeile gesetzt ist, wird diese Zeile je nach
         * Zahlencode X-mal in die XImage-Daten kopiert 
         */
        if (on_flag == 1)
        {
            source = lineoffset;
            for (target = source - theImage->bytes_per_line, k = 1; k < numcode_y[iy]; 
                    k++, target -= theImage->bytes_per_line)
                memcpy(theImage->data + target, theImage->data + source, 
                                       (size_t) theImage->bytes_per_line);
                                       
            /*  ganze Zeile kopieren(theImage->bytes_per_line) da altes XImage 
                ueberschrieben werden muss */
        }
        lineoffset -= (numcode_y[iy] - 1) * theImage->bytes_per_line;  /* Offset neu berechnen */
    }
}

/*
 * Diese Prozedur setzt "np" schwarze Punkte in einer Zeile ("lineoffset") des XImage 
 * (Typ IMG_MSB_4) ab dem Punkt "ex" (wird nur in img4expand verwendet).
 */
void    makeline(XImage *theImage, unsigned long BPxl, unsigned long lineoffset, long ex, int np)
{
    int     ix;
    byte   *ptr;
    unsigned long offset;                /* Offset in XImage-Daten  */

    ptr = (byte *)theImage->data;        /* Zeiger auf XImage-Daten */
    offset = lineoffset + (ex >> 1);     /* 4-bits-per-pixel => vertikalen Offset halbieren */
    if (ex & 0x1)
    {                                    /* ex ungerade */
        ptr[offset] = (ptr[offset] & 0xf0) | BPxl;      /* Pixel-Farbwerte kombiniert
                                                         * in XImage schreiben */
        np--;                            /* Zahl der Punkte vermindert sich */
        offset++;                        /* Neuer Punkt liegt im naechsten Byte */
    }
    /* Eine gerade Anzahl ("np>>1") der restlichen Punkte setzen */
    for (ix = 1; ix <= (np >> 1); ix++)
        ptr[offset + (ix >> 1)] = (BPxl << 4) | BPxl;   /* Pixel-Farbwerte kombiniert
                                                         * in XImage schreiben */
    offset += np >> 1;               /* Offset berechnen */
    if (np & 0x1)                    /* Falls "np" noch ungerade, ist noch ein Punkt zu setzen */
        ptr[offset] = (BPxl << 4) | (ptr[offset] & 0x0f);       /* Pixel-Farbwerte
                                                                 * kombiniert in XImage
                                                                 * schreiben */
}

/*
 * Vorberechnung der Summen, die in den Prozeduren "fastexpand", "fastshrink" bzw.
 * "img4expand" zum vertikalen Offset addiert werden, falls innerhalb eines Bytes des
 * Framebuffers "frame_buffer" kein Bit gesetzt ist. 
 */
void    init_sr_x(int length)
{
    register int rx, ix, i;
    for (rx = 0, ix = 0; ix < length; ix++, rx += 8)    /* Zahl der Summen = length */
    {
        sr[ix] = 0;
        for (i = 0; i < 8; i++)            /* innerhalb eines Bytes d.h. also 8 Bit */
            sr[ix] += numcode_x[rx + i];
    }
}

/*
 * Hier werden die fuer die Convolution verwendeten Filter initialisiert. Falls
 * NORMFILTER definiert ist, wird ueber die Filter eine Normalverteilung gelegt und
 * daraus die Filterdaten bestimmt ("norm_data"). Andernfalls handelt es sich um
 * Boxfilter ("make_data"). 
 */
int     init_filter(unsigned long num_colors)
{
    int     fnum, i, j;
    UTYPE   y[VECTORXY], x[VECTORXY];
    UTYPE   dst, size_y, size_x, *data = NULL;
    size_t  num;

#ifdef NORMFILTER
    stretch = MAXVALUE / makemax();                 /* Skalierungsvariable berechnen */
#endif
    for (i = 0; i < NUMFILTER; i++)                 /* ps[][] loeschen               */
        for (j = 0; j < MAXPS; j++)
            ps[i][j] = 0;

    for (fnum = 0; fnum < NUMFILTER; fnum++)
    {
        for (i = 0; i < VECTORXY; i++)              /* y[],x[] loeschen              */
            y[i] = x[i] = 0;
            
        dst = fnum + 1;                             /* Zielgroesse <-> Quellgroesse  */

        if (dst <= SRC)                             /* Verkleinern und Identitaet    */
            size_y = size_x = (SRC % dst != 0) ?
                    (SRC / dst + 1) : (SRC / dst);  /* reelle Filtergroesse aufrunden  
                                                     * falls nicht ganzzahlig */
        else                                        /* Vergroessern */
            size_y = size_x = 2;                   

        /* Speicher fuer Filterdaten alloziieren 
         */
        num = size_x * dst * size_y * dst;
        if ((data = (UTYPE *) calloc(num, (size_t) sizeof(UTYPE))) == NULL)
            return (1);

        if (dst <= SRC)
        {                                /* Verkleinerungsfilter berechnen */
            if (make_ss_xy(y, x, dst, size_y) != 0)
                return (1);

#ifdef NORMFILTER

            norm_data(y, x, data, (UTYPE) (size_y * dst), (UTYPE) (size_x * dst), size_y, size_x);
#else
            make_data(y, x, data, (UTYPE) (size_y * dst), (UTYPE) (size_x * dst));
#endif
        }
        else
        {                                /* Vergroesserungsfilter berechnen */
            if (ex_ss_xy(y, x, dst, size_y) != 0)
                return (1);

#ifdef NORMFILTER

            norm_data(y, x, data, (UTYPE) (size_y * dst), (UTYPE) (size_x * dst), size_y, size_x);
#else
            make_data(y, x, data, (UTYPE) (size_y * dst), (UTYPE) (size_x * dst));

#endif
        }

        /* Filterstruktur fuellen */
        filter[fnum].dst = dst;          /* Zielgroesse <-> Quellgroesse */
        filter[fnum].size_y = size_y;    /* Groesse einer Filtereinheit in y-Richtung */
        filter[fnum].size_x = size_x;    /* Groesse einer Filtereinheit in x-Richtung */
        filter[fnum].cyss = &ps[fnum][0];/* Zeiger auf Verschiebung in y-Richtung */
        filter[fnum].cxss = &ps[fnum][0];/* Zeiger auf Verschiebung in x-Richtung */
        filter[fnum].data = data;        /* Zeiger auf Filterdaten */
    }
    if (num_colors > 0)
        init_tab(num_colors);            /* Farbuebersetzungstabelle initialisieren */
    else
    {
        printf("ERROR num_colors=%d\n", num_colors);
        return (2);
    }
    return (0);
}

/*
 * Hier werden die Vektoren x und y fuer einen Filter und die Werte, um die dieser
 * verschoben wird "ps[][]" berechnet. Daraus ergeben sich die Eintraege der
 * Filterdaten vgl. "make_data" bzw. "norm_data". Mithilfe der exakten reellen Werte
 * auf die der Filter plaziert werden muesste wird der geeignetste ganzzahlige Wert
 * bestimmt, auf den der Filter bzw. der Master dann tatsaechlich plaziert wird. Je
 * groesser die Ueberdeckung des exakten Intervalls umso geeigneter ist der Wert fuer
 * die Plazierung. Aus dem Schnitt jeder Filterzelle mit dem exakten Intervall
 * resultiert dann der Eintrag in den Vektor x bzw. y. Dabei ergibt die Summe von
 * "size_y" Eintraegen immer "SRC". 
 */
int     make_ss_xy(UTYPE * y, UTYPE * x, UTYPE dst, UTYPE size_y)
{
    register unsigned int i;
    double  a0, a1, z0, z1, m, m_old;    /* Filterenden; Plazierungen */
    double  l0, l1, r0, r1, f0, f1;      /* Fehler                    */
    double  l_exact, r_exact;            /* Exakte Plazierungen       */
    int     r;
    register unsigned int k, sy, mi, f_sum;
    double  rl;
    double  old, new;                    /* zur Filterverschiebung */
    k = 0;
    r = 0;
    old = 0;
    for (m_old = 1, i = 0; i < dst; i++)
    {

        l_exact = (i) * (double) (SRC) / (double) dst;  /* Exakte Plazierungen */
        r_exact = (i + 1) * (double) (SRC) / (double) dst;

        a0 = m_old - 1;                  /* linkes Filterende  */
        a1 = m_old;

        z0 = a0 + size_y;                /* rechtes Filterende */
        z1 = a1 + size_y;

        l0 = fabs(a0 - l_exact);         /* Fehler links       */
        l1 = fabs(a1 - l_exact);

        r0 = fabs(r_exact - z0);         /* Fehler rechts      */
        r1 = fabs(r_exact - z1);

        f0 = l0 + r0;                    /* Gesamtfehler       */
        f1 = l1 + r1;

        f_sum = SRC;                     /* Summe von "size_y" Filterzellen = SRC   */

        if (f0 <= f1)
        {
            m = a0;                      /* Plazierung links   */
            rl = l_exact - m;
            rl = 1 - rl;
            if (fabs(rl) < EPS)
                rl = 1.0;
            sy = rl * dst + EPS;         /* Wert fuer Filterzelle = Vektor x bzw. y */
            f_sum -= sy;
            x[k] = y[k] = sy;
            k++;
            for (mi = 2; mi < size_y; mi++)     /* Mittlere Filterzellen auffuellen */
            {
                sy = 1 * dst + EPS;      /* Fuellen mit maximalem Wert */
                f_sum -= sy;
                x[k] = y[k] = sy;
                k++;
            }

        }
        else
        {
            m = a1;                           /* Plazierung rechts */
            rl = m - l_exact;
            sy = (rl * dst + 1 * dst) + EPS;  /* Wert fuer Filterzelle = Vektor x bzw. y */
            f_sum -= sy;
            x[k] = y[k] = sy;
            k++;
            for (mi = 2; mi < size_y; mi++)   /* Mittlere Filterzellen auffuellen */
            {
                sy = 1 * dst + EPS;           /* Fuellen mit maximalem Wert       */
                f_sum -= sy;
                x[k] = y[k] = sy;
                k++;
            }

        }
        /* Falls noch Rest von "f_sum" uebrig, wird weitere Filterzelle gefuellt */
        if (f_sum != 0)
        {
            sy = f_sum;          /* Wert fuer Filterzelle = Vektor x bzw. y */
            x[k] = y[k] = sy;
            k++;
        }
        else
        {                        /* printf("Warning\n"); return(2); */
        }

        new = m;                 /* berechneter Wert fuer Plazierung des Filters bzw. Masters */
        if (r > 0)
            ps[dst - 1][r - 1] = (UTYPE) (new - old);      /* Verschiebungsvektor beschreiben */
        else
            ps[dst - 1][r] = 0;  /* Startwert fuer Verschiebung = 0 */
        old = new;
        r++;
        m_old = m + size_y; /* Ganzzahliges rechtes Ende des soeben berecheten Filterstueckes */
    }                                    
    return (0);
}



/*
 * Vergleiche "make_ss_xy". Hier werden allerdings die Filter behandelt, die zur
 * Vergroesserung verwendet werden . 
 */
int     ex_ss_xy(UTYPE * y, UTYPE * x, UTYPE dst, UTYPE size_y)
{
    register unsigned int i, j;
    double  a0, a1, z0, z1, m, m_old;    /* Filterenden; Plazierungen */
    double  s, s0, s1;                   /* Fehler bzw. Schnitte      */
    double  l_exact, r_exact;            /* Exakte Plazierungen       */
    UTYPE   sy;
    int     r;
    register unsigned int k;
    double  old, new;                    /* zur Filterverschiebung    */
    k = 0;
    r = 0;
    old = 0;
    for (m_old = 1, i = 0; i < dst; i++)
    {

        l_exact = (i) * (double) (SRC) / (double) dst;  /* Exakte Plazierungen */
        r_exact = (i + 1) * (double) (SRC) / (double) dst;

        a0 = m_old - 1;                  /* linkes Filterende  */
        a1 = m_old;

        z0 = a0 + size_y;                /* rechtes Filterende */
        z1 = a1 + size_y;

        /* Exakte Intervalle mit Filter schneiden */
        s0 = schnitt(l_exact, r_exact, (double) a0, (double) z0);
        s1 = schnitt(l_exact, r_exact, (double) a1, (double) z1);
        /*
         * Je nach Schnitt von [l_exact,r_exact] mit [a0,z0] bzw. [a1,z1] entscheidet
         * sich, wie der Filter weitergeschoben wird 
         */
        if (s0 >= s1)
        {
            m = a0;                      /* Plazierung links   */
        }
        else
        {
            m = a1;                      /* Plazierung rechts  */
        }
        for (j = 0; j < size_y; j++)
        {                                /* Schneide Filterzellen mit exaktem Intervall  */
            s = schnitt(l_exact, r_exact, m + j, m + j + 1);
            sy = s * dst + EPS;          /* Wert fuer Filterzelle = Vektor x bzw. y      */
            x[k] = y[k] = sy;
            k++;
        }

        new = m;                         /* berechneter Wert fuer Plazierung des
                                          * Filters bzw. Masters */
        if (r > 0)
            ps[dst - 1][r - 1] = (UTYPE) (new - old); /* Verschiebungsvektor beschreiben */
        else
            ps[dst - 1][r] = (UTYPE) 0.0;/* Startwert fuer Verschiebung = 0 */
        old = new;
        r++;
        m_old = m + 1;                   /* neuer Wert zur Entscheidung ueber Plazierung */
    }                                    
    return (0);
}

#ifdef NORMFILTER
/*
 * Berechnung der Eintraege in die Filterdaten (my = Gesamthoehe des Filters mx =
 * Gesamtbreite des Filters). Die Gaussfunktion "f()" liegt ueber dem Filter . Es wird
 * ueber Teilflaechen der Filtergesamtflaeche  integriert. Daraus ergeben
 * sich die Eintraege der Filterdaten 
 */
void    norm_data(UTYPE * y, UTYPE * x, UTYPE * data, UTYPE my, UTYPE mx,
                UTYPE size_y, UTYPE size_x)
{
    UTYPE   iy, ix, xx, yy;
    double  sxneu, syneu, erg, a, b, c, d;
    for (yy = 0; yy < my; yy += size_y)
    {
        syneu = 0;
        for (iy = yy; iy < size_y + yy; iy++)
        {
            c = syneu;                          /* Gitterpunkte y-Achse */
            syneu += y[iy];
            d = syneu;
            c = (double) (IL * c) / SRC - (double) IL / 2; /* Skalieren */
            d = (double) (IL * d) / SRC - (double) IL / 2;

            for (xx = 0; xx < mx; xx += size_x)
            {
                sxneu = 0;
                for (ix = xx; ix < size_x + xx; ix++)
                {
                    a = sxneu;           /* Gitterpunkte x-Achse */
                    sxneu += x[ix];
                    b = sxneu;
                    a = (double) (IL * a) / SRC - (double) IL / 2; /* Skalieren   */
                    b = (double) (IL * b) / SRC - (double) IL / 2;

                    erg = simpson(a, b, c, d, 99999.9) * stretch;  /* Integrieren */
                    if (erg > MAXVALUE)
                        erg = (double) MAXVALUE;
                    *data++ = (UTYPE) erg;      /* Filtereintraege setzen */
                }
            }
        }
    }
}
/* Integration */
double  simpson(double a, double b, double c, double d, double old)
{
    double  z0, z1, z, hx, hy, x0, x1, x2, y0, y1, y2;
    hx = (b - a) / 2; hy = (d - c) / 2; 
    x0 = a; x1 = (a + b) / 2; x2 = b;
    y0 = c; y1 = (c + d) / 2; y2 = d;
    z = ((hx * hy) / 9) * (f(x0, y0) + f(x2, y0) + f(x2, y2) + 4 * (f(x0, y1) + f(x1, y0) + f(x2, y1) + f(x1, y2)) +
            16 * f(x1, y1));             
    if (fabs(z - old) > 0.1)
    {                                    
        z0 = simpson(a,(a+b)/2,c,(c+d)/2,z/4)+simpson((a+b)/2,b,c,(c+d)/2,z/4);
        z1 = simpson(a,(a+b)/2,(c+d)/2,d,z/4)+simpson((a+b)/2,b,(c+d)/2,d,z/4);
        z = z1 + z0;
        return (z);
    }
    else
        return (z);
}
/* Liefert Werte der dichte der Normalverteilung */
double  f(double x, double y)
{
    double  z;
    z = ((double) 1 / (2 * PI)) * exp(-0.5 * (x * x + y * y));
    return (z);
}
/* Berechnet den maximalen Wert des Rauminhaltes (exakter Wert=1) unter der
 * Dichtefunktion. Bei der Berechnung normalverteilter Filter mittels
 * Integration wird dieser Wert zur Skalierung benoetigt.*/
double  makemax(void)
{
    double  s, a, b, c, d, x, y;
    s = 0;
    for (y = 0; y < SRC; y++)
    {                                    /* Skalierte Gitterpunkte y-Achse */
        c = (double) (IL * y) / SRC - (double) IL / 2;
        d = (double) (IL * (y + 1)) / SRC - (double) IL / 2;
        for (x = 0; x < SRC; x++)
        {                                /* Skalierte Gitterpunkte x-Achse */
            a = (double) (IL * x) / SRC - (double) IL / 2;
            b = (double) (IL * (x + 1)) / SRC - (double) IL / 2;
            s += simpson(a, b, c, d, 99999.9);  /* Aufsummieren der Teilrauminhalte */
        }
    }
    return (s);
}
#else
/*
 * Berechnung der Eintraege in die Filterdaten aus den Vektoren y und x Parameter: y =
 * Zeiger auf Vektor y x = Zeiger auf Vektor x data = Zeiger auf Filterdaten my =
 * Gesamthoehe des Filters mx = Gesamtbreite des Filters 
 */
void    make_data(UTYPE * y, UTYPE * x, UTYPE * data, UTYPE my, UTYPE mx)
{
    UTYPE   iy, ix;
    for (iy = 0; iy < my; iy++)
        for (ix = 0; ix < mx; ix++)
            *data++ = y[iy] * x[ix];
}
#endif

/* Gibt den Inhalt des Filters f und die Werte, um die dieser verschoben wird, aus */
void    out_filter(struct Filter f)
{
    register unsigned int m, ix, iy;
    int     wert;
    printf("PIXELSTART ");
    for (m = 0; m < MAXPS; m++)
        printf("%d ", *(f.cxss + m));    /* Verschieben des Filters */
    printf("\n");
    printf("\nFILTER: %d --> %d\n", SRC, f.dst);
    for (iy = 0; iy < f.size_y * f.dst; iy++)
    {
        for (ix = 0; ix < f.size_x * f.dst; ix++)
        {
            wert = *f.data++;
            printf("%02d ", wert);       /* Filterwerte             */
        }
        printf("\n");
    }
}

/*
 * Berechnung einer internen Tabelle zur Uebersetzung der Convolutions-Summen in Farbwerte
 * Dabei wird das Intervall [0,MAXVALUE] d.h. die Werte der Convolution auf das 
 * Intervall [0,nc-1] d.h. die Farbindizes abgebildet. Parameter: 
 * nc = Zahl der Farben auf die abzubilden ist. 
 */
void    init_tab(unsigned long nc)
{
    long    i;
    double  y;
    for (i = 0; i <= MAXVALUE; i++)
    {
        /* linear */
        y = ((double) (nc - 1) * i) / (double) MAXVALUE;
        y = y + 0.5;
        coltab[i] = (UTYPE) y;
    }
}

/*
 * Freigabe des in init_filter alloziierten Speichers der Filterdaten aller Filter 
 */
void    kill_filter(void)
{
    int     m;
    for (m = 0; m < NUMFILTER; m++)
        if (filter[m].data != NULL)
            free(filter[m].data);
}
