/*
  usage: hz2ps [-s <spec file>] file
  
  Version 1.0
  
  Copyright (C) 1989  Fung Fung Lee (leeumunhum.stanford.edu)
  
  A program to generate a PostScript output file from a HZ file.
  HZ is a 7-bit data format for mixed Chinese (GB) and ASCII text.
  
  The font files and spec files should all be under the directory
  HZDIR (setenv HZDIR the-hanzi-directory).
  
  This program supports both horizontal and vertical printing mode.
  This program does not handle well (English) fonts with variable width.
  This is a simple HZ to PS driver. There are known ways to reduce the sizes
  of PS output files in general by building font dictionaries.

  This program is free for general distribution.  

  This program runs on UNIX. You are welcome to port it to other operating
  systems.

  09/08/89 FFL	Release version 1.0
  09/11/89 FFL	Support "tab", add fputTab(), vfputTab().

*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#define true 1
#define false 0

char *getenv();

extern char *HZbitmap();
extern loadHZfont();
int hzbyte;	/* number of bytes per hanzi image */

char hzdir[80], libdir[100], fontdir[100];
char pmode;
float leftmargin, rightmargin, topmargin, bottommargin;
float hpos, vpos;
int HZbm;
float HZpoint, HZspacing, HZskip;
float Epoint, Espacing, Eskip, Evskip, Ekerning, Espace;
float vskip, hskip;
float hanziPerTab, HZtab;

static char HZDIR[] = "HZDIR";
static char HEADER[] = "hz2ps.prep";
static char SPEC[] = "hz2ps.spec";
static char nullName[] = "";
char *progname, *inputName;
char headerName[100], specName[100];
char hzfont[30], hzName[30];
char engfont[30];
FILE *spec, *fin;

#define DB(hi,lo)	(((hi)&0xFF) << 8 | (lo)&0xFF)
#define isGB(c)		((c)>=0x21 && (c)<=0x7E)


warning()
{
  fprintf(stderr, "usage: %s [-s <spec-file>] hz-file\n", progname); 
  exit(1);
}


#define LEN 100

main(argc, argv)
     int argc;
     char *argv[];
{
  char s[LEN];
  FILE *header;
  
  if (getenv(HZDIR) != NULL)
    {
      strcpy(hzdir, getenv(HZDIR));
      strcpy(libdir, hzdir);
      strcat(libdir, "/lib/");
      strcpy(fontdir, hzdir);
      strcat(fontdir, "/font/");
    }
  else
    {
      fprintf(stderr, "Please setenv %s appropriately.\n", HZDIR);
      exit(1);
    }
  strcpy(headerName, libdir);
  strcat(headerName, HEADER);
  strcpy(specName, libdir);
  strcat(specName, SPEC);
  inputName = nullName;
  
  progname = argv[0];
  switch (argc)
    {
    case 1: break;
    case 2: inputName = argv[1]; break;
    case 4: inputName = argv[3];
    case 3: 
      if (strcmp(argv[1], "-s") != 0) warning();
      strcpy(specName, libdir);
      strcat(specName, argv[2]);
      break;
    default: warning(); break;
    }
  
  if ((spec = fopen(specName, "r")) == 0)
    {
      fprintf(stderr, "Cannot open spec file %s\n", specName);
      exit(1);
    }
  
  fgets(s, LEN, spec);
  sscanf(s, "%c", &pmode);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &leftmargin);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &rightmargin);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &topmargin);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &bottommargin);
  fgets(s, LEN, spec);
  sscanf(s, "%s %d", hzName, &HZbm);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &HZpoint);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &HZspacing);
  HZskip = HZpoint + HZspacing;
  fgets(s, LEN, spec);
  sscanf(s, "%f", &vskip);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &hskip);
  fgets(s, LEN, spec);
  sscanf(s, "%s", engfont);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Epoint);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Eskip);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Espace);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Espacing);
  Evskip = Epoint + Espacing;
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Ekerning);
  fgets(s, LEN, spec);
  if (sscanf(s, "%f", &hanziPerTab) == 1)	/* optional */
    HZtab = hanziPerTab * HZskip;
  else
    HZtab = 4.0 * HZskip;

  sprintf(s, "%d", HZbm);
  strcpy(hzfont, hzName);
  strcat(hzfont, s);
  
  hzbyte = HZbm * HZbm / 8;
  loadHZfont();
  
  if ((header = fopen(headerName, "r")) == 0)
    {
      fprintf(stderr, "Cannot open header file %s\n", headerName);
      exit(1);
    }
  if (strcmp(inputName, nullName) == 0)
    fin = stdin;
  else if ((fin = fopen(inputName, "r")) == 0)
    {
      fprintf(stderr, "Cannot open input file %s\n", inputName);
      exit(1);
    }
  prHeader(header, stdout);
  fclose(header);
  if (pmode == 'h') 
    {     
      hpos = leftmargin;
      vpos = topmargin;
      fputNewLine(stdout);
      scanner(fin, stdout);
    }  
  else if (pmode == 'v')
    {
      hpos = rightmargin;
      vpos = topmargin;
      vfputNewLine(stdout);
      vscanner(fin, stdout);
    }
  else
    {
      fprintf(stderr, "unknown printing mode %c\n", pmode);
      exit(1);
    }
}

prHeader(fin, f)
     FILE *fin, *f;
{
  int c;
  
  fprintf(f, "%%!\n");
  fprintf(f, "%%%%\n");
  fprintf(f, "%%%%Title: %s\n", inputName);
  fprintf(f, "%%%%Creator: hz2ps\n");
  fprintf(f, "%%%%EndComments\n");
  
  while ((c=fgetc(fin)) != EOF)
    putc(c, f);
  
  fprintf(f, "/HZbm %d def\n", HZbm);
  fprintf(f, "/HZpoint %6.2f def\n", HZpoint);
  fprintf(f, "/HZspacing %6.2f def\n", HZspacing);
  fprintf(f, "/HZskip %6.2f def\n", HZskip);
  fprintf(f, "/Epoint %6.2f def\n", Epoint);
  fprintf(f, "/Espacing %6.2f def\n", Espacing);
  fprintf(f, "/Evskip %6.2f def\n", Evskip);
  fprintf(f, "/Ekerning %6.2f def\n", Ekerning);
  fprintf(f, "\/%s Epoint chooseFont\n", engfont);
  fprintf(f, "%%%%\n");
}

GBrange(hi, lo)
int hi, lo;
{
  if (!isGB(hi) || !isGB(lo))
    {
      fprintf(stderr, "GB code out of range: %c%c (hex: %2x%2x)\n",
	      hi, lo, hi, lo);
      exit(1);
    }
}

#define SLEN 80		/* maximum English string length */

scanner(fin, fout)
     FILE *fin, *fout;
{
  int c1, c2, code, i;
  char *image, s[SLEN];
  int GBmode = false;
  
  while ((c1=fgetc(fin)) != EOF)
    {
      if (GBmode)
	{
	  c2 = fgetc(fin);
	  GBrange(c1, c2);
	  code = DB(c1, c2);
	  switch (code)
	    {
	    case 0x7E7D: /* DB('~','}') */
	      GBmode = false; break;
	    case 0x212A: fputDash(0.5, fout); break;
	    case 0x235F: fputDash(0.0, fout); break;
	    case 0x237E: fputDash(1.0, fout); break;
	    default:
	      image = HZbitmap(code);
	      fputHanzi(image, fout);
	      break;
	    }
	}
      /* not in GBmode */
      else if (c1 == '~')
	{
	  c2 = fgetc(fin);
	  switch (c2)
	    {
	    case '~': fputAscii('~', fout); break;
	    case '{': GBmode = true; break;
	    case '\n': break;	/* line-continuation marker */
	    default:
	      fprintf(stderr, "unexpected escape sequence: ~%c\n", c2);
	      /* optional error recovery: */
	      fputAscii('~', fout); ungetc(c2, fin);
	      break;
	    }
	}
      else if (isalnum(c1))
	{
	  s[0] = c1;
	  i = 1;
	  while (i<SLEN-1 && (c1=fgetc(fin)) && isalnum(c1))
	    s[i++] = c1;
	  s[i++] = '\0';
	  if (i<SLEN)
	    ungetc(c1, fin);
	  fputWord(s, fout);
	}
      else
	  fputAscii(c1, fout);
    }
  fprintf(fout, "printpage\n");
}

fputNewPage(f)
     FILE *f;
{
  fprintf(f, "printpage\n");
  hpos = leftmargin;
  vpos = topmargin;
  fputNewLine(f);
}

fputNewLine(f)
     FILE *f;
{
  hpos = leftmargin;
  vpos -= vskip;
  if (vpos < bottommargin)
    fputNewPage(f);
  else
    fprintf(f, "%6.2f %6.2f moveto\n", hpos, vpos);
}

fputHanzi(image, f)
     char *image;	/* image points to hzbyte bytes of bitmap */
     FILE *f;
{
  char c;
  int i;
  
  if (hpos + HZskip > rightmargin)
    fputNewLine(f);
  fprintf(f, "<");
  for (i=0; i<hzbyte; i++)
    {
      c = *image++;
      fprintf(f, "%x%x", (c&0xF0)>>4, c&0x0F);
    }
  fprintf(f, "> HZshow\n");
  hpos += HZskip;
}

fputWord(s, f)
     char *s;
     FILE *f;
{
  if (hpos + strlen(s) * Eskip > rightmargin)
    fputNewLine(f);
  fprintf(f, "(%s) Eshow\n", s);
  hpos += strlen(s) * Eskip;
}

fputTab(f)
     FILE *f;
{
  hpos = (int) ((hpos-leftmargin)/HZtab + 1) * HZtab + leftmargin;
  if (hpos > rightmargin)
    fputNewLine(f);
  else
    fprintf(f, "%6.2f %6.2f moveto\n", hpos, vpos);
}

fputAscii(c, f)
     char c;
     FILE *f;
{
  if (c=='\n' || c=='\r')
    {
      fputNewLine(f);
      return;
    }
  if (c=='\f')
    {
      fputNewPage(f);
      return;
    }
  if (c=='\t')
    {
      fputTab(f);
      return;
    }
  if (hpos + Eskip > rightmargin)
    fputNewLine(f);
  if (c==' ')
    {
      fprintf(f, "( ) Eshow\n");
      hpos += Eskip;
    }
  else if (c=='(' ||  c==')')
    {
      fprintf(f,  "(\\%c) Eshow\n", c);
      hpos += Eskip;
    }
  else if (c=='\\')
    {
      fprintf(f,  "(\\\\) Eshow\n");
      hpos += Eskip;
    }
  else if (isgraph(c))
    {
      fprintf(f, "(%c) Eshow\n", c);
      hpos += Eskip;
    }
  /* ignore other non-graphical (unprintable) ascii characters */	
}

fputDash(scale, f)
     float scale;
     FILE *f;
{
  if (hpos + HZskip > rightmargin)
    fputNewLine(f);
  fprintf(f, "%3f dash\n", scale);
  hpos += HZskip;
}


/* similar code for vertical printing mode */

char vtest(code)
     unsigned int code;
{
  char mode;

  switch (code)
    {
    case 0x212A:	/* middle-dash */
    case 0x212B:	/*  ~  */
    case 0x212D:	/* ... */
    case 0x2132:
    case 0x2133:
    case 0x2134:
    case 0x2135:
    case 0x2136:
    case 0x2137:
    case 0x2138:
    case 0x2139:
    case 0x213A:
    case 0x213B:
    case 0x213C:
    case 0x213D:
    case 0x213E:
    case 0x213F:
    case 0x217A:	/* --> */
    case 0x217B:	/* <-- */
    case 0x217C:	/* up-arrow */
    case 0x217D:	/* down-arrow */
    case 0x2328:	/* ( */
    case 0x2329:	/* ) */
    case 0x235B:	/* [ */
    case 0x235D:	/* ] */
    case 0x235F:	/* lower-dash */
    case 0x237B:	/* { */
    case 0x237C:	/* | */
    case 0x237D:	/* } */
    case 0x237E:	/* upper-dash */
      mode = 'R'; break;
    case 0x2122:	/* dun-hao */
    case 0x2123:	/* period */
    case 0x2321:	/* ! */
    case 0x232C:	/* , */
    case 0x232E:	/* . */
    case 0x233A:	/* : */
    case 0x233B:	/* ; */
    case 0x233F:	/* ? */
      mode = 'Q'; break;
    default:
      mode = 'V'; break;
    }
  return mode;
}

unsigned int replace(code)
     unsigned int code;
{
  switch (code)
    {
    case 0x212E: return 0x2138; break;
    case 0x212F: return 0x2139; break;
    case 0x2130: return 0x2138; break;
    case 0x2131: return 0x2139; break;
    default: return code; break;
    }
}

vscanner(fin, fout)
     FILE *fin, *fout;
{
  int c1, c2, code, i;
  char *image, s[SLEN];
  int GBmode = false, vmode = false;
  
  while ((c1=fgetc(fin)) != EOF)
    {
      if (GBmode)
	{
	  c2 = fgetc(fin);
	  GBrange(c1, c2);
	  code = DB(c1, c2);
	  switch (code)
	    {
	    case 0x7E7D: /* DB('~','}') */
	      GBmode = false; break;
	    case 0x212A: vfputDash(0.5, fout); break;
	    case 0x235F: vfputDash(0.0, fout); break;
	    case 0x237E: vfputDash(1.0, fout); break;
	    default:
	      code = replace(code);
	      image = HZbitmap(code);
	      vmode = vtest(code);
	      vfputHanzi(image, fout, vmode);
	      break;
	    }
	}
      /* not in GB mode */
      else if (c1 == '~')
	{
	  c2 = fgetc(fin);
	  switch (c2)
	    {
	    case '~': vfputAscii('~', fout); break;
	    case '{': GBmode = true; break;
	    case '\n': break;	/* line-continuation marker */
	    default:
	      fprintf(stderr, "unexpected escape sequence: ~%c\n", c2);
	      /* optional error recovery: */
	      vfputAscii('~', fout); ungetc(c2, fin);
	      break;
	    }
	}
      else if (isalnum(c1))
	{
	  s[0] = c1;
	  i = 1;
	  while (i<SLEN-1 && (c1=fgetc(fin)) && isalnum(c1))
	    s[i++] = c1;
	  s[i++] = '\0';
	  if (i<SLEN)
	    ungetc(c1, fin);
	  vfputWord(s, fout);
	}
      else
	  vfputAscii(c1, fout);
    }
  fprintf(fout, "printpage\n");
}

vfputNewPage(f)
FILE *f;
{
    fprintf(f, "printpage\n");
    hpos = rightmargin;
    vpos = topmargin;
    vfputNewLine(f);
}

vfputNewLine(f)
FILE *f;
{
    vpos = topmargin - HZskip;
    hpos -= hskip;
    if (hpos < leftmargin)
    	vfputNewPage(f);
    else
	fprintf(f, "%6.2f %6.2f moveto\n", hpos, vpos);
}

vfputHanzi(image, f, vmode)
char *image;	/* image points to hzbyte bytes of bitmap */
FILE *f;
char vmode;
{
    char c;
    int i;
    
    if (vpos < bottommargin)
        vfputNewLine(f);
    fprintf(f, "<");
    for (i=0; i<hzbyte; i++)
    {
	c = *image++;
	fprintf(f, "%x%x", (c&0xF0)>>4, c&0x0F);
    }
    switch (vmode)
      {
      case 'R': fprintf(f, "> HZvshowR\n"); break;
      case 'Q': fprintf(f, "> HZvshowQ\n"); break;
      default:  fprintf(f, "> HZvshow\n"); break;
      }
    vpos -= HZskip;
}

vfputWord(s, f)
     char *s;
     FILE *f;
{
    if (vpos + HZskip - Epoint - (strlen(s)-1) * Evskip < bottommargin)
        vfputNewLine(f);
    fprintf(f, "(%s) Evshow\n", s);
    vpos -= Epoint + (strlen(s)-1) * Evskip;
}

vfputTab(f)
     FILE *f;
{
  vpos = topmargin - (int)((topmargin-vpos-HZskip)/HZtab + 1)*HZtab - HZskip;
  if (vpos < bottommargin)
    vfputNewLine(f);
  else
    fprintf(f, "%6.2f %6.2f moveto\n", hpos, vpos);
}

vfputAscii(c, f)
     char c;
     FILE *f;
{
  if (c=='\n' || c=='\r')
    {
      vfputNewLine(f);
      return;
    }
  if (c=='\f')
    {
      vfputNewPage(f);
      return;
    }
  if (c=='\t')
    {
      vfputTab(f);
      return;
    }
  if (vpos + HZskip - Epoint - Evskip < bottommargin)
    vfputNewLine(f);
  if (c==' ')
    {
      fprintf(f, "( ) Evshow\n");
    }
  else if (c=='(' ||  c==')')
    {
      fprintf(f,  "(\\%c) Evshow\n", c);
      vpos -= Epoint;
    }
  else if (c=='\\')
    {
      fprintf(f,  "(\\\\) Evshow\n");
      vpos -= Epoint;
    }
  else if (isgraph(c))
    {
      fprintf(f, "(%c) Evshow\n", c);
      vpos -= Epoint;
    }
  /* ignore the unprintable ascii characters */	
}

vfputDash(scale, f)
     float scale;
     FILE *f;
{
  if (vpos < bottommargin)
    vfputNewLine(f);
  fprintf(f, "%3f vdash\n", scale);
  vpos -= HZskip;
}

