// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: bitvec.C,v 1.5 91/02/22 15:58:31 hmgr Exp $";

// An array of bits
//
// By  Parag Patel

#include <stdio.h>
#include "bitvec.h"


// the following are machine-dependant - change them if necessary
// sizeof long == 32 bits == 2**5 --> 5, therefore ...
const int NUM = 32;
const int SHIFT = 5;
const int MASK = 0x1F;
const unsigned long EMPTY = 0L;


// generic minimum function
inline long MIN(long i1, long i2)
{
    return i1 < i2 ? i1 : i2;
}

// this determines which particular "long" in the array has this element
inline long ELT(long e)
{
    return e >> SHIFT;
}

// this gets the bit position in a "long" for this vec element
inline unsigned long BIT(long e)
{
    return ((unsigned long)(1L << (e & MASK)));
}

// this adds an element to an array of longs
inline void ADD(unsigned long *elts, long e)
{
    elts[ELT(e)] |= BIT(e);
}


// return a new vec of elts - clear it
static unsigned long *newelts(long largest)
{
    register unsigned long *elts = new unsigned long[ELT(largest) + 1];

    for (register long i = ELT(largest); i >= 0; i--)
	elts[i] = EMPTY;
    return elts;
}

// bump the size of a vec
void Bitvec::bumpsize(long largest)
{
    if (largest <= num)
	return;

    // only re-allocate our array if we are out of ELTs
    if (ELT(largest) != ELT(num))
    {
	register unsigned long *ne = newelts(largest);
	for (register long i = ELT(num); i >= 0; i--)
	    ne[i] = elts[i];
	delete elts;
	elts = ne;
    }
    num = largest;
}

// various constructors:

// make a new vec of the specified size
Bitvec::Bitvec(long size)
{
	if (size <= 1)
		size = 1;
	elts = newelts(num = size - 1);
}

// dup a vec
Bitvec::Bitvec(const Bitvec &s)
{
    elts = newelts(num = s.num);
    for (register long i = ELT(s.num); i >= 0; i--)
	elts[i] = s.elts[i];
}

// assign a vec to a vec - also a dup
Bitvec &Bitvec::operator=(const Bitvec &s)
{
    register long i, t;

    BUMPSIZE(s.num);
    for (i = t = ELT(s.num); i >= 0; i--)
	elts[i] = s.elts[i];
    for (i = ELT(num); i > t; i--)
	elts[i] = EMPTY;
    return *this;
}


// add an element to a vec
Bitvec &Bitvec::add(long elt)
{
    if (elt < 0)
	return *this;
    BUMPSIZE(elt);
    elts[ELT(elt)] |= BIT(elt);
    return *this;
}

// delete an element from a vec
Bitvec &Bitvec::remove(long elt)
{
    if (elt < 0)
	return *this;
    elts[ELT(elt)] &= ~BIT(elt);
    return *this;
}

// is vec same as s?
boolean Bitvec::operator==(const Bitvec &s) const
{
    register long i = MIN(num, s.num);

    for (i = ELT(i); i >= 0; i--)
	if (elts[i] != s.elts[i])
	    return FALSE;
    if (num == s.num)
	return TRUE;

    register long t;
    if (num < s.num)
    {
	for (t = ELT(num), i = ELT(s.num); i > t; i--)
	    if (s.elts[i])
		return FALSE;
    }
    else
    {
	for (t = ELT(s.num), i = ELT(num); i > t; i--)
	    if (elts[i])
		return FALSE;
    }
    return TRUE;
}


// return the number of elements in the vec (cardinality)
long Bitvec::size() const
{
    register long n = 0;

    for (register long i = ELT(num); i >= 0; i--)
	for (register unsigned long m = 1; m; m <<= 1)
	    if (elts[i] & m)
		n++;
    return n;
}

// clear the vec of all elements
void Bitvec::clear()
{
    for (register long i = ELT(num); i >= 0; i--)
	elts[i] = EMPTY;
}
