IMPLEMENTATION MODULE VT220VDU;

(* Authors:        John Mann and Andrew Trevorrow
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   September, 1985

   Description:
   Implements the InitVT220 procedure that initializes the generic VDU routines
   and parameters used in DVItoVDU.
   DVItoVDU assumes text lines start at 1 and increase down the screen.
   When updating the window in graphics mode (using ShowChar and ShowRectangle),
   DVItoVDU assumes the top left screen pixel is (0,0); h coordinates increase
   to the right and v coordinates increase down.

   Revised:
   January, 1986
 - Chunky bitmap graphics are used in ShowRectangle.  The VT220 is switched
   to 132 column mode and is assumed to be in VT200 mode with 7 bit controls.
   The bottom 20 rows in bitmap represent a 132 by 20 character window region.
   Each character is made up of 5 vertical "pixels" and so 32 (=2^5) characters
   can be down-loaded and used to draw all possible 1 by 5 pixel rectangles.
   December, 1986
 - Tidied up a few minor problems with use of rectcount and ShowBitmap.
*)

FROM VDUInterface IMPORT
   DVIstatusl, windowstatusl, messagel, commandl, bottoml,
   windowh, windowv, windowwd, windowht,
   TeXtoASCII,
   StartText, MoveToTextLine, ClearTextLine, ClearScreen,
   StartGraphics, LoadFont, ShowChar, ShowRectangle,
   ResetVDU;

FROM ScreenIO IMPORT
   Write, WriteString, WriteCard, WriteBuffer;

FROM ANSIVDU IMPORT
   MoveAbs, ANSIMoveToTextLine, ANSIClearTextLine, ANSILoadFont;

CONST
   ESC = 33C;
   SO  = 16C;
   SI  = 17C;
   hpixels = 1;                  (* horizontal pixels per char position *)
   vpixels = 5;                  (* vertical pixels per char position *)
   hscreenmax = 131;             (* max horizontal char coordinate *)
   vscreenmax = 23;              (* max vertical char coordinate *)

TYPE
   bitvalue = [0..vpixels];      (* vpixels is only used as a flag *)
   byteset = SET OF bitvalue;

VAR
   cursrow, curscol,             (* VT220ShowChar remembers cursor location *)
   rectcount : CARDINAL;         (* keeps a count of ShowRectangle calls *)
   bitmap :
      ARRAY [0..hscreenmax],[0..vscreenmax] OF byteset;

(******************************************************************************)

PROCEDURE InitVT220;

(* The dialogue region is the top 4 lines.
   The window region is the remaining area of the screen
   (i.e., the bottom 20 rows in bitmap).
*)

BEGIN
DVIstatusl    := 1;
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := vscreenmax + 1;
(* DVItoVDU's coordinate scheme is the same as the VT220 scheme. *)
windowh  := 0;
windowv  := 4 * vpixels;           (* = height of 4 dialogue lines *)
windowwd := (hscreenmax + 1) * hpixels;
windowht := (vscreenmax + 1) * vpixels - windowv;

StartText      := ShowBitmap;
MoveToTextLine := ANSIMoveToTextLine;
ClearTextLine  := ANSIClearTextLine;
ClearScreen    := VT220ClearScreen;
StartGraphics  := VT220StartGraphics;
LoadFont       := ANSILoadFont;
ShowChar       := VT220ShowChar;
ShowRectangle  := VT220ShowRectangle;
ResetVDU       := VT220ResetVDU;

Write(ESC); WriteString('[?3h');   (* switch to 132 column mode *)
LoadPixels;
rectcount := 0;                    (* for initial ShowBitmap *)
END InitVT220;

(******************************************************************************)

PROCEDURE LoadPixels;

(* Down-load the chunky graphics character set into the VT220. *)

BEGIN
Write(ESC); Write('P');
WriteString('1;1;2;3;2;1{&%C');
WriteString('BBBBBBBB/????????;');
WriteString('KKKKKKKK/????????;');
WriteString('NNNNNNNN/????????;');
WriteString('oooooooo/????????;');
WriteString('rrrrrrrr/????????;');
WriteString('{{{{{{{{/????????;');
WriteString('~~~~~~~~/????????;');
WriteString('????????/BBBBBBBB;');
WriteString('BBBBBBBB/BBBBBBBB;');
WriteString('KKKKKKKK/BBBBBBBB;');
WriteString('NNNNNNNN/BBBBBBBB;');
WriteString('oooooooo/BBBBBBBB;');
WriteString('rrrrrrrr/BBBBBBBB;');
WriteString('{{{{{{{{/BBBBBBBB;');
WriteString('~~~~~~~~/BBBBBBBB;');
WriteString('????????/KKKKKKKK;');
WriteString('BBBBBBBB/KKKKKKKK;');
WriteString('KKKKKKKK/KKKKKKKK;');
WriteString('NNNNNNNN/KKKKKKKK;');
WriteString('oooooooo/KKKKKKKK;');
WriteString('rrrrrrrr/KKKKKKKK;');
WriteString('{{{{{{{{/KKKKKKKK;');
WriteString('~~~~~~~~/KKKKKKKK;');
WriteString('????????/NNNNNNNN;');
WriteString('BBBBBBBB/NNNNNNNN;');
WriteString('KKKKKKKK/NNNNNNNN;');
WriteString('NNNNNNNN/NNNNNNNN;');
WriteString('oooooooo/NNNNNNNN;');
WriteString('rrrrrrrr/NNNNNNNN;');
WriteString('{{{{{{{{/NNNNNNNN;');
WriteString('~~~~~~~~/NNNNNNNN');
Write(ESC); Write('\');
Write(ESC); WriteString(')&%C');   (* set as G1 character set *)
END LoadPixels;

(******************************************************************************)

PROCEDURE ShowBitmap;

(* Display only the flagged characters in the bitmap. *)

VAR h, v : CARDINAL;

BEGIN
IF rectcount > 0 THEN
   Write(SO);
   FOR v := 4 TO vscreenmax DO            (* ignore dialogue lines *)
      FOR h := 0 TO hscreenmax DO
         IF vpixels IN bitmap[h,v] THEN   (* send flagged character *)
            EXCL(bitmap[h,v], vpixels);   (* clear flag *)
            MoveQuick(h,v);
            Write(CHAR(CARDINAL(bitmap[h,v]) + 32));
         END;
      END;
   END;
   Write(SI);
   WriteBuffer;
   rectcount := 0;
END;
END ShowBitmap;

(******************************************************************************)

PROCEDURE VT220ClearScreen;

VAR h, v : CARDINAL;

BEGIN
Write(ESC);
WriteString('[2J');            (* erase entire screen *)
FOR v := 4 TO vscreenmax DO    (* clear bitmap, ignoring dialogue lines 0..3 *)
   FOR h := 0 TO hscreenmax DO
      bitmap[h,v] := byteset{};
   END;
END;
END VT220ClearScreen;

(******************************************************************************)

PROCEDURE VT220StartGraphics;

(* Note that DVItoVDU will only call LoadFont, ShowChar and ShowRectangle
   while in graphics mode.
*)

BEGIN
rectcount := 0;
cursrow := 0;     (* for MoveQuick *)
END VT220StartGraphics;

(******************************************************************************)

PROCEDURE MoveQuick (screenh, screenv : CARDINAL);

(* Move cursor to given screen position.
   We remember the cursor position (cursrow,curscol) so we can reduce the
   output bytes needed to do the next MoveQuick.
   StartGraphics resets the position to an undefined state (cursrow = 0).
   We also reset when the cursor reaches the right edge (= windowwd) to
   avoid possibility of any auto wrap.
*)

VAR amount : CARDINAL;

BEGIN
(* first translate DVItoVDU coordinates into actual screen location *)
INC(screenh);
INC(screenv);
IF cursrow = screenv THEN
   (* The cursor is on the same line as in previous MoveQuick call so we only
      need to move left or right, and probably just a small amount (if at all).
   *)
   IF screenh = curscol THEN       (* cursor in correct location *)
      INC(curscol);                (* cursor will move right when ch written *)
   ELSIF screenh < curscol THEN    (* move cursor left *)
      amount := curscol - screenh;
      Write(ESC); Write('[');
      IF amount > 1 THEN           (* default is 1 col *)
         WriteCard(amount);
         DEC(curscol,amount-1);    (* no need to do this if amount = 1 *)
      END;
      Write('D');
   ELSE                            (* move cursor right *)
      amount := screenh - curscol;
      Write(ESC); Write('[');
      IF amount > 1 THEN WriteCard(amount) END;   (* default is 1 col *)
      INC(curscol,amount+1);
      Write('C');
   END;
ELSE                               (* cursrow undefined or ch on a new line *)
   MoveAbs(screenv,screenh);
   cursrow := screenv;
   curscol := screenh + 1;         (* cursor will move right when ch written *)
END;
IF screenh = CARDINAL(windowwd) THEN   (* ch will be written at right edge *)
   cursrow := 0;                       (* so avoid auto wrap next time around *)
END;
END MoveQuick;

(******************************************************************************)

PROCEDURE VT220ShowChar (screenh, screenv : CARDINAL; ch : CHAR);

(* Show the given Terse character (mapped to ASCII) at the given position. *)

BEGIN
IF rectcount > 0 THEN   (* flush bitmap if ShowRectangle/s are pending *)
   ShowBitmap;
END;
MoveQuick(screenh, screenv DIV vpixels);
Write(TeXtoASCII[ch]);
END VT220ShowChar;

(******************************************************************************)

PROCEDURE VT220ShowRectangle (screenh, screenv,          (* top left pixel *)
                              width, height : CARDINAL;  (* size of rectangle *)
                              ch : CHAR);                (* black pixel *)

(* Set the given rectangular bitmap region.
   DVItoVDU ensures the top left position is visible and the given
   dimensions do not go beyond the window edges.
*)

VAR h, v, vrow : CARDINAL;

BEGIN
FOR v := screenv TO screenv + height - 1 DO
   FOR h := screenh TO screenh + width - 1 DO
      (* set bit h,v in bitmap *)
      vrow := v DIV vpixels;
      INCL(bitmap[h,vrow], v MOD vpixels);
      INCL(bitmap[h,vrow], vpixels);   (* flag so char will be sent *)
   END;
END;
INC(rectcount);
IF rectcount = 400 THEN   (* avoid too much of a pause before flushing bitmap *)
   ShowBitmap;
END;
END VT220ShowRectangle;

(******************************************************************************)

PROCEDURE VT220ResetVDU;

(* We don't do a hardware reset, but try to leave VDU gracefully. *)

BEGIN
Write(ESC); WriteString('[?3l');   (* switch back to 80 column mode *)
END VT220ResetVDU;

(******************************************************************************)

BEGIN
END VT220VDU.
