/*
 * init.c --
 *
 *      This file contains the initialization procedure for the Dvi
 *      Tcl extension, as well as Tcl command definitions so the
 *      features of the Dvi extension are available from within
 *      Tcl. This is needed so the extension can be dynamically
 *      loaded.
 *
 * Copyright  1998 by Anselm Lingnau. All rights reserved.
 * See the LICENSE file for distribution conditions.
 */

#include <string.h>
#include "kpathsea/kpathsea.h"
#include "dviInt.h"

#ifndef lint
static char rcsid[] VAR_UNUSED = "$Id: init.c,v 1.3 1999/06/13 23:22:27 lingnau Exp $";
#endif /* lint */

typedef struct DviFileCookie {
    Dvi_File *dviFilePtr;
    Tcl_Interp *interp;
    Tcl_Obj *reloadCmd;
} DviFileCookie;

static Tcl_HashTable * InitCookieToFileTable _ANSI_ARGS_((Tcl_Interp *interp));

static void DeleteCookieToFileTable _ANSI_ARGS_((ClientData clientData,
						 Tcl_Interp *interp));

static void CookieSetReloadCmd _ANSI_ARGS_((Tcl_Interp *,
					    Tcl_Obj *cookie, Tcl_Obj *cmd));
static Tcl_Obj *CookieGetReloadCmd _ANSI_ARGS_((Tcl_Interp *,
						Tcl_Obj *cookie));
static void InvokeReloadCmds _ANSI_ARGS_((ClientData clientData,
					  Dvi_File *dviFile));

static int DviFileCmd _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
				   int objc, Tcl_Obj * CONST objv[]));
#if DVI_DEBUG
static int DviPageSpecCmd _ANSI_ARGS_((ClientData clientData,
				       Tcl_Interp *interp,
				       int objc, Tcl_Obj * CONST objv[]));
static int DviPageFindCmd _ANSI_ARGS_((ClientData clientData,
				       Tcl_Interp *interp,
				       int objc, Tcl_Obj * CONST objv[]));
static int DviInterpCmd _ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp,
				     int objc, Tcl_Obj * CONST objv[]));
#endif
static int DviGetPixelsCmd _ANSI_ARGS_((ClientData clientData,
					Tcl_Interp *interp,
					int objc, Tcl_Obj * CONST objv[]));
static int DviGetDistanceCmd _ANSI_ARGS_((ClientData clientData,
					  Tcl_Interp *interp,
					  int objc, Tcl_Obj * CONST objv[]));
static int DviFontCmd _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
				   int objc, Tcl_Obj * CONST objv[]));

/*
 * ------------------------------------------------------------------------
 *
 * DviFileCmd --
 *
 *      Implements the `::dvi::file' command.
 *      See the user documentation for details.
 *
 * ------------------------------------------------------------------------
 */

static int
DviFileCmd (clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    static enum {
	DVIF_OPEN, DVIF_CLOSE, DVIF_CHANGED, DVIF_INFO, DVIF_PAGENUMBERS,
	DVIF_RELOAD, DVIF_RELOADCMD,
#if DVI_DEBUG
	DVIF_PURGE,
#endif /* DVI_DEBUG */
    } idx;
    static char *subCmds[] = {
	"open", "close", "changed", "info", "pagenumbers",
	"reload", "reloadcmd",
#if DVI_DEBUG
	"_purge",
#endif /* DVI_DEBUG */
	(char *)0
    };
    Dvi_File *dviFilePtr;

    /*
     * See below for details of the cookie scheme.
     */

    static int cookieCount = 0;	/* for generation of new cookies */
    char cookieBuf[1024];
    char *cookie = cookieBuf;
    int cookieLen;
    int changed;
    Tcl_HashTable *cookieToFileTablePtr = (Tcl_HashTable *)clientData;
    Tcl_HashEntry *cookieEntry;

    Tcl_Obj *pageListPtr;	/* for page number listing */

    /*
     * The usual preliminaries -- check argument count and obtain
     * subcommand index.
     */

    if (objc == 1) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?parameters?");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subCmds, "option",
			    TCL_EXACT, (int *)&idx) != TCL_OK) {
	return TCL_ERROR;
    }

    switch (idx) {

	/*
	 * Open a DVI file mentioned by name.
	 */

    case DVIF_OPEN:
	if (objc != 3 && objc != 4) {
	    Tcl_WrongNumArgs(interp, 2, objv, "?name? filename");
	    return TCL_ERROR;
	}

	dviFilePtr = Dvi_OpenFile(interp,
				  Tcl_GetStringFromObj(objv[objc - 1],
						       (int *)0),
				  InvokeReloadCmds, (ClientData)0);
	if (dviFilePtr == (Dvi_File *)0) {
	    return TCL_ERROR;
	}

	/*
	 * Figure out the cookie to use for this file. The DVI file
	 * routines load a file just once for the whole application;
	 * the cookies are a per-interpreter affair and thus the same
	 * DVI file may be accessible from multiple interpreters using
	 * different cookies. However, we try to re-use cookies even
	 * in the same interpreter, thus if the user didn't specify a
	 * cookie and the file is currently in use in this interpreter,
	 * we give him the existing cookie for that file -- otherwise
	 * we make one up. If the user asks for a particular cookie,
	 * he gets that cookie regardless of whether the file already
	 * has another one.
	 */

	if (objc == 3) {
	    int new = 0;
	    while (!new) {
		sprintf(cookieBuf, "dvif%d", cookieCount++);
		cookieLen = strlen(cookieBuf);
		cookieEntry = Tcl_CreateHashEntry(cookieToFileTablePtr,
						  cookieBuf, &new);
		if (new) {
		    DviFileCookie *cookiePtr
			= ckalloc(sizeof(DviFileCookie));
		    cookiePtr->dviFilePtr = dviFilePtr;
		    cookiePtr->interp = interp;
		    cookiePtr->reloadCmd = (Tcl_Obj *)0;
		    /* cookiePtr->refCount = 1; */
		    dviFilePtr->clientData = (ClientData)cookiePtr;
		    Tcl_SetHashValue(cookieEntry, cookiePtr);
		}
	    }
	} else {
	    /*
	     * The user wants a specific cookie.
	     * See whether the user's cookie is already in use. If so,
	     * bail out.
	     */
	    int new;
	    DviFileCookie *cookiePtr;
	    
	    cookie = Tcl_GetStringFromObj(objv[2], &cookieLen);
	    cookieEntry = Tcl_CreateHashEntry(cookieToFileTablePtr,
					      cookie, &new);
	    if (!new) {
		/*
		 * TODO: check whether cookie in question refers to
		 * another instance of the file we just loaded. If so,
		 * then all is well!
		 */

		Dvi_CloseFile(dviFilePtr);
		Tcl_AppendStringsToObj(resultPtr, "DVI file cookie \"",
				       cookie, "\" is already in use",
				       (char *)0);
		return TCL_ERROR;
	    }
	    cookiePtr = ckalloc(sizeof(DviFileCookie));
	    cookiePtr->dviFilePtr = dviFilePtr;
	    cookiePtr->interp = interp;
	    cookiePtr->reloadCmd = (Tcl_Obj *)0;
	    /*    cookiePtr->refCount = 1;*/
	    dviFilePtr->clientData = (ClientData)cookiePtr; /* Pfui! */
	    Tcl_SetHashValue(cookieEntry, cookiePtr);
	}
	Tcl_SetStringObj(resultPtr, cookie, cookieLen);
	break;

	/*
	 * `Close' the DVI file identified by the given cookie. The file
	 * isn't really closed until the last reference to it is
	 * removed.
	 */

    case DVIF_CLOSE:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "cookie");
	    return TCL_ERROR;
	}
	
	cookie = Tcl_GetStringFromObj(objv[2], (int *)0);
	cookieEntry = Tcl_FindHashEntry(cookieToFileTablePtr, cookie);
	if (cookieEntry == (Tcl_HashEntry *)0) {
	    Tcl_SetResult(interp, "no such cookie", TCL_STATIC);
	    return TCL_ERROR;
	}

	dviFilePtr = Dvi_GetFileByCookie(interp, cookie, TCL_LEAVE_ERR_MSG);
	if (dviFilePtr == (Dvi_File *)0) {
	    return TCL_ERROR;
	}

	Dvi_CloseFile(dviFilePtr);

	ckfree((char *)Tcl_GetHashValue(cookieEntry));
	Tcl_DeleteHashEntry(cookieEntry);
	break;

	/*
	 * Return true if file has changed since it was opened, else false.
	 */

    case DVIF_CHANGED:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "cookie");
	    return TCL_ERROR;
	}
	
	cookie = Tcl_GetStringFromObj(objv[2], (int *)0);
	cookieEntry = Tcl_FindHashEntry(cookieToFileTablePtr, cookie);
	if (cookieEntry == (Tcl_HashEntry *)0) {
	    Tcl_SetResult(interp, "no such cookie", TCL_STATIC);
	    return TCL_ERROR;
	}

	dviFilePtr = Dvi_GetFileByCookie(interp, cookie, TCL_LEAVE_ERR_MSG);
	if (dviFilePtr == (Dvi_File *)0) {
	    return TCL_ERROR;
	}

	if ((changed = Dvi_FileChanged(dviFilePtr)) < 0) {
	    Tcl_SetResult(interp, Tcl_PosixError(interp), TCL_VOLATILE);
	    return TCL_ERROR;
	}

	Tcl_SetObjResult(interp, Tcl_NewBooleanObj(changed));
	break;

	/*
	 * Display information about all DVI files or the DVI file
	 * identified by the given cookie.
	 */

    case DVIF_INFO:
	if (objc > 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "?cookie?");
	    return TCL_ERROR;
	}
	if (objc == 2) {
	    Tcl_HashSearch searchBuf;
	    Dvi_FileInfo *dviInfoPtr;

	    cookieEntry = Tcl_FirstHashEntry(cookieToFileTablePtr, &searchBuf);
	    while (cookieEntry != (Tcl_HashEntry *)0) {
		char *key = Tcl_GetHashKey(cookieToFileTablePtr, cookieEntry);
		DviFileCookie *dviFileCookiePtr
		    = Tcl_GetHashValue(cookieEntry);

		Tcl_Obj *infoPtr;
		Tcl_Obj *infoObjs[3];
		infoObjs[0] = Tcl_NewStringObj(key, strlen(key));
		dviInfoPtr = dviFileCookiePtr->dviFilePtr->infoPtr;
		infoObjs[1] = Tcl_NewStringObj(dviInfoPtr->name,
					       strlen(dviInfoPtr->name));
		infoObjs[2] = Tcl_NewIntObj(dviInfoPtr->refCount);
		infoPtr = Tcl_NewListObj(3, infoObjs);
		if (Tcl_ListObjAppendElement(interp, resultPtr, infoPtr)
		    != TCL_OK) {
		    return TCL_ERROR;
		}
		cookieEntry = Tcl_NextHashEntry(&searchBuf);
	    }
	} else {
	    char buf[20];
	    Dvi_FileInfo *dviInfoPtr;

	    cookie = Tcl_GetStringFromObj(objv[2], (int *)0);
	    dviFilePtr = Dvi_GetFileByCookie(interp, cookie,
					     TCL_LEAVE_ERR_MSG);
	    if (dviFilePtr == (Dvi_File *)0) {
		return TCL_ERROR;
	    }

	    dviInfoPtr = dviFilePtr->infoPtr;
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(dviInfoPtr->name,
					      strlen(dviInfoPtr->name)));
	    sprintf(buf, "%d", dviInfoPtr->refCount);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%u", dviInfoPtr->pageCount);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%lu", dviInfoPtr->fileSize);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
#if DVI_DEBUG
	    sprintf(buf, "%lu", (unsigned long)dviInfoPtr->num);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%lu", (unsigned long)dviInfoPtr->den);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%lu", (unsigned long)dviInfoPtr->mag);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%u", dviInfoPtr->stackSize);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%d", dviInfoPtr->fileDesc);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
	    sprintf(buf, "%p", dviInfoPtr->code);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewStringObj(buf, strlen(buf)));
#endif /* DVI_DEBUG */
	}
	break;

	/*
	 * Generate a list with all the page numbers that occur in the
	 * DVI file identified by the given cookie.
	 */

    case DVIF_PAGENUMBERS:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "cookie");
	    return TCL_ERROR;
	}
	
	cookie = Tcl_GetStringFromObj(objv[2], (int *)0);
	dviFilePtr = Dvi_GetFileByCookie(interp, cookie, TCL_LEAVE_ERR_MSG);
	if (dviFilePtr == (Dvi_File *)0) {
	    return TCL_ERROR;
	}

	pageListPtr = Dvi_ListPageNumbers(dviFilePtr);
	if (pageListPtr == (Tcl_Obj *)0) {
	    Tcl_SetResult(interp, "no page table", TCL_STATIC);
	    return TCL_ERROR;
	}
	Tcl_SetObjResult(interp, pageListPtr);
	break;

    case DVIF_RELOADCMD:
	if (objc != 3 && objc != 4) {
	    Tcl_WrongNumArgs(interp, 2, objv, "cookie ?proc?");
	    return TCL_ERROR;
	}

	if (objc == 3) {
	    Tcl_Obj *cmd = CookieGetReloadCmd(interp, objv[2]);

	    if (cmd != (Tcl_Obj *)0) {
		Tcl_SetObjResult(interp, cmd);
	    }
	} else {
	    int length;
	    (void)Tcl_GetStringFromObj(objv[3], &length);
	    if (length == 0) {
		CookieSetReloadCmd(interp, objv[2], (Tcl_Obj *)0);
	    } else {
		CookieSetReloadCmd(interp, objv[2], objv[3]);
	    }
	    Tcl_SetObjResult(interp, objv[3]);
	}
	break;

    case DVIF_RELOAD:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "cookie");
	    return TCL_ERROR;
	}
	
	cookie = Tcl_GetStringFromObj(objv[2], (int *)0);
	dviFilePtr = Dvi_GetFileByCookie(interp, cookie, TCL_LEAVE_ERR_MSG);
	if (dviFilePtr == (Dvi_File *)0) {
	    return TCL_ERROR;
	}

	Dvi_ReloadFile(interp, dviFilePtr);
	break;

#if DVI_DEBUG
    case DVIF_PURGE:
	if (objc != 2) {
	    Tcl_WrongNumArgs(interp, 1, objv, "");
	    return TCL_ERROR;
	}

	Dvi_PurgeFiles();
	Tcl_DeleteHashTable(cookieToFileTablePtr);
	Tcl_InitHashTable(cookieToFileTablePtr, TCL_STRING_KEYS);
	cookieCount = 0;
	break;
#endif /* DVI_DEBUG */
    }

    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *
 * InitCookieToFileTable --
 *
 *      Initializes a table mapping DVI file `cookies' to Dvi_File
 *      pointers.
 *
 * Results:
 *      A pointer to the newly-created hash table or 0.
 *
 * Side Effects:
 *      The hash table is created and initialized. A pointer to the
 *      table is installed in the current interpreter.
 *
 * ------------------------------------------------------------------------
 */

static Tcl_HashTable *
InitCookieToFileTable (interp)
    Tcl_Interp *interp;
{
    Tcl_HashTable *cookieToFileTablePtr
	= (Tcl_HashTable *)ckalloc(sizeof(Tcl_HashTable));
    if (cookieToFileTablePtr == (Tcl_HashTable *)0) {
	Tcl_SetResult(interp, "not enough memory", TCL_STATIC);
	return (Tcl_HashTable *)0;
    }
    Tcl_InitHashTable(cookieToFileTablePtr, TCL_STRING_KEYS);
    Tcl_SetAssocData(interp, DVI_NAME, DeleteCookieToFileTable,
		     (ClientData)cookieToFileTablePtr);
    return cookieToFileTablePtr;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_GetFileByCookie --
 *
 *      Returns the Dvi_File pointer of a file given a DVI file cookie.
 *
 * Results:
 *      A pointer to the Dvi_File structure of the file identified by
 *      `cookie' or a null pointer if the cookie doesn't currently
 *      exist. In the latter case, if `flags' is TCL_LEAVE_ERR_MSG,
 *      then an * error message appears as the Tcl result.
 *
 * Side Effects:
 *      None.
 *
 * ------------------------------------------------------------------------
 */

Dvi_File *
Dvi_GetFileByCookie (interp, cookie, flags)
    Tcl_Interp *interp;		/* Current Tcl interpreter */
    const char *cookie;		/* DVI file cookie in question */
    const int flags;		/* 0, or TCL_LEAVE_ERR_MSG */
{
    Tcl_HashTable *cookieToFileTablePtr;
    Tcl_HashEntry *cookieEntry;

    cookieToFileTablePtr =
	(Tcl_HashTable *)Tcl_GetAssocData(interp, DVI_NAME,
					  (Tcl_InterpDeleteProc **)0);
    cookieEntry = Tcl_FindHashEntry(cookieToFileTablePtr, cookie);
    if (cookieEntry == (Tcl_HashEntry *)0) {
	if (flags & TCL_LEAVE_ERR_MSG) {
	    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
	    Tcl_AppendStringsToObj(resultPtr, "DVI cookie \"", cookie,
				   "\" does not exist", (char *)0);
	}
	return (Dvi_File *)0;
    }
    return ((DviFileCookie *)Tcl_GetHashValue(cookieEntry))->dviFilePtr;
}

/*
 * ------------------------------------------------------------------------
 *
 * DeleteCookieToFileTable --
 *
 *      Removes the cookie-to-Dvi_File-pointer translation table.
 *
 * This procedure is called when the Tcl interpreter is deleted. It
 * removes the hash table used to map cookies to Dvi_File pointers.
 *
 * ------------------------------------------------------------------------
 */

static void
DeleteCookieToFileTable (clientData, interp)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
{
    Tcl_HashTable *cookieToFileTablePtr;

    cookieToFileTablePtr =
	(Tcl_HashTable *)Tcl_GetAssocData(interp, DVI_NAME,
					  (Tcl_InterpDeleteProc **)0);
    if (cookieToFileTablePtr != 0) {
	Tcl_DeleteHashTable(cookieToFileTablePtr);
    }
    Tcl_DeleteAssocData(interp, "Dvi");
}

static void
CookieSetReloadCmd (interp, cookie, reloadCmd)
    Tcl_Interp *interp;
    Tcl_Obj *cookie;
    Tcl_Obj *reloadCmd;
{
    Tcl_HashTable *cookieToFileTablePtr;
    Tcl_HashEntry *cookieEntry;
    DviFileCookie *cookiePtr;

    cookieToFileTablePtr =
	(Tcl_HashTable *)Tcl_GetAssocData(interp, DVI_NAME,
					  (Tcl_InterpDeleteProc **)0);
    cookieEntry = Tcl_FindHashEntry(cookieToFileTablePtr,
				    Tcl_GetStringFromObj(cookie, (int *)0));
    if (cookieEntry == (Tcl_HashEntry *)0) {
	Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
	Tcl_AppendStringsToObj(resultPtr, "DVI cookie \"", cookie,
			       "\" does not exist", (char *)0);
	return;
    }

    cookiePtr = ((DviFileCookie *)Tcl_GetHashValue(cookieEntry));

    if (cookiePtr->reloadCmd != (Tcl_Obj *)0) {
	Tcl_DecrRefCount(cookiePtr->reloadCmd);
    }
    if (reloadCmd != (Tcl_Obj *)0) {
	cookiePtr->reloadCmd = reloadCmd;
	Tcl_IncrRefCount(reloadCmd);
    } else {
	cookiePtr->reloadCmd = (Tcl_Obj *)0;
    }
}

static Tcl_Obj *
CookieGetReloadCmd (interp, cookie)
    Tcl_Interp *interp;
    Tcl_Obj *cookie;
{
    Tcl_HashTable *cookieToFileTablePtr;
    Tcl_HashEntry *cookieEntry;

    cookieToFileTablePtr =
	(Tcl_HashTable *)Tcl_GetAssocData(interp, DVI_NAME,
					  (Tcl_InterpDeleteProc **)0);
    cookieEntry = Tcl_FindHashEntry(cookieToFileTablePtr,
				    Tcl_GetStringFromObj(cookie, (int *)0));
    if (cookieEntry == (Tcl_HashEntry *)0) {
	Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
	Tcl_AppendStringsToObj(resultPtr, "DVI cookie \"", cookie,
			       "\" does not exist", (char *)0);
	return (Tcl_Obj *)0;
    }
    return ((DviFileCookie *)Tcl_GetHashValue(cookieEntry))->reloadCmd;
}

static void
InvokeReloadCmds (clientData, dviFile)
    ClientData clientData;
    Dvi_File *dviFile;
{
    DviFileCookie *cookiePtr = (DviFileCookie *)clientData;

    if (cookiePtr->reloadCmd) {
	Tcl_GlobalEvalObj(cookiePtr->interp, cookiePtr->reloadCmd);
    }
}

/*
 * ------------------------------------------------------------------------
 *
 * DviPageSpecCmd --
 *
 *     This command exposes the Dvi_GetPageSpec() function for use in
 *     the test suite.
 *
 * ------------------------------------------------------------------------
 */

#if DVI_DEBUG
static int
DviPageSpecCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    Dvi_PageSpec pageSpec;
    char careVec[10];
    Tcl_Obj *tmpObj;
    int i;

    if (objc != 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "pagespec");
	return TCL_ERROR;
    }

    if (Dvi_GetPageSpec(interp, Tcl_GetStringFromObj(objv[1], (int *)0),
			&pageSpec) != TCL_OK) {
	return TCL_ERROR;
    }

    if (pageSpec.countersUsed == DVI_PS_ABSOLUTE) {
	tmpObj = Tcl_NewStringObj("Abs", -1);
	if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	    return TCL_ERROR;
	}
	tmpObj = Tcl_NewIntObj(pageSpec.number[0]);
	if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	tmpObj = Tcl_NewIntObj(pageSpec.countersUsed);
	if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	    return TCL_ERROR;
	}

	sprintf(careVec, "%0x", pageSpec.careVector);
	tmpObj = Tcl_NewStringObj(careVec, -1);
	if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	    return TCL_ERROR;
	}

	tmpObj = Tcl_NewIntObj(pageSpec.occurrences);
	if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	    return TCL_ERROR;
	}

	for (i = 0; i < pageSpec.countersUsed; i++) {
	    tmpObj = Tcl_NewIntObj(pageSpec.number[i]);
	    if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj)!=TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }
    return TCL_OK;
}
#endif /* DVI_DEBUG */

/*
 * ------------------------------------------------------------------------
 *
 * DviPageFindCmd --
 *
 *     This command exposes the Dvi_FindPage() function for use in
 *     the test suite.
 *
 * ------------------------------------------------------------------------
 */

#if DVI_DEBUG
static int
DviPageFindCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    Dvi_File *dviFilePtr;
    Dvi_PageSpec pageSpec;
    unsigned int currPageNo;
    char pointerStr[10];
    U8 *pagePtr;
    Tcl_Obj *tmpObj;

    if (objc != 4) {
	Tcl_WrongNumArgs(interp, 1, objv, "file pagespec currpage");
	return TCL_ERROR;
    }

    dviFilePtr = Dvi_GetFileByCookie(interp,
				     Tcl_GetStringFromObj(objv[1], (int *)0),
				     TCL_LEAVE_ERR_MSG);
    if (dviFilePtr == (Dvi_File *)0) {
	return TCL_ERROR;
    }

    if (Dvi_GetPageSpec(interp, Tcl_GetStringFromObj(objv[2], (int *)0),
			&pageSpec) != TCL_OK) {
	return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, objv[3], (int *)&currPageNo) != TCL_OK) {
	return TCL_ERROR;
    }

    pagePtr = Dvi_FindPage(dviFilePtr, &pageSpec, &currPageNo);

    sprintf(pointerStr, "%lu", pagePtr == 0
	    ? 0
	    : (unsigned long)(pagePtr - dviFilePtr->infoPtr->code));
    tmpObj = Tcl_NewStringObj(pointerStr, -1);
    if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	return TCL_ERROR;
    }

    tmpObj = Tcl_NewIntObj(currPageNo);
    if (Tcl_ListObjAppendElement(interp, resultPtr, tmpObj) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}
#endif /* DVI_DEBUG */

/*
 * ------------------------------------------------------------------------
 *
 * DviGetPixelsCmd --
 *
 *      Implements the `::dvi::pixels' command. See the user documentation
 *      for details.
 *
 * ------------------------------------------------------------------------
 */

static int
DviGetPixelsCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    int resolution;		/* Resolution for pixel conversion */
    int result;			/* Resulting distance */

    if (objc != 3) {
	Tcl_WrongNumArgs(interp, 1, objv, "resolution distance");
	return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, objv[1], &resolution) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Dvi_GetPixels(interp, resolution,
		      Tcl_GetStringFromObj(objv[2], (int *)0),
		      &result, TCL_LEAVE_ERR_MSG) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetObjResult(interp, Tcl_NewIntObj(result));
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *
 * DviGetDistanceCmd --
 *
 *      Implements the `::dvi::distance' command. See the user
 *      documentation for details.
 *
 * ------------------------------------------------------------------------
 */

static int
DviGetDistanceCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    int resolution;		/* Resolution for pixel conversion */
    double pixels;		/* Pixel quantity to be converted */
    double result;		/* Result buffer */

    if (objc != 4) {
	Tcl_WrongNumArgs(interp, 1, objv, "resolution pixels unit");
	return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, objv[1], &resolution) != TCL_OK
	|| Tcl_GetDoubleFromObj(interp, objv[2], &pixels) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Dvi_GetDistance(interp, resolution, pixels,
			Tcl_GetStringFromObj(objv[3], (int *)0),
			&result, TCL_LEAVE_ERR_MSG) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetObjResult(interp, Tcl_NewDoubleObj(result));
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *
 * DviFontCmd --
 *
 *      Implements the `::dvi::font' command. See the user documentation
 *      for details.
 *
 * ------------------------------------------------------------------------
 */

static int
DviFontCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    static enum {
	DVIFN_FINDFILE, DVIFN_LIST, DVIFN_PURGE,
#if DVI_DEBUG
	DVIFN_INTERP, DVIFN_LOAD, DVIFN_INFO, DVIFN_FREE,
#endif /* DVI_DEBUG */
    } idx;
    static char *subCmds[] = {
	"findfile", "list", "purge",
#if DVI_DEBUG
	"_interp", "_load", "_info", "_free",
#endif /* DVI_DEBUG */
	(char *)0
    };

    unsigned int res;
    Dvi_FontType type;
    char *fontName;
    int fontNameLength;
    char *fileName;
    char buf[20];
#if DVI_DEBUG
    U32 check;
    U32 fontScale;
    U32 designSize;
    U32 resolution;
    static Dvi_Interp *dviInterp = 0;
    static Tcl_HashTable *fontNamesPtr = 0;
    Dvi_Font *fontPtr;
#endif /* DVI_DEBUG */
    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);
    
    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?parameters?");
        return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subCmds, "option",
			    TCL_EXACT, (int *)&idx) != TCL_OK) {
	return TCL_ERROR;
    }

    switch (idx) {

	/*
	 * Find the file name, actual resolution and font type used
	 * for the given font at the given resolution.
	 */

    case DVIFN_FINDFILE:
	if (Tcl_GetIntFromObj(interp, objv[3], (int *)&res) != TCL_OK) {
            return TCL_ERROR;
        }
	fontName = Tcl_GetStringFromObj(objv[2], &fontNameLength);
        fileName = Dvi_FindFontFile((unsigned int)fontNameLength, fontName,
                                &res, &type);
        if (fileName == (char *)0) {
            Tcl_AppendResult(interp, "Font \"", fontName, "\" at ",
			     Tcl_GetStringFromObj(objv[3], (int *)0),
			     " dpi not found", (char *)0);
            return TCL_ERROR;
        }
        Tcl_ListObjAppendElement(interp, resultPtr,
				 Tcl_NewStringObj(fileName, -1));
        sprintf(buf, "%u", res);
	Tcl_ListObjAppendElement(interp, resultPtr,
				 Tcl_NewStringObj(buf, -1));
        sprintf(buf, "%d", type);
	Tcl_ListObjAppendElement(interp, resultPtr,
				 Tcl_NewStringObj(buf, -1));
        ckfree(fileName);
	break;

#if DVI_DEBUG
	/*
	 * List all fonts currently in use.
	 */

    case DVIFN_LIST:
	Tcl_SetObjResult(interp, Dvi_FontDumpAll(interp));
	break;

	/*
	 * Initialize a DVI interpreter, which is necessary for
	 * explicit font loading.
	 */

    case DVIFN_INTERP:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "resolution");
	    return TCL_ERROR;
	}

	if (Tcl_GetIntFromObj(interp, objv[2], (int *)&resolution) != TCL_OK) {
	    return TCL_ERROR;
	}

	if (dviInterp != 0) {
	    ckfree((char *)dviInterp);
	}
	dviInterp = Dvi_CreateInterp(interp, resolution, resolution,
				     32, 25400000, 473628672, 1000);
	break;

	/*
	 * Load a named font.
	 */

    case DVIFN_LOAD:
	if (objc != 6) {
	    Tcl_WrongNumArgs(interp, 2, objv, "name check scale designSize");
	    return TCL_ERROR;
	}

	if (dviInterp == 0) {
	    Tcl_SetResult(interp, "must initialize DVI interpreter first",
			  TCL_STATIC);
	    return TCL_ERROR;
	}

	if (fontNamesPtr == 0) {
	    fontNamesPtr = ckalloc(sizeof(Tcl_HashTable));
	    Tcl_InitHashTable(fontNamesPtr, TCL_STRING_KEYS);
	}

	fontName = Tcl_GetStringFromObj(objv[2], &fontNameLength);
	if (Tcl_GetIntFromObj(interp, objv[3], (int *)&check) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_GetIntFromObj(interp, objv[4], (int *)&fontScale) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_GetIntFromObj(interp, objv[5], (int *)&designSize) != TCL_OK) {
	    return TCL_ERROR;
	}
	
	fontPtr = Dvi_FontFind(dviInterp, check, fontScale, designSize,
			       fontNameLength, fontName);
	if (fontPtr == (Dvi_Font *)0) {
	    return TCL_ERROR;
	} else {
	    char key[20];
	    Tcl_HashEntry *entryPtr;
	    int new;
	    sprintf(key, "fn%lx", (unsigned long)fontPtr);
	    entryPtr = Tcl_CreateHashEntry(fontNamesPtr, key, &new);
	    if (new) {
		Tcl_SetHashValue(entryPtr, (ClientData)fontPtr);
	    }
	    Tcl_SetResult(interp, key, TCL_VOLATILE);
	}
	break;

	/*
	 * Free a font, or display info about a font.
	 */

    case DVIFN_INFO: case DVIFN_FREE:
	if (fontNamesPtr == 0) {
	    fontNamesPtr = ckalloc(sizeof(Tcl_HashTable));
	    Tcl_InitHashTable(fontNamesPtr, TCL_STRING_KEYS);
	}

	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "fontId");
	    return TCL_ERROR;
	} else {
	    char *fontId = Tcl_GetStringFromObj(objv[2], (int *)0);
	    Tcl_HashEntry *entryPtr = Tcl_FindHashEntry(fontNamesPtr, fontId);
	    if (entryPtr == 0) {
		Tcl_SetResult(interp, "font ID does not exist", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (idx == DVIFN_FREE) {
		Dvi_FontFree((Dvi_Font *)Tcl_GetHashValue(entryPtr));
	    } else {
		Tcl_SetObjResult(interp, Dvi_FontDump(interp,
				 (Dvi_Font *)Tcl_GetHashValue(entryPtr)));
	    }
	}
	break;
#endif /* DVI_DEBUG */

	/*
	 * Remove all fonts that are currently unused.
	 */

    case DVIFN_PURGE:
        Dvi_FontPurge();
	break;
    }
    return TCL_OK;
}

static int
DviInterpCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
    static enum {
	DVII_FONTS, DVII_GETFONTS, DVII_INIT, DVII_PARAMS,
	DVII_RENDER, DVII_RESET,
    } idx;
    static char *subCmds[] = {
	"fonts", "getfonts", "init", "parameters",
	"render", "reset",
	(char *)0
    };

    U32 resolution;
    char *cookie;
    Dvi_File *dviFile;
    static Dvi_Interp *dviInterp = 0;
    char buf[40];
    int full;
    Dvi_FontList *listPtr;
    char *pageSpecStr;
    Dvi_PageSpec pageSpec;
    unsigned int currPage = 1;
    U8 *code;
    Tcl_DString dviCode;

    Tcl_Obj *resultPtr = Tcl_GetObjResult(interp);

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?parameters?");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subCmds, "option",
			    TCL_EXACT, (int *)&idx) != TCL_OK) {
	return TCL_ERROR;
    }

    switch (idx) {

    case DVII_INIT:
	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 2, objv, "resolution dvi_file_cookie");
	    return TCL_ERROR;
	}

	if (Tcl_GetIntFromObj(interp, objv[2], (int *)&resolution) != TCL_OK) {
	    return TCL_ERROR;
	}

	cookie = Tcl_GetStringFromObj(objv[3], (int *)0);
	dviFile = Dvi_GetFileByCookie(interp, cookie, TCL_LEAVE_ERR_MSG);
	if (dviFile == (Dvi_File *)0) {
	    return TCL_ERROR;
	}

	if (dviInterp != 0) {
	    ckfree((char *)dviInterp);
	}
	dviInterp = Dvi_CreateInterpForFile(interp, resolution, resolution,
					    dviFile);
	break;

    case DVII_PARAMS:
	if (objc != 2) {
	    Tcl_WrongNumArgs(interp, 2, objv, "");
	    return TCL_ERROR;
	}

	if (dviInterp == 0) {
	    Tcl_SetResult(interp, "DVI interpreter uninitialized", TCL_STATIC);
	    return TCL_ERROR;
	}

#define APPEND(buf) Tcl_ListObjAppendElement(interp, resultPtr, \
					     Tcl_NewStringObj((buf), -1));
	sprintf(buf, "%lu", (unsigned long)dviInterp->xResolution);
	APPEND(buf);
	sprintf(buf, "%lu", (unsigned long)dviInterp->yResolution);
	APPEND(buf);
	sprintf(buf, "%.17e", (double)dviInterp->tfmConv); APPEND(buf);
	sprintf(buf, "%.17e", (double)dviInterp->xConv); APPEND(buf);
	sprintf(buf, "%.17e", (double)dviInterp->yConv); APPEND(buf);
	sprintf(buf, "%.17e", (double)dviInterp->trueXConv); APPEND(buf);
	sprintf(buf, "%.17e", (double)dviInterp->trueYConv); APPEND(buf);
	sprintf(buf, "%lu", (unsigned long)dviInterp->maxDrift); APPEND(buf);
	sprintf(buf, "%u", dviInterp->stackSize); APPEND(buf);
	break;

    case DVII_GETFONTS:
	if (objc != 2) {
	    Tcl_WrongNumArgs(interp, 2, objv, "");
	    return TCL_ERROR;
	}

	if (dviInterp == 0) {
	    Tcl_SetResult(interp, "DVI interpreter uninitialized", TCL_STATIC);
	    return TCL_ERROR;
	}

	if (Dvi_FontsFromPostamble(dviInterp, dviInterp->dviFile) != TCL_OK) {
	    return TCL_ERROR;
	}
	break;

    case DVII_FONTS:
	if (objc != 2 && objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "?-full?");
	    return TCL_ERROR;
	}

	full = 0;
	if (objc == 3) {
	    char *flag = Tcl_GetStringFromObj(objv[2], (int *)0);
	    if (strcmp(flag, "-full") != 0) {
		Tcl_SetResult(interp, "invalid option", TCL_STATIC);
		return TCL_ERROR;
	    } else {
		full = 1;
	    }
	}

	if (dviInterp == 0) {
	    Tcl_SetResult(interp, "DVI interpreter uninitialized", TCL_STATIC);
	    return TCL_ERROR;
	}

	for (listPtr = dviInterp->fonts; listPtr; listPtr = listPtr->nextPtr) {
	    Tcl_Obj *item[2] = { 0, 0 };
	    item[0] = Tcl_NewLongObj((long)listPtr->fontNum);
	    item[1] = full
		? Dvi_FontDump(interp, listPtr->fontPtr)
		    : Tcl_NewStringObj(listPtr->fontPtr->fontName, -1);
	    Tcl_ListObjAppendElement(interp, resultPtr,
				     Tcl_NewListObj(2, item));
	}
	break;

    case DVII_RENDER:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "pagespec");
	    return TCL_ERROR;
	}

	if (dviInterp == 0) {
	    Tcl_SetResult(interp, "DVI interpreter uninitialized", TCL_STATIC);
	    return TCL_ERROR;
	}

	pageSpecStr = Tcl_GetStringFromObj(objv[2], (int *)0);
	if (Dvi_GetPageSpec(interp, pageSpecStr, &pageSpec) != TCL_OK) {
	    return TCL_ERROR;
	}

	if ((code = Dvi_FindPage(dviInterp->dviFile, &pageSpec, &currPage))
	    == 0) {
	    Tcl_AppendResult(interp, "Couldn't find page \"", pageSpecStr,
			     "\"", (char *)0);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&dviCode);
	dviInterp->procData = (ClientData)&dviCode;
	Dvi_ResetInterp(dviInterp, 0);
	Dvi_Interpret(dviInterp, code + 44);
	Tcl_DStringResult(interp, &dviCode);
	break;

    case DVII_RESET:
	if (objc != 2) {
	    Tcl_WrongNumArgs(interp, 2, objv, "");
	    return TCL_ERROR;
	}

	if (dviInterp != 0) {
	    Dvi_ResetInterp(dviInterp, 1);
	}
	break;
    }
    return TCL_OK;
}   

static int
DviAssocCmd (clientData, interp, objc, objv)
    ClientData clientData __attribute__((unused));
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj * CONST objv[];
{
  char buf[20];
  sprintf(buf, "%p",
	  (void *)Tcl_GetAssocData(interp, "Dvi", (Tcl_InterpDeleteProc **)0));
  Tcl_SetResult(interp, buf, TCL_VOLATILE);
  return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_Init --
 *
 *      Main installation entry point for Tcl extension.
 *
 * Results:
 *      A standard Tcl result, depending on whether the installation
 *      succeeded or failed. If it failed, an error message may be
 *      left as the result string.
 *
 * Side effects:
 *      The commands and data structures for the Dvi extension are
 *      made available. If the Tk option is compiled in, a Tk image
 *      type for rendering DVI files is defined.
 *
 * ------------------------------------------------------------------------
 */

int
Dvi_Init (interp)
    Tcl_Interp *interp;
{
    Tcl_HashTable *cookieToFileTablePtr;
    Tcl_Obj *resultPtr;
    Tcl_Obj *objPtr, *dviObjPtr, *namePtr;
    char *progName;
    char *envSuffix;
    int makePK;
    char *mfMode;
    char *defaultFont;
    int resolution;

    Dvi_CreateFontType_PK();
    Dvi_CreateFontType_VF();
    Dvi_CreateFontType_TFM();

    if (Tcl_PkgProvide(interp, DVI_NAME, DVI_VERSION) != TCL_OK) {
        return TCL_ERROR;
    }

    /*
     * Try to determine the program name for the benefit of Kpathsea.
     */

    if (Tcl_EvalObj(interp, Tcl_NewStringObj("info script", -1)) != TCL_OK) {
	return TCL_ERROR;
    }
    resultPtr = Tcl_GetObjResult(interp);
    progName = Tcl_GetStringFromObj(resultPtr, (int *)0);
    if (strlen(progName) == 0) {
	if (Tcl_EvalObj(interp, Tcl_NewStringObj("info nameofexecutable", -1))
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	resultPtr = Tcl_GetObjResult(interp);
	progName = Tcl_GetStringFromObj(resultPtr, (int *)0);
    }

    /*
     * The Kpathsea library lets us specify names of the form XYZ.foo,
     * where `foo' is a suffix specific to a certain program, to give
     * various alternatives for the variable XYZ. TkDVI uses the value
     * of the Tcl variable dvi(progname), or (by default) "tkdvi".
     */

    dviObjPtr = Tcl_NewStringObj("dvi", 3);
    namePtr = Tcl_NewStringObj("progname", 8);
    objPtr = Tcl_ObjGetVar2(interp, dviObjPtr, namePtr, TCL_GLOBAL_ONLY);
    if (objPtr == (Tcl_Obj *)0) {
	envSuffix = "tkdvi";
    } else {
	envSuffix = Tcl_GetStringFromObj(objPtr, (int *)0);
    }

    kpse_set_program_name(progName, envSuffix);

    /*
     * Try to get the (horizontal) resolution from dvi(xresolution),
     * the desired Metafont mode from dvi(mfmode), and the font to be
     * used in emergencies from dvi(defaultfont), for the purposes of
     * locating and generating fonts.
     */

    namePtr = Tcl_NewStringObj("xresolution", 11);
    objPtr = Tcl_ObjGetVar2(interp, dviObjPtr, namePtr, TCL_GLOBAL_ONLY);
    if (objPtr == (Tcl_Obj *)0) {
	resolution = 600;
    } else {
	if (Tcl_GetIntFromObj(interp, objPtr, &resolution) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    Tcl_SetStringObj(namePtr, "makepk", 6);
    objPtr = Tcl_ObjGetVar2(interp, dviObjPtr, namePtr, TCL_GLOBAL_ONLY);
    if (objPtr == (Tcl_Obj *)0) {
	makePK = 1;
    } else {
	if (Tcl_GetBooleanFromObj(interp, objPtr, &makePK) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    Tcl_SetStringObj(namePtr, "mfmode", 6);
    objPtr = Tcl_ObjGetVar2(interp, dviObjPtr, namePtr, TCL_GLOBAL_ONLY);
    if (objPtr == (Tcl_Obj *)0) {
	mfMode = "ljfour";
    } else {
	mfMode = Tcl_GetStringFromObj(objPtr, (int *)0);
    }

    Tcl_SetStringObj(namePtr, "defaultfont", 11);
    objPtr = Tcl_ObjGetVar2(interp, dviObjPtr, namePtr, TCL_GLOBAL_ONLY);
    if (objPtr == (Tcl_Obj *)0) {
	defaultFont = "cmr10";
    } else {
	defaultFont = Tcl_GetStringFromObj(objPtr, (int *)0);
    }

    kpse_init_prog(envSuffix, resolution, mfMode, defaultFont);
    kpse_set_program_enabled(kpse_pk_format, makePK, kpse_src_cmdline);


    /*
     * Finish the initialization by creating a number of Tcl commands
     * and internal hash tables for mapping file and font indentifiers
     * to internal data structures.
     */

    Tcl_ResetResult(interp);

    Tcl_CreateObjCommand(interp, "::dvi::pixels", DviGetPixelsCmd,
		      (ClientData)0, (Tcl_CmdDeleteProc *)0);
    Tcl_CreateObjCommand(interp, "::dvi::distance", DviGetDistanceCmd,
		      (ClientData)0, (Tcl_CmdDeleteProc *)0);

#if DVI_DEBUG
    Tcl_CreateObjCommand(interp, "::dvi::pagespec", DviPageSpecCmd,
		      (ClientData)0, (Tcl_CmdDeleteProc *)0);

    Tcl_CreateObjCommand(interp, "::dvi::pagefind", DviPageFindCmd,
		      (ClientData)0, (Tcl_CmdDeleteProc *)0);

    Tcl_CreateObjCommand(interp, "::dvi::interp", DviInterpCmd,
			 (ClientData)0, (Tcl_CmdDeleteProc *)0);

    Tcl_CreateObjCommand(interp, "::dvi::assocData", DviAssocCmd,
			 (ClientData)0, (Tcl_CmdDeleteProc *)0);
#endif /* DVI_DEBUG */

    cookieToFileTablePtr = InitCookieToFileTable(interp);
    if (cookieToFileTablePtr == (Tcl_HashTable *)0) {
	return TCL_ERROR;
    }

    Tcl_CreateObjCommand(interp, "::dvi::file", DviFileCmd,
			 (ClientData)cookieToFileTablePtr,
			 (Tcl_CmdDeleteProc *)0);

    Tcl_CreateObjCommand(interp, "::dvi::font", DviFontCmd,
		      (ClientData)0, (Tcl_CmdDeleteProc *)0);

#if ENABLE_TK
    if (Dviimg_Init(interp) != TCL_OK) {
	return TCL_ERROR;
    }
#endif

    return TCL_OK;
}
