//===-- Collection of utils for sinf/cosf/sincosf ---------------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC_MATH_GENERIC_SINCOSF_UTILS_H
#define LLVM_LIBC_SRC_MATH_GENERIC_SINCOSF_UTILS_H

#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/macros/config.h"
#include "src/__support/macros/properties/cpu_features.h" // LIBC_TARGET_CPU_HAS_FMA

#if defined(LIBC_TARGET_CPU_HAS_FMA)
#include "range_reduction_fma.h"
// using namespace LIBC_NAMESPACE::fma;
using LIBC_NAMESPACE::fma::FAST_PASS_BOUND;
using LIBC_NAMESPACE::fma::large_range_reduction;
using LIBC_NAMESPACE::fma::small_range_reduction;

#else
#include "range_reduction.h"
// using namespace LIBC_NAMESPACE::generic;
using LIBC_NAMESPACE::generic::FAST_PASS_BOUND;
using LIBC_NAMESPACE::generic::large_range_reduction;
using LIBC_NAMESPACE::generic::small_range_reduction;
#endif // LIBC_TARGET_CPU_HAS_FMA

namespace LIBC_NAMESPACE_DECL {

// Lookup table for sin(k * pi / 32) with k = 0, ..., 63.
// Table is generated with Sollya as follow:
// > display = hexadecimal;
// > for k from 0 to 63 do { D(sin(k * pi/32)); };
const double SIN_K_PI_OVER_32[64] = {
    0x0.0000000000000p+0,  0x1.917a6bc29b42cp-4,  0x1.8f8b83c69a60bp-3,
    0x1.294062ed59f06p-2,  0x1.87de2a6aea963p-2,  0x1.e2b5d3806f63bp-2,
    0x1.1c73b39ae68c8p-1,  0x1.44cf325091dd6p-1,  0x1.6a09e667f3bcdp-1,
    0x1.8bc806b151741p-1,  0x1.a9b66290ea1a3p-1,  0x1.c38b2f180bdb1p-1,
    0x1.d906bcf328d46p-1,  0x1.e9f4156c62ddap-1,  0x1.f6297cff75cbp-1,
    0x1.fd88da3d12526p-1,  0x1.0000000000000p+0,  0x1.fd88da3d12526p-1,
    0x1.f6297cff75cbp-1,   0x1.e9f4156c62ddap-1,  0x1.d906bcf328d46p-1,
    0x1.c38b2f180bdb1p-1,  0x1.a9b66290ea1a3p-1,  0x1.8bc806b151741p-1,
    0x1.6a09e667f3bcdp-1,  0x1.44cf325091dd6p-1,  0x1.1c73b39ae68c8p-1,
    0x1.e2b5d3806f63bp-2,  0x1.87de2a6aea963p-2,  0x1.294062ed59f06p-2,
    0x1.8f8b83c69a60bp-3,  0x1.917a6bc29b42cp-4,  0x0.0000000000000p+0,
    -0x1.917a6bc29b42cp-4, -0x1.8f8b83c69a60bp-3, -0x1.294062ed59f06p-2,
    -0x1.87de2a6aea963p-2, -0x1.e2b5d3806f63bp-2, -0x1.1c73b39ae68c8p-1,
    -0x1.44cf325091dd6p-1, -0x1.6a09e667f3bcdp-1, -0x1.8bc806b151741p-1,
    -0x1.a9b66290ea1a3p-1, -0x1.c38b2f180bdb1p-1, -0x1.d906bcf328d46p-1,
    -0x1.e9f4156c62ddap-1, -0x1.f6297cff75cbp-1,  -0x1.fd88da3d12526p-1,
    -0x1.0000000000000p+0, -0x1.fd88da3d12526p-1, -0x1.f6297cff75cbp-1,
    -0x1.e9f4156c62ddap-1, -0x1.d906bcf328d46p-1, -0x1.c38b2f180bdb1p-1,
    -0x1.a9b66290ea1a3p-1, -0x1.8bc806b151741p-1, -0x1.6a09e667f3bcdp-1,
    -0x1.44cf325091dd6p-1, -0x1.1c73b39ae68c8p-1, -0x1.e2b5d3806f63bp-2,
    -0x1.87de2a6aea963p-2, -0x1.294062ed59f06p-2, -0x1.8f8b83c69a60bp-3,
    -0x1.917a6bc29b42cp-4,
};

static LIBC_INLINE void sincosf_poly_eval(int64_t k, double y, double &sin_k,
                                          double &cos_k, double &sin_y,
                                          double &cosm1_y) {
  // After range reduction, k = round(x * 32 / pi) and y = (x * 32 / pi) - k.
  // So k is an integer and -0.5 <= y <= 0.5.
  // Then sin(x) = sin((k + y)*pi/32)
  //             = sin(y*pi/32) * cos(k*pi/32) + cos(y*pi/32) * sin(k*pi/32)

  sin_k = SIN_K_PI_OVER_32[k & 63];
  // cos(k * pi/32) = sin(k * pi/32 + pi/2) = sin((k + 16) * pi/32).
  // cos_k = cos(k * pi/32)
  cos_k = SIN_K_PI_OVER_32[(k + 16) & 63];

  double ysq = y * y;

  // Degree-6 minimax even polynomial for sin(y*pi/32)/y generated by Sollya
  // with:
  // > Q = fpminimax(sin(y*pi/32)/y, [|0, 2, 4, 6|], [|D...|], [0, 0.5]);
  sin_y =
      y * fputil::polyeval(ysq, 0x1.921fb54442d18p-4, -0x1.4abbce625abb1p-13,
                           0x1.466bc624f2776p-24, -0x1.32c3a619d4a7ep-36);
  // Degree-6 minimax even polynomial for cos(y*pi/32) generated by Sollya with:
  // > P = fpminimax(cos(x*pi/32), [|0, 2, 4, 6|], [|1, D...|], [0, 0.5]);
  // Note that cosm1_y = cos(y*pi/32) - 1.
  cosm1_y = ysq * fputil::polyeval(ysq, -0x1.3bd3cc9be430bp-8,
                                   0x1.03c1f070c2e27p-18, -0x1.55cc84bd942p-30);
}

LIBC_INLINE void sincosf_eval(double xd, uint32_t x_abs, double &sin_k,
                              double &cos_k, double &sin_y, double &cosm1_y) {
  int64_t k;
  double y;

  if (LIBC_LIKELY(x_abs < FAST_PASS_BOUND)) {
    k = small_range_reduction(xd, y);
  } else {
    fputil::FPBits<float> x_bits(x_abs);
    k = large_range_reduction(xd, x_bits.get_exponent(), y);
  }

  sincosf_poly_eval(k, y, sin_k, cos_k, sin_y, cosm1_y);
}

// Return k and y, where
//   k = round(x * 32) and y = (x * 32) - k.
//   => pi * x = (k + y) * pi / 32
static LIBC_INLINE int64_t range_reduction_sincospi(double x, double &y) {
  double kd = fputil::nearest_integer(x * 32);
  y = fputil::multiply_add<double>(x, 32.0, -kd);

  return static_cast<int64_t>(kd);
}

LIBC_INLINE void sincospif_eval(double xd, double &sin_k, double &cos_k,
                                double &sin_y, double &cosm1_y) {
  double y;
  int64_t k = range_reduction_sincospi(xd, y);
  sincosf_poly_eval(k, y, sin_k, cos_k, sin_y, cosm1_y);
}

} // namespace LIBC_NAMESPACE_DECL

#endif // LLVM_LIBC_SRC_MATH_GENERIC_SINCOSF_UTILS_H
