package converter

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"gitlab.com/gitlab-org/security-products/analyzers/trivy-k8s-wrapper/data/analyzers/report"
	"gitlab.com/gitlab-org/security-products/analyzers/trivy-k8s-wrapper/prototool"
)

func TestToIdentifiers(t *testing.T) {
	proto := []*prototool.Identifier{
		{Type: "cve", Name: "name1", Value: "val1", Url: "url1"},
		{Type: "cve", Name: "name2", Value: "val2", Url: "url2"},
	}
	c := Converter{}
	assert.Equal(t, []report.Identifier{{Type: "cve", Name: "name1", Value: "val1", URL: "url1"},
		{Type: "cve", Name: "name2", Value: "val2", URL: "url2"}}, c.ToIdentifiers(proto))
}

func TestToLinks(t *testing.T) {
	proto := []*prototool.Link{
		{Name: "name1", Url: "url1"},
		{Name: "name2", Url: "url2"},
	}
	c := Converter{}
	assert.Equal(t, []report.Link{{Name: "name1", URL: "url1"},
		{Name: "name2", URL: "url2"}}, c.ToLinks(proto))
}

func TestToLocation(t *testing.T) {
	proto := &prototool.Location{
		Dependency: &prototool.Dependency{
			Package: &prototool.Package{Name: "name"},
			Version: "231",
		},
		KubernetesResource: &prototool.KubernetesResource{
			Namespace:     "n1",
			Name:          "n2",
			Kind:          "n3",
			AgentId:       "4",
			ContainerName: "n5",
		},
		Image:           "image",
		OperatingSystem: "os",
	}
	c := Converter{}
	assert.Equal(t, report.Location{
		Dependency: &report.Dependency{
			Package: report.Package{Name: "name"},
			Version: "231",
		},
		KubernetesResource: &report.KubernetesResource{
			Namespace:     "n1",
			Name:          "n2",
			Kind:          "n3",
			AgentID:       "4",
			ContainerName: "n5",
		},
		Image:           "image",
		OperatingSystem: "os",
	}, c.ToLocation(proto))
}

func TestToSeverity(t *testing.T) {

	tcs := []struct {
		severity    string
		expSeverity report.SeverityLevel
	}{
		{
			severity:    "CRITICAL",
			expSeverity: report.SeverityLevelCritical,
		},
		{
			severity:    "HIGH",
			expSeverity: report.SeverityLevelHigh,
		},
		{
			severity:    "MEDIUM",
			expSeverity: report.SeverityLevelMedium,
		},
		{
			severity:    "LOW",
			expSeverity: report.SeverityLevelLow,
		},
		{
			severity:    "NONE",
			expSeverity: report.SeverityLevelInfo,
		},
		{
			severity:    "UNKNOWN",
			expSeverity: report.SeverityLevelUnknown,
		},
	}
	for _, tc := range tcs {
		t.Run(tc.severity, func(t *testing.T) {
			c := Converter{}
			assert.Equal(t, tc.expSeverity, c.ToSeverity(tc.severity))
		})
	}

}
