/***************************************************************************
 *   Copyright (C) 2005-2006 Gao Xianchao                                  *
 *                 2007 Gao Xianchao gnap_an linux_lyb ahlongxp            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*
 * Author:	gxc
 * Create data:	2005-10-14 20:09
 */
 
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include "PeerLink.h"
#include "log.h"
#include "utils.h"

#define REQUEST_BLOCK_SIZE 16*1024
#define RECF_BUFFER_SIZE 8*1024

CPeerLink::CPeerLink()
: _state(PS_INIT)
, _accepted(false)
, _bitSetRecved(false)
, _canRead(false)
, _canWrite(true)
{
}

CPeerLink::~CPeerLink()
{
}

void CPeerLink::connect(const char* ip, unsigned short port)
{
	CSocket::createTCPSocket();
	CSocket::setReactor(_manager->getBTTask()->getSocketReactor());

	_ip = ip;
	_port = port;
	
	//LOG_DEBUG("CPeerLink::connect, fd = "<<getHandle()<<" ip = "<<_ip<<" port = "<<_port);
	
	maskWrite(true);
	CSocket::connect(ip, port);
	_state = PS_CONNECTING;
	
	_connectTimeoutTimer = getReactor()->addTimer(this, 10000, true);
}

void CPeerLink::attach(int handle, const char* ip, unsigned short port, IPeerManager* manager)
{
	_ip = ip;
	_port = port;	
	_accepted = true;
	_state = PS_ESTABLISHED;
	_connectTimeoutTimer = 0;
	CSocket::attach(handle);
	setPeerManager(manager);
	CSocket::setReactor(_manager->getBTTask()->getSocketReactor());
	onConnect();
}

bool CPeerLink::isAccepted()
{
	return _accepted;
}

void CPeerLink::closeLink()
{
	handleClose();
}

void CPeerLink::onConnect()
{
	//LOG_DEBUG("CPeerLink::onConnect, fd = "<<getHandle()<<" ip = "<<_ip<<" port = "<<_port);
	
	maskRead(true);
	_sendBuffer.clear();
	_recvBuffer.clear();
	_handShaked = false;
	
	_pieceRequest.alloc(NONE_PIECE_INDEX, 0, 0);
	
	 _amChoking = true;
	_amInterested = false;
	_peerChoking = true;;
	_peerInterested = false;
	
	_downloadCount = 0;
	_uploadCount = 0;
	_lastCountSpeedTime = GetTickCount();
	_lastDownloadCount = 0;
	_lastUploadCount = 0;
	_uploadSpeed = 0;
	_downloadSpeed = 0;
	
	_canRead = false;
	_canWrite = true;
	
	LOG_DEBUG("+++++ Add RateMeasure Client, ip="<<_ip);
	_manager->getBTTask()->getRateMeasure()->addClient(this);
		
	sendHandshake();
}

void CPeerLink::onConnectFailed()
{
	//LOG_DEBUG("CPeerLink::onConnectFailed, fd = "<<getHandle()<<" ip = "<<_ip<<" port = "<<_port);
}

void CPeerLink::onConnectClosed()
{
	//LOG_DEBUG("CPeerLink::onConnectClose, fd = "<<getHandle()<<" ip = "<<_ip<<" port = "<<_port);
	LOG_DEBUG("+++++ remove RateMeasure Client, ip="<<_ip);
	_manager->getBTTask()->getRateMeasure()->removeClient(this);
	
	_manager->getBTTask()->getStorage()->abandonPieceTask(_pieceRequest.getPieceIndex());
	_pieceRequest.alloc(NONE_PIECE_INDEX, 0, 0);
}

void CPeerLink::onSendComplete()
{
	doPieceSend();
}

int CPeerLink::handleRead()
{
	LOG_DEBUG("CPeerLink::handleRead, ip = "<<_ip<<", _canRead="<<_canRead);
	_canRead = true;
	
	return 0;
}

int CPeerLink::handleWrite()
{
	if(_state == PS_CONNECTING)
	{
		_state = PS_ESTABLISHED;
		maskWrite(false);
		onConnect();
		
		return 0;
	}
	
	LOG_DEBUG("CPeerLink::handleWrite, ip = "<<_ip<<", _canWrite="<<_canWrite);
	
	_canWrite = true;
			
	return 0;
}

void CPeerLink::handleClose()
{
	//LOG_DEBUG("CPeerLink::handleClose, fd = "<<getHandle());
	
	if(_state == PS_CONNECTING)
	{
		_state = PS_CONNECTFAILED;
		onConnectFailed();
	}
	
	if(_state == PS_ESTABLISHED)
	{
		_state = PS_CLOSED;
		onConnectClosed();		
	}
	
	if(_connectTimeoutTimer != 0)
	{
	 	getReactor()->removeTimer(_connectTimeoutTimer);
	 	_connectTimeoutTimer = 0;
	}	
	
	_canRead = false;
	_canWrite = false;

	setReactor(NULL);
	CSocket::close();	
}

void CPeerLink::blockWrite(bool block)
{
	if(!block)
	{
		if((_sendBuffer.size() > 0)
			&& !_canWrite)
		{
			if(!maskWrite())
			{
				maskWrite(true);
			}	
		}
		return;
	}
	
	if(block
		&& maskWrite())
	{
		maskWrite(false);
	}	
}

void CPeerLink::blockRead(bool block)
{
	if(block
		&& maskRead())
	{
		maskRead(false);
		return;
	}
	
	if(!block
		&& !maskRead())
	{
		maskRead(true);
	}
}

void CPeerLink::setWritePriority(unsigned int Priority)
{
	_writePriority = Priority;
}

unsigned int CPeerLink::getWritePriority()
{
	return _writePriority;
}

void CPeerLink::setReadPriority(unsigned int Priority)
{
	_readPriority = Priority;
}

unsigned int CPeerLink::getReadPriority()
{
	return _readPriority;
}

bool CPeerLink::canWrite()
{
	return _canWrite && (_sendBuffer.size() > 0);
}

bool CPeerLink::canRead()
{
	return _canRead;
}

int CPeerLink::doWrite(unsigned int count)
{	
	LOG_DEBUG("+++ doWrite, count = "<<count<<" ip="<<_ip<<" _sendBuffer.size()="<<_sendBuffer.size());
	unsigned int sendCount = 0;
	
	if(_state == PS_ESTABLISHED)
	{
		for(;(_sendBuffer.size()>0) && (sendCount<count);)
		{	
			size_t sendSize = _sendBuffer.size();
			if(sendSize > count)
			{
				sendSize = count;
			}
			int ret = send(getHandle(), _sendBuffer.data(), sendSize, 0);
			if(ret == -1)
			{
				if(errno == EAGAIN
					|| errno == EINTR)
				{
					//LOG_DEBUG("+++ _canWrite = false ip="<<_ip);
					_canWrite = false;			
				}
				else
				{
					//error
					closeLink();
					return sendCount;
				}
				
				break;
			}
			
			sendCount += ret;
			_uploadCount += ret;
			_manager->getBTTask()->incUploadCount(ret);
			
			_sendBuffer.erase(0, ret);
			
			if(_sendBuffer.size() == 0)
			{
				onSendComplete();
				break;
			}
		}
	}
	
	//LOG_DEBUG("_canWrite="<<_canWrite<<", maskWrite()="<<maskWrite());
	
	if(_canWrite)
	{
		if(maskWrite())
		{
			//LOG_DEBUG("++++ maskwrite(false), ip="<<_ip<<" sendCount="<<sendCount);
			maskWrite(false);
		}
	}
	else
	{
		if(!maskWrite())
		{
			//LOG_DEBUG("++++ maskwrite(true), ip="<<_ip<<" sendCount="<<sendCount);
			maskWrite(true);
		}
	}
	
	return 	sendCount;
}

int CPeerLink::doRead(unsigned int count)
{
	LOG_DEBUG("+++ doRead, count = "<<count<<" ip="<<_ip);
	unsigned int readCount = 0;
	
	char* buf = new char[RECF_BUFFER_SIZE];
	
	for(;readCount<count;)
	{
		size_t readSize = RECF_BUFFER_SIZE;
		if(readSize > count - readCount)
		{
			readSize = count - readCount;
		}
		
		int ret = recv(getHandle(), buf, readSize, 0);
		if(ret == 0)
		{
			closeLink();
			//LOG_DEBUG("recv return 0, fd="<<getHandle());
			delete[] buf;
			return readCount;
		}
		
		if(ret == -1)
		{
			if(errno == EAGAIN)
			{
				LOG_DEBUG("+++ _canRead = false ip="<<_ip);
				_canRead = false;
				break;
			}
			
			if(errno == EINTR)
			{
				//LOG_DEBUG("recv return EINTR, fd="<<getHandle());
				continue;
			}
			
			closeLink();
			//error
			delete[] buf;
			return readCount;
		}
		
		//LOG_DEBUG("recv "<<ret<<" bytes, ip="<<_ip);
		
		if(ret > 0)
		{
			readCount += ret;
			_downloadCount += ret;
			_manager->getBTTask()->incDownlaodCount(ret);
			
			_recvBuffer.append((const char*)buf, ret);
		}
	}
	
	delete[] buf;
	
	processRecvData();	
	
	LOG_DEBUG("_canRead="<<_canRead<<", maskRead()="<<maskRead()<<", readCount="<<readCount);
	
	if(_canRead)
	{
		if(maskRead())
		{
			LOG_DEBUG("++++ maskRead(false), ip="<<_ip);
			maskRead(false);
		}
	}
	else
	{
		if(!maskRead())
		{
			LOG_DEBUG("++++ maskRead(true), ip="<<_ip);
			maskRead(true);
		}
	}
	
	return readCount;
}

TPeerState CPeerLink::getState()
{
	return _state;
}

void CPeerLink::setPeerManager(IPeerManager* manager)
{
	_manager = manager;
}

const char* CPeerLink::getIP()
{
	return _ip.c_str();
}

unsigned short CPeerLink::getPort()
{
	return _port;
}

void CPeerLink::onTimer(unsigned int id)
{
	if(id == _connectTimeoutTimer)
	{
		_connectTimeoutTimer = 0;
		
		if(_state == PS_CONNECTING)
		{
			//LOG_DEBUG("connection timeout, fd = "<<getHandle());
			closeLink();
		}
	}
}

void CPeerLink::sendHandshake()
{
	char buf[68];
	memset(buf, 0, sizeof(buf));
	
	buf[0] = 19;
	strncpy(buf+1, "BitTorrent protocol", 19);
	
	strncpy(buf+28, (const char*)_manager->getBTTask()->getTorrentFile()->getInfoHash(), 20);
	strncpy(buf+48, (const char*)_manager->getBTTask()->getPeerID().c_str(), 20);
	
	sendData(buf, sizeof(buf));
}

void CPeerLink::sendChoke(bool chocke)
{
	char buf[5];
	*((int*)buf) = htonl(1);
	
	if(chocke)
	{
		*((char*)(buf+4)) = 0;
		_amChoking = true;
	}
	else
	{
		*((char*)(buf+4)) = 1;
		_amChoking = false;
	}
	
	sendData(buf, sizeof(buf));
}

void CPeerLink::sendInterested(bool interested)
{
	char buf[5];
	*((int*)buf) = htonl(1);
	
	if(interested)
	{
		*((char*)(buf+4)) = 2;
		_amInterested = true;
	}
	else
	{
		*((char*)(buf+4)) = 3;
		_amInterested = false;
	}
	
	sendData(buf, sizeof(buf));	
}

void  CPeerLink::sendBitfield()
{
	std::string bitfield = _manager->getBTTask()->getStorage()->getBitfield();
	if(bitfield.size() == 0)
	{
		return;
	}
	
	std::string buf;
	char header[5];	
	*((int*)(header)) =  htonl(1 + bitfield.size());
	*((char*)(header+4))  = 5;
	
	buf.append(header, 5);
	buf.append(bitfield.data(), bitfield.size());
	
	sendData(buf.data(), buf.size());
}

void CPeerLink::sendHave(unsigned int pieceIndex)
{
	char buf[9];
	*((unsigned int*)buf) = htonl(5);
	*((unsigned char*)(buf+4)) = 4;
	*((unsigned int*)(buf+5)) = htonl(pieceIndex);
	
	sendData(buf, sizeof(buf));
}

void CPeerLink::sendPieceRequest(unsigned int pieceIndex, unsigned int offset, unsigned int len)
{
	char buf[17];
	
	*((unsigned int*)buf) = htonl(13);
	*((unsigned char*)(buf+4)) = 6;
	*((unsigned int*)(buf+5)) = htonl(pieceIndex);
	*((unsigned int*)(buf+9)) = htonl(offset);
	*((unsigned int*)(buf+13)) = htonl(len);
	
	sendData(buf, sizeof(buf));
}

void CPeerLink::sendPieceData(unsigned int pieceIndex, unsigned int offset, std::string& data)
{
	char buf[13];
	*((unsigned int*)buf) = htonl(9+data.size());
	*((unsigned char*)(buf+4)) = 7;
	*((unsigned int*)(buf+5)) = htonl(pieceIndex);
	*((unsigned int*)(buf+9)) = htonl(offset);	
	
	std::string pkg;
	pkg.append((const char*)buf, sizeof(buf));
	pkg.append(data.c_str(), data.size());
	
	sendData(pkg.data(), pkg.size());
}

void CPeerLink::sendPieceCancel(unsigned int pieceIndex, unsigned int offset, unsigned int len)
{
	char buf[17];
	
	*((unsigned int*)buf) = htonl(13);
	*((unsigned char*)(buf+4)) = 8;
	*((unsigned int*)(buf+5)) = htonl(pieceIndex);
	*((unsigned int*)(buf+9)) = htonl(offset);
	*((unsigned int*)(buf+13)) = htonl(len);
	
	sendData(buf, sizeof(buf));	
}

void CPeerLink::sendData(const void* data, size_t len)
{
	_sendBuffer.append((const char*)data, len);
}

int CPeerLink::processRecvData()
{
	if(!_handShaked)
	{
		if(_recvBuffer.size() >= 68)
		{
			std::string handShakeStr = _recvBuffer.substr(0, 68);
			_recvBuffer.erase(0, 68);		
			if(!checkHandshake(handShakeStr))
			{
				return -1;
			}
		}
	}
	else
	{
		for(;_recvBuffer.size() >= 4;)
		{
			unsigned int len = *((unsigned int*)_recvBuffer.data());
			len = ntohl(len);
			if(len == 0)
			{
				//LOG_DEBUG("++++++++++++++ "<<_ip<< " ++++++++++recv cmd keepalive");
				_recvBuffer.erase(0, 4);
				continue;
			}
			
			if(_recvBuffer.size() < 5)
			{
				break;
			}
			unsigned char cmdID = _recvBuffer[4];
			if(_recvBuffer.size() >= size_t(len+4))
			{
				if(-1 == processCmd(cmdID, (void*)(_recvBuffer.data()+5), len-1))
				{
					return -1;
				}
				
				_recvBuffer.erase(0, len+4);
			}
			else
			{
				break;
			}
		}		
	}
	
	return 0;
}

bool CPeerLink::checkHandshake(std::string info)
{
	if(info[0] != 19)
	{
		LOG_DEBUG("handshake[0] != 19, ip="<<_ip);
		return false;
	}
	
	std::string str = info.substr(1, 19);
	if(str != "BitTorrent protocol")
	{
		LOG_DEBUG("handshake[protocol] != BitTorrent protocol");
		return false;
	}
	
	str = info.substr(28, 20);
	std::string str2;
	str2.append((const char*)_manager->getBTTask()->getTorrentFile()->getInfoHash(), 20);
	if(str != str2)
	{
		LOG_DEBUG("handshake[info_hash] error");
		return false;
	}
	
	_id = info.substr(48, 20);
		
	//LOG_DEBUG("+++++++++++++++++ handshake ok, ip="<<_ip);
	_handShaked = true;
	
	sendBitfield();
	sendInterested(true);
	
	//alloc peer's bitSet
	_bitSet.alloc(_manager->getBTTask()->getTorrentFile()->getPieceCount());
	
	return true;
}

int CPeerLink::processCmd(unsigned cmd, void* data, size_t dataLen)
{
//	LOG_DEBUG("++++++++++++++ "<<_ip<< " ++++++++++recv cmd "<<cmd<<", dateLen="<<dataLen);
	
	switch(cmd)
	{
		case 0:
			if(processCmdChoke(data, dataLen) == -1)
			{
				return -1;
			}
			break;			
		case 1:
			if(processCmdUnchoke(data, dataLen) == -1)
			{
				return -1;
			}
			break;						
		case 2:
			if(processCmdInterested(data, dataLen) == -1)
			{
				return -1;
			}
			break;				
		case 3:
			if(processCmdNotInterested(data, dataLen) == -1)
			{
				return -1;
			}
			break;			
		case 4:
			if(processCmdHave(data, dataLen) == -1)
			{
				return -1;
			}
			break;
		case 5:
			if(processCmdBitfield(data,dataLen) == -1)
			{
				return -1;
			}
			break;
		case 6:
			if(processCmdRequest(data,dataLen) == -1)
			{
				return -1;
			}
			break;			
		case 7:
			if(processCmdPiece(data,dataLen) == -1)
			{
				return -1;
			}
			break;			
		case 8:
			if(processCmdCancel(data,dataLen) == -1)
			{
				return -1;
			}
			break;							
	}
	return 0;
}

int CPeerLink::processCmdChoke(void* data, size_t dataLen)
{
	LOG_DEBUG("choked by " << _ip);
	_peerChoking = true;
	
	//_manager->getBTTask()->getStorage()->abandonPieceTask(_pieceRequest.getPieceIndex());
	//_pieceRequest.alloc(NONE_PIECE_INDEX, 0, 0);
	_pieceRequest.clearRequest();
	return 0;
}

int CPeerLink::processCmdUnchoke(void* data, size_t dataLen)
{
	LOG_DEBUG("unchoked by " << _ip);
	_peerChoking = false;
	
	 getNewPieceTask();
	 doPieceRequest();
	
	return 0;
}

int CPeerLink::processCmdInterested(void* data, size_t dataLen)
{
	LOG_DEBUG("interested by " << _ip);
	_peerInterested = true;
	return 0;
}

int CPeerLink::processCmdNotInterested(void* data, size_t dataLen)
{
	LOG_DEBUG("uninterested by " << _ip);
	_peerInterested = false;
	return 0;
}

int CPeerLink::processCmdHave(void* data, size_t dataLen)
{
	if(dataLen != 4)
	{
		return -1;
	}
	
	unsigned int index = *((unsigned int*)data);
	index = ntohl(index);
	_bitSet.set(index, true);
	
	if(!_peerChoking)
	{
		getNewPieceTask();
		doPieceRequest();
	}	
	
	return 0;
}

int CPeerLink::processCmdBitfield(void* data, size_t dataLen)
{
	//LOG_DEBUG("CPeerLink::processCmdBitfield");
	
	std::string bitset;
	bitset.append((const char*)data, dataLen);
	_bitSet.alloc(bitset, _manager->getBTTask()->getTorrentFile()->getPieceCount());
	
	std::string log;
	for(unsigned int i =0; i<_bitSet.getSize(); ++i)
	{
		if(_bitSet.isSet(i))
		{
			log.append("1");
		}
		else
		{
			log.append("0");
		}
	}
	_bitSetRecved = true;
	
	//LOG_DEBUG("++++++bitset "<<log);
	
	return 0;
}

int CPeerLink::processCmdRequest(void* data, size_t dataLen)
{
	if(dataLen != 12)
	{
		return -1;
	}
	
	if(_amChoking)
	{
		//I am choking this client
		return 0;
	}
	
	unsigned int index = *((unsigned int*)data);
	index = ntohl(index);
	unsigned int offset = *((unsigned int*)((char*)data+4));
	offset = ntohl(offset);	
	unsigned int len = *((unsigned int*)((char*)data+8));
	len = ntohl(len);		
	
	//LOG_DEBUG(">>>>>>>>>>>> PieceRequest, index="<<index<<" offset="<<offset<<" len="<<len<<" ip="<<_ip);
	
	TPeerPieceRequestList::iterator iter = _peerRequestList.begin();
	for(;iter!=_peerRequestList.end();++iter)
	{
		if(iter->index == index
			&& iter->offset == offset)
		{
			break;
		}
	}
	
	if(iter == _peerRequestList.end())
	{
		TPeerPieceRequest peerRequest;
		peerRequest.index = index;
		peerRequest.offset = offset;
		peerRequest.len = len;
		
		_peerRequestList.push_back(peerRequest);
	}
	
	doPieceSend();
	
	return 0;
}

void CPeerLink::doPieceSend()
{
	if(!_amChoking
		&& _sendBuffer.size() == 0
		&& _peerRequestList.size() > 0)
	{
		TPeerPieceRequest peerRequest = _peerRequestList.front();
		_peerRequestList.pop_front();
		
		std::string pieceDdata = _manager->getBTTask()->getStorage()->readData(peerRequest.index, peerRequest.offset, peerRequest.len);
		if(pieceDdata.size() == peerRequest.len)
		{
			//LOG_DEBUG(">>>>>>>>>> send piece data, index="<<peerRequest.index<<" offset="<<peerRequest.offset<<" len="<<peerRequest.len<<" ip="<<_ip);
			sendPieceData(peerRequest.index, peerRequest.offset, pieceDdata);
		}		
	}
}


int CPeerLink::processCmdPiece(void* data, size_t dataLen)
{
	if(dataLen <= 8)
	{
		return -1;
	}
	
	unsigned int index = *((unsigned int*)data);
	index = ntohl(index);
	unsigned int offset = *((unsigned int*)((char*)data+4));
	offset = ntohl(offset);
	
	unsigned len = dataLen - 8;
	
	if(index != _pieceRequest.getPieceIndex())
	{
		return -1;
	}

	if(!_pieceRequest.addPieceData(offset, (const char*)data+8, len))
	{
		return -1;	
	}
	else
	{
		//LOG_DEBUG("<<<<<<<<< recv piece data, index="<<index<<" offset="<<offset<<" len="<<len);
	}
	
	if(	_pieceRequest.complete())
	{
		std::string pieceData = _pieceRequest.getPiece();
		
		if(shaString((char*)pieceData.data(), pieceData.size()) 
			== _manager->getBTTask()->getTorrentFile()->getPieceHash(index))
		{
			//LOG_DEBUG("<<<<<<<<<< piece "<<index<<" download completed");
			
			 _manager->getBTTask()->getStorage()->writePiece(index, pieceData);
			 _manager->broadcastHave(index);
			 
			 getNewPieceTask();
		}
		else
		{
			LOG_DEBUG("<<<<<<<<  piece hash error! "<<index);
			_pieceRequest.alloc(NONE_PIECE_INDEX,  0, 0);
			closeLink();
		}
	}
	
	doPieceRequest();
	
	return 0;
}

int CPeerLink::processCmdCancel(void* data, size_t dataLen)
{
	if(dataLen != 12)
	{
		return -1;
	}
	
	unsigned int index = *((unsigned int*)data);
	index = ntohl(index);
	unsigned int offset = *((unsigned int*)((char*)data+4));
	offset = ntohl(offset);	
	unsigned int len = *((unsigned int*)((char*)data+8));
	len = ntohl(len);	
			
	TPeerPieceRequestList::iterator iter = _peerRequestList.begin();
	for(;iter!=_peerRequestList.end();++iter)
	{
		if(iter->index == index
			&& iter->offset == offset)
		{
			_peerRequestList.erase(iter);
			break;
		}
	}
	
	return 0;
}

void CPeerLink::notifyHavePiece(unsigned int pieceIndex)
{
	if(_handShaked)
	{
		if(_pieceRequest.getPieceIndex() == pieceIndex)
		{
			cancelPieceRequest(pieceIndex);
		}
		sendHave(pieceIndex);
	}
}

void CPeerLink::cancelPieceRequest(unsigned int pieceIndex)
{
	if(_pieceRequest.getPieceIndex() == pieceIndex)
	{
		unsigned int index;
		unsigned int offset;
		unsigned int len;
		
		while(_pieceRequest.cancelPendingRequest(index, offset, len))
		{
			sendPieceCancel(index, offset, len);
		}
		
		_pieceRequest.alloc(NONE_PIECE_INDEX, 0, 0);
	
		if(!_peerChoking)
		{
			getNewPieceTask();
			doPieceRequest();
		}	
	}
}

void CPeerLink::getNewPieceTask()
{
	if(!_pieceRequest.complete())
	{
		return;
	}
	
	if(_manager->getBTTask()->getStorage()->finished())
	{
		return;
	}
	
	if(_peerChoking)
	{
		return;
	}	
	
	 //get new task;
	unsigned int pieceIndex = _manager->getBTTask()->getStorage()->getPieceTask(&_bitSet);
	if(	pieceIndex != NONE_PIECE_INDEX)
	{
		if(!_amInterested)
		{
			sendInterested(true);
		}
		LOG_DEBUG("new piece task, index="<< pieceIndex<<" ip="<<_ip);
		_pieceRequest.alloc(pieceIndex,
			 _manager->getBTTask()->getStorage()->getPieceLength(pieceIndex),
			 REQUEST_BLOCK_SIZE);
	}
	else
	{
		if(_amInterested)
                {
		        sendInterested(false);
                }
	}
}

void CPeerLink::doPieceRequest()
{
	if(_peerChoking)
	{
		return;
	}
		
	//do request
	for(;_pieceRequest.getPendingCount()<5;)
	{
		unsigned int index, offset, len;
		if(!_pieceRequest.getRequest(index, offset, len))
		{
			break;
		}
		
		//LOG_DEBUG("<<<<<<<<<< doPieceRequest, index="<<index<<" offset="<<offset<<" len="<<len<<" ip="<<_ip);
		sendPieceRequest(index, offset, len);
	}	
}

bool CPeerLink::peerChoked()
{
	return _amChoking;
}

bool CPeerLink::peerInterested()
{
	return _peerInterested;
}

void CPeerLink::chokePeer(bool choke)
{
	if(choke)
	{
		LOG_DEBUG("********chokePeer, ip="<<_ip<<" downloadspeed="<<_downloadSpeed);
		_peerRequestList.clear();
	}
	else
	{
		LOG_DEBUG("********unchokePeer, ip="<<_ip<<" downloadspeed="<<_downloadSpeed);
	}
	sendChoke(choke);
}

unsigned int CPeerLink::getDownloadCount()
{
	return _downloadCount;
}

unsigned int CPeerLink::getUploadCount()
{
	return _uploadCount;
}

void CPeerLink::countSpeed()
{
	_uploadSpeed = (_uploadCount - _lastUploadCount)*1000/(GetTickCount()-_lastCountSpeedTime+1);
	_downloadSpeed = (_downloadCount - _lastDownloadCount)*1000/(GetTickCount()-_lastCountSpeedTime+1);
	
	//LOG_DEBUG("+++++++++ "<<_ip<<"  _uploadSpeed="<<_uploadSpeed<<" _downloadSpeed="<<_downloadSpeed);
	
	_lastCountSpeedTime = GetTickCount();
	_lastUploadCount = _uploadCount;
	_lastDownloadCount = _downloadCount;
}

bool CPeerLink::checkNeedClose()
{
	if(!_bitSetRecved)
	{
		return false;
	}

	if( _manager->getBTTask()->getStorage()->finished())
	{
		if(_bitSet.isAllSet())
		{
			return true;
		}
	}
	
	for(int i=0; i<_bitSet.getSize(); ++i)
	{
		if(_manager->getBTTask()->getStorage()->getBanedBitSet()->isSet(i))
		{
			continue;
		}
		
		if(_bitSet.isSet(i)
			&& (!_manager->getBTTask()->getStorage()->getBitSet()->isSet(i)))
			{
				return false;
			}
	}
	return true;
}

unsigned int CPeerLink::getUploadSpeed()
{
	return _uploadSpeed;
}

unsigned int CPeerLink::getDownloadSpeed()
{
	return _downloadSpeed;
}

void CPeerLink::onDownloadComplete()
{
	sendInterested(false);
	_pieceRequest.alloc(NONE_PIECE_INDEX, 0, 0);
}

	
