/* Vuls - Vulnerability Scanner
Copyright (C) 2016  Future Corporation , Japan.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package oval

import (
	"reflect"
	"testing"

	"github.com/future-architect/vuls/config"
	"github.com/future-architect/vuls/models"
	"github.com/future-architect/vuls/util"
	ovalmodels "github.com/kotakanbe/goval-dictionary/models"
)

func TestParseCvss2(t *testing.T) {
	type out struct {
		score  float64
		vector string
	}
	var tests = []struct {
		in  string
		out out
	}{
		{
			in: "5/AV:N/AC:L/Au:N/C:N/I:N/A:P",
			out: out{
				score:  5.0,
				vector: "AV:N/AC:L/Au:N/C:N/I:N/A:P",
			},
		},
		{
			in: "",
			out: out{
				score:  0,
				vector: "",
			},
		},
	}
	for _, tt := range tests {
		s, v := RedHatBase{}.parseCvss2(tt.in)
		if s != tt.out.score || v != tt.out.vector {
			t.Errorf("\nexpected: %f, %s\n  actual: %f, %s",
				tt.out.score, tt.out.vector, s, v)
		}
	}
}

func TestParseCvss3(t *testing.T) {
	type out struct {
		score  float64
		vector string
	}
	var tests = []struct {
		in  string
		out out
	}{
		{
			in: "5.6/CVSS:3.0/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L",
			out: out{
				score:  5.6,
				vector: "CVSS:3.0/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L",
			},
		},
		{
			in: "",
			out: out{
				score:  0,
				vector: "",
			},
		},
	}
	for _, tt := range tests {
		s, v := RedHatBase{}.parseCvss3(tt.in)
		if s != tt.out.score || v != tt.out.vector {
			t.Errorf("\nexpected: %f, %s\n  actual: %f, %s",
				tt.out.score, tt.out.vector, s, v)
		}
	}
}

func TestPackNamesOfUpdate(t *testing.T) {
	var tests = []struct {
		in       models.ScanResult
		defPacks defPacks
		out      models.ScanResult
	}{
		{
			in: models.ScanResult{
				ScannedCves: models.VulnInfos{
					"CVE-2000-1000": models.VulnInfo{
						AffectedPackages: models.PackageStatuses{
							{Name: "packA"},
							{Name: "packB", NotFixedYet: false},
						},
					},
				},
			},
			defPacks: defPacks{
				def: ovalmodels.Definition{
					Advisory: ovalmodels.Advisory{
						Cves: []ovalmodels.Cve{
							{
								CveID: "CVE-2000-1000",
							},
						},
					},
				},
				actuallyAffectedPackNames: map[string]bool{
					"packB": true,
				},
			},
			out: models.ScanResult{
				ScannedCves: models.VulnInfos{
					"CVE-2000-1000": models.VulnInfo{
						AffectedPackages: models.PackageStatuses{
							{Name: "packA"},
							{Name: "packB", NotFixedYet: true},
						},
					},
				},
			},
		},
	}

	util.Log = util.NewCustomLogger(config.ServerInfo{})
	for i, tt := range tests {
		RedHat{}.update(&tt.in, tt.defPacks)
		e := tt.out.ScannedCves["CVE-2000-1000"].AffectedPackages
		a := tt.in.ScannedCves["CVE-2000-1000"].AffectedPackages
		if !reflect.DeepEqual(a, e) {
			t.Errorf("[%d] expected: %v\n  actual: %v\n", i, e, a)
		}
	}
}
