#include "RFMxx.h"
#include <SPI.h>

// Usage with ESP8266: to use this lib with ESP8266, we need a: #define ESP8266 
// This can be done in some IDEs like visual micro on the project level.
// In Arduino IDE it's not possible.
// In this case, uncomment the following line:
//// #define ESP8266

volatile byte RFMxx::_mode;       // current transceiver state
volatile int RFMxx::RSSI;
volatile int16_t RFMxx::FEI;

RFMxx* RFMxx::selfPointer;

// constructor
RFMxx::RFMxx(bool isPrimary) {
  m_debug = false;
  m_payloadReady = false;

  // No radio found until now
  m_radioType = RFMxx::None;
}

void RFMxx::InitialzeVantage(byte freqBand) {
#ifdef _DEBUG
#define fName "RFMxx::InitialzeVantage: "
  m_debug = true;
  Serial.print(fName);
  Serial.println("enter");
#endif

  // frequency map and index
  m_freqBand = freqBand;
  m_freqBandIdx = 0;

#ifdef _DEBUG
  Serial.print(fName);
  Serial.println("SPI");
#endif
  pinMode(SPI_CS, OUTPUT);
  SPI.setDataMode(SPI_MODE0);
  SPI.setBitOrder(MSBFIRST);
  SPI.setClockDivider(SPI_CLOCK_DIV2); //max speed, except on Due which can run at system clock speed
  SPI.begin();

  // Is there a RFM69 ?
#ifdef _DEBUG
  Serial.print(fName);
  Serial.println("radio");
#endif
  WriteReg(REG_PAYLOADLENGTH, 0xA);
  if (ReadReg(REG_PAYLOADLENGTH) == 0xA) {
    WriteReg(REG_PAYLOADLENGTH, 0x40);
    if (ReadReg(REG_PAYLOADLENGTH) == 0x40) {
      m_radioType = RFMxx::RFM69CW;
    }
  }

#ifdef _DEBUG
  Serial.print(fName);
  Serial.println("sync");
#endif
  do
	  WriteReg(REG_SYNCVALUE1, 0xaa);
  while (ReadReg(REG_SYNCVALUE1) != 0xaa);
  do
	  WriteReg(REG_SYNCVALUE1, 0x55);
  while (ReadReg(REG_SYNCVALUE1) != 0x55);

  /* 0x01 */ WriteReg(REG_OPMODE, RF_OPMODE_SEQUENCER_ON | RF_OPMODE_LISTEN_OFF | RF_OPMODE_STANDBY);
  /* 0x02 */ WriteReg(REG_DATAMODUL, RF_DATAMODUL_DATAMODE_PACKET | RF_DATAMODUL_MODULATIONTYPE_FSK | RF_DATAMODUL_MODULATIONSHAPING_10); // Davis uses Gaussian shaping with BT=0.5
  /* 0x03 */ WriteReg(REG_BITRATEMSB, RF_BITRATEMSB_19200); // Davis uses a datarate of 19.2 KBPS
  /* 0x04 */ WriteReg(REG_BITRATELSB, RF_BITRATELSB_19200);
  /* 0x05 */ WriteReg(REG_FDEVMSB, RF_FDEVMSB_9900); // Davis uses a deviation of 9.9 kHz
  /* 0x06 */ WriteReg(REG_FDEVLSB, RF_FDEVLSB_9900);
  /* 0x07 to 0x09 are REG_FRFMSB to LSB. No sense setting them here. Done in main routine. */
  /* 0x0B */ WriteReg(REG_AFCCTRL, RF_AFCLOWBETA_OFF); // TODO: Should use LOWBETA_ON, but having trouble getting it working
  /* 0x12 */ WriteReg(REG_PARAMP, RF_PARAMP_25); // xxx
  // looks like PA1 and PA2 are not implemented on RFM69W, hence the max output power is 13dBm
  // +17dBm and +20dBm are possible on RFM69HW
  // +13dBm formula: Pout=-18+OutputPower (with PA0 or PA1**)
  // +17dBm formula: Pout=-14+OutputPower (with PA1 and PA2)**
  // +20dBpaym formula: Pout=-11+OutputPower (with PA1 and PA2)** and high power PA settings (section 3.3.7 in datasheet)
  /* 0x11 */ WriteReg(REG_PALEVEL, RF_PALEVEL_PA0_ON | RF_PALEVEL_PA1_OFF | RF_PALEVEL_PA2_OFF | 31);
  ///* 0x13 */ WriteReg(REG_OCP, RF_OCP_ON | RF_OCP_TRIM_95); //over current protection (default is 95mA)
  /* 0x18 */ WriteReg(REG_LNA, RF_LNA_ZIN_50 | RF_LNA_GAINSELECT_AUTO); // Not sure which is correct!
  // RXBW defaults are { REG_RXBW, RF_RXBW_DCCFREQ_010 | RF_RXBW_MANT_24 | RF_RXBW_EXP_5} (RxBw: 10.4khz)
  /* 0x19 */ WriteReg(REG_RXBW, RF_RXBW_DCCFREQ_010 | RF_RXBW_MANT_20 | RF_RXBW_EXP_3); // Use 25 kHz BW (BitRate < 2 * RxBw)
  // for REG_RXBW 50 kHz seems to work better and fixes console retransmit reception, too
  /* 0x1A */ WriteReg(REG_AFCBW, RF_RXBW_DCCFREQ_010 | RF_RXBW_MANT_20 | RF_RXBW_EXP_2); // Use double the bandwidth during AFC as reception
  /* 0x1B - 0x1D These registers are for OOK.  Not used */
  /* 0x1E */ WriteReg(REG_AFCFEI, RF_AFCFEI_AFCAUTOCLEAR_ON | RF_AFCFEI_AFCAUTO_ON);
  /* 0x1F & 0x20 AFC MSB and LSB values, respectively */
  /* 0x21 & 0x22 FEI MSB and LSB values, respectively */
  /* 0x23 & 0x24 RSSI MSB and LSB values, respectively */
  /* 0x25 */ WriteReg(REG_DIOMAPPING1, RF_DIOMAPPING1_DIO0_01); //DIO0 is the only IRQ we're using
  /* 0x26 RegDioMapping2 */
  /* 0x27 RegIRQFlags1 */
  /* 0x28 */ WriteReg(REG_IRQFLAGS2, RF_IRQFLAGS2_FIFOOVERRUN); // Reset the FIFOs. Fixes a problem I had with bad first packet.
  /* 0x29 */ WriteReg(REG_RSSITHRESH, 190); //must be set to dBm = (-Sensitivity / 2) - default is 0xE4=228 so -114dBm
  /* 0x2a & 0x2b RegRxTimeout1 and 2, respectively */
  /* 0x2c RegPreambleMsb - use zero default */
  /* 0x2d */ WriteReg(REG_PREAMBLELSB, 0x4); // Davis has four preamble bytes 0xAAAAAAAA -- use 6 for TX for this setup
  /* 0x2e */ WriteReg(REG_SYNCCONFIG, RF_SYNC_ON | RF_SYNC_FIFOFILL_AUTO | RF_SYNC_SIZE_2 | RF_SYNC_TOL_2);  // Allow a couple erros in the sync word
  /* 0x2f */ WriteReg(REG_SYNCVALUE1, 0xcb); // Davis ISS first sync byte. http://madscientistlabs.blogspot.ca/2012/03/first-you-get-sugar.html
  /* 0x30 */ WriteReg(REG_SYNCVALUE2, 0x89); // Davis ISS second sync byte.
  /* 0x31 - 0x36  REG_SYNCVALUE3 - 8 not used */
  /* 0x37 */ WriteReg(REG_PACKETCONFIG1, RF_PACKET1_FORMAT_FIXED | RF_PACKET1_DCFREE_OFF | RF_PACKET1_CRC_OFF | RF_PACKET1_CRCAUTOCLEAR_OFF | RF_PACKET1_ADRSFILTERING_OFF); // Fixed packet length and we'll check our own CRC
  /* 0x38 */ WriteReg(REG_PAYLOADLENGTH, PAYLOADSIZE); // Davis sends 8 bytes of payload, including CRC that we check manually.
  //* 0x39 */ { REG_NODEADRS, nodeID); // Turned off because we're not using address filtering
  //* 0x3a */ { REG_BROADCASTADRS, RF_BROADCASTADDRESS_VALUE); // Not using this
  /* 0x3b REG_AUTOMODES - Automatic modes are not used in this implementation. */
  /* 0x3c */ WriteReg(REG_FIFOTHRESH, RF_FIFOTHRESH_TXSTART_FIFOTHRESH | (PAYLOADSIZE - 1));
  /* 0x3d */ WriteReg(REG_PACKETCONFIG2, RF_PACKET2_RXRESTARTDELAY_2BITS | RF_PACKET2_AUTORXRESTART_ON | RF_PACKET2_AES_OFF); //RXRESTARTDELAY must match transmitter PA ramp-down time (bitrate dependent)
  /* 0x3e - 0x4d  AES Key not used in this implementation */
  /* 0x6F */ WriteReg(REG_TESTDAGC, RF_DAGC_IMPROVED_LOWBETA0); // // TODO: Should use LOWBETA_ON, but having trouble getting it working
  /* 0x71 */ WriteReg(REG_TESTAFC, 0); // AFC Offset for low mod index systems

  setMode(RF69_MODE_STANDBY);
  // Wait for ModeReady
#ifdef _DEBUG
  Serial.print(fName);
  Serial.print("ModeReady");
#endif
  while ((ReadReg(REG_IRQFLAGS1) & RF_IRQFLAGS1_MODEREADY) == 0x00) {
#ifdef _DEBUG
	Serial.print(".");
#endif
	  ;
  }
#ifdef _DEBUG
   Serial.println();
#endif

#ifdef _DEBUG
   Serial.print(fName);
   Serial.println("interrupt");
#endif
  selfPointer = this;
  userInterrupt = NULL;
  attachInterrupt(0, RFMxx::isr0, RISING);

  SetFrequency(m_freqBandIdx);
  SetDataRate();

  ClearFifo();
#ifdef _DEBUG
  Serial.print(fName);
  Serial.println("leave");
#endif

  // notify wrong radio type detection
  checkRadioType();
}

byte RFMxx::GetChannel() {
	return m_freqBandIdx;
}

void RFMxx::SetChannel(byte channel) {
	SetFrequency(channel);
}

void RFMxx::Receive() {
	if (ReadReg(REG_IRQFLAGS2) & RF_IRQFLAGS2_PAYLOADREADY) {
		RSSI = GetRSSI();
		FEI = word(ReadReg(REG_FEIMSB), ReadReg(REG_FEILSB));
	    setMode(RF69_MODE_STANDBY);
	    select();   // Select RFM69 module, disabling interrupts
	    SPI.transfer(REG_FIFO & 0x7f);

	    for (byte i = 0; i < PAYLOADSIZE; i++)
	    	m_payload[i] = reverseBits(SPI.transfer(0));

	    m_payloadReady = true;
	    if (userInterrupt != NULL)
	    	(*userInterrupt)();
	    unselect();  // Unselect RFM69 module, enabling interrupts
	}
}

void RFMxx::GetPayload(byte *data) {
  m_payloadReady = false;
  for (int i = 0; i < PAYLOADSIZE; i++) {
    data[i] = m_payload[i];
  }
}


void RFMxx::SetDataRate() {
	word r = ((32000000UL + (m_dataRate / 2)) / m_dataRate);
	WriteReg(0x03, r >> 8);
	WriteReg(0x04, r & 0xFF);
}


int RFMxx::GetRSSI(bool forceTrigger) {
  int rssi = 0;
  if (forceTrigger)
  {
    //RSSI trigger not needed if DAGC is in continuous mode
    WriteReg(REG_RSSICONFIG, RF_RSSI_START);
    while ((ReadReg(REG_RSSICONFIG) & RF_RSSI_DONE) == 0x00); // Wait for RSSI_Ready
  }
  rssi = -ReadReg(REG_RSSIVALUE);
  rssi >>= 1;
  return rssi;
}

void RFMxx::EnableReceiver(bool enable){
  setMode((enable ? RF69_MODE_RX : RF69_MODE_STANDBY));
}

void RFMxx::EnableTransmitter(bool enable){
  setMode(enable ? RF69_MODE_TX : RF69_MODE_STANDBY);
}

bool RFMxx::PayloadIsReady() {
  return m_payloadReady;
}


void RFMxx::ClearFifo() {
    WriteReg(REG_IRQFLAGS2, 16);
}

void RFMxx::PowerDown(){
    WriteReg(REG_OPMODE, (ReadReg(REG_OPMODE) & 0xE3) | RF_OPMODE_SLEEP);
}



byte RFMxx::ReadReg(byte addr) {
  select();
  SPI.transfer(addr & 0x7F);
  byte regval = SPI.transfer(0);
  unselect();
  return regval;
}

void RFMxx::WriteReg(byte addr, byte value) {
  select();
  SPI.transfer(addr | 0x80);
  SPI.transfer(value);
  unselect();
}

RFMxx::RadioType RFMxx::GetRadioType() {
  return m_radioType;
}

String RFMxx::GetRadioName() {
  switch (GetRadioType()) {
//  case RFMxx::RFM12B:
//    return String("RFM12");
//    break;
  case RFMxx::RFM69CW:
    return String("RFM69");
    break;
  default:
    return String("None");
  }
}

bool RFMxx::IsConnected() {
  return m_radioType != RFMxx::None;
}

void RFMxx::SetDebugMode(boolean mode) {
  m_debug = mode;
}

unsigned long RFMxx::GetDataRate() {
  return m_dataRate;
}

void RFMxx::GetChannelInfo() {
	Serial.print(" b:");
	Serial.print(m_freqBand);
	Serial.print(" c:");
	Serial.print(m_freqBandIdx);
	Serial.print(" f:");
	Serial.print(GetFrequencyValue());
}

byte RFMxx::GetBand() {
	return m_freqBand;
}

byte RFMxx::GetBandFrequencies() {
	return bandTabLengths[m_freqBand];
}

void RFMxx::SetBand(byte freqBand) {
	m_freqBand = freqBand;
}

unsigned long RFMxx::GetFrequencyValue() {
	byte r07 = ReadReg(0x07), r08 = ReadReg(0x08), r09 = ReadReg(0x09);

	unsigned long cf = ((unsigned long)r07 << 16) + ((unsigned long)r08 << 8) + ((unsigned long)r09);

	return (unsigned long)(((((cf) >> 6) * (32000000L >> 11)) >> 2) / 1000);
}

void RFMxx::SetFrequency(byte freqBandIdx) {
	// manage overflow
	m_freqBandIdx = (freqBandIdx > bandTabLengths[m_freqBand] ? 0 : freqBandIdx);

#ifdef _DEBUG
#define fName "RFMxx::SetFrequency: "
	if (Serial.available()) {
		Serial.print(fName);
		Serial.print(m_freqBand);
		Serial.print(" ");
		Serial.print(m_freqBandIdx);
	}
#endif
	WriteReg(0x07, pgm_read_byte(&bandTab[m_freqBand][m_freqBandIdx][0]));
	WriteReg(0x08, pgm_read_byte(&bandTab[m_freqBand][m_freqBandIdx][1]));
	WriteReg(0x09, pgm_read_byte(&bandTab[m_freqBand][m_freqBandIdx][2]));
#ifdef _DEBUG
	if (Serial.available()) {
	    Serial.print(" -> ");
		Serial.print(GetFrequencyValue());
	    Serial.println(" kHz");
	}
#endif
}

void RFMxx::isr0() {
	selfPointer->Receive();
}

void RFMxx::setUserInterrupt(void (*function)()) {
    userInterrupt = function;
}

bool RFMxx::checkRadioType() {
	bool result = IsConnected();

	if (!result)
		Serial.println("\ncouldn't detect supported radio type!");

	return result;
}

void RFMxx::setMode(byte newMode) {
#ifdef _DEBUG
	 if (Serial.available()) {
	 	Serial.print("RFMxx::setMode: enter ");
	    Serial.println(newMode);
	 }
#endif
    if (newMode == _mode) {
#ifdef _DEBUG
    if (Serial.available())
    	Serial.println("RFMxx::setMode: leave");
#endif
    	return;
    }

    // don't start radio if not supported
    if (!checkRadioType() && !(newMode == RF69_MODE_STANDBY || newMode == RF69_MODE_SLEEP))
  	  return;

    switch (newMode) {
      case RF69_MODE_TX:
        WriteReg(REG_OPMODE, (ReadReg(REG_OPMODE) & 0xE3) | RF_OPMODE_TRANSMITTER);
//        if (_isRFM69HW) setHighPowerRegs(true);
        break;
      case RF69_MODE_RX:
        WriteReg(REG_OPMODE, (ReadReg(REG_OPMODE) & 0xE3) | RF_OPMODE_RECEIVER);
//        if (_isRFM69HW) setHighPowerRegs(false);
        break;
      case RF69_MODE_SYNTH:
        WriteReg(REG_OPMODE, (ReadReg(REG_OPMODE) & 0xE3) | RF_OPMODE_SYNTHESIZER);
        break;
      case RF69_MODE_STANDBY:
        WriteReg(REG_OPMODE, (ReadReg(REG_OPMODE) & 0xE3) | RF_OPMODE_STANDBY);
        break;
      case RF69_MODE_SLEEP:
        WriteReg(REG_OPMODE, (ReadReg(REG_OPMODE) & 0xE3) | RF_OPMODE_SLEEP);
        break;
      default:
    	return;
    }

    // we are using packet mode, so this check is not really needed
    // but waiting for mode ready is necessary when going from sleep because the FIFO may not be immediately available from previous mode
    while (_mode == RF69_MODE_SLEEP && (ReadReg(REG_IRQFLAGS1) & RF_IRQFLAGS1_MODEREADY) == 0x00); // Wait for ModeReady

    _mode = newMode;
#ifdef _DEBUG
    if (Serial.available()) {
        Serial.print("RFMxx::setMode: leave ");
        Serial.println(_mode);
    }
#endif
  }

  // Select the transceiver
  void RFMxx::select() {
    noInterrupts();
    digitalWrite(SPI_CS, LOW);
  }

  // Unselect the transceiver chip
  void RFMxx::unselect() {
    digitalWrite(SPI_CS, HIGH);
    interrupts();
  }

  void RFMxx::setHighPowerRegs(bool onOff) {
    WriteReg(REG_TESTPA1, onOff ? 0x5D : 0x55);
    WriteReg(REG_TESTPA2, onOff ? 0x7C : 0x70);
  }

  byte RFMxx::reverseBits(byte b) {
    b = ((b & 0b11110000) >>4 ) | ((b & 0b00001111) << 4);
    b = ((b & 0b11001100) >>2 ) | ((b & 0b00110011) << 2);
    b = ((b & 0b10101010) >>1 ) | ((b & 0b01010101) << 1);

    return(b);
  }

  unsigned int RFMxx::GetCRC16CCITT(volatile byte *buf, byte len, unsigned int initCrc)
  {
    unsigned int crc = initCrc;
    while( len-- ) {
      int i;
      crc ^= *(char *)buf++ << 8;
      for( i = 0; i < 8; ++i ) {
        if( crc & 0x8000 )
          crc = (crc << 1) ^ 0x1021;
        else
          crc = crc << 1;
      }
    }
    return crc;
  }
