/*
  Copyright (C) 2009, 2010 and 2012 Chris Vine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#ifndef MOUNTER_H
#define MOUNTER_H

#include "prog_defs.h"

#include <string>
#include <map>
#include <utility>

#include <gio/gio.h>
#include <gio/gunixmounts.h>

#define UDISKS_API_IS_SUBJECT_TO_CHANGE 1
#include <udisks/udisks.h>

#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/emitter.h>
#include <c++-gtk-utils/gobj_handle.h>

namespace MounterCB {
  extern "C" void mounter_mounts_changed(GUnixMountMonitor*, void*);
}

class Mounter {
public:
  enum Direction {do_mount, do_unmount};
private:

  std::map<std::string, std::string> mounted_device_map;
  GobjHandle<GUnixMountMonitor> monitor;

  void fill_cache();
  static void raw_mount(const std::string&, Direction, Callback::Callback*);
  static void raw_mount_thread(char*, Direction, Callback::Callback*);
  static void get_udisks_object(const std::string&, Direction, Callback::Callback*);
  static void udisks_mount(const GobjHandle<UDisksObject>&, Direction, Callback::Callback*);

  // Objects of this class are not for copying
  Mounter(const Mounter&);
  Mounter& operator=(const Mounter&);
public:
  friend void MounterCB::mounter_mounts_changed(GUnixMountMonitor*, void*);

  // this Emitter will be emitted by Mounter when it detects that
  // there has been a change of mounted devices. If a call to
  // Mounter::mount() or Mounter::unmount() has failed, then the
  // on_fail callback passed to those methods will be executed instead.
  Emitter monitor_cb;

  // returns true if device mounted
  bool is_mounted(const std::string& device) const {
    return mounted_device_map.find(device) != mounted_device_map.end();
  }

  // returns mount point if device mounted, otherwise empty string
  std::string get_mount_point(const std::string& device) const;

  // Mounter::mount() and Mounter::unmount() will take ownership of the on_fail
  // callback argument, which will be executed if the mount or unmount fails.
  // Successful mounting/unmounting will be detected via the monitor callback.
  // If no fail callback is wanted, pass NULL to on_fail callback argument
  void mount(const std::string& device, Callback::Callback* on_fail);
  void unmount(const std::string& device, Callback::Callback* on_fail);

  Mounter();
};

#endif
