/* $Id$ */

/*
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  Preferences dialog
 *
 */

/* Radar map on by default. */
#define RADARMAP

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <locale.h>

#include <mate-panel-applet.h>
#include <gio/gio.h>

#define MATEWEATHER_I_KNOW_THIS_IS_UNSTABLE

#include <libmateweather/mateweather-xml.h>
#include "mateweather.h"
#include "mateweather-pref.h"
#include "mateweather-applet.h"
#include "mateweather-dialog.h"

#define NEVER_SENSITIVE "never_sensitive"

struct _MateWeatherPrefPrivate {
	GtkWidget* notebook;

	GtkWidget* basic_temp_combo;
	GtkWidget* basic_speed_combo;
	GtkWidget* basic_dist_combo;
	GtkWidget* basic_pres_combo;
	GtkWidget* find_entry;
	GtkWidget* find_next_btn;

#ifdef RADARMAP
	GtkWidget* basic_radar_btn;
	GtkWidget* basic_radar_url_btn;
	GtkWidget* basic_radar_url_hbox;
	GtkWidget* basic_radar_url_entry;
#endif /* RADARMAP */

#ifdef HAVE_LIBNOTIFY
	GtkWidget* basic_show_notifications_btn;
#endif /* HAVE_LIBNOTIFY */

	GtkWidget* basic_update_spin;
	GtkWidget* basic_update_btn;
	GtkWidget* tree;

	GtkTreeModel* model;

	MateWeatherApplet* applet;
};

enum {
	PROP_0,
	PROP_MATEWEATHER_APPLET,
};

G_DEFINE_TYPE_WITH_PRIVATE (MateWeatherPref, mateweather_pref, GTK_TYPE_DIALOG);

/* set sensitive and setup NEVER_SENSITIVE appropriately */
static void hard_set_sensitive(GtkWidget* w, gboolean sensitivity)
{
	gtk_widget_set_sensitive(w, sensitivity);
	g_object_set_data(G_OBJECT(w), NEVER_SENSITIVE, GINT_TO_POINTER(!sensitivity));
}

/* set sensitive, but always insensitive if NEVER_SENSITIVE is set */
static void soft_set_sensitive(GtkWidget* w, gboolean sensitivity)
{
	if (g_object_get_data(G_OBJECT(w), NEVER_SENSITIVE))
	{
		gtk_widget_set_sensitive(w, FALSE);
	}
	else
	{
		gtk_widget_set_sensitive(w, sensitivity);
	}
}

/* sets up ATK Relation between the widgets */
static void add_atk_relation(GtkWidget* widget1, GtkWidget* widget2, AtkRelationType type)
{
	AtkObject* atk_obj1;
	AtkObject* atk_obj2;
	AtkRelationSet* relation_set;
	AtkRelation* relation;

	atk_obj1 = gtk_widget_get_accessible(widget1);

	if (!GTK_IS_ACCESSIBLE(atk_obj1))
	{
		return;
	}

	atk_obj2 = gtk_widget_get_accessible(widget2);

	relation_set = atk_object_ref_relation_set(atk_obj1);
	relation = atk_relation_new(&atk_obj2, 1, type);
	atk_relation_set_add(relation_set, relation);
	g_object_unref(G_OBJECT(relation));
}

/* sets accessible name and description */
void set_access_namedesc(GtkWidget* widget, const gchar* name, const gchar* desc)
{
	AtkObject* obj = gtk_widget_get_accessible(widget);

	if (!GTK_IS_ACCESSIBLE(obj))
	{
		return;
	}

	if (desc)
	{
		atk_object_set_description(obj, desc);
	}

	if (name)
	{
		atk_object_set_name(obj, name);
	}
}

/* sets accessible name, description, CONTROLLED_BY
 * and CONTROLLER_FOR relations for the components
 * in mateweather preference dialog.
 */
static void mateweather_pref_set_accessibility(MateWeatherPref* pref)
{
    /* Relation between components in General page */
    add_atk_relation(pref->priv->basic_update_btn, pref->priv->basic_update_spin, ATK_RELATION_CONTROLLER_FOR);
    add_atk_relation(pref->priv->basic_radar_btn, pref->priv->basic_radar_url_btn, ATK_RELATION_CONTROLLER_FOR);
    add_atk_relation(pref->priv->basic_radar_btn, pref->priv->basic_radar_url_entry, ATK_RELATION_CONTROLLER_FOR);

    add_atk_relation(pref->priv->basic_update_spin, pref->priv->basic_update_btn, ATK_RELATION_CONTROLLED_BY);
    add_atk_relation(pref->priv->basic_radar_url_btn, pref->priv->basic_radar_btn, ATK_RELATION_CONTROLLED_BY);
    add_atk_relation(pref->priv->basic_radar_url_entry, pref->priv->basic_radar_btn, ATK_RELATION_CONTROLLED_BY);

    /* Accessible Name and Description for the components in Preference Dialog */
    set_access_namedesc(pref->priv->tree, _("Location view"), _("Select Location from the list"));
    set_access_namedesc(pref->priv->basic_update_spin, _("Update spin button"), _("Spinbutton for updating"));
    set_access_namedesc(pref->priv->basic_radar_url_entry, _("Address Entry"), _("Enter the URL"));
}

/* Update pref dialog from mateweather_pref */
static gboolean update_dialog(MateWeatherPref* pref)
{
    MateWeatherApplet* gw_applet = pref->priv->applet;

    g_return_val_if_fail(gw_applet->mateweather_pref.location != NULL, FALSE);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(pref->priv->basic_update_spin), gw_applet->mateweather_pref.update_interval / 60);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pref->priv->basic_update_btn), gw_applet->mateweather_pref.update_enabled);
    soft_set_sensitive(pref->priv->basic_update_spin, gw_applet->mateweather_pref.update_enabled);

    gtk_combo_box_set_active (GTK_COMBO_BOX (pref->priv->basic_temp_combo),
                              (gint) gw_applet->mateweather_pref.temperature_unit - 2);

    gtk_combo_box_set_active (GTK_COMBO_BOX (pref->priv->basic_speed_combo),
                              (gint) gw_applet->mateweather_pref.speed_unit - 2);

    gtk_combo_box_set_active (GTK_COMBO_BOX (pref->priv->basic_pres_combo),
                              (gint) gw_applet->mateweather_pref.pressure_unit - 2);

    gtk_combo_box_set_active (GTK_COMBO_BOX (pref->priv->basic_dist_combo),
                              (gint) gw_applet->mateweather_pref.distance_unit - 2);

	#ifdef RADARMAP
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pref->priv->basic_radar_btn), gw_applet->mateweather_pref.radar_enabled);

		soft_set_sensitive(pref->priv->basic_radar_url_btn, gw_applet->mateweather_pref.radar_enabled);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pref->priv->basic_radar_url_btn), gw_applet->mateweather_pref.use_custom_radar_url);
		soft_set_sensitive(pref->priv->basic_radar_url_hbox, gw_applet->mateweather_pref.radar_enabled);

		if (gw_applet->mateweather_pref.radar)
		{
			gtk_entry_set_text(GTK_ENTRY(pref->priv->basic_radar_url_entry), gw_applet->mateweather_pref.radar);
		}

	#endif /* RADARMAP */

	#ifdef HAVE_LIBNOTIFY
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pref->priv->basic_show_notifications_btn), gw_applet->mateweather_pref.show_notifications);
	#endif
    return TRUE;
}

static void
on_row_selected (GtkTreeSelection *selection,
                 MateWeatherPref  *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	GtkTreeModel* model;
	WeatherLocation* loc = NULL;
	GtkTreeIter iter;

	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
		return;

	gtk_tree_model_get(model, &iter, MATEWEATHER_XML_COL_POINTER, &loc, -1);

	if (!loc)
	{
		return;
	}

	g_settings_set_string (gw_applet->settings, "location1", loc->code);
	g_settings_set_string (gw_applet->settings, "location2", loc->zone);
	g_settings_set_string (gw_applet->settings, "location3", loc->radar);
	g_settings_set_string (gw_applet->settings, "location4", loc->name);
	g_settings_set_string (gw_applet->settings, "coordinates", loc->coordinates);

	if (gw_applet->mateweather_pref.location)
	{
		weather_location_free(gw_applet->mateweather_pref.location);
	}

	gw_applet->mateweather_pref.location =
		weather_location_new (loc->name, loc->code, loc->zone, loc->radar, loc->coordinates,
			NULL, NULL);

	mateweather_update(gw_applet);
}

static gboolean compare_location(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer user_data)
{
    MateWeatherPref* pref = user_data;
    WeatherLocation* loc;
    GtkTreeView* view;

    gtk_tree_model_get(model, iter, MATEWEATHER_XML_COL_POINTER, &loc, -1);

    if (!loc)
    {
		return FALSE;
	}

    if (!weather_location_equal(loc, pref->priv->applet->mateweather_pref.location))
    {
		return FALSE;
	}

    view = GTK_TREE_VIEW(pref->priv->tree);
    gtk_tree_view_expand_to_path(view, path);
    gtk_tree_view_set_cursor(view, path, NULL, FALSE);
    gtk_tree_view_scroll_to_cell(view, path, NULL, TRUE, 0.5, 0.5);

    return TRUE;
}

static void load_locations(MateWeatherPref* pref)
{
	GtkTreeView* tree = GTK_TREE_VIEW(pref->priv->tree);
	GtkTreeViewColumn* column;
	GtkCellRenderer* cell_renderer;

	/* Add a column for the locations */
	cell_renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes("not used", cell_renderer, "text", MATEWEATHER_XML_COL_LOC, NULL);
	gtk_tree_view_append_column(tree, column);
	gtk_tree_view_set_expander_column(GTK_TREE_VIEW(tree), column);

	/* load locations from xml file */
	pref->priv->model = mateweather_xml_load_locations();

	if (!pref->priv->model)
	{
		GtkWidget* d = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, _("Failed to load the Locations XML database.  Please report this as a bug."));
		gtk_dialog_run(GTK_DIALOG(d));
		gtk_widget_destroy(d);
	}

	gtk_tree_view_set_model (tree, pref->priv->model);

	if (pref->priv->applet->mateweather_pref.location)
	{
		/* Select the current (default) location */
		gtk_tree_model_foreach(GTK_TREE_MODEL(pref->priv->model), compare_location, pref);
	}
}

static void
on_show_notifications_toggled (GtkToggleButton *button,
                               MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;

	gboolean toggled = gtk_toggle_button_get_active(button);

	if (toggled != gw_applet->mateweather_pref.show_notifications)
	{
		/* sync with mateweather_pref struct */
		gw_applet->mateweather_pref.show_notifications = toggled;

		/* sync with gsettings */
		g_settings_set_boolean (gw_applet->settings, "show-notifications", toggled);
	}
}

static void
on_auto_update_toggled (GtkToggleButton *button,
                        MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	gboolean toggled;
	gint nxtSunEvent;

	toggled = gtk_toggle_button_get_active(button);
	gw_applet->mateweather_pref.update_enabled = toggled;
	soft_set_sensitive(pref->priv->basic_update_spin, toggled);
	g_settings_set_boolean (gw_applet->settings, "auto-update", toggled);

	if (gw_applet->timeout_tag > 0)
	{
		g_source_remove(gw_applet->timeout_tag);
	}

	if (gw_applet->suncalc_timeout_tag > 0)
	{
		g_source_remove(gw_applet->suncalc_timeout_tag);
	}

	if (gw_applet->mateweather_pref.update_enabled)
	{
		gw_applet->timeout_tag
			= g_timeout_add_seconds ((guint) gw_applet->mateweather_pref.update_interval,
		                                 timeout_cb, gw_applet);

		nxtSunEvent = weather_info_next_sun_event (gw_applet->mateweather_info);
		if (nxtSunEvent >= 0)
		{
			gw_applet->suncalc_timeout_tag
				= g_timeout_add_seconds ((guint) nxtSunEvent,
				                         suncalc_timeout_cb, gw_applet);
		}
	}
}

static void
on_temp_combo_changed (GtkComboBox     *combo,
                       MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	TempUnit new_unit, old_unit;

	g_return_if_fail(gw_applet != NULL);

	new_unit = gtk_combo_box_get_active(combo) + 2;

	old_unit = gw_applet->mateweather_pref.temperature_unit;

	if (new_unit == old_unit)
	{
		return;
	}

	gw_applet->mateweather_pref.temperature_unit = new_unit;

	g_settings_set_enum (gw_applet->settings, "temperature-unit", new_unit);

	gtk_label_set_text(GTK_LABEL(gw_applet->label), weather_info_get_temp_summary(gw_applet->mateweather_info));

	if (gw_applet->details_dialog)
	{
		mateweather_dialog_update(MATEWEATHER_DIALOG(gw_applet->details_dialog));
	}
}

static void
on_speed_combo_changed (GtkComboBox     *combo,
                        MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	SpeedUnit new_unit, old_unit;

	g_return_if_fail(gw_applet != NULL);

	new_unit = gtk_combo_box_get_active(combo) + 2;

	old_unit = gw_applet->mateweather_pref.speed_unit;

	if (new_unit == old_unit)
	{
		return;
	}

	gw_applet->mateweather_pref.speed_unit = new_unit;

	g_settings_set_enum (gw_applet->settings, "speed-unit", new_unit);

	if (gw_applet->details_dialog)
	{
		mateweather_dialog_update(MATEWEATHER_DIALOG(gw_applet->details_dialog));
	}
}

static void
on_pres_combo_changed (GtkComboBox     *combo,
                       MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	PressureUnit new_unit, old_unit;

	g_return_if_fail(gw_applet != NULL);

	new_unit = gtk_combo_box_get_active(combo) + 2;

	old_unit = gw_applet->mateweather_pref.pressure_unit;

	if (new_unit == old_unit)
	{
		return;
	}

	gw_applet->mateweather_pref.pressure_unit = new_unit;

	g_settings_set_enum (gw_applet->settings, "pressure-unit", new_unit);

	if (gw_applet->details_dialog)
	{
		mateweather_dialog_update(MATEWEATHER_DIALOG(gw_applet->details_dialog));
	}
}

static void
on_dist_combo_changed (GtkComboBox     *combo,
                       MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	DistanceUnit new_unit, old_unit;

	g_return_if_fail(gw_applet != NULL);

	new_unit = gtk_combo_box_get_active(combo) + 2;

	old_unit = gw_applet->mateweather_pref.distance_unit;

	if (new_unit == old_unit)
	{
		return;
	}

	gw_applet->mateweather_pref.distance_unit = new_unit;

	g_settings_set_enum (gw_applet->settings, "distance-unit", new_unit);

	if (gw_applet->details_dialog)
	{
		mateweather_dialog_update(MATEWEATHER_DIALOG(gw_applet->details_dialog));
	}
}

static void
on_radar_toggled (GtkToggleButton *button,
                  MateWeatherPref *pref)
{
    MateWeatherApplet* gw_applet = pref->priv->applet;
    gboolean toggled;

    toggled = gtk_toggle_button_get_active(button);
    gw_applet->mateweather_pref.radar_enabled = toggled;
    g_settings_set_boolean (gw_applet->settings, "enable-radar-map", toggled);
    soft_set_sensitive(pref->priv->basic_radar_url_btn, toggled);

    if (toggled == FALSE || gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (pref->priv->basic_radar_url_btn)) == TRUE)
    {
		soft_set_sensitive (pref->priv->basic_radar_url_hbox, toggled);
	}
}

static void
on_use_radar_url_toggled (GtkToggleButton *button,
                          MateWeatherPref *pref)
{
    MateWeatherApplet* gw_applet = pref->priv->applet;
    gboolean toggled;

    toggled = gtk_toggle_button_get_active(button);
    gw_applet->mateweather_pref.use_custom_radar_url = toggled;
    g_settings_set_boolean (gw_applet->settings, "use-custom-radar-url", toggled);
    soft_set_sensitive(pref->priv->basic_radar_url_hbox, toggled);
}

static gboolean
on_radar_url_changed  (GtkWidget       *widget,
                       GdkEventFocus   *event,
                       MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;
	gchar *text;

	text = gtk_editable_get_chars(GTK_EDITABLE(widget), 0, -1);

	if (gw_applet->mateweather_pref.radar)
	{
		g_free(gw_applet->mateweather_pref.radar);
	}

	if (text)
	{
		gw_applet->mateweather_pref.radar = g_strdup(text);
		g_free (text);
	}
	else
	{
		gw_applet->mateweather_pref.radar = NULL;
	}

	g_settings_set_string (gw_applet->settings, "radar", gw_applet->mateweather_pref.radar);

	return FALSE;
}

static void
on_update_interval_changed (GtkSpinButton   *button,
                            MateWeatherPref *pref)
{
	MateWeatherApplet* gw_applet = pref->priv->applet;

	gw_applet->mateweather_pref.update_interval = gtk_spin_button_get_value_as_int(button)*60;
	g_settings_set_int (gw_applet->settings, "auto-update-interval", gw_applet->mateweather_pref.update_interval);

	if (gw_applet->timeout_tag > 0)
	{
		g_source_remove(gw_applet->timeout_tag);
	}

	if (gw_applet->mateweather_pref.update_enabled)
	{
		gw_applet->timeout_tag
			= g_timeout_add_seconds ((guint) gw_applet->mateweather_pref.update_interval,
			                         timeout_cb, gw_applet);
	}
}

static gboolean free_data(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer data)
{
	WeatherLocation* location;

	gtk_tree_model_get(model, iter, MATEWEATHER_XML_COL_POINTER, &location, -1);

	if (!location)
	{
	   return FALSE;
	}

	weather_location_free(location);

	return FALSE;
}

static GtkWidget* create_hig_category(GtkWidget* main_box, gchar* title)
{
	GtkWidget* vbox;
	GtkWidget* vbox2;
	GtkWidget* hbox;
	GtkWidget*label;
	gchar* tmp;

	vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 6);
	gtk_box_pack_start (GTK_BOX (main_box), vbox, FALSE, FALSE, 0);

	tmp = g_strdup_printf ("<b>%s</b>", title);
	label = gtk_label_new (NULL);
	gtk_label_set_xalign (GTK_LABEL (label), 0.0);
	gtk_label_set_markup (GTK_LABEL (label), tmp);
	g_free (tmp);
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);

	hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	vbox2 = gtk_box_new (GTK_ORIENTATION_VERTICAL, 6);
	gtk_box_pack_start (GTK_BOX (hbox), vbox2, TRUE, TRUE, 0);

	return vbox2;
}

static gboolean find_location(GtkTreeModel* model, GtkTreeIter* iter, const gchar* location, gboolean go_parent)
{
	GtkTreeIter iter_child;
	GtkTreeIter iter_parent;
	gchar* aux_loc;
	gboolean valid;
	size_t len;

	if ((len = strlen (location)) == 0)
		return FALSE;

	do {

		gtk_tree_model_get (model, iter, MATEWEATHER_XML_COL_LOC, &aux_loc, -1);

		if (g_ascii_strncasecmp (aux_loc, location, len) == 0)
		{
			g_free (aux_loc);
			return TRUE;
		}

		if (gtk_tree_model_iter_has_child(model, iter))
		{
			gtk_tree_model_iter_nth_child(model, &iter_child, iter, 0);

			if (find_location (model, &iter_child, location, FALSE))
			{
				/* Manual copying of the iter */
				iter->stamp = iter_child.stamp;
				iter->user_data = iter_child.user_data;
				iter->user_data2 = iter_child.user_data2;
				iter->user_data3 = iter_child.user_data3;

				g_free (aux_loc);

				return TRUE;
			}
		}

		g_free (aux_loc);

		valid = gtk_tree_model_iter_next(model, iter);

	} while (valid);

	if (go_parent)
	{
		iter_parent = *iter;

		while (gtk_tree_model_iter_parent (model, iter, &iter_parent))
		{
			if (gtk_tree_model_iter_next (model, iter))
			{
				return find_location (model, iter, location, TRUE);
			}

			iter_parent = *iter;
		}
	}

	return FALSE;
}

static void
on_find_next_clicked (GtkButton       *button,
                      MateWeatherPref *pref)
{
	GtkTreeView *tree;
	GtkTreeModel *model;
	GtkEntry *entry;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeIter iter_parent;
	GtkTreePath *path;
	const gchar *location;

	tree = GTK_TREE_VIEW (pref->priv->tree);
	model = gtk_tree_view_get_model (tree);
	entry = GTK_ENTRY (pref->priv->find_entry);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));

	if (gtk_tree_selection_count_selected_rows (selection) >= 1)
	{
		gtk_tree_selection_get_selected (selection, &model, &iter);
		/* Select next or select parent */
		if (!gtk_tree_model_iter_next (model, &iter))
		{
			iter_parent = iter;

			if (!gtk_tree_model_iter_parent (model, &iter, &iter_parent) || !gtk_tree_model_iter_next (model, &iter))
			{
				gtk_tree_model_get_iter_first (model, &iter);
			}
		}

	}
	else
	{
		gtk_tree_model_get_iter_first (model, &iter);
	}

	location = gtk_entry_get_text (entry);

	if (find_location (model, &iter, location, TRUE))
	{
		gtk_widget_set_sensitive (pref->priv->find_next_btn, TRUE);

		path = gtk_tree_model_get_path (model, &iter);
		gtk_tree_view_expand_to_path (tree, path);
		gtk_tree_selection_select_path (selection, path);
		gtk_tree_view_scroll_to_cell (tree, path, NULL, TRUE, 0.5, 0);

		gtk_tree_path_free (path);
	}
	else
	{
		gtk_widget_set_sensitive (pref->priv->find_next_btn, FALSE);
	}
}

static void
find_entry_changed (GtkEditable     *entry,
                    MateWeatherPref *pref)
{
	GtkTreeView *tree;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreePath *path;
	const gchar *location;

	tree = GTK_TREE_VIEW (pref->priv->tree);
	model = gtk_tree_view_get_model (tree);

	g_return_if_fail (model != NULL);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
	gtk_tree_model_get_iter_first (model, &iter);

	location = gtk_entry_get_text (GTK_ENTRY (entry));

	if (find_location (model, &iter, location, TRUE))
	{
		gtk_widget_set_sensitive (pref->priv->find_next_btn, TRUE);

		path = gtk_tree_model_get_path (model, &iter);
		gtk_tree_view_expand_to_path (tree, path);
		gtk_tree_selection_select_iter (selection, &iter);
		gtk_tree_view_scroll_to_cell (tree, path, NULL, TRUE, 0.5, 0);

		gtk_tree_path_free (path);
	}
	else
	{
		gtk_widget_set_sensitive (pref->priv->find_next_btn, FALSE);
	}
}

static void help_cb(GtkDialog* dialog, MateWeatherPref* pref)
{
	gint current_page;
	gchar *uri;
	GError* error = NULL;

	current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (pref->priv->notebook));
	uri = g_strdup_printf ("help:mateweather/mateweather-prefs#mateweather-%s", (current_page == 0) ? "metric" : "change-location");
	gtk_show_uri_on_window (GTK_WINDOW (dialog),
	                        uri,
	                        gtk_get_current_event_time (),
	                        &error);
	g_free (uri);

	if (error)
	{
		GtkWidget* error_dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, _("There was an error displaying help: %s"), error->message);
		g_signal_connect (error_dialog, "response",
		                  G_CALLBACK (gtk_widget_destroy),
		                  NULL);
		gtk_window_set_resizable (GTK_WINDOW (error_dialog), FALSE);
		gtk_window_set_screen (GTK_WINDOW (error_dialog), gtk_widget_get_screen (GTK_WIDGET (dialog)));
		gtk_widget_show (error_dialog);
		g_error_free (error);
		error = NULL;
	}
}

static void
on_response (GtkDialog       *dialog,
             gint             id,
             MateWeatherPref *pref)
{
	if (id == GTK_RESPONSE_HELP)
	{
		help_cb(dialog, pref);
	}
	else
	{
		gtk_widget_destroy(GTK_WIDGET(dialog));
	}
}

static void
mateweather_pref_create (MateWeatherPref* pref)
{
	GtkWidget* pref_vbox;
	#ifdef RADARMAP
		GtkWidget* radar_toggle_hbox;
	#endif /* RADARMAP */
	GtkWidget* pref_basic_update_lbl;
	GtkWidget* pref_basic_update_hbox;
	GtkAdjustment* pref_basic_update_spin_adj;
	GtkWidget* pref_basic_update_sec_lbl;
	GtkWidget* pref_basic_note_lbl;
	GtkWidget* pref_loc_hbox;
	GtkWidget* pref_loc_note_lbl;
	GtkWidget* scrolled_window;
	GtkWidget* label;
	GtkWidget* value_hbox;
	GtkWidget* tree_label;
	GtkTreeSelection *selection;
	GtkWidget* pref_basic_vbox;
	GtkWidget* vbox;
	GtkWidget* frame;
	GtkWidget* temp_label;
	GtkWidget* temp_combo;
	GtkWidget* speed_label;
	GtkWidget* speed_combo;
	GtkWidget* pres_label;
	GtkWidget* pres_combo;
	GtkWidget* dist_label;
	GtkWidget* dist_combo;
	GtkWidget* unit_grid;
	GtkWidget* pref_find_label;
	GtkWidget* pref_find_hbox;
	GtkWidget* image;

	g_object_set (pref, "destroy-with-parent", TRUE, NULL);
	gtk_window_set_title (GTK_WINDOW (pref), _("Weather Preferences"));
	gtk_dialog_add_buttons (GTK_DIALOG (pref), "gtk-close", GTK_RESPONSE_CLOSE, "gtk-help", GTK_RESPONSE_HELP, NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (pref), GTK_RESPONSE_CLOSE);
	gtk_container_set_border_width (GTK_CONTAINER (pref), 5);
	gtk_window_set_resizable (GTK_WINDOW (pref), TRUE);
	gtk_window_set_screen (GTK_WINDOW (pref), gtk_widget_get_screen (GTK_WIDGET (pref->priv->applet->applet)));

	pref_vbox = gtk_dialog_get_content_area (GTK_DIALOG (pref));
	gtk_box_set_spacing (GTK_BOX (pref_vbox), 2);
	gtk_widget_show (pref_vbox);

	pref->priv->notebook = gtk_notebook_new ();
	gtk_container_set_border_width (GTK_CONTAINER (pref->priv->notebook), 5);
	gtk_widget_show (pref->priv->notebook);
	gtk_box_pack_start (GTK_BOX (pref_vbox), pref->priv->notebook, TRUE, TRUE, 0);

  /*
   * General settings page.
   */

	pref_basic_vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 18);
	gtk_container_set_border_width (GTK_CONTAINER (pref_basic_vbox), 12);
	gtk_container_add (GTK_CONTAINER (pref->priv->notebook), pref_basic_vbox);

	pref->priv->basic_update_btn = gtk_check_button_new_with_mnemonic (_("_Automatically update every:"));
	gtk_widget_set_halign (pref->priv->basic_update_btn, GTK_ALIGN_START);
	gtk_widget_set_vexpand (pref->priv->basic_update_btn, TRUE);
	gtk_widget_show (pref->priv->basic_update_btn);

	if (!g_settings_is_writable (pref->priv->applet->settings, "auto-update"))
	{
		hard_set_sensitive (pref->priv->basic_update_btn, FALSE);
	}

	/*
	 * Units settings page.
	 */

	/* Temperature Unit */
	temp_label = gtk_label_new_with_mnemonic (_("_Temperature unit:"));
	gtk_label_set_use_markup (GTK_LABEL (temp_label), TRUE);
	gtk_label_set_justify (GTK_LABEL (temp_label), GTK_JUSTIFY_LEFT);
	gtk_label_set_xalign (GTK_LABEL (temp_label), 0.0);
	gtk_widget_show (temp_label);

	temp_combo = gtk_combo_box_text_new ();
	pref->priv->basic_temp_combo = temp_combo;
	gtk_label_set_mnemonic_widget (GTK_LABEL (temp_label), temp_combo);
	//gtk_combo_box_append_text (GTK_COMBO_BOX (temp_combo), _("Default"));
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (temp_combo), _("Kelvin"));
	/* TRANSLATORS: Celsius is sometimes referred Centigrade */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (temp_combo), _("Celsius"));
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (temp_combo), _("Fahrenheit"));
	gtk_widget_show (temp_combo);

	if ( ! g_settings_is_writable (pref->priv->applet->settings, "temperature-unit"))
	{
		hard_set_sensitive (pref->priv->basic_temp_combo, FALSE);
	}

	/* Speed Unit */
	speed_label = gtk_label_new_with_mnemonic (_("_Wind speed unit:"));
	gtk_label_set_use_markup (GTK_LABEL (speed_label), TRUE);
	gtk_label_set_justify (GTK_LABEL (speed_label), GTK_JUSTIFY_LEFT);
	gtk_label_set_xalign (GTK_LABEL (speed_label), 0.0);
	gtk_widget_show (speed_label);

	speed_combo = gtk_combo_box_text_new ();
	pref->priv->basic_speed_combo = speed_combo;
	gtk_label_set_mnemonic_widget (GTK_LABEL (speed_label), speed_combo);
	//gtk_combo_box_append_text (GTK_COMBO_BOX (speed_combo), _("Default"));
	/* TRANSLATOR: The wind speed unit "meters per second" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (speed_combo), _("m/s"));
	/* TRANSLATOR: The wind speed unit "kilometers per hour" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (speed_combo), _("km/h"));
	/* TRANSLATOR: The wind speed unit "miles per hour" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (speed_combo), _("mph"));
	/* TRANSLATOR: The wind speed unit "knots" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (speed_combo), _("knots"));
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (speed_combo), _("Beaufort scale"));
	gtk_widget_show (speed_combo);

	if (!g_settings_is_writable (pref->priv->applet->settings, "speed-unit"))
	{
		hard_set_sensitive (pref->priv->basic_speed_combo, FALSE);
	}

	/* Pressure Unit */
	pres_label = gtk_label_new_with_mnemonic (_("_Pressure unit:"));
	gtk_label_set_use_markup (GTK_LABEL (pres_label), TRUE);
	gtk_label_set_justify (GTK_LABEL (pres_label), GTK_JUSTIFY_LEFT);
	gtk_label_set_xalign (GTK_LABEL (pres_label), 0.0);
	gtk_widget_show (pres_label);

	pres_combo = gtk_combo_box_text_new ();
	pref->priv->basic_pres_combo = pres_combo;
	gtk_label_set_mnemonic_widget (GTK_LABEL (pres_label), pres_combo);
	//gtk_combo_box_append_text (GTK_COMBO_BOX (pres_combo), _("Default"));
	/* TRANSLATOR: The pressure unit "kiloPascals" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (pres_combo), _("kPa"));
	/* TRANSLATOR: The pressure unit "hectoPascals" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (pres_combo), _("hPa"));
	/* TRANSLATOR: The pressure unit "millibars" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (pres_combo), _("mb"));
	/* TRANSLATOR: The pressure unit "millibars of mercury" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (pres_combo), _("mmHg"));
	/* TRANSLATOR: The pressure unit "inches of mercury" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (pres_combo), _("inHg"));
	/* TRANSLATOR: The pressure unit "atmospheres" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (pres_combo), _("atm"));
	gtk_widget_show (pres_combo);

	if (!g_settings_is_writable(pref->priv->applet->settings, "pressure-unit"))
	{
		hard_set_sensitive(pref->priv->basic_pres_combo, FALSE);
	}

	/* Distance Unit */
	dist_label = gtk_label_new_with_mnemonic (_("_Visibility unit:"));
	gtk_label_set_use_markup (GTK_LABEL (dist_label), TRUE);
	gtk_label_set_justify (GTK_LABEL (dist_label), GTK_JUSTIFY_LEFT);
	gtk_label_set_xalign (GTK_LABEL (dist_label), 0.0);
	gtk_widget_show (dist_label);

	dist_combo = gtk_combo_box_text_new ();
	pref->priv->basic_dist_combo = dist_combo;
	gtk_label_set_mnemonic_widget (GTK_LABEL (dist_label), dist_combo);
	//gtk_combo_box_append_text (GTK_COMBO_BOX (dist_combo), _("Default"));
	/* TRANSLATOR: The distance unit "meters" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (dist_combo), _("meters"));
	/* TRANSLATOR: The distance unit "kilometers" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (dist_combo), _("km"));
	/* TRANSLATOR: The distance unit "miles" */
	gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (dist_combo), _("miles"));
	gtk_widget_show (dist_combo);

	if ( ! g_settings_is_writable (pref->priv->applet->settings, "distance-unit"))
		hard_set_sensitive (pref->priv->basic_dist_combo, FALSE);

	unit_grid = gtk_grid_new ();
	gtk_grid_set_row_spacing(GTK_GRID(unit_grid), 6);
	gtk_grid_set_column_spacing(GTK_GRID(unit_grid), 12);
	gtk_widget_set_halign (temp_label, GTK_ALIGN_START);
	gtk_grid_attach(GTK_GRID(unit_grid), temp_label, 0, 0, 1, 1);
	gtk_grid_attach(GTK_GRID(unit_grid), temp_combo,  1, 0, 1, 1);
	gtk_widget_set_halign (speed_label, GTK_ALIGN_START);
	gtk_grid_attach(GTK_GRID(unit_grid), speed_label, 0, 1, 1, 1);
	gtk_grid_attach(GTK_GRID(unit_grid), speed_combo, 1, 1, 1, 1);
	gtk_widget_set_halign (pres_label, GTK_ALIGN_START);
	gtk_grid_attach(GTK_GRID(unit_grid), pres_label, 0, 2, 1, 1);
	gtk_grid_attach(GTK_GRID(unit_grid), pres_combo,  1, 2, 1, 1);
	gtk_widget_set_halign (dist_label, GTK_ALIGN_START);
	gtk_grid_attach(GTK_GRID(unit_grid), dist_label, 0, 3, 1, 1);
	gtk_grid_attach(GTK_GRID(unit_grid), dist_combo,  1, 3, 1, 1);
	gtk_widget_show(unit_grid);

	#ifdef RADARMAP
		pref->priv->basic_radar_btn = gtk_check_button_new_with_mnemonic (_("Enable _radar map"));
		gtk_widget_show (pref->priv->basic_radar_btn);

		if (!g_settings_is_writable (pref->priv->applet->settings, "enable-radar-map"))
		{
			hard_set_sensitive(pref->priv->basic_radar_btn, FALSE);
		}

		radar_toggle_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 12);
		gtk_widget_show(radar_toggle_hbox);

		label = gtk_label_new ("    ");
		gtk_widget_show (label);
		gtk_box_pack_start (GTK_BOX (radar_toggle_hbox), label, FALSE, FALSE, 0);

		pref->priv->basic_radar_url_btn = gtk_check_button_new_with_mnemonic (_("Use _custom address for radar map"));
		gtk_widget_show (pref->priv->basic_radar_url_btn);
		gtk_box_pack_start (GTK_BOX (radar_toggle_hbox), pref->priv->basic_radar_url_btn, FALSE, FALSE, 0);

		if ( ! g_settings_is_writable (pref->priv->applet->settings, "use-custom-radar-url"))
		{
			hard_set_sensitive (pref->priv->basic_radar_url_btn, FALSE);
		}

		pref->priv->basic_radar_url_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 12);
		gtk_widget_show (pref->priv->basic_radar_url_hbox);

		label = gtk_label_new ("    ");
		gtk_widget_show (label);
		gtk_box_pack_start (GTK_BOX (pref->priv->basic_radar_url_hbox), label, FALSE, FALSE, 0);

		label = gtk_label_new_with_mnemonic (_("A_ddress:"));
		gtk_widget_show (label);
		gtk_box_pack_start (GTK_BOX (pref->priv->basic_radar_url_hbox), label, FALSE, FALSE, 0);
		pref->priv->basic_radar_url_entry = gtk_entry_new ();
		gtk_widget_show (pref->priv->basic_radar_url_entry);
		gtk_box_pack_start (GTK_BOX (pref->priv->basic_radar_url_hbox), pref->priv->basic_radar_url_entry, TRUE, TRUE, 0);

		if ( ! g_settings_is_writable (pref->priv->applet->settings, "radar"))
		{
			hard_set_sensitive (pref->priv->basic_radar_url_entry, FALSE);
		}
	#endif /* RADARMAP */

#ifdef HAVE_LIBNOTIFY
	/* setup show-notifications button */
	pref->priv->basic_show_notifications_btn = gtk_check_button_new_with_mnemonic (_("Show _notifications"));

	if (!g_settings_is_writable (pref->priv->applet->settings, "show-notifications"))
	{
		hard_set_sensitive (pref->priv->basic_show_notifications_btn, FALSE);
	}
#endif

	frame = create_hig_category (pref_basic_vbox, _("Update"));

	pref_basic_update_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 12);

	pref_basic_update_lbl = gtk_label_new_with_mnemonic (_("_Automatically update every:"));
	gtk_widget_show (pref_basic_update_lbl);

	gtk_widget_show (pref_basic_update_hbox);

	pref_basic_update_spin_adj = gtk_adjustment_new (30, 1, 3600, 5, 25, 1);
	pref->priv->basic_update_spin = gtk_spin_button_new (pref_basic_update_spin_adj, 1, 0);
	gtk_widget_show (pref->priv->basic_update_spin);

	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->priv->basic_update_spin), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (pref->priv->basic_update_spin), GTK_UPDATE_IF_VALID);

	pref_basic_update_sec_lbl = gtk_label_new (_("minutes"));
	gtk_widget_show (pref_basic_update_sec_lbl);

	if ( ! g_settings_is_writable (pref->priv->applet->settings, "auto-update-interval"))
	{
		hard_set_sensitive (pref->priv->basic_update_spin, FALSE);
		hard_set_sensitive (pref_basic_update_sec_lbl, FALSE);
	}

	value_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 6);

	gtk_box_pack_start (GTK_BOX (pref_basic_update_hbox), pref->priv->basic_update_btn, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (pref_basic_update_hbox), value_hbox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (value_hbox), pref->priv->basic_update_spin, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (value_hbox), pref_basic_update_sec_lbl, FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (frame), pref_basic_update_hbox);

	frame = create_hig_category (pref_basic_vbox, _("Display"));

	vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 6);

	gtk_box_pack_start (GTK_BOX (vbox), unit_grid, TRUE, TRUE, 0);

	#ifdef RADARMAP
		gtk_box_pack_start (GTK_BOX (vbox), pref->priv->basic_radar_btn, TRUE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), radar_toggle_hbox, TRUE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), pref->priv->basic_radar_url_hbox, TRUE, TRUE, 0);
	#endif /* RADARMAP */

	#ifdef HAVE_LIBNOTIFY
		/* add the show-notification toggle button to the vbox of the display section */
		gtk_box_pack_start (GTK_BOX (vbox), pref->priv->basic_show_notifications_btn, TRUE, TRUE, 0);
	#endif

	gtk_container_add (GTK_CONTAINER (frame), vbox);

	pref_basic_note_lbl = gtk_label_new (_("General"));
	gtk_widget_show (pref_basic_note_lbl);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (pref->priv->notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (pref->priv->notebook), 0), pref_basic_note_lbl);

  /*
   * Location page.
   */
	pref_loc_hbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 6);
	gtk_container_set_border_width (GTK_CONTAINER (pref_loc_hbox), 12);
	gtk_container_add (GTK_CONTAINER (pref->priv->notebook), pref_loc_hbox);

	tree_label = gtk_label_new_with_mnemonic (_("_Select a location:"));
	gtk_label_set_xalign (GTK_LABEL (tree_label), 0.0);
	gtk_box_pack_start (GTK_BOX (pref_loc_hbox), tree_label, FALSE, FALSE, 0);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	pref->priv->tree = gtk_tree_view_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (tree_label), GTK_WIDGET (pref->priv->tree));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (pref->priv->tree), FALSE);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pref->priv->tree));

	gtk_container_add (GTK_CONTAINER (scrolled_window), pref->priv->tree);
	gtk_widget_show (pref->priv->tree);
	gtk_widget_show (scrolled_window);
	gtk_box_pack_start (GTK_BOX (pref_loc_hbox), scrolled_window, TRUE, TRUE, 0);
	load_locations(pref);

	pref_find_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 6);
	pref_find_label = gtk_label_new (_("_Find:"));
	gtk_label_set_use_underline (GTK_LABEL (pref_find_label), TRUE);

	pref->priv->find_entry = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (pref_find_label), pref->priv->find_entry);

	pref->priv->find_next_btn = gtk_button_new_with_mnemonic (_("Find _Next"));
	gtk_widget_set_sensitive (pref->priv->find_next_btn, FALSE);

	image = gtk_image_new_from_icon_name ("edit-find", GTK_ICON_SIZE_BUTTON);
	gtk_button_set_image (GTK_BUTTON (pref->priv->find_next_btn), image);

	gtk_container_set_border_width (GTK_CONTAINER (pref_find_hbox), 0);
	gtk_box_pack_start (GTK_BOX (pref_find_hbox), pref_find_label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (pref_find_hbox), pref->priv->find_entry, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (pref_find_hbox), pref->priv->find_next_btn, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (pref_loc_hbox), pref_find_hbox, FALSE, FALSE, 0);

	if ( ! g_settings_is_writable (pref->priv->applet->settings, "location0"))
	{
		hard_set_sensitive (scrolled_window, FALSE);
	}

	pref_loc_note_lbl = gtk_label_new (_("Location"));
	gtk_widget_show (pref_loc_note_lbl);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (pref->priv->notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (pref->priv->notebook), 1), pref_loc_note_lbl);

	mateweather_pref_set_accessibility (pref);
	gtk_label_set_mnemonic_widget (GTK_LABEL (pref_basic_update_sec_lbl), pref->priv->basic_update_spin);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), pref->priv->basic_radar_url_entry);

    /* Set the initial values */
    update_dialog (pref);

    /* signals */
    g_signal_connect (temp_combo,  "changed", G_CALLBACK (on_temp_combo_changed), pref);
    g_signal_connect (speed_combo, "changed", G_CALLBACK (on_speed_combo_changed), pref);
    g_signal_connect (dist_combo,  "changed", G_CALLBACK (on_dist_combo_changed), pref);
    g_signal_connect (pres_combo,  "changed", G_CALLBACK (on_pres_combo_changed), pref);
    g_signal_connect (pref->priv->basic_update_btn, "toggled", G_CALLBACK (on_auto_update_toggled), pref);
#ifdef RADARMAP
    g_signal_connect (pref->priv->basic_radar_btn, "toggled", G_CALLBACK (on_radar_toggled), pref);
    g_signal_connect (pref->priv->basic_radar_url_btn, "toggled", G_CALLBACK (on_use_radar_url_toggled), pref);
    g_signal_connect (pref->priv->basic_radar_url_entry, "focus-out-event", G_CALLBACK (on_radar_url_changed), pref);
#endif /* RADARMAP */
#ifdef HAVE_LIBNOTIFY
    g_signal_connect (pref->priv->basic_show_notifications_btn, "toggled", G_CALLBACK (on_show_notifications_toggled), pref);
#endif /* HAVE_LIBNOTIFY */
    g_signal_connect (pref->priv->find_next_btn, "clicked", G_CALLBACK (on_find_next_clicked), pref);
    g_signal_connect (pref->priv->find_entry, "changed", G_CALLBACK (find_entry_changed), pref);
    g_signal_connect (selection, "changed", G_CALLBACK (on_row_selected), pref);
    g_signal_connect (pref->priv->basic_update_spin, "value-changed", G_CALLBACK (on_update_interval_changed), pref);
    g_signal_connect (pref, "response", G_CALLBACK (on_response), pref);
}

static void mateweather_pref_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec *pspec)
{
    MateWeatherPref* pref = MATEWEATHER_PREF(object);

    switch (prop_id) {
        case PROP_MATEWEATHER_APPLET:
            pref->priv->applet = g_value_get_pointer(value);
            break;
        default:
            /* We don't have any other property... */
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void mateweather_pref_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec)
{
    MateWeatherPref* pref = MATEWEATHER_PREF(object);

    switch (prop_id) {
        case PROP_MATEWEATHER_APPLET:
            g_value_set_pointer(value, pref->priv->applet);
            break;
        default:
            /* We don't have any other property... */
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void mateweather_pref_init(MateWeatherPref* self)
{
	self->priv = mateweather_pref_get_instance_private(self);
}

static GObject*
mateweather_pref_constructor (GType                  type,
                              guint                  n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;
    MateWeatherPref *self;

    object = G_OBJECT_CLASS (mateweather_pref_parent_class)->constructor(type, n_construct_params, construct_params);
    self = MATEWEATHER_PREF (object);

    mateweather_pref_create (self);

    return object;
}

GtkWidget* mateweather_pref_new(MateWeatherApplet* applet)
{
    return g_object_new(MATEWEATHER_TYPE_PREF, "mateweather-applet", applet, NULL);
}

static void mateweather_pref_finalize(GObject* object)
{
	MateWeatherPref* self = MATEWEATHER_PREF(object);

	gtk_tree_model_foreach(self->priv->model, free_data, NULL);
	g_object_unref(G_OBJECT(self->priv->model));

	G_OBJECT_CLASS(mateweather_pref_parent_class)->finalize(object);
}

static void mateweather_pref_class_init(MateWeatherPrefClass* klass)
{
    GObjectClass* object_class = G_OBJECT_CLASS(klass);

    mateweather_pref_parent_class = g_type_class_peek_parent(klass);

    object_class->set_property = mateweather_pref_set_property;
    object_class->get_property = mateweather_pref_get_property;
    object_class->constructor = mateweather_pref_constructor;
    object_class->finalize = mateweather_pref_finalize;

    /* This becomes an OBJECT property when MateWeatherApplet is redone */
    g_object_class_install_property(object_class, PROP_MATEWEATHER_APPLET, g_param_spec_pointer("mateweather-applet", "MateWeather Applet", "The MateWeather Applet", G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}
