/*
 * $Id: truerand.c,v 1.1.1.1 2002/12/16 23:32:43 kravietz Exp $
 *
 * Generate truly random bytes: VonNeumannBytes() version 1.0
 * written by Adam L. Young, Copyright (c) 2002 Adam L. Young
 * 
 * July 27, 2002
 * 
 * This is the source file for VonNeumannBytes() which returns
 * a byte stream of random bytes. It is based on truerand()
 * of M. Blaze, and J. Lacy, D. Mitchell, and W. Schell [1], since it
 * assumes that the underlying motherboard has a cpu crystal
 * and real-time crystal and attempts to extract physical randomness
 * from fluctuations in their frequencies of oscillation. The
 * basic idea is to set a timer interrupt using an OS call with a 
 * call-back routine and add unity to an initially zero
 * counter ad infinitum until the interrupt occurs. The call-back
 * routine causes the infinite increment loop to terminate. The lower
 * order 16 bits of the counter are assumed to contain some true
 * randomness. The real-time crystal defines the time until
 * the timer interrupt occurs. The instruction speed determines
 * the number of increments to the counter in this time. The
 * theory behind truerand is that the randomness of this process
 * is reflected in the least significant bits of the counter
 * value.
 * 
 * Unlike truerand however, VonNeumannBytes() takes the further
 * step of passing this randomness through an algorithm devised by
 * John von Neumann [2] to remove any potential bias in the randomness.
 * Neumann's method for unbiasing a biased coin is used as follows.
 * Two timer interrupts are called to obtain two sets of 16 least
 * significant bits from the two counter values. The least
 * significant bits of each are used as a von Neumann trial, then the
 * two penultimate bits are used as a second trial, and so on.
 * In a trial, a 00 or 11 is disregarded. A 01 result is interpreted
 * as a random binary 0, and a 10 result is interpreted as
 * a binary 1. Let p_{i,j} denote the probability that a binary
 * 1 occurs in the jth bit of the ith trial, where j = 0,1,...15
 * and i = 1,2,3,4,.... This code therefore makes the following
 * assumptions:
 * 
 * 1) That the motherboard has separate timer and cpu clock
 * crystals.
 * 
 * 2) That p_{i,j} = p_{i+1,j} for i = 1,3,5,7,... and
 * j = 0,1,2,...,15.
 * 
 * 3) That for a non-negligible number of iterations i, where
 * i is contained in {1,3,5,7,....}, there exists a j
 * contained in {0,1,2,...,15} such that p_{i,j} is neither
 * overwhelming nor negligible (and hence that differences in
 * the crystal frequencies are capable of providing randomness
 * sufficiently often that is measurable in poly-time).
 * 
 * 
 * If the above hold, then this routine outputs truly random
 * byte sequences. Note that (2) essentially states that the
 * behavior of the 16 bit counter must be that same across pairs
 * of calls. For example, suppose that p_{i,15} = 1 for
 * i = 1,2,3,... In this case the random number generator will
 * never output a bit based on bit position j = 15.
 * 
 * Note that we would expect the bias to get worse as the
 * significance of the bit increases. Von Neumann's method is
 * used to correct the bias.
 * 
 * 
 * Acknowledgements:
 * 
 * I learned about truerand() during my Masters degree at Columbia
 * University in 1995 in a course taught by Matt Blaze. Jack
 * Lacy was an invited speaker who spoke about truerand(). I
 * have advocated its use ever since, and I hope that this work
 * is regarded as building on it in a positive way.
 * 
 * 
 * References:
 * 
 * [1] J.B. Lacy, D.P. Mitchell and W.M. Schell, "CryptoLib:
 * Cryptography in Software", Proceedings of UNIX Security
 * Symposium IV, USENIX Association, 1993.	
 * 
 * [2] J. von Neumann, "Various techniques for use in connection
 * with random digits", in von Neumann's Collected Works,
 * Pergamon, New York, 1963, pages 768-770.
 * 
 * 
 * Unfortunately there are those who would steal this without
 * attributing credit, and even those who would try to hold
 * me accountable for any errors. So, below is the license
 * agreement:
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain the copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 * 
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgement:  This product
 *    includes software developed by Adam L. Young.
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * This routine was tested with FIPS140.c written by Greg Rose of
 * Qualcomm. It performs favorably when run. Please send comments,
 * suggestions, bug reports, and hate mail to ayoung@cs.columbia.edu
 */

#include <stdio.h>
#include <sys/time.h>
#include <signal.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifndef HAVE_SETITIMER
# error "Must have setitimer()!"
#endif

static volatile int gQuit;

int VonNeumannBytes (unsigned char *buff, unsigned int bufflen);

/* VonNeumannBytes() fills the buffer buff with
bufflen truly random bytes. It returns 0 on success and -1
on failure. */
static void
interrupt ()
{
  gQuit = 1;
}
static unsigned int
ReturnPhysical2Bytes (void)
{
  struct itimerval it, oit;
  register unsigned long count;
  timerclear (&it.it_interval);
  it.it_value.tv_sec = 0;
  it.it_value.tv_usec = 15666;
  if (setitimer (ITIMER_REAL, &it, &oit) < 0)
    perror ("setitimer error occured");
  signal (SIGALRM, interrupt);
  count = 0;
  while (!gQuit)
    count++;
  gQuit = 0;
  return count % 0xFFFF;
}
static unsigned int
VonNeumann2Bytes (void)
{
  register unsigned char i, numbitsobtained = 0, toss1, toss2;
  register unsigned int bytes1, bytes2, retval = 0;

/* extract 16 bits at a time from crystals per alarm interrupt. */
  for (;;)

    {
      bytes1 = ReturnPhysical2Bytes ();
      bytes2 = ReturnPhysical2Bytes ();
      for (i = 0; i < 16; i++)

	{
	  toss1 = bytes1 % 2;
	  toss2 = bytes2 % 2;
	  if (toss1 != toss2)

	    {
	      retval = retval | toss1;
	      numbitsobtained++;
	      if (numbitsobtained == 16)
		goto EndVonNeumann2Bytes;
	      retval <<= 1;
	    }
	  bytes1 >>= 1;
	  bytes2 >>= 1;
	}
    }
EndVonNeumann2Bytes:return retval;
}

int
VonNeumannBytes (unsigned char *buff, unsigned int bufflen)
{
  register unsigned char isodd;
  register unsigned int i, localbufflen, twobytes;

/* VonNeumannBytes() fills the buffer buff with
bufflen truly random bytes. It returns 0 on success and -1
on failure. */
  if ((!bufflen) || (!buff))
    return -1;
  isodd = bufflen % 2;
  localbufflen = bufflen - isodd;
  for (i = 0; i < localbufflen;)

    {
      twobytes = VonNeumann2Bytes ();
      buff[i] = twobytes % 0xFF;
      twobytes >> 8;
      i++;
      buff[i] = twobytes;
      i++;
    }
  if (isodd)
    buff[i] = VonNeumann2Bytes () % 0xFF;
  return 0;
}
