%{
  
/*

    confparse.l - ConfParse parser rules for flex
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <errno.h>
#include <string.h>


#include "confparse.h"


static struct confvars *conf_vars;  /* local pointer to the array of confvars */



static int	i = 0,			/* all purpose var */
		lineno = 1,		/* row number for error reporting */
		current_var = -1,	/* current var parsed, index in conf_vars */
		str_len = 0;		/* length of string currently reading */

static char	*dst_ptr = NULL,	/* string building local pointer */
		*sbuf = NULL;		/* local buffer */
		      

		      

%}


 /* octal, decimal and hex integer definition */
INTERO		("0x"[0-9a-f]+)|[+-]?[0-9]+

 /* float decimal definition */
DECIMALE	[+-]?[0-9]*[\.][0-9]*

 /* var identifier */
IDENT	  [[:alpha:]_][[:alnum:]_]*


%option noyywrap

%x identifier_found int_expect ignore_line dec_expect string_expect in_string
%x intcall_expect strcall_expect
%s end_of_line


%%


<ignore_line>.+   {
  /* Rule to ignore the rest of the line, used in comments and errors */
#ifdef DEBUG
  printf("Line %d: ignore remaining chars\n",lineno);
#endif
}


<INITIAL,end_of_line>#    {
  /* Rule for comments, beginning for '#' */
#ifdef DEBUG
  printf("Line %d: found comment '#'\n",lineno);
#endif
  BEGIN(ignore_line);
}


<identifier_found,int_expect,dec_expect>#    {
  /* Error detection rule, misplaced comment */
  fprintf(stderr,"Error line %d: misplaced comment '#'\n",lineno);
  BEGIN(ignore_line);
}
 


<INITIAL,identifier_found,int_expect,dec_expect,end_of_line,string_expect,intcall_expect,strcall_expect>[ \t]+  {
  /* 
   * Rule ignoring whitespaces at the line beginning, after identifier and 
   * after '=' operator
   */
#ifdef DEBUG
  printf("Line %d: ignoring whitespace(s)\n",lineno);
#endif
}


<INITIAL,end_of_line,ignore_line>\n	{
  /* Line counting rule. For error reporting and debug */
#ifdef DEBUG
  printf("Line %d: end of line.\n",lineno);
#endif
  lineno++;
  BEGIN(INITIAL);   /* one assignment per line allowed */
}


<identifier_found>"="  {
  /* Assignment operator found, now wait for value */
  switch (conf_vars[current_var].type) {
    case VAR_INT: 
      BEGIN(int_expect);
      break;
    case VAR_STR:
      BEGIN(string_expect);
      break;
    case VAR_DEC:
      BEGIN(dec_expect);
      break;
    case CALLBK_INT:
      BEGIN(intcall_expect);
      break;
    case CALLBK_STR:
      BEGIN(strcall_expect);
      break;
    default:
      fprintf(stderr, "Error line %d: var type unknown\n",lineno);
      BEGIN(ignore_line);
  }
#ifdef DEBUG
  printf("Line %d: found operator '='\n",lineno);
#endif
}


<dec_expect>{DECIMALE}|{INTERO}	{
  /* Rule for decimal float */
  *((double*) conf_vars[current_var].varp) = (double) atof(yytext);
#ifdef DEBUG
  printf("Line %d: readed value %s for var '%s'\n",lineno,
  yytext,conf_vars[current_var].name);
#endif
  BEGIN(end_of_line);
  current_var = -1;
}



<int_expect>{INTERO}	{
  /* Rule for decimal, octal or hex integer */
  *((int*) conf_vars[current_var].varp) = (int) strtol(yytext,NULL,0);
#ifdef DEBUG
  printf("Line %d: readed value %s for var '%s'\n",lineno,
  yytext,conf_vars[current_var].name);
#endif
  BEGIN(end_of_line);
  current_var = -1;
}


<intcall_expect>{INTERO}  {
  /* Rule for callback with single int parameter */
  i = (int) strtol(yytext,NULL,0);
#ifdef DEBUG
  printf("Line %d: readed value %s for callbk '%s'\n",lineno,
  yytext,conf_vars[current_var].name);
#endif
  ((void(*)())conf_vars[current_var].varp)(i);
  BEGIN(end_of_line);
  current_var = -1;
}



<strcall_expect,string_expect>\"   {
  /* Found a double quote, start string reading */
  str_len = 0;		/* char readed counter */
  sbuf = (char*) malloc(conf_vars[current_var].size);
  if (sbuf == NULL) 
  {
    fprintf(stderr,"Line %d: memory allocation failure for var %s\n",
	lineno,conf_vars[current_var].name);
    BEGIN(ignore_line);
  }
  BEGIN(in_string);	/* now we are inside a string */
  dst_ptr = sbuf;	/* buffer pointer */
#ifdef DEBUG
  printf("Line %d: string start\n",lineno);
#endif
}



<in_string>[^\n\\\"]+	{
  char *src_ptr = yytext;

  /* Rule that read string up to maximum-1 to allow terminating zero
     even if readed string is too long */
  while (*src_ptr && (str_len < (conf_vars[current_var].size) - 1)) {
    *dst_ptr++ = *src_ptr++;
    str_len++;
  }
#ifdef DEBUG
  printf("Line %d: chars added '%s'\n",lineno,yytext);
#endif
}



<in_string>\\\"|\\\\	  {
  /* Rule to read escaped backslash (\\) and double quote (\") in strings */
  if (str_len < (conf_vars[current_var].size - 1)) {
    *dst_ptr++ = yytext[1];
    str_len++;
  }
#ifdef DEBUG
  printf("Line %d: add char %c\n",lineno,yytext[1]);
#endif
}



<in_string>\\t	  {
  /* Rule for TAB char (\t) */
  if (str_len < (conf_vars[current_var].size - 1)) {
    *dst_ptr++ = '\t';
    str_len++;
  }
#ifdef DEBUG
  printf("Line %d: add TAB\n",lineno);
#endif
}



<in_string>\\n	  {
  /* Rule for newline char (\n) */
  if (str_len < (conf_vars[current_var].size - 1)) {
    *dst_ptr++ = '\n';
    str_len++;
  }
#ifdef DEBUG
  printf("Line %d: add newline\n",lineno);
#endif
}



<in_string>\\r	  {
  /* Rule for carriage return (\r) */
  if (str_len < (conf_vars[current_var].size - 1)) {
    *dst_ptr++ = '\r';
    str_len++;
  }
#ifdef DEBUG
  printf("Line %d: add CR\n",lineno);
#endif
}


  /* TODO escape sequences like \nnn or \xNN TODO */



<in_string>\n	{
  /* warning rule, unterminated string */
  *dst_ptr = 0;	    /* terminate string */
  BEGIN(INITIAL);
  fprintf(stderr,"Warn line %d: undelimited string, unespected end of line.\n",lineno);
  lineno++;
  if (conf_vars[current_var].type == VAR_STR)
  {
    strcpy((char*)conf_vars[current_var].varp,sbuf);
  }
  else if (conf_vars[current_var].type == CALLBK_STR)
  {
    ((void(*)())conf_vars[current_var].varp)(sbuf);
  }
  free(sbuf);
#ifdef DEBUG
  printf("Line %d: string end\n",lineno);
#endif
}


<in_string>\"	{
  /* Rule for terminating string with double quote */
  *dst_ptr = 0;	    /* terminate string */
  BEGIN(end_of_line);
  if (conf_vars[current_var].type == VAR_STR)
  {
    strcpy((char*)conf_vars[current_var].varp,sbuf);
  }
  else if (conf_vars[current_var].type == CALLBK_STR)
  {
    ((void(*)())conf_vars[current_var].varp)(sbuf);
  }
  free(sbuf);
#ifdef DEBUG
  printf("Line %d: string end\n",lineno);
#endif
}



{INTERO}|{DECIMALE}	{
  /* Error rule: misplaced numeric value */
  fprintf(stderr,"Error line %d: misplaced numeric value.\n", lineno);
}


<INITIAL>{IDENT}	{   
  /*
    Rule for identifiers, verifies that the name is known in the conf_vars list
  */
#ifdef DEBUG
  printf("Line %d: found an identifier '%s'\n",lineno,yytext);
#endif
  i = 0;
  current_var = -1;
  while (conf_vars[i].varp != NULL) {
    if (strcasecmp(yytext,conf_vars[i].name) == 0) {
      current_var = i;
      break;
    }
    i++;
  }
  if (current_var >= 0) {
    BEGIN(identifier_found);
  }
  else {
    fprintf(stderr,"Error: unknown var '%s' in line %d.\n",
    yytext, lineno);
    BEGIN(ignore_line);
  }
}


"="	{
  /* Error rule for misplaced '=' operator */
  fprintf(stderr,"Error line %d: misplaced operator '='.\n",
    lineno);
  BEGIN(ignore_line);
}


<int_expect,dec_expect,identifier_found,end_of_line>{IDENT}	{
  /* Error rule for misplaced identifier */
  fprintf(stderr,"Error line %d: misplaced identifier '%s'\n",
    lineno, yytext);
  BEGIN(ignore_line);
}


\"	{
  /* Error rule for misplaced double quote (") */
  fprintf(stderr,"Error line %d: misplaced double quote (\")\n",
    lineno);
  BEGIN(ignore_line);
}
  


<dec_expect,int_expect,identifier_found>\n   {
  fprintf(stderr,"Error line %d: unespected end of line.\n",lineno);
  lineno++;
  BEGIN(INITIAL);   /* only one assignment per line */
}


<end_of_line>.	  |
.	{
  /* Error rule for generic garbage */
  fprintf(stderr,"Warn line %d: ignored char(s)\n",lineno);
  BEGIN(ignore_line);
}



%%


/* 
 * Parser main function. Requires two parameter:
 * cv: array of confvars structs, terminated by a 0-filled struct.
 * fp: FILE * to the configuration file, ready for parsing.
 */

int confparse(struct confvars *cv, FILE* fp)
{

  if (cv == NULL) {
    errno = EINVAL;
    return -1;
  }
  if (fp == NULL) {
    errno = EBADF;
    return -1;
  }
  conf_vars = cv;
  yyin = fp;
  return yylex();
}



