//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//                  Network Security Analysis Tool                          //
//            SockSet.cpp - connection management functions                 //
//                                                                          //
//   Copyright (C) 1999-2002 by Mixter and 2xs ltd. <mixter@2xs.co.il>      //
//                                                                          //
// This program is free software; you can redistribute it and/or modify     //
// it under the terms of the GNU General Public License as published by     //
// the Free Software Foundation; either version 2 of the License, or        //
// (at your option) any later version.                                      //
//                                                                          //
// This program is distributed in the hope that it will be useful,          //
// but WITHOUT ANY WARRANTY; without even the implied warranty of           //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            //
// GNU General Public License for more details.                             //
//                                                                          //
// You should have received a copy of the GNU General Public License        //
// along with this program; if not, write to the Free Software              //
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#pragma implementation "sockset"
#include "SockSet.h"
#include "services.h"

#ifdef EBUG
extern ofstream dbug;

#endif

extern ProgressIndicator pi;

SockSet::SockSet(void)
{
  timeout = pi.timeout;
  highest_fd = -1;
  sflags = -1;
  siginterrupt(SIGALRM, 1);
  for (int i = 0; i < SetSize; i++)
    tSet[i] = 0;
  FD_ZERO(&sfds);
}

SockSet::~SockSet(void)
{
  int i;

  for (i = 0; i <= SetSize; i++)
    if (tSet[i])
      close(tSet[i]);
}

int SockSet::sInit(void)
{
poor_trick:
  int opt = MAXCAPLEN, one = 1, i = tsock();

  if (sflags < 0)
    sflags = fcntl(i, F_GETFL, 0);

  if (i < 0)
    {
#ifdef EBUG
      dbug << "socket() failing: " << strerror(errno) << ", waiting to try again...\n";
#endif
      sleep(pi.timeout * 5);
      goto poor_trick;
    }

  setsockopt(i, SOL_SOCKET, SO_RCVBUF, (char *) &opt, sizeof(int));

  ioctl(i, FIONBIO, &one);
  fcntl(i, F_SETFL, O_NONBLOCK | O_NDELAY);
  return i;
}

void SockSet::sCon(int sc, unsigned short sn)
{
  sin.sin_family = AF_INET;
  sin.sin_port = htons(sn);
  sin.sin_addr.s_addr = target;
  memset(&sin.sin_zero, 0, 8);

  tSet[sc] = sInit();
  int test = connect(tSet[sc], (struct sockaddr *) &sin, SALen);

  if (test == -1)
    switch (errno)
      {
      case EINPROGRESS:
	test = 1;
	break;
      case EINVAL:
      case EBADF:
      case ECONNREFUSED:
      case ETIMEDOUT:
      case ENETUNREACH:
	test = -2;
	SCLOSE(tSet[sc]);
	tSet[sc] = 0;
	break;
      case EALREADY:
	test = 2;
	break;
#ifdef EBUG
      default:
	dbug << "Strange: test(" << sc << ") == " << test << " (" << strerror(errno) << ")\n";
#endif
      }

  if (highest_fd < tSet[sc])
    highest_fd = tSet[sc];

  FD_SET(tSet[sc], &sfds);
}

int SockSet::check(int ref)
{
  int s = tSet[ref];
  unsigned int err = 0, erl = 0;

  if (getsockopt(s, SOL_SOCKET, SO_ERROR, &err, (socklen_t *)&erl) == 0)
    {
      if (err == 0)
	{
	  int null = 0;

	  ioctl(s, FIONBIO, &null);
	  fcntl(s, F_SETFL, sflags);
#ifdef EBUG
	  dbug << "check OK: " << s << ENTER;
#endif

	  if (getpeername(s, (struct sockaddr *) &sin,(socklen_t *) &sl) == -1)
	    {
#ifdef EBUG
	      dbug << "getpeername failed: " << strerror(errno) << ENTER;
#endif
	      SCLOSE(s);
	      return (0);
	    }

	  if (!firstopen)
	    firstopen = sin.sin_port;
	  return (1);
	}
#ifdef EBUG
      else
	{
	  dbug << "getsockopt OK, but: " << strerror(err) << ENTER;
	}
    }
  else
    dbug << "getsockopt ERROR: " << " " << strerror(errno) << ENTER;
#else
    }
#endif

  SCLOSE(s);
  return (0);
}

void SockSet::con(char *host)
{
  target = resolve(host);

  if ((target == 0) || ((long) target == -1))
    {
#ifdef EBUG
      dbug << "Unknown host: " << host << ": " << h_errno << " target: " << (long) target << ENTER;
#endif
      return;
    }

  con(target);
}

void SockSet::con(unsigned int addr)
{
  target = addr;

  if ((target == 0) || ((long) target == -1))
    {
#ifdef EBUG
      dbug << "Unknown host: " << ntoa(addr) << ": " << h_errno << " target: " << (long) target << ENTER;
#endif
      return;
    }

#ifdef EBUG
  dbug << "Scan of host: " << Hostname() << ENTER;
#endif

  // wave 0: icmp querying

  if ((pi.ScanIcmp) && (!geteuid()))
    {
	  if ((icmptest() == 0) && pi.pingonly)
	    return;
    }

  if (pi.ScanNetstat)
    sCon(1, S_NSTAT);
  if (pi.ScanFtp)
    sCon(2, S_FTP);
  if (pi.ScanSsh)
    sCon(3, S_SSH);
  if (pi.ScanTelnet)
    sCon(4, S_TELNET);
  if (pi.ScanSmtp)
    sCon(5, S_SMTP);
  if (pi.ScanBind)
    sCon(6, S_BIND);
  if (pi.ScanFinger)
    sCon(7, S_FINGER);
  if (pi.ScanWeb)
    sCon(8, S_HTTP);
  if (pi.ScanPop2)
    sCon(9, S_POP2);
  if (pi.ScanPop3)
    sCon(10, S_POP3);
  if (pi.ScanRpc)
    sCon(11, S_PMAP);
  if (pi.ScanNntp)
    sCon(12, S_NNTP);
  if (pi.ScanSmb)
    sCon(13, S_SMB);
  if (pi.ScanImap)
    sCon(14, S_IMAP);
  if (pi.ScanNlps)
    sCon(15, S_NLPS);
  if (pi.ScanIrcd)
    sCon(16, S_IRCD);
  if (pi.ScanXWin)
    sCon(17, S_XWIN);

  int k = 0, j = 0;

  for (k = 100; k < 2000; k++)
    {
      if (pi.Ports[j] > 0)
	sCon(k, pi.Ports[j]);
      if (j++ >= SMALLB)
	break;
    }

  if (pi.PortRange[0])
    {
      if (pi.PortRange[0] > pi.PortRange[1])
	{
	  int tmp = pi.PortRange[0];

	  pi.PortRange[0] = pi.PortRange[1];
	  pi.PortRange[1] = tmp;
	}

      j = pi.PortRange[0];
      for (k = 2000; k < SetSize; k++)
	{
	  if (j >= pi.PortRange[1])
	    break;
	  sCon(k, j);
	  j++;
	}
    }

  probe(pi.timeout);
  return;
}

void SockSet::probe(int to)
{
#ifdef EBUG
  int i;

  dbug << "Sockets: ";
  for (i = 0; i < SetSize; i++)
    if (tSet[i])
      dbug << " " << tSet[i];
  dbug << ENTER;
#endif

  if (to)
    timeout = to;

// prepare for select()ion

  struct timeval t = {timeout * 10, 0};

  if (!select(highest_fd + 1, NULL, &sfds, NULL, &t))
    {
#ifdef EBUG
      dbug << "select() failed: " << strerror(errno) << ENTER;
#endif

     /* Ok, no open TCP port so we just do connectionless stuff. */

     if (pi.ScanNetBios)
       nbname();
     if (pi.ScanSnmp)
       snmp();
     if (pi.ScanBackOrifice)
       bo();
#ifdef WITH_XPROBE
     if ((pi.ScanOS) && (!geteuid()))
       os();
#endif
      
      return;
    }

  // inherited AuditSet class functions
  // wave 1 services: port scan

  int k = 0, j = 0;

  for (k = 100; k < 2000; k++)
    {
      if (pi.Ports[j] > 0)
	if (check(k))
	  plog(tSet[k]);
      if (j++ >= SMALLB)
	break;
    }

// wave 2 services: read input

  if (pi.ScanBind)
    if (check(6))
      nbind(tSet[6]);
  if (pi.ScanNetstat)
    if (check(1))
      nstat(tSet[1]);
  if (pi.ScanSsh)
    if (check(3))
      ssh(tSet[3]);
  if (pi.ScanPop2)
    if (check(9))
      pop2(tSet[9]);
  if (pi.ScanPop3)
    if (check(10))
      pop3(tSet[10]);
  if (pi.ScanNntp)
    if (check(12))
      nntp(tSet[12]);
  if (pi.ScanImap)
    if (check(14))
      imap(tSet[14]);

// wave 3 services: security check

  if (pi.ScanFinger)
    if (check(7))
      finger(tSet[7]);
  if (pi.ScanTelnet)
    if (check(4))
      telnet(tSet[4]);		// also wingate

  if (pi.ScanSmtp)
    if (check(5))
      smtp(tSet[5]);
  if (pi.ScanRpc)
    if (check(11))
      pmap(tSet[11]);		// start rpcscan

  if (pi.ScanSmb)
    if (check(13))
      smb(tSet[13]);
  if (pi.ScanNlps)
    if (check(15))
      nlps(tSet[15]);		// try overflow

  if (pi.ScanIrcd)
    if (check(16))
      ircd(tSet[16]);		// iline, version

  if (pi.ScanXWin)
    if (check(17))
      xwin(tSet[17]);		// sniff test

  if (pi.ScanFtp)
    if (check(2))
      ftp(tSet[2]);
  if (pi.ScanWeb)
    if (check(8))
      http(tSet[8]);

// wave 4 services: connectionless

  if (pi.ScanNetBios)
    nbname();
  if (pi.ScanSnmp)
    snmp();
  if (pi.ScanBackOrifice)
    bo();
#ifdef WITH_XPROBE
  if ((pi.ScanOS) && (!geteuid()))
    os();
#endif

// wave 5: port range scan

  if (pi.PortRange[0])
    {
      j = pi.PortRange[0];
      for (k = 2000; k < SetSize; k++)
	{
	  if (j >= pi.PortRange[1])
	    break;
	  if (check(k))
	    plog(k);
	  j++;
	}
    }

  return;
}
