/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifndef LIBIDNKITLITE
#include <iconv.h>
#endif

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/localconverter.h>
#include <idn/localencoding.h>
#include <idn/utf8.h>
#include <idn/util.h>

struct idn__localconverter {
	char *from_name;
	char *to_name;
	int flags;
	idn__localconverter_openproc_t openproc;
	idn__localconverter_convproc_t convproc;
	idn__localconverter_closeproc_t closeproc;
	void *private_data;
};

static idn_result_t	roundtrip_check(idn__localconverter_t ctx,
					const char *from, const char *to);

#ifndef LIBIDNKITLITE
static idn_result_t
       localconverter_iconv_open(idn__localconverter_t ctx,
				 void **privdata);
static idn_result_t
       localconverter_iconv_conv(idn__localconverter_t ctx,
				 void *privdata, const char *from,
				 char *to, size_t tolen);
static idn_result_t
       localconverter_iconv_close(idn__localconverter_t ctx, void *privdata);
static idn_result_t	iconv_initialize_privdata(void **privdata);
static void		iconv_finalize_privdata(void *privdata);
#endif /* LIBIDNKITLITE */

static idn_result_t
       localconverter_none_conv(idn__localconverter_t ctx,
				void *privdata, const char *from,
				char *to, size_t tolen);

static idn_result_t
       localconverter_uescape_convfromutf8(idn__localconverter_t ctx,
					   void *privdata, const char *from,
					   char *to, size_t tolen);
static idn_result_t
       localconverter_uescape_convtoutf8(idn__localconverter_t ctx,
					 void *privdata, const char *from,
					 char *to, size_t tolen);

/*
 * Local converters.
 */
static const struct idn__localconverter
localconverters[] = {
	{IDN__UTF8_ENCODINGNAME,
	 "U-escape",
	 0,
	 NULL,
	 localconverter_uescape_convfromutf8,
	 NULL,
	 NULL},
	{"U-escape",
	 IDN__UTF8_ENCODINGNAME,
	 0,
	 NULL,
	 localconverter_uescape_convtoutf8,
	 NULL,
	 NULL},
	{IDN__UTF8_ENCODINGNAME,
	 IDN__UTF8_ENCODINGNAME,
	 0,
	 NULL,
	 localconverter_none_conv,
	 NULL,
	 NULL},
#ifndef LIBIDNKITLITE
	{"*",
	 "*",
	 0,
	 localconverter_iconv_open,
	 localconverter_iconv_conv,
	 localconverter_iconv_close,
	 NULL},
#endif
	{NULL, NULL, 0, NULL, NULL, NULL, NULL}
};

idn_result_t
idn__localconverter_create(const char *from_name, const char *to_name, 
			   idn__localconverter_t *ctxp, int flags) {
	idn_result_t r = idn_success;
	const char *from_realname = NULL;
	const char *to_realname = NULL;
	idn__localconverter_t ctx = NULL;
	idn__localencoding_t from_ctx = NULL;
	idn__localencoding_t to_ctx = NULL;
	const struct idn__localconverter *cp;

	assert(ctxp != NULL);

	TRACE(("idn__localconverter_create(from=\"%s\", to=\"%s\")\n",
	       idn__debug_xstring(from_name), idn__debug_xstring(to_name)));

	r = idn__localencoding_create(&from_ctx);
	if (r != idn_success)
		goto ret;
	r = idn__localencoding_create(&to_ctx);
	if (r != idn_success)
		goto ret;

	if (from_name != NULL) {
		idn__localencoding_setname(from_ctx, from_name);
		from_realname = idn__localencoding_getname(from_ctx);
		if (r != idn_success)
			goto ret;
	} else {
		from_realname = IDN__UTF8_ENCODINGNAME;
	}

	if (to_name != NULL) {
		idn__localencoding_setname(to_ctx, to_name);
		to_realname = idn__localencoding_getname(to_ctx);
		if (r != idn_success)
			goto ret;
	} else {
		to_realname = IDN__UTF8_ENCODINGNAME;
	}

	/* Allocate memory for a converter context and the name. */
	ctx = (idn__localconverter_t)malloc(sizeof(*ctx));
	if (ctx == NULL) {
		r = idn_nomemory;
		goto ret;
	}

	ctx->from_name    = idn__util_strdup(from_realname);
	ctx->to_name      = idn__util_strdup(to_realname);
	ctx->flags        = flags;
	ctx->openproc     = NULL;
	ctx->convproc     = NULL;
	ctx->closeproc    = NULL;
	ctx->private_data = NULL;

	if (ctx->from_name == NULL || ctx->to_name == NULL) {
		r = idn_nomemory;
		goto ret;
	}

	for (cp = localconverters; cp->convproc != NULL; cp++) {
		if ((strcmp(cp->from_name, "*") == 0 ||
		    strcmp(cp->from_name, from_realname) == 0) &&
		    (strcmp(cp->to_name, "*") == 0 ||
		    strcmp(cp->to_name, to_realname) == 0)) {
			ctx->openproc  = cp->openproc;
			ctx->convproc  = cp->convproc;
			ctx->closeproc = cp->closeproc;
			break;
		}
	}

	if (ctx->convproc == NULL) {
		r = idn_invalid_name;
		goto ret;
	}

	if (ctx->openproc != NULL) {
		r = (*ctx->openproc)(ctx, &(ctx->private_data));
		if (r != idn_success) {
			WARNING(("idn__localconverter_create(): open failed "
			     "(%s->%s)\n", from_realname, to_realname));
			goto ret;
		}
	}

	*ctxp = ctx;

ret:
	if (r == idn_success) {
		TRACE(("idn__localconverter_create(): success (%s->%s)\n", 
		       from_realname, to_realname));
	} else {
		TRACE(("idn__localconverter_create(): %s\n",
		       idn_result_tostring(r)));
		if (ctx != NULL) {
			free(ctx->from_name);
			free(ctx->to_name);
			free(ctx);
		}
	}
	if (from_ctx != NULL)
		idn__localencoding_destroy(from_ctx);
	if (to_ctx != NULL)
		idn__localencoding_destroy(to_ctx);

	return (r);
}

void
idn__localconverter_destroy(idn__localconverter_t ctx) {
	assert(ctx != NULL);

	TRACE(("idn__localconverter_destroy(ctx=%s->%s)\n",
	       ctx->from_name, ctx->to_name));

	if (ctx->closeproc != NULL)
		(*ctx->closeproc)(ctx, ctx->private_data);

	TRACE(("idn__localconverter_destroy(): the object is destroyed\n"));

	free(ctx->from_name);
	free(ctx->to_name);
	free(ctx);
}

idn_result_t
idn__localconverter_convert(idn__localconverter_t ctx,
			    const char *from, char *to, size_t tolen) {
	idn_result_t r = idn_success;

	assert(ctx != NULL && from != NULL && to != NULL);

	TRACE(("idn__localconverter_convert(ctx=%s->%s, from=\"%s\", "
	       "tolen=%d)\n",
	       ctx->from_name, ctx->to_name, idn__debug_xstring(from),
	       (int)tolen));

	r = (*ctx->convproc)(ctx, ctx->private_data, from, to, tolen);
	if (r != idn_success)
		goto ret;
	if ((ctx->flags & IDN__LOCALCONVERTER_RTCHECK) != 0) {
		r = roundtrip_check(ctx, from, to);
		if (r != idn_success)
			goto ret;
	}
	
ret:
	if (r == idn_success) {
		TRACE(("idn__localconverter_convert(): success (to=\"%s\")\n",
		       idn__debug_xstring(to)));
	} else {
		TRACE(("idn__localconverter_convert(): %s\n",
		       idn_result_tostring(r)));
	}
	return (r);
}

/*
 * Round trip check.
 */
static idn_result_t
roundtrip_check(idn__localconverter_t ctx, const char *from, const char *to) {
	/*
	 * One problem with iconv() convertion is that
	 * iconv() doesn't signal an error if the input
	 * string contains characters which are valid but
	 * do not have mapping to the output codeset.
	 * (the behavior of iconv() for that case is defined as
	 * `implementation dependent')
	 * One way to check this case is to perform round-trip
	 * conversion and see if it is same as the original string.
	 */
	idn_result_t r = idn_success;
	char *back;
	char backbuf[256];
	size_t fromlen;
	size_t backlen;

	TRACE(("idn__localconverter_convert: round-trip checking "
	       "(from=\"%s\")\n",
	       idn__debug_xstring(from)));

	/* Allocate enough buffer. */
	fromlen = strlen(from) + 1;
	if (fromlen * sizeof(long) <= sizeof(backbuf)) {
		backlen = sizeof(backbuf);
		back = backbuf;
	} else {
		backlen = fromlen;
		back = (char *)malloc(backlen * sizeof(long));
		if (back == NULL)
			return (idn_nomemory);
	}

	/*
	 * Perform backward conversion.
	 */
	r = idn__localconverter_convert(ctx, to, back, backlen);
	switch (r) {
	case idn_success:
		if (memcmp(back, from, sizeof(*from) * fromlen) != 0)
			r = idn_nomapping;
		break;
	case idn_invalid_encoding:
	case idn_buffer_overflow:
		r = idn_nomapping;
		break;
	default:
		break;
	}

	if (back != backbuf)
		free(back);

	if (r != idn_success) {
		TRACE(("round-trip check failed: %s\n",
		       idn_result_tostring(r)));
	}

	return (r);
}

/*
 * General conversion procedures.
 */
#ifndef LIBIDNKITLITE

static idn_result_t
localconverter_iconv_open(idn__localconverter_t ctx, void **privdata) {
	idn_result_t r = idn_success;
	iconv_t *ictxp;

	assert(ctx != NULL);

	r = iconv_initialize_privdata(privdata);
	if (r != idn_success)
		return (r);

	ictxp = (iconv_t *)*privdata;
	*ictxp = iconv_open(ctx->to_name, ctx->from_name);
	if (*ictxp == (iconv_t)(-1)) {
		free(*privdata);
		*privdata = NULL;
		switch (errno) {
		case ENOMEM:
			return (idn_nomemory);
		case EINVAL:
			return (idn_invalid_name);
		default:
			WARNING(("iconv_open failed with errno %d\n", errno));
			return (idn_failure);
		}
	}

	return (idn_success);
}

static idn_result_t
iconv_initialize_privdata(void **privdata) {
	if (*privdata == NULL) {
		*privdata = malloc(sizeof(iconv_t));
		if (*privdata == NULL)
			return (idn_nomemory);
		*((iconv_t *)*privdata) = (iconv_t)(-1);
	}

	return (idn_success);
}

static void
iconv_finalize_privdata(void *privdata) {
	iconv_t *ictxp;
	
	if (privdata != NULL) {
		ictxp = (iconv_t *)privdata;
		if (*ictxp != (iconv_t)(-1))
			iconv_close(*ictxp);
		free(privdata);
	}
}

static idn_result_t
localconverter_iconv_conv(idn__localconverter_t ctx, void *privdata,
			  const char *from, char *to, size_t tolen) {
	idn_result_t r = idn_success;
	iconv_t ictx;
	size_t sz;
	size_t inleft;
	size_t outleft;
	const char *inbuf;
	char *outbuf;

	assert(ctx != NULL && from != NULL && to != NULL);

	if (tolen <= 0) {
		r = idn_buffer_overflow;	/* need space for NUL */
		goto ret;
	}

	ictx = ((iconv_t *)privdata)[0];

	/*
	 * Reset internal state.
	 * 
	 * The following code should work according to the SUSv2 spec,
	 * but causes segmentation fault with Solaris 2.6.
	 * So.. a work-around.
	 * 
	 * (void)iconv(ictx, (const char **)NULL, (size_t *)NULL, 
	 * 	    (char **)NULL, (size_t *)NULL);
	 */
	inleft = 0;
	outbuf = NULL;
	outleft = 0;
	iconv(ictx, (const char **)NULL, &inleft, &outbuf, &outleft);

	inleft = strlen(from);
	inbuf = from;
	outleft = tolen - 1;	/* reserve space for terminating NUL */
	sz = iconv(ictx, (const char **)&inbuf, &inleft, &to, &outleft);

	if (sz == (size_t)(-1) || inleft > 0) {
		switch (errno) {
		case EILSEQ:
		case EINVAL:
			/*
			 * We already checked the validity of the input
			 * string.  So we assume a mapping error.
			 */
			r = idn_nomapping;
			goto ret;
		case E2BIG:
			r = idn_buffer_overflow;
			goto ret;
		default:
			WARNING(("iconv failed with errno %d\n", errno));
			r = idn_failure;
			goto ret;
		}
	}

	/*
	 * Append a sequence of state reset.
	 */
	inleft = 0;
	sz = iconv(ictx, (const char **)NULL, &inleft, &to, &outleft);
	if (sz == (size_t)(-1)) {
		switch (errno) {
		case EILSEQ:
		case EINVAL:
			r = idn_invalid_encoding;
			goto ret;
		case E2BIG:
			r = idn_buffer_overflow;
			goto ret;
		default:
			WARNING(("iconv failed with errno %d\n", errno));
			r = idn_failure;
			goto ret;
		}
	}
	*to = '\0';

	return (idn_success);

ret:
	if (r != idn_buffer_overflow) {
		WARNING(("idn_iconv_convfromutf8(): %s\n",
			 idn_result_tostring(r)));
	}
	return (r);

}

static idn_result_t
localconverter_iconv_close(idn__localconverter_t ctx, void *privdata) {
	assert(ctx != NULL);

	iconv_finalize_privdata(privdata);

	return (idn_success);
}

#endif /* LIBIDNKITLITE */

static idn_result_t
localconverter_none_conv(idn__localconverter_t ctx, void *privdata,
			  const char *from, char *to, size_t tolen) {
	assert(ctx != NULL && from != NULL && to != NULL);

	return (idn__util_strcpy(to, tolen, from));
}

/*
 * Special conversion procedures for U-escape.
 *
 * Conversion to/from unicode escape string.
 * Arbitrary UTF-32 characters can be specified by a special sequence
 *	\u{XXXXXX}
 * where XXXXX denotes any hexadecimal string up to FFFFFFFF.
 * This is designed for debugging.
 */
static idn_result_t
localconverter_uescape_convfromutf8(idn__localconverter_t ctx, void *privdata,
				    const char *from, char *to, size_t tolen) {
	idn_result_t r = idn_success;
	unsigned long v;
	int width;

	while (*from != '\0') {
		width = idn__utf8_mbtowc(from, &v);
		if (width == 0) {
			r = idn_invalid_encoding;
			goto ret;
		}
		from += width;

		if (0x20 <= v && v <= 0x7e) {
			if (tolen < 1) {
				r = idn_buffer_overflow;
				goto ret;
			}
			*to++ = v;
			tolen--;
		} else if (v <= 0xffffffffUL) {
			char tmp[20];
			int len;

			(void)sprintf(tmp, "\\u{%lx}", v);
			len = strlen(tmp);
			if (tolen < len) {
				r = idn_buffer_overflow;
				goto ret;
			}
			(void)memcpy(to, tmp, len);
			to += len;
			tolen -= len;
		} else {
			r = idn_invalid_encoding;
			goto ret;
		}
	}

	if (tolen <= 0) {
		r = idn_buffer_overflow;
		goto ret;
	}
	*to = '\0';

	return (idn_success);

ret:
	if (r != idn_buffer_overflow) {
		WARNING(("idn_uescape_convfromutf8(): %s\n",
			 idn_result_tostring(r)));
	}
	return (r);
}

static idn_result_t
localconverter_uescape_convtoutf8(idn__localconverter_t ctx, void *privdata,
				  const char *from, char *to, size_t tolen) {
	idn_result_t r = idn_success;
	size_t fromlen = strlen(from);
	char mb[16];  /* large enough */
	int mblen;

	while (*from != '\0') {
		if (tolen <= 0) {
			r = idn_buffer_overflow;
			goto ret;
		}
		if (strncmp(from, "\\u{", 3) == 0 ||
		    strncmp(from, "\\U{", 3) == 0) {
			size_t ullen;
			unsigned long v;
			char *end;

			v = strtoul(from + 3, &end, 16);
			ullen = end - (from + 3);

			if (*end == '}' && ullen >= 1 && ullen <= 6) {
				mblen = idn__utf8_wctomb(mb, sizeof(mb), v);
				if (mblen == 0) {
					r = idn_invalid_encoding;
					goto ret;
				}
				r = idn__util_strncpy(to, tolen, mb, mblen);
				if (r != idn_success)
					goto ret;

				from = end + 1;
				fromlen -= ullen;
				to += mblen;
				tolen -= mblen;
			} else {
				*to = '\\';
				from++;
				fromlen--;
				to++;
				tolen--;
			}
		} else {
			mblen = idn__utf8_mblen(from);
			if (mblen == 0) {
				r = idn_invalid_encoding;
				goto ret;
			}
			r = idn__util_strncpy(to, tolen, from, mblen);
			if (r != idn_success)
				goto ret;

			from += mblen;
			fromlen -= mblen;
			to += mblen;
			tolen -= mblen;
		}
	}

	if (tolen <= 0) {
		r = idn_buffer_overflow;
		goto ret;
	}
	*to = '\0';

	return (idn_success);

ret:
	if (r != idn_buffer_overflow) {
		WARNING(("idn_uescape_convtoutf8(): %s\n",
			 idn_result_tostring(r)));
	}
	return (r);
}
