
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Most of this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: download.c 2513 2007-07-16 13:27:11Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>

#ifdef HAVE_CURL
#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>
#endif

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mutex.h"
#include "download.h"
#include "environment.h"
#include "oxine.h"
#include "utils.h"

#ifdef HAVE_CURL

extern oxine_t *oxine;

#define USER_AGENT  PACKAGE_NAME"/"PACKAGE_VERSION
#define USER_PWD    "anonymous:"PACKAGE_BUGREPORT

static double percent = 0.0;


download_t *
download_new (const char *filename)
{
    download_t *download = ho_new (download_t);

    download->buffer = NULL;
    download->size = 0;
    download->file = NULL;

    if (filename) {
        download->file = fopen (filename, "w");
        if (!download->file) {
            error (_("Could not open '%s': %s!"), filename, strerror (errno));
            ho_free (download);
        }
    }

    percent = 0.0;

    return download;
}


void
download_free (download_t * download, bool free_buffer)
{
    if (download) {
        if (download->file) {
            fclose (download->file);
        }
        if (free_buffer) {
            ho_free (download->buffer);
        }
        ho_free (download);
    }

    percent = 0.0;
}


static int
progress_cb (void *userdata, double dltotal, double dlnow,
             double ultotal, double ulnow)
{
    percent = (dltotal > 0.0) ? (dlnow * 100.0 / dltotal) : 0.0;

    debug ("Downloading... %6.2f %%", percent);

    oxine_event_t ev;
    ev.type = OXINE_EVENT_OTK_UPDATE;
    odk_oxine_event_send (oxine->odk, &ev);

    return 0;
}


static size_t
download_write_cb (void *ptr, size_t size, size_t nmemb, void *userdata)
{
    download_t *download = (download_t *) userdata;

    size_t wsize = 0;
    size_t rsize = size * nmemb;
    size_t bsize = download->size + rsize + 1;
    if (download->size == 0) {
        download->buffer = (char *) ho_malloc (bsize);
    }
    else {
        download->buffer = (char *) ho_realloc (download->buffer, bsize);
    }

    if (download->buffer) {
        memcpy (&(download->buffer[download->size]), ptr, rsize);
        download->size += rsize;
        download->buffer[download->size] = 0;
        wsize = rsize;
    }

    return wsize;
}


bool
network_download (const char *url, download_t * download)
{
    assert (url);
    assert (download);

    debug ("Starting download of %s", url);

    CURL *curl = NULL;
    CURLcode res = CURLE_FAILED_INIT;

    if ((curl = curl_easy_init ()) != NULL) {
#ifdef DEBUG
        curl_easy_setopt (curl, CURLOPT_VERBOSE, true);
#endif

        curl_easy_setopt (curl, CURLOPT_URL, url);
        curl_easy_setopt (curl, CURLOPT_USERAGENT, USER_AGENT);
        curl_easy_setopt (curl, CURLOPT_USERPWD, USER_PWD);
        curl_easy_setopt (curl, CURLOPT_NOSIGNAL, true);
        curl_easy_setopt (curl, CURLOPT_FAILONERROR, true);
        curl_easy_setopt (curl, CURLOPT_NETRC, CURL_NETRC_OPTIONAL);
        curl_easy_setopt (curl, CURLOPT_FOLLOWLOCATION, true);

        curl_easy_setopt (curl, CURLOPT_NOPROGRESS, false);
        curl_easy_setopt (curl, CURLOPT_PROGRESSFUNCTION, progress_cb);
        curl_easy_setopt (curl, CURLOPT_PROGRESSDATA, download);

        if (download->file) {
            curl_easy_setopt (curl, CURLOPT_WRITEDATA, download->file);
        }
        else {
            curl_easy_setopt (curl, CURLOPT_WRITEFUNCTION, download_write_cb);
            curl_easy_setopt (curl, CURLOPT_WRITEDATA, download);
        }

        if ((res = curl_easy_perform (curl)) != CURLE_OK) {
            error ("Downloading of '%s' failed: %s!",
                   url, curl_easy_strerror (res));
        }

        curl_easy_cleanup (curl);
    }
    else {
        error ("Downloading of '%s' failed: %s!", url, _("CURL initialization failed"));
    }

    return (res == CURLE_OK);
}


char *
download_to_cache (const char *url, const char *filename, bool force_update)
{
    /* Create the URL of the cache. */
    int prefix_len = 0;
    if (strncasecmp (url, "http://", 7) == 0) {
        prefix_len = 7;
    }
    else if (strncasecmp (url, "ftp://", 6) == 0) {
        prefix_len = 6;
    }
    else {
        error (_("Unknown prefix to URL '%s'!"), url);
        return NULL;
    }

    char *cache_file = NULL;
    if (filename) {
        cache_file = ho_strdup_printf ("%s/%s", get_dir_oxine_tmp (),
                                       filename);
    }
    else {
        cache_file = ho_strdup_printf ("%s/%s", get_dir_oxine_tmp (),
                                       (url + prefix_len));
    }

    /* If the file is already in the cache we quit. */
    if (!force_update && file_exists (cache_file)) {
        return cache_file;
    }

    /* Create the cache directory */
    char *cache_dir = get_dirname (cache_file);
    if (!mkdir_recursive (cache_dir, 0700)) {
        ho_free (cache_dir);
        ho_free (cache_file);
        return NULL;
    }
    ho_free (cache_dir);

    debug ("Downloading '%s' to '%s'...", url, cache_file);

    /* Download the file. */
    download_t *download = download_new (cache_file);
    if (!download) {
        ho_free (cache_file);
        return NULL;
    }

    if (!network_download (url, download)) {
        download_free (download, false);
        ho_free (cache_file);
        return NULL;
    }

    download_free (download, false);
    chmod (cache_file, S_IRUSR | S_IWUSR);

    return cache_file;
}


static size_t
redirect_header_cb (void *ptr, size_t size, size_t nmemb, void *userdata)
{
    download_t *download = (download_t *) userdata;

    size_t rsize = size * nmemb;
    char *buffer = (char *) ptr;

    buffer[rsize - 1] = 0;
    if (starts_with (buffer, "Location: ")) {
        char *location = trim_whitespace (buffer + 10);
        download->buffer = ho_strdup (location);
        debug ("found redirection '%s'", location);
        return -1;
    }

    return rsize;
}


static size_t
redirect_write_cb (void *ptr, size_t size, size_t nmemb, void *userdata)
{
    return -1;
}


char *
get_redirect_url (const char *url)
{
    download_t *download = download_new (NULL);
    if (!download) {
        return NULL;
    }

    debug ("searching redirection for '%s'...", url);

    CURL *curl = NULL;
    if ((curl = curl_easy_init ()) != NULL) {
#ifdef DEBUG
        curl_easy_setopt (curl, CURLOPT_VERBOSE, true);
#endif

        curl_easy_setopt (curl, CURLOPT_URL, url);
        curl_easy_setopt (curl, CURLOPT_USERAGENT, USER_AGENT);
        curl_easy_setopt (curl, CURLOPT_USERPWD, USER_PWD);
        curl_easy_setopt (curl, CURLOPT_NOSIGNAL, true);
        curl_easy_setopt (curl, CURLOPT_FAILONERROR, true);

        curl_easy_setopt (curl, CURLOPT_HEADERFUNCTION, redirect_header_cb);
        curl_easy_setopt (curl, CURLOPT_HEADERDATA, download);

        curl_easy_setopt (curl, CURLOPT_WRITEFUNCTION, redirect_write_cb);
        curl_easy_setopt (curl, CURLOPT_WRITEDATA, download);

        curl_easy_perform (curl);
        curl_easy_cleanup (curl);
    }
    else {
        error ("Downloading of '%s' failed: %s!", url, _("CURL initialization failed"));
    }

    char *location = download->buffer;
    download_free (download, false);

    return location;
}


#else

char *
download_to_cache (const char *url, const char *filename, bool force_update)
{
    return NULL;
}

#endif /* HAVE_CURL */


bool
is_downloadable (const char *url)
{
    return (url && ((strncasecmp (url, "http://", 7) == 0)
                    || (strncasecmp (url, "ftp://", 6) == 0)));
}


int
download_get_percent (void *p)
{
#ifdef HAVE_CURL
    return (int) percent;
#else
    return 0;
#endif
}


void
downloader_init (void)
{
#ifdef HAVE_CURL
    curl_global_init (CURL_GLOBAL_DEFAULT);
#endif
}


void
downloader_free (void)
{
#ifdef HAVE_CURL
    curl_global_cleanup ();
#endif
}
