/* gkrellm-bluez - A BlueZ monitor plug-in for GKrellM2
 *
 * Copyright (C) 2006 Ludovic Cintrat <lcintrat@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "gkrellm-bluez.h"

#include "gkrellm-bluez-alert.h"
#include "gkrellm-bluez-monitor.h"

#include "gkrellm-bluez-preferences.h"


#define TEXT_HIDE_KEYWORD    "text_hide"
#define TEXT_FORMAT_KEYWORD  "text_format"

#define CHART_ENABLE_KEYWORD "enabled"
#define CHART_FORCE_KEYWORD  "forced"

#define LAUNCHER_CMD_KEYWORD "launch"
#define LAUNCHER_TIP_KEYWORD "tooltip"


static GtkWidget *format_combo;


/*  prototypes  */

static void enable_button_toggled_callback (GtkToggleButton *togglebutton,
                                            gpointer         user_data);
/* static void alert_button_clicked_callback  (GtkWidget       *button, */
/*                                             gpointer         user_data); */


/*  public functions  */

void
gkrellm_bluez_preferences_show (GtkWidget *tabs_vbox)
{
  GkrellmBlueZMonitor *bluezmon;
  GtkWidget          *tabs;
  GtkWidget          *vbox;
  GtkWidget          *hbox;
  GtkWidget          *frame;
  GtkWidget          *text;
  GtkWidget          *label;
  gchar              *tmp;
  GList              *format_list = NULL;
  GList              *list;
  GtkWidget          *table;
  gchar              *info[] = {
    "<h>BlueZ Monitor\n",
    N_("This plug-in monitors the Bluetooth adapters in your computer and\n"),
    N_("displays a graph of the rx/tx bytes for each adapter.\n"),
    "\n",
    N_("The plug-in requires the kernel to have support for the Linux BlueZ\n"),
    N_("stack (CONFIG_BT).\n"),
    "\n",
    N_("<h>Chart Labels\n"),
    N_("Substitution variables for the chart labels format string:\n"),
    N_("\t$M\tmaximum chart value\n"),
    N_("\t$T\treceive + transmit bytes\n"),
    N_("\t$r\treceive bytes\n"),
    N_("\t$t\ttransmit bytes\n"),
    N_("\t$O\tcumulative receive + transmit bytes\n"),
    N_("\t$i\tcumulative receive bytes\n"),
    N_("\t$o\tcumulative transmit bytes\n"),
    N_("\t$c\tnumber of connections\n"),
    N_("\t$l\tthe hci interface name\n"),
  };

  g_assert (tabs_vbox != NULL);

  tabs = gtk_notebook_new ();
  gtk_box_pack_start (GTK_BOX (tabs_vbox), tabs, TRUE, TRUE, 0);

  /*  device tabs  */
  for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
      bluezmon = list->data;

      vbox = gkrellm_gtk_framed_notebook_page (tabs, bluezmon->interface);

      hbox = gtk_hbox_new (FALSE, 0);

      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);

      tmp = g_strdup_printf (_("Enable %s"), bluezmon->interface);

      bluezmon->enable_button = gtk_check_button_new_with_label (tmp);

      g_free (tmp);

      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (bluezmon->enable_button),
                                    bluezmon->enabled);

      g_signal_connect (bluezmon->enable_button,
                        "toggled",
                        G_CALLBACK (enable_button_toggled_callback),
                        bluezmon);

      gtk_box_pack_start (GTK_BOX (hbox), bluezmon->enable_button, FALSE, FALSE, 0);

      /* TODO: implement alerts */

/*       gkrellm_gtk_alert_button (hbox, &bluezmon->alert_button, */
/*                                 FALSE, FALSE, 4, FALSE, */
/*                                 alert_button_clicked_callback, bluezmon); */

/*       gtk_widget_set_sensitive (bluezmon->alert_button, bluezmon->enabled); */

      hbox = gtk_hbox_new (FALSE, 0);

      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

      bluezmon->force_button = gtk_check_button_new_with_label (_("Force chart to be shown even "
								  "if interface is not detected"));

      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (bluezmon->force_button),
                                    bluezmon->forced);

      gtk_widget_set_sensitive (bluezmon->force_button, bluezmon->enabled);

      gtk_box_pack_start (GTK_BOX (hbox), bluezmon->force_button, FALSE, FALSE, 0);

      frame = gkrellm_gtk_framed_vbox_end (vbox, _("Launch Command"),
                                           4, FALSE, 0, 2);

      table = gkrellm_gtk_launcher_table_new (frame, 1);

      gkrellm_gtk_config_launcher (table, 0,
                                   &bluezmon->command_entry,
                                   &bluezmon->tooltip_entry,
                                   bluezmon->interface,
                                   &bluezmon->launcher);

      gtk_widget_set_sensitive (bluezmon->command_entry, bluezmon->enabled);

      gtk_widget_set_sensitive (bluezmon->tooltip_entry, bluezmon->enabled);
    }

  /*  setup tab  */
  vbox = gkrellm_gtk_framed_notebook_page (tabs, _("Setup"));

  frame = gkrellm_gtk_framed_vbox (vbox, _("Chart Labels Format String"),
                                   4, FALSE, 0, 4);

  format_combo = gtk_combo_new ();

  format_list = g_list_append (format_list, GKRELLM_BLUEZ_FORMAT_STRING1);
  format_list = g_list_append (format_list, GKRELLM_BLUEZ_FORMAT_STRING2);
  format_list = g_list_append (format_list, GKRELLM_BLUEZ_FORMAT_STRING3);
  format_list = g_list_append (format_list, GKRELLM_BLUEZ_FORMAT_STRING4);
  format_list = g_list_append (format_list, GKRELLM_BLUEZ_FORMAT_STRING5);
  format_list = g_list_append (format_list, GKRELLM_BLUEZ_FORMAT_STRING6);

  gtk_combo_set_popdown_strings (GTK_COMBO (format_combo), format_list);

  g_list_free (format_list);

  gtk_box_pack_start (GTK_BOX (frame), format_combo, TRUE, TRUE, 0);

  /*  info tab  */
  vbox = gkrellm_gtk_framed_notebook_page (tabs, _("Info"));

  text = gkrellm_gtk_scrolled_text_view (vbox, NULL, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  gkrellm_gtk_text_view_append_strings (text, info, sizeof (info) / sizeof (gchar *));

  /*  about tab  */
  vbox = gkrellm_gtk_framed_notebook_page (tabs, _("About"));

  tmp = g_strdup_printf (_("%s %s\n"
                           "GKrellM2 BlueZ Monitor plug-in for Linux\n\n"
                           "Copyright (C) 2006 Ludovic Cintrat <lcintrat@users.sourceforge.net>\n\n"
                           "Released under the GNU General Public License"),
                         GKRELLM_BLUEZ_PLUGIN_NAME, PACKAGE_VERSION);

  label = gtk_label_new (tmp);
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);

  g_free (tmp);
}

void
gkrellm_bluez_preferences_apply (void)
{
  GkrellmBlueZMonitor *bluezmon;
  GList   *list;

  if (gkrellm_bluez_format_string)
    g_free (gkrellm_bluez_format_string);

  gkrellm_bluez_format_string = g_strdup (gkrellm_gtk_entry_get_text (&GTK_COMBO (format_combo)->entry));

  for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
      bluezmon = list->data;

      bluezmon->enabled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bluezmon->enable_button));
      bluezmon->forced  = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bluezmon->force_button));

      if (bluezmon->chart)
        {
          gkrellm_apply_launcher (&bluezmon->command_entry,
                                  &bluezmon->tooltip_entry,
                                  bluezmon->chart->panel,
                                  &bluezmon->launcher,
                                  gkrellm_launch_button_cb);

          gkrellm_refresh_chart (bluezmon->chart);
        }
      else
        {
          if (bluezmon->launcher.command)
            g_free (bluezmon->launcher.command);

          bluezmon->launcher.command = g_strdup (gkrellm_gtk_entry_get_text (&bluezmon->command_entry));

          if (bluezmon->launcher.tooltip_comment)
            g_free (bluezmon->launcher.tooltip_comment);

          bluezmon->launcher.tooltip_comment = g_strdup (gkrellm_gtk_entry_get_text (&bluezmon->tooltip_entry));
        }
    }
}

void
gkrellm_bluez_preferences_save (FILE *file)
{
  GkrellmBlueZMonitor *bluezmon;
  GList   *list;
  gchar   *devid;

  g_assert (file != NULL);

  for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
      bluezmon = list->data;

      devid = g_strdup_printf ("%d", bluezmon->devid);

      gkrellm_save_chartconfig (file, bluezmon->config,
                                GKRELLM_BLUEZ_PLUGIN_KEYWORD, devid);

      gkrellm_save_alertconfig (file, bluezmon->alert,
                                GKRELLM_BLUEZ_PLUGIN_KEYWORD, devid);

      g_free (devid);

      fprintf (file, "%s %s %d %d\n",
               GKRELLM_BLUEZ_PLUGIN_KEYWORD,
               TEXT_HIDE_KEYWORD,
               bluezmon->devid,
               bluezmon->hide_text);

      fprintf (file, "%s %s %d %d\n",
               GKRELLM_BLUEZ_PLUGIN_KEYWORD,
               CHART_ENABLE_KEYWORD,
               bluezmon->devid,
               bluezmon->enabled);

      fprintf (file, "%s %s %d %d\n",
               GKRELLM_BLUEZ_PLUGIN_KEYWORD,
               CHART_FORCE_KEYWORD,
               bluezmon->devid,
               bluezmon->forced);

      if (bluezmon->launcher.command &&
          strlen (bluezmon->launcher.command) > 0)
        {
          fprintf (file, "%s %s %d %s\n",
                   GKRELLM_BLUEZ_PLUGIN_KEYWORD,
                   LAUNCHER_CMD_KEYWORD,
                   bluezmon->devid,
                   bluezmon->launcher.command);
        }

      if (bluezmon->launcher.tooltip_comment &&
          strlen (bluezmon->launcher.tooltip_comment) > 0)
        {
          fprintf (file, "%s %s %d %s\n",
                   GKRELLM_BLUEZ_PLUGIN_KEYWORD,
                   LAUNCHER_TIP_KEYWORD,
                   bluezmon->devid,
                   bluezmon->launcher.tooltip_comment);
        }
    }

  if (gkrellm_bluez_format_string)
    {
      fprintf (file, "%s %s %s\n",
               GKRELLM_BLUEZ_PLUGIN_KEYWORD,
               TEXT_FORMAT_KEYWORD,
               gkrellm_bluez_format_string);
    }
}

void
gkrellm_bluez_preferences_load (gchar *line)
{
  GkrellmBlueZMonitor  *bluezmon;
  gchar    *keyword;
  gchar    *data;
  gint      devid;
  gchar    *config;
  gboolean  choice;
  gint      items;

  g_assert (line != NULL);

  keyword   = g_new (gchar, strlen (line) + 1);
  data      = g_new (gchar, strlen (line) + 1);
  config    = g_new (gchar, strlen (line) + 1);

  items = sscanf (line, "%s %[^\n]", keyword, data);

  if (items == 2)
    {
      if (g_ascii_strcasecmp (keyword, GKRELLM_CHARTCONFIG_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %[^\n]", &devid, config);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);

              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              gkrellm_load_chartconfig (&bluezmon->config, config, 1);
            }
        }
      if (g_ascii_strcasecmp (keyword, GKRELLM_ALERTCONFIG_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %[^\n]", &devid, config);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);

              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              if (! bluezmon->alert)
                gkrellm_bluez_alert_create (bluezmon);

              gkrellm_load_alertconfig (&bluezmon->alert, config);
            }
        }
      else if (g_ascii_strcasecmp (keyword, TEXT_HIDE_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %d\n", &devid, &choice);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);

              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              bluezmon->hide_text = choice;
            }
        }
      else if (g_ascii_strcasecmp (keyword, CHART_ENABLE_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %d\n", &devid, &choice);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);

              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              bluezmon->enabled = choice;
            }
        }
      else if (g_ascii_strcasecmp (keyword, CHART_FORCE_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %d\n", &devid, &choice);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);


              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              bluezmon->forced = choice;
            }
        }
      else if (g_ascii_strcasecmp (keyword, LAUNCHER_CMD_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %[^\n]", &devid, config);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);

              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              if (bluezmon->launcher.command)
                g_free (bluezmon->launcher.command);

              bluezmon->launcher.command = g_strdup (config);
            }
        }
      else if (g_ascii_strcasecmp (keyword, LAUNCHER_TIP_KEYWORD) == 0)
        {
          items = sscanf (data, "%d %[^\n]", &devid, config);

          if (items == 2)
            {
              bluezmon = gkrellm_bluez_monitor_find (devid);

              if (! bluezmon)
                bluezmon = gkrellm_bluez_monitor_create (devid);

              if (bluezmon->launcher.tooltip_comment)
                g_free (bluezmon->launcher.tooltip_comment);

              bluezmon->launcher.tooltip_comment = g_strdup (config);
            }
        }
      else if (g_ascii_strcasecmp (keyword, TEXT_FORMAT_KEYWORD) == 0)
        {
          if (gkrellm_bluez_format_string)
            g_free (gkrellm_bluez_format_string);

          gkrellm_bluez_format_string = g_strdup (data);
        }
    }

  g_free (keyword);
  g_free (data);
  g_free (config);
}


/*  private functions  */

static void
enable_button_toggled_callback (GtkToggleButton *togglebutton,
                                gpointer         user_data)
{
  GkrellmBlueZMonitor *bluezmon;
  gboolean            sensitive;

  g_assert (togglebutton != NULL);
  g_assert (user_data != NULL);

  bluezmon = user_data;

  sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bluezmon->enable_button));

  gtk_widget_set_sensitive (bluezmon->force_button, sensitive);
  gtk_widget_set_sensitive (bluezmon->alert_button, sensitive);
  gtk_widget_set_sensitive (bluezmon->command_entry, sensitive);
  gtk_widget_set_sensitive (bluezmon->tooltip_entry, sensitive);
}

/* static void */
/* alert_button_clicked_callback (GtkWidget *button, */
/*                                gpointer   user_data) */
/* { */
/*   GkrellmBlueZMonitor *bluezmon; */

/*   g_assert (button != NULL); */
/*   g_assert (user_data != NULL); */

/*   bluezmon = user_data; */

/*   if (! bluezmon->alert) */
/*     gkrellm_bluez_alert_create (bluezmon); */

/*   gkrellm_alert_config_window (&bluezmon->alert); */
/* } */
