/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "urtreepararray.h"

#ifdef PARALLEL

/**
	constructor, which duplicates the tree mpi_size times
*/
UrTreeParArray::UrTreeParArray(UrTree &tree) {
	setLimit(mpi_size);
	for (int cnt = 0; cnt < mpi_size; cnt++) {
		UrTreePar *item = new UrTreePar(tree);
		add(item);
	}
	responded_index = 0;
	responded_procs.set(mpi_size, 0);
}

/**
	constructor
*/
UrTreeParArray::UrTreeParArray() {
	setLimit(mpi_size);
	for (int cnt = 0; cnt < mpi_size; cnt++) {
		UrTreePar *item = new UrTreePar();
		add(item);
	}
	responded_index = 0;
	responded_procs.set(mpi_size, 0);
}


/**
	start to receive from all slaves
*/
void UrTreeParArray::receiveFromSlaves() {
	for (int cnt = 0; cnt < mpi_size; cnt ++) {
		if (cnt != mpi_master_rank) {
			getItem(cnt)->receiveFromProcess(cnt);
		}
	}
}

/**
	wait until some slaves send tree back
	@return the rank of completed process
	
*/
int UrTreeParArray::waitForSlaves() {

	if (responded_procs.getSize() > responded_index) {
		return responded_procs[responded_index++];
	}
	int cnt;
	responded_index = 0;
	responded_procs.setSize(0);
	do {
		for (cnt = 0; cnt < mpi_size; cnt++) 
		if (cnt != mpi_master_rank)	{
			int flag;
			MPI_Status status;
			MPI_Test(getItem(cnt)->getReceiveRequest(), &flag, &status);
			if (flag) responded_procs.add(cnt);
		}
		if (responded_procs.getSize() == 0) {
			usleep(100000); // sleep for 0.1 seconds before scanning again
		}
	} while (responded_procs.getSize() == 0);
	//if (responded_procs.getSize() > 1) 
	//	cout << responded_procs.getSize() << " workers responded" << endl;
	return responded_procs[responded_index++];
}

/**
	wait until some slaves send tree back
	@param tree_index (OUT) the indices of completed process 
	@return the number of completed process
	
*/
/*
int UrTreeParArray::testSomeSlaves(int *tree_index) {

	MPI_Request *requests = new MPI_Request[mpi_size];
	int count = mpi_size;
	for (int i = 0; i < mpi_size; i++)
		requests[i] = *getItem(i)->getReceiveRequest();

	int index;
	int tree_count;
	MPI_Testsome(count, requests, &tree_count, tree_index, MPI_STATUSES_IGNORE);
	delete [] requests;	
	if (tree_count == MPI_UNDEFINED)
		Utl::announceError("testSomeSlaves failed!");
	return tree_count;
}
*/

/**
	broadcast a tree to all slaves
	@param process the process which creates the tree
*/
void UrTreeParArray::broadcastToSlaves(UrTree &tree, int process) {
	for (int cnt = 0; cnt < mpi_size; cnt++)
		if (cnt != mpi_master_rank && cnt != process) {

			// cancel the sending request if necessary
			if (*getItem(cnt)->getSendRequest() != MPI_REQUEST_NULL) {
				int flag;
				MPI_Status status;
				MPI_Test(getItem(cnt)->getSendRequest(), &flag, &status);
				if (!flag)
					MPI_Cancel(getItem(cnt)->getSendRequest());
			}

			// now do non blocking send to slave
			getItem(cnt)->clone(tree);
			getItem(cnt)->sendToProcess(cnt);
		}
}

/**
	cancel all the attached receive or send of all elements
*/
void UrTreeParArray::cancelCommunications() {
	for (int cnt = 0; cnt < mpi_size; cnt++)
		getItem(cnt)->cancelCommunications();
}

/**
	return TRUE of all slaves done their job
*/
bool UrTreeParArray::allSlavesDone() {
	for (int cnt = 0; cnt < mpi_size; cnt++)
		if (cnt != mpi_master_rank && *getItem(cnt)->getReceiveRequest() != MPI_REQUEST_NULL) {
			return false;
		}
	return true;
}

/**
	return the number of running slaves
*/
int UrTreeParArray::runningSlaves() {
	int result = 0;
	for (int cnt = 0; cnt < mpi_size; cnt++)
		if (cnt != mpi_master_rank && *getItem(cnt)->getReceiveRequest() != MPI_REQUEST_NULL) {
			result++;
		}
	return result;
	
}


UrTreeParArray::~UrTreeParArray() {
	for (int cnt = mpi_size-1; cnt >= 0; cnt--) {
		delete getItem(cnt);
	}
}


#endif // PARALLEL
