// Button.hh for BBCD - a CD player for X11 / BlackBox
// Copyright (c) 2002 Bertrand Duret <bertrand.duret at libertysurf.fr>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef __BUTTON_HH
#define __BUTTON_HH

#include <X11/Xlib.h>

#include "Texture.hh"

// Forward declaration
class ButtonDrawer;


// A class to manage a button
class Button
{
public:
  Button(unsigned int width, unsigned int height, int x, int y,
      Display* display, Window parent, ButtonDrawer* bd);
  ~Button();

  inline void resize(unsigned int w, unsigned int h)
    { width = w; height = h; XResizeWindow(display, buttonW, width, height); }
  inline void move(int mx, int my)
    { x = mx; y = my; XMoveWindow(display, buttonW, x, y); }

  inline unsigned int getWidth() const { return width; }
  inline unsigned int getHeight() const { return height; }
  inline int getX() const { return x; }
  inline int getY() const { return y; }
  inline Window getXWindow() const { return buttonW; }
  inline bool containsPoint(int px, int py) const
    { return px>=x && px<x+(long)width && py>=y && py<y+(long)height; }

  inline void setDrawer(ButtonDrawer* bd) { bDrawer = bd; }
  inline ButtonDrawer* getDrawer() const { return bDrawer; }
  void draw(bool pressed = false, bool forced = false); // Call drawer::draw
  inline void setPressed(bool p) { pressed = p; }
  inline bool isPressed() const { return pressed; }

  inline Pixmap getPixmap() const { return background; }
  inline void setPixmap(Pixmap pm)
    { XSetWindowBackgroundPixmap(display, buttonW, pm);
      background = pm; }

protected:
  Display* display;
  Window buttonW;
  Pixmap background;
  unsigned int width, height;
  int x, y;
  ButtonDrawer* bDrawer;
  bool pressed;
}; // class Button




// Kind of operator, to draw the face of a button
class ButtonDrawer{
public:
  // A ButtonDrawer may draw a button in a released or pressed state
  ButtonDrawer(Display*, GC*, GC*, BTexture*, BTexture*);
  virtual ~ButtonDrawer();
  virtual void draw(Button*, bool pressed = false, bool = false) const;
protected:
  Display* display;
  GC* gc, * gca;
  BTexture* releasedT, * pressedT;
}; // class ButtonDrawer


#endif // __BUTTON_HH
