/* callbacks.c
 * Copyright (C) 2004, 2005 Sylvain Cresto <scresto@gmail.com>
 *
 * This file is part of graveman!
 *
 * graveman! is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or
 * (at your option) any later version.
 *
 * graveman! is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with program; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * URL: http://www.nongnu.org/graveman/
 *
 */

#include "graveman.h"

/* affichage boite de dialogue au sujet de ... */
void on_about (GtkMenuItem *Amenuitem, gpointer Adata)
{
  about_graveman();
}


gboolean on_closeabout (GtkWidget *Awidget, GdkEventButton *Aevent, gpointer Adata)
{
  return FALSE;
}

/* callback appele pour l'ajout d'une piste audio */
void _add_a_piste(gchar *Anom, GtkListStore *Aliststore, Tgrave *Ag)
{
  guint32 *Laudiolen = (guint32 *)sc_grave_get_data(Ag, "audiolen");
  GtkTreeIter Liter;
  gchar *LTitle = NULL, *LAlbum = NULL, *LArtist = NULL;
  guint32 Llennbr = 0;
  gchar *Lfullnameutf8 = NULL, *Lnameutf8 = NULL, *Llengthutf8 = NULL;
  gchar *Lname;
  gboolean Lstatus = FALSE;

  /* on reconnai le type fichier grace a son extension */

#ifdef ENABLE_MP3
  if (GsupportMp3 && sc_str_has_casesuffix(Anom, MP3)) {
    Lstatus = getMp3Info(Anom, &LTitle, &LAlbum, &LArtist, &Llennbr, NULL);
  } else
#endif    
#ifdef ENABLE_OGG
  if (GsupportOgg && sc_str_has_casesuffix(Anom, OGG)) {
    Lstatus = getOggInfo(Anom, &LTitle, &LAlbum, &LArtist, &Llennbr, NULL);
  } else
#endif
#ifdef ENABLE_FLAC
  if (GsupportFlac && sc_str_has_casesuffix(Anom, FLAC)) {
    Lstatus = getFlacInfo(Anom, &Llennbr, NULL);
  } else
#endif
  if (sc_str_has_casesuffix(Anom, WAV)) {
    Lstatus = getWavInfo(Anom, &Llennbr, NULL);
  }
  if (Lstatus == FALSE) {
    /* erreur format inconnue ou incorecte */
    gchar *Ltmp;
    GtkWidget *Lerrdiag;
    
    Lnameutf8 = _UTF8(Anom);
    Ltmp = g_strdup_printf(_("Cannot add track '%s', this is an unknown or invalid audio file."),
        Lnameutf8);
    g_free(Lnameutf8);

    Lerrdiag = gtk_message_dialog_new(NULL,
                        GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_CANCEL, Ltmp);
    gtk_dialog_run(GTK_DIALOG(Lerrdiag));
    gtk_widget_destroy(Lerrdiag);

    g_free(Ltmp);

    return;
  }

  if ((!LTitle && !LAlbum && !LArtist) ||
      (LTitle && LAlbum && LArtist && !*LTitle && !*LAlbum && !*LArtist)) {
    gchar *slash = strrchr(Anom, '/');
    Lname = g_strdup(slash ? ++slash : Anom);
    Lnameutf8 = _UTF8(Lname);
  } else {
    Lname = NULL;
    Lnameutf8 = g_strdup_printf("%s - %s - %s", LArtist, LAlbum, LTitle);
  }  
  *(Laudiolen)=*(Laudiolen) + Llennbr;
  gtk_list_store_append(Aliststore, &Liter);
  Llengthutf8 = format_length(Llennbr);
  Lfullnameutf8 = _UTF8(Anom);

  gtk_list_store_set(Aliststore, &Liter, 0, TYPE_MP3, 1, get_image("AudioIcon"), 2, Lnameutf8, 3,
                      Llengthutf8, 4, Lfullnameutf8, 5, Anom, 6, Llennbr, -1);
  g_free(Lfullnameutf8); g_free(Llengthutf8); g_free(Lnameutf8); g_free(Lname);

  g_free(LArtist);
  g_free(LAlbum); g_free(LTitle);
}

/* callback appele pour l'ajout de chaques fichiers */
gint _add_a_file(gchar *Anom, GtkTreeStore *ATreeStore, GtkTreeIter *Aparent, Tgrave *Ag, guint64 *Adatasize, GtkTreeIter *Aiter)
{
  GtkTreeIter Liter;
  gchar *LOnlyName;
  struct stat Lfi;
  GDir *Lgdir;
  const gchar *Ldirname;
  gchar *Lfullname;
  gchar *Lonlynameutf8, *Lnomutf8;
  gchar *Lformatsizeutf8;
  gboolean Lret;

  if ((LOnlyName=strrchr(Anom, '/'))) {
    LOnlyName++;
  } else {
    LOnlyName = Anom;
  }

  if (stat(Anom, &Lfi) == -1) {
    _WARN("erreur lecture fichier '%s' err [%d]", Anom, errno);
    return errno;
  }

  gtk_tree_store_append(ATreeStore, &Liter, Aparent); 

  if (S_ISDIR(Lfi.st_mode)) {
    /* si c'est un repertoire alors on lis tout son contenu et on ajoute
     * les fichiers qu'il contient */

    Lonlynameutf8 = _UTF8(LOnlyName);
    Lnomutf8 = _UTF8(Anom);
    gtk_tree_store_set(ATreeStore, &Liter, 0, TYPE_DIR, 1, get_image("DirIcon"), 2, Lonlynameutf8, 3, "", 4, Lnomutf8, 5, LOnlyName, 6, 0, -1);
    g_free(Lnomutf8);
    g_free(Lonlynameutf8);

    Lgdir = g_dir_open(Anom, 0, NULL);
    if (!Lgdir) {
      _WARN("erreur ouverture repertoire '%s'", Anom);
      return errno;
    }

    while ((Ldirname = g_dir_read_name(Lgdir))) {
      if (!strcmp(Ldirname, ".") || !strcmp(Ldirname, "..")) continue;

      Lfullname = g_strdup_printf("%s/%s", Anom, Ldirname);
      Lret = _add_a_file(Lfullname, ATreeStore, &Liter, Ag, Adatasize, NULL);
      g_free(Lfullname);

      if (Lret == ELOOP || Lret == EMLINK) break;
    }

    g_dir_close(Lgdir);
    
  } else {
    Lonlynameutf8 = _UTF8(LOnlyName);
    Lnomutf8 = _UTF8(Anom);
    Lformatsizeutf8 = format_size(Lfi.st_size);
    gtk_tree_store_set(ATreeStore, &Liter, 0, TYPE_FILE, 1, get_image("FileIcon"), 2, Lonlynameutf8,
                                           3, Lformatsizeutf8, 4, Lnomutf8, 5, Anom, 6, Lfi.st_size, -1);
    g_free(Lformatsizeutf8);
    g_free(Lnomutf8);
    g_free(Lonlynameutf8);

    *(Adatasize)=*(Adatasize) + Lfi.st_size;
  }

  if (Aiter) *Aiter = Liter;

  return Lret;
}

/* creer un repertoire */
void gtk_data_creerrepertoire(GtkWidget *btn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(Lliste);
  GtkTreeIter Liter, Lparent;
  GtkWidget *Lboite, *Lentry;
  GtkTreePath *Lpath = NULL;
  const gchar *Lname;

  /* on recupere l'element selectionne */
  gtk_tree_view_get_cursor(Lliste, &Lpath, NULL);
  if (!Lpath) return;
  gtk_tree_model_get_iter(Ltreemodel, &Lparent, Lpath);
  gtk_tree_path_free(Lpath);

  Lboite= gtk_dialog_new_with_buttons(_("Enter the new directory name"), NULL, GTK_DIALOG_MODAL,
                  GTK_STOCK_OK,GTK_RESPONSE_OK, GTK_STOCK_CANCEL,GTK_RESPONSE_CANCEL, NULL);
  
  Lentry = gtk_entry_new();

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Lboite)->vbox), Lentry, TRUE, FALSE, 0);
  gtk_widget_show_all(GTK_DIALOG(Lboite)->vbox);

  if (gtk_dialog_run(GTK_DIALOG(Lboite)) == GTK_RESPONSE_OK) {
    /* creation d'un repertoire */
    Lname = gtk_entry_get_text(GTK_ENTRY(Lentry));
    if (!Lname || !*Lname) return;

    gtk_tree_store_prepend(GTK_TREE_STORE(Ltreemodel), &Liter, &Lparent); 
    gtk_tree_store_set(GTK_TREE_STORE(Ltreemodel), &Liter, 0, TYPE_DIR, 1, get_image("DirIcon"), 2, Lname, 3, "", 4, "", 5, 0, -1);

    /* on developpe pour qu'il soit visible */
    if ((Lpath = gtk_tree_model_get_path(Ltreemodel, &Liter))) {
      gtk_tree_view_expand_to_path(Lliste, Lpath);
      gtk_tree_path_free(Lpath);
    }
  }
  gtk_widget_destroy(Lboite); 
}

/* developper tout le contenu de l'arbre */
void gtk_data_expandelement(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Llist = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));

  if (!GTK_IS_TREE_VIEW(Llist)) return;

  expand_list_icons(Llist, TRUE);
}

gboolean callback_simpledialog_eventkeyrelease(GtkWidget *Aentry, GdkEventKey *Aevent, gpointer Adata)
{
  GtkDialog *Ldialog = GTK_DIALOG(Adata);

  if (!Aevent) return FALSE;
  if (Aevent->keyval == GDK_Return) {
    gtk_dialog_response(Ldialog, GTK_RESPONSE_OK);
    return TRUE;
  }

  return FALSE;
}

/* renommer un element (fichier ou repertoire) */
void callbacks_data_renameitem(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
  GtkWidget *Lboite, *Lentry;
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(Lliste);
  GtkTreePath *Lpath = NULL;
  gchar *Ltitle;
  gchar *Lname;
  const gchar *Lselname;
  GtkTreeIter Liter;

  /* on recupere le nom de l'element selectionne */
  gtk_tree_view_get_cursor(Lliste, &Lpath, NULL);
  if (!Lpath) return;
  gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
  gtk_tree_path_free(Lpath);

  gtk_tree_model_get(Ltreemodel, &Liter, 2, &Lname, -1);

  Ltitle = g_strdup_printf(Ltitle, _("Enter the new name for %s"), Lname);
 
  Lboite = gtk_dialog_new_with_buttons(Ltitle, NULL, GTK_DIALOG_MODAL,
        GTK_STOCK_OK,GTK_RESPONSE_OK, GTK_STOCK_CANCEL,GTK_RESPONSE_CANCEL, NULL);
  g_free(Ltitle);
  gtk_dialog_set_default_response(GTK_DIALOG(Lboite), GTK_RESPONSE_OK);

  Lentry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(Lentry), Lname);
  g_free(Lname);
  g_signal_connect(G_OBJECT(Lentry), "key-release-event", G_CALLBACK(callback_simpledialog_eventkeyrelease), Lboite);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Lboite)->vbox), Lentry, TRUE, FALSE, 0);
  gtk_widget_show_all(GTK_DIALOG(Lboite)->vbox);

  if (gtk_dialog_run(GTK_DIALOG(Lboite)) == GTK_RESPONSE_OK) {
    Lselname = gtk_entry_get_text(GTK_ENTRY(Lentry));
    if (Lselname && *Lselname) {

      /* on definie le nouveau nom dans la liste */
      gtk_tree_store_set(GTK_TREE_STORE(Ltreemodel), &Liter, 2, Lselname, -1);
    }
  }

  gtk_widget_destroy(Lboite); 
}

/* retourne l'element selectionne ou bien le premier si aucun n'est
 * selectionne */
void get_parent_iter(GtkTreeView *Aliste, GtkTreeIter *Aiter, gint *Anb, gboolean *Aisroot)
{
  GtkTreeModel *LTreeData = gtk_tree_view_get_model(GTK_TREE_VIEW(Aliste));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Aliste));
  GList *Lfiles = NULL;

  *(Aisroot) = 0;
  *(Anb) = 0;
  gtk_tree_model_get_iter_from_string(LTreeData, Aiter, "0");

  if (Lselection) {
    if (!gtk_tree_selection_iter_is_selected(Lselection, Aiter)) {
      *(Anb) = gtk_tree_selection_count_selected_rows(Lselection);
      Lfiles = gtk_tree_selection_get_selected_rows(Lselection, &LTreeData);
    } else {
      *(Aisroot)=1;
    }
  }

  if (Lfiles) {
    gtk_tree_model_get_iter(LTreeData, Aiter, (GtkTreePath*)(Lfiles->data));
    g_list_free(Lfiles);
  }
}

/* enlever des pistes a graver */
void gtk_audio_removetrack(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "LISTEAUDIO"));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
  GtkProgressBar *Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "audiolenbar"));
  guint32 *Laudiolen = (guint32 *)sc_grave_get_data(Lg, "audiolen");
  guint32 *Laudiomaxlen = (guint32 *)sc_grave_get_data(Lg, "audiomaxlen");
  GtkTreeIter Liter;
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(Lliste);
  guint32 Llennbr = 0;
  GList *Ltracks = gtk_tree_selection_get_selected_rows(Lselection, &Ltreemodel);
  GList *Lrefs = NULL, *Lnode = NULL;
  gpointer Lproxy = g_object_newv(G_TYPE_OBJECT, 0, NULL);

  for (Lnode = Ltracks; Lnode; Lnode = Lnode->next) {
    Lrefs = g_list_append(Lrefs, gtk_tree_row_reference_new_proxy(Lproxy, Ltreemodel, Lnode->data));
    gtk_tree_path_free(Lnode->data);
  }
  g_list_free(Ltracks);

  for (Lnode = Lrefs; Lnode; Lnode = Lnode->next) {
    GtkTreePath *Lpath = gtk_tree_row_reference_get_path(Lnode->data);
    gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
 
    gtk_tree_model_get(Ltreemodel, &Liter, 6, &Llennbr, -1);
    gtk_list_store_remove(GTK_LIST_STORE(Ltreemodel), &Liter);

    *Laudiolen-=Llennbr;
    g_assert((*Laudiolen) >= 0);

    gtk_tree_row_reference_deleted(Lproxy, Lpath);
    gtk_tree_path_free(Lpath);
    gtk_tree_row_reference_free(Lnode->data);
  }
 
  g_list_free(Lrefs);
  g_object_unref(Lproxy);

  switch_pbar_status(Lcontenance, *Laudiomaxlen, *Laudiolen, PB_UPDATE_AUDIO);
}

/* parcourir tous les fils d'un element et recuperer leur taille que l'on enleve de "cursize" */
void remove_children_size(GtkTreeModel *Atreedata, GtkTreeIter *Aiter, guint64 *Acursize, Tgrave *Ag);
void remove_children_size(GtkTreeModel *Atreedata, GtkTreeIter *Aiter, guint64 *Acursize, Tgrave *Ag)
{
  GtkTreeIter Lcuriter;
  guint64 Lsizenbr = 0;
  int Ltype;

  if (gtk_tree_model_iter_children(Atreedata, &Lcuriter, Aiter)) {
    do {
      gtk_tree_model_get(Atreedata, &Lcuriter, 0, &Ltype, 6, &Lsizenbr, -1);
      *Acursize-=Lsizenbr;
#if 0
      if (Ltype == TYPE_IMAGEBOOT) {
        g_hash_table_replace(Lg, "file_imageboot", gtk_tree_iter_copy(&Lnewiter));
      }
#endif

      remove_children_size(Atreedata, &Lcuriter, Acursize, Ag);
    } while (gtk_tree_model_iter_next(Atreedata, &Lcuriter));
  }
}

/* clique sur le popup pour la  mise a jour de la taille des icones utilisees */
void gtk_display_updatesize(GtkWidget *Abtn, gpointer Adata)
{
  const gchar *Lname = glade_get_widget_name(Abtn);
  Tgrave *Lg = (Tgrave *) Adata;
  gchar *Lsize = strrchr(Lname, '_');

  g_assert(Lsize);

  conf_store_value("iconsize", ++Lsize);

  manage_interface(Lg, MENU_UPDATE);
}

/* clique sur le popup pour l'affichage ou non de la status bar */
void gtk_display_switchstatusbar(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *) Adata;
  gboolean Ldispsb = conf_get_boolean("statusbar");

  conf_store_value("statusbar", Ldispsb ? "0" : "1");

  update_status_bar_display(Lg);
}

/* enlever des fichiers a graver */
void gtk_data_removefiles(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
  GtkProgressBar *Lcontenance;
  guint64 *Ldatasize;
  guint64 *Ldatamaxsize;

  _REMPLISSAGE_VALUE(Lg, Lliste, Ldatasize, Ldatamaxsize, Lcontenance);
  
  /* image de boot */
#if 0  
  GtkTreeIter *Literboot = sc_grave_get_data(Lg, "file_imageboot");
#endif
  GtkTreeModel *Ltreedata = gtk_tree_view_get_model(GTK_TREE_VIEW(Lliste));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
  GList *Lfiles = gtk_tree_selection_get_selected_rows(Lselection, &Ltreedata);
  GList *Lrefs = NULL;
  GList *Lnode = NULL;
  gint Ltype;
  guint32 Lsizenbr = 0;
  GtkTreeIter Liter;
  gpointer Lproxy = g_object_newv(G_TYPE_OBJECT, 0, NULL);

  for (Lnode = Lfiles; Lnode; Lnode = Lnode->next) {
    Lrefs = g_list_append(Lrefs, gtk_tree_row_reference_new_proxy(Lproxy, Ltreedata, Lnode->data));
    gtk_tree_path_free(Lnode->data); /* Free unneeded tree path */
  }
  g_list_free(Lfiles); /* Emptied list */

  for (Lnode = Lrefs; Lnode; Lnode = Lnode->next) {
    GtkTreePath *Lpath = gtk_tree_row_reference_get_path(Lnode->data);

    if (Lpath) {
      gtk_tree_model_get_iter(Ltreedata, &Liter, Lpath);

      gtk_tree_model_get(Ltreedata, &Liter, 0, &Ltype, 6, &Lsizenbr, -1);

      /* on enleve pas l'element racine */
      if (Ltype != TYPE_CD) {
        *Ldatasize-=Lsizenbr;
        g_assert((*Ldatasize) >= 0);

        /* on enleve la taille des eventuels enfants */
        remove_children_size(Ltreedata, &Liter, Ldatasize, Lg);

        gtk_tree_store_remove(GTK_TREE_STORE(Ltreedata), &Liter);
        gtk_tree_row_reference_deleted(Lproxy, Lpath);
        gtk_tree_path_free(Lpath);
      } 
    }
    gtk_tree_row_reference_free(Lnode->data); /* No longer needed */
  }
  g_list_free(Lrefs);

  g_object_unref(Lproxy);

  switch_pbar_status(Lcontenance, *Ldatamaxsize, *Ldatasize, PB_UPDATE_DATA);

  /* on selectionne l'icone CD */
  if (gtk_tree_model_get_iter_first(Ltreedata, &Liter)) {
    gtk_tree_selection_select_iter(Lselection, &Liter);
    gtk_widget_grab_focus(GTK_WIDGET(Lliste));
  }
}

/* ajouter des pistes a graver */
void gtk_audio_addtrack(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "LISTEAUDIO"));
  guint32 *Laudiolen = (guint32 *)sc_grave_get_data(Lg, "audiolen");
  guint32 *Laudiomaxlen = (guint32 *)sc_grave_get_data(Lg, "audiomaxlen");
  GtkProgressBar *Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "audiolenbar"));
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(Lliste);
  GtkWidget *Lpistesel = create_addpisteselection(NULL);
  GSList *Lfiles, *Lcur;

  use_last_dir(GTK_FILE_CHOOSER(Lpistesel));
  if (gtk_dialog_run(GTK_DIALOG(Lpistesel))==GTK_RESPONSE_ACCEPT) {
    save_last_dir(GTK_FILE_CHOOSER(Lpistesel));

    Lfiles = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(Lpistesel));

    set_cursor(Lwindow1, GDK_WATCH);
    for (Lcur = Lfiles; Lcur && Lcur->data; Lcur = Lcur->next) {
      _add_a_piste((gchar *) Lcur->data, GTK_LIST_STORE(Ltreemodel), Lg);
    }
    unset_cursor(Lwindow1);

    g_slist_free(Lfiles);
  }

  gtk_widget_destroy(Lpistesel);

  switch_pbar_status(Lcontenance, *Laudiomaxlen, *Laudiolen, PB_UPDATE_AUDIO);
}

/* ajouter des repertoires a graver */
void gtk_data_adddirectory(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Lliste));
  GtkWidget *Lfilesel = create_adddirselection(NULL);
  GtkTreeIter Liter;
  GSList *Lfiles, *Lcur;
  GtkProgressBar *Lcontenance;
  guint64 *Ldatasize;
  guint64 *Ldatamaxsize;
  gint Lnumsel = gtk_tree_selection_count_selected_rows(Lselection);
  gint Ltype;
  GtkTreePath *Lpath = NULL;

  _REMPLISSAGE_VALUE(Lg, Lliste, Ldatasize, Ldatamaxsize, Lcontenance);

  /* focus sur le premier element si pas de focus */
  if (!Lnumsel) {
    if (!gtk_tree_model_get_iter_first(Ltreemodel, &Liter)) return;

    gtk_tree_selection_select_iter(Lselection, &Liter);
    gtk_widget_grab_focus(GTK_WIDGET(Lliste));
  } else {
    /* si plusieurs elements de selectionne alors on pointe sur celui etant
     * sous le curseur */
    gtk_tree_view_get_cursor(GTK_TREE_VIEW(Lliste), &Lpath, NULL);
    if (!Lpath) return;
    gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
    gtk_tree_model_get(Ltreemodel, &Liter, 0, &Ltype, -1);
    if (Ltype == TYPE_FILE) {
      gtk_tree_path_up(Lpath);
      gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
    }
    gtk_tree_path_free(Lpath);
  }

  use_last_dir(GTK_FILE_CHOOSER(Lfilesel));
  if (gtk_dialog_run(GTK_DIALOG(Lfilesel))==GTK_RESPONSE_ACCEPT) {
    set_cursor(Lwindow1, GDK_WATCH);
    save_last_dir(GTK_FILE_CHOOSER(Lfilesel));

    Lfiles = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(Lfilesel));
	
    for (Lcur = Lfiles; Lcur && Lcur->data; Lcur = Lcur->next) {
      _add_a_file((gchar *) Lcur->data, (GtkTreeStore *)Ltreemodel, &Liter, Lg, Ldatasize, NULL);

    }
    g_slist_free(Lfiles);

    if ((Lpath = gtk_tree_model_get_path(Ltreemodel, &Liter))) {
      gtk_tree_view_expand_row(Lliste, Lpath, TRUE);
      gtk_tree_path_free(Lpath);
    }
    unset_cursor(Lwindow1);
  }
  gtk_widget_destroy(Lfilesel);

  switch_pbar_status(Lcontenance, *Ldatamaxsize, *Ldatasize, PB_UPDATE_DATA);
}


/* ajouter des fichiers a graver */
void gtk_data_addfile(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
  guint64 *Ldatasize;
  guint64 *Ldatamaxsize;
  GtkProgressBar *Lcontenance;
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Lliste));
  GtkWidget *Lfilesel = create_addfileselection(NULL);
  GtkTreeIter Liter;
  GSList *Lfiles, *Lcur;
  gint Ltype;
  gint Lnumsel = gtk_tree_selection_count_selected_rows(Lselection);
  GtkTreePath *Lpath = NULL;

  _REMPLISSAGE_VALUE(Lg, Lliste, Ldatasize, Ldatamaxsize, Lcontenance);

  /* focus sur le premier element si pas de focus */
  if (!Lnumsel) {
    if (!gtk_tree_model_get_iter_first(Ltreemodel, &Liter)) return;

    gtk_tree_selection_select_iter(Lselection, &Liter);
    gtk_widget_grab_focus(GTK_WIDGET(Lliste));
  } else {
    /* si plusieurs elements de selectionne alors on pointe sur celui etant
     * sous le curseur */
    gtk_tree_view_get_cursor(GTK_TREE_VIEW(Lliste), &Lpath, NULL);
    if (!Lpath) return;
    gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
    gtk_tree_model_get(Ltreemodel, &Liter, 0, &Ltype, -1);
    if (Ltype == TYPE_FILE) {
      gtk_tree_path_up(Lpath);
      gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
    }
    gtk_tree_path_free(Lpath);
  }

  use_last_dir(GTK_FILE_CHOOSER(Lfilesel));
  if (gtk_dialog_run(GTK_DIALOG(Lfilesel))==GTK_RESPONSE_ACCEPT) {
    save_last_dir(GTK_FILE_CHOOSER(Lfilesel));

    Lfiles = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(Lfilesel));
    for (Lcur = Lfiles; Lcur && Lcur->data; Lcur = Lcur->next) {
      _add_a_file((gchar *) Lcur->data, GTK_TREE_STORE(Ltreemodel), &Liter, Lg, Ldatasize, NULL);

    }
    g_slist_free(Lfiles);

    if ((Lpath = gtk_tree_model_get_path(Ltreemodel, &Liter))) {
      gtk_tree_view_expand_row(Lliste, Lpath, TRUE);
      gtk_tree_path_free(Lpath);
    }
  }
  gtk_widget_destroy(Lfilesel);

  switch_pbar_status(Lcontenance, *Ldatamaxsize, *Ldatasize, PB_UPDATE_DATA);
}

/* l'utilisateur veut editer un device */
void gtk_devices_edit(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *) Adata;
  GtkWidget *Lparent = GTK_WIDGET(sc_grave_get_data(Lg, "window"));
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "periphdata"));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
  GtkTreeIter Liter;
  GtkTreeModel *Ltreemodel;

  if (gtk_tree_selection_get_selected(Lselection, &Ltreemodel, &Liter)) {
    Tgrave *Ldialoghash;
    GtkWidget *Lwineditdevice;
    Tdriveinfo *Ldrive;
    gint Lnum;
    gint Lbus;
    
    gtk_tree_model_get(Ltreemodel, &Liter, 0, &Lnum, -1);
    Ldrive = (Tdriveinfo *)g_slist_nth_data(Glistdrives, Lnum);
    Ldialoghash = create_dialog_edit_device(Lparent, Ldrive);
    Lwineditdevice = sc_grave_get_data(Ldialoghash, "window");

    if (gtk_dialog_run(GTK_DIALOG(Lwineditdevice)) != GTK_RESPONSE_CANCEL) {
      GtkEntry *Ldevicename = GTK_ENTRY(sc_grave_get_widget(Ldialoghash, "devicename"));

      Lbus = get_combo_intvalue(sc_grave_get_widget(Ldialoghash, "devicebus"));
      Ldrive->actuperiph = Lbus;

      /* on remplace le nom dans la liste des lecteurs */
      g_free(Ldrive->name);
      Ldrive->name = g_strdup(gtk_entry_get_text(Ldevicename));
      /* ainsi que dans la liste affiche */
      gtk_list_store_set(GTK_LIST_STORE(Ltreemodel), &Liter, 2, (gchar *)g_slist_nth_data(Ldrive->dev, Ldrive->actuperiph),
          4, Ldrive->name, -1);
    }
    gtk_widget_destroy(Lwineditdevice);
    sc_grave_destroy(Ldialoghash);
  }

}

/* l'utilisateur a cliquer sur supprimer un device */
void gtk_devices_remove(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  GtkWidget *Lconfirm;
  gint Lrep;
  
  /* confirmation debut de l'operation */ 
  Lconfirm = gtk_message_dialog_new(GTK_WINDOW(Lwindow1),
                      GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                      GTK_BUTTONS_YES_NO,
                      _("Do you really want to remove this device ?"));
  Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));
  gtk_widget_destroy(Lconfirm);

  if (Lrep == GTK_RESPONSE_YES) {
    /* oui on supprime un device */
    GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "periphdata"));
    GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
    GtkTreeIter Liter;
    GtkTreeModel *Ltreemodel;
    gint Lnum;

    if (gtk_tree_selection_get_selected(Lselection, &Ltreemodel, &Liter)) {
      gtk_tree_model_get(Ltreemodel, &Liter, 0, &Lnum, -1);

      _DEB("ON SUPRIMME LE LEC = %d\n", Lnum);
      Glistdrives = g_slist_remove(Glistdrives, g_slist_nth_data(Glistdrives, Lnum));
      gtk_list_store_remove(GTK_LIST_STORE(Ltreemodel), &Liter);
    } 
  }

}

/* ajouter manuelement un device */
void gtk_devices_add(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *) Adata;
  GtkWidget *Lparent = GTK_WIDGET(sc_grave_get_data(Lg, "window"));
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "periphdata"));
  Tgrave *Ldialoghash = create_dialog_add_device(Lparent);
  GtkWidget *Lwinadddevice = sc_grave_get_data(Ldialoghash, "window");
  GtkWidget *Lmessage;

  while (1) {
    if (gtk_dialog_run(GTK_DIALOG(Lwinadddevice)) != GTK_RESPONSE_CANCEL) {
      GtkEntry *Ldevicebus = GTK_ENTRY(sc_grave_get_widget(Ldialoghash, "devicebus"));
      GtkEntry *Ldevicename = GTK_ENTRY(sc_grave_get_widget(Ldialoghash, "devicename"));
      gchar *Lbus = (gchar *)gtk_entry_get_text(Ldevicebus);
      Tdriveinfo *Ldrive = g_malloc0(sizeof(Tdriveinfo));
 
      proc_init_cdrominfo(NULL);
      matos_maj_drive_info(Ldrive, Lbus, NULL);
      proc_free_cdrominfo();

      if (Ldrive->type == 0) {
        Lmessage = gtk_message_dialog_new(GTK_WINDOW(Lparent),
                                     GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                                     _("Cannot add this drive !"));
      } else {
        Lmessage = gtk_message_dialog_new(GTK_WINDOW(Lparent),
                                     GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
                                     _("Drive succesfully added !"));
      }
      
      gtk_dialog_run(GTK_DIALOG(Lmessage));
      gtk_widget_destroy(Lmessage);

      if (Ldrive->type > 0) {
        /* on complate la structure drive avant de l'ajouter */
        Ldrive->manual = TRUE;
        Ldrive->actuperiph = 0;
        Ldrive->name = g_strdup(gtk_entry_get_text(Ldevicename));
        Ldrive->dev = g_slist_append(Ldrive->dev, g_strdup(Lbus));
        Ldrive->bus = g_slist_append(Ldrive->bus, g_strdup("MANU"));

        Glistdrives = g_slist_append(Glistdrives, Ldrive);

        /* mise a jour de la liste des peripheriques */
        remplirelistegraveur(Lliste);
        break;
      }
      g_free(Ldrive);
    } else {
      break;
    }
  }
  gtk_widget_destroy(Lwinadddevice);
  sc_grave_destroy(Ldialoghash);
}

  
/* selection d'une image iso */
void gtk_file_sel(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  const gchar *Lname = glade_get_widget_name(Abtn);
  GtkWidget *Ltextdest = (GtkWidget *) sc_grave_get_widgetv(Lg, "%svalue", Lname);
  gint Lresp;
  GtkWidget *Lfilesel;
  GtkFileChooserAction Ldialogtype = !strncmp(Lname, "src", 3) ?
                                        GTK_FILE_CHOOSER_ACTION_OPEN : GTK_FILE_CHOOSER_ACTION_SAVE;
_DEB("=>%s\n", Lname);
  if (!strncmp(Lname, "dir", 3)) {
    Ldialogtype = GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER;
  }
  Lfilesel = create_isoselection(NULL, Ldialogtype);
  use_last_dir(GTK_FILE_CHOOSER(Lfilesel));
  Lresp=gtk_dialog_run(GTK_DIALOG(Lfilesel));
  gtk_widget_hide (Lfilesel);
  while (gtk_events_pending())
    gtk_main_iteration();

  if (Lresp == GTK_RESPONSE_ACCEPT) {
    gchar *Lfilenamebrut = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER ( Lfilesel));
    gchar *Lfilename = _FILENAME(Lfilenamebrut);
    save_last_dir(GTK_FILE_CHOOSER(Lfilesel));
    gtk_entry_set_text(GTK_ENTRY(Ltextdest), Lfilename);
    g_free(Lfilename);
    g_free(Lfilenamebrut);
  }

  gtk_widget_destroy(Lfilesel);
}

/* lors de la selection d'un lecteur / graveur */
void on_selectgraveur(GtkWidget *Acombo, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lpanel;
  GtkWidget *Lvitesse, *Lobj;
  GtkTreeModel *Lmodel = gtk_combo_box_get_model(GTK_COMBO_BOX(Acombo));
  GtkTreeIter Liter;
  gboolean Lisimgiso = FALSE;
  const gchar *Lname = glade_get_widget_name(Acombo);
  gchar *Laskname, *Lspeedname;
  gchar *Llecteur = NULL;
  guint Lident = 0;
  gint i;
  Tdriveinfo *Ldrive;

  if (g_str_has_suffix(Lname, "combo")==FALSE) return;
        
  if (!gtk_combo_box_get_active_iter(GTK_COMBO_BOX(Acombo), &Liter)) return;

  gtk_tree_model_get(Lmodel, &Liter, 0, &Ldrive, 3, &Lident, -1);

  Lisimgiso = !Ldrive;

  Lspeedname = g_strdup(Lname);
  _SWAP_COMBO(Lspeedname, "speed");
  Lvitesse = (GtkWidget *)sc_grave_get_widget(Lg, Lspeedname);

  Laskname = g_strdup(Lname);
  _SWAP_COMBO(Laskname, "table");
  Lpanel = (GtkWidget *)sc_grave_get_widget(Lg, Laskname);
  g_free(Laskname);

  /* on cache ou montre la zone de saisie du nom de l'image iso */
  if (GTK_IS_WIDGET(Lpanel)) {

    if (Lisimgiso) {
      /* on montre le champ de selection de l'image iso */
      gtk_widget_show(Lpanel);
    } else {
      /* on le cache */
      gtk_widget_hide(Lpanel);
    }
    if (GTK_IS_WIDGET(Lvitesse)) {
      /* selection de la vitesse non active lorsque l'on a selectionne une image iso .. */
      gtk_widget_set_sensitive(Lvitesse, Lisimgiso ? FALSE : TRUE);
    }
  } 
  if (!Lisimgiso && GTK_IS_WIDGET(Lvitesse)) {
    remplire_combo_vitesse(GTK_COMBO_BOX(Lvitesse), Ldrive, conf_get_string(Lspeedname)); 
  }

  for (i=0; ((Lobj = sc_grave_get_datav(Lg, "%s_isotodisabled%d", Lname, i)) && GTK_IS_WIDGET(Lobj)); i++) {
    /* disable si image iso, enable sinon */
    gtk_widget_set_sensitive (Lobj, !Lisimgiso);
  }

  for (i=0; ((Lobj = sc_grave_get_datav(Lg, "%s_notcdrwtodisabled%d", Lname, i)) && GTK_IS_WIDGET(Lobj)); i++) {
    /* disable si lecteur non cdrw */
    gtk_widget_set_sensitive (Lobj, DRIVE_WRITER(Ldrive));
  }

  for (i=0; ((Lobj = sc_grave_get_datav(Lg, "%s_notdvdrwtodisabled%d", Lname, i)) && GTK_IS_WIDGET(Lobj)); i++) {
    /* disable si lecteur non dvdrw */
    gtk_widget_set_sensitive (Lobj, DRIVE_DVDWRITER(Ldrive));
  }

  g_free(Lspeedname);
  g_free(Llecteur);
}

/* confirmer l'annulation de la gravure */
gboolean abort_gravure(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *) Adata;
  gboolean *Labort = (gboolean *)sc_grave_get_data(Lg, "gabort");
  GtkWidget *Lwindow = GTK_WIDGET(sc_grave_get_widget(Lg, "window_burn"));
  gint Lret;

  g_assert(Labort);

  GtkWidget *Lconfirm = gtk_message_dialog_new(GTK_WINDOW(Lwindow),
                      GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                      GTK_BUTTONS_YES_NO, _("Do you really want to abort operation ?"));

  Lret = gtk_dialog_run(GTK_DIALOG(Lconfirm));
      
  if (Lret == GTK_RESPONSE_YES) {
    gint *Lpid = (gint *)sc_grave_get_data(Lg, "pid");
    if (*Lpid) {
      /* on envoi un TERM au process (cdrecord...) pour qu'il s'arrete ! */
_DEB("kill de %d\n", *Lpid);
      kill(*Lpid, SIGINT);
    }
    *Labort = TRUE;
  }

  gtk_widget_destroy(Lconfirm);

  return Lret == GTK_RESPONSE_YES ? TRUE : FALSE;
}

gboolean on_deleteevent_gravure(GtkWidget *Awidget, GdkEvent *Aevent, gpointer Adata)
{

  abort_gravure(NULL, Adata);

  return TRUE;
}

/* confirmer la cloture d'un cd */
void confirm_fixcd(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  GtkWidget *Lconfirm;
  gint Lrep;
  GSList *Llmediarequis = g_slist_append(NULL, GINT_TO_POINTER(_MEDIA_CDR + _MEDIA_NOBLANK + _MEDIA_APPENDABLE));
  gboolean Lstatus;
  
  Lstatus = waiting_for_user(_MEDIA_TITLE_BLANKORAPPENDABLE, Lg, Llmediarequis, NULL, matos_get_drive_info(Lg, "dstothercombo"));
  g_slist_free(Llmediarequis);
  
  if (Lstatus==FALSE) return;
  
  /* confirmation debut de l'operation */ 
  Lconfirm = gtk_message_dialog_new(GTK_WINDOW(Lwindow1),
                      GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                      GTK_BUTTONS_YES_NO,
                      _("Do you really want to fix the CD-R ?"));
  Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));

  gtk_widget_destroy(Lconfirm);

  if (Lrep == GTK_RESPONSE_YES) {
    /* ok go ! */
    show_fixcdwindow(Lg);
  }
}

/* affichage du popup "formater un dvdrw" */
void popup_formatdvdrw(GtkWidget *Abtn, gpointer Adata)
{
  GtkMenu *Lpopupm = create_gtkpopupmenuformatdvd(Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Abtn, 0, 0);

}

/* confirmation d'une operation de formatage sur un dvd */
void gtk_confirm_dvdoperation(GtkWidget *Abtn, gpointer Adata)
{
  const gchar *Lname = glade_get_widget_name(Abtn);
  gchar *s, *Ltitle, *Lmessage;
  GtkWidget *Lconfirm;
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  gint Lrep;
  Ttypeformatdvd Lcode;
 
  g_assert(Lname);
  if (!(s = strrchr(Lname, '_')) || !*(++s)) return;

  if (!strcmp(s, "format")) {
    Ltitle = _("Formatting DVD...");
    Lmessage = _("Do you really want to format this media ?");
    Lcode = DVD_FORMAT;
  } else if (!strcmp(s, "blank")) {
    Ltitle = _("Blanking DVD...");
    Lmessage = _("Do you really want to fast blanking this DVD-RW ?\nBe carreful you could only write in DAO mode after a fast blanking.");
    Lcode = DVD_BLANK;
  } else if (!strcmp(s, "blankfull")) {
    Ltitle = _("Full blanking DVD...");
    Lmessage = _("Do you really want to blank this DVD-RW ?\nThis could take 1 hour with an 1x media.");
    Lcode = DVD_FULLBLANK;
  } else {
    return;
  }
  
  Lconfirm = gtk_message_dialog_new(GTK_WINDOW(Lwindow1),
                      GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                      GTK_BUTTONS_YES_NO, Lmessage);

  Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));
  gtk_widget_destroy(Lconfirm);
  if (Lrep == GTK_RESPONSE_YES) {
    show_dvdformatwindow(Lg, Ltitle, Lcode);
  }
}

/* confirmer l'effacement d'un cdrw */
void confirm_erasecdrw(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  Tgrave *Ldialoghash;
  GtkWidget *Lconfirm;
  GSList *Llmediarequis = g_slist_append(NULL, GINT_TO_POINTER(_MEDIA_CDRW + _MEDIA_NOBLANK));
  gboolean Lstatus;
  Lstatus = waiting_for_user(_MEDIA_TITLE_BLANKORAPPENDABLE, Lg, Llmediarequis, NULL, matos_get_drive_info(Lg, "dstothercombo"));
  g_slist_free(Llmediarequis);
  
  if (Lstatus==FALSE) return;

  Ldialoghash = create_dialog_confirm_operation(Lwindow1, _("Do you really want to erase the CD-RW ?"),
        _("_Fast blank disk when erasing CD-RW"), conf_get_boolean("fastblank"));
  Lconfirm = sc_grave_get_data(Ldialoghash, "window");

  /* confirmation debut de l'operation */ 
  if (gtk_dialog_run(GTK_DIALOG(Lconfirm)) == GTK_RESPONSE_YES) {
    gboolean Lusefastblank = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sc_grave_get_widget(Ldialoghash, "checkbox")));
    gtk_widget_destroy(Lconfirm);
    
    sc_grave_set_data(Lg, &Lusefastblank, "usefastblank");
    show_erasecdrwwindow(Lg);
    sc_grave_del_data(Lg, "usefastblank");
  } else {
    gtk_widget_destroy(Lconfirm);
  }
  sc_grave_destroy(Ldialoghash);

}

/* confirmer la debut de la gravure */
void confirm_burn (GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  Tgrave *Ldialoghash;
  GtkWidget *Lconfirm;
  gint Lrep;
  gboolean Luseautoblank = FALSE;
  gchar *Ltypeburn = NULL;
  const gchar *Lname;
  gchar *Ltxt = NULL;
  gchar *Ltitle = NULL;
  GtkWidget *Lwindow1 = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  GSList *Llmediarequis1 = NULL;
  GSList *Llmediarequis2 = NULL;
  gint Lmediatitle1 = _MEDIA_NONE, Lmediatitle2 = _MEDIA_NONE;
  gint Lmediadetect1 = _MEDIA_NONE, Lmediadetect2 = _MEDIA_NONE;
  Tdriveinfo *Ldevicename1 = NULL, *Ldevicename2 = NULL;
  gboolean Lwritercdrw;
  gboolean Liso = FALSE;
  Ttypeoperation Loperation = WRITE_CD;

  Lname = gtk_widget_get_name(Abtn);

  if (!Lname) return;
  if (g_str_has_prefix(Lname, "btnburnaudio")) {
    GtkTreeIter Liter;
    GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(
        GTK_TREE_VIEW(sc_grave_get_widget(Lg, "LISTEAUDIO")));
    
    /* gravure cd audio, on verifie d'abord qu'il y ai au
     * moins une piste dans la liste .. */
    if (!gtk_tree_model_get_iter_first(Ltreemodel, &Liter)) {
      Lconfirm = gtk_message_dialog_new(GTK_WINDOW(Lwindow1),
                                        GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING,
                                        GTK_BUTTONS_CANCEL, 
                                        _("You must add at least one track to write !"));
      gtk_dialog_run(GTK_DIALOG(Lconfirm));
      gtk_widget_destroy(Lconfirm);
      return;
    }
    Ltxt = _("Do you really want to create an audio cd ?");
    Ltitle = _("Writing audio cd in progress...");
    Ldevicename1 = matos_get_drive_info(Lg, "dstaudiocombo");
    Llmediarequis1 = manage_mediarequis(NULL,
                      _MEDIA_CDR + _MEDIA_BLANK, _MEDIA_CDR + _MEDIA_APPENDABLE,
                      _MEDIA_CDRW + _MEDIA_BLANK, _MEDIA_CDRW + _MEDIA_APPENDABLE,
                      _MEDIA_CDRW + _MEDIA_NOBLANK, -1);
    Lmediatitle1 = _MEDIA_TITLE_BLANKORAPPENDABLE;
  } else if (g_str_has_prefix(Lname, "btnburndata") || g_str_has_prefix(Lname, "btnburndvddata")) {
    GtkTreeIter Liter;
    GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list")));
    /* gravure cd de donnee, on verifie qu'il y ai au
     * moins un fichier/repertoire dans la liste */
    if (g_str_has_prefix(Lname, "btnburndvddata")) {
      Loperation = WRITE_DVD;
    }
    
    if (!gtk_tree_model_get_iter_first(Ltreemodel, &Liter) ||
        !gtk_tree_model_iter_has_child(Ltreemodel, &Liter)) {

      Lconfirm = gtk_message_dialog_new(GTK_WINDOW(Lwindow1),
                                        GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING,
                                        GTK_BUTTONS_CANCEL, 
                              _("You must add at least one file to write !"));
      gtk_dialog_run(GTK_DIALOG(Lconfirm));
      gtk_widget_destroy(Lconfirm);
      return;
    }

    if (Loperation == WRITE_CD) {
      Ldevicename1 = matos_get_drive_info(Lg, "dstdatacombo");
      Liso = !Ldevicename1;
      if (Liso) {
        Ltxt = _("Do you really want to create an iso image ?");
      } else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sc_grave_get_widget(Lg, "datamulti")))) {
        Loperation = WRITE_CD_MULTI;
        Ltxt = _("Do you really want to start or continue a multi session data cd ?");
      } else {
        Ltxt = _("Do you really want to create a data cd ?");
      }
      if (!Liso) {
        Ltitle = _("Writing data cd in progress...");
        Llmediarequis1 = manage_mediarequis(NULL,
                    _MEDIA_CDR + _MEDIA_BLANK, _MEDIA_CDR + _MEDIA_APPENDABLE,
                    _MEDIA_CDRW + _MEDIA_BLANK, _MEDIA_CDRW + _MEDIA_APPENDABLE,
                    _MEDIA_CDRW + _MEDIA_NOBLANK, -1);
        Lmediatitle1 = _MEDIA_TITLE_BLANKORAPPENDABLE;
      } else {
        Ltitle = _("Writing iso image in progress...");
      }
    } else {
      Ldevicename1 = matos_get_drive_info(Lg, "dstdvddatacombo");
      Liso = !Ldevicename1;
      if (!Liso) {
        Ltxt = _("Do you really want to create a data dvd ?");
        Ltitle = _("Writing data dvd in progress...");
      
        Llmediarequis1 = manage_mediarequis(NULL,
                    _MEDIA_DVDR_P + _MEDIA_BLANK, _MEDIA_DVDR_P + _MEDIA_APPENDABLE,
                    _MEDIA_DVDR_M + _MEDIA_BLANK, _MEDIA_DVDR_M + _MEDIA_APPENDABLE,
                    _MEDIA_DVD_RAM, _MEDIA_DVDP_RW, _MEDIA_DVDM_RW, -1);
        Lmediatitle1 = _MEDIA_TITLE_DVDRew;
      } else {
        /* creation image iso */
        Loperation = WRITE_CD;
        Ltxt = _("Do you really want to create an iso image ?");
      }
    }
  } else {
    /* copie de cd */

    Ltxt = _("Do you really want to start the copy ?");
    Ltitle = _("Copying in progress...");
    Ldevicename1 = matos_get_drive_info(Lg, "srccopycombo");
    Liso = !Ldevicename1;
    if (!Liso) {
      Llmediarequis1 = manage_mediarequis(NULL, _MEDIA_CD + _MEDIA_NOBLANK, -1);
      Lmediatitle1 = _MEDIA_CD;
    }
    Ldevicename2 = matos_get_drive_info(Lg, "dstcopycombo");

    if (Ldevicename2) {
      Llmediarequis2 = manage_mediarequis(NULL, 
                  _MEDIA_CDR + _MEDIA_BLANK, _MEDIA_CDR + _MEDIA_APPENDABLE,
                  _MEDIA_CDRW + _MEDIA_BLANK, _MEDIA_CDRW + _MEDIA_APPENDABLE,
                  _MEDIA_CDRW + _MEDIA_NOBLANK, -1);

      Lmediatitle2 = _MEDIA_TITLE_BLANKORAPPENDABLE;
    }
  }

  /* attente que l'utilisateur ai bien insere un cd */
  if (waiting_for_user(Lmediatitle1, Lg, Llmediarequis1, &Lmediadetect1, Ldevicename1)==FALSE) return;
  if (waiting_for_user(Lmediatitle2, Lg, Llmediarequis2, &Lmediadetect2, Ldevicename2)==FALSE) return;

  /* confirmation debut de l'operation */ 
  if (Loperation != WRITE_DVD) {
    /* on affiche l'option "effacer le CDRW que lorsque c'est un graveur qui le permet */    
    Lwritercdrw = Ldevicename1 && DRIVE_CDRW_WRITER(Ldevicename1) ? TRUE : FALSE;
    Ldialoghash = create_dialog_confirm_operation(Lwindow1, Ltxt,
       Lwritercdrw == TRUE &&
        ((Lmediadetect1 & _MEDIA_CDRW) && (Lmediadetect1 & _MEDIA_NOBLANK)) ?
        _("Blank CD-RW before writing") : NULL, conf_get_boolean("autoblank"));
  } else {
    /* confirmation gravure dvd */
    Ldialoghash = create_dialog_select_dvdoperation(Lwindow1, Lmediadetect1);
  }
  Lconfirm = sc_grave_get_data(Ldialoghash, "window");

  Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));

  if (Lrep == GTK_RESPONSE_YES) {
    /* ok go ! */
    if (Loperation != WRITE_DVD) {
      GtkCheckButton *Labcheck = GTK_CHECK_BUTTON(sc_grave_get_widget(Ldialoghash, "checkbox"));

      g_assert(GTK_IS_WIDGET(Labcheck));
      if (Lwritercdrw == TRUE) {
        Luseautoblank = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Labcheck));
      }
      sc_grave_set_data(Lg, &Luseautoblank, "useautoblank");
    } else {
      GtkRadioButton *Labradio = GTK_RADIO_BUTTON(sc_grave_get_widget(Ldialoghash, "burninitial"));
      Ltypeburn = g_strdup(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Labradio)) ? "-Z" : "-M");
      sc_grave_set_data(Lg, Ltypeburn, "typeburn");
    }

    gtk_widget_destroy(Lconfirm);

    sc_grave_set_data(Lg, Llmediarequis1, "mediarequis1");
    sc_grave_set_data(Lg, &Lmediatitle1, "mediatitle1");
    sc_grave_set_data(Lg, Llmediarequis2, "mediarequis2");
    sc_grave_set_data(Lg, &Lmediatitle2, "mediatitle2");
    sc_grave_set_data(Lg, Ltxt, "medialabel");
    show_graverwindow(Lg, (gchar *)Lname, Ltitle);
    sc_grave_del_data(Lg, "medialabel");
    sc_grave_del_data(Lg, "mediatitle2");
    sc_grave_del_data(Lg, "mediarequis2");
    sc_grave_del_data(Lg, "mediatitle1");
    sc_grave_del_data(Lg, "mediarequis1");

    if (Loperation != WRITE_DVD) {
      sc_grave_del_data(Lg, "useautoblank");
    } else {
      sc_grave_del_data(Lg, "typeburn");
      g_free(Ltypeburn);
    }
  } else {
    gtk_widget_destroy(Lconfirm);
  }

  g_slist_free(Llmediarequis2);
  g_slist_free(Llmediarequis1);

  sc_grave_destroy(Ldialoghash);
}

/* popup ajouter des fichiers */
void view_popupmenu_addfile(GtkWidget *Aliste, GdkEventButton *Aevent, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenudata(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, Aevent->button, Aevent->time);
}
void key_popupmenu_addfile(GtkWidget *Aliste, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenudata(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, 0, 0);
}

/* popup affichage */
void view_popupmenu_display(GtkWidget *Aliste, GdkEventButton *Aevent, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenudisplay(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, Aevent->button, Aevent->time);
}
void key_popupmenu_display(GtkWidget *Aliste, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenudisplay(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, 0, 0);
}

/* popup ajouter des pistes audio */
void view_popupmenu_addaudio(GtkWidget *Aliste, GdkEventButton *Aevent, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenuaudio(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, Aevent->button, Aevent->time);
}
void key_popupmenu_addaudio(GtkWidget *Aliste, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenuaudio(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, 0, 0);
}


/* popup gestion des peripheriques */
void view_popupmenu_devices(GtkWidget *Aliste, GdkEventButton *Aevent, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenudevices(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, Aevent->button, Aevent->time);
}
void key_popupmenu_devices(GtkWidget *Aliste, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenudevices(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, 0, 0);
}

/* popup gestion des charsets */
void view_popupmenu_charset(GtkWidget *Aliste, GdkEventButton *Aevent, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenucharset(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, Aevent->button, Aevent->time);
}
void key_popupmenu_charset(GtkWidget *Aliste, gpointer Adata) {
  GtkMenu *Lpopupm = create_gtkpopupmenucharset(Aliste, Adata);
  gtk_menu_popup(Lpopupm, NULL, NULL, NULL, Aliste, 0, 0);
}

/* appuis sur une touche, gestion de la suppresion des elements dans les treeview */
gboolean gtk_do_eventkeyrelease(GtkWidget *Aliste, GdkEventKey *Aevent, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *) Adata;
  const gchar *Lname = glade_get_widget_name(Aliste);
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Aliste));
  
  if (!Aevent) return FALSE;

  /* appuis sur la touche supprimer */
  if (Aevent->keyval == GDK_Delete) {
    if (!gtk_tree_selection_count_selected_rows(Lselection)) return TRUE;

    if (!strcmp(Lname, "periphdata")) {
      gtk_devices_remove(NULL, Lg); 
      return TRUE;
    } else if (!strcmp(Lname, "charsetdata")) {
      gtk_charset_remove(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "LISTEAUDIO")) {
      gtk_audio_removetrack(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "LISTEDATA")) {
      gtk_data_removefiles(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "LISTEDVDDATA")) {
      gtk_data_removefiles(NULL, Lg);
      return TRUE;
    }
  } else if (Aevent->keyval == GDK_Insert) {
    /* appuis sur la touche insert */
    if (!strcmp(Lname, "periphdata")) {
      gtk_devices_add(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "charsetdata")) {
      gtk_charset_add(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "LISTEAUDIO")) {
      gtk_audio_addtrack(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "LISTEDATA")) {
      gtk_data_addfile(NULL, Lg);
      return TRUE;
    } else if (!strcmp(Lname, "LISTEDVDDATA")) {
      gtk_data_addfile(NULL, Lg);
      return TRUE;
    }

  } else if (Aevent->keyval == GDK_F2) {
    /* appuis sur F2, renommer un element */
    if (!strcmp(Lname, "LISTEDATA") || !strcmp(Lname, "LISTEDVDDATA")) {
      callbacks_data_renameitem(NULL, Lg);
    }
  } else if ((Aevent->keyval >= GDK_exclam && Aevent->keyval <= GDK_EuroSign) ||
             (Aevent->keyval >= GDK_KP_0 && Aevent->keyval <= GDK_KP_9)) {
    guint Lkey = Aevent->keyval >=GDK_KP_0 ? Aevent->keyval - 0xFF80 : Aevent->keyval;

    callbacks_try_to_select(Lkey, GTK_TREE_VIEW(Aliste), Lg);
  }

  return FALSE;
}

/* clique sur la souris, gestion des popups */
gboolean gtk_do_eventbuttonpress(GtkWidget *Aliste, GdkEventButton *Aevent, gpointer Adata)
{
  if (!Aevent) return FALSE;

  if (Aevent->type == GDK_BUTTON_PRESS) {
    if (Aevent->button == 3) {
      const gchar *Lname = glade_get_widget_name(Aliste);
      if (!strcmp(Lname, "LISTEDATA")) {
        view_popupmenu_addfile(Aliste, Aevent, Adata);
      } else if (!strcmp(Lname, "LISTEDVDDATA")) {
        view_popupmenu_addfile(Aliste, Aevent, Adata);
      } else if (!strcmp(Lname, "LISTEAUDIO")) {
        view_popupmenu_addaudio(Aliste, Aevent, Adata);
      } else if (!strcmp(Lname, "periphdata")) {
        view_popupmenu_devices(Aliste, Aevent, Adata);
      } else if (!strcmp(Lname, "charsetdata")) {
        view_popupmenu_charset(Aliste, Aevent, Adata);
      } else if (!strcmp(Lname, "topmenu")) {
        view_popupmenu_display(Aliste, Aevent, Adata);
      }
      return TRUE;
    }
  } else if (Aevent->type == GDK_2BUTTON_PRESS) {
    if (Aevent->button == 1) {
      const gchar *Lname = glade_get_widget_name(Aliste);
      /* double clique sur la fenetre des lecteurs, on edite le lecteur */
      if (!strcmp(Lname, "periphdata")) {
        gtk_devices_edit(NULL, Adata);
      } else if (!strcmp(Lname, "charsetdata")) {
        gtk_charset_edit(NULL, Adata);
      }
      return TRUE;
    }
  }
  return FALSE;
}

/* callback lorsque l'on selectionne un fichier pour activer/desactiver
 * le bouton "ajouter des fichiers" */
gboolean on_select_files (GtkTreeSelection *Aselection, GtkTreeModel *Amodel,
                          GtkTreePath *Apath, gboolean Apath_currently_selected,
                          gpointer Adata)
{
  GtkTreeIter Liter;
  GtkWidget *Lbtn = (GtkWidget *) Adata; 
  gboolean Ltodisable;
  
  if (Lbtn && !Apath_currently_selected && gtk_tree_model_get_iter(Amodel, &Liter, Apath)) {
    gint Ltype;

    gtk_tree_model_get(Amodel, &Liter, 0, &Ltype, -1);

    Ltodisable = (Ltype != TYPE_FILE);

    gtk_widget_set_sensitive(GTK_WIDGET(Lbtn), Ltodisable);
  }
  return TRUE;
}

/* passage a l'onglet suivante */
void gonextpage(GtkWidget *Antn, GtkNotebook *Anotebook)
{
  gtk_notebook_next_page(Anotebook);
}

/* callback pour verifier la longueur des textes buffer */
void text_buffer_callback(GtkTextBuffer *Aobj, gpointer Adata)
{
  GtkTextIter Lstart_iter;
  GtkTextIter Lend_iter;
  gchar *Ltext;
  gint Lmax_len;

  Lmax_len = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (Aobj), "max-length"));
  if (Lmax_len == 0) return;

  gtk_text_buffer_get_bounds (Aobj, &Lstart_iter, &Lend_iter);
  Ltext = gtk_text_buffer_get_text (Aobj, &Lstart_iter, &Lend_iter, FALSE);

  if (strlen(Ltext)>Lmax_len) {
    gtk_text_buffer_get_iter_at_offset (Aobj, &Lstart_iter, Lmax_len);
    gtk_text_buffer_delete (Aobj, &Lstart_iter, &Lend_iter);
  }

  g_free(Ltext);
}

/* clique sur le menu, on active un onglet */
void menu_active_panel(GtkWidget *Amenuitem, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Ltopmenu = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "topmenu"));
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(Ltopmenu);
  GtkTreeModel *Lmodel = gtk_tree_view_get_model(Ltopmenu);
  GtkTreeIter Liter;
  const gchar *Lname = glade_get_widget_name(Amenuitem);
  gchar *Lrequest = g_strdup_printf("body%s", Lname+strlen("menu"));
  gchar *Lcuritem;
  gchar Lcont = gtk_tree_model_get_iter_first(Lmodel, &Liter);
  while (Lcont == TRUE) {
    gtk_tree_model_get(Lmodel, &Liter, 0, &Lcuritem, -1);

    if (!strcmp(Lcuritem, Lrequest)) {
      gtk_tree_selection_select_iter(Lselection, &Liter);
      gtk_toggle_top_menu(Ltopmenu, Lg);
      
      g_free(Lcuritem);
      break;
    }
    g_free(Lcuritem);

    Lcont = gtk_tree_model_iter_next(Lmodel, &Liter);
  }
}

/* changement d'onglet */
void switchpage(GtkNotebook *Anotebook, GtkNotebookPage *Anotebookpage, guint Apagenum, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Ldatavoli;
  GtkTreeModel *Ltreemodel;
  GtkTreeIter Liter;
  gchar *Lnom;

  if (Apagenum != 0 && Apagenum != 2) return;
  if (!strcmp(glade_get_widget_name(GTK_WIDGET(Anotebook)), "bodycddata")) {
    Ldatavoli = sc_grave_get_widget(Lg, "datavoli");
    Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(sc_grave_get_widget(Lg, "LISTEDATA")));
  } else {
    Ldatavoli = sc_grave_get_widget(Lg, "dvddatavoli");
    Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(sc_grave_get_widget(Lg, "LISTEDVDDATA")));
  }


  
  /* il devrait normalement toujours y avoir une entree dans la liste, au moins le nom du cd ! */
  if (!gtk_tree_model_get_iter_first(Ltreemodel, &Liter)) return;
  if (Apagenum == 2) {
    /* mise a jour du champ "Nom du volume" */
    gtk_tree_model_get(Ltreemodel, &Liter, 2, &Lnom, -1);
    gtk_entry_set_text(GTK_ENTRY(Ldatavoli), Lnom);
    g_free(Lnom);
  } else if (Apagenum == 0) {
    /* mise a jour de l'entre contenant le nom du volume */
    gtk_tree_store_set(GTK_TREE_STORE(Ltreemodel), &Liter, 2, gtk_entry_get_text(GTK_ENTRY(Ldatavoli)), -1);
  }
}

/* activer la boite de dialogue proprietes */
void menu_active_proprietes(GtkWidget *Amenuitem, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *) Adata;
  GtkWidget *Lparent = GTK_WIDGET(sc_grave_get_widget(Lg, "topwindow"));
  Tgrave *Ldialoghash = create_dialog_proprietes(Lparent);
  GtkWidget *Lwinproprietes = sc_grave_get_data(Ldialoghash, "window");
  GError *Aerror = NULL;
  GtkWidget *Lptr;
  TProgRequis *Lcurentry;
  gchar Ltmp[256];
  gboolean Lval;

  sc_tooltips(Lwinproprietes, conf_get_boolean("tooltips"));

  gtk_dialog_run(GTK_DIALOG(Lwinproprietes));

  /* recuperation des valeurs a sauver */
  Lptr = sc_grave_get_widget(Ldialoghash, "dirtmpvalue");
  conf_store_value("tmpdir", (gchar *)gtk_entry_get_text(GTK_ENTRY(Lptr)));
  Lptr = sc_grave_get_widget(Ldialoghash, "overburn");
  conf_store_value("overburn", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
  Lptr = sc_grave_get_widget(Ldialoghash, "fastblank");
  conf_store_value("fastblank", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
  Lptr = sc_grave_get_widget(Ldialoghash, "autoblank");
  conf_store_value("autoblank", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
  Lptr = sc_grave_get_widget(Ldialoghash, "eject");
  conf_store_value("eject", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
  Lptr = sc_grave_get_widget(Ldialoghash, "statusbar");
  conf_store_value("statusbar", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
  Lptr = sc_grave_get_widget(Ldialoghash, "iconsizelist");
  conf_store_value("iconsize", get_combo_value((Lptr)));
  Lptr = sc_grave_get_widget(Ldialoghash, "saveconfig");
  conf_store_value("saveconfig", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
  Lptr = sc_grave_get_widget(Ldialoghash, "inputcharset");
  conf_store_value("charsetdata", get_combo_value_pos(Lptr, 2));
  Lptr = sc_grave_get_widget(Ldialoghash, "themelist");
  conf_store_value("theme", get_combo_value_pos(Lptr, 3));

  /* tooltips ! */
  Lptr = sc_grave_get_widget(Ldialoghash, "tooltips");
  Lval = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr));
  conf_store_value("tooltips", _BOOLEAN_CONF(Lval));
  sc_tooltips(Lparent, Lval);

#if 0  
  Lptr = sc_grave_get_widget(Ldialoghash, "extended");
  conf_store_value("extended", _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lptr))));
#endif


  /* on sauve les parametres utilisateurs a passer aux programmes externes */
  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (Lcurentry->type != CONF_PROG) continue;
    g_snprintf(Ltmp, sizeof(Ltmp)-1, "%spara", Lcurentry->entry);

    Lptr = sc_grave_get_widget(Ldialoghash, Ltmp);
    if (!GTK_IS_WIDGET(Lptr)) continue;

    conf_store_value(Ltmp, (gchar *)gtk_entry_get_text(GTK_ENTRY(Lptr)));
  }
  
  /* sauvegarde configuration */
  write_conf(&Aerror);

  /* mise a jour des listes contenant les lecteurs */
  update_combo_graveur(Lg);
  update_status_bar_display(Lg);
  manage_interface(Lg, MENU_UPDATE + COMBO_UPDATE );

  gtk_widget_destroy(Lwinproprietes);

  sc_grave_destroy(Ldialoghash);
}

gboolean encours_callback(gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkProgressBar *Lprog = GTK_PROGRESS_BAR(sc_grave_get_data(Lg, "progress"));
  gint *Lcont = (gint *)sc_grave_get_data(Lg, "cont");

  gtk_progress_bar_pulse(Lprog);

  if ((*Lcont) == 1) (*Lcont) = 0;
  
  return (*Lcont) > 0 ? TRUE : FALSE;
}

/* chercher (a nouveau) les programmes */
void chercheprog(GtkWidget *Aobj, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Lstatusbox;
  GtkWidget *Lwindow1 = sc_grave_get_data(Lg, "window");

  /* mise a jour de la configuration */
  manage_config(Lg, SEARCH_PROG, NULL);
  clean_config();

  maj_proprietes_prog(Lg);

  /* boite de dialoge operation terminee */
  Lstatusbox = gtk_message_dialog_new(GTK_WINDOW(Lwindow1), GTK_DIALOG_MODAL,
      GTK_MESSAGE_INFO, GTK_BUTTONS_OK, _("configuration updated."));
  gtk_dialog_run(GTK_DIALOG(Lstatusbox));
  gtk_widget_destroy(Lstatusbox);
}


/* chercher (a nouveau) les lecteurs */
void cherchelecteur(GtkWidget *Aobj, gpointer Adata)
{
  Tgrave *Lparenthash = (Tgrave *)Adata;
  GtkWidget *Lwindow1 = sc_grave_get_data(Lparenthash, "window");
  Tgrave *Lg = create_window_encours(Lwindow1);
  GtkWidget *Lstatusbox;
  GtkWidget *Lwin;
  GtkTreeView *Ldata;
  gboolean Labort = FALSE;
  gint Lcont = 2;
  gchar *Ltxt;
  GSList *Lcur, *Lnext;
  Tdriveinfo *Ldrive;
  guint Ltimeout;
  
  Lwin = sc_grave_get_widget(Lg, "window_inprogress"); 

  sc_grave_set_data(Lg, &Lcont, "cont");
  sc_grave_set_data(Lg, &Labort, "gabort");
  sc_grave_ref_wtd(Lg, "progress", "progress");

  /* bouton annuler */
  g_signal_connect(G_OBJECT(sc_grave_get_widget(Lg, "cancelbtn")), "clicked", G_CALLBACK(abort_encours), (GtkWidget *)Lg);

  gtk_widget_show(Lwin);
  Ltimeout = g_timeout_add(500, encours_callback, Lg);

  proc_init_cdrominfo(NULL);
  cdrecord_scan_for_drives(Lg, NULL);

  if (Labort == FALSE) {
    for (Lcur = Glistdrives; Lcur; Lcur = Lcur ? g_slist_next(Lcur) : Lnext) {
      Ldrive = (Tdriveinfo *)Lcur->data;
      if (g_slist_length(Ldrive->dev)>0) {
        Ltxt = g_strdup_printf(_("Scanning drive \"%s\"..."), Ldrive->name);
        gtk_label_set_text(GTK_LABEL(sc_grave_get_widget(Lg, "labelava")), Ltxt); g_free(Ltxt);
        while(gtk_events_pending()) gtk_main_iteration();

        matos_maj_drive_info(Ldrive, NULL, NULL);

        /* on n'ajoute que les lecteurs/graveurs */
        if (Ldrive->type == 0) {
_DEB("le type == %d = %s\n", Ldrive->type, Ldrive->name);
          Lnext = g_slist_next(Lcur);
          Glistdrives = g_slist_delete_link(Glistdrives, Lcur);
          Lcur = NULL;
        }
      }
    }
  }
  proc_free_cdrominfo();

  /* on attend la fin d'execution de la popup d'attente */
  g_source_remove(Ltimeout);

  gtk_widget_destroy(Lwin);
  if (GTK_IS_WIDGET(Aobj) && Labort == FALSE) {
    Ldata = GTK_TREE_VIEW(sc_grave_get_widget(Lparenthash, "periphdata"));
    remplirelistegraveur(Ldata);

    /* boite de dialoge operation terminee */
    Lstatusbox = gtk_message_dialog_new(GTK_WINDOW(Lwindow1), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
        _("configuration updated."));
    gtk_dialog_run(GTK_DIALOG(Lstatusbox));
    gtk_widget_destroy(Lstatusbox);
  }

  sc_grave_destroy(Lg);
}

void abort_encours(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  gint *Lcont = (gint *)sc_grave_get_data(Lg, "cont");
  gboolean *Labort = (gboolean *)sc_grave_get_data(Lg, "gabort");
  gint *Lpid = (gint *)sc_grave_get_data(Lg, "pid");
printf(" on va kill !\n");
  if (Lpid && *Lpid) {
    /* on envoi un TERM au process (cdrecord...) pour qu'il s'arrete ! */
_DEB("kill de %d\n", *Lpid);
    kill(*Lpid, SIGINT);
  }

  (*Labort) = TRUE;
  (*Lcont) = 1; /* on passe Lcont de 2 a 1 */
}
void delete_event_encours(GtkWidget *Awidget, GdkEvent *Aevent, gpointer Adata)
{
  abort_encours(NULL, Adata);
}

/* on quitte l'application .. */
void menu_quit(GtkWidget *Aobj1, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  gint Loperation = UPDATE_SIZE + WRITE_CONFIG;

  if (conf_get_boolean("saveconfig")==TRUE) {
    Loperation+=UPDATE_SAVEONEXIT_VALUE;
  }

  /* on sauve la position et taille de la fenetre */
  manage_config(Lg, Loperation, NULL);

  gtk_main_quit();
}

void delete_event_quit(GtkWidget *Awidget, GdkEvent *Aevent, gpointer Adata)
{
  menu_quit(Awidget, Adata);
}

#if 0
void on_addmediaformat(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lmediahash = addmediaformat(Adata);
  GtkDialog *Ldialog = GTK_DIALOG(sc_grave_get_widget(Lmediahash, "dialog_media"));

  gtk_dialog_run(Ldialog);
  gtk_widget_destroy(GTK_WIDGET(Ldialog));

  sc_grave_destroy(Lmediahash);
}
#endif

/* mise a jour type cdr destination, on mets donc a jour la barre de remplissage */
void change_remplissage(GtkWidget *Acombo, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  const gchar *Lname = glade_get_widget_name(Acombo);
  gchar *Lvalue = get_combo_value(Acombo);
  GtkProgressBar *Lcontenance;
  guint64 *Lmaxlen;
  guint64 *Llen;

  if (!strcmp(Lname, "listeremplidata")) {  /* cd data */
    Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "datasizebar"));
    Lmaxlen = (guint64 *)sc_grave_get_data(Lg, "datamaxsize");
    Llen = (guint64 *)sc_grave_get_data(Lg, "datasize");
    (*Lmaxlen) = (unsigned long long)strtoul(Lvalue, NULL, 10) * 1024 * 1024;
  } else if (!strcmp(Lname, "listeremplidvddata")) {  /* dvd data */
    Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "dvddatasizebar"));
    Lmaxlen = (guint64 *)sc_grave_get_data(Lg, "dvddatamaxsize");
    Llen = (guint64 *)sc_grave_get_data(Lg, "dvddatasize");
    (*Lmaxlen) = (unsigned long long)strtoul(Lvalue, NULL, 10) * 1024 * 1024;
    
  } else {  /* cd audio */
    Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "audiolenbar"));
    Lmaxlen = (guint64 *)sc_grave_get_data(Lg, "audiomaxlen");
    Llen = (guint64 *)sc_grave_get_data(Lg, "audiolen");
    (*Lmaxlen) = (unsigned long long)strtoul(Lvalue, NULL, 10);
  }

  switch_pbar_status(Lcontenance, *Lmaxlen, *Llen, PB_NONE);
}

/* gestion drag and drop */
void audio_dnd_data_received (GtkWidget *Aliste, GdkDragContext *Acontext, gint Ax, gint Ay, GtkSelectionData *Aseldata, guint Ainfo, guint Atime, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Aliste));
  guint64 *Laudiolen = (guint64 *)sc_grave_get_data(Lg, "audiolen");
  guint64 *Laudiomaxlen = (guint64 *)sc_grave_get_data(Lg, "audiomaxlen");
  GtkProgressBar *Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "audiolenbar"));

  if ((Aseldata->length >= 0) && (Aseldata->format == 8))
  {
    gchar *Lunfic;
    gint i;
    gchar **Lallname = g_strsplit(Aseldata->data, "\r\n", 0);
    gchar *Lfilename = NULL;

    for (i=0; ((Lallname[i])); i++) {
      if ((!(Lunfic = g_strchomp(Lallname[i]))) || !*Lunfic) continue;

      /* on ne gere que les fichiers locaux pour le moment, et c'est deja a !! */
      if (g_ascii_strncasecmp(Lunfic, "file://", 7) || (
            ! (
#ifdef ENABLE_MP3
             (GsupportMp3 && sc_str_has_casesuffix(Lunfic, MP3)) || 
#endif
#ifdef ENABLE_OGG             
             (GsupportOgg && sc_str_has_casesuffix(Lunfic, OGG)) ||
#endif
#ifdef ENABLE_FLAC
             (GsupportFlac && sc_str_has_casesuffix(Lunfic, FLAC)) ||
#endif
             sc_str_has_casesuffix(Lunfic, WAV)))) continue;

      if (!(Lfilename = g_filename_from_uri(Lunfic, NULL, NULL))) continue;
      _add_a_piste(Lfilename, GTK_LIST_STORE(Ltreemodel), Lg);
      g_free(Lfilename);
    }

    g_strfreev(Lallname);
    
    /* mise a jour barre de remplissage */
    switch_pbar_status(Lcontenance, *Laudiomaxlen, *Laudiolen, PB_UPDATE_AUDIO);

    gtk_drag_finish (Acontext, TRUE, FALSE, Atime);
    return;
  } else {
    GtkTreePath *Lpathsrc = NULL;
    GtkTreePath *Lpathdst = NULL;
    GtkTreeIter Litersrc;
    GtkTreeIter Literdst;
    gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(Aliste), Ax, Ay, &Lpathdst, NULL); 
    if (Lpathdst) {
      gtk_tree_view_get_cursor(GTK_TREE_VIEW(Aliste), &Lpathsrc, NULL);
      if (Lpathsrc) {
        gtk_tree_model_get_iter(Ltreemodel, &Litersrc, Lpathsrc);
        gtk_tree_model_get_iter(Ltreemodel, &Literdst, Lpathdst);
        gtk_list_store_move_after(GTK_LIST_STORE(Ltreemodel), &Litersrc, &Literdst); 
        gtk_tree_path_free(Lpathsrc);
        gtk_drag_finish (Acontext, TRUE, FALSE, Atime);
      }
      gtk_tree_path_free(Lpathdst);
    }
  }
  gtk_drag_finish (Acontext, FALSE, FALSE, Atime);
}

void data_dnd_data_received (GtkWidget *Aliste, GdkDragContext *Acontext, gint Ax, gint Ay, GtkSelectionData *Aseldata, guint Ainfo, guint Atime, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  guint64 *Ldatasize;
  guint64 *Ldatamaxsize;
  GtkProgressBar *Lcontenance;
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Aliste));

  _REMPLISSAGE_VALUE(Lg, Aliste, Ldatasize, Ldatamaxsize, Lcontenance);

  if ((Aseldata->length >= 0) && (Aseldata->format == 8))
  {
    gchar *Lunfic;
    gint i;
    gchar **Lallname = g_strsplit(Aseldata->data, "\n", 0);
    gchar *Lfilename = NULL;
    GtkTreeIter Liter, Lfirstiter;
    GtkTreePath *Lpath = NULL;

    gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(Aliste), Ax, Ay, &Lpath, NULL); 
    if (!Lpath) {
      if (!gtk_tree_model_get_iter_first(Ltreemodel, &Liter)) {
        gtk_drag_finish (Acontext, FALSE, FALSE, Atime);
        return;
      }
    } else {
      gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath);
      gtk_tree_path_free(Lpath);
    }

    for (i=0; ((Lallname[i])); i++) {
      if ((!(Lunfic = g_strchomp(Lallname[i]))) || !*Lunfic) continue;
      if (g_ascii_strncasecmp(Lunfic, "file://", 7)) continue;
      if (!(Lfilename = g_filename_from_uri(Lunfic, NULL, NULL))) continue;

      _add_a_file(Lfilename, GTK_TREE_STORE(Ltreemodel), &Liter, Lg, Ldatasize,
          i == 0 ? &Lfirstiter : NULL);
      
      g_free(Lfilename);
    }

    g_strfreev(Lallname);

    /* expand element ajoute */
    if ((Lpath = gtk_tree_model_get_path(Ltreemodel, &Lfirstiter))) {
      gtk_tree_view_expand_to_path(GTK_TREE_VIEW(Aliste), Lpath);
      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(Aliste), Lpath, NULL, FALSE, 0, 0);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(Aliste), Lpath, NULL, TRUE);
      gtk_tree_path_free(Lpath);
    }

    /* mise a jour barre de remplissage */
    switch_pbar_status(Lcontenance, *Ldatamaxsize, *Ldatasize, PB_UPDATE_DATA);
    
    gtk_drag_finish (Acontext, TRUE, FALSE, Atime);
    return;
  } else {
    GtkTreePath *Lpathsrc = NULL;
    GtkTreePath *Lpathdst = NULL;
    GtkTreeIter Litersrc, Literdst, Liternew;
    gint Ltype;
    gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(Aliste), Ax, Ay, &Lpathdst, NULL); 
    if (Lpathdst) {
      gtk_tree_view_get_cursor(GTK_TREE_VIEW(Aliste), &Lpathsrc, NULL);
      if (Lpathsrc) {
        if (gtk_tree_path_is_descendant(Lpathdst, Lpathsrc)) {

          gtk_tree_path_free(Lpathsrc);
          gtk_tree_path_free(Lpathdst);
          gtk_drag_finish (Acontext, FALSE, FALSE, Atime);
          return;
        }
        
        gtk_tree_model_get_iter(Ltreemodel, &Litersrc, Lpathsrc);
        gtk_tree_model_get_iter(Ltreemodel, &Literdst, Lpathdst);

        gtk_tree_model_get(Ltreemodel, &Literdst, 0, &Ltype, -1);
     
        /* on ne deplace pas le CD root */ 
        if (Ltype == TYPE_CD) return;

        if (Ltype == TYPE_FILE) {
          gtk_tree_store_insert_after(GTK_TREE_STORE(Ltreemodel), &Liternew, NULL, &Literdst);
        } else {
          gtk_tree_store_insert_after(GTK_TREE_STORE(Ltreemodel), &Liternew, &Literdst, NULL);
        }

        /* copie recursive de l'element et des enfants */
        sc_line_data_copy(Ltreemodel, &Litersrc, &Liternew);

        gtk_tree_path_free(Lpathsrc);
        gtk_tree_path_free(Lpathdst);
        gtk_drag_finish (Acontext, TRUE, TRUE, Atime);

        return;
      }
      gtk_tree_path_free(Lpathdst);
    }
  }

  gtk_drag_finish (Acontext, FALSE, FALSE, Atime);
}

#if 0
void update_media_unit_range(gchar *Aunit)
{
//  gtk_spin_button_set_range

}
    
void update_media_unit(GtkWidget *Acombo, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkWidget *Ldataentry = GTK_WIDGET(g_hash_table_lookup(Lg, "mediadata"));

  gtk_widget_grab_focus(Ldataentry);
}

/* mise a jour des champs dans la fenetre "configuration des medias" */
gboolean update_media_fields(GtkWidget *Awidget, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  const gchar *Lname = glade_get_widget_name(GTK_WIDGET(Awidget));
  gulong Lnewsize;
  gulong Lasksize;
  gulong *Lsize = (gulong *)g_hash_table_lookup(Lg, "fullsize");
  GtkComboBox *Lmediadataunit = GTK_COMBO_BOX(g_hash_table_lookup(Lg, "mediadataunit"));
  gchar *Lcharunit = get_combo_value(GTK_WIDGET(Lmediadataunit));

 // printf("l'unite est => %s\n", Lcharunit);
  Lasksize = (gulong) gtk_spin_button_get_value(GTK_SPIN_BUTTON(Awidget));
  
//      printf("oui [%c]\n", *Lcharunit);
    /* mise a jour depuis la taille */
//  printf("asjsize avant => %ld\n", Lasksize);
    if (*Lcharunit == 'g') {
      Lasksize*=1024 * 1024 * 1024;
    } else if (*Lcharunit == 'm') {
      Lasksize*=1024 * 1024;
     } else {
 //     printf("oui kb\n");
      Lasksize*= 1024;
   }

    gtk_spin_button_set_value(Lmediaaudio, Lasksize );
    *Lsize = Lasksize;
//printf("la taille est donc de = %ld\n", *Lsize);

    g_free(Lcharunit);
  }

  return FALSE;
}
#endif

/* charger un fichier, importer une playlist */
void menu_load(GtkMenuItem *Amenuitem, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkFileFilter *filterm3u, *filterpts, *filterall;
  GtkWidget *Lparent = sc_grave_get_widget(Lg, "topwindow");
  guint32 *Laudiolen = (guint32 *)sc_grave_get_data(Lg, "audiolen");
  guint32 *Laudiomaxlen = (guint32 *)sc_grave_get_data(Lg, "audiomaxlen");
  GtkProgressBar *Lcontenance = GTK_PROGRESS_BAR(sc_grave_get_widget(Lg, "audiolenbar"));
  GError *Lerror = NULL;
  GtkWidget *Lfileselection;
  gint Lresp;
  gboolean LnewPl = (strcmp(glade_get_widget_name(GTK_WIDGET(Amenuitem)), "popupaudio_append"));
  
  /* filtres pour la selection de fichiers */
  filterall = gtk_file_filter_new ();
  gtk_file_filter_set_name (filterall, _("All playlists files (*.m3u *.pts)"));
  gtk_file_filter_add_pattern(filterall, "*.m3u");
  gtk_file_filter_add_pattern(filterall, "*.pts");
  filterm3u = gtk_file_filter_new ();
  gtk_file_filter_set_name (filterm3u, _("M3u and m3u extended playlists only (*.m3u)"));
  gtk_file_filter_add_pattern(filterm3u, "*.m3u");
  filterpts = gtk_file_filter_new ();
  gtk_file_filter_set_name (filterpts, _("Pts playlists only (*.pts)"));
  gtk_file_filter_add_pattern(filterpts, "*.pts");

  Lfileselection = gtk_file_chooser_dialog_new(LnewPl ? _("Import audio play list") : _("Append audio play list"),
      GTK_WINDOW(Lparent), GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

  gtk_dialog_set_default_response (GTK_DIALOG (Lfileselection), GTK_RESPONSE_ACCEPT);
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (Lfileselection), filterall);
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (Lfileselection), filterm3u);
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (Lfileselection), filterpts);

  use_last_dir(GTK_FILE_CHOOSER(Lfileselection));
  Lresp=gtk_dialog_run(GTK_DIALOG(Lfileselection));

  gtk_widget_hide (Lfileselection);

  if (Lresp == GTK_RESPONSE_ACCEPT) {
    gchar *Lfilenamebrut = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER (Lfileselection));
    gchar *Lfilename = _FILENAME(Lfilenamebrut);

    save_last_dir(GTK_FILE_CHOOSER(Lfileselection));

    if (sc_str_has_casesuffix(Lfilename, M3U)) {
      m3u_import(Lfilename, Lg, LnewPl, &Lerror);
    } else if (sc_str_has_casesuffix(Lfilename, PLS)) {
      pls_import(Lfilename, Lg, LnewPl, &Lerror);
    }
    
    g_free(Lfilename);
    g_free(Lfilenamebrut);

    if (Lerror) {
      /* erreur */
      GtkWidget *Lstatusbox = gtk_message_dialog_new(GTK_WINDOW(Lparent),
                                GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                                Lerror->message); 
      gtk_dialog_run(GTK_DIALOG(Lstatusbox));

      gtk_widget_destroy(Lstatusbox);
      g_error_free(Lerror);
    } else {
      /* on active le panel "cd audio" si on n'y est pas */
      menu_active_panel(sc_grave_get_widget(Lg, "menucdaudio"), Lg);
    }
  }

  gtk_widget_destroy(Lfileselection);
  switch_pbar_status(Lcontenance, *Laudiomaxlen, *Laudiolen, PB_UPDATE_AUDIO);
}

/* debuter un nouveau projet */
void menu_start_new(GtkMenuItem *Amenuitem, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Ltopmenu = GTK_TREE_VIEW(sc_grave_get_widget(Lg, "topmenu"));
  gchar *Lname;
  GtkTreeIter Liter;
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(Ltopmenu);
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(Ltopmenu);

  if (!gtk_tree_selection_get_selected(Lselection, &Ltreemodel, &Liter)) return;
  gtk_tree_model_get(Ltreemodel, &Liter, 0, &Lname, -1);

  if (!strcmp(Lname, "bodycdaudio")) {
    /* on est sur l'onglet cd audio, on vide la liste */
    clear_list_audio(Lg);
  } else if (!strcmp(Lname, "bodycddata") || !strcmp(Lname, "bodydvddata")) {
    /* on est sur l'onglet cd data, on vide la liste */
    GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
    GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Lliste));
    GtkProgressBar *Lcontenance;
    guint64 *Ldatasize;
    guint64 *Ldatamaxsize;
    GtkTreeIter Liter;
  
    _REMPLISSAGE_VALUE(Lg, Lliste, Ldatasize, Ldatamaxsize, Lcontenance);
    (*Ldatasize) = 0;

    gtk_tree_store_clear(GTK_TREE_STORE(Ltreemodel));
    /* icone de base dans la liste des fichiers */
    gtk_tree_store_append(GTK_TREE_STORE(Ltreemodel), &Liter, NULL);
    gtk_tree_store_set(GTK_TREE_STORE(Ltreemodel), &Liter, 0, TYPE_CD, 1, get_image("CDsmallIcon"), 2,
        !strcmp(Lname, "bodycddata") ? _("New CD") : _("New DVD"), 3, "", 4, "", 5, 0, -1);
    gtk_tree_view_columns_autosize(Lliste); 
    switch_pbar_status(Lcontenance, *Ldatamaxsize, *Ldatasize, PB_UPDATE_DATA);
  }

  g_free(Lname);
}

/* lorsque l'on coche/decoche "continuer un cd multi session" */
void change_datamulti(GtkWidget *Amulti, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  gboolean Lact = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Amulti));
  GtkSpinButton *Lspin = GTK_SPIN_BUTTON(sc_grave_get_widget(Lg, "nbrcddata"));

  if (Lact) gtk_spin_button_set_value(Lspin, 1);
  gtk_widget_set_sensitive(GTK_WIDGET(Lspin), ! Lact);
}

/* select items begining with key user has pressed */
gboolean callbacks_try_to_select(guint Akey, GtkTreeView *Aliste, Tgrave *Ag)
{
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Aliste));
  GtkTreePath *Lpath = NULL;
  gchar *Lname;
  GtkTreeIter Liter;
  gboolean Ltrouve = FALSE;
  gboolean Lfirstiteration = TRUE;
  gboolean Lskip = FALSE;

  gtk_tree_view_get_cursor(Aliste, &Lpath, NULL);
  if (!Lpath) return FALSE;

  while (1) {
    while (1) {
      if (Lskip == FALSE) {
        gtk_tree_path_next(Lpath);
      } else {
        Lskip = FALSE;
      }
      if (!gtk_tree_model_get_iter(Ltreemodel, &Liter, Lpath)) break;

      gtk_tree_model_get(Ltreemodel, &Liter, 2, &Lname, -1);
      if (!Lname) continue;

      if (*(tools_next_no_char(Lname, (gchar ) GDK_space)) == Akey) {
        Ltrouve = TRUE;
        break;
      }
    }

    if (Lfirstiteration == FALSE || Ltrouve == TRUE) {
      break;
    } else {
      Lfirstiteration = FALSE;
      while (gtk_tree_path_prev(Lpath)) ;
      Lskip = TRUE;
    }
  }

  if (Ltrouve) {
    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(Aliste), Lpath, NULL, FALSE, 0, 0);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(Aliste), Lpath, NULL, TRUE);
  }

  gtk_tree_path_free(Lpath);

  return Ltrouve;
}

/* gestion des tris */
void callbacks_data_asksort(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(sc_grave_get_data(Lg, "_current_list"));
  GtkTreeModel *Ltreemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(Lliste));
  GtkTreeSortable *Lsort = NULL;
/*  gint Lcolid;
  GtkSortType Lorder;*/
  const gchar *Lname;

  if (!GTK_IS_WIDGET(Abtn) || !GTK_IS_WIDGET(Lliste)) return;
  Lsort = GTK_TREE_SORTABLE(Ltreemodel);

  if ((!(Lname = gtk_widget_get_name(Abtn))) || strlen(Lname)<10) return;

/*  if (!gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(Lmodel), &Lcolid, &Lorder)) return;*/
  if (strstr(Lname, "_name")) {
    gtk_tree_sortable_set_sort_column_id(Lsort, 2, GTK_SORT_ASCENDING);
  } else if (strstr(Lname, "_size")) {
    gtk_tree_sortable_set_sort_column_id(Lsort, 3, GTK_SORT_ASCENDING);
  } else {
    gtk_tree_sortable_set_sort_column_id(Lsort, 0, GTK_SORT_ASCENDING);
  }

}

/* definir/enlever l'image de boot */
#if 0
void gtk_def_imageboot(GtkWidget *Abtn, gpointer Adata)
{
  Tgrave *Lg = (Tgrave *)Adata;
  GtkTreeView *Lliste = GTK_TREE_VIEW(g_hash_table_lookup(Lg, "LISTEDATA"));
  GtkTreeModel *Ltreemodel;
  GtkTreeSelection *Lselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(Lliste));
  GtkTreeIter *Liter = g_hash_table_lookup(Lg, "file_imageboot");
  GtkTreeIter Lnewiter;

  if (!gtk_tree_selection_get_selected(Lselection, &Ltreemodel, &Lnewiter)) return;

  if (Liter) {
    gtk_tree_store_set(GTK_TREE_STORE(Ltreemodel), Liter, 1, GFileIcone, -1);
    gtk_tree_iter_free(Liter);
  }

  gtk_tree_store_set(GTK_TREE_STORE(Ltreemodel), &Lnewiter, 0, TYPE_IMAGEBOOT, 1, GFileImage, -1);

  g_hash_table_replace(Lg, "file_imageboot", gtk_tree_iter_copy(&Lnewiter));
}
#endif

/*
 * vim:et:ts=8:sts=2:sw=2
 */
