/*
 * ChainSawHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2002 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.1
 * Created by suhler on 99/04/08
 * Last modified by suhler on 02/10/01 16:36:27
 */

package sunlabs.brazil.handler;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat; 
import java.util.Date; 
import sunlabs.brazil.server.ChainHandler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;

/**
 * Variant of the chain handler for doing standard logging.
 * Don't use on fine furniture.
 * <p>
 * Output is a variant of the common logfile format.
 * The common logfile format is as follows: 
 * <pre>
 * remotehost rfc931 authuser [date] "request" status bytes
 * </pre>
 * <dl>
 * <dt>remotehost
 * <dd>Remote hostname (or IP number if DNS hostname is not available, or if DNSLookup is Off. 
 * <dt>rfc931
 * <dd>The remote logname of the user. 
 * <dt>authuser
 * <dd>The username as which the user has authenticated himself. 
 * <dt>[date]
 * <dd>Date and time of the request. 
 * <dt>"request"
 * <dd>The request line exactly as it came from the client. 
 * <dt>status
 * <dd>The HTTP status code returned to the client. 
 * <dt>bytes
 * <dd>The content-length of the document transferred. 
 * <dt> "referrer" (optional)
 * <dd> the referring url
 * <dt> "user agent" (optional)
 * <dd> "The user agent making the request
 * </dl>
 * <p>
 * Additional Configuration options:
 * <dl class=props>
 * <dt> logFile <dd> The name of the file to write the logs to.
 * <dt> flush   <dd> The number of requests between flushes to the file
 * </dl>
 * If the logFile is removed, the server creates a new one.
 * Thus logs may be truncated by periodically moving them to
 * another name (at least on unix).
 * <p>
 * See the {@link LogHandler} handler for generating logs whose
 * contents are configurable.
 *
 * @author		Stephen Uhler
 * @version		2.1, 02/10/01
 */

public class ChainSawHandler extends ChainHandler {
    static public final String LOG = "logFile";
    static public final String FLUSH = "flush";
    static public final int BUFSIZE = 4096;
    static SimpleDateFormat sdf;	// standard log format date string
    DataOutputStream log; 	// where to log the output to
    File file;
    int count = 0;		// count flushing interval
    int flush = 1;		// how may requests to flush at

    static {
       sdf = new SimpleDateFormat("dd/MMM/yyyy:hh:mm:ss Z"); 
    }

    public boolean
    init(Server server, String prefix) {
	String logFile = server.props.getProperty(prefix + LOG,
		server.hostName + "-" + server.listen.getLocalPort() + ".log");
	try {
	    flush = Integer.parseInt(
	            server.props.getProperty(prefix + FLUSH, "1"));
	} catch (NumberFormatException e) {
	// ignore
	}

	try {
	    file = new File(logFile);
	    log = new DataOutputStream(new BufferedOutputStream(
		    new FileOutputStream(file),BUFSIZE));
	} catch (IOException e) {
	    server.log(Server.LOG_WARNING, prefix, e.toString());
	    return false;
	}
	server.log(Server.LOG_DIAGNOSTIC, prefix, "Log file: " + logFile);
	return super.init(server, prefix);
    }

    /**
     * Run the chain-handler, counting the # of bytes of output generated
     * by its chained handlers.
     */

    public boolean
    respond(Request request) throws IOException {
    	String req = "\"" + request.method + " " + request.url + " " +
		request.protocol + "\"";
	String now = sdf.format(new Date());
    	boolean result = false;
	IOException oops = null;

    	try {
	    result = super.respond(request);
	} catch (IOException e) {
	    oops = e;
	}
	int code =  200;
	if (!result) {
	    code = 404;
	}
	request.out.flush();
    	
	String refer;
	if (request.headers.get("Referer") != null) {
	    refer = "\"" + request.headers.get("Referer") + "\"";
	} else {
	    refer = "-";
	}

	String agent;
	if (request.headers.get("User-Agent") != null) {
	    agent = "\"" + request.headers.get("User-Agent") + "\"";
	} else {
	    agent = "-";
	}

    	count = request.out.bytesWritten;
    	String msg = request.getSocket().getInetAddress().getHostAddress() +
    		" - - [" + now + "]" + " " + req + " " +
    		code + " " + count + " " + refer + " " + agent + "\n";

        /*
         * Write to the log file.  If there's a problem, try to open
	 * the file again.
         */
        
        if (!file.exists()) {
	    log.flush();
	    request.log(Server.LOG_WARNING, "Log file went away!");
	    log = new DataOutputStream(new BufferedOutputStream(
	    	    new FileOutputStream(file), BUFSIZE));
	    count = 0;
        }

	log.writeBytes(msg);
	request.log(Server.LOG_DIAGNOSTIC, msg);
	if (count++ >= flush) {
	    log.flush();
	    count = 0;
	}
    	if (oops != null) {
	    throw oops;
	}
    	return result;
    }
}
