/*  amplitude.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef AMPLITUDE_H_
#define AMPLITUDE_H_

#include <ginac/ginac.h>

namespace Reduze {

// helper function too keep Lorentz structure together
DECLARE_FUNCTION_1P( Lorentz)

class FeynmanRules;
class ClosedFermionChain;

/// holds the analytical expressions of a open fermion chain, sorted against the charge flow (eg vbar, propagators, u)
class OpenFermionChain {
public:
	/*OpenFermionChain();*/
	/// create a chain
	// left is a conjugated spinor (ubar, vbar) or a conjugated ghost
	OpenFermionChain(const GiNaC::ex& left, const GiNaC::exvector& line,
			const GiNaC::ex& right);

	// getters
	const GiNaC::ex& left() const {
		return left_;
	}
	const GiNaC::ex& right() const {
		return right_;
	}
	const GiNaC::ex& value() const {
		return value_;
	}

	/// complex conjugation and revert chain
	OpenFermionChain conjugate() const;
	/// substitute
	OpenFermionChain
	substitute(const GiNaC::exmap& m, unsigned int opt = 0) const;

	/// sort the open fermion chains in multiple lists, each like:  { {u_1_bar X u_2}, {u_2_bar X u_3}, ... {u_n_bar X u_1} }
	/// such that they are ready for applying the completeness relations (spin summation)
	/// verify each spinor/ghost has a conjugated partner
	/// input: the fermion chains of two Amplitudes (Matrixelements) to be multiplied
	static std::vector<std::vector<OpenFermionChain> > sort_open_fermion_chains(
			const std::vector<OpenFermionChain>& ofcs1,
			const std::vector<OpenFermionChain>& ofcs2);

private:
	OpenFermionChain(const GiNaC::ex& left, const GiNaC::ex& value,
			const GiNaC::ex& right);

	GiNaC::ex left_; // indexed object containing spinorbar, (ghost) conjugated polarization;
	GiNaC::ex right_; //indexed object containing spinor, (ghost) polarization;
	/// non-commutative value
	GiNaC::ex value_;

	friend std::ostream& operator<<(std::ostream& os,
			const OpenFermionChain& line);
};

/// holds the analytical expressions of a closed spinor chain, sorted against the charge flow
class ClosedFermionChain {
public:
	ClosedFermionChain();
	ClosedFermionChain(const GiNaC::exvector& line);

	ClosedFermionChain conjugate() const;
	GiNaC::ex value() const;
	ClosedFermionChain
	substitute(const GiNaC::exmap& m, unsigned int opt = 0) const;

	/// closes the open fermion chains from eg two Amplitudes by applying the spin-summation rules from the FeynmanRules
	static std::vector<ClosedFermionChain>
	close_open_fermion_chains(const std::vector<OpenFermionChain>& ofcs1,
			const std::vector<OpenFermionChain>& ofcs2, const FeynmanRules* fr);

private:
	explicit ClosedFermionChain(const GiNaC::ex& value);

	GiNaC::ex value_;

	friend std::ostream& operator <<(std::ostream& os,
			const ClosedFermionChain& line);
};

inline GiNaC::ex ClosedFermionChain::value() const {
	return value_;
}

class Sector;
class LinearCombination;

/// class for storing the analytical expression of a diagram
class Amplitude {
public:
	Amplitude();

	void insert(const OpenFermionChain& e);
	void insert(const ClosedFermionChain& e);
	void set_external_bosons(const GiNaC::exvector& e);
	void set_amp(const GiNaC::ex& e);
	void set_sector(const Sector& sector);
	void set_loop_momenta(const GiNaC::lst& l) {
		loop_momenta_ = l;
	}

	void set_local_indices(const std::map<std::string, GiNaC::symbol>& indices);
	void set_name(const std::string& name);

	const Sector& sector() const;

	const std::string& name() const;
	const GiNaC::lst& loop_momenta() const {
		return loop_momenta_;
	}

	const std::vector<OpenFermionChain>& open_fermion_chains() const {
		return open_fermion_chains_;
	}
	const std::vector<ClosedFermionChain>& closed_fermion_chains() const {
		return closed_fermion_chains_;
	}
	const GiNaC::exvector& external_bosons() const {
		return external_bosons_;
	}
	const GiNaC::ex& amp() const {
		return amp_;
	}

	Amplitude conjugate() const;

	/// introduces new, unique index names
	void make_new_index_names();

private:
	/// name of the amplitude
	std::string name_;
	/// open chains, spinors or ghosts
	std::vector<OpenFermionChain> open_fermion_chains_;
	/// closed chains, spinors or ghosts
	std::vector<ClosedFermionChain> closed_fermion_chains_;
	/// external bosonic legs
	GiNaC::exvector external_bosons_;
	/// rest of the amplitude (commutative factors)
	GiNaC::ex amp_;
	/// loop momenta
	GiNaC::lst loop_momenta_;
	/// one or zero element list of sectors
	std::list<Sector> sector_;
	/// local indices by name
	std::map<std::string, GiNaC::symbol> local_indices_;

	friend std::ostream& operator <<(std::ostream& os, const Amplitude& amp);
};

/// interference term infos
struct EquationInfo {
	EquationInfo() : max_color_order_(0), valid_equation_(true) {}
	int max_color_order_;
	std::string amplitude_name_;
	bool valid_equation_;
};

/// class to store and calculate multiplied amplitudes
class AmplitudeProd {
public:
	/// Constructs the product of the two amplitudes
	AmplitudeProd(const Amplitude& amp1, const Amplitude& amp2,
			const FeynmanRules* fr);

	const std::string& name() const;
	const Sector& sector() const;

	void set_sector(const Sector& sector);

	/// returns the amplitude as a LinearCombination, calculates: Dirac traces, Lorentz contractions
	EquationInfo calculate(LinearCombination& lincomb) const;

private:
	/// a name
	std::string name_;
	/// contains the closed spinor loops
	GiNaC::exvector spinor_traces_;
	/// commutative part of the amplitude
	GiNaC::ex amp_;

	GiNaC::lst loop_momenta_;
	std::list<Sector> sector_;

	const FeynmanRules* fr_;

private:
	/// inserts the fermion chains, return commutative overall factors
	GiNaC::ex insert(const std::vector<ClosedFermionChain>& lines);

	/// calculates the dirac traces in the vector
	GiNaC::ex calc_dirac(const GiNaC::exvector& traces,
			const GiNaC::ex& prefactor) const;
	//GiNaC::ex calc_lorentz(const GiNaC::ex& lor, bool is_expanded = false) const;
	GiNaC::ex calc_color(const GiNaC::ex& col, int& Nc_Degree) const;

	friend std::ostream
	& operator <<(std::ostream& os, const AmplitudeProd& amp);

};

inline const std::string& AmplitudeProd::name() const {
	return name_;
}

} // namespace Reduze

#endif /* AMPLITUDE_H_ */
