/*  dfspalmtree.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef DFSPALMTREE_H_
#define DFSPALMTREE_H_

#include "topology.h"

namespace Reduze {

/// A palm tree representing a depth-first-search of a Topology
/** This is a helper class for linear biconnected and triconnected component
 ** decompositions, see SplitComponents. It implements the algorithms described
 ** by Hopcroft, Tarjan (1973) and corrected by Gutwenger, Mutzel (2001). */
class DFSPalmTree {
public:
	/// creates a depth-first-search tree for a biconnected graph
	/** The graph 't' must be biconnected, the nodes and edges must have
	 ** non-negative ids (maximal ids should not differ too much from total
	 ** numbers), the graph must not have multiple edges. */
	DFSPalmTree(const Topology& t, int start_node, int max_node_id,
			int max_edge_id, int max_original_edge_id);
	/// determines split components of a biconnected graph
	/** procedure 6 of HT73,
	 ** Returned components are triple bonds, triangles and triconnected
	 ** graphs.
	 ** Note: call only once, repeated calls undefined */
	std::list<std::list<Edge> > find_split_components();
	/// return true if the egde id belongs to a virtual edge
	bool is_virtual_edge(int id);
	/// return maximal edge id including virtual edges
	int max_edge_id() const;
	/// return maximal id of real edges
	int max_original_edge_id() const;
	/// produces a DOT file for graphical representation of the tree
	void print_dot(const std::string& filename) const;
	/// produces a postscript file with graphics for the tree
	void print_postscript(const std::string& filename) const;
	/// find paths (changes numbering etc)
	/** procedure 5 of HT73 */
	const std::list<std::list<Edge> >& find_paths();

private:
	struct Triple {
		Triple();
		Triple(int h, int a, int b);
		int h, a, b;
		// invalid values are used as tstack separators and default values
		bool is_valid() const;
	};

	// note: virtual edges start at edge_type_.size()
	size_t num_nodes() const;
	/// return maximal node id
	int max_node_id() const;
	/// from and to in returned Edge is ordered according to palm tree
	Edge directed_edge(const Edge& e) const;
	/// construct ordered adjacency lists, invalidates various members
	/** procedure 4 in HT73 **/
	void construct_acceptable_adjacencies();
	/// updates lowpoints lo1 and lo2 for a directed path to cand
	void update_min(int& lo1, int& lo2, int cand);
	/// recursive build of DFS palm tree from topology
	/** procedure 1 + procedure 3 in HT73 **/
	void build_DFS_palm_tree(const Topology& topo, int v /*node*/);
	/// recursive helper for find_paths()
	void find_paths(int node, int& last_num_assigned, bool& is_path_started);
	int first_son(int v) const;
	const Edge& father_arc(int v) const;
	int father(int v) const;
	int highpos(int w) const;
	static void print_estack(const std::stack<Edge>& estack);
	static void print_tstack(const std::stack<Triple>& tstack);
	// implementation of HT73 algorithm:
	// we use some implementation tricks found in OGDF package,
	// class TriComp, by Carsten Gutwenger, 2010
	void find_split_components(int v,//
			std::stack<Triple>& tstack,//
			std::stack<Edge>& estack,//
			std::list<std::list<Edge> >& comps);
	// helpers for find_split_components():
	void check_for_type_2_pairs(int, std::stack<Triple>&, std::stack<Edge>&,
			std::list<std::list<Edge> >&, int&);
	void check_for_type_1_pair(int, std::stack<Triple>&, std::stack<Edge>&,
			std::list<std::list<Edge> >&, int&);
	Edge new_vedges(int from, int to, std::list<Edge>& comp);
	void move_edge(Edge e, std::list<Edge>& comp);
	void make_tree_arc(const Edge& e);

	enum EdgeType {
		NotVisited, TreeArc, PalmFrond
	};

	// note: we use original node indices (and not number[]) for lowpt1 etc.,
	// so we need no update in second DFS search
	int num_numbers_;
	int root_;
	std::vector<int> pos_; // position wrt DFS ("number/newnumber" in HT73)
	std::vector<int> index_of_pos_; // inversion of pos_
	std::vector<int> lowpt1_;
	std::vector<int> lowpt2_;
	std::vector<int> nd_; // number of descendants
	int max_edge_id_; // highest assigned edge id within find_split_components()
	int max_orig_edge_id_; // highest id of non-virtual edges
	// edge stored for tail node (defines direction in DFSTree, Edge.from/to in original order!):
	std::vector<std::list<Edge> > outgoing_; // outgoing edges of a vertex
	std::vector<EdgeType> edge_type_;
	std::vector<bool> is_first_edge_of_path_;
	std::list<std::list<Edge> > paths_;
	// set up and used in find_split_components() only:
	std::vector<int> degree_;
	std::vector<std::list<Edge> > incoming_; // incoming edges of a vertex
	std::vector<std::list<Edge>::iterator> incoming_it_; // element in incoming_ based on edge.id
	std::vector<std::list<Edge>::iterator> outgoing_it_; // element in outgoing_ based on edge.id
	std::vector<int> num_unvisited_outarcs_;
};

} // namespace Reduze

#endif /* DFSPALMTREE_H_ */
