/*  diagram.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "ginac/ginac.h"
#include "graph.h"
#include "diagram.h"
#include "amplitude.h"
#include "functions.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "files.h"
#include "integralfamily.h"
#include "polarization.h"
#include "feynmanrules.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// register type
namespace {
//YAMLProxy<Field> dummy1;
YAMLProxy<PropagatorFields> dummy2;
YAMLProxy<VertexFields> dummy3;
YAMLProxy<ExternalField> dummy4;
YAMLProxy<Diagram> dummy5;
}

Field::Field() :
	name_("undef"), field_type_(FieldType::undef), anti_commuting_(false),
			momentum_(0), mass_(0), field_index_(0) {
}

Field::~Field() {
}

void Field::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << Flow << BeginSeq;
	os << name_;
	os << field_type_;
	if (anti_commuting_)
		os << -1;
	else
		os << 1;
	os << momentum_;
	os << mass_;
	os << field_index_;
	os << EndSeq;
}

void Field::read(const YAML::Node& n, const GiNaC::lst& symbols) {
	//verify_yaml_spec(n);
	if (n.Type() != YAML::NodeType::Sequence && n.size() != 6)
		throw runtime_error("node is not a sequence with 6 entries "
				+ position_info(n));
	int ac;
	n[0] >> name_;
	n[1] >> field_type_;
	n[2] >> ac;
	anti_commuting_ = ac < 0 ? true : false;
	Reduze::read(n[3], momentum_, symbols);
	Reduze::read(n[4], mass_, symbols);
	n[5] >> field_index_;
}

void Field::set_field_index(int new_field_index) {
	field_index_ = new_field_index;
}

void Field::substitute_momenta(const GiNaC::exmap& m, unsigned options) {
	momentum_ = momentum_.subs(m, options).expand();
}

void Field::substitute_mass(const GiNaC::exmap& m, unsigned options) {
	mass_ = mass_.subs(m, options).expand();
}

void Field::read_stream(std::istream& stream, const GiNaC::lst& symbols) {
	string mom, mass;
	int ac;
	stream >> name_ >> field_type_ >> ac >> mom >> mass >> field_index_;
	anti_commuting_ = ac < 0 ? true : false;
	momentum_ = ex(mom, symbols);
	mass_ = ex(mass, symbols);
}

void Field::write_stream(std::ostream& s) const {
	s << name_ << " " << field_type_ << " " << (anti_commuting_ ? -1 : 1)
			<< " " << momentum_ << " " << mass_ << " " << field_index_;
}

//
//
//

FieldTuple::FieldTuple() {
}

FieldTuple::~FieldTuple() {
}

void FieldTuple::set_symbols(const GiNaC::lst& symbols) {
	symbols_ = symbols;
}

void FieldTuple::substitute_momenta(const GiNaC::exmap& m, unsigned options) {
	for (vector<Field>::iterator it = fields_.begin(); it != fields_.end(); ++it)
		it->substitute_momenta(m, options);
}

void FieldTuple::substitute_mass(const GiNaC::exmap& m, unsigned options) {
	for (vector<Field>::iterator it = fields_.begin(); it != fields_.end(); ++it)
		it->substitute_mass(m, options);
}

void FieldTuple::replace_field_indices(const std::map<int, int>& old_to_new) {
	map<int, int>::const_iterator found;
	for (vector<Field>::iterator f = fields_.begin(); f != fields_.end(); ++f) {
		found = old_to_new.find(f->field_index());
		if (found != old_to_new.end())
			f->set_field_index(found->second);
	}
}

bool FieldTuple::has_anti_commuting_field() const {
	for (vector<Field>::const_iterator f = fields_.begin(); f != fields_.end(); ++f) {
		if (f->anti_commuting())
			return true;
	}
	return false;
}

/// creates a new symbol from a string, or returns the symbol from the map
static GiNaC::symbol symbol_factory(const string& s,
		map<string, GiNaC::symbol>& m) {
	map<string, symbol>::iterator it = m.find(s);
	if (it != m.end())
		return it->second;
	symbol s_symb(s);
	m.insert(make_pair(s, s_symb));
	return s_symb;
}

GiNaC::exmap FieldTuple::mom_mass_indices_map(const FeynmanRules* fr, map<
		string, GiNaC::symbol>& indices) const {
	exmap res;
	ASSERT(!fields_.empty());
	if (int(fields_.size()) > fr->max_vertex_degree()) {
		string n_max = to_string(fr->max_vertex_degree());
		string n_vert = to_string(fields_.size());
		throw runtime_error("Encountered vertex with more fields (=" + n_vert
				+ ") than max_vertex_degree (=" + n_max + ") allows.");
	}
	const map<int, string>& ind_pref = fr->index_prefixes();
	for (unsigned j = 0; j < fields_.size(); ++j) {
		const Field& f = fields_[j];
		int i = j + 1;
		res[fr->mom(i)] = f.momentum();
		res[fr->mass(i)] = f.mass();
		if (ft_is_a<ExternalField> (this))
			res[fr->polarization_orthogonal_vector(i)] = ft_to<ExternalField> (
					this)->polarization_orthogonal_vector();
		// replace the all template indices with prefix p of field f by a symbol prefix + field index
		map<int, string>::const_iterator p;
		for (p = ind_pref.begin(); p != ind_pref.end(); ++p) {
			// take negative field indices for the internal fields (dummy indices)
			int suffix = -f.field_index();
			string s_str = p->second + to_string(suffix);
			symbol s = symbol_factory(s_str, indices);
			res[fr->index(p->second, i)] = s;
		}
	}
	return res;
}

GiNaC::ex FieldTuple::feynman_rule_value(const FeynmanRules* fr, std::map<
		std::string, GiNaC::symbol>& indices) const {
	ex value = fr->rule(fr_id_);
	exmap subs = mom_mass_indices_map(fr, indices);
	return value.subs(subs);
}

GiNaC::ex FieldTuple::momentum(int i) const {
	if (i < 1 || i > int(fields_.size()))
		ABORT("Field " << i << " does not exist");
	return fields_[i - 1].momentum();
}

GiNaC::ex FieldTuple::mass(int i) const {
	if (i < 1 || i > int(fields_.size()))
		ABORT("Field " << i << " does not exist");
	return fields_[i - 1].mass();
}

int FieldTuple::field_index(int i) const {
	if (i < 1 || i > int(fields_.size()))
		ABORT("Field " << i << " does not exist");
	return fields_[i - 1].field_index();
}

const GiNaC::lst& FieldTuple::symbols() const {
	return symbols_;
}

void operator>>(const YAML::Node& node, FieldTuple& fn) {
	if (node.Type() != YAML::NodeType::Map || node.size() != 1)
		throw runtime_error("node is not a 1-element map "
				+ position_info(node));
	YAML::Iterator it = node.begin();
	it.first() >> fn.fr_id_;
	const YAML::Node& n = it.second();
	if (n.Type() != YAML::NodeType::Sequence || n.size() == 0)
		throw runtime_error("Node is not a non-empty sequence "
				+ position_info(n));
	fn.fields_.clear();
	fn.fields_.reserve(n.size());
	for (YAML::Iterator i = n.begin(); i != n.end(); ++i) {
		Field f;
		f.read(*i, fn.symbols_);
		fn.fields_.push_back(f);
		if (f.field_type() == FieldType::incoming || f.field_type()
				== FieldType::outgoing) {
			VERIFY(f.field_index() < 0);
		} else {
			VERIFY(f.field_index() > 0);
		}
	}
}

YAML::Emitter& operator<<(YAML::Emitter& os, const FieldTuple& fn) {
	os << YAML::BeginMap << YAML::Key << fn.fr_id_;
	os << YAML::Value << YAML::BeginSeq << YAML::Flow;
	vector<Field>::const_iterator i;
	for (i = fn.fields_.begin(); i != fn.fields_.end(); ++i)
		i->print(os);
	os << YAML::EndSeq << YAML::EndMap;
	return os;
}

void FieldTuple::read_stream(std::istream& stream) {
	stream >> fr_id_;
	string tmp;
	stream >> tmp;
	int size = fields_.size();
	fields_.clear();
	fields_.reserve(size);
	while (tmp == "field") {
		Field f;
		f.read_stream(stream, symbols_);
		fields_.push_back(f);
		if (f.field_type() == FieldType::incoming || f.field_type()
				== FieldType::outgoing) {
			VERIFY(f.field_index() < 0);
		} else {
			VERIFY(f.field_index() > 0);
		}
		stream >> tmp;
	}
	VERIFY(tmp == "end");
}

void FieldTuple::write_stream(std::ostream& s) const {
	s << fr_id_;
	for (unsigned i = 0; i < fields_.size();) {
		s << "  field ";
		fields_[i].write_stream(s);
		if (++i == fields_.size())
			s << " end";
	}
}

//
//
//


ExternalField::ExternalField() :
	from_vertex_(0), to_vertex_(0), polarization_orthogonal_vector_(0) {
	fields_ = vector<Field> (1, Field());
}

ExternalField::~ExternalField() {
}

ExternalField* ExternalField::clone() const {
	return new ExternalField(*this);
}

void ExternalField::read_stream(std::istream& stream) {
	stream >> from_vertex_ >> to_vertex_;
	FieldTuple::read_stream(stream);
	string pol;
	stream >> pol;
	polarization_orthogonal_vector_ = ex(pol, symbols_);
}

void ExternalField::write_stream(std::ostream& s) const {
	s << from_vertex_ << " " << to_vertex_ << " ";
	FieldTuple::write_stream(s);
	s << " " << polarization_orthogonal_vector_;
}

void ExternalField::read(const YAML::Node& n) {
	//verify_yaml_spec(n);
	if (n.Type() != YAML::NodeType::Sequence || (n.size() != 2 && n.size() != 3))
		throw runtime_error("node is not a 2 (or 3)-element sequence "
				+ position_info(n));
	if (n[0].Type() != YAML::NodeType::Sequence || n[0].size() != 2)
		throw runtime_error("node is not a 2-element sequence "
				+ position_info(n[0]));
	n[0][0] >> from_vertex_;
	n[0][1] >> to_vertex_;
	n[1] >> dynamic_cast<FieldTuple&> (*this);
	VERIFY(fields_.size() == 1);
	if (n.size() == 3)
		Reduze::read(n[2], polarization_orthogonal_vector_, symbols_);
}

void ExternalField::substitute_momenta(const GiNaC::exmap& m, unsigned options) {
	FieldTuple::substitute_momenta(m, options);
	polarization_orthogonal_vector_ = polarization_orthogonal_vector_.subs(m,
			options).expand();
}

void ExternalField::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << Flow << BeginSeq;
	os << BeginSeq;
	os << from_vertex_;
	os << to_vertex_;
	os << EndSeq;
	os << dynamic_cast<const FieldTuple&> (*this);
	if (!polarization_orthogonal_vector_.is_zero())
		os << polarization_orthogonal_vector_;
	os << EndSeq;
}

bool ExternalField::is_incoming() const {
	return (fields_[0].field_type() == FieldType::incoming) ? true : false;
}

bool ExternalField::begin_fermion_line(const FeynmanRules* fr) const {
	if (!has_anti_commuting_field())
		return false;
	ex rule = fr->rule(fr_id_);
	ASSERT(is_a<indexed>(rule));
	if (is_a<Spinor> (rule.op(0))) {
		Spinor spinor = ex_to<Spinor> (rule.op(0));
		if ((spinor.is_u() && is_incoming()) //
				|| (spinor.is_v() && !is_incoming()))
			return true;
		VERIFY((spinor.is_ubar() && !is_incoming()) || (spinor.is_vbar() && is_incoming()));
		return false;
	} else if (is_a<Polarization> (rule.op(0))) {
		Polarization pol = ex_to<Polarization> (rule.op(0));
		return !pol.is_conjugated();
	} else {
		ABORT("Unknown external particle.");
	}
}

bool ExternalField::end_fermion_line(const FeynmanRules* fr) const {
	if (!has_anti_commuting_field())
		return false;
	return !begin_fermion_line(fr);
}

//
//
//

PropagatorFields::PropagatorFields() :
	from_vertex_(0), to_vertex_(0) {
	fields_ = vector<Field> (2, Field());
}

PropagatorFields::~PropagatorFields() {
}

PropagatorFields* PropagatorFields::clone() const {
	return new PropagatorFields(*this);
}

void PropagatorFields::read(const YAML::Node& n) {
	//verify_yaml_spec(n);
	if (n.Type() != YAML::NodeType::Sequence || n.size() != 2)
		throw runtime_error("node is not a 2-element sequence "
				+ position_info(n));
	if (n[0].Type() != YAML::NodeType::Sequence || n[0].size() != 2)
		throw runtime_error("node is not a 2-element sequence "
				+ position_info(n[0]));
	n[0][0] >> from_vertex_;
	n[0][1] >> to_vertex_;
	n[1] >> dynamic_cast<FieldTuple&> (*this);
	VERIFY(fields_.size() == 2);
}

void PropagatorFields::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << Flow << BeginSeq;
	os << BeginSeq;
	os << from_vertex_;
	os << to_vertex_;
	os << EndSeq;
	os << dynamic_cast<const FieldTuple&> (*this);
	os << EndSeq;
}

void PropagatorFields::read_stream(std::istream& stream) {
	stream >> from_vertex_ >> to_vertex_;
	FieldTuple::read_stream(stream);
	VERIFY(fields_.size() == 2);
}

void PropagatorFields::write_stream(std::ostream& s) const {
	s << from_vertex_ << " " << to_vertex_ << " ";
	FieldTuple::write_stream(s);
}

//
//
//

VertexFields::VertexFields() {
}

VertexFields::~VertexFields() {
}

VertexFields* VertexFields::clone() const {
	return new VertexFields(*this);
}

void VertexFields::read(const YAML::Node& n) {
	//verify_yaml_spec(n);
	if (n.Type() != YAML::NodeType::Sequence || n.size() != 1)
		throw runtime_error("node is not a 1-element sequence "
				+ position_info(n));
	n[0] >> dynamic_cast<FieldTuple&> (*this);
	if (fields_.size() < 3)
		WARNING("Vertex of degree " << fields_.size() << " detected");
}

void VertexFields::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << Flow << BeginSeq;
	os << dynamic_cast<const FieldTuple&> (*this);
	os << EndSeq;
}

void VertexFields::read_stream(std::istream& stream) {
	FieldTuple::read_stream(stream);
	if (fields_.size() < 3)
		WARNING("Vertex of degree " << fields_.size() << " detected");
}

void VertexFields::write_stream(std::ostream& s) const {
	FieldTuple::write_stream(s);
}

//
//
//


Diagram::Diagram(const Kinematics* kin) :
	num_loops_(0), num_legs_in_(0), num_legs_out_(0), num_propagators_(0),
			num_vertices_(0), kinematics_(kin) {
	// get the external momenta and kinematical invariants from Kinematics
	if (kinematics_ != 0) {
		symbols_ = kinematics_->get_external_momenta_and_kinematics();
		symbols_without_loop_momenta_ = symbols_;
	}
}

Diagram::Diagram() :
	num_loops_(0), num_legs_in_(0), num_legs_out_(0), num_propagators_(0),
			num_vertices_(0), kinematics_(0) {
}

Diagram::~Diagram() {
	clear_field_tuples();
}

Diagram::Diagram(const Diagram& other) :
	process_name_(other.process_name_), //
			name_(other.name_), //
			num_loops_(other.num_loops_), //
			num_legs_in_(other.num_legs_in_), //
			num_legs_out_(other.num_legs_out_), //
			num_propagators_(other.num_propagators_), //
			num_vertices_(other.num_vertices_), //
			symmetry_factor_(other.symmetry_factor_), //
			sector_(other.sector_), //
			tuple_by_vertex_(other.tuple_by_vertex_), //
			kinematics_(other.kinematics_), //
			loop_momenta_(other.loop_momenta_), //
			symbols_without_loop_momenta_(other.symbols_without_loop_momenta_), //
			symbols_(other.symbols_) {
	// deep copy of the pointers
	vector<FieldTuple*>::const_iterator it;
	field_tuples_.reserve(other.field_tuples_.size());
	for (it = other.field_tuples_.begin(); it != other.field_tuples_.end(); ++it)
		field_tuples_.push_back((*it)->clone());
}

Diagram& Diagram::operator=(const Diagram& other) {
	if (this != &other) {
		// clear the field vertices
		clear_field_tuples();
		// deep copy of the pointers
		vector<FieldTuple*>::const_iterator c;
		for (c = other.field_tuples_.begin(); c != other.field_tuples_.end(); ++c)
			field_tuples_.push_back((*c)->clone());

		// copy the rest
		process_name_ = other.process_name_;
		name_ = other.name_;
		num_loops_ = other.num_loops_;
		num_legs_in_ = other.num_legs_in_;
		num_legs_out_ = other.num_legs_out_;
		num_propagators_ = other.num_propagators_;
		num_vertices_ = other.num_vertices_;
		symmetry_factor_ = other.symmetry_factor_;
		sector_ = other.sector_;
		tuple_by_vertex_ = other.tuple_by_vertex_;
		kinematics_ = other.kinematics_;
		loop_momenta_ = other.loop_momenta_;
		symbols_without_loop_momenta_ = other.symbols_without_loop_momenta_;
		symbols_ = other.symbols_;
	}
	return *this;
}

void Diagram::swap(Diagram& other) {
	process_name_.swap(other.process_name_);
	name_.swap(other.name_);
	std::swap(num_loops_, other.num_loops_);
	std::swap(num_legs_in_, other.num_legs_in_);
	std::swap(num_legs_out_, other.num_legs_out_);
	std::swap(num_propagators_, other.num_propagators_);
	std::swap(num_vertices_, other.num_vertices_);
	field_tuples_.swap(other.field_tuples_);
	std::swap(symmetry_factor_, other.symmetry_factor_);
	sector_.swap(other.sector_);
	tuple_by_vertex_.swap(other.tuple_by_vertex_);
	std::swap(kinematics_, other.kinematics_);
	std::swap(loop_momenta_, other.loop_momenta_);
	std::swap(symbols_without_loop_momenta_,
			other.symbols_without_loop_momenta_);
	std::swap(symbols_, other.symbols_);
}

void Diagram::clear_field_tuples() {
	vector<FieldTuple*>::iterator it;
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (*it != 0) {
			delete *it;
			*it = 0;
		}
	}
	field_tuples_.clear();
}

void Diagram::rename_negative_field_indices() {
	LOGX("rename negative field indices");

	// we assume (and don't test) correct distribution of the field indices as given by QGRAF:
	// external legs have negative field indices
	// internal edges don't introduce new negative indices

	map<int, int> old_to_new;
	int leg_index(0);
	for (unsigned i = 0; i < field_tuples_.size(); ++i)
		if (ft_is_a<ExternalField> (field_tuples_[i]))
			old_to_new[field_tuples_[i]->field_index(1)] = --leg_index;

	for (unsigned i = 0; i < field_tuples_.size(); ++i)
		field_tuples_[i]->replace_field_indices(old_to_new);
}

void Diagram::init() {
	if (kinematics_ != 0)
		substitute_momenta(kinematics_->rule_momentum_conservation());
	tuple_by_vertex_.clear();
	int nr(0);
	for (unsigned i = 0; i < field_tuples_.size(); ++i)
		if (ft_is_a<VertexFields> (field_tuples_[i]))
			tuple_by_vertex_[++nr] = i;
	rename_negative_field_indices();
}

int Diagram::insert(FieldTuple* field_tuple) {
	field_tuples_.push_back(field_tuple);
	return field_tuples_.size() - 1;
}

void Diagram::substitute_momenta(const GiNaC::exmap& substitutions,
		unsigned options) {
	for (unsigned i = 0; i < field_tuples_.size(); ++i)
		field_tuples_[i]->substitute_momenta(substitutions, options);
}

void Diagram::substitute_mass(const GiNaC::exmap& substitutions,
		unsigned options) {
	for (unsigned i = 0; i < field_tuples_.size(); ++i)
		field_tuples_[i]->substitute_mass(substitutions, options);
}

void Diagram::set_loop_momenta(const GiNaC::lst& loop_momenta) {
	loop_momenta_ = loop_momenta;
	symbols_ = symbols_without_loop_momenta_;
	lst::const_iterator it;
	for (it = loop_momenta.begin(); it != loop_momenta.end(); ++it)
		symbols_.append(*it);
}

void Diagram::set_name(const std::string& s) {
	name_ = s;
}

void Diagram::set_process_name(const std::string& s) {
	process_name_ = s;
}

// fermion lines

std::map<int, int> Diagram::find_following_fermion(const FeynmanRules* fr) const {
	// we assume the Diagram has no isolated vertices
	// Want to build the map of field node (from_node -> to_node) to be
	// able to follow the FieldTuples of a anti-commuting fields (spinors, ghosts).
	// e.g. incoming fermion -> vertex -> fermion propagator -> vertex -> outgoing fermion

	map<int, int> next;
	for (unsigned i = 0; i < field_tuples_.size(); ++i) {
		const FieldTuple* fn = field_tuples_[i];
		if (ft_is_a<VertexFields> (fn) || !fn->has_anti_commuting_field())
			continue;
		int this_node = i;
		if (ft_is_a<ExternalField> (fn)) {
			int from_vertex = ft_to<ExternalField> (fn)->from_vertex();
			int to_vertex = ft_to<ExternalField> (fn)->to_vertex();
			// determine particle flow (is equal to momentum flow for U and Ubar)
			// only take internal vertices (to_vertex > 0, from_vertex > 0)
			if (to_vertex > 0) { // incoming (anti-) spinor
				int to_node = tuple_of_vertex(to_vertex);
				if (ft_to<ExternalField> (fn)->begin_fermion_line(fr)) // incoming spinor U
					next[this_node] = to_node;
				else if (ft_to<ExternalField> (fn)->end_fermion_line(fr)) // incoming spinor vbar, opposite particle flow
					next[to_node] = this_node;
				else
					ERROR("Incoming Fermion ExternalField with field node " << this_node
							<< " is neither starting nor ending a fermion line.");
			} else if (from_vertex > 0) { // outgoing spinor
				int from_node = tuple_of_vertex(from_vertex);
				if (ft_to<ExternalField> (fn)->begin_fermion_line(fr)) // spinor v (opposite particle flow)
					next[this_node] = from_node;
				else if (ft_to<ExternalField> (fn)->end_fermion_line(fr)) // outgoing spinor Ubar
					next[from_node] = this_node;
				else
					ERROR("Outgoing Fermion ExternalField with field node " << this_node
							<< " is neither starting nor ending a fermion line.");
			} else {
				ERROR("External Fermion has neither from_vetex > 0 nor to_vertex > 0");
			}
		} else if (ft_is_a<PropagatorFields> (fn)) {
			int from_vertex = ft_to<PropagatorFields> (fn)->from_vertex();
			int to_vertex = ft_to<PropagatorFields> (fn)->to_vertex();
			// momentum flow is equal to particle flow
			int from_node = tuple_of_vertex(from_vertex);
			int to_node = tuple_of_vertex(to_vertex);
			next[this_node] = to_node;
			next[from_node] = this_node;
		} else {
			ABORT("Unknown Fieldnode");
		}
	}
	return next;
}

void Diagram::find_sorted_ids(list<list<int> >& open_fl,
		list<list<int> >& closed_fl, list<int>& rest, const FeynmanRules* fr) const {
	map<int, int> follower = find_following_fermion(fr);
	open_fl.clear();
	closed_fl.clear();
	rest.clear();
	set<int> visited;
	// open fermionic chains
	set<int> start = begin_fermion_lines(fr);
	for (set<int>::iterator it = start.begin(); it != start.end(); ++it) {
		list<int> chain;
		int next = *it;
		do {
			ASSERT(visited.find(next) == visited.end());
			chain.push_back(next);
			visited.insert(next);
			if (follower.find(next) != follower.end())
				next = follower.find(next)->second;
			else
				next = 0;
		} while (next != 0);
		open_fl.push_back(chain);
	}
	// closed fermion chains
	for (unsigned i = 0; i < field_tuples_.size(); ++i) {
		if (visited.find(i) != visited.end()
				|| !field_tuples_[i]->has_anti_commuting_field())
			continue;
		list<int> chain;
		int first = i;
		int next = first;
		do {
			ASSERT(visited.find(next) == visited.end());
			chain.push_back(next);
			visited.insert(next);
			ASSERT(follower.find(next) != follower.end());
			next = follower.find(next)->second;
		} while (next != first);
		closed_fl.push_back(chain);
	}
	// rest
	for (unsigned i = 0; i < field_tuples_.size(); ++i) {
		if (visited.find(i) != visited.end())
			continue;
		visited.insert(i);
		rest.push_back(i);
	}
	ASSERT(visited.size() == field_tuples_.size());
}

GiNaC::ex Diagram::feynman_rule_value(int id, const FeynmanRules* fr, map<
		string, GiNaC::symbol>& indices) const {
	ex rule = field_tuple(id)->feynman_rule_value(fr, indices);
	LOGX("Requested Feynman rule: " << rule);
	return rule;
}

Graph Diagram::get_Graph() const {
	Graph graph;
	string name = name_;
	if (!sector_.empty())
		name += "_" + sector().get_safe_string_for_filename();
	graph.set_name(name);
	graph.set_loop_momenta(loop_momenta_);
	vector<FieldTuple*>::const_iterator it;

	// set up all edges
	set<int> ext_edges;
	int nr1 = 0;
	int nr2 = num_legs_in_ + num_legs_out_;
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (ft_is_a<ExternalField> (*it)) {
			int from = ft_to<ExternalField> (*it)->from_vertex();
			int to = ft_to<ExternalField> (*it)->to_vertex();
			ext_edges.insert(graph.insert_edge(Edge(from, to, ++nr1)));
		} else if (ft_is_a<PropagatorFields> (*it)) {
			int from = ft_to<PropagatorFields> (*it)->from_vertex();
			int to = ft_to<PropagatorFields> (*it)->to_vertex();
			graph.insert_edge(Edge(from, to, ++nr2));
		}
	}

	// assign momenta so that they can be checked for momentum conservation
	nr1 = 0;
	nr2 = num_legs_in_ + num_legs_out_;
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (ft_is_a<ExternalField> (*it)) {
			++nr1;
			graph.set_momentum(nr1, (*it)->momentum(1));
			graph.set_squaredmass(nr1, pow((*it)->mass(1), 2));
		} else if (ft_is_a<PropagatorFields> (*it)) {
			++nr2;
			ex mom = (*it)->momentum(1);
			graph.set_momentum(nr2, mom);
			graph.set_squaredmass(nr2, pow((*it)->mass(1), 2));
		}
	}
	return graph;
}

Amplitude Diagram::get_amplitude(const FeynmanRules* fr) const {
	// res collects all commutative factors
	ex res(1);
	map<string, symbol> indices;
	Amplitude amplitude;
	list<list<int> > open_fl, closed_fl;
	list<int> rest;
	find_sorted_ids(open_fl, closed_fl, rest, fr);
	list<list<int> >::const_iterator lit;
	list<int>::const_reverse_iterator it;
	// open fermion lines
	for (lit = open_fl.begin(); lit != open_fl.end(); ++lit) {
		exvector line;
		ASSERT(lit->size() > 1);
		line.reserve(lit->size() - 1);
		ex spinorbar;
		bool first = true;
		for (it = lit->rbegin(); it != lit->rend(); ++it) {
			if (first) {
				spinorbar = feynman_rule_value(*it, fr, indices);
				first = false;
			} else {
				line.push_back(feynman_rule_value(*it, fr, indices));
			}
		}
		ex spinor = line.back();
		line.pop_back();
		amplitude.insert(OpenFermionChain(spinorbar, line, spinor));
	}
	// closed fermion lines
	for (lit = closed_fl.begin(); lit != closed_fl.end(); ++lit) {
		exvector line;
		line.reserve(lit->size());
		list<pair<int, string> > tag; // store ids for all Feynman rules used in the fermion loop
		for (it = lit->rbegin(); it != lit->rend(); ++it) {
			string s = field_tuple(*it)->fr_id();
			tag.push_back(make_pair(fr->num_by_id(s), s));
			line.push_back(feynman_rule_value(*it, fr, indices));
		}
		tag.sort();
		tag.unique();
		lst ntag;
		LOGN("  tagged fermion loop: (");
		for (list<pair<int,string> >::iterator i = tag.begin(); i != tag.end() ; ++i) {
			// the fr_id() is actually a nice string rep, but inconvenient in expr
			ntag.append(i->first);
			LOGN(" " << i->first << ":" << i->second);
		}
		LOG(" )");
		res *= Tag(ntag);
		amplitude.insert(ClosedFermionChain(line));
	}
	// rest of the propagators and vertices (assuming commutative objects)
	exvector extbosons;
	for (it = rest.rbegin(); it != rest.rend(); ++it) {
		ex value = feynman_rule_value(*it, fr, indices);
		ASSERT(value.return_type() == return_types::commutative);
		if (ft_is_a<ExternalField> (field_tuple(*it)))
			extbosons.push_back(value);
		else
			res *= value;
	}
	// symmetry factor
	res *= symmetry_factor_;

	if (!sector_.empty())
		amplitude.set_sector(sector());
	amplitude.set_loop_momenta(loop_momenta_);
	amplitude.set_external_bosons(extbosons);
	amplitude.set_amp(res);
	amplitude.set_local_indices(indices);
	amplitude.set_name(name());

	return amplitude;
}

void Diagram::set_sector(const Sector& s) {
	sector_.clear();
	sector_.push_front(s);
}

void Diagram::set_polarization_orthogonal_vector(int tuple,
		const GiNaC::ex& ortho) {
	VERIFY(ft_is_a<ExternalField>(field_tuple(tuple)));
	static_cast<ExternalField*> (field_tuples_[tuple])->set_polarization_orthogonal_vector(
			ortho);
}

int Diagram::tuple_of_vertex(int vertex) const {
	ASSERT(vertex > 0);
	map<int, int>::const_iterator it = tuple_by_vertex_.find(vertex);
	if (it == tuple_by_vertex_.end())
		ABORT("Vertex number " << vertex << " does not exist");
	return it->second;
}

const FieldTuple* Diagram::field_tuple(int node) const {
	if (node < 0 || node >= int(field_tuples_.size()))
		ABORT("FieldTuple with node " << node << " does not exist");
	return field_tuples_[node];
}

set<int> Diagram::begin_fermion_lines(const FeynmanRules* fr) const {
	set<int> res;
	for (unsigned i = 0; i < field_tuples_.size(); ++i) {
		if (ft_is_a<ExternalField> (field_tuples_[i]) && ft_to<ExternalField> (
				field_tuples_[i])->begin_fermion_line(fr))
			res.insert(i);
	}
	return res;
}

const Sector& Diagram::sector() const {
	if (sector_.empty())
		throw std::runtime_error("No sector available");
	return sector_.front();
}

//
//
//

std::ostream& operator<<(std::ostream& os, const Diagram& dia) {
	YAML::Emitter es;
	es << dia;
	os << es.c_str();
	return os;
}

void Diagram::read(const YAML::Node& n) {
	verify_yaml_spec(n);
	using namespace YAML;
	YAML::Iterator it;
	if (n.FindValue("process_name") != 0)
		n["process_name"] >> process_name_;
	n["name"] >> name_;

	n["num_loops"] >> num_loops_;
	if (num_loops() > 0 && n.FindValue("loop_momenta") != 0) {
		list<string> s;
		n["loop_momenta"] >> s;
		set_loop_momenta(create_symbols(s));
	}
	if (num_loops() > 0 && loop_momenta().nops() == 0)
		ABORT("No loop momenta defined");

	n["num_legs_in"] >> num_legs_in_;
	n["num_legs_out"] >> num_legs_out_;
	n["num_propagators"] >> num_propagators_;
	n["num_vertices"] >> num_vertices_;

	int num_legs = num_legs_in_ + num_legs_out_;
	if (num_legs > 0 && n.FindValue("external_legs") != 0) {
		const Node& ext_node = n["external_legs"];
		for (it = ext_node.begin(); it != ext_node.end(); ++it) {
			FieldTuple* field = new ExternalField();
			field->set_symbols(symbols_);
			*it >> dynamic_cast<ExternalField&> (*field);
			insert(field);
		}
	}
	if (num_propagators_ > 0 && n.FindValue("propagators") != 0) {
		const Node& edge_node = n["propagators"];
		for (Iterator it = edge_node.begin(); it != edge_node.end(); ++it) {
			FieldTuple* prop = new PropagatorFields();
			prop->set_symbols(symbols_);
			*it >> dynamic_cast<PropagatorFields&> (*prop);
			insert(prop);
		}
	}
	if (num_vertices_ > 0 && n.FindValue("vertices") != 0) {
		const Node& vert_node = n["vertices"];
		for (Iterator it = vert_node.begin(); it != vert_node.end(); ++it) {
			FieldTuple* vertex = new VertexFields();
			vertex->set_symbols(symbols_);
			*it >> dynamic_cast<VertexFields&> (*vertex);
			insert(vertex);
		}
	}
	if (n.FindValue("sector")) {
		try {
			Sector s(n["sector"]);
			sector_.push_front(s);
		} catch (exception& e) {
			sector_.clear();
		}
	}

	Reduze::read(n["symmetry_factor"], symmetry_factor_, lst());
	init();
}

void Diagram::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;
	os << Key << "process_name" << Value << process_name_;
	os << Key << "name" << Value << name_;

	vector<FieldTuple*>::const_iterator it;
	os << Key << "external_legs" << Value;
	os << BeginSeq;
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (ft_is_a<ExternalField> (*it))
			os << *(ft_to<ExternalField> (*it));
	}
	os << EndSeq;

	os << Key << "propagators" << Value;
	os << BeginSeq;
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (ft_is_a<PropagatorFields> (*it))
			os << *(ft_to<PropagatorFields> (*it));
	}
	os << EndSeq;

	os << Key << "vertices" << Value;
	os << BeginSeq;
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (ft_is_a<VertexFields> (*it))
			os << *(ft_to<VertexFields> (*it));
	}
	os << EndSeq;

	os << Key << "symmetry_factor" << Value << symmetry_factor_;
	os << Key << "num_legs_in" << Value << num_legs_in_;
	os << Key << "num_legs_out" << Value << num_legs_out_;
	os << Key << "num_loops" << Value << num_loops_;
	os << Key << "num_propagators" << Value << num_propagators_;
	os << Key << "num_vertices" << Value << num_vertices_;
	os << Key << "loop_momenta" << Value << Flow << loop_momenta_;
	os << Key << "sector" << Value;
	if (!sector_.empty())
		os << sector_.front();
	else
		os << sector_; // print empty list (for doc)

	os << EndMap;
}

void Diagram::read_stream(std::istream& in) {
	string token;
	num_legs_in_ = num_legs_out_ = num_propagators_ = num_vertices_ = 0;
	while (in >> token) {
		if (token == "name") {
			in >> name_;
		} else if (token == "process_name") {
			in >> process_name_;
		} else if (token == "loop_momenta") {
			list<string> s;
			string mom;
			in >> mom;
			while (mom != "end") {
				s.push_back(mom);
				in >> mom;
			}
			set_loop_momenta(create_symbols(s));
		} else if (token == "external_leg") {
			FieldTuple* f = new ExternalField();
			f->set_symbols(symbols_);
			f->read_stream(in);
			if (ft_to<ExternalField> (f)->is_incoming())
				++num_legs_in_;
			else
				++num_legs_out_;
			insert(f);
		} else if (token == "propagator") {
			FieldTuple* f = new PropagatorFields();
			f->set_symbols(symbols_);
			f->read_stream(in);
			++num_propagators_;
			insert(f);
		} else if (token == "vertex") {
			FieldTuple* f = new VertexFields();
			f->set_symbols(symbols_);
			f->read_stream(in);
			++num_vertices_;
			insert(f);
		} else if (token == "symmetry_factor") {
			string tmp;
			in >> tmp;
			symmetry_factor_ = ex(tmp, lst());
		} else if (token == "num_loops") {
			in >> num_loops_;
		} else if (token == "sector") {
			string fam;
			int id;
			in >> fam >> id;
			const IntegralFamily*f = Files::instance()->integralfamily(fam);
			sector_.push_back(Sector(f, id));
		} else if (token == "end") {
			init();
			return;
		} else {
			ABORT("Failed to read diagram: unknown tag " << token);
		}
	}
	ABORT("Never reached");
	return;
}

void Diagram::write_stream(std::ostream& s) const {
	vector<FieldTuple*>::const_iterator it;
	s << "diagram\n";
	s << "  name " << name_ << "\n";
	if (!process_name_.empty())
		s << "  process_name " << process_name_ << "\n";
	s << "  loop_momenta ";
	for (GiNaC::lst::const_iterator l = loop_momenta_.begin(); l
			!= loop_momenta_.end(); ++l)
		s << *l << " ";
	s << "end\n";
	for (it = field_tuples_.begin(); it != field_tuples_.end(); ++it) {
		if (ft_is_a<ExternalField> (*it))
			s << "  external_leg ";
		else if (ft_is_a<PropagatorFields> (*it))
			s << "  propagator ";
		else if (ft_is_a<VertexFields> (*it))
			s << "  vertex ";
		else
			ABORT("unknown field");
		(*it)->write_stream(s);
		s << "\n";
	}
	s << "  symmetry_factor " << symmetry_factor_ << "\n";
	s << "  num_loops " << num_loops_ << "\n";
	if (!sector_.empty())
		s << "  sector " << sector_.back().integralfamily()->name() << " "
				<< sector_.back().id() << "\n";
	s << "  end\n\n";
}

//std::ostream& operator<<(std::ostream& os, const Diagram& dia) {
//	dia.write_stream(os);
//	return os;
//}

//
//
//


void qgraf_globals::set_num_loops() {
	list<string>::const_iterator it;
	for (it = globals_.begin(); it != globals_.end(); ++it) {
		string key("loops=");
		string::size_type loc = it->find(key);
		if (loc == string::npos)
			continue;
		string::size_type end = it->find(";");
		if (end == string::npos)
			ABORT("No terminating ';' in line: " << *it);
		string s = it->substr(key.size(), end - key.size());
		num_loops_ = to_int(s);
		VERIFY(num_loops_ >= 0);
		return;
	}
	ABORT("Failed to find the number of loops in the qgraf output");
}

void qgraf_globals::set_loop_momenta() {
	if (num_loops_ < 0)
		set_num_loops();
	string loop_momentum_prefix;
	list<string>::const_iterator it;
	for (it = globals_.begin(); it != globals_.end(); ++it) {
		string key("loop_momentum=");
		string::size_type loc = it->find(key);
		if (loc == string::npos)
			continue;
		string::size_type end = it->find(";");
		if (end == string::npos)
			ABORT("No terminating ';' in line: " << *it);
		loop_momentum_prefix = it->substr(key.size(), end - key.size());
		VERIFY(!loop_momentum_prefix.empty());
		loop_momenta_ = create_symbols(loop_momentum_prefix, num_loops_);
		return;
	}
	ABORT("Failed to find the loop momentum prefix in the qgraf output");
}

//static std::set<std::string> extract_momenta(const std::string& s) {
//	set<string> res;
//	string::size_type b = s.find("[", 0), e;
//	while (b != string::npos) {
//		e = s.find("]", b); // b+2
//		if (e == string::npos)
//			ABORT("Failed to extract momenta from string " << s);
//		string mom = s.substr(b + 1, e - b - 1);
//		VERIFY(!mom.empty());
//		res.insert(mom);
//		b = s.find("[", e);
//	}
//	return res;
//}

//void qgraf_globals::set_ext_momenta() {
//	set<string> mom;
//	list<string>::const_iterator it;
//	set<string> moms;
//	for (it = globals_.begin(); it != globals_.end(); ++it) {
//		string::size_type loc = it->find("in=");
//		if (loc == string::npos)
//			continue;
//		set<string> m = extract_momenta(*it);
//		moms.insert(m.begin(), m.end());
//	}
//	for (it = globals_.begin(); it != globals_.end(); ++it) {
//		string::size_type loc = it->find("out=");
//		if (loc == string::npos)
//			continue;
//		set<string> m = extract_momenta(*it);
//		moms.insert(m.begin(), m.end());
//	}
//	list<string> lmoms(moms.begin(), moms.end());
//	ext_momenta_ = create_symbols(lmoms);
//}

void qgraf_globals::set_process_name() {
	list<string>::const_iterator it;
	for (it = globals_.begin(); it != globals_.end(); ++it) {
		string::size_type loc = it->find("output");
		if (loc == string::npos)
			continue;
		string prefix = it->substr(loc);
		prefix.replace(0, 6, "");
		for (string::iterator sit = prefix.begin(); sit != prefix.end(); ++sit) {
			if (*sit == ' ' || *sit == '=' || *sit == ';' || *sit == '\'')
				continue;
			process_name_ += *sit;
		}
		process_name_ = get_filename_without_suffix(process_name_, ".yaml");
		ASSERT(!process_name_.empty());
	}
}

void operator >>(const YAML::Node& node, qgraf_globals& qgraf) {
	YAML::Iterator it;
	for (it = node.begin(); it != node.end(); ++it) {
		string s;
		*it >> s;
		// remove spaces
		string::size_type pos = s.find(" ");
		while (pos != string::npos) {
			s.replace(pos, 1, "");
			pos = s.find(" ", pos);
		}
		qgraf.globals_.push_back(s);
	}
	qgraf.set_num_loops();
	qgraf.set_loop_momenta();
	//qgraf.set_ext_momenta();
	qgraf.set_process_name();
}

// functions

void read_diagrams(std::list<Diagram>& diagrams, const std::string& fn,
		const std::set<std::string>& names, const Kinematics* kin) {
#ifdef HAVE_NEW_REDUZE_STYLE_FILE
	read_diagrams_stream(diagrams, fn, names, kin);
#else
	read_diagrams_yaml(diagrams, fn, names, kin);
#endif /* HAVE_NEW_REDUZE_STYLE_FILE */
}

Diagram read_diagram(const std::string& fn, int pos) {
#ifdef HAVE_NEW_REDUZE_STYLE_FILE
	return read_diagram_stream(fn, pos);
#else
	return read_diagram_yaml(fn, pos);
#endif /* HAVE_NEW_REDUZE_STYLE_FILE */
}

void write_diagrams(const std::list<Diagram>& diagrams, const std::string& fn) {
#ifdef HAVE_NEW_REDUZE_STYLE_FILE
	write_diagrams_stream(diagrams, fn);
#else
	write_diagrams_yaml(diagrams, fn);
#endif /* HAVE_NEW_REDUZE_STYLE_FILE */
}

void read_diagrams_yaml(list<Diagram>& diagrams, const string& fn,
		const std::set<std::string>& names, const Kinematics* kina) {
	Timer timer;
	LOG("Reading diagrams from file: " << fn);
	ifstream fin(fn.c_str());
	if (!fin.good())
		ABORT("cannot open file: " << fn);
	YAML::Parser parser(fin);
	YAML::Node doc;
	GiNaC::lst loop_momenta/*, ext_momenta*/;
	string process_name;
	string key;
	bool first = true;
	const Kinematics* kin;
	if (kina == 0)
		kin = Files::instance()->kinematics();
	else
		kin = kina;
	while (parser.GetNextDocument(doc)) {
		ASSERT(doc.size() == 1);
		ASSERT(doc.Type() == YAML::NodeType::Map);
		doc.begin().first() >> key;
		if (first && key == "qgraf_globals") {
			qgraf_globals qgraf;
			doc["qgraf_globals"] >> qgraf;
			loop_momenta = qgraf.loop_momenta();
			//ext_momenta = qgraf.get_ext_momenta();
			process_name = qgraf.process_name();
			first = false;
			continue;
		} else if (key == "diagram") {
			const YAML::Node& node = doc.begin().second();
			Diagram dia(kin);
			dia.set_loop_momenta(loop_momenta);
			dia.set_process_name(process_name);
			//dia.append_external_momenta(ext_momenta);
			dia.read(node);
			if (names.empty() || names.find(dia.name()) != names.end()) {
				LOGX("Loaded diagram " << dia.name());
				diagrams.push_back(dia);
			} else {
				LOGX("Discarded diagram " << dia.name());
			}
		} else {
			ABORT("expected diagram or qgraf_globals but got: " << key);
		}
	}
	fin.close();
	LOG("Diagrams loaded. Time: " << timer.get_wall_time() << "s");
}

Diagram read_diagram_yaml(const std::string& fn, int pos) {
	using namespace YAML;
	Timer timer;
	LOG("Reading diagram number " << pos << " from file " << fn);
	const Kinematics* kin = Files::instance()->kinematics();
	ifstream in(fn.c_str());
	if (!in.good())
		ABORT("cannot open file: " << fn);
	// first read qgraf_globals if present
	string l;
	while (getline(in, l) && l != "---") {
	}
	GiNaC::lst loop_momenta/*, ext_momenta*/;
	string process_name;
	if (getline(in, l) && l.size() >= 14 && l.substr(0, 14) == "qgraf_globals:") {
		LOGX("  found qgraf_globals");
		in.seekg(0, ios::beg);
		YAML::Parser yp(in);
		YAML::Node d;
		if (!yp.GetNextDocument(d) || d.size() != 1 || d.Type() != NodeType::Map)
			ABORT("syntax error in file " << fn);
		qgraf_globals qgraf;
		d["qgraf_globals"] >> qgraf;
		loop_momenta = qgraf.loop_momenta();
		//ext_momenta = qgraf.get_ext_momenta();
		process_name = qgraf.process_name();
	}
	in.seekg(0, ios::beg);

	if (pos < 1)
		ABORT("in read_diagram: position " << pos << " is not positive");
	int num = 0;
	size_t file_pos = in.tellg();
	while (num != pos && in.good()) {
		// skip to new YAML doc
		while (file_pos = in.tellg(), getline(in, l) && l != "---") {
		}
		// does YAML doc contain a diagram ?
		if (getline(in, l) && l.size() >= 8 && l.substr(0, 8) == "diagram:")
			++num; // found diagram at position = new num value
	}
	if (num != pos)
		ABORT("cannot find diagram number " << pos << " in " << fn);
	in.seekg(file_pos);
	YAML::Parser yp(in);
	YAML::Node d;
	if (!yp.GetNextDocument(d) || d.size() != 1 || d.Type() != NodeType::Map)
		ABORT("syntax error in file " << fn);
	Diagram dia(kin);
	dia.set_loop_momenta(loop_momenta);
	dia.set_process_name(process_name);
	//dia.append_external_momenta(ext_momenta);
	d["diagram"] >> dia;
	return dia;
}

void write_diagrams_yaml(const std::list<Diagram>& dias, const std::string& fn) {
	using namespace YAML;
	string tmpfn = fn + ".tmp";
	ofstream out(tmpfn.c_str());
	if (!out.good())
		ABORT("cannot open file: " << fn);
	for (list<Diagram>::const_iterator d = dias.begin(); d != dias.end(); ++d) {
		YAML::Emitter ye;
	    ye << YAML::BeginDoc;
		ye << BeginMap << Key << "diagram" << Value << *d << EndMap;
		out << ye.c_str() << "\n";
	}
	out.close();
	LOG("Finalizing output file " << fn);
	rename(tmpfn, fn);
}

// new functions

void read_diagrams_stream(std::list<Diagram>& diagrams, const std::string& fn,
		const std::set<std::string>& names, const Kinematics* kina) {
	Timer timer;
	LOG("Reading diagrams from file: " << fn);
	ifstream fin(fn.c_str());
	if (!fin.good())
		ABORT("cannot open file: " << fn);
	GiNaC::lst loop_momenta;
	string process_name;
	bool first = true;
	const Kinematics* kin;
	if (kina == 0)
		kin = Files::instance()->kinematics();
	else
		kin = kina;
	string token;
	while (fin >> token) {
		if (first && token == "qgraf_globals") {
			string qglob, line;
			while (getline(fin, line)) {
				if (line == "end")
					break;
				qglob += line + "\n";
			}
			istringstream qglobstream(qglob);
			YAML::Parser parser(qglobstream);
			YAML::Node doc;
			parser.GetNextDocument(doc);
			qgraf_globals qgraf;
			doc["qgraf_globals"] >> qgraf;
			loop_momenta = qgraf.loop_momenta();
			//ext_momenta = qgraf.get_ext_momenta();
			process_name = qgraf.process_name();
			first = false;
			continue;
		} else if (token == "diagram") {
			first = false;
			Diagram dia(kin);
			dia.set_loop_momenta(loop_momenta);
			dia.set_process_name(process_name);
			//dia.append_external_momenta(ext_momenta);
			dia.read_stream(fin);
			if (names.empty() || names.find(dia.name()) != names.end()) {
				LOGX("Loaded diagram " << dia.name());
				diagrams.push_back(dia);
			} else {
				LOGX("Discarded diagram " << dia.name());
			}
		} else if (token == "#") {
			string rest;
			getline(fin, rest);
		} else {
			ABORT("expected diagram or qgraf_globals but got: " << token);
		}
	}
	fin.close();
	LOG("Diagrams loaded. Time: " << timer.get_wall_time() << "s");
}

void write_diagrams_stream(const std::list<Diagram>& dias,
		const std::string& fn) {
	string tmpfn = fn + ".tmp";
	ofstream out(tmpfn.c_str());
	if (!out.good())
		ABORT("cannot open file: " << fn);
	for (list<Diagram>::const_iterator d = dias.begin(); d != dias.end(); ++d)
		d->write_stream(out);
	out.close();
	LOG("Finalizing output file " << fn);
	rename(tmpfn, fn);
}

// global functions

// \todo test me!
Diagram read_diagram_stream(const std::string& fn, int pos) {
	LOG("Reading diagram number " << pos << " from file " << fn);
	const Kinematics* kin = Files::instance()->kinematics();
	ifstream in(fn.c_str());
	if (!in.good())
		ABORT("cannot open file: " << fn);
	if (pos < 1)
		ABORT("in read_diagram: position " << pos << " is not positive");
	bool first = true;
	Diagram dia(kin);
	string token;
	int num = 0;
	while (in >> token) {
		if (first && token == "qgraf_globals") {
			string qglob, line;
			while (getline(in, line)) {
				if (line == "end")
					break;
				qglob += line + "\n";
			}
			istringstream qglobstream(qglob);
			YAML::Parser parser(qglobstream);
			YAML::Node doc;
			parser.GetNextDocument(doc);
			qgraf_globals qgraf;
			doc["qgraf_globals"] >> qgraf;
			dia.set_loop_momenta(qgraf.loop_momenta());
			dia.set_process_name(qgraf.process_name());
			first = false;
			continue;
		} else if (token == "diagram") {
			first = false;
			if (++num != pos)
				continue;
			dia.read_stream(in);
			in.close();
			return dia;
		}
	}
	in.close();
	ABORT("cannot find diagram number " << pos << " in " << fn);
	return dia;
}

size_t find_graphs_of_diagrams( //
		const std::string& qgraf_filename, //
		const std::set<std::string>& names, //
		const std::string& kinematics, //
		bool amputate_external_legs, //
		bool join_external_nodes, //
		bool minimize_graphs_by_twists, //
		std::list<std::pair<Graph, Diagram> >& graph_dia_pairs) {

	LOG("Read diagrams from file '" << qgraf_filename << "'");
	const Kinematics* kin = Files::instance()->kinematics(kinematics);
	list<Diagram> alldias;
	read_diagrams(alldias, qgraf_filename, names, kin);
	const size_t num_dias = alldias.size();
	ProgressBar pbar(2, "preparing graphs:", alldias.size());
	pbar.start();
	while (!alldias.empty()) {
		Diagram& dia = *alldias.begin();
		Graph graph = dia.get_Graph();
		if (amputate_external_legs)
			graph.remove_external_legs();
		if (join_external_nodes)
			graph.join_external_nodes();
		graph.contract_multiple_internal_propagators();
		graph.contract_bridges();
		graph.disconnect_vacuum_components();
		if (minimize_graphs_by_twists)
			graph.minimize_by_twists();
		VERIFY(graph.get_propagators_with_loop_momenta().size()
				== graph.num_internal_edges()); // \todo remove this test
		graph_dia_pairs.push_back(make_pair(graph, dia));
		alldias.erase(alldias.begin());
		pbar.print();
	}
	pbar.end();
	LOG("Loaded " << num_dias << " diagrams.");
	return num_dias;
}

size_t find_graphs_of_diagrams(
		const std::string& qgraf_filename, //
		const std::set<std::string>& names, //
		const std::string& kinematics, //
		bool amputate_external_legs, //
		bool join_external_nodes, //
		bool minimize_graphs_by_twists, //
		std::map<int, std::list<std::pair<Graph, Diagram> > >& graph_dia_pairs_by_loop) {

	std::list<std::pair<Graph, Diagram> > graph_dia_pairs;
	size_t num_dias = find_graphs_of_diagrams(qgraf_filename, names,
			kinematics, amputate_external_legs, join_external_nodes,
			minimize_graphs_by_twists, graph_dia_pairs);
	while (!graph_dia_pairs.empty()) {
		int loop = graph_dia_pairs.begin()->first.loop_momenta().nops();
		graph_dia_pairs_by_loop[loop].push_back(*graph_dia_pairs.begin());
		graph_dia_pairs.erase(graph_dia_pairs.begin());
	}
	return num_dias;
}

size_t find_graphs_of_diagrams(
		const std::string& qgraf_filename, //
		const std::set<std::string>& names, //
		const std::string& kinematics, //
		bool amputate_external_legs, //
		bool join_external_nodes, //
		bool minimize_graphs_by_twists, //
		std::map<int, std::map<int, std::list<std::pair<Graph, Diagram> > > > & graph_dia_pairs_by_t_by_loop,
		bool rename_diagrams) {

	std::list<std::pair<Graph, Diagram> > graph_dia_pairs;
	size_t num_dias = find_graphs_of_diagrams(qgraf_filename, names,
			kinematics, amputate_external_legs, join_external_nodes,
			minimize_graphs_by_twists, graph_dia_pairs);
	int count = 0;
	while (!graph_dia_pairs.empty()) {
		int loop = graph_dia_pairs.begin()->first.loop_momenta().nops();
		int t = graph_dia_pairs.begin()->first.num_internal_edges();
		if (rename_diagrams) {
			ostringstream ss;
			ss << setfill('0') << setw(number_of_digits(num_dias)) << ++count;
			graph_dia_pairs.begin()->second.set_name(ss.str());
		}
		graph_dia_pairs_by_t_by_loop[loop][t].push_back(
				*graph_dia_pairs.begin());
		graph_dia_pairs.erase(graph_dia_pairs.begin());
	}
	return num_dias;
}

} // namespace


