/*  equationlist.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include <sstream>
#include <fstream>

#include <ginac/ginac.h>

#include "integralfamily.h"
#include "int.h"
#include "functions.h"
#include "equationlist.h"
#include "kinematics.h"
#include "files.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// EqBasicList

template<class TE>
typename EqBasicList<TE>::iterator EqBasicList<TE>::it_to_shortest(
		const integral_type & max) {
	iterator it_simple = l.end();
	iterator it = l.begin();
	// get the first equation eq with eq.max_INT() == max
	while (it != l.end()) {
		if (it->empty()) {
			++it;
		} else if (it->max_INT() == max) {
			it_simple = it++;
			break; // break if first equation found with max_INT() == max
		} else {
			++it;
		}
	}
	// find a shorter equations with it->max_INT() == max
	while (it != l.end()) {
		if (it->empty()) {
			++it;
		} else if (it->max_INT() == max//
				&& (it->size() < it_simple->size() || //
						(it->size() == it_simple->size() && *it_simple < *it))) {
			it_simple = it++;
		} else {
			++it;
		}
	}
	return it_simple;
}

template<class TE>
typename EqBasicList<TE>::iterator EqBasicList<TE>::it_to_most_reduced(
		const integral_type & max) {
	iterator it_to_simple = l.end();
	iterator it = l.begin();
	// get the first equation eq with eq.max_INT() == max
	while (it != l.end()) {
		if (it->empty()) {
			++it;
		} else if (it->max_INT() == max) {
			it_to_simple = it++;
			break; // break if first equation found with max_INT() == max
		} else {
			++it;
		}
	}
	// find a more reduced equation with it->max_INT() == max
	while (it != l.end()) {
		if (it->empty()) {
			++it;
		} else if (it->max_INT() == max && *it < *it_to_simple) {
			it_to_simple = it++;
		} else {
			++it;
		}
	}
	return it_to_simple;
}

template<class TE>
int EqBasicList<TE>::remove_empty_equations() {
	int lauf = 0;
	iterator it = l.begin();
	while (it != l.end()) {
		if (it->empty()) {
			it = l.erase(it);
			++lauf;
		} else {
			++it;
		}
	}
	return lauf;
}

template<class TE>
void EqBasicList<TE>::remove_same_leading_INT_multiples() {
	remove_empty_equations();
	sort();
	iterator it = l.begin();
	while (it != l.end()) {
		iterator last = it;
		++it;
		if (it != l.end() && last->max_INT() == it->max_INT())
			it = l.erase(it);
	}
}

// EqAlgebraicList

template<class TE>
void EqAlgebraicList<TE>::apply(const GiNaC::exmap& m, unsigned options) {
	for (iterator it = this->l.begin(); it != this->l.end(); ++it)
		it->apply(m, options);
}

template<class TE>
void EqAlgebraicList<TE>::replace(const eq_type & toreplace, bool normalize) {
	for (iterator it = this->l.begin(); it != this->l.end(); ++it)
		it->replace(toreplace, normalize);
}

template<class TE>
void EqAlgebraicList<TE>::replace(const EqAlgebraicList<TE> & toreplace,
		bool normalize) {
	iterator it1 = this->l.begin();
	const_iterator it2;
	while (it1 != this->l.end()) {
		if (it1->empty()) {
			it1 = this->l.erase(it1);
		} else {
			for (it2 = toreplace.l.begin(); it2 != toreplace.l.end(); ++it2)
				it1->replace(*it2, normalize);
			++it1;
		}
	}
}

template<class TE>
bool EqAlgebraicList<TE>::gives_reduction_zero(const integral_type & in) const {
	for (const_iterator t = this->l.begin(); t != this->l.end(); ++t)
		if (t->size() == 1 && t->max_INT() == in)
			return true;
	return false;
}

template<class TE>
bool EqAlgebraicList<TE>::gives_reduction_zero(const set<integral_type> & in) const {
	typename set<integral_type>::const_iterator sit = in.begin();
	for (sit = in.begin(); sit != in.end(); ++sit)
		if (!gives_reduction_zero(*sit))
			return false;
	return true;
}

template<class TE>
void EqAlgebraicList<TE>::forward_eliminate() {
	LOGXX("  triangularize,     equations : " << this->l.size());
	EqAlgebraicList<TE> solved;
	set<integral_type> myset;
	this->find_INT(myset);
	LOGXX("                     integrals : " << myset.size());
	typename set<integral_type>::const_reverse_iterator sit;
	for (sit = myset.rbegin(); sit != myset.rend(); ++sit) {
		iterator it = this->it_to_most_reduced(*sit);
		if (it == this->l.end())
			continue;
		solved.l.splice(solved.l.end(), this->l, it); // move element *it from this->l to solved.l
		this->replace(solved.l.back(), true); // insert equation into others
	}
	this->l.swap(solved.l);
	this->l.reverse();
	// equations in this->l are in ascending order
}

template<class TE>
void EqAlgebraicList<TE>::forward_eliminate_leading(EqAlgebraicList<TE>& lower) {
	LOGXX("  forward eliminate leading, equations : " << this->l.size());
	EqAlgebraicList<TE> solved;
	set<integral_type> maxints;
	this->find_max_INT(maxints);
	LOGXX("                             integrals : " << maxints.size());
	typename set<integral_type>::const_reverse_iterator sit;
	for (sit = maxints.rbegin(); sit != maxints.rend(); ++sit) {
		iterator it = this->it_to_most_reduced(*sit);
		if (it == this->l.end())
			continue;
		solved.l.splice(solved.l.end(), this->l, it); // move *it from this->l to solved.l
		this->replace(solved.l.back(), false); // insert equation into others
	}
	this->solve();
	this->l.sort();
	lower.l.splice(lower.l.end(), this->l);
	lower.solve();
	solved.l.reverse();
	this->l.swap(solved.l);
	// this->l and lower.l are sorted in ascending order
}

template<class TE>
void EqAlgebraicList<TE>::back_substitute() {
#ifdef DEBUG
	LOGXX("  back substitute,   equations : " << this->l.size());
	set<integral_type> allints;
	this->find_INT(allints);
	LOGXX("                     integrals : " << allints.size());
#endif
	for (iterator s = this->l.begin(); s != this->l.end(); ++s) {
		iterator e = s;
		for (++e; e != this->l.end();) {
			bool normalize = true;
			e->replace(*s, normalize);
			if (e->empty())
				this->l.erase(e++);
			else
				++e;
		}
	}
}

template<class TE>
void EqAlgebraicList<TE>::reduze() {
	forward_eliminate();
	back_substitute();
}

template<class TE>
void EqAlgebraicList<TE>::solve() {
	iterator it = this->l.begin();
	while (it != this->l.end()) {
		it->solve();
		if (it->empty())
			this->l.erase(it++);
		else
			++it;
	}
}

template<class TE>
bool EqAlgebraicList<TE>::has_symbol(const GiNaC::symbol & symb) const {
	const_iterator it = this->begin();
	while (it != this->end()) {
		if (it->has_symbol(symb))
			return true;
		else
			++it;
	}
	return false;
}

template<class TE>
int EqAlgebraicList<TE>::number_of_symbols() const {
	const lst& syms = //
					Files::instance()->kinematics()->kinematic_invariants_and_dimension();
	return number_of_symbols(syms);
}

template<class TE>
int EqAlgebraicList<TE>::number_of_symbols(const GiNaC::lst & alist) const {
	int res = 0;
	for (unsigned int i = 0; i < alist.nops(); ++i) {
		if (!is_a<symbol> (alist.op(i)))
			ABORT(alist.op(i) << " is not a symbol!");
		symbol symb = ex_to<symbol> (alist.op(i));
		if (has_symbol(symb))
			++res;
	}
	return res;
}

// EquationList

EquationList::EquationList() {
}


EquationList::EquationList(const EquationLightList& other) {
	const lst& symbols = //
					Files::instance()->kinematics()->kinematic_invariants_and_dimension();
	EquationList eqlist(other, symbols);
	eqlist.swap(*this);
}

EquationList::EquationList(const EquationLightList& other, const GiNaC::lst& s) {
	EquationLightList::const_iterator it;
	for (it = other.begin(); it != other.end(); ++it)
		push_back(Equation(*it, s));
}

// IdentityList

IdentityList::IdentityList() {
}

double IdentityList::average_length() {
	iterator it = l.begin();
	double sum = 0.0;
	if (l.size() == 0)
		return 0.0;
	double length = l.size();
	while (it != l.end())
		sum += (it++)->size();
	sum /= length;
	return sum;
}

double IdentityList::average_coeff_length() {
	iterator it = l.begin();
	double sum = 0.0;
	double length = size();
	while (it != l.end())
		sum += (it++)->average_coeff_length() / length;
	return sum;
}

void IdentityList::reconstruct_symbol_replaced_by_one() {
	for (iterator e = l.begin(); e != l.end(); ++e)
		e->reconstruct_symbol_replaced_by_one();
}

void IdentityList::remove_zero_integral() {
	for (list<Identity>::iterator i = l.begin(); i != l.end();) {
		i->remove_zero_integral();
		if (i->empty())
			l.erase(i++);
		else
			++i;
	}
}

void IdentityList::toggle_metric_convention() {
	list<Identity>::iterator it = l.begin();
	for (it = l.begin(); it != l.end(); ++it)
		it->toggle_metric_convention();
}

// explicit instantiations
template class EqBasicList<EquationXLight> ;
template class EqBasicList<EquationHLight> ;
template class EqBasicList<EquationLight> ;
template class EqBasicList<Identity> ;
template class EqBasicList<IdentityGeneric> ;
template class EqBasicList<LinearCombination> ;
template class EqBasicList<LinearCombinationGeneric> ;
template class EqBasicList<Equation> ;
template class EqAlgebraicList<Identity> ;
template class EqAlgebraicList<IdentityGeneric> ;
template class EqAlgebraicList<Equation> ;

} // namespace Reduze
