/*  ginacutils.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "ginacutils.h"
#include "functions.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// basic helpers

bool nonneg_integer(const ex & e, int & i) {
	numeric n;
	if (is_a<numeric> (e)) {
		n = ex_to<numeric> (e);
		if (n.is_nonneg_integer()) {
			i = n.to_int();
			return true;
		} else {
			return false;
		}
	} else {
		return false;
	}
}

bool nonneg_integer(const ex & e, long & i) {
	numeric n;
	if (is_a<numeric> (e)) {
		n = ex_to<numeric> (e);
		if (n.is_nonneg_integer()) {
			i = n.to_long();
			return true;
		} else {
			return false;
		}
	} else {
		return false;
	}
}

int to_int(const ex & e) {
	if (is_a<numeric> (e)) {
		numeric n = ex_to<numeric> (e);
		if (n.is_integer()) {
			return n.to_int();
		} else {
			cerr << "error in to_int(ex): " << n << " is not an integer"
					<< endl;
			abort();
		}
	} else {
		cerr << "error in to_int(ex): " << e << " is not a number" << endl;
		abort();
	}
	return 0; // never executed
}

int to_int(const string & str) {
	ex e = ex(str, lst());
	return to_int(e);
}

int to_nonneg_int(const string & str) {
	ex e = ex(str, lst());
	int i;
	bool bu = nonneg_integer(e, i);
	if (!bu) {
		cerr << str << " : is not a non negative integer" << endl;
		abort();
	}
	return i;
}

GiNaC::lst add_lst(const GiNaC::lst& l1, const GiNaC::lst& l2) {
	lst l = l1;
	for (lst::const_iterator it = l2.begin(); it != l2.end(); ++it)
		l.append(*it);
	return l;
}

GiNaC::lst set2lst(const GiNaC::exset& s) {
	lst l;
	for (exset::const_iterator it = s.begin(); it != s.end(); ++it)
		l.append(*it);
	return l;
}

GiNaC::exset lst2set(const GiNaC::lst& l) {
	exset s;
	for (lst::const_iterator it = l.begin(); it != l.end(); ++it)
		s.insert(*it);
	return s;
}

bool freeof(const GiNaC::ex& e, const GiNaC::lst& l) {
	for (lst::const_iterator it = l.begin(); it != l.end(); ++it) {
		if (e.has(*it))
			return false;
	}
	return true;
}

int compare(const exmap& a, const exmap& b) {
	if (a.size() != b.size())
		return a.size() - b.size();
	exmap::const_iterator ia = a.begin();
	exmap::const_iterator ib = b.begin();
	for (; ia != a.end(); ++ia, ++ib) {
		int cmp;
		if ((cmp = ia->first.compare(ib->first)) != 0)
			return cmp;
		if ((cmp = ia->second.compare(ib->second)) != 0)
			return cmp;
	}
	return 0;
}

bool exmap_is_less::operator()(const exmap& a, const exmap& b) const {
	return compare(a, b) < 0;
}

GiNaC::ex toggle_non_numeric_varidx(const GiNaC::ex& e) {
	exset found;
	exmap m;
	gather_objects<varidx> (e, found);
	for (exset::const_iterator f = found.begin(); f != found.end(); ++f) {
		if (!is_a<numeric> (ex_to<varidx> (*f).get_value()))
			m[*f] = ex_to<varidx> (*f).toggle_variance();
	}
	return e.subs(m);
}

bool has_symbol_name(const GiNaC::lst& l, const std::string& str) {
	for (lst::const_iterator it = l.begin(); it != l.end(); ++it) {
		if (is_a<symbol> (*it)) {
			symbol s = ex_to<symbol> (*it);
			if (s.get_name() == str)
				return true;
		}
	}
	return false;
}

void provide_alternative_symbols(const GiNaC::lst& symbols,
		GiNaC::lst& alt_symbols, GiNaC::exmap& sym2alt, GiNaC::exmap& alt2sym) {
	using namespace GiNaC;
	for (lst::const_iterator s = symbols.begin(); s != symbols.end(); ++s) {
		if (!is_exactly_a<symbol> (*s))
			ABORT("This is no symbol: " << *s);
		symbol alt(ex_to<symbol> (*s).get_name() + "alt");
		sym2alt[*s] = alt;
		alt2sym[alt] = *s;
		alt_symbols.append(alt);
	}
}

GiNaC::exmap substitute_rhs(const GiNaC::exmap& a, const GiNaC::exmap& b,
		unsigned options) {
	GiNaC::exmap result;
	for (GiNaC::exmap::const_iterator s = a.begin(); s != a.end(); ++s)
		result[s->first] = s->second.subs(b, options);
	return result;
}

GiNaC::lst substitute_lhs(const GiNaC::exmap& a, const GiNaC::exmap& b,
		unsigned options) {
	GiNaC::lst result;
	for (GiNaC::exmap::const_iterator s = a.begin(); s != a.end(); ++s)
		result.append(s->first.subs(b, options) == s->second);
	return result;
}

GiNaC::exmap substitute_lhs_map(const GiNaC::exmap& a, const GiNaC::exmap& b,
		unsigned options) {
	GiNaC::exmap result;
	for (GiNaC::exmap::const_iterator s = a.begin(); s != a.end(); ++s) {
		ex lhs = s->first.subs(b, options);
		ASSERT(is_a<symbol>(lhs));
		result[lhs] = s->second;
	}
	return result;
}

bool lhs_freeof(const GiNaC::exmap& m, const GiNaC::ex& e) {
	for (GiNaC::exmap::const_iterator it = m.begin(); it != m.end(); ++it) {
		if (it->first.has(e))
			return false;
	}
	return true;
}

bool rhs_freeof(const GiNaC::exmap& m, const GiNaC::ex& e) {
	for (GiNaC::exmap::const_iterator it = m.begin(); it != m.end(); ++it) {
		if (it->second.has(e))
			return false;
	}
	return true;
}

GiNaC::exmap make_translation(const GiNaC::lst& a, const GiNaC::lst& b) {
	ASSERT(a.nops() == b.nops());
	GiNaC::exmap trans;
	GiNaC::lst::const_iterator i, j;
	for (i = a.begin(), j = b.begin(); i != a.end() && j != b.end(); ++i, ++j)
		trans[*i] = *j;
	return trans;
}

bool can_be_parsed_with(const GiNaC::ex& e, const GiNaC::lst& symbols) {
	ostringstream os;
	os << e;
	string s = os.str();
	try {
		ex dummy(s, symbols);
	} catch (exception& excep) {
		return false;
	}
	return true;
}

// create symbols


void assert_unique_symbols_and_valid_names(const GiNaC::lst& symb) {
	lst l = symb;
	set<string> names;
	for (lst::const_iterator s = symb.begin(); s != symb.end(); ++s) {
		ostringstream sname;
		sname << *s;
		if (!is_a<symbol> (*s))
			ABORT("is not a symbol: " << sname.str());
		assert_valid_symbol_name(sname.str());
		pair<set<string>::iterator, bool> info = names.insert(sname.str());
		if (!info.second)
			throw runtime_error("symbol name '" + sname.str()
					+ "' appears more than once.");
	}
}

void assert_valid_symbol_name(const string& name) {
	static const char* allowed =
			"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
	if (name.find_first_not_of(allowed, 0) != string::npos || name.empty())
		throw runtime_error("Invalid symbol name: '" + name
				+ "'. Allowed characters are: " + string(allowed));
	if (name.empty() || isalpha(name[0]) == 0)
		throw runtime_error(
				"Invalid symbol name: '" + name
						+ "'. First character must be alphabetic (even a lower case letter if Fermat is used)");
}

void assert_valid_prefix_name(const std::string& name) {
	static const char* allowed = "abcdefghijklmnopqrstuvwxyz";
	if (name.find_first_not_of(allowed, 0) != string::npos || name.empty())
		throw runtime_error("Invalid prefix symbol name: '" + name
				+ "'. Allowed characters are: " + string(allowed));
}

GiNaC::symbol create_symbol(const std::string& name) {
	assert_valid_symbol_name(name);
	return symbol(name);
}
GiNaC::realsymbol create_real_symbol(const std::string& name) {
	assert_valid_symbol_name(name);
	return realsymbol(name);
}

GiNaC::lst create_symbols(const std::string& prefix, int n) {
	ASSERT(!prefix.empty() && n >= 0);
	lst l;
	for (int i = 1; i <= n; ++i)
		l.append(create_symbol(prefix + to_string(i)));
	return l;
}

GiNaC::lst create_symbols(const std::list<std::string>& s) {
	lst l;
	for (list<string>::const_iterator it = s.begin(); it != s.end(); ++it)
		l.append(create_symbol(*it));
	return l;
}

GiNaC::lst create_real_symbols(const std::list<std::string>& s) {
	lst l;
	for (list<string>::const_iterator it = s.begin(); it != s.end(); ++it)
		l.append(create_real_symbol(*it));
	return l;
}

// substitutions


GiNaC::lst subs_in_lst(const GiNaC::lst& v, const GiNaC::exmap& m,
		unsigned options) {
	lst result;
	for (lst::const_iterator it = v.begin(); it != v.end(); ++it)
		result.append(it->subs(m, options));
	return result;
}

exvector subs_in_exvector(const exvector& v, const ex& m, unsigned options) {
	exvector result;
	result.reserve(v.size());
	for (exvector::const_iterator it = v.begin(); it != v.end(); ++it)
		result.push_back(it->subs(m, options));
	return result;
}

exvector subs_in_exvector(const exvector& v, const exmap& m, unsigned options) {
	exvector result;
	result.reserve(v.size());
	for (exvector::const_iterator it = v.begin(); it != v.end(); ++it)
		result.push_back(it->subs(m, options));
	return result;
}

exvector conjugate_in_exvector(const exvector& v) {
	exvector result;
	result.reserve(v.size());
	for (exvector::const_reverse_iterator it = v.rbegin(); it != v.rend(); ++it)
		result.push_back(it->conjugate());
	return result;
}

// algebraic manipulations

GiNaC::ex factor_rational(const GiNaC::ex& e) {
	ex nd = e.numer_denom();
	ASSERT(is_a<lst> (nd) && nd.nops() == 2);
	ex n = nd[0];
	ex d = nd[1];
	// factor doesn't like I
	ex nfac = (n.expand().has(I) ? -I * factor(expand(I * n)) : factor(n));
	ex dfac = (d.expand().has(I) ? -I * factor(expand(I * d)) : factor(d));
	return nfac / dfac;
}

GiNaC::lst extract_numerical_factor(const GiNaC::ex& e) {
	ex num(1), rest(1);
	if (is_a<numeric> (e)) {
		num = e;
	} else if (is_a<mul> (e)) {
		for (size_t i = 0; i < e.nops(); ++i)
			if (is_a<numeric> (e.op(i)))
				num *= e.op(i);
			else
				rest *= e.op(i);
	} else {
		rest = e;
	}
#ifdef NEW_GINAC
    lst result({num, rest});
#else
    lst result(num, rest);
#endif
	return result;
}

// this is just a helper class to get access to protected GiNaC::matrix methods
class HelperMatrix: public GiNaC::matrix {
GINAC_DECLARE_REGISTERED_CLASS(HelperMatrix, GiNaC::matrix)
public:
	HelperMatrix(size_t m, size_t n) :
		matrix(m, n) {
	}
	// this is a slight modification of GiNaC's matrix::solve implementation
	// to allow for partially solving over-determined systems
	matrix solve_overdetermined(const matrix & vars, const matrix & rhs,
			unsigned algo) const;
};
GINAC_IMPLEMENT_REGISTERED_CLASS(HelperMatrix, GiNaC::matrix)

int HelperMatrix::compare_same_type(const GiNaC::basic& other) const {
	return matrix::compare_same_type(other);
}

HelperMatrix::HelperMatrix() {
}

/** Solve a linear system consisting of a m x n matrix and a m x p right hand
 *  side by applying an elimination scheme to the augmented matrix.
 *  In case the system is overdetermined, constraints on rhs are just ignored.
 *
 *  @param vars n x p matrix, all elements must be symbols
 *  @param rhs m x p matrix
 *  @param algo selects the solving algorithm
 *  @return n x p solution matrix
 *  @exception logic_error (incompatible matrices)
 *  @exception invalid_argument (1st argument must be matrix of symbols)
 *  @exception runtime_error (inconsistent linear system)
 *  @see       solve_algo */
matrix HelperMatrix::solve_overdetermined(const matrix & vars,
		const matrix & rhs, unsigned algo) const {
	const unsigned m = this->rows();
	const unsigned n = this->cols();
	const unsigned p = rhs.cols();

	// syntax checks
	if ((rhs.rows() != m) || (vars.rows() != n) || (vars.cols() != p))
		throw(std::logic_error("matrix::solve(): incompatible matrices"));
	for (unsigned ro = 0; ro < n; ++ro)
		for (unsigned co = 0; co < p; ++co)
			if (!vars(ro, co).info(info_flags::symbol))
				throw(std::invalid_argument(
						"matrix::solve(): 1st argument must be matrix of symbols"));

	// build the augmented matrix of *this with rhs attached to the right
	HelperMatrix aug(m, n + p);
	for (unsigned r = 0; r < m; ++r) {
		for (unsigned c = 0; c < n; ++c)
			aug.m[r * (n + p) + c] = this->m[r * n + c];
		for (unsigned c = 0; c < p; ++c)
			aug.m[r * (n + p) + c + n] = rhs(r, c); // = rhs.m[r*p+c];
	}

	// Gather some statistical information about the augmented matrix:
	bool numeric_flag = true;
	exvector::const_iterator r = aug.m.begin(), rend = aug.m.end();
	while (r != rend && numeric_flag == true) {
		if (!r->info(info_flags::numeric))
			numeric_flag = false;
		++r;
	}

	// Here is the heuristics in case this routine has to decide:
	if (algo == solve_algo::automatic) {
		// Bareiss (fraction-free) elimination is generally a good guess:
		algo = solve_algo::bareiss;
		// For m<3, Bareiss elimination is equivalent to division free
		// elimination but has more logistic overhead
		if (m < 3)
			algo = solve_algo::divfree;
		// This overrides any prior decisions.
		if (numeric_flag)
			algo = solve_algo::gauss;
	}

	// Eliminate the augmented matrix:
	switch (algo) {
	case solve_algo::gauss:
		aug.gauss_elimination();
		break;
	case solve_algo::divfree:
		aug.division_free_elimination();
		break;
	case solve_algo::bareiss:
	default:
		aug.fraction_free_elimination();
	}

	// assemble the solution matrix:
	matrix sol(n, p);
	for (unsigned co = 0; co < p; ++co) {
		unsigned last_assigned_sol = n + 1;
		for (int r = m - 1; r >= 0; --r) {
			unsigned fnz = 1; // first non-zero in row
			while ((fnz <= n) && (aug.m[r * (n + p) + (fnz - 1)].is_zero()))
				++fnz;
			if (fnz > n) {
				// row consists only of zeros, corresponding rhs must be 0, too
				//
				// modified: just ignore it
				//
				//if (!aug.m[r*(n+p)+n+co].is_zero()) {
				//	throw (std::runtime_error("matrix::solve(): inconsistent linear system"));
				//}
			} else {
				// assign solutions for vars between fnz+1 and
				// last_assigned_sol-1: free parameters
				for (unsigned c = fnz; c < last_assigned_sol - 1; ++c)
					sol(c, co) = vars(c, co); // = vars.m[c*p+co];
				ex e = aug.m[r * (n + p) + n + co];
				for (unsigned c = fnz; c < n; ++c)
					e -= aug.m[r * (n + p) + c] * sol(c, co); // *sol.m[c*p+co];
				sol(fnz - 1, co)
						= (e / (aug.m[r * (n + p) + (fnz - 1)])).normal();
				last_assigned_sol = fnz;
			}
		}
		// assign solutions for vars between 1 and
		// last_assigned_sol-1: free parameters
		for (unsigned ro = 0; ro < last_assigned_sol - 1; ++ro)
			sol(ro, co) = vars(ro, co);
	}

	return sol;
}

ex lsolve_overdetermined(const ex& eqns, const ex& symbols, unsigned options) {
	// solve a system of linear equations
	if (eqns.info(info_flags::relation_equal)) {
		if (!symbols.info(info_flags::symbol))
			throw(std::invalid_argument(
					"lsolve(): 2nd argument must be a symbol"));
#ifdef NEW_GINAC
        const ex sol = lsolve(lst({eqns}), lst({symbols}));
#else
        const ex sol = lsolve(lst(eqns), lst(symbols));
#endif

		GINAC_ASSERT(sol.nops()==1);
		GINAC_ASSERT(is_exactly_a<relational>(sol.op(0)));

		return sol.op(0).op(1); // return rhs of first solution
	}

	// syntax checks
	if (!eqns.info(info_flags::list)) {
		throw(std::invalid_argument(
				"lsolve(): 1st argument must be a list or an equation"));
	}
	for (size_t i = 0; i < eqns.nops(); i++) {
		if (!eqns.op(i).info(info_flags::relation_equal)) {
			throw(std::invalid_argument(
					"lsolve(): 1st argument must be a list of equations"));
		}
	}
	if (!symbols.info(info_flags::list)) {
		throw(std::invalid_argument(
				"lsolve(): 2nd argument must be a list or a symbol"));
	}
	for (size_t i = 0; i < symbols.nops(); i++) {
		if (!symbols.op(i).info(info_flags::symbol)) {
			throw(std::invalid_argument(
					"lsolve(): 2nd argument must be a list of symbols"));
		}
	}

	// build matrix from equation system
	HelperMatrix sys(eqns.nops(), symbols.nops());
	matrix rhs(eqns.nops(), 1);
	matrix vars(symbols.nops(), 1);

	for (size_t r = 0; r < eqns.nops(); r++) {
		const ex eq = eqns.op(r).op(0) - eqns.op(r).op(1); // lhs-rhs==0
		ex linpart = eq;
		for (size_t c = 0; c < symbols.nops(); c++) {
			const ex co = eq.coeff(ex_to<symbol> (symbols.op(c)), 1);
			linpart -= co * symbols.op(c);
			sys(r, c) = co;
		}
		linpart = linpart.expand();
		rhs(r, 0) = -linpart;
	}

	// test if system is linear and fill vars matrix
	for (size_t i = 0; i < symbols.nops(); i++) {
		vars(i, 0) = symbols.op(i);
		if (sys.has(symbols.op(i)))
			throw(std::logic_error("lsolve: system is not linear"));
		if (rhs.has(symbols.op(i)))
			throw(std::logic_error("lsolve: system is not linear"));
	}

	matrix solution;
	try {
		solution = sys.solve_overdetermined(vars, rhs, options);
	} catch (const std::runtime_error & e) {
		LOG("we got exception: " << e.what());
		// Probably singular matrix or otherwise overdetermined system:
		// It is consistent to return an empty list
		return lst();
	}
	GINAC_ASSERT(solution.cols()==1);
	GINAC_ASSERT(solution.rows()==symbols.nops());

	// return list of equations of the form lst(var1==sol1,var2==sol2,...)
	lst sollist;
	for (size_t i = 0; i < symbols.nops(); i++)
		sollist.append(symbols.op(i) == solution(i, 0));

	return sollist;
}

GiNaC::exmap lsolve_result_to_rules(const GiNaC::ex& sol,
		const GiNaC::ex& symbols) {
	if (!is_a<lst> (symbols)) {
		stringstream ss;
		ss << "symbols are not given as a list: " << symbols;
		throw runtime_error(ss.str());
	}
	lst symlst = ex_to<lst> (symbols);
	// convert to rules
	exmap rules;
	try {
		if (!is_a<lst> (sol))
			throw runtime_error("Failed to solve system (result not a list)");
		lst l = ex_to<lst> (sol);
		//		if (l.nops() == 0)
		//			throw runtime_error(
		//					"Failed to solve system (problematic singular matrix)");
		for (lst::const_iterator e = l.begin(); e != l.end(); ++e) {
			if (!is_a<relational> (*e))
				throw runtime_error(
						"Failed to solve system (unknown structure)");
			lst::const_iterator i;
			for (i = symlst.begin(); i != symlst.end(); ++i)
				if (e->has(*i))
					break;
			if (i == symlst.end()) // relation is free of indet. => ignore it
				continue;
			for (i = symlst.begin(); i != symlst.end(); ++i)
				if (e->op(0).match(*i))
					break;
			if (i == symlst.end())
				throw runtime_error("Failed to solve system (unknown equation)");
			for (i = symlst.begin(); i != symlst.end(); ++i)
				if (e->op(1).has(*i))
					break;
			if (i != symlst.end())
				throw runtime_error("Failed to solve system (under-determined)");
			rules[e->op(0)] = e->op(1);
		}
		if (rules.size() < symlst.nops())
			throw runtime_error("Failed to solve system (too few solutions)");
		if (rules.size() > symlst.nops())
			throw runtime_error("Failed to solve system (too many solutions)");
	} catch (exception& e) {
		stringstream ss;
		ss << "equations:\n" << sol << "\n";
		ss << " " << symlst.nops() << " indeterminants:\n" << symlst;
		throw runtime_error(string(e.what()) + "\n" + ss.str());
	}
	return rules;
}

GiNaC::exmap lsolve_pattern(const GiNaC::lst& eqns, const GiNaC::ex& pattern) {
	// find the indeterminants
	exset indets; // indeterminants
	for (lst::const_iterator e = eqns.begin(); e != eqns.end(); ++e)
		e->find(pattern, indets);
	// try to solve the system (introduce temp. symbols to make lsolve work)
	int count = 0;
	lst syms;
	exmap i2s, s2i;
	for (exset::const_iterator i = indets.begin(); i != indets.end(); ++i) {
		stringstream symbolname;
		symbolname << "tmp" << (++count);
		symbol s(symbolname.str());
		i2s[*i] = s;
		s2i[s] = *i;
		syms.append(s);
	}
	ex sol = lsolve(eqns.subs(i2s), syms, solve_algo::gauss).subs(s2i);
	// convert to rules
	exmap rules;
	try {
		if (!is_a<lst> (sol))
			throw runtime_error("Failed to solve system (result not a list)");
		lst l = ex_to<lst> (sol);
		for (lst::const_iterator e = l.begin(); e != l.end(); ++e) {
			if (!is_a<relational> (*e))
				throw runtime_error(
						"Failed to solve system (unknown structure)");
			if (!e->op(0).match(pattern))
				throw runtime_error("Failed to solve system (unknown equation)");
			if (e->op(1).has(pattern))
				throw runtime_error("Failed to solve system (under-determined)");
			rules[e->op(0)] = e->op(1);
		}
		if (rules.size() != syms.nops())
			throw runtime_error("Failed to solve system (over-determined)");
	} catch (exception& e) {
		stringstream ss;
		ss << eqns.nops() << " equations:\n" << eqns << "\n";
		ss << syms.nops() << " indeterminants:\n" << indets;
		throw runtime_error(string(e.what()) + "\n" + ss.str());
	}
	return rules;
}

GiNaC::lst to_equations(const GiNaC::exmap& m) {
	lst result;
	for (exmap::const_iterator i = m.begin(); i != m.end(); ++i)
		result.append(i->first == i->second);
	return result;
}

// convert GinaC::lst of equations { a = b, c == d } to map a->b, c->d
GiNaC::exmap equations_to_substitutions(const GiNaC::lst& eqs,
		GiNaC::exmap lhs_substitutions, bool discard_identities) {
	using namespace GiNaC;
	exmap m;
	for (lst::const_iterator e = eqs.begin(); e != eqs.end(); ++e) {
		if (!is_exactly_a<relational> (*e))
			ABORT("This is no equation: " << *e);
		relational r = ex_to<relational> (*e);
		ex key = r.lhs().subs(lhs_substitutions);
		ex val = r.rhs();
		if (!discard_identities || !val.is_equal(key))
			m[key] = val;
	}
	return m;
}

GiNaC::ex jacobi_determinant(const GiNaC::lst& old2new,
		const GiNaC::lst& new_vars) {
	using namespace GiNaC;
	if (old2new.nops() != new_vars.nops())
		return 0;
	vector<ex> old_of_new;
	vector<symbol> new_symbols;
	for (lst::const_iterator e = old2new.begin(); e != old2new.end(); ++e) {
		if (!is_exactly_a<relational> (*e))
			ABORT("This is no equation: " << *e);
		old_of_new.push_back(ex_to<relational> (*e).rhs());
	}
	for (lst::const_iterator v = new_vars.begin(); v != new_vars.end(); ++v) {
		if (!is_exactly_a<symbol> (*v))
			ABORT("This is no symbol: " << *v);
		new_symbols.push_back(ex_to<symbol> (*v));
	}
	matrix jacmat(old_of_new.size(), new_symbols.size());
	for (unsigned int i = 0; i < old_of_new.size(); ++i)
		for (unsigned int j = 0; j < new_symbols.size(); ++j)
			jacmat(i, j) = old_of_new[i].diff(new_symbols[j]);
	return jacmat.determinant();
}

// form output functions

print_form::print_form() :
	print_dflt(std::cout) {
}

GINAC_IMPLEMENT_PRINT_CONTEXT(print_form, print_dflt)

void print_pseries_as_form(const pseries & s, const print_form& c,
		unsigned level) {
	ex es = s.convert_to_poly(true);
	if (s.precedence() <= level)
		c.s << '(';
	if (s.is_terminating()) {
		es.print(c);
	} else {
		es.print(c);
		c.s << " + Unknown(" << s.gethash() << ")*";
		pow(s.get_var(), s.degree(s.get_var())).print(c);
	}
	if (s.precedence() <= level)
		c.s << ')';
}

void print_power_as_form(const power& p, const print_form& c, unsigned level) {
	const ex& base = p.op(0);
	const ex& expo = p.op(1);
	if (expo == numeric(1, 2)) {
		c.s << "sqrt(" << base << ")";
	} else if (is_a<numeric> (expo) && ex_to<numeric> (expo).is_negative()) {
		c.s << "Den(";
		base.print(c);
		c .s << ')';
		if (!expo.is_equal(-1)) {
			c.s << '^';
			(-expo).print(c, p.precedence());
		}
	} else {
		if (p.precedence() <= level)
			c.s << '(';
		base.print(c, p.precedence());
		c.s << '^';
		expo.print(c, p.precedence());
		if (p.precedence() <= level)
			c.s << ')';
	}
}

// mathematica output functions

print_mma::print_mma() :
	print_dflt(std::cout) {
}

GINAC_IMPLEMENT_PRINT_CONTEXT(print_mma, print_dflt)

GiNaC::ex subs_conjugate::operator()(const GiNaC::ex &e) {
	if (GiNaC::is_a<GiNaC::function>(e) && //
			GiNaC::ex_to<GiNaC::function>(e).get_name() == "conjugate")
		return ConjugateRedef((*this)(e.op(0)));
	else if (GiNaC::is_a<GiNaC::pseries>(e))
		return e;
	else
		return e.map(*this);
}

void print_pseries_as_mma(const pseries & s, const print_mma& c, unsigned level) {
	ex es = s.convert_to_poly(true);
	if (s.precedence() <= level)
		c.s << '(';
	if (s.is_terminating()) {
		es.print(c);
	} else {
		es.print(c);
		c.s << " + Unknown[" << s.gethash() << "]*";
		s.get_var().print(c);
		c.s << '^' << s.degree(s.get_var());
	}
	if (s.precedence() <= level)
		c.s << ')';
}

void print_power_as_mma(const power& p, const print_mma& c, unsigned level) {
	const ex& base = p.op(0);
	const ex& expo = p.op(1);
	if (expo == numeric(1, 2)) {
		c.s << "Sqrt[" << base << "]";
	} else {
		if (p.precedence() <= level)
			c.s << '(';
		base.print(c, p.precedence());
		c.s << '^';
		expo.print(c, p.precedence());
		if (p.precedence() <= level)
			c.s << ')';
	}
}

static void ConjugateRedef_print_mma(const GiNaC::ex& arg,
		const print_context& c) {
	c.s << "Conjugate[";
	arg.print(c);
	c.s << "]";
}

REGISTER_FUNCTION(ConjugateRedef, print_func<print_mma>(ConjugateRedef_print_mma))

REGISTER_FUNCTION(Ident, dummy())

static void Color_print_mma(const GiNaC::ex& arg, const GiNaC::print_context& c) {
	c.s << "Color[";
	arg.print(c);
	c.s << "]";
}
REGISTER_FUNCTION(Color, print_func<print_mma>(Color_print_mma))

static void Tag_print_mma(const GiNaC::ex& arg, const print_context& c) {
	c.s << "Tag[";
	arg.print(c);
	c.s << "]";
}

REGISTER_FUNCTION(Tag, print_func<print_mma>(Tag_print_mma))

// maple output functions

print_maple::print_maple() :
	print_dflt(std::cout) {
}

GINAC_IMPLEMENT_PRINT_CONTEXT(print_maple, print_dflt)

void print_pseries_as_maple(const pseries & s, const print_maple& c,
		unsigned level) {
	ex es = s.convert_to_poly(true);
	if (s.precedence() <= level)
		c.s << '(';
	if (s.is_terminating()) {
		es.print(c);
	} else {
		es.print(c);
		c.s << " + Unknown(" << s.gethash() << ")*";
		s.get_var().print(c);
		c.s << '^' << s.degree(s.get_var());
	}
	if (s.precedence() <= level)
		c.s << ')';
}

void print_power_as_maple(const power& p, const print_maple& c, unsigned level) {
	const ex& base = p.op(0);
	const ex& expo = p.op(1);
	if (expo == numeric(1, 2)) {
		c.s << "sqrt(" << base << ")";
	} else {
		if (p.precedence() <= level)
			c.s << '(';
		base.print(c, p.precedence());
		c.s << '^';
		expo.print(c, p.precedence());
		if (p.precedence() <= level)
			c.s << ')';
	}
}

void print_as_mma(const GiNaC::exmap& m, std::ostream& os, bool linebreaks) {
	os << "{" << (linebreaks ? "\n  " : "");
	for (exmap::const_iterator p = m.begin(); p != m.end(); ++p) {
		if (p != m.begin())
			os << "," << (linebreaks ? "\n  " : "");
		p->first.print(print_mma(os));
		os << "->";
		p->second.print(print_mma(os));
	}
	os << (linebreaks ? "\n}" : "}");
}

void init_ginacutils() {
	// set up print functions for builtin GiNaC classes
	set_print_func<pseries, print_form> (print_pseries_as_form);
	set_print_func<power, print_form> (print_power_as_form);
	set_print_func<pseries, print_mma> (print_pseries_as_mma);
	set_print_func<power, print_mma> (print_power_as_mma);
	set_print_func<pseries, print_maple> (print_pseries_as_maple);
	set_print_func<power, print_maple> (print_power_as_maple);
}

} // namespace Reduze

