/*  int.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_INT_H
#define REDUZE_INT_H

#include "functions.h"
#include "ginac/ginac.h"
#include "yamlconfigurable.h"

#ifdef USE_HASH_TABLE
#include <tr1/unordered_map>
#include <tr1/unordered_set>
#endif

namespace Reduze {

class IntegralFamily;
class SectorMappings;
class INTGeneric;
class LinearCombination;
class Sector;
class Permutation;

/// Feynman integral
/** An integral is a list of exponents, where positive (negative) entries
 *  belong to propagators in the denominator (numerator). ID is the
 *  identification number of the sector, t the number of the propagators
 *  in the denominator and r (s) the absolute value of the sum of the exponents
 *  in the denominator (numerator).
 */
class INT {
public:
	typedef std::vector<int8>::const_iterator const_iterator;

	/// generates an integral with all values set to 0
	/** default INT is the trivial zero integral for #loop momenta > 0,
	 ** or defined as 1 for #loop momenta == 0 **/
	INT(const IntegralFamily* m);
	/// generates the integral of topology ID_ with no additional propagators
	explicit INT(const IntegralFamily* m, int id);
	/// generates the integral of topology ID_ with no additional propagators
	explicit INT(const Sector& s);
	/// generates an integral from the vector v
	explicit INT(const IntegralFamily* m, const std::vector<int8> & v);
	/// generates an integral from a string in Reduze format
	explicit INT(const std::string & str);
	/// generates an integral from a YAML node
	explicit INT(const YAML::Node& node);
	/// generates an integral from an generic integral using the map to replace symbols by numbers
	explicit INT(const INTGeneric& in2, const GiNaC::exmap& m);
	virtual ~INT() {
	}

	/// returns an INT constructed from Mathematica-type string
	static INT from_mma(const std::string& s);

	// getter/setter

	/// returns the IntegralFamily the integral belongs to
	inline const IntegralFamily* integralfamily() const {
		return integralfamily_;
	}
	/// returns the sector the INT belongs to
	Sector get_sector() const;

	/// returns the length of the vector 'v_'
	size_t size() const {
		return v_.size();
	}
	const_iterator begin() const {
		return v_.begin();
	}
	const_iterator end() const {
		return v_.end();
	}
	const std::vector<int8>& v() const {
		return v_;
	}
	int8 v_i(int index) const {
		VERIFY(0 <= index && index < int(size()));
		return v_[index];
	}

	/// returns the ID number
	int id() const {
		return id_;
	}
	/// returns t
	int8 t() const {
		return t_;
	}
	/// returns r
	int8 r() const {
		return r_;
	}
	/// returns s
	int8 s() const {
		return s_;
	}
	/// sets the metric convention the integral is defined in
	/** b==false: g=diag(1,-1,-1,-1), b==true: g=diag(-1,1,1,1) **/
	void set_metric_mostly_plus(bool b) {
		metric_mostly_plus_ = b;
	}
	bool uses_metric_mostly_plus() const {
		return metric_mostly_plus_;
	}

	/// returns the minimal equivalent integral of integral 'integral'
	/** employs the permutation symmetries of the integral family */
	INT get_equivalent() const;

//	LinearCombination apply_shift(const GiNaC::exmap& shift,
//			const IntegralFamily* f, int mult, bool set_subsec_to_zero) const;

	// properties of the integral

	/// returns true if the integral is obviously zero, false otherwise
	/** returns also true if a propagator, which is declared as a cut propagator
	 *	appears in the integral with an exponent <= 0 */
	bool is_obviously_zero() const;
	/// returns the mass dimension of the integrand: 2*(s-r)
	int mass_dimension() const {
		return 2 * (static_cast<int> (s_) - static_cast<int> (r_));
	}
	/// returns the integrand in terms of instances of Propagator
	GiNaC::ex get_integrand() const;
	/// returns a hash value
	size_t hash_value() const;

	// comparison

	bool operator ==(const INT& other) const;
	bool operator !=(const INT& other) const;
	bool operator <(const INT& other) const {
		bool preferred = is_preferred(*this);
		if (preferred != is_preferred(other))
			return preferred;
		return (*ordering)(*this, other);
	}
	bool operator >(const INT& other) const {
		return other < *this;
	}

	/// sets the global ordering for instances of INT
	static void set_ordering(const std::string& type);

	// preferred integrals

	/// sets integral and its crossed partners to be preferred masters
	/** change ordering: set integral lower than all which are not preferred */
	static void set_preferred(const INT&);
	/// set_preferred() for all integrals contained in a file
	/** empty filename is ignored **/
	static void load_preferred(const std::string& filename);
	/// whether integral or crossed partner is preferred master
	/** the integral with r = s = 0 is always preferred **/
	static bool is_preferred(const INT&);

	// input/output

	///  writes the integral to the stream 'strm' in FORM format
	void to_form_stream(std::ostream & strm) const;
	///  writes the integral to the stream 'strm' in Maple format
	void to_maple_stream(std::ostream & strm) const;
	///  writes the integral to the stream 'strm' in Mathematica format
	void to_mma_stream(std::ostream & strm) const;
	/// format is a space separated row of the exponents
	friend std::ostream & operator <<(std::ostream& strm, const INT & in);
	/// reads from YAML
	friend void operator >>(const YAML::Node& node, INT& in);
	/// writes to YAML
	friend YAML::Emitter& operator <<(YAML::Emitter& os, const INT& in);

private:
	/// the integral family
	const IntegralFamily* integralfamily_;
	/// the entries of the vector 'v_' are the exponents of the propagators
	std::vector<int8> v_;
	/// the identification number for the sector
	int id_;
	/// t is the number of different propagators in the denominator
	int8 t_;
	/// r is the sum of the exponents in the denominator
	int8 r_;
	/// s is the sum of the exponents in the numerator, (absolut value)
	int8 s_;
	/// metric convention the integral is defined in, default: false
	/** b==false: g=diag(1,-1,-1,-1), b==true: g=diag(-1,1,1,1) **/
	bool metric_mostly_plus_;

private:
	/// initialize the integral: calculates the parameters t_, r_, s_, id_ from v_
	void init();
	/// applies (the inverse of) the permutation to the integral
	INT& transform(const Permutation&);

	// different orderings, ignoring 'preferred' feature
	static bool less_rs(const INT&, const INT&);
	static bool less_sr(const INT&, const INT&);
	static bool less_degrlex_r(const INT&, const INT&);
	static bool less_degrlex_s(const INT&, const INT&);

	/// comparison class for set of preferred INTs
	struct CompareNoPreferred {
		bool operator()(const INT& a, const INT& b) {
			return (*ordering)(a, b);
		}
	};

	/// pointer to less function to define the ordering for instances of INT
	static bool (*ordering)(const INT&, const INT&);
	/// integrals which are preferred
	/** store only uncrossed INTs here */
	static std::set<INT, CompareNoPreferred> preferred_integrals;
};

// GLOBAL FUNCTIONS
/// operator for printing an integral
std::ostream & operator<<(std::ostream& os, const INT& in);

//
//
//


class LinearCombinationGeneric;
class OP;
class OPPROD;
class OPSUM;

/// integral with general exponents
class INTGeneric {
public:
	typedef std::vector<GiNaC::ex>::const_iterator const_iterator;

	/// construct an integral with general propagator exponents a1, ..., an
	INTGeneric(const IntegralFamily* m);
	/// construct from string
	INTGeneric(const std::string& str);

	virtual ~INTGeneric() {
	}

	// getters

	/// returns the IntegralFamily the integral belongs to
	inline const IntegralFamily* integralfamily() const {
		return integralfamily_;
	}
	/// returns the length of the vector 'v_'
	size_t size() const {
		return v_.size();
	}
	const_iterator begin() const {
		return v_.begin();
	}
	const_iterator end() const {
		return v_.end();
	}

	/// applies the shift operator on the integral
	INTGeneric shift(const OP& op) const;
	/// applies the shift operator product on the integral
	INTGeneric shift(const OPPROD& op) const;
	/// applies the shift operators on the integral
	LinearCombinationGeneric shift(const OPSUM& op) const;

	// comparison
	bool operator <(const INTGeneric& other) const;
	bool operator ==(const INTGeneric& other) const;
	bool operator !=(const INTGeneric& other) const;

	// output
	void to_form_stream(std::ostream& os) const;
	void to_maple_stream(std::ostream& os) const;
	void to_mma_stream(std::ostream& os) const;

private:
	/// the integral family
	const IntegralFamily* integralfamily_;
	/// the entries of the vector 'v_' are the exponents of the propagators
	std::vector<GiNaC::ex> v_;
};

// GLOBAL FUNCTIONS
/// operator for printing an integral
std::ostream& operator <<(std::ostream& os, const INTGeneric& in);

//
//
//


class RSPoint;
class RSFiniteGenericSelection;

class SeedGeneratorOptions: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("seed_generator_options");
		s.set_short_description("Options for the seed generator.");
		s.set_long_description(""//
					"Options for the seed generator.");
		s.add_option("reduce_to_equivalent", false, "boolean", ""//
					"Whether to replace integrals by lowest equivalent"
					" according to permutation symmetries.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	SeedGeneratorOptions() :
		reduce_to_equivalent_(true) {
	}
	virtual ~SeedGeneratorOptions() {
	}

	/// whether permutation symmetries of the integralfamily should be used
	void set_reduce_to_equivalent(bool b);
	virtual void print(YAML::Emitter& os) const;
	virtual void read(const YAML::Node&);

private:
	bool reduce_to_equivalent_;
	friend class SeedGenerator;
};

// operators
inline void operator >>(const YAML::Node& node, SeedGeneratorOptions& opt) {
	opt.read(node);
}
inline YAML::Emitter& operator <<(YAML::Emitter& os,
		const SeedGeneratorOptions& opt) {
	opt.print(os);
	return os;
}

/// class to generate seed integrals
class SeedGenerator {
public:
	SeedGenerator(const IntegralFamily* ic, const SeedGeneratorOptions& options) :
		integralfamily_(ic), /**/
		options_(options) {
	}
	virtual ~SeedGenerator() {
	}

	/// generate all integrals of topology id and class (r,s)
	/** integrals are appended to out */
	void generate(int id, int r, int s, std::set<INT>& out) const;
	/// generate all integrals of topology id and with r in (r_min,r_max) and s in (s_min,s_max)
	/** integrals are appended to out */
	void generate(int id, int r_min, int r_max, int s_min, int s_max, std::set<
			INT>& out) const;
	/// generate all integrals of topology id which are contained in the range rs_generic
	/** integrals are appended to out */
	void generate(int id, const RSFiniteGenericSelection& rs_generic, std::set<
			INT>& out) const;
	/// generate all integrals of topology id which are contained in the ranges rs_generic
	/** integrals are appended to out */
	void generate(int id,
			const std::list<RSFiniteGenericSelection>& rs_generic,
			std::set<INT>& out) const;

private:
	/// never unbound
	SeedGenerator() :
		integralfamily_(0) {
	}

	const IntegralFamily* integralfamily_;
	SeedGeneratorOptions options_;
};

} // namespace Reduze


#ifdef USE_HASH_TABLE
namespace std {
	namespace tr1 {
		template<>
		struct hash<Reduze::INT> {
			size_t operator()(const Reduze::INT& i) const {
				return i.hash_value();
			}
		};
	}
}
#endif /* USE_HASH_TABLE */

#endif /* REDUZE_INT_H */

