/*  job_applycrossing.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_applycrossing.h"
#include "files.h"
#include "filedata.h"
#include "kinematics.h"
#include "equation.h"
#include "identitygenerator.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<ApplyCrossing> dummy;
}

void ApplyCrossing::run_serial() {
	Crossing crossing(crossing_);
	InFileEquations in(input_filename_);
	OutFileEquations out(output_filename_);
	Identity ident;
	bool have_eqs = true;
	// replace individual integrals also by minimal equivalent crossings
	bool replace_by_min_crossing = true;
	while (in.get(ident)) {
		Identity xid = crossing.transform(ident, replace_by_min_crossing);
		// exploit special shifts of crossed sectors to their uncrossed version
		set<INT> ints;
		xid.find_INT(ints);
		for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i) {
			Identity red;
			red = IdentityGenerator::get_shift_identity(*i, have_eqs, false);
			xid.replace(red, false);
		}
		out << xid;
	}
	in.close();
	out.finalize();
}

void ApplyCrossing::print_manual_options(YAML::Emitter& os) const {
	using namespace YAML;
	os << Key << "crossing" << Value << crossing_;
}

void ApplyCrossing::read_manual_options(const YAML::Node& node) {
	Kinematics* kin = Files::instance()->kinematics();
	crossing_ = Crossing(kin);
	node["crossing"] >> crossing_;
}

bool ApplyCrossing::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(input_filename_);
	out.push_back(output_filename_);
	return true;
}

std::string ApplyCrossing::get_description() const {
	return string("apply crossing ") + crossing_.name() + " to "
			+ short_filename(input_filename_);
}

void ApplyCrossing::init() {
	if (input_filename_.empty() || output_filename_.empty())
		throw runtime_error("input or output file name undefined.");
	if (input_filename_ == output_filename_)
		throw runtime_error("same input and output file name.");
}

}
