/*  job_findfiniteintegrals.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_FINDFINITEINTEGRALS_H_
#define JOB_FINDFINITEINTEGRALS_H_

#include "job.h"

namespace Reduze {

/// search for quasi-finite integrals
class FindFiniteIntegrals : public Reduze::Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("find_finite_integrals");
		s.set_short_description("Find quasi-finite integrals.");
		s.set_long_description("Job to determine quasi-finite or finite integrals,"
				" implementing the algorithm described in arXiv:1411.7392."
				" Note that this function will search only consider integrals"
				" with dots and ignore the possibility of (quasi-)finite integrals"
				" with additional numerators.");
		s.add_option("input_file", true, "string", ""
				"Name of input file containing integrals to consider.");
		s.add_option("output_file", true, "string", ""
				"Name of output file with quasi-finite integrals.");
		s.add_option("reference_dimension" , true, "integer", ""
				"Dimension for which target integrals should be quasi-finite, e.g. 4.");
		s.add_option("num_candidates", false, "integer", ""
				"Number of integrals to construct per sector.");
		s.add_option("require_finiteness", false, "boolean", ""
				"Whether truly finite integrals should be selected. If disabled,"
				" also quasi-finite integrals are generated, which have a 1/eps"
				" divergence in the Gamma function prefactor in the Feynman"
				" parameter representation, see also arXiv:1510.06758.");
        s.add_option("min_omega", false, "integer", ""
                     "Minimal value of omega_JK. min_omega=0 selects convergent integrations, min_omega>0 suppresses the integrand at the boundaries x=0 and x=infinity.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	FindFiniteIntegrals() : reference_dimension_(4), num_candidates_(1), require_finiteness_(false),
        min_omega_(0) {
		add_auto_options();
	}
	virtual ~FindFiniteIntegrals() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;
protected:
	virtual void add_auto_options() {
		add_auto_io("input_file", input_file_);
		add_auto_io("output_file", output_file_);
		add_auto_io("reference_dimension", reference_dimension_);
		add_auto_io("num_candidates", num_candidates_);
		add_auto_io("require_finiteness", require_finiteness_);
        add_auto_io("min_omega", min_omega_);
	}
	virtual void init();
	//virtual void print_manual_options(YAML::Emitter& os) const;
	//virtual void read_manual_options(const YAML::Node&);
private:
    std::string input_file_, output_file_;
    int reference_dimension_, num_candidates_;
    bool require_finiteness_;
    int min_omega_;
};

}

#endif /* JOB_FINDFINITEINTEGRALS_H_ */
