/*  job_insertreductions.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_insertreductions.h"
#include "functions.h"
#include "sector.h"
#include "amplitude.h"
#include "equation.h"
#include "equationlist.h"
#include "files.h"
#include "filedata.h"
#include "sectormappings.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<InsertReductions> dummy;
}

bool InsertReductions::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(input_filename_);
	in.push_back(reductions_filename_);
	if (!preferred_masters_filename_.empty())
		in.push_back(preferred_masters_filename_);
	out.push_back(output_filename_);
	return true;
}

void InsertReductions::init() {
	if (input_filename_.empty())
		throw runtime_error("no valid amplitudes_file defined");
	if (reductions_filename_.empty())
		throw runtime_error("no valid reductions_file defined");
	if (output_filename_.empty())
		throw runtime_error("no valid output_file defined");
	if (output_filename_ == input_filename_)
		throw runtime_error("input and output files are identical: "
				+ input_filename_);
}

std::string InsertReductions::get_description() const {
	return string("insert reductions in ") + short_filename(input_filename_);
}

void InsertReductions::run_serial() {
	using namespace GiNaC;

	INT::load_preferred(preferred_masters_filename_);

	LOG("Checking type of input file " << input_filename_);
	bool have_eqs = false;
	if (InFileEquations::check_filetype(input_filename_))
		have_eqs = true;
	else if (!InFileLinearCombinations::check_filetype(input_filename_))
		throw runtime_error("unknown contents in file " + input_filename_);
	LOG("=> file contains " << (have_eqs ? "equations" : "linear combinations"));

	LOG("Loading reductions from file " << reductions_filename_);
	InFileEquations inred(reductions_filename_);
	size_t s = inred.file_size();
	ProgressBar pbar(2, "loading " + to_string(s) + " bytes", s);
	pbar.start();
	map<INT, Identity> reductions;
	Identity id;
	while (inred.get(id)) {
		pbar.print(inred.tellg());
		if (!id.empty()) {
			if (!have_eqs)
				id.reconstruct_symbol_replaced_by_one();
			reductions.insert(make_pair(id.max_INT(), id));
		}
	}
	inred.close();
	pbar.end();

	LOG("Reading file " << input_filename_);
	InFileLinearCombinations in(input_filename_.c_str());
	OutFileLinearCombinations out(output_filename_.c_str());

	Timer timer_all;
	set<INT> resints;
	LinearCombination lc;
	while (in.get(lc)) {
		LOG("Processing " << lc.name());
		set<INT> ints;
		lc.find_INT(ints);
		LOG("  found " << ints.size() << " integrals");
		IdentityList reds;
		Timer timer_replace;
		timer_replace.pause();
		LOGN("  ");
		for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i) {
			map<INT, Identity>::const_iterator r = reductions.find(*i);
			if (r != reductions.end()) {
				LOGN("r");
				timer_replace.unpause();
				bool normalize = false;
				lc.replace(r->second, normalize);
				timer_replace.pause();
			} else {
				LOGN(".");
			}
		}
		lc.find_INT(resints);
		LOG("");

		LOG("  writing result");
		out << lc;

		LOG("  time to process amplitude: " << timer_all.get_wall_time() << " s");
		LOG("  time to replace integrals: " << timer_replace.get_wall_time() << " s");
		LOG("");
		timer_all.restart();
	}
	in.close();
	out.finalize();

	LOG("\nIntegrals:");
	LOG("  " << resints.size() << " different integrals remain in the result in total");
	set<INT> nomi;
	for (set<INT>::const_iterator i = resints.begin(); i != resints.end(); ++i)
		if (!INT::is_preferred(*i))
			nomi.insert(*i);
	LOG("  " << nomi.size() << " of them are no known masters (unreduced)");

	LOGX("\nIntegrals which are not known masters:");
	for (set<INT>::const_iterator i = nomi.begin(); i != nomi.end(); ++i) {
		stringstream ss;
		i->to_mma_stream(ss);
		LOGX("  " << ss.str());
	}
}

}
