/*  job_reducefiles.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_reducefiles.h"
#include "functions.h"
#include "equationlist.h"
#include "files.h"
#include "filedata.h"
#include "job_runreduction.h"
#include "yamlutils.h"
#include <functional>

using namespace std;

namespace Reduze {

// register job type at JobFactory
namespace {
JobProxy<ReduceFiles> dummy;
}

bool ReduceFiles::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	find_dependencies_reductions(outothers, in);
	in.insert(in.end(), equation_files_.begin(), equation_files_.end());
	in.insert(in.end(), substitution_files_.begin(), substitution_files_.end());
	out.push_back(output_file_);
	if (!preferred_masters_file_.empty())
		in.push_back(preferred_masters_file_);

	string tmp_dir = get_tmp_directory();
	make_directory(tmp_dir);
	list<string> delete_files, delete_directories;
	RunReduction* job = new RunReduction(tmp_dir, equation_files_,
			substitution_files_, output_file_, reduzer_options_,
			preferred_masters_file_, delete_files, delete_directories);
	job->set_conditional(is_conditional());
#ifdef HAVE_MPI
	if (!use_full_parallelization_)
	job->set_num_workers_range(0, 0);
	else
	job->set_num_workers_range(min_workers_, max_workers_);
#endif
	auxjobs.push_back(job);
	in.push_back(output_file_);
	return true;
}

void ReduceFiles::run_serial() {
	remove_directory_with_files(get_tmp_directory());
}

std::string ReduceFiles::get_description() const {
	return string("reduce files ")//
			+ (equation_files_.empty() ? "" : short_filename(equation_files_.front()))//
			+ (equation_files_.size() == 1 ? "" : " + ...");
}

void ReduceFiles::init() {
	list<string>::iterator fn;
	for (fn = equation_files_.begin(); fn != equation_files_.end(); ++fn) {
		*fn = get_canonical_filename(*fn);
		if (fn->empty())
			throw runtime_error("input file undefined");
	}
	output_file_ = get_canonical_filename(output_file_);
	if (output_file_.empty())
		throw runtime_error("output file undefined");
#ifdef HAVE_MPI
	if (min_workers_ < 1)
	ERROR("min_workers must be 1 or higher");
	if (max_workers_ < min_workers_)
	ERROR("max_workers must be equal to min_workers or higher");
#endif
}

string ReduceFiles::get_tmp_directory() const {
	Files* files = Files::instance();
	string tmp_dir = files->get_tmp_directory()
			+ get_filename_without_directory(output_file_) + "_dir/";
	return tmp_dir;
}

}
