/*  triconnected.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef TRICONNECTED_H_
#define TRICONNECTED_H_

#include "topology.h"

namespace Reduze {

class SplitComponents;

/// Decomposition of an arbitrary multigraph into tri-connected components
/** Employs a SplitComponents instance for each bi-connected component. **/
class TriConnectedComponents {
public:
	/// performs the decomposition
	/** Nodes and edges must have non-negative ids (maximal ids
	 ** should not differ too much from total numbers). The flag controls
	 ** whether the original edges should be considered indistinguishable
	 ** (i.e. have the same mass). */
	TriConnectedComponents(const Topology& t, bool suppress_edge_permutations);
	/// performs next twist of underlying graph
	/** must call init_twist() first. returns false for last **/
	bool set_next_twist();
	/// returns the current twisted graph
	/** nodes without any edge are dropped **/
	Topology merge_to_graph() const;
	/// returns set of edge ids for edges with reverse orientation
	std::set<int> flipped_edges() const;
	/// returns list of 2-isomorphic graphs
	/** 2-isomorphic graphs are related by an isomorphism of one graph
	 ** to a twisted version of the other graph. Any graph reachable
	 ** by twistings is isomorphic to at least one graph in the returned
	 ** list, the list may contain redundant entries. Second entry in
	 ** the returned pairs contains ids of all edges whose orientation were
	 ** reversed*/
	std::list<std::pair<Topology, std::set<int> > >//
	find_topologies_of_matroid();
private:
	std::vector<SplitComponents> comps_;
	Topology nonbiconnected_;
};

/// Decomposition of a biconnected multigraph into tree of split components
/** allows to iterate over twisted versions of the multigraph **/
class SplitComponents {
public:
	enum ComponentType {
		Bond /*    */= (1u << 0), // P
		Polygon /* */= (1u << 1), // S (cycles)
		TriComponent = (1u << 2)
	// R (3-vertex connected)
	};

	/// generates a tree of fully split components of a biconnected graph
	/** This fully split decomposition is not unique. To generate the actual
	 ** unique 3-vertex-connected components, call merge_to_SPQR() subsequently.
	 ** The graph 't' must be biconnected, the nodes and edges must have
	 ** non-negative ids (maximal ids should not differ too much from total
	 ** numbers), multiple edges are allowed. */
	SplitComponents(const Topology& t);
	/// merges to so-called tri-connected components (SPQR tree)
	/** this gives a unique decomposition with virtual edges for Tutte splits */
	void merge_to_SPQR();
	/// returns whether the edge is virtual
	bool is_virtual_edge(int edge_id) const;
	/// returns the combination of all components joined at the virtual edges
	Topology merge_to_graph() const;
	/// returns whether edge occurs and has reverse orientation
	std::set<int> flipped_edges() const;
	/// prepares iteration over twisted graphs
	/** argument controls whether permutations of original edges within
	 ** 'polygons' should be suppressed */
	void init_twist(bool are_original_edges_indistinguishable);
	/// performs next twist of underlying graph
	/** The iteration will not necessarily produce all possible twisted graphs,
	 ** but at least one representative of all possible isomorphism classes.
	 ** Must call init_twist() first. Returns false for last. **/
	bool set_next_twist();
	/// performs a twist of the underlying graph at a virtual edge
	/** exchanges two vertices in one of two subgraphs of underlying graph **/
	void perform_twist(int vid, const Edge& te);
	/// returns ordered list of non-isomorphic but 2-isomorphic graphs
	/** 2-isomorphic graphs are related by an isomorphism of one graph
	 ** to a twisted version of the other graph **/
	//std::list<Topology> find_matroid_isomorphism_classes();
	/// produces a DOT file for a graphical representation
	void print_dot(const std::string& filename) const;
	/// produces a postscript file with graphics
	void print_postscript(const std::string& filename) const;

protected:
	std::string name_;
	/// combines adjacent components of given types
	void merge(int type_mask);

private:
	/// converts a component into a Topology
	Topology convert_to_topology(const std::list<Edge>& edges) const;

	// use Topology for now, we want to use same (high) virtual edge ids
	// for our edges as in original graph
	Topology tree_; // tree of components (direct access to virtual edges)
	std::vector<std::list<Edge> > comps_;
	std::vector<ComponentType> comp_type_;
	int first_virtual_edge_id_, last_virtual_edge_id_;

	// iteration over all possible twists of underlying graph, realize this by:
	// a) "Tutte twists" at virtual edges of "tri-connected components" and
	// b) rearrangements of virtual edges in polygons

	// data members for twist status
	// edge_pos_: position within polygon for each edge (reverse id order)
	std::vector<std::vector<int> > edge_pos_; // order of edges within polygons (for iteration)
	std::vector<std::vector<int> > nodes_; // node ids within polygons
	std::vector<size_t> num_vedges_; // number of virtual edges within polygons
	std::map<int, bool> twist_; // selection of Tutte twists (current virtual edges)
	std::map<int, bool> edge_orientation_; // if direction was changed for edge
	bool are_original_edges_indistinguishable_; // suppress permutations of original edges

	// a) "Tutte twists" at virtual edges:

	/// performs next twist restricted by current virtual edge order in polygons
	/** return false for last **/
	bool set_next_Tutte_twist();

	// b) inside polygon twists:
	// implement via rearrangement of virtual edges: instead of performing
	// a sequence of actual twists of polygon, observe:
	// 1.) any ordering of edge ids is reachable by twisting
	// 2.) we are interested only in positions of virtual edges
	// therefore we iterate over different relevant edge orderings and
	// reconstruct the effect of arriving there by a sequence of twists

	/// sets next permutation of polygon virtual edges
	/** returns false for last **/
	bool set_next_polygon_edge_combination();
	/// updates components corresponding to some chosen polygon edge combination
	void update_polygon_edge_combination(int component,
			const std::vector<int>& new_edge_pos,
			const std::map<int, bool>& initial_flip);
};

} // namespace Reduze

#endif /* TRICONNECTED_H_ */
