/*  yamlconfigurable.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "yamlconfigurable.h"
#include <iomanip>
#include <stdexcept>
#include "functions.h"

using namespace std;

namespace Reduze {

void YAMLSpec::set_keyword(const string& s) {
	keyword_ = s;
}

void YAMLSpec::set_short_description(const string& s) {
	short_description_ = s;
}

void YAMLSpec::set_long_description(const string& s) {
	long_description_ = s;
}

const string& YAMLSpec::keyword() const {
	return keyword_;
}

const string& YAMLSpec::short_description() const {
	return short_description_;
}

const string& YAMLSpec::long_description() const {
	return long_description_;
}

void YAMLSpec::add_option(const std::string& keyword, bool mandatory,
		const std::string& type, const std::string& help) {
	optindex_[keyword] = opts_.size();
	OptHelp h = { keyword, type, help, mandatory };
	opts_.push_back(h);
	if (mandatory)
		mandatory_options_.insert(keyword);
}

void YAMLSpec::add_options(const YAMLSpec& other) {
	vector<OptHelp>::const_iterator o;
	for (o = other.opts_.begin(); o != other.opts_.end(); ++o)
		add_option(o->keyword, o->mandatory, o->type, o->help);
}

bool YAMLSpec::has_option(const std::string& keyword) const {
	return optindex_.find(keyword) != optindex_.end();
}

const set<string>& YAMLSpec::mandatory_options() const {
	return mandatory_options_;
}

void YAMLSpec::print_help(std::ostream& os) const {
	os << "List of options (asterix * denotes mandatory options):\n\n";
	vector<OptHelp>::const_iterator o;
	for (o = opts_.begin(); o != opts_.end(); ++o) {
		os << (o->mandatory ? "* " : "  ") << setw(26) << left << o->keyword
				<< ": " << o->type << "\n" << setw(30) << "";
		string desc = o->help;
		list<string> ds = split_string(desc, 50);
		for (list<string>::const_iterator s = ds.begin(); s != ds.end(); ++s) {
			if (s != ds.begin())
				os << setw(30) << "";
			os << *s << '\n';
		}
	}
}

void YAMLConfigurable::verify_yaml_spec(const YAML::Node& n) {
	if (n.Type() != YAML::NodeType::Map)
		throw runtime_error("expected a map " + position_info(n));
	YAMLSpec d = yaml_spec_link();
	const set<string>& m = d.mandatory_options();
	for (set<string>::const_iterator k = m.begin(); k != m.end(); ++k)
		if (!n.FindValue(*k))
			throw runtime_error(d.keyword() + ": missing mandatory option "
					+ *k);
	for (YAML::Iterator i = n.begin(); i != n.end(); ++i) {
		string k;
		i.first() >> k;
		if (!d.has_option(k))
			throw runtime_error(d.keyword() + ": unknown option " + k + " "
					+ position_info(i.first()));
	}
}

void YAMLConfigurable::print_with_type(YAML::Emitter& os) const {
	os << YAML::BeginMap << YAML::Key << yaml_spec_link().keyword();
	os << YAML::Value;
	print(os);
	os << YAML::EndMap;
}

YAML::Node* YAMLConfigurable::create_yaml_rep() const {
	YAML::Emitter ye;
	print(ye);
	stringstream ss(ye.c_str());
	YAML::Parser p(ss);
	YAML::Node n;
	p.GetNextDocument(n);
	return n.Clone().release();
}

YAMLConfigurable* YAMLFactory::create_object(const std::string& keyword) {
	if (specs_.find(keyword) == specs_.end())
		throw std::runtime_error(std::string("unknown type ") + keyword);
	return specs_[keyword]->create_object();
}

set<string> YAMLFactory::keywords() const {
	set<string> k;
	map<string, IYAMLProxy*>::const_iterator s;
	for (s = specs_.begin(); s != specs_.end(); ++s)
		k.insert(s->first);
	return k;
}

void YAMLFactory::print_help_topics(std::ostream& os) const {
	os << "\nList of help topics:\n\n";
	map<string, IYAMLProxy*>::const_iterator s;
	for (s = specs_.begin(); s != specs_.end(); ++s) {
		os << setw(35) << left << (s->first + ": ");
		os << s->second->yaml_spec().short_description() << '\n';
	}
	os << '\n';
}

void YAMLFactory::print_help(const string& keyword, std::ostream& os) const {
	map<string, IYAMLProxy*>::const_iterator s = specs_.find(keyword);
	if (s == specs_.end()) {
		os << "\nNo help available for unknown keyword \"" << keyword << "\"\n";
		return;
	}
	os << "\n" << keyword << ":\n\n";
	string ld = s->second->yaml_spec().long_description();
	list<string> lds = split_string(ld, 80);
	std::copy(lds.begin(), lds.end(), ostream_iterator<string> (os, "\n"));
	os << "\n";
	s->second->yaml_spec().print_help(os);
	os << "\n";
	YAMLConfigurable* c = s->second->create_object();
	YAML::Emitter ye;
	ye << YAML::BeginMap << YAML::Key << "template with default values"
			<< YAML::Value;
	ye << YAML::BeginSeq;
	c->print_with_type(ye);
	ye << YAML::EndSeq;
	ye << YAML::EndMap;
	os << ye.c_str() << "\n\n";
	//	os << "default values:\n" << ye.c_str() << "\n\n";
	delete c;
}

YAMLAutoConfigurable::YAMLAutoConfigurable(const YAMLAutoConfigurable& o) {
}

YAMLAutoConfigurable& YAMLAutoConfigurable::operator=(
		const YAMLAutoConfigurable& o) {
	clear_auto_io();
	return *this;
}

void YAMLAutoConfigurable::clear_auto_io() {
	list<pair<string, IYAMLReader*> >::const_iterator i;
	for (i = readers_.begin(); i != readers_.end(); ++i)
		delete i->second;
	list<pair<string, IYAMLPrinter*> >::const_iterator j;
	for (j = printers_.begin(); j != printers_.end(); ++j)
		delete j->second;
	readers_.clear();
	printers_.clear();
}

YAMLAutoConfigurable::~YAMLAutoConfigurable() {
	clear_auto_io();
}

void YAMLAutoConfigurable::read_auto_options(const YAML::Node& n) {
	list<pair<string, IYAMLReader*> >::iterator r;
	for (r = readers_.begin(); r != readers_.end(); ++r)
		if (n.FindValue(r->first))
			r->second->read(n[r->first]);
}

void YAMLAutoConfigurable::print_auto_options(YAML::Emitter& os) const {
	using namespace YAML;
	list<pair<string, IYAMLPrinter*> >::const_iterator p;
	for (p = printers_.begin(); p != printers_.end(); ++p) {
		os << Key << p->first << Value;
		p->second->print(os);
	}
}

void YAMLAutoConfigurable::print(YAML::Emitter& os) const {
	os << YAML::BeginMap;
	print_auto_options(os);
	print_manual_options(os);
	os << YAML::EndMap;
}

void YAMLAutoConfigurable::read(const YAML::Node& n) {
	verify_yaml_spec(n);
	read_auto_options(n);
	read_manual_options(n);
	init();
}

} // namespace Reduze

