import { IObservableDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
import { IPoll } from './index';
/**
 * A class that wraps an asynchronous function to poll at a regular interval
 * with exponential increases to the interval length if the poll fails.
 *
 * @typeparam T - The resolved type of the factory's promises.
 * Defaults to `any`.
 *
 * @typeparam U - The rejected type of the factory's promises.
 * Defaults to `any`.
 *
 * @typeparam V - An optional type to extend the phases supported by a poll.
 * Defaults to `standby`, which already exists in the `Phase` type.
 */
export declare class Poll<T = any, U = any, V extends string = 'standby'> implements IObservableDisposable, IPoll<T, U, V> {
    /**
     * Instantiate a new poll with exponential backoff in case of failure.
     *
     * @param options - The poll instantiation options.
     */
    constructor(options: Poll.IOptions<T, U, V>);
    /**
     * The name of the poll.
     */
    readonly name: string;
    /**
     * A signal emitted when the poll is disposed.
     */
    readonly disposed: ISignal<this, void>;
    /**
     * The polling frequency parameters.
     */
    frequency: IPoll.Frequency;
    /**
     * Whether the poll is disposed.
     */
    readonly isDisposed: boolean;
    /**
     * Indicates when the poll switches to standby.
     */
    standby: Poll.Standby | (() => boolean | Poll.Standby);
    /**
     * The poll state, which is the content of the current poll tick.
     */
    readonly state: IPoll.State<T, U, V>;
    /**
     * A promise that resolves when the poll next ticks.
     */
    readonly tick: Promise<this>;
    /**
     * A signal emitted when the poll ticks and fires off a new request.
     */
    readonly ticked: ISignal<this, IPoll.State<T, U, V>>;
    /**
     * Dispose the poll.
     */
    dispose(): void;
    /**
     * Refreshes the poll. Schedules `refreshed` tick if necessary.
     *
     * @returns A promise that resolves after tick is scheduled and never rejects.
     *
     * #### Notes
     * The returned promise resolves after the tick is scheduled, but before
     * the polling action is run. To wait until after the poll action executes,
     * await the `poll.tick` promise: `await poll.refresh(); await poll.tick;`
     */
    refresh(): Promise<void>;
    /**
     * Schedule the next poll tick.
     *
     * @param next - The next poll state data to schedule. Defaults to standby.
     *
     * @param next.cancel - Cancels state transition if function returns `true`.
     *
     * @returns A promise that resolves when the next poll state is active.
     *
     * #### Notes
     * This method is not meant to be invoked by user code typically. It is public
     * to allow poll instances to be composed into classes that schedule ticks.
     */
    schedule(next?: Partial<IPoll.State<T, U, V> & {
        cancel: (last: IPoll.State<T, U, V>) => boolean;
    }>): Promise<void>;
    /**
     * Starts the poll. Schedules `started` tick if necessary.
     *
     * @returns A promise that resolves after tick is scheduled and never rejects.
     */
    start(): Promise<void>;
    /**
     * Stops the poll. Schedules `stopped` tick if necessary.
     *
     * @returns A promise that resolves after tick is scheduled and never rejects.
     */
    stop(): Promise<void>;
    /**
     * Execute a new poll factory promise or stand by if necessary.
     */
    private _execute;
    private _disposed;
    private _factory;
    private _frequency;
    private _standby;
    private _state;
    private _tick;
    private _ticked;
    private _timeout;
}
/**
 * A namespace for `Poll` types, interfaces, and statics.
 */
export declare namespace Poll {
    /**
     * A promise factory that returns an individual poll request.
     *
     * @typeparam T - The resolved type of the factory's promises.
     *
     * @typeparam U - The rejected type of the factory's promises.
     *
     * @typeparam V - The type to extend the phases supported by a poll.
     */
    type Factory<T, U, V extends string> = (state: IPoll.State<T, U, V>) => Promise<T>;
    /**
     * Indicates when the poll switches to standby.
     */
    type Standby = 'never' | 'when-hidden';
    /**
     * Instantiation options for polls.
     *
     * @typeparam T - The resolved type of the factory's promises.
     *
     * @typeparam U - The rejected type of the factory's promises.
     *
     * @typeparam V - The type to extend the phases supported by a poll.
     */
    interface IOptions<T, U, V extends string> {
        /**
         * Whether to begin polling automatically; defaults to `true`.
         */
        auto?: boolean;
        /**
         * A factory function that is passed a poll tick and returns a poll promise.
         */
        factory: Factory<T, U, V>;
        /**
         * The polling frequency parameters.
         */
        frequency?: Partial<IPoll.Frequency>;
        /**
         * The name of the poll.
         * Defaults to `'unknown'`.
         */
        name?: string;
        /**
         * Indicates when the poll switches to standby or a function that returns
         * a boolean or a `Poll.Standby` value to indicate whether to stand by.
         * Defaults to `'when-hidden'`.
         *
         * #### Notes
         * If a function is passed in, for any given context, it should be
         * idempotent and safe to call multiple times. It will be called before each
         * tick execution, but may be called by clients as well.
         */
        standby?: Standby | (() => boolean | Standby);
    }
    /**
     * An interval value that indicates the poll should tick immediately.
     */
    const IMMEDIATE = 0;
    /**
     * Delays are 32-bit integers in many browsers so intervals need to be capped.
     *
     * #### Notes
     * https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/setTimeout#Maximum_delay_value
     */
    const MAX_INTERVAL = 2147483647;
    /**
     * An interval value that indicates the poll should never tick.
     */
    const NEVER: number;
}
//# sourceMappingURL=poll.d.ts.map