//**************************************************************************************************
//                                         CmdGnuCapOPT.cpp                                        *
//                                        ------------------                                       *
// Started     : 2006-09-11                                                                        *
// Last Update : 2020-10-20                                                                        *
// Copyright   : (C) 2006-2020 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "CmdGnuCapOPT.hpp"

//**************************************************************************************************
// Constructor.

CmdGnuCapOPT::CmdGnuCapOPT( void ) : CmdBase( )
{
  bSetDefaults( );
}

//**************************************************************************************************
// Destructor.

CmdGnuCapOPT::~CmdGnuCapOPT( )
{
}

//**************************************************************************************************
// Check that the object attributes are valid.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  CmdGnuCapOPT::bValidate( void )
{
  wxString  os1="Invalid value for ";
  double    df1;
  long      li1;

  CmdBase::bValidate( );

  if( ! CnvtType::bStrToFlt( m_osABSTOL  , &df1 ) ) SetErrMsg( os1 + "ABSTOL."       );
  if( ! CnvtType::bStrToFlt( m_osCHGTOL  , &df1 ) ) SetErrMsg( os1 + "CHGTOL."       );
  if( ! CnvtType::bStrToFlt( m_osDAMPMAX , &df1 ) ) SetErrMsg( os1 + "DAMPMAX."      );
  if( ! CnvtType::bStrToFlt( m_osDAMPMIN , &df1 ) ) SetErrMsg( os1 + "DAMPMIN."      );
  if( ! CnvtType::bStrToFlt( m_osDEFAD   , &df1 ) ) SetErrMsg( os1 + "DEFAD."        );
  if( ! CnvtType::bStrToFlt( m_osDEFAS   , &df1 ) ) SetErrMsg( os1 + "DEFAS."        );
  if( ! CnvtType::bStrToFlt( m_osDEFL    , &df1 ) ) SetErrMsg( os1 + "DEFL."         );
  if( ! CnvtType::bStrToFlt( m_osDEFW    , &df1 ) ) SetErrMsg( os1 + "DEFW."         );
  if( ! CnvtType::bStrToFlt( m_osDTMIN   , &df1 ) ) SetErrMsg( os1 + "DTMIN."        );
  if( ! CnvtType::bStrToFlt( m_osDTRATIO , &df1 ) ) SetErrMsg( os1 + "DTRATIO."      );
  if( ! CnvtType::bStrToFlt( m_osFLOOR   , &df1 ) ) SetErrMsg( os1 + "FLOOR."        );
  if( ! CnvtType::bStrToFlt( m_osGMIN    , &df1 ) ) SetErrMsg( os1 + "GMIN."         );
  if( ! CnvtType::bStrToFlt( m_osRELTOL  , &df1 ) ) SetErrMsg( os1 + "RELTOL."       );
  if( ! CnvtType::bStrToFlt( m_osROUND   , &df1 ) ) SetErrMsg( os1 + "ROUND."        );
  if( ! CnvtType::bStrToFlt( m_osSHORT   , &df1 ) ) SetErrMsg( os1 + "SHORT."        );
  if( ! CnvtType::bStrToFlt( m_osTEMP    , &df1 ) ) SetErrMsg( os1 + "TEMP."         );
  if( ! CnvtType::bStrToFlt( m_osTNOM    , &df1 ) ) SetErrMsg( os1 + "TNOM."         );
  if( ! CnvtType::bStrToFlt( m_osTRREJECT, &df1 ) ) SetErrMsg( os1 + "TRREJECT."     );
  if( ! CnvtType::bStrToFlt( m_osTRSTEPG , &df1 ) ) SetErrMsg( os1 + "TRSTEPGROW."   );
  if( ! CnvtType::bStrToFlt( m_osTRSTEPH , &df1 ) ) SetErrMsg( os1 + "TRSTEPHOLD."   );
  if( ! CnvtType::bStrToFlt( m_osTRSTEPS , &df1 ) ) SetErrMsg( os1 + "TRSTEPSHRINK." );
  if( ! CnvtType::bStrToFlt( m_osTRTOL   , &df1 ) ) SetErrMsg( os1 + "TRTOL."        );
  if( ! CnvtType::bStrToFlt( m_osVFLOOR  , &df1 ) ) SetErrMsg( os1 + "VFLOOR."       );
  if( ! CnvtType::bStrToFlt( m_osVMAX    , &df1 ) ) SetErrMsg( os1 + "VMAX."         );
  if( ! CnvtType::bStrToFlt( m_osVMIN    , &df1 ) ) SetErrMsg( os1 + "VMIN."         );
  if( ! CnvtType::bStrToFlt( m_osVNTOL   , &df1 ) ) SetErrMsg( os1 + "VNTOL."        );

  if( ! CnvtType::bStrToInt( m_osDAMPST  , &li1 ) ) SetErrMsg( os1 + "DAMPST."       );
  if( ! CnvtType::bStrToInt( m_osHARMS   , &li1 ) ) SetErrMsg( os1 + "HARMONICS."    );
  if( ! CnvtType::bStrToInt( m_osITERMIN , &li1 ) ) SetErrMsg( os1 + "ITERMIN."      );
  if( ! CnvtType::bStrToInt( m_osITL1    , &li1 ) ) SetErrMsg( os1 + "ITL1."         );
  if( ! CnvtType::bStrToInt( m_osITL2    , &li1 ) ) SetErrMsg( os1 + "ITL2."         );
  if( ! CnvtType::bStrToInt( m_osITL3    , &li1 ) ) SetErrMsg( os1 + "ITL3."         );
  if( ! CnvtType::bStrToInt( m_osITL4    , &li1 ) ) SetErrMsg( os1 + "ITL4."         );
  if( ! CnvtType::bStrToInt( m_osITL5    , &li1 ) ) SetErrMsg( os1 + "ITL5."         );
  if( ! CnvtType::bStrToInt( m_osITL6    , &li1 ) ) SetErrMsg( os1 + "ITL6."         );
  if( ! CnvtType::bStrToInt( m_osITL7    , &li1 ) ) SetErrMsg( os1 + "ITL7."         );
  if( ! CnvtType::bStrToInt( m_osITL8    , &li1 ) ) SetErrMsg( os1 + "ITL8."         );
  if( ! CnvtType::bStrToInt( m_osNUMDGT  , &li1 ) ) SetErrMsg( os1 + "NUMDGT."       );
  if( ! CnvtType::bStrToInt( m_osRECURS  , &li1 ) ) SetErrMsg( os1 + "RECURS."       );
  if( ! CnvtType::bStrToInt( m_osTRANSITS, &li1 ) ) SetErrMsg( os1 + "TRANSITS."     );

  if( m_osMETHOD.IsEmpty( ) )                       SetErrMsg( os1 + "METHOD."       );
  if( m_osMODE  .IsEmpty( ) )                       SetErrMsg( os1 + "MODE."         );
  if( m_osORDER .IsEmpty( ) )                       SetErrMsg( os1 + "ORDER."        );
  if( m_osPHASE .IsEmpty( ) )                       SetErrMsg( os1 + "PHASE."        );

  return( bIsValid( ) );
}

//**************************************************************************************************
// Set the object attributes to they're default values.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  CmdGnuCapOPT::bSetDefaults( void )
{
  CmdBase::bSetDefaults( );

  m_eSimEngine = eSIMR_GNUCAP;
  m_eCmdType   = eCMD_OPT;

  m_osABSTOL   = GCP_ABSTOL;
  m_osCHGTOL   = GCP_CHGTOL;
  m_osDAMPMAX  = GCP_DAMPMAX;
  m_osDAMPMIN  = GCP_DAMPMIN;
  m_osDAMPST   = GCP_DAMPST;
  m_osDEFAD    = GCP_DEFAD;
  m_osDEFAS    = GCP_DEFAS;
  m_osDEFL     = GCP_DEFL;
  m_osDEFW     = GCP_DEFW;
  m_osDTMIN    = GCP_DTMIN;
  m_osDTRATIO  = GCP_DTRATIO;
  m_osFLOOR    = GCP_FLOOR;
  m_osGMIN     = GCP_GMIN;
  m_osRELTOL   = GCP_RELTOL;
  m_osROUND    = GCP_ROUND;
  m_osSHORT    = GCP_SHORT;
  m_osTEMP     = GCP_TEMP;
  m_osTNOM     = GCP_TNOM;
  m_osTRREJECT = GCP_TRREJECT;
  m_osTRSTEPG  = GCP_TRSTEPG;
  m_osTRSTEPH  = GCP_TRSTEPH;
  m_osTRSTEPS  = GCP_TRSTEPS;
  m_osTRTOL    = GCP_TRTOL;
  m_osVFLOOR   = GCP_VFLOOR;
  m_osVMAX     = GCP_VMAX;
  m_osVMIN     = GCP_VMIN;
  m_osVNTOL    = GCP_VNTOL;

  m_osHARMS    = GCP_HARMS;
  m_osITL1     = GCP_ITL1;
  m_osITL2     = GCP_ITL2;
  m_osITL3     = GCP_ITL3;
  m_osITL4     = GCP_ITL4;
  m_osITL5     = GCP_ITL5;
  m_osITL6     = GCP_ITL6;
  m_osITL7     = GCP_ITL7;
  m_osITL8     = GCP_ITL8;
  m_osITERMIN  = GCP_ITERMIN;
  m_osNUMDGT   = GCP_NUMDGT;
  m_osRECURS   = GCP_RECURS;
  m_osTRANSITS = GCP_TRANSITS;

  m_osMETHOD   = GCP_METHOD;
  m_osMODE     = GCP_MODE;
  m_osORDER    = GCP_ORDER;
  m_osPHASE    = GCP_PHASE;

  m_bBYPASS    = GCP_BYPASS;
  m_bCSTRAY    = GCP_CSTRAY;
  m_bFBBYPASS  = GCP_FBBYPASS;
  m_bINCMODE   = GCP_INCMODE;
  m_bLUBYPASS  = GCP_LUBYPASS;
  m_bOPTS      = GCP_OPTS;
  m_bQUITCONV  = GCP_QUITCONV;
  m_bRSTRAY    = GCP_RSTRAY;
  m_bTRACEL    = GCP_TRACEL;

  return( true );
}

//**************************************************************************************************
// Parse the OPTIONS command string.
//
// Eg.s :
//   .OPTIONS NOOPTS
//   .OPTIONS ABSTOL=1.20p CHGTOL=12.00f DAMPMAX=900.00m DAMPMIN=600.00m DEFAD=11.00f DEFAS=11.00f
//            DEFL=120.00u DEFW=120.00u DTMIN=1.10p DTRATIO=1.10Giga FLOOR=1.10E-21 GMIN=1.20p
//            RELTOL=3.00m ROUND=1.0E-10 SHORT=110.00n TEMP=30.00 TNOM=35.00 TRREJECT=600.00m
//            TRSTEPGROW=3.00 TRSTEPHOLD=100.00 TRSTEPSHRINK=9.00 TRTOL=9.00 VFLOOR=1.10f VMAX=40.00
//            VMIN=-20.00 VNTOL=3.00u DAMPST=1 HARMONICS=10 ITL1=200 ITL2=70 ITL3=10 ITL4=30 ITL5=2
//            ITL6=2 ITL7=2 ITL8=200 ITERMIN=2 NUMDGT=6 RECURS=10 TRANSITS=3 NOBYP NOCSTRAY NOFBBYP
//            NOINC NOLUBYP NOOPTS QUITCONV RSTRAY NOTRACEL METHOD=EULER MODE=ANALOG ORDER=FORWARD
//            PHASE=RADIANS
//
// Return Values :
//   true  - Success
//   false - Failure

bool  CmdGnuCapOPT::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1, os2;
  size_t             sz1;

  // Clear the object attributes
  os1 = (wxString &) *this;
  bSetDefaults( );
  assign( os1 );

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) < 2 ) return( bValidate( ) );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 4 ).Upper( );
  if( os1 != ".OPT" )            return( bValidate( ) );

  // Extract each parameter value
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the field name and the associated value
    os1 = ostk1.GetNextToken( );
    os2 = "";
    if( ( sz1=os1.find( "=" ) ) != wxString::npos )
    {
      os2 = os1.Right( os1.Length( )-sz1-1 );
      os1 = os1.Left( sz1 );
    }

    // Set the object attribute values
    if(      os1.StartsWith( "ABSTOL"   ) ) m_osABSTOL   = os2;
    else if( os1.StartsWith( "CHGTOL"   ) ) m_osCHGTOL   = os2;
    else if( os1.StartsWith( "DAMPMAX"  ) ) m_osDAMPMAX  = os2;
    else if( os1.StartsWith( "DAMPMIN"  ) ) m_osDAMPMIN  = os2;
    else if( os1.StartsWith( "DEFAD"    ) ) m_osDEFAD    = os2;
    else if( os1.StartsWith( "DEFAS"    ) ) m_osDEFAS    = os2;
    else if( os1.StartsWith( "DEFL"     ) ) m_osDEFL     = os2;
    else if( os1.StartsWith( "DEFW"     ) ) m_osDEFW     = os2;
    else if( os1.StartsWith( "DTMIN"    ) ) m_osDTMIN    = os2;
    else if( os1.StartsWith( "DTRATIO"  ) ) m_osDTRATIO  = os2;
    else if( os1.StartsWith( "FLOOR"    ) ) m_osFLOOR    = os2;
    else if( os1.StartsWith( "GMIN"     ) ) m_osGMIN     = os2;
    else if( os1.StartsWith( "RELTOL"   ) ) m_osRELTOL   = os2;
    else if( os1.StartsWith( "ROUND"    ) ) m_osROUND    = os2;
    else if( os1.StartsWith( "SHORT"    ) ) m_osSHORT    = os2;
    else if( os1.StartsWith( "TEMP"     ) ) m_osTEMP     = os2;
    else if( os1.StartsWith( "TNOM"     ) ) m_osTNOM     = os2;
    else if( os1.StartsWith( "TRREJECT" ) ) m_osTRREJECT = os2;
    else if( os1.StartsWith( "TRSTEPGR" ) ) m_osTRSTEPG  = os2;
    else if( os1.StartsWith( "TRSTEPHO" ) ) m_osTRSTEPH  = os2;
    else if( os1.StartsWith( "TRSTEPSH" ) ) m_osTRSTEPS  = os2;
    else if( os1.StartsWith( "TRTOL"    ) ) m_osTRTOL    = os2;
    else if( os1.StartsWith( "VFLOOR"   ) ) m_osVFLOOR   = os2;
    else if( os1.StartsWith( "VMAX"     ) ) m_osVMAX     = os2;
    else if( os1.StartsWith( "VMIN"     ) ) m_osVMIN     = os2;
    else if( os1.StartsWith( "VNTOL"    ) ) m_osVNTOL    = os2;

    else if( os1.StartsWith( "DAMPST"   ) ) m_osDAMPST   = os2;
    else if( os1.StartsWith( "HARMONIC" ) ) m_osHARMS    = os2;
    else if( os1.StartsWith( "ITL1"     ) ) m_osITL1     = os2;
    else if( os1.StartsWith( "ITL2"     ) ) m_osITL2     = os2;
    else if( os1.StartsWith( "ITL3"     ) ) m_osITL3     = os2;
    else if( os1.StartsWith( "ITL4"     ) ) m_osITL4     = os2;
    else if( os1.StartsWith( "ITL5"     ) ) m_osITL5     = os2;
    else if( os1.StartsWith( "ITL6"     ) ) m_osITL6     = os2;
    else if( os1.StartsWith( "ITL7"     ) ) m_osITL7     = os2;
    else if( os1.StartsWith( "ITL8"     ) ) m_osITL8     = os2;
    else if( os1.StartsWith( "ITERMIN"  ) ) m_osITERMIN  = os2;
    else if( os1.StartsWith( "NUMDGT"   ) ) m_osNUMDGT   = os2;
    else if( os1.StartsWith( "RECURS"   ) ) m_osRECURS   = os2;
    else if( os1.StartsWith( "TRANSITS" ) ) m_osTRANSITS = os2;

    else if( os1.StartsWith( "METHOD"   ) ) m_osMETHOD   = os2;
    else if( os1.StartsWith( "MODE"     ) ) m_osMODE     = os2;
    else if( os1.StartsWith( "ORDER"    ) ) m_osORDER    = os2;
    else if( os1.StartsWith( "PHASE"    ) ) m_osPHASE    = os2;

    else if( os1.StartsWith( "BYPASS"   ) ) m_bBYPASS    = true;
    else if( os1.StartsWith( "NOBYP"    ) ) m_bBYPASS    = false;
    else if( os1.StartsWith( "CSTRAY"   ) ) m_bCSTRAY    = true;
    else if( os1.StartsWith( "NOCSTRAY" ) ) m_bCSTRAY    = false;
    else if( os1.StartsWith( "FBBYPASS" ) ) m_bFBBYPASS  = true;
    else if( os1.StartsWith( "NOFBBYP"  ) ) m_bFBBYPASS  = false;
    else if( os1.StartsWith( "INCMODE"  ) ) m_bINCMODE   = true;
    else if( os1.StartsWith( "NOINC"    ) ) m_bINCMODE   = false;
    else if( os1.StartsWith( "LUBYPASS" ) ) m_bLUBYPASS  = true;
    else if( os1.StartsWith( "NOLUBYP"  ) ) m_bLUBYPASS  = false;
    else if( os1.StartsWith( "OPTS"     ) ) m_bOPTS      = true;
    else if( os1.StartsWith( "NOOPTS"   ) ) m_bOPTS      = false;
    else if( os1.StartsWith( "QUITCONV" ) ) m_bQUITCONV  = true;
    else if( os1.StartsWith( "NOQUITCO" ) ) m_bQUITCONV  = false;
    else if( os1.StartsWith( "RSTRAY"   ) ) m_bRSTRAY    = true;
    else if( os1.StartsWith( "NORSTRAY" ) ) m_bRSTRAY    = false;
    else if( os1.StartsWith( "TRACEL"   ) ) m_bTRACEL    = true;
    else if( os1.StartsWith( "NOTRACEL" ) ) m_bTRACEL    = false;
  }

  return( bValidate( ) );
}

//**************************************************************************************************
// Format the OPTIONS command string.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  CmdGnuCapOPT::bFormat( void )
{
  wxString  os1, os2;
  double    df1, df2;
  long      li1, li2;

  os1 = ".OPTIONS";

  // Compare numerical values
  CnvtType::bStrToFlt( m_osABSTOL  , &df1 );
  CnvtType::bStrToFlt( GCP_ABSTOL  , &df2 );
  if( df1 != df2 ) os1 << " ABSTOL="       << m_osABSTOL;

  CnvtType::bStrToFlt( m_osCHGTOL  , &df1 );
  CnvtType::bStrToFlt( GCP_CHGTOL  , &df2 );
  if( df1 != df2 ) os1 << " CHGTOL="       << m_osCHGTOL;

  CnvtType::bStrToFlt( m_osDAMPMAX , &df1 );
  CnvtType::bStrToFlt( GCP_DAMPMAX , &df2 );
  if( df1 != df2 ) os1 << " DAMPMAX="      << m_osDAMPMAX;

  CnvtType::bStrToFlt( m_osDAMPMIN , &df1 );
  CnvtType::bStrToFlt( GCP_DAMPMIN , &df2 );
  if( df1 != df2 ) os1 << " DAMPMIN="      << m_osDAMPMIN;

  CnvtType::bStrToFlt( m_osDEFAD   , &df1 );
  CnvtType::bStrToFlt( GCP_DEFAD   , &df2 );
  if( df1 != df2 ) os1 << " DEFAD="        << m_osDEFAD;

  CnvtType::bStrToFlt( m_osDEFAS   , &df1 );
  CnvtType::bStrToFlt( GCP_DEFAS   , &df2 );
  if( df1 != df2 ) os1 << " DEFAS="        << m_osDEFAS;

  CnvtType::bStrToFlt( m_osDEFL    , &df1 );
  CnvtType::bStrToFlt( GCP_DEFL    , &df2 );
  if( df1 != df2 ) os1 << " DEFL="         << m_osDEFL;

  CnvtType::bStrToFlt( m_osDEFW    , &df1 );
  CnvtType::bStrToFlt( GCP_DEFW    , &df2 );
  if( df1 != df2 ) os1 << " DEFW="         << m_osDEFW;

  CnvtType::bStrToFlt( m_osDTMIN   , &df1 );
  CnvtType::bStrToFlt( GCP_DTMIN   , &df2 );
  if( df1 != df2 ) os1 << " DTMIN="        << m_osDTMIN;

  CnvtType::bStrToFlt( m_osDTRATIO , &df1 );
  CnvtType::bStrToFlt( GCP_DTRATIO , &df2 );
  if( df1 != df2 ) os1 << " DTRATIO="      << m_osDTRATIO;

  CnvtType::bStrToFlt( m_osFLOOR   , &df1 );
  CnvtType::bStrToFlt( GCP_FLOOR   , &df2 );
  if( df1 != df2 ) os1 << " FLOOR="        << m_osFLOOR;

  CnvtType::bStrToFlt( m_osGMIN    , &df1 );
  CnvtType::bStrToFlt( GCP_GMIN    , &df2 );
  if( df1 != df2 ) os1 << " GMIN="         << m_osGMIN;

  CnvtType::bStrToFlt( m_osRELTOL  , &df1 );
  CnvtType::bStrToFlt( GCP_RELTOL  , &df2 );
  if( df1 != df2 ) os1 << " RELTOL="       << m_osRELTOL;

  CnvtType::bStrToFlt( m_osROUND   , &df1 );
  CnvtType::bStrToFlt( GCP_ROUND   , &df2 );
  if( df1 != df2 ) os1 << " ROUND="        << m_osROUND;

  CnvtType::bStrToFlt( m_osSHORT   , &df1 );
  CnvtType::bStrToFlt( GCP_SHORT   , &df2 );
  if( df1 != df2 ) os1 << " SHORT="        << m_osSHORT;

  CnvtType::bStrToFlt( m_osTEMP    , &df1 );
  CnvtType::bStrToFlt( GCP_TEMP    , &df2 );
  if( df1 != df2 ) os1 << " TEMP="         << m_osTEMP;

  CnvtType::bStrToFlt( m_osTNOM    , &df1 );
  CnvtType::bStrToFlt( GCP_TNOM    , &df2 );
  if( df1 != df2 ) os1 << " TNOM="         << m_osTNOM;

  CnvtType::bStrToFlt( m_osTRREJECT, &df1 );
  CnvtType::bStrToFlt( GCP_TRREJECT, &df2 );
  if( df1 != df2 ) os1 << " TRREJECT="     << m_osTRREJECT;

  CnvtType::bStrToFlt( m_osTRSTEPG , &df1 );
  CnvtType::bStrToFlt( GCP_TRSTEPG , &df2 );
  if( df1 != df2 ) os1 << " TRSTEPGROW="   << m_osTRSTEPG;

  CnvtType::bStrToFlt( m_osTRSTEPH , &df1 );
  CnvtType::bStrToFlt( GCP_TRSTEPH , &df2 );
  if( df1 != df2 ) os1 << " TRSTEPHOLD="   << m_osTRSTEPH;

  CnvtType::bStrToFlt( m_osTRSTEPS , &df1 );
  CnvtType::bStrToFlt( GCP_TRSTEPS , &df2 );
  if( df1 != df2 ) os1 << " TRSTEPSHRINK=" << m_osTRSTEPS;

  CnvtType::bStrToFlt( m_osTRTOL   , &df1 );
  CnvtType::bStrToFlt( GCP_TRTOL   , &df2 );
  if( df1 != df2 ) os1 << " TRTOL="        << m_osTRTOL;

  CnvtType::bStrToFlt( m_osVFLOOR  , &df1 );
  CnvtType::bStrToFlt( GCP_VFLOOR  , &df2 );
  if( df1 != df2 ) os1 << " VFLOOR="       << m_osVFLOOR;

  CnvtType::bStrToFlt( m_osVMAX    , &df1 );
  CnvtType::bStrToFlt( GCP_VMAX    , &df2 );
  if( df1 != df2 ) os1 << " VMAX="         << m_osVMAX;

  CnvtType::bStrToFlt( m_osVMIN    , &df1 );
  CnvtType::bStrToFlt( GCP_VMIN    , &df2 );
  if( df1 != df2 ) os1 << " VMIN="         << m_osVMIN;

  CnvtType::bStrToFlt( m_osVNTOL   , &df1 );
  CnvtType::bStrToFlt( GCP_VNTOL   , &df2 );
  if( df1 != df2 ) os1 << " VNTOL="        << m_osVNTOL;

  CnvtType::bStrToInt( m_osDAMPST  , &li1 );
  CnvtType::bStrToInt( GCP_DAMPST  , &li2 );
  if( li1 != li2 ) os1 << " DAMPST="       << m_osDAMPST;

  CnvtType::bStrToInt( m_osHARMS   , &li1 );
  CnvtType::bStrToInt( GCP_HARMS   , &li2 );
  if( li1 != li2 ) os1 << " HARMONICS="    << m_osHARMS;

  CnvtType::bStrToInt( m_osITL1    , &li1 );
  CnvtType::bStrToInt( GCP_ITL1    , &li2 );
  if( li1 != li2 ) os1 << " ITL1="         << m_osITL1;

  CnvtType::bStrToInt( m_osITL2    , &li1 );
  CnvtType::bStrToInt( GCP_ITL2    , &li2 );
  if( li1 != li2 ) os1 << " ITL2="         << m_osITL2;

  CnvtType::bStrToInt( m_osITL3    , &li1 );
  CnvtType::bStrToInt( GCP_ITL3    , &li2 );
  if( li1 != li2 ) os1 << " ITL3="         << m_osITL3;

  CnvtType::bStrToInt( m_osITL4    , &li1 );
  CnvtType::bStrToInt( GCP_ITL4    , &li2 );
  if( li1 != li2 ) os1 << " ITL4="         << m_osITL4;

  CnvtType::bStrToInt( m_osITL5    , &li1 );
  CnvtType::bStrToInt( GCP_ITL5    , &li2 );
  if( li1 != li2 ) os1 << " ITL5="         << m_osITL5;

  CnvtType::bStrToInt( m_osITL6    , &li1 );
  CnvtType::bStrToInt( GCP_ITL6    , &li2 );
  if( li1 != li2 ) os1 << " ITL6="         << m_osITL6;

  CnvtType::bStrToInt( m_osITL7    , &li1 );
  CnvtType::bStrToInt( GCP_ITL7    , &li2 );
  if( li1 != li2 ) os1 << " ITL7="         << m_osITL7;

  CnvtType::bStrToInt( m_osITL8    , &li1 );
  CnvtType::bStrToInt( GCP_ITL8    , &li2 );
  if( li1 != li2 ) os1 << " ITL8="         << m_osITL8;

  CnvtType::bStrToInt( m_osITERMIN , &li1 );
  CnvtType::bStrToInt( GCP_ITERMIN , &li2 );
  if( li1 != li2 ) os1 << " ITERMIN="      << m_osITERMIN;

  CnvtType::bStrToInt( m_osNUMDGT  , &li1 );
  CnvtType::bStrToInt( GCP_NUMDGT  , &li2 );
  if( li1 != li2 ) os1 << " NUMDGT="       << m_osNUMDGT;

  CnvtType::bStrToInt( m_osRECURS  , &li1 );
  CnvtType::bStrToInt( GCP_RECURS  , &li2 );
  if( li1 != li2 ) os1 << " RECURS="       << m_osRECURS;

  CnvtType::bStrToInt( m_osTRANSITS, &li1 );
  CnvtType::bStrToInt( GCP_TRANSITS, &li2 );
  if( li1 != li2 ) os1 << " TRANSITS="     << m_osTRANSITS;

  // Case-insensitive string comparison, return a non-zero value if the strings aren't equal
  os2 = m_osMETHOD.Upper( );
  if( os2.CmpNoCase( GCP_METHOD ) != 0 ) os1 << " METHOD=" << os2;

  os2 = m_osMODE  .Upper( );
  if( os2.CmpNoCase( GCP_MODE   ) != 0 ) os1 << " MODE="   << os2;

  os2 = m_osORDER .Upper( );
  if( os2.CmpNoCase( GCP_ORDER  ) != 0 ) os1 << " ORDER="  << os2;

  os2 = m_osPHASE .Upper( );
  if( os2.CmpNoCase( GCP_PHASE  ) != 0 ) os1 << " PHASE="  << os2;

  // Compare boolean values
  if( m_bBYPASS   != GCP_BYPASS   ) os1 << (m_bBYPASS   ? " BYPASS"   : " NOBYP"   );
  if( m_bCSTRAY   != GCP_CSTRAY   ) os1 << (m_bCSTRAY   ? " CSTRAY"   : " NOCSTRAY");
  if( m_bFBBYPASS != GCP_FBBYPASS ) os1 << (m_bFBBYPASS ? " FBBYPASS" : " NOFBBYP" );
  if( m_bINCMODE  != GCP_INCMODE  ) os1 << (m_bINCMODE  ? " INCMODE"  : " NOINC"   );
  if( m_bLUBYPASS != GCP_LUBYPASS ) os1 << (m_bLUBYPASS ? " LUBYP"    : " NOLUBYP" );
  if( m_bOPTS     != GCP_OPTS     ) os1 << (m_bOPTS     ? " OPTS"     : " NOOPTS"  );
  if( m_bQUITCONV != GCP_QUITCONV ) os1 << (m_bQUITCONV ? " QUITCONV" : " NOQUITCO");
  if( m_bRSTRAY   != GCP_RSTRAY   ) os1 << (m_bRSTRAY   ? " RSTRAY"   : " NORSTRAY");
  if( m_bTRACEL   != GCP_TRACEL   ) os1 << (m_bTRACEL   ? " TRACEL"   : " NOTRACEL");

  assign( os1 );

  return( bValidate( ) );
}

//**************************************************************************************************
// Copy the contents of a CmdNgSpiceOPT object.
//
// Argument List :
//   roCmdOPT - A reference to a CmdNgSpiceOPT object
//
// Return Values :
//   A reference to this object

CmdGnuCapOPT & CmdGnuCapOPT::operator = ( const CmdNgSpiceOPT & roCmdOPT )
{
  (CmdBase &) *this = (CmdBase &) roCmdOPT;

  m_osABSTOL   = roCmdOPT.m_osABSTOL;
  m_osCHGTOL   = roCmdOPT.m_osCHGTOL;
//m_osDAMPMAX  = roCmdOPT.m_osDAMPMAX;
//m_osDAMPMIN  = roCmdOPT.m_osDAMPMIN;
//m_osDAMPST   = roCmdOPT.m_osDAMPST;
  m_osDEFAD    = roCmdOPT.m_osDEFAD;
  m_osDEFAS    = roCmdOPT.m_osDEFAS;
  m_osDEFL     = roCmdOPT.m_osDEFL;
  m_osDEFW     = roCmdOPT.m_osDEFW;
//m_osDTMIN    = roCmdOPT.m_osDTMIN;
//m_osDTRATIO  = roCmdOPT.m_osDTRATIO;
//m_osFLOOR    = roCmdOPT.m_osFLOOR;
  m_osGMIN     = roCmdOPT.m_osGMIN;
  m_osRELTOL   = roCmdOPT.m_osRELTOL;
//m_osROUND    = roCmdOPT.m_osROUND;
//m_osSHORT    = roCmdOPT.m_osSHORT;
  m_osTEMP     = roCmdOPT.m_osTEMP;
  m_osTNOM     = roCmdOPT.m_osTNOM;
//m_osTRREJECT = roCmdOPT.m_osTRREJECT;
//m_osTRSTEPG  = roCmdOPT.m_osTRSTEPG;
//m_osTRSTEPH  = roCmdOPT.m_osTRSTEPH;
//m_osTRSTEPS  = roCmdOPT.m_osTRSTEPS;
  m_osTRTOL    = roCmdOPT.m_osTRTOL;
//m_osVFLOOR   = roCmdOPT.m_osVFLOOR;
//m_osVMAX     = roCmdOPT.m_osVMAX;
//m_osVMIN     = roCmdOPT.m_osVMIN;
  m_osVNTOL    = roCmdOPT.m_osVNTOL;

//m_osHARMS    = roCmdOPT.m_osHARMS;
  m_osITL1     = roCmdOPT.m_osITL1;
  m_osITL2     = roCmdOPT.m_osITL2;
  m_osITL3     = roCmdOPT.m_osITL3;
  m_osITL4     = roCmdOPT.m_osITL4;
  m_osITL5     = roCmdOPT.m_osITL5;
  m_osITL6     = roCmdOPT.m_osITL6;
//m_osITL7     = roCmdOPT.m_osITL7;
//m_osITL8     = roCmdOPT.m_osITL8;
//m_osITERMIN  = roCmdOPT.m_osITERMIN;
//m_osNUMDGT   = roCmdOPT.m_osNUMDGT;
//m_osRECURS   = roCmdOPT.m_osRECURS;
//m_osTRANSITS = roCmdOPT.m_osTRANSITS;

  m_osMETHOD   = roCmdOPT.m_osMETHOD;
//m_osMODE     = roCmdOPT.m_osMODE;
//m_osORDER    = roCmdOPT.m_osORDER;
//m_osPHASE    = roCmdOPT.m_osPHASE;

//m_bBYPASS    = roCmdOPT.m_osBYPASS;
//m_bCSTRAY    = roCmdOPT.m_osCSTRAY;
//m_bFBBYPASS  = roCmdOPT.m_osFBBYPASS;
//m_bINCMODE   = roCmdOPT.m_osINCMODE;
//m_bLUBYPASS  = roCmdOPT.m_osLUBYPASS;
//m_bOPTS      = roCmdOPT.m_osOPTS;
//m_bQUITCONV  = roCmdOPT.m_osQUITCONV;
//m_bRSTRAY    = roCmdOPT.m_osRSTRAY;
//m_bTRACEL    = roCmdOPT.m_osTRACEL;

  bFormat( );

  return( *this );
}

//**************************************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  CmdGnuCapOPT::Print( const wxString & rosPrefix )
{
  CmdBase::Print( rosPrefix + "CmdBase::" );

  std::cout << rosPrefix.mb_str( ) << "m_osABSTOL   : " << m_osABSTOL  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osCHGTOL   : " << m_osCHGTOL  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDAMPMAX  : " << m_osDAMPMAX .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDAMPMIN  : " << m_osDAMPMIN .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDEFAD    : " << m_osDEFAD   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDEFAS    : " << m_osDEFAS   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDEFL     : " << m_osDEFL    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDEFW     : " << m_osDEFW    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDTMIN    : " << m_osDTMIN   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osDTRATIO  : " << m_osDTRATIO .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osFLOOR    : " << m_osFLOOR   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osGMIN     : " << m_osGMIN    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osRELTOL   : " << m_osRELTOL  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osROUND    : " << m_osROUND   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osSHORT    : " << m_osSHORT   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTEMP     : " << m_osTEMP    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTNOM     : " << m_osTNOM    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTRREJECT : " << m_osTRREJECT.mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTRSTEPG  : " << m_osTRSTEPG .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTRSTEPH  : " << m_osTRSTEPH .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTRSTEPS  : " << m_osTRSTEPS .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTRTOL    : " << m_osTRTOL   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osVFLOOR   : " << m_osVFLOOR  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osVMAX     : " << m_osVMAX    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osVMIN     : " << m_osVMIN    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osVNTOL    : " << m_osVNTOL   .mb_str( ) << '\n';

  std::cout << rosPrefix.mb_str( ) << "m_osDAMPST   : " << m_osDAMPST  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osHARMS    : " << m_osHARMS   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL1     : " << m_osITL1    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL2     : " << m_osITL2    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL3     : " << m_osITL3    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL4     : " << m_osITL4    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL5     : " << m_osITL5    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL6     : " << m_osITL6    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL7     : " << m_osITL7    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITL8     : " << m_osITL8    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osITERMIN  : " << m_osITERMIN .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osNUMDGT   : " << m_osNUMDGT  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osRECURS   : " << m_osRECURS  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osTRANSITS : " << m_osTRANSITS.mb_str( ) << '\n';

  std::cout << rosPrefix.mb_str( ) << "m_osMETHOD   : " << m_osMETHOD  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osMODE     : " << m_osMODE    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osORDER    : " << m_osORDER   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_osPHASE    : " << m_osPHASE   .mb_str( ) << '\n';

  std::cout << rosPrefix.mb_str( ) << "m_bBYPASS    : " << (m_bBYPASS   ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bCSTRAY    : " << (m_bCSTRAY   ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bFBBYPASS  : " << (m_bFBBYPASS ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bINCMODE   : " << (m_bINCMODE  ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bLUBYPASS  : " << (m_bLUBYPASS ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bOPTS      : " << (m_bOPTS     ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bQUITCONV  : " << (m_bQUITCONV ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bRSTRAY    : " << (m_bRSTRAY   ? "true" : "false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "m_bTRACEL    : " << (m_bTRACEL   ? "true" : "false") << '\n';
}

//**************************************************************************************************
//                                          Test Utility                                           *
//**************************************************************************************************

#ifdef TEST_CMDGNUCAPOPT

using  namespace  std;

// Function prototypes

void  Usage( char * psAppName );

//**************************************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == "-h" ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else              { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdGnuCapOPT Test Utility"
       << "\n     Version 1.05 (2015-01-03)\n";

  // Create a NG-SPICE OPTIONS command object
  CmdGnuCapOPT  oCmd_OPT;

  // Use the following command example to check the formatter and the parser :
  osCmd << ".OPTIONS ABSTOL=1.20p CHGTOL=12.00f DAMPMAX=900.00m DAMPMIN=600.00m DEFL=120.00u "
        << "DEFW=120.00u DEFAD=11.00f DEFAS=11.00f DTMIN=1.10p DTRATIO=1.10G FLOOR=1.10E-21 "
        << "GMIN=1.20p RELTOL=3.00m ROUND=1.0E-10 SHORT=110.00n TEMP=30.00 TNOM=35.00 "
        << "TRREJECT=600.00m TRSTEPGROW=3.00 TRSTEPHOLD=100.00 TRSTEPSHRINK=9.00 TRTOL=9.00 "
        << "VFLOOR=1.10f VMAX=40.00 VMIN=-20.00 VNTOL=3.00u DAMPST=1 HARMONICS=10 ITL1=200 "
        << "ITL2=70 ITL3=10 ITL4=30 ITL5=2 ITL6=2 ITL7=2 ITL8=200 ITERMIN=2 NUMDGT=6 "
        << "RECURS=10 TRANSITS=3 NOBYP NOCSTRAY NOFBBYP NOINC NOLUBYP NOOPTS QUITCONV RSTRAY "
        << "NOTRACEL METHOD=EULER MODE=ANALOG ORDER=FORWARD PHASE=RADIANS";

  // Set things up for a formatter test
  oCmd_OPT.m_osABSTOL   = "1.20p";
  oCmd_OPT.m_osCHGTOL   = "12.00f";
  oCmd_OPT.m_osDAMPMAX  = "900.00m";
  oCmd_OPT.m_osDAMPMIN  = "600.00m";
  oCmd_OPT.m_osDEFAD    = "11.00f";
  oCmd_OPT.m_osDEFAS    = "11.00f";
  oCmd_OPT.m_osDEFL     = "120.00u";
  oCmd_OPT.m_osDEFW     = "120.00u";
  oCmd_OPT.m_osDTMIN    = "1.10p";
  oCmd_OPT.m_osDTRATIO  = "1.10Giga";
  oCmd_OPT.m_osFLOOR    = "1.10E-21";
  oCmd_OPT.m_osGMIN     = "1.20p";
  oCmd_OPT.m_osRELTOL   = "3.00m";
  oCmd_OPT.m_osROUND    = "1.0E-10";
  oCmd_OPT.m_osSHORT    = "110.00n";
  oCmd_OPT.m_osTEMP     = "30.00";
  oCmd_OPT.m_osTNOM     = "35.00";
  oCmd_OPT.m_osTRREJECT = "600.00m";
  oCmd_OPT.m_osTRSTEPG  = "3.00";
  oCmd_OPT.m_osTRSTEPH  = "100.00";
  oCmd_OPT.m_osTRSTEPS  = "9.00";
  oCmd_OPT.m_osTRTOL    = "9.00";
  oCmd_OPT.m_osVFLOOR   = "1.10f";
  oCmd_OPT.m_osVMAX     = "40.00";
  oCmd_OPT.m_osVMIN     = "-20.00";
  oCmd_OPT.m_osVNTOL    = "3.00u";

  oCmd_OPT.m_osDAMPST   = "1";
  oCmd_OPT.m_osHARMS    = "10";
  oCmd_OPT.m_osITL1     = "200";
  oCmd_OPT.m_osITL2     = "70";
  oCmd_OPT.m_osITL3     = "10";
  oCmd_OPT.m_osITL4     = "30";
  oCmd_OPT.m_osITL5     = "2";
  oCmd_OPT.m_osITL6     = "2";
  oCmd_OPT.m_osITL7     = "2";
  oCmd_OPT.m_osITL8     = "200";
  oCmd_OPT.m_osITERMIN  = "2";
  oCmd_OPT.m_osNUMDGT   = "6";
  oCmd_OPT.m_osRECURS   = "10";
  oCmd_OPT.m_osTRANSITS = "3";

  oCmd_OPT.m_osMETHOD   = "EULER";
  oCmd_OPT.m_osMODE     = "ANALOG";
  oCmd_OPT.m_osORDER    = "FORWARD";
  oCmd_OPT.m_osPHASE    = "RADIANS";

  oCmd_OPT.m_bBYPASS    = false;
  oCmd_OPT.m_bCSTRAY    = false;
  oCmd_OPT.m_bFBBYPASS  = false;
  oCmd_OPT.m_bINCMODE   = false;
  oCmd_OPT.m_bLUBYPASS  = false;
  oCmd_OPT.m_bOPTS      = false;
  oCmd_OPT.m_bQUITCONV  = true;
  oCmd_OPT.m_bRSTRAY    = true;
  oCmd_OPT.m_bTRACEL    = false;

  cout << "\nRun Formatter     : " << ( oCmd_OPT.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_OPT.bSetString( osCmd );
  cout << "\nRun Parser        : " << ( oCmd_OPT.bParse( ) ? "OK" : "FAULT" );
  oCmd_OPT.bFormat( );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//**************************************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_CMDGNUCAPOPT

//**************************************************************************************************
