/*
  File: bootcreator.c

  Copyright (C) 2007
  Marcin 'Morgoth' Kurek <morgoth6@box43.pl>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include <sys/stat.h>

#include "bootcreator.h"
#include "menu.h"
#include "options.h"
#include "list.h"

/*
 * Private
 */

static int create_menu(void);

static bool prepare_creator(void);
static void cleanup_creator(void);

static int identify_section(char *buffer);
static int identify_option(char *buffer);

static bool verifyline(char *line);

struct MenuDescriptorNode *select_menu_type(int version);

void generate_command(char *src, char *dst);

/*
 * Args
 */

extern struct Args args;

/*
 * BootMenu
 */

struct BootMenu menu;

/*
 * create_menu
 */

int create_menu(void)
{
	int res = RETURN_OK;

	/* Sanity ... */
	if(menu.nodes)
	{
		if(menu.def <= menu.nodes)
		{
			char buffer[FILE_BUF_SIZE];
			struct MenuDescriptorNode *src_menu = select_menu_type(menu.version);

			if(src_menu)
			{
				char tmp_buf[255];
				int x = 0;

				while((src_menu[x].id != ID_END) && (res == RETURN_OK))
				{
					switch (src_menu[x].id)
					{
						case ID_ID:
						{
							snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, VERSION_STRING);

							SAVELINE(args.destination_fh, buffer, strlen(buffer));

							/* End */
							break;
						}
						case ID_BOOTNUM:
						{
							snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, menu.nodes);

							SAVELINE(args.destination_fh, buffer, strlen(buffer));

							/* End */
							break;
						}
						case ID_BOOTDEF:
						{
							snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, menu.def);

							SAVELINE(args.destination_fh, buffer, strlen(buffer));

							/* End */
							break;
						}
						case ID_BOOTDELAY:
						{
							snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, menu.timeout * 10);

							SAVELINE(args.destination_fh, buffer, strlen(buffer));

							/* End */
							break;
						}
						case ID_TITLEP:
						{
							if((menu.title[0] != '\n') && (menu.title[0] != '\0'))
							{
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line);

								SAVELINE(args.destination_fh, buffer, strlen(buffer));
							}

							/* End */
							break;
						}
						case ID_TITLE:
						{
							if((menu.title[0] != '\n') && (menu.title[0] != '\0'))
							{
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, menu.title);

								SAVELINE(args.destination_fh, buffer, strlen(buffer));
							}

							/* End */
							break;
						}
						case ID_LABELS:
						{
							int counter = 1;

							struct MenuNode *node = (struct MenuNode *) menu.list.Head;
							struct MenuNode *nextnode = NULL;

							while(nextnode = (struct MenuNode *) (node->node.Succ))
							{
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, counter, node->title);

								SAVELINE(args.destination_fh, buffer, strlen(buffer));

								/* Next */
								node = nextnode;
								++counter;
							}

							/* End */
							break;
						}
						case ID_COMMANDS:
						{
							int counter = 1;

							struct MenuNode *node = (struct MenuNode *) menu.list.Head;
							struct MenuNode *nextnode = NULL;

							while(nextnode = (struct MenuNode *) (node->node.Succ))
							{
								generate_command(node->command, tmp_buf);
								
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, menu.version ? convert_case(counter) : convert_case(counter) + 30, tmp_buf);

								SAVELINE(args.destination_fh, buffer, strlen(buffer));

								/* Next */
								node = nextnode;
								++counter;
							}

							/* End */
							break;
						}
						case ID_ABORT:
						{
							if(menu.abortok)
							{
							    if(!menu.rem_v2)
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, "TTY.HOME_CLR abort ( -- )");
							    else
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, " abort ( -- )");
							}
							else
							{
								snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line, "2drop 0  \\ loop-inc = 0");
							}

							SAVELINE(args.destination_fh, buffer, strlen(buffer));

							/* End */
							break;
						}
						case ID_REM_V2:
						{
						    if(!menu.rem_v2)
						    {
                                            		snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line);

                                                	SAVELINE(args.destination_fh, buffer, strlen(buffer));
						    }

						    /* End */
						    break;
						}
						default:
						{
							snprintf(buffer, FILE_BUF_SIZE, src_menu[x].line);

							SAVELINE(args.destination_fh, buffer, strlen(buffer));

							/* End */
							break;
						}
					}

					/* Next */
					++x;
				}
			}
		}
		else
		{
			/* Wrong default */
			printf("can't apply default boot entry\n");

			res = RETURN_FAILURE;
		}
	}
	else
	{
		/* Can't create menu without any sections */
		printf("can't create menu without any sections\n");

		res = RETURN_FAILURE;
	}

	/* End */
	return (res);

  write_err:

	printf("error writing to destination file\n");

	/* End */
	return (res);
}

/*
 * BootCreator
 */

int bootcreator(void)
{
	int res = RETURN_OK;

	if(prepare_creator())
	{
		int getl_res, stat_line = 0, inside_section = SECTION_UNKNOWN, loaded_sections = 0, loaded_options = 0;
		char buffer[FILE_BUF_SIZE];

		while((res == RETURN_OK) && ((getl_res = get_line(args.source_fh, FILE_BUF_SIZE, buffer, &stat_line, false)) > 0))
		{

  /* Used to fallback to identify section if required */
  identify_section_label:

			/* Ignore comments (Default to lines begining with '#') */
			if(*buffer != PARSE_COMMENT)
			{
				if((inside_section = identify_section(buffer)) != SECTION_UNKNOWN)
				{
					if(inside_section != SECTION_IGNORE)
						VERBOSE(printf("detected section %s at line %d\n", Sections[inside_section], stat_line));

					switch (inside_section)
					{
						case SECTION_IGNORE:
						{
							VERBOSE(printf("ignoring section %s at line %d\n", buffer, stat_line));

							/* End */
							break;
						}
						case SECTION_VERSION:
						{
							/* Read next line */
							INSIDE_READLINE;

							if(!(loaded_sections & BSections[SECTION_VERSION]))
							{
								/* Parse version */
								menu.version = load_i(buffer);

								VERBOSE(printf("output file format version set to %d\n", menu.version));

								/* Sanity check */
								if((menu.version != 0) && (menu.version != 1) && (menu.version != 2))
								{
									/* Only version 0, 1 and 2 are currently supported */
									PARSE_MSG(printf("unsupported output file version %d\n", menu.version));

									res = RETURN_FAILURE;
								}
								else
								{
									/* For version 2 we will use v1 template with only a few changes */
									if(menu.version == 2)
									{
									    menu.version = 1;
									    menu.rem_v2 = true;
									}
									
									/* Mark ... */
									loaded_sections |= BSections[SECTION_VERSION];
								}
							}
							else
							{
								/* Version detected twice ? */
								PARSE_MSG(printf("section %s is already loaded\n", Sections[inside_section]));
							}

							/* End */
							break;
						}
						case SECTION_SETTINGS:
						{
							/* Read next line */
							INSIDE_READLINE;

							if(!(loaded_sections & BSections[SECTION_SETTINGS]))
							{
								struct bcvar var;

								/* Mark ... */
								loaded_sections |= BSections[SECTION_SETTINGS];

								while(buffer[0] != PARSE_SECTION)
								{
									/* Try to interpret the line */
									if(get_var(buffer, &var))
									{
										int option = identify_option(var.var);

										switch (option)
										{
											case OPTION_IGNORE:
											{
												VERBOSE(printf("ignore settings line %s\n", buffer));

												/* End */
												break;
											}
											case OPTION_DEFAULT:
											{
												if(!(loaded_options & BOptions[OPTION_DEFAULT]))
												{
													if((menu.def = load_i(var.val)) != ~0)
													{
														VERBOSE(printf("default set to %d\n", menu.def));

														loaded_options |= BOptions[OPTION_DEFAULT];
													}
													else
													{
														/* Problem when reading timeout value */
														PARSE_MSG(printf("can't read default value\n"));

														res = RETURN_FAILURE;
													}
												}
												else
												{
													/* Timeout detected twice ? */
													PARSE_MSG(printf("option %s is already loaded\n", Options[option]));
												}

												/* End */
												break;
											}
											case OPTION_TIMEOUT:
											{
												if(!(loaded_options & BOptions[OPTION_TIMEOUT]))
												{
													if((menu.timeout = load_i(var.val)) != ~0)
													{
														VERBOSE(printf("timeout set to %d\n", menu.timeout));

														loaded_options |= BOptions[OPTION_TIMEOUT];
													}
													else
													{
														/* Problem when reading timeout value */
														PARSE_MSG(printf("can't read timeout value\n"));

														res = RETURN_FAILURE;
													}
												}
												else
												{
													/* Timeout detected twice ? */
													PARSE_MSG(printf("option %s is already loaded\n", Options[option]));
												}

												/* End */
												break;
											}
											case OPTION_AOK:
											{
												if(!(loaded_options & BOptions[OPTION_AOK]))
												{
													if((menu.abortok = load_b(var.val)) != ~0)
													{
														VERBOSE(printf("abort on key set to %s\n", menu.abortok ? "yes" : "no"));

														loaded_options |= BOptions[OPTION_AOK];
													}
													else
													{
														/* Problem when reading aok value */
														PARSE_MSG(printf("can't read abort on key value\n"));

														res = RETURN_FAILURE;
													}
												}
												else
												{
													/* aok detected twice ? */
													PARSE_MSG(printf("option %s is already loaded\n", Options[option]));
												}


												/* End */
												break;
											}
											case OPTION_UNKNOWN:
											{
												PARSE_MSG(printf("unknown option %s\n", var.var));

												res = RETURN_FAILURE;

												/* End */
												break;
											}
											default:
											{
												PARSE_MSG(printf("internal program failure at identify_option() block '%s'\n", buffer));

												res = RETURN_FAILURE;

												/* End */
												break;
											}
										}
									}
									else
									{
										/* Can't parse line ... */
										PARSE_MSG(printf("error while interpreting settings line '%s'\n", buffer));

										res = RETURN_FAILURE;

										break;
									}

									/* Get next line ... */
									if(res == RETURN_OK)
									{
										INSIDE_READLINE;
									}
									else
										break;
								}

								/* Ugly ! */
								if(res == RETURN_OK)
									goto identify_section_label;
							}
							else
							{
								/* Title detected twice ? */
								PARSE_MSG(printf("section %s is already loaded\n", Sections[inside_section]));
							}

							/* End */
							break;
						}
						case SECTION_TITLE:
						{
							/* Read next line */
							INSIDE_PURELINE;

							if(!(loaded_sections & BSections[SECTION_TITLE]))
							{
								/* Verify */
								VERIFYLINE(buffer);

								/* Copy */
								strncpy(menu.title, buffer, FILE_BUF_SIZE);

								VERBOSE(printf("menu title set to %s\n", menu.title));

								/* Mark ... */
								loaded_sections |= BSections[SECTION_TITLE];
							}
							else
							{
								/* Title detected twice ? */
								PARSE_MSG(printf("section %s is already loaded\n", Sections[inside_section]));
							}

							/* End */
							break;
						}
						case SECTION_SECTION:
						{

							if(menu.nodes < SECTION_LIMIT)
							{
								struct MenuNode *node;

								/* Read next line */
								INSIDE_PURELINE;

								/* OK, in theory we have a title. Create a node */
								if(node = malloc(sizeof(struct MenuNode)))
								{
									/* Verify */
									VERIFYLINE(buffer);

									/* Copy title */
									strncpy(node->title, buffer, FILE_BUF_SIZE);

									/* Get the command */
									INSIDE_READLINE;

									/* Verify */
									VERIFYLINE(buffer);

									/* Copy command */
									strncpy(node->command, buffer, FILE_BUF_SIZE);

									/* Add to list */
									add_node(&menu.list, &node->node);

									/* Counter */
									++menu.nodes;

									/* Msg ... */
									VERBOSE(printf("section - %s [%s]\n", node->title, node->command));
								}
								else
								{
									/* Out of memory */
									PARSE_MSG(printf("not enough memory for node [%d bytes]\n", sizeof(struct MenuNode)));

									res = RETURN_FAILURE;
								}
							}
							else
							{
								/* Too many sections ... */
								PARSE_MSG(printf("too many sections ... section ignored\n"));

								INSIDE_PURELINE;	/* Inore title */
								INSIDE_READLINE;	/* Ignore command */
							}

							/* End */
							break;
						}
						default:
						{
							/* Internal error ... */
							PARSE_MSG(printf("internal program failure at identify_section() block '%s'\n", buffer));

							res = RETURN_FAILURE;

							/* End */
							break;
						}
					}
				}
				else
				{
					/* Unknown Section */
					PARSE_MSG(printf("unknown section '%s'\n", buffer));

					res = RETURN_FAILURE;
				}
			}
		}
	}
	else
	{
		/* Failure */
		res = RETURN_FAILURE;
	}

	/* Create a result file ... */
	if(res == RETURN_OK)
		res = create_menu();

	/* Cleanup */
	cleanup_creator();

	/* End */
	return (res);

  unexpected_eof:

	printf("unexpected end of file\n");

	/* Cleanup */
	cleanup_creator();

	/* End */
	return (res);
}

/*
 * Private
 */

static bool prepare_creator(void)
{
	bool res = true;

	if((args.source_fh = open(args.source_file, O_RDONLY, 0)) != -1)
		args.destination_fh = open(args.destination_file, O_CREAT | O_WRONLY | O_TRUNC, DEF_PERMS);

	/* Sanity */
	if((args.source_fh == -1) || (args.destination_fh == -1))
	{
		/* Canrt open|create files */
		printf("can't %s file %s\n", args.source_fh == -1 ? "open" : "create", args.source_fh == -1 ? args.source_file : args.destination_file);

		res = false;
	}

	/* Fill BootMenu with default arguments */
	if(res)
	{
		menu.version = DEF_VERSION;
		menu.timeout = DEF_TIMEOUT;
		menu.abortok = DEF_ABORTOK;
		menu.def = DEF_DEFAULT;
		menu.rem_v2 = DEF_REM_V2;

		strcpy(menu.title, DEF_TITLE);

		new_list(&menu.list);
	}

	/* End */
	return (res);
}

static void cleanup_creator(void)
{
	/* Free all allocated nodes */
	{
		struct MenuNode *node = (struct MenuNode *) menu.list.Head;
		struct MenuNode *nextnode = NULL;

		while(nextnode = (struct MenuNode *) (node->node.Succ))
		{
			/* Free */
			free(node);

			/* Next */
			node = nextnode;

			/* Counter */
			--menu.nodes;
		}
	}

	/* Destination file */
	if(args.destination_fh)
		close(args.destination_fh);

	/* Source file */
	if(args.source_fh)
		close(args.source_fh);

	/* End */
	return;
}

static int identify_section(char *buffer_a)
{
	int res = SECTION_UNKNOWN;
	char buffer[FILE_BUF_SIZE];

	/* Copy ... */
	strncpy(buffer, buffer_a, FILE_BUF_SIZE);

	/* Unify the section name */
	to_upper(buffer);

	if(buffer[0] == PARSE_SECTION)
	{
		/* Identify .... */
		if(strcmp(buffer, Sections[SECTION_VERSION]) == 0)
			res = SECTION_VERSION;
		else if(strcmp(buffer, Sections[SECTION_SETTINGS]) == 0)
			res = SECTION_SETTINGS;
		else if(strcmp(buffer, Sections[SECTION_TITLE]) == 0)
			res = SECTION_TITLE;
		else if(strcmp(buffer, Sections[SECTION_SECTION]) == 0)
			res = SECTION_SECTION;
	}

	/* End */
	return (res);
}

static int identify_option(char *buffer_a)
{
	int res = OPTION_UNKNOWN;
	char buffer[FILE_BUF_SIZE];

	/* Copy ... */
	strncpy(buffer, buffer_a, FILE_BUF_SIZE);

	/* Unify the option name */
	to_upper(buffer);

	/* Identify ... */
	if(buffer[0] != PARSE_COMMENT)
	{
		/* Identify ... */
		if(strncmp(buffer, Options[OPTION_TIMEOUT], strlen(Options[OPTION_TIMEOUT])) == 0)
			res = OPTION_TIMEOUT;
		else if(strncmp(buffer, Options[OPTION_AOK], strlen(Options[OPTION_AOK])) == 0)
			res = OPTION_AOK;
		else if(strncmp(buffer, Options[OPTION_DEFAULT], strlen(Options[OPTION_DEFAULT])) == 0)
			res = OPTION_DEFAULT;
	}
	else
		res = OPTION_IGNORE;

	/* End */
	return (res);

}

static bool verifyline(char *line)
{
	bool res = true;

	if(line)
	{
		/* Seek for " */
		if(strchr(line, '"'))
		{
			char buffer[FILE_BUF_SIZE], *x;

			strcpy(buffer, line);
			x = buffer;

			do
			{
				if(*x == '"')
				{
					char escape[] = "\"(22)", *y;
					y = escape;

					while(*y)
					{
						*line = *y;
						++line;
						++y;
					}
				}
				else
				{
					*line = *x;
					++line;
				}
			}
			while(*x++);

			*x = '\0';
		}
	}

	/* End */
	return (res);
}


struct MenuDescriptorNode *select_menu_type(int version)
{
	struct MenuDescriptorNode *res = NULL;

	if(version == 0)
		res = (void *) Menu0;
	else if(version == 1)
		res = (void *) Menu1;

	/* End */
	return (res);
}

void generate_command(char *src, char *dst)
{
    if(!strcmp(src, "-"))
	snprintf(dst, 255, "abort");
    else if(menu.rem_v2 || (menu.version == 0))
	snprintf(dst, 255, "\" boot %s \" eval", src); /* v0 / v2 */
    else
	snprintf(dst, 255, "\" %s \"", src); /* v1 */

    /* End */
    return;
}
