/** @file cdstatus_cddb.c Implementation file for all routines pertaining to
  * cddb.  This file performs the work of calculating the disc ID, communicating
  * with the cddb server, and populating the track information.
  */
#include "gimme_config_h.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "cdstatus_cddb.h"
#include "make_pretty_string.h"
#include "cddb_perform_lookup.h"
#include "cddb_hearken.h"
#include "cddb_calc_discid.h"
#include "cddb_connect_to_server.h"
#include "cddb_handshake.h"
#include "cddb_handle_multiple_matches.h"
#include "cddb_fetch_and_parse_results.h"
#include "cdstatus_output.h"

#ifndef DIE_CLEAN
#define DIE_CLEAN if(buffer) { free(buffer); } if(scratchbuffer) { free(scratchbuffer); } exit(EXIT_FAILURE);
#endif


/** The system-provided error return code for system calls */
extern int errno;

/** read and handle server response. 
  * This is a small convenience function to handle the socket read with some
  * error checking in place.
  * @param[in] client_sock the socket descriptor to read on
  * @param[out] buffer the buffer to which to copy the server response
  * @return the server response code (from the cddb protocol), -1 on error
  */

void cddb_query(unsigned int tracks, const cd_toc_info * cdtocinfo, int noMangle, const char *cddb_site, unsigned int first_inexact)
{
	/* disc related */
	unsigned int discid;
	char category[64];
	unsigned int index;
	
	/* network socket */
	int client_sock;

	/* buffer and status parsing */
	int result;
	char *scratchbuffer;
	void *buffer;

	memset(category, 0, 64);
	discid = calcDiscId(tracks, cdtocinfo);

	for (index = 0; index < 99; ++index)
	{
		trackinfo[index].artist[0] = '\0';
		trackinfo[index].title[0] = '\0';
	}
	buffer = calloc((size_t) 1, (size_t) BUFFSIZE);
	if (buffer == NULL)
	{
		conditional_puts(CRITICAL, "Unable to allocate buffer for connection with cddb site. Exiting\n");
		exit(EXIT_FAILURE);
	}
	else
	{
		conditional_puts(VERBOSE_DEBUG, "Buffer allocated successfully.");
	}

	scratchbuffer = calloc(1, SCRATCHBUFFER_SIZE);
	if (scratchbuffer == NULL)
	{
		conditional_puts(WARNING, "Unable to allocate scratch buffer. Lookup aborted.");
		free(buffer);
		return;
	}
	else
	{
		conditional_puts(VERBOSE_DEBUG, "scratch buffer allocated successfully.");
	}

	client_sock=cddbConnectToServer(cddb_site);

	if(client_sock==0) { DIE_CLEAN }
	if(cddbHandshake(client_sock, buffer)==0) { DIE_CLEAN }
	result=cddbPerformLookup(client_sock, buffer, scratchbuffer, discid, tracks, cdtocinfo, category, first_inexact);
	if(result==0) { DIE_CLEAN }

	if ((result == 211) || (result == 210))
	{
		if(handleMultipleMatches(buffer, first_inexact, category, &discid)==0)
		{
			DIE_CLEAN
		}
	}
	if(fetchAndParseResults(buffer, category, &discid, client_sock)==0)
	{
		DIE_CLEAN
	}
	free(scratchbuffer);
	free(buffer);

	if(album_name==NULL)
	{
		conditional_puts(CRITICAL, "Error, got null album_name, exiting.");
		exit(EXIT_FAILURE);
	}
	/* beautify names to make them easy on the filesystem */
	if (noMangle==0)
	{
		makePrettyString(album_name);
		makePrettyString(artist_name);
		for (index = 1; index <= tracks; ++index)
		{
			makePrettyString(trackinfo[index].artist);
			makePrettyString(trackinfo[index].title);
		}
	}
	return;
}

