/*****************************************************************************
 * JustePort.cs: JustePort 0.2
 *****************************************************************************
 * Copyright (C) 2005 Jon Lech Johansen <jon@nanocrew.net>
 *
 * BitBuffer and EncodeALAC was contributed by Anonymous
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

using System;
using System.IO;
using System.Text;

class BitBuffer
{
    byte [] buffer;

    byte [] masks =
    {
        0x01, 0x03, 0x07, 0x0F,
        0x1F, 0x3F, 0x7F, 0xFF
    };

    int bitOffset = 0;
    int byteOffset = 0;

    public byte [] Buffer
    {
        get
        {
            return this.buffer;
        }
    }

    public BitBuffer( int length )
    {
        buffer = new byte[ length ];
    }

    public void WriteBits( int data, int numbits )
    {
        if( bitOffset != 0 && bitOffset + numbits > 8 )
        {
            int numwritebits = 8 - bitOffset;
            byte bitstowrite = (byte)((data >> (numbits - numwritebits)) <<
                               (8 - bitOffset - numwritebits));
            buffer[ byteOffset ] |= bitstowrite;
            numbits -= numwritebits;
            bitOffset = 0;
            byteOffset++;
        }

        while( numbits >= 8 )
        {
            byte bitstowrite = (byte)((data >> (numbits - 8)) & 0xFF);
            buffer[ byteOffset ] |= bitstowrite;
            numbits -= 8;
            bitOffset = 0;
            byteOffset++;
        }

        if( numbits > 0 )
        {
            byte bitstowrite = (byte)((data & masks[ numbits ]) <<
                               (8 - bitOffset - numbits));
            buffer[ byteOffset ] |= bitstowrite;
            bitOffset += numbits;
            if( bitOffset == 8 )
            {
                byteOffset++;
                bitOffset = 0;
            }
        }
    }
}

public class JustePort
{
    private static int [] GetChunkSamplesTable( STSC stsc, int ChunkCount )
    {
        int [] Table = new int[ ChunkCount ];
        int Last = ChunkCount;
        int Index = stsc.FirstChunk.Length;

        while( Index-- > 0 )
        {
            for( int Chunk = stsc.FirstChunk[ Index ] - 1;
                 Chunk < Last; Chunk++ )
            {
                Table[ Chunk ] = stsc.SamplesPerChunk[ Index ];
            }

            Last = stsc.FirstChunk[ Index ] - 1;
        }

        return Table;
    }

    private static byte [] EncodeALAC( byte [] buffer )
    {
        // Frame size is set as 4096 samples, stereo
        BitBuffer bitbuf = new BitBuffer( (4096 * 2 * 2) + 3 );

        bitbuf.WriteBits(1, 3);  // channels -- 0 mono, 1 stereo
        bitbuf.WriteBits(0, 4);  // unknown
        bitbuf.WriteBits(0, 12); // unknown
        bitbuf.WriteBits(0, 1);  // 'has size' flag
        bitbuf.WriteBits(0, 2);  // unknown
        bitbuf.WriteBits(1, 1);  // 'no compression' flag

        for( int i = 0; i < buffer.Length; i += 2 )
        {
            // endian swap 16 bit samples
            bitbuf.WriteBits(buffer[i+1], 8);
            bitbuf.WriteBits(buffer[i], 8);
        }

        return bitbuf.Buffer;
    }

    public static void Main( string [] Args )
    {
        FileStream fs = null;
        RAOPClient at = null;

        if( Args.Length < 2 || Args.Length > 3 )
        {
            string u = "Usage: JustePort input ipaddress [volume]" +
                       Environment.NewLine +
                       "(volume range is -144 to 0, default: " +
                       RAOPClient.VOLUME_DEF.ToString( "N" ) + ")";
            Console.WriteLine( u );
            return;
        }

        try
        {
            at = new RAOPClient( Args[ 1 ] );
            if( Args.Length == 3 )
                at.Volume = Convert.ToDouble( Args[ 2 ] );
            at.Connect();
        }
        catch( Exception e )
        {
            Console.WriteLine( "Connect failed: {0}", e.Message );
            return;
        }

        string s = String.Format( "JackStatus: {0}{1}JackType: {2}{3}",
            at.JackStatus == RAOPClient.JACK_STATUS_CONNECTED ?
            "connected" : "disconnected", Environment.NewLine,
            at.JackType == RAOPClient.JACK_TYPE_DIGITAL ?
            "digital" : "analog", Environment.NewLine );
        Console.WriteLine( s );

        try
        {
            if( Args[ 0 ].Equals( "-" ) )
            {
                BinaryReader br;

                br = new BinaryReader( Console.OpenStandardInput() );

                while( true )
                {
                    byte [] raw = br.ReadBytes( 4096 * 2 * 2 );

                    if( raw.Length == 0 )
                        break;

                    byte [] alac = EncodeALAC( raw );
                    at.SendSample( alac, 0, alac.Length );
                }
            }
            else
            {
                MP4Box box;

                fs = new FileStream( Args[ 0 ], FileMode.Open,
                                     FileAccess.Read );
                MP4Root root = new MP4Root();
                root.Parse( fs );

                if( root[ MP4Types.ALAC ] == null )
                    throw new Exception( "File is not an Apple Lossless file" );

                box = root[ MP4Types.STSZ ];
                if( box == null )
                    throw new Exception( "STSZ box is not present" );
                int [] SampleSizeTable = ((STSZ)box.Data).Table;

                box = root[ MP4Types.STCO ];
                if( box == null )
                    throw new Exception( "STCO box is not present" );
                int [] ChunkOffsetTable = ((STCO)box.Data).Table;

                box = root[ MP4Types.STSC ];
                if( box == null )
                    throw new Exception( "STSC box is not present" );
                int [] ChunkSamples = GetChunkSamplesTable( (STSC)box.Data,
                    ChunkOffsetTable.Length );

                box = root[ MP4Types.MDAT,
                    Convert.ToUInt64( ChunkOffsetTable[ 0 ] ) ];
                if( box == null )
                    throw new Exception( "MDAT box is not present" );

                int PosMDAT = Convert.ToInt32( box.Pos + 8 );

                for( int Chunk = 0, Sample = 0;
                     Chunk < ChunkOffsetTable.Length; Chunk++ )
                {
                    int Pos = ChunkOffsetTable[ Chunk ];

                    for( int i = 0; i < ChunkSamples[ Chunk ]; i++ )
                    {
                        at.SendSample( (byte [])box.Data, Pos - PosMDAT,
                                       SampleSizeTable[ Sample ] );

                        Pos += SampleSizeTable[ Sample++ ];
                    }
                }
            }
        }
        catch( Exception e )
        {
            Console.WriteLine( "Error: {0}", e.Message );
        }
        finally
        {
            if( fs != null )
                fs.Close();
        }

        at.Disconnect();
    }
}
