//=============================================
//  wmxkbg - WM XKB groups
// --------------------------------------------
//  Michael Glickman <wmalms@yahooo.com>
//  03-Oct-03
//=============================================
#include "wmxkb.h"

#if HAVE_XINPUT
#include <X11/extensions/XInput.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

extern Window wndRoot;

extern Display *disp;

char XkbRunVersion[16];

#if HAVE_XINPUT
int  KbdDevOrder;
#endif

extern FunctionKey  GroupSetKeys[];
extern FunctionKey  GroupLatchKeys[];
extern FunctionKey  GroupSwitchKeys[2];

extern int GroupTitleSource;
extern int  GroupCodeCount;
extern int  SetKeyCount, LatchKeyCount, SwitchKeyCount;
extern Bool FlexyGroups;

int CurrentGroupXKBNo, CurrentGroupResNo, GroupCount;
static int BaseEventCode, BaseErrorCode;
static Bool HasFunctionKeys;

char *SymbolNames[XkbNumKbdGroups];
char *GroupNames[XkbNumKbdGroups];
extern char *CustomNames[];
extern char *GroupScripts[];
extern char *GroupCodes[];


static int deviceID;
#if HAVE_XINPUT
static XDevice *xdev;
static int key_press_type;
static XEventClass key_event_class;
#endif

//=============================================================================
Bool getXKBVersion(void)
{
	int major, minor, oppcode;
	Bool status; 

	major = XkbMajorVersion;
	minor = XkbMinorVersion; 

	XkbIgnoreExtension(False); 
	status = XkbQueryExtension(disp, &oppcode, &BaseEventCode, &BaseErrorCode, &major, &minor);

	if (status == False) {
		fprintf(stderr, "XKB open error, oppcode %d (disabled ? inconsistent ?)\n", oppcode);
		strncpy (XkbRunVersion, "Unknown", sizeof(XkbRunVersion));
	} else
#if HAVE_SNPRINTF
		snprintf (XkbRunVersion, sizeof(XkbRunVersion), "%d.%d", major, minor);
#else
		sprintf (XkbRunVersion, "%d.%d", major, minor);
#endif		
	
	return status;
}

//=============================================================================
static int GroupLookup(int sourceValue, char *fromTexts[], char *toTexts[], int count)
{

  if (FlexyGroups) {
    const char *sourceText = fromTexts[sourceValue];
  
	if (sourceText != NULL) {
  	  const char *targetText;
	  int i;
  
  	  for (i=0; i<count; i++) {
		targetText = toTexts[i];
		if (strcasecmp(sourceText, targetText) == 0) {
		  sourceValue = i;
		  break;
		}	  
  	  }
	}
  }
  
  return sourceValue;  

}

static int GroupNoResToXKB(int GroupResNo)
{
  return GroupLookup(GroupResNo, GroupCodes, SymbolNames, GroupCount);
}

static int GroupXKBToRes(int GroupXKBNo)
{
  return GroupLookup(GroupXKBNo, SymbolNames, GroupCodes, GroupCodeCount);
}


static char * getSymbolNameByResNo(int groupResNo)
{
	return SymbolNames[GroupNoResToXKB(groupResNo)];
}

static char * getGroupNameByResNo(int groupResNo)
{
	return GroupNames[GroupNoResToXKB(groupResNo)];
}

//=============================================================================
#if HAVE_XINPUT
static Bool examineKeyboardDevice(void)
{
	XDeviceInfo *devInfo;
	int ndevs, i, j;
	

	xdev = NULL;

    if (XQueryExtension(disp, "XInputExtension", &ndevs, &i, &j) == False) {
	  fprintf(stderr, "XInput Extension is not present\n");
	  return False;
	}  

	devInfo = XListInputDevices(disp, &ndevs);
	j = KbdDevOrder;
	
	for (i=0; i<ndevs; i++) {

		// Select only keyboard devices
		if (devInfo[i].inputclassinfo->class == 0)	{
			if (j<=0) {		
				deviceID = devInfo[i].id;
				j =  devInfo[i].use;
				break;
			}
			j--;	
	  	}
	}

	XFreeDeviceList(devInfo);

	if (i >= ndevs) {
	  fprintf (stderr, "Keyboard device '%i' is not present!\n", KbdDevOrder);
	  return False;
	
	}	

	// We can't apply XOpenDevice to core keyboard		
	// devinfo[i].use must not be IsXKeyboard
	if (j != IsXKeyboard)  { // == IsExtensionDevice) {
		xdev = XOpenDevice(disp, deviceID);
		if (xdev != NULL) {
		    DeviceKeyPress(xdev, key_press_type, key_event_class);
		    XSelectExtensionEvent(disp, wndRoot, &key_event_class, 1);
		}	
	}
	
	return True;
}

#endif

//=============================================================================
static void initGroupNames() 
{
	int i;
	for (i=0; i< GroupCount; i++) {
		GroupNames[i] = NULL;	
		SymbolNames[i] = NULL;	
	}		
}

static void deinitGroupNames() 
{
	int i;
	for (i=0; i< GroupCount; i++) {
		if (GroupNames[i] != NULL) {
		  free(GroupNames[i]);
		  GroupNames[i] = NULL;
		}  
		if (SymbolNames[i] != NULL) {
		  free(SymbolNames[i]);
		  SymbolNames[i] = NULL;
		}  
	}		
}

static void collectGroupNames(const XkbDescRec *kbd_desc_ptr) 
{
	int i;
	const Atom *groupSource = kbd_desc_ptr->names->groups;
	Atom curGroupAtom;
	char *ptr;

	for (i=0; i< GroupCount; i++) {
		if ((curGroupAtom = groupSource[i]) != None) {
			GroupNames[i] = ptr = XGetAtomName(disp, curGroupAtom);
			if (ptr != NULL && (ptr=strchr(ptr, '(')) != NULL)
				*ptr = '\0';
		}
	}

}


static void checkGroupNames(void) 
{
	int i;
	
	int count = (GroupTitleSource == 2) ? GroupCodeCount : GroupCount;



	for (i=0; i< count; i++) {
	  if (FlexyGroups && GroupCodes[i] == NULL) {
		fprintf(stderr, "\nCode is not specified for Group %i !\n", i+1);
		fprintf(stderr, "Flexy mode is ignored\n");
		FlexyGroups = False;
	  }

	  switch(GroupTitleSource)
	  {
		case 1:	
			// Group name
		    if (GroupNames[i] == NULL) {
				char *name = getSymbolNameByResNo(i);
				if (name == NULL) name = "U/A";
				fprintf(stderr, "\nGroup Name %i is undefined, set to '%s' !\n", i+1, name);
				GroupNames[i] = strdup(name);		
			}
			break;	
		
		case 2:
			// Gustom name
		    if (CustomNames[i] == NULL) {
				const char *name = getSymbolNameByResNo(i);
				if (name == NULL) name = getGroupNameByResNo(i);
				if (name == NULL) name = "U/A";
				fprintf(stderr, "\nCustom Name %i is undefined, set to '%s' !\n", i+1, name);
				CustomNames[i] = strdup(name);		

			}	
			break;
			
		default:
			// Symbolic name (0), No title source but can be used for images (3)
		    if (SymbolNames[i] == NULL) {
				fprintf(stderr, "\nGroup Symbol %i is undefined, set to 'U/A' !\n", i+1);
				SymbolNames[i] = strdup("U/A");		
			}
			break;
			
	  }
	}
}

/*
static Bool lookupAppendedName(const char *text)
{
  const char *appendices[] = 
  {  "phonetic", "qwerty", "yawery", "latin", "enhanced", NULL} ;
  
  const char **appendPtr = appendices, *appendix;

  
  while ((appendix = *appendPtr++) != NULL) {
	if (memcmp(appendix, text, strlen(appendix)) == 0) return True;
  }
  
  return False;	

}


static void collectSymbolNames(const XkbDescRec *kbd_desc_ptr) 
{
	Atom symNameAtom = kbd_desc_ptr->names->symbols;
	char *symName;
	if (symNameAtom != None &&
	  (symName = XGetAtomName(disp, symNameAtom)) != NULL) {

	  unsigned char *ptr, *ptr1, *ptr2;
	  int i, j;
	  
	  // Get rid of all (..)
	  ptr = symName;
	  
	  for (;;) {
		ptr1 = strchr(ptr, '(');
		if (ptr1 == NULL) break;
		ptr = strchr(ptr1+1, ')');
		if (ptr == NULL) *ptr1 = '\0';
		else strcpy(ptr1, ptr+1);
		ptr = ptr1; 
	  }
	  	  
	  // Get rid of all that follows _group	  	  
	  ptr1 = strstr(symName, "_group");
	  if (ptr1 != NULL) *ptr1 = '\0';
	  
	  // Count number of '_'-s
	  ptr = symName;
	  j = 0;
	  for(;;) {
		// We accept only lower case letters, unless the first
		if (j==0 || (islower(*ptr) && !lookupAppendedName(ptr))) j++;
		ptr=strchr(ptr, '_');
		if (ptr == NULL) break;
		ptr++;
	  }
	  
	  j -= GroupCount;	  
	  i = 0;
	  // This will cater for treally odd cases
	  while (j<0) {
		SymbolNames[i++] = strdup("us");
		j++;
	  }
	  
	  // Now process the names!
	  ptr = symName;
	  while (i<GroupCount) {
		ptr1=strchr(ptr, '_');

		if (islower(*ptr) && !lookupAppendedName(ptr)) {
		  if (j>0) j--;
		  else {
			if (ptr1 != NULL) {
				// Cater for 5-char code
				ptr2 = ptr1+1;
				if (isupper(*ptr2)) // || lookupAppendedName(ptr2)) 
					ptr1=strchr(ptr2, '_');
				// Cut off current code	
				*ptr1 =  '\0';
			}					

			// Remove path from code
			ptr2 = strrchr(ptr, '/');
			if (ptr2 != NULL) ptr = ptr2+1;		  
			SymbolNames[i++] = strdup(ptr);
		  }
		}

		if (ptr1 == NULL) break;
		ptr = ptr1+1;
	  }
			
	  XFree(symName);
	  
	  if (GroupCount == 1 && GroupNames[0] == NULL &&
	      strcmp(SymbolNames[0], "jp") == 0)  {
		  GroupCount = 2;
		  SymbolNames[1] = SymbolNames[0];
		  SymbolNames[0] = strdup("us");
		  GroupNames[0] = strdup("US/ASCII");
		  GroupNames[1] = strdup("Japanese");
	  }	  
	}

}	
*/
static void collectSymbolNames(const XkbDescRec *kbd_desc_ptr) 
{
	Atom symNameAtom = kbd_desc_ptr->names->symbols;
	char *symName;
	char *ptr, *ptr1;
	int  count;
	
	if (symNameAtom == None ||
	  (symName = XGetAtomName(disp, symNameAtom)) == NULL) return;

	count = 0;
	
	for(ptr = strtok(symName, "+"); ptr != NULL; ptr = strtok(NULL, "+"))
	{
		ptr1 = strchr(ptr, '(');
		if (ptr1 != NULL) *ptr1 = '\0';
		ptr1 = strchr(ptr, '_');
		if (ptr1 != NULL && !isupper(*(ptr1+1))) *ptr1 = '\0';
		ptr1 = strchr(ptr, ':');
		if (ptr1 != NULL) *ptr1 = '\0';

  	    ptr1 = strrchr(ptr, '/');
	    if (ptr1 != NULL) {
		  // Filter out cases like pc/pc
		  if (memcmp(ptr, ptr1+1, ptr1-ptr) == 0) continue;			
		  
		  ptr = ptr1+1;
		}

		if (strncmp(ptr, "group", 5) == 0) continue;

		SymbolNames[count++] = strdup(ptr);	
	}

	if (count == 1 && GroupNames[0] == NULL &&
	      strcmp(SymbolNames[0], "jp") == 0)  {
		GroupCount = 2;
		SymbolNames[1] = SymbolNames[0];
		SymbolNames[0] = strdup("us");
		GroupNames[0] = strdup("US/ASCII");
		GroupNames[1] = strdup("Japanese");
	}	  
	else
	if (count<GroupCount) {
		int j=count, k=GroupCount;
		while(--j>=0) SymbolNames[--k] = SymbolNames[j];
		while(--k>=0) SymbolNames[k] = strdup("en_US");	
	}

}

/*

// Custom name collection
// Source string will be damaged !
static void collectCustomNames(char *nameStr) 
{
	int i = 0;
	char *ptr, *ptr1;
	const char *seps = ",;)+";
	
	ptr = strtok(nameStr, seps);

	while (i<GroupCount && ptr != NULL && *ptr != '\0') {
		if (*ptr == '_') ptr++;
		ptr1 = strchr(ptr, '(');
		if (ptr1 != NULL) *ptr1 = '\0';
		if (AllGroupNames[i] == NULL)
			AllGroupNames[i] = strdup(ptr);
		i++;	
		ptr = strtok(NULL, seps);

	}
}


// Gets rid of all remaining NULLs 
static void finalizeNames(void) 
{
	int i;
	const char *ptr = "U/A";

	for (i=0; i<GroupCount; i++) {
		if (AllGroupNames[i] == NULL) AllGroupNames[i] = strdup(ptr);
	}
}
*/

//=============================================================================
Bool initXKB(void)
{
	return getXKBVersion();
}	




void grabFunctionKey(FunctionKey *key)
{
	if (key->active) {
		int keyscan = key->scan;
		int modifier = key->modifier;	
		
		HasFunctionKeys = True;
#if HAVE_XINPUT
		if (xdev != NULL)
			XGrabDeviceKey(disp, xdev, keyscan, modifier, xdev, wndRoot, False,
			      1, &key_event_class,   GrabModeAsync, GrabModeAsync);
		else {			
#endif
	   		XGrabKey(disp, keyscan, modifier, wndRoot, False, GrabModeAsync, GrabModeAsync);
	}	
}

void ungrabFunctionKey(FunctionKey *key)
{
	if (key->active) {
		int keyscan = key->scan;
		int modifier = key->modifier;	
#if HAVE_XINPUT
		if (xdev != NULL)
			XUngrabDeviceKey(disp, xdev, keyscan, modifier, xdev, wndRoot);
		else						
#endif
	  	  XUngrabKey(disp, keyscan, modifier, wndRoot);
	}	
}



void prepareFunctionKeys(void)
{
	int i, j;
	
	for (i=0; i<GroupCount; i++) {
		j = GroupXKBToRes(i);

		if (j>=0)  {
			if (j < SetKeyCount)
				grabFunctionKey(GroupSetKeys + j);

			if (j < LatchKeyCount)
				grabFunctionKey(GroupLatchKeys + j);
		}	
	
	}

	for (i=0; i<SwitchKeyCount; i++) 
		grabFunctionKey(GroupSwitchKeys + i);

}


void unprepareFunctionKeys(void)
{
	int i, j;

	for (i=0; i<XkbNumKbdGroups; i++) {
		j = GroupXKBToRes(i);
		if (j>=0)  {
			if (j < SetKeyCount)
				ungrabFunctionKey(GroupSetKeys + j);

			if (j < LatchKeyCount)
				ungrabFunctionKey(GroupLatchKeys + j);
		}	
	
	}

	for (i=0; i<SwitchKeyCount; i++) 
		ungrabFunctionKey(GroupSwitchKeys + i);


}

//=============================================================================
void accomodateGroupXKB(void)
{
	CurrentGroupResNo = GroupXKBToRes(CurrentGroupXKBNo);
	updateIconTitle();
	runScriptNow(GroupScripts[CurrentGroupResNo]);
}

//=============================================================================

Bool prepareXKB(void)
{
	int status = False;
	XkbStateRec xkb_state;
	XkbDescRec *kbd_desc_ptr = NULL;
	const Atom *groupSource;	
	unsigned long mask;

	GroupCount = 0;
	HasFunctionKeys = False;

	deviceID = XkbUseCoreKbd;
#if HAVE_XINPUT
	if (KbdDevOrder >= 0 && KbdDevOrder < 256 &&
	   examineKeyboardDevice() == False)
	  return False;
#endif		
	
	/* ============================================================================= */
	/* ============================================================================= */
	/* ============================================================================= */
	/*            We don't check status ever after because of XKB bugs !             */
	/* ============================================================================= */
	/* ============================================================================= */
	/* ============================================================================= */

	//  This doesn't work
	//kbd_desc_ptr = XkbGetKeyboard(disp, XkbControlsMask | XkbNamesMask, deviceID); 

	//  This works but rather unstable
	//if (kbd_desc_ptr == NULL)
//		kbd_desc_ptr = XkbGetKeyboard(disp, XkbAllComponentsMask, deviceID);

	// A way to overcome the bugs and get exactly what we need - too risky though
	kbd_desc_ptr = XkbAllocKeyboard();
	if (kbd_desc_ptr == NULL) {
		fprintf(stderr, "Failed to get keyboard description\n");
		goto HastaLaVista;
	}	

	kbd_desc_ptr->dpy = disp;
	if (deviceID != XkbUseCoreKbd) kbd_desc_ptr->device_spec = deviceID;

	XkbGetControls(disp, XkbAllControlsMask, kbd_desc_ptr);
	XkbGetNames(disp, XkbSymbolsNameMask, kbd_desc_ptr);
	XkbGetNames(disp, XkbGroupNamesMask, kbd_desc_ptr);
	
	if (kbd_desc_ptr->names == NULL) {
		fprintf(stderr, "Failed to get keyboard description\n");
		goto HastaLaVista;
	}
	
	groupSource = kbd_desc_ptr->names->groups;

	// And more bug patches !
	if (kbd_desc_ptr->ctrls != NULL)
		GroupCount = kbd_desc_ptr->ctrls->num_groups;
	else {
		for (GroupCount=0; 
		     GroupCount<XkbNumKbdGroups && groupSource[GroupCount] != None;
			 GroupCount++);
	}	

    if (GroupCount == 0) GroupCount=1;
	
	
	

	
	initGroupNames();
	collectGroupNames(kbd_desc_ptr);
	collectSymbolNames(kbd_desc_ptr);
//	finalizeNames();

	checkGroupNames();		
	  

	XkbGetState(disp, deviceID, &xkb_state);
	CurrentGroupXKBNo = xkb_state.group;


	/* Register for event processing */
	mask = XkbStateNotifyMask;
	XkbSelectEvents(disp, deviceID, mask, mask);
	mask = XkbGroupStateMask;
	XkbSelectEventDetails(disp, deviceID, XkbStateNotify, mask, mask);


	prepareFunctionKeys();
//	if (HasFunctionKeys)
//		XSelectInput(disp, wndRoot, KeyPressMask);



	status = True;


HastaLaVista:
	// The following will free all items (if .. is actually redundant)
	if (kbd_desc_ptr) XkbFreeKeyboard(kbd_desc_ptr, 0, True);
	return status;
}

//=============================================================================
void deinitXKB(void)
{
	deinitGroupNames();
	unprepareFunctionKeys();

#if HAVE_XINPUT
	if (xdev != NULL)
		XCloseDevice(disp, xdev);
#endif		

}

//=============================================================================

const char * getGroupName(int groupNo) 
{
	char *result;
	switch(GroupTitleSource) {
	  case 0:
		result = getSymbolNameByResNo(groupNo);
		break;
		
	  case 1:	
		result = GroupNames[groupNo];
		break;

	  case 2:	
		result = CustomNames[groupNo]; 		  
		break;
		
	  default:	// None
	  	return NULL; 		
	  		
	}	

	if (result == NULL) result = "U/A";
	return result;
}


const char * getCurrentGroupName() 
{
	return getGroupName(CurrentGroupResNo); 
}


//=============================================================================

Bool processEventXKB(XkbEvent *xkbev)
{
	/* The following test has become redundant due to event selection
		if (xkbev->type == BaseEventCode &&
				xkbev->any.xkb_type == XkbStateNotify &&
	*/
	if (xkbev->type == BaseEventCode) {
		int newGroupNo;
		
		// Redundant, but left for safety
		if (xkbev->any.xkb_type == XkbStateNotify &&
  		   //(xkbev->state.changed & XkbGroupStateMask) != 0 && // Left out for redundancy
			(newGroupNo = xkbev->state.group) != CurrentGroupXKBNo) {
			CurrentGroupXKBNo = newGroupNo;
			accomodateGroupXKB();
			processGroupX11();
		}
			
	//	return True;
	}	

	return False;	// Not an XKB event	
}

static Bool latchGroupByXKBNo(int latchedGroupNo)
{
	return (latchedGroupNo != CurrentGroupResNo) &&
			XkbLatchGroup(disp, deviceID, latchedGroupNo);

}

static Bool setGroupByXKBNo(int newGroupNo)
{
	return (newGroupNo != CurrentGroupResNo) &&
			XkbLockGroup(disp, deviceID, newGroupNo);

}

Bool changeGroup(int increment)
{
  if (GroupCount <= 1) return False;
  XkbLockGroup(disp, deviceID, 
	 (CurrentGroupXKBNo + GroupCount + increment) % GroupCount);
  return True;
}


static int lookupFunctionKeyByResNo(const FunctionKey *fkey, const FunctionKey *allKeys, int count)
{
	int i, j;
	const FunctionKey *curKey;
	
	for (i=0; i<GroupCount; i++) {
		j = GroupXKBToRes(i);
		if (j > count) continue;		
		curKey = allKeys + j;
		if (curKey->active &&
		    curKey->scan == fkey->scan &&
			curKey->modifier == fkey->modifier) return i;
	}
	
	return -1;
}

static int lookupFunctionKey(const FunctionKey *fkey, const FunctionKey *allKeys, int count)
{
	int i;
	
	
	for (i=0; i<count; i++) {
		if (allKeys->active &&
		    allKeys->scan == fkey->scan &&
			allKeys->modifier == fkey->modifier) return i;
		allKeys++;
	}
	
	return -1;
}


Bool processFunctionKey(XEvent *xev)
{
	FunctionKey fkey;
	int index = xev->type;

#if HAVE_XINPUT
	if (xdev != NULL && index == key_press_type) {
		XDeviceKeyEvent *xdevkey = 	(XDeviceKeyEvent *) xev;

		fkey.scan = xdevkey->keycode;
		fkey.modifier = xdevkey->state;
	}
	else
#endif	
	if (index == KeyPress) {
		XKeyEvent *xkey = 	(XKeyEvent *) xev;
		
		fkey.scan = xkey->keycode;
		fkey.modifier = xkey->state;
	}
	else
		return False;

		
		 
	if ((index = lookupFunctionKeyByResNo(&fkey, GroupSetKeys, SetKeyCount)) >= 0)
		setGroupByXKBNo(index);
			
	
	if (index < 0 &&
		(index = lookupFunctionKeyByResNo(&fkey, GroupLatchKeys, LatchKeyCount)) >= 0) 
		latchGroupByXKBNo(index);
	

	if (index < 0 &&
		(index = lookupFunctionKey(&fkey, GroupSwitchKeys,	SwitchKeyCount)) >= 0) 
		changeGroup((index > 0) ? -1 : 1);  

		
	return False;

}


void ringBell(int volume)
{
#if HAVE_XINPUT
	if (xdev != NULL)
		XDeviceBell(disp, xdev, BellFeedbackClass,  0, volume);
	else						
#endif
		XBell(disp, volume);
}
