package netflow

import (
	"bytes"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestDecodeNetFlowV9(t *testing.T) {
	templates := CreateTemplateSystem()

	// Decode a template
	template := []byte{
		0x00, 0x09, 0x00, 0x01, 0xb3, 0xbf, 0xf6, 0x83, 0x61, 0x8a, 0xa3, 0xa8, 0x32, 0x01, 0xee, 0x98,
		0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x64, 0x01, 0x04, 0x00, 0x17, 0x00, 0x02, 0x00, 0x04,
		0x00, 0x01, 0x00, 0x04, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0c, 0x00, 0x04, 0x00, 0x0a, 0x00, 0x04,
		0x00, 0x0e, 0x00, 0x04, 0x00, 0x15, 0x00, 0x04, 0x00, 0x16, 0x00, 0x04, 0x00, 0x07, 0x00, 0x02,
		0x00, 0x0b, 0x00, 0x02, 0x00, 0x10, 0x00, 0x04, 0x00, 0x11, 0x00, 0x04, 0x00, 0x12, 0x00, 0x04,
		0x00, 0x09, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01,
		0x00, 0x05, 0x00, 0x01, 0x00, 0x3d, 0x00, 0x01, 0x00, 0x59, 0x00, 0x01, 0x00, 0x30, 0x00, 0x02,
		0x00, 0xea, 0x00, 0x04, 0x00, 0xeb, 0x00, 0x04,
	}
	buf := bytes.NewBuffer(template)
	dec, err := DecodeMessage(buf, templates)
	assert.Nil(t, err)
	assert.NotNil(t, dec)
	decNfv9 := dec.(NFv9Packet)
	assert.Equal(t,
		NFv9Packet{
			Version:        9,
			Count:          1,
			SystemUptime:   0xb3bff683,
			UnixSeconds:    0x618aa3a8,
			SequenceNumber: 838987416,
			SourceId:       256,
			FlowSets: []interface{}{
				TemplateFlowSet{
					FlowSetHeader: FlowSetHeader{Id: 0x0, Length: 100},
					Records: []TemplateRecord{
						{
							TemplateId: 260,
							FieldCount: 23,
							Fields: []Field{
								{PenProvided: false, Type: 0x2, Length: 0x4},
								{PenProvided: false, Type: 0x1, Length: 0x4},
								{PenProvided: false, Type: 0x8, Length: 0x4},
								{PenProvided: false, Type: 0xc, Length: 0x4},
								{PenProvided: false, Type: 0xa, Length: 0x4},
								{PenProvided: false, Type: 0xe, Length: 0x4},
								{PenProvided: false, Type: 0x15, Length: 0x4},
								{PenProvided: false, Type: 0x16, Length: 0x4},
								{PenProvided: false, Type: 0x7, Length: 0x2},
								{PenProvided: false, Type: 0xb, Length: 0x2},
								{PenProvided: false, Type: 0x10, Length: 0x4},
								{PenProvided: false, Type: 0x11, Length: 0x4},
								{PenProvided: false, Type: 0x12, Length: 0x4},
								{PenProvided: false, Type: 0x9, Length: 0x1},
								{PenProvided: false, Type: 0xd, Length: 0x1},
								{PenProvided: false, Type: 0x4, Length: 0x1},
								{PenProvided: false, Type: 0x6, Length: 0x1},
								{PenProvided: false, Type: 0x5, Length: 0x1},
								{PenProvided: false, Type: 0x3d, Length: 0x1},
								{PenProvided: false, Type: 0x59, Length: 0x1},
								{PenProvided: false, Type: 0x30, Length: 0x2},
								{PenProvided: false, Type: 0xea, Length: 0x4},
								{PenProvided: false, Type: 0xeb, Length: 0x4},
							},
						},
					},
				},
			},
		}, decNfv9)
	assert.Equal(t,
		`Flow Packet
------------
  Version: 9
  Count:  1
  SystemUptime: 3015702147
  UnixSeconds: 2021-11-09 16:36:56 +0000 UTC
  SequenceNumber: 838987416
  SourceId: 256
  FlowSets (1):
    - TemplateFlowSet 0:
       Id 0
       Length: 100
       Records (1 records):
       - 0. Record:
            TemplateId: 260
            FieldCount: 23
            Fields (23):
            - 0. IN_PKTS (2/false): 4
            - 1. IN_BYTES (1/false): 4
            - 2. IPV4_SRC_ADDR (8/false): 4
            - 3. IPV4_DST_ADDR (12/false): 4
            - 4. INPUT_SNMP (10/false): 4
            - 5. OUTPUT_SNMP (14/false): 4
            - 6. LAST_SWITCHED (21/false): 4
            - 7. FIRST_SWITCHED (22/false): 4
            - 8. L4_SRC_PORT (7/false): 2
            - 9. L4_DST_PORT (11/false): 2
            - 10. SRC_AS (16/false): 4
            - 11. DST_AS (17/false): 4
            - 12. BGP_IPV4_NEXT_HOP (18/false): 4
            - 13. SRC_MASK (9/false): 1
            - 14. DST_MASK (13/false): 1
            - 15. PROTOCOL (4/false): 1
            - 16. TCP_FLAGS (6/false): 1
            - 17. SRC_TOS (5/false): 1
            - 18. DIRECTION (61/false): 1
            - 19. FORWARDING STATUS (89/false): 1
            - 20. FLOW_SAMPLER_ID (48/false): 2
            - 21. Unassigned (234/false): 4
            - 22. Unassigned (235/false): 4
`,
		decNfv9.String())

	// Decode some data using the above template
	data := []byte{
		0x00, 0x09, 0x00, 0x15, 0xb3, 0xbf, 0xf6, 0x83, 0x61, 0x8a, 0xa3, 0xa8, 0x32, 0x01, 0xee, 0x9c,
		0x00, 0x00, 0x01, 0x00, 0x01, 0x04, 0x05, 0x5c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc,
		0xc6, 0x26, 0x78, 0xde, 0x58, 0x79, 0xd9, 0xd0, 0x00, 0x00, 0x01, 0x62, 0x00, 0x00, 0x01, 0x30,
		0xb3, 0xbf, 0xe6, 0xf9, 0xb3, 0xbf, 0xe6, 0xf9, 0x01, 0xbb, 0x3b, 0x50, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x0e, 0x06, 0x10, 0x00, 0x00, 0x40, 0x00,
		0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x0b,
		0xb8, 0x6d, 0x47, 0xa2, 0xc4, 0x5b, 0xad, 0x61, 0xe0, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01,
		0x30, 0xb3, 0xbf, 0xe8, 0x1c, 0xb3, 0xbf, 0xe6, 0xf9, 0x01, 0xbb, 0x7b, 0x99, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x0d, 0x06, 0x10, 0x48, 0x00, 0x40,
		0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
		0x05, 0xdc, 0xc6, 0x26, 0x78, 0xd3, 0x5b, 0xa5, 0xd2, 0xee, 0x00, 0x00, 0x01, 0x62, 0x00, 0x00,
		0x01, 0x75, 0xb3, 0xbf, 0xe6, 0xfc, 0xb3, 0xbf, 0xe6, 0xfc, 0x00, 0x50, 0x8f, 0xb8, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x95, 0xae, 0x3b, 0x18, 0x0e, 0x06, 0x10, 0x00, 0x00,
		0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
		0x00, 0x05, 0xdc, 0x5f, 0x64, 0x56, 0x42, 0x5b, 0xa9, 0x1a, 0xbe, 0x00, 0x00, 0x01, 0x61, 0x00,
		0x00, 0x01, 0x31, 0xb3, 0xbf, 0xe6, 0xfc, 0xb3, 0xbf, 0xe6, 0xfc, 0x00, 0x50, 0xbf, 0xc3, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x0e, 0x06, 0x10, 0x28,
		0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
		0x00, 0x00, 0x05, 0xdc, 0xc6, 0x26, 0x78, 0xc6, 0x5b, 0xab, 0x33, 0x34, 0x00, 0x00, 0x01, 0x62,
		0x00, 0x00, 0x01, 0x31, 0xb3, 0xbf, 0xe6, 0xfc, 0xb3, 0xbf, 0xe6, 0xfc, 0x01, 0xbb, 0xf9, 0xd5,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x0e, 0x06, 0x10,
		0x00, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x05, 0xdc, 0xc6, 0x26, 0x78, 0x83, 0x4e, 0xf2, 0x8c, 0x81, 0x00, 0x00, 0x01,
		0x62, 0x00, 0x00, 0x01, 0x31, 0xb3, 0xbf, 0xe6, 0xfe, 0xb3, 0xbf, 0xe6, 0xfe, 0x01, 0xbb, 0xb3,
		0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x18, 0x06,
		0x10, 0x00, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0xc6, 0x26, 0x78, 0xb8, 0x5b, 0xaa, 0xab, 0x01, 0x00, 0x00,
		0x01, 0x62, 0x00, 0x00, 0x01, 0x31, 0xb3, 0xbf, 0xe6, 0xff, 0xb3, 0xbf, 0xe6, 0xff, 0x01, 0xbb,
		0xe5, 0xe5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x0e,
		0x06, 0x10, 0x00, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0xc6, 0x26, 0x78, 0xc5, 0x5b, 0xa5, 0x22, 0x65, 0x00,
		0x00, 0x01, 0x62, 0x00, 0x00, 0x01, 0x69, 0xb3, 0xbf, 0xe7, 0x00, 0xb3, 0xbf, 0xe7, 0x00, 0x01,
		0xbb, 0x3c, 0xb4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x95, 0xae, 0x31, 0x18,
		0x0e, 0x06, 0x10, 0x00, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x02,
		0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0x8f, 0xf4, 0x38, 0x1a, 0x5b, 0xa4, 0xc7, 0x3a,
		0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x75, 0xb3, 0xbf, 0xe7, 0x01, 0xb3, 0xbf, 0xe7, 0x01,
		0x01, 0xbb, 0x49, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x95, 0xae, 0x3b,
		0x17, 0x0e, 0x06, 0x10, 0x28, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00,
		0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xb0, 0xc7, 0xe8, 0xb2, 0x49, 0x5b, 0xaf, 0x83,
		0x0c, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x30, 0xb3, 0xbf, 0xe7, 0x02, 0xb3, 0xbf, 0xe7,
		0x02, 0x01, 0xbb, 0x96, 0x4a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00,
		0x00, 0x16, 0x0d, 0x06, 0x10, 0x28, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0xc6, 0x26, 0x78, 0xd8, 0x58, 0x7c,
		0x1f, 0x58, 0x00, 0x00, 0x01, 0x62, 0x00, 0x00, 0x01, 0x30, 0xb3, 0xbf, 0xe7, 0x02, 0xb3, 0xbf,
		0xe7, 0x02, 0x01, 0xbb, 0x16, 0x7b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf,
		0x00, 0x00, 0x18, 0x0e, 0x06, 0x10, 0x00, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0xc6, 0x26, 0x78, 0xdc, 0x5b,
		0xaf, 0x13, 0x88, 0x00, 0x00, 0x01, 0x62, 0x00, 0x00, 0x01, 0x30, 0xb3, 0xbf, 0xe7, 0x02, 0xb3,
		0xbf, 0xe7, 0x02, 0x01, 0xbb, 0x79, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc,
		0xdf, 0x00, 0x00, 0x18, 0x0d, 0x06, 0x10, 0x00, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02,
		0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0xcd, 0xea, 0xaf, 0x66,
		0x5b, 0xa1, 0xfc, 0x11, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x69, 0xb3, 0xbf, 0xe7, 0x03,
		0xb3, 0xbf, 0xe7, 0x03, 0x01, 0xbb, 0x79, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0xc2, 0x95, 0xae, 0x31, 0x18, 0x0e, 0x06, 0x10, 0x28, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00,
		0x02, 0x60, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x0b, 0x20, 0x8a, 0xc7, 0x10,
		0xcc, 0x5b, 0xa6, 0xb0, 0x14, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x69, 0xb3, 0xbf, 0xe7,
		0x04, 0xb3, 0xbf, 0xe4, 0xba, 0x04, 0xaa, 0x22, 0xd9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0xc2, 0x95, 0xae, 0x31, 0x18, 0x0e, 0x11, 0x00, 0x28, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00,
		0x00, 0x02, 0x60, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x03, 0xd8, 0xb9, 0x21,
		0xdc, 0x64, 0x5b, 0xac, 0x7f, 0x22, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x30, 0xb3, 0xbf,
		0xe7, 0x04, 0xb3, 0xbf, 0xe7, 0x04, 0x01, 0xbb, 0x1b, 0xac, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x16, 0x0d, 0x06, 0x18, 0x28, 0x00, 0x40, 0x00, 0x01, 0x60,
		0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0xdc, 0xb9,
		0x15, 0x3d, 0x5c, 0x4e, 0xe8, 0x7a, 0x02, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x30, 0xb3,
		0xbf, 0xe7, 0x05, 0xb3, 0xbf, 0xe7, 0x05, 0x88, 0xb3, 0xd0, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x16, 0x16, 0x06, 0x10, 0x28, 0x00, 0x40, 0x00, 0x01,
		0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x28,
		0xd4, 0x20, 0xfe, 0x7b, 0x5b, 0xab, 0x61, 0x86, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00, 0x01, 0x31,
		0xb3, 0xbf, 0xe7, 0x06, 0xb3, 0xbf, 0xe7, 0x06, 0xd3, 0xc9, 0xc3, 0x50, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x13, 0x0e, 0x06, 0x10, 0x28, 0x00, 0x40, 0x00,
		0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x05,
		0x90, 0xc6, 0x26, 0x78, 0xc3, 0x25, 0xa5, 0xad, 0xb8, 0x00, 0x00, 0x01, 0x62, 0x00, 0x00, 0x01,
		0x31, 0xb3, 0xbf, 0xe7, 0x08, 0xb3, 0xbf, 0xe7, 0x08, 0x01, 0xbb, 0x58, 0x64, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x12, 0x06, 0x10, 0x00, 0x00, 0x40,
		0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
		0x05, 0xdc, 0x8f, 0xf4, 0x39, 0x32, 0x4e, 0xe6, 0x08, 0xb1, 0x00, 0x00, 0x01, 0x61, 0x00, 0x00,
		0x01, 0x30, 0xb3, 0xbf, 0xe7, 0x08, 0xb3, 0xbf, 0xe7, 0x08, 0x01, 0xbb, 0xb2, 0x9a, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x17, 0x17, 0x06, 0x10, 0x28, 0x00,
		0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
		0x00, 0x05, 0x3c, 0xc6, 0x26, 0x78, 0xb6, 0x25, 0xa4, 0xf7, 0xb2, 0x00, 0x00, 0x01, 0x62, 0x00,
		0x00, 0x01, 0x30, 0xb3, 0xbf, 0xe7, 0x09, 0xb3, 0xbf, 0xe7, 0x09, 0x01, 0xbb, 0xdd, 0x06, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x12, 0x06, 0x10, 0x00,
		0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
		0x00, 0x00, 0x05, 0x64, 0xcd, 0xb9, 0xd8, 0x12, 0x52, 0x8e, 0x0d, 0x65, 0x00, 0x00, 0x01, 0x61,
		0x00, 0x00, 0x01, 0x30, 0xb3, 0xbf, 0xe7, 0x09, 0xb3, 0xbf, 0xe7, 0x09, 0x00, 0x50, 0x94, 0x3c,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xdf, 0x00, 0x00, 0x18, 0x14, 0x06, 0x10,
		0x28, 0x00, 0x40, 0x00, 0x01, 0x60, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	}
	buf = bytes.NewBuffer(data[:89]) // truncate: we don't want to test for everything
	dec, err = DecodeMessage(buf, templates)
	assert.Nil(t, err)
	assert.NotNil(t, dec)
	decNfv9 = dec.(NFv9Packet)
	assert.Equal(t,
		NFv9Packet{
			Version:        9,
			Count:          21,
			SystemUptime:   3015702147,
			UnixSeconds:    1636475816,
			SequenceNumber: 838987420,
			SourceId:       256,
			FlowSets: []interface{}{
				DataFlowSet{
					FlowSetHeader: FlowSetHeader{
						Id:     260,
						Length: 1372,
					},
					Records: []DataRecord{
						// Truncated!
						{
							Values: []DataField{
								{
									PenProvided: false,
									Type:        2,
									Pen:         0,
									Value:       []uint8{0x00, 0x00, 0x00, 0x01},
								},
								{
									PenProvided: false,
									Type:        1,
									Pen:         0,
									Value:       []uint8{0x00, 0x00, 0x05, 0xdc},
								},
								{
									PenProvided: false,
									Type:        8,
									Pen:         0,
									Value:       []uint8{0xc6, 0x26, 0x78, 0xde},
								},
								{
									PenProvided: false,
									Type:        12,
									Pen:         0,
									Value:       []uint8{0x58, 0x79, 0xd9, 0xd0},
								},
								{
									PenProvided: false,
									Type:        10,
									Pen:         0,
									Value:       []uint8{0x00, 0x00, 0x01, 0x62},
								},
								{
									PenProvided: false,
									Type:        14,
									Pen:         0,
									Value:       []uint8{0x00, 0x00, 0x01, 0x30},
								},
								{
									PenProvided: false,
									Type:        21,
									Pen:         0,
									Value:       []uint8{0xb3, 0xbf, 0xe6, 0xf9},
								},
								{
									PenProvided: false,
									Type:        22,
									Pen:         0,
									Value:       []uint8{0xb3, 0xbf, 0xe6, 0xf9},
								},
								{
									PenProvided: false,
									Type:        7,
									Pen:         0,
									Value:       []uint8{0x01, 0xbb},
								},
								{
									PenProvided: false,
									Type:        11,
									Pen:         0,
									Value:       []uint8{0x3b, 0x50},
								},
								{
									PenProvided: false,
									Type:        16,
									Pen:         0,
									Value:       []uint8{0x00, 0x00, 0x00, 0x00},
								},
								{
									PenProvided: false,
									Type:        17,
									Pen:         0,
									Value:       []uint8{0x00, 0x00, 0x00, 0x00},
								},
								{
									PenProvided: false,
									Type:        18,
									Pen:         0,
									Value:       []uint8{0xfc, 0xdf, 0x00, 0x00},
								},
								{
									PenProvided: false,
									Type:        9,
									Pen:         0,
									Value:       []uint8{0x18},
								},
								{
									PenProvided: false,
									Type:        13,
									Pen:         0,
									Value:       []uint8{0x0e},
								},
								{
									PenProvided: false,
									Type:        4,
									Pen:         0,
									Value:       []uint8{0x06},
								},
								{
									PenProvided: false,
									Type:        6,
									Pen:         0,
									Value:       []uint8{0x10},
								},
								{
									PenProvided: false,
									Type:        5,
									Pen:         0,
									Value:       []uint8{0x00},
								},
								{
									PenProvided: false,
									Type:        61,
									Pen:         0,
									Value:       []uint8{0x00},
								},
								{
									PenProvided: false,
									Type:        89,
									Pen:         0,
									Value:       []uint8{0x40},
								},
								{
									PenProvided: false,
									Type:        48,
									Pen:         0,
									Value:       []uint8{0x00, 0x01},
								},
								{
									PenProvided: false,
									Type:        234,
									Pen:         0,
									Value:       []uint8{0x60, 0x00, 0x00, 0x02},
								},
								{
									PenProvided: false,
									Type:        235,
									Pen:         0,
									Value:       []uint8{0x60, 0x00, 0x00, 0x00},
								},
							},
						},
					},
				},
			},
		}, decNfv9)
	assert.Equal(t,
		`Flow Packet
------------
  Version: 9
  Count:  21
  SystemUptime: 3015702147
  UnixSeconds: 2021-11-09 16:36:56 +0000 UTC
  SequenceNumber: 838987420
  SourceId: 256
  FlowSets (1):
    - DataFlowSet 0:
       Id 260
       Length: 1372
       Records (1 records):
       - Record 0:
            Values (23):
            - 0. IN_PKTS (2): [0 0 0 1]
            - 1. IN_BYTES (1): [0 0 5 220]
            - 2. IPV4_SRC_ADDR (8): [198 38 120 222]
            - 3. IPV4_DST_ADDR (12): [88 121 217 208]
            - 4. INPUT_SNMP (10): [0 0 1 98]
            - 5. OUTPUT_SNMP (14): [0 0 1 48]
            - 6. LAST_SWITCHED (21): [179 191 230 249]
            - 7. FIRST_SWITCHED (22): [179 191 230 249]
            - 8. L4_SRC_PORT (7): [1 187]
            - 9. L4_DST_PORT (11): [59 80]
            - 10. SRC_AS (16): [0 0 0 0]
            - 11. DST_AS (17): [0 0 0 0]
            - 12. BGP_IPV4_NEXT_HOP (18): [252 223 0 0]
            - 13. SRC_MASK (9): [24]
            - 14. DST_MASK (13): [14]
            - 15. PROTOCOL (4): [6]
            - 16. TCP_FLAGS (6): [16]
            - 17. SRC_TOS (5): [0]
            - 18. DIRECTION (61): [0]
            - 19. FORWARDING STATUS (89): [64]
            - 20. FLOW_SAMPLER_ID (48): [0 1]
            - 21. Unassigned (234): [96 0 0 2]
            - 22. Unassigned (235): [96 0 0 0]
`,
		decNfv9.String())
}
