/*
 * QTPixmap GTK engine
 *
 * This is a hacked/modified version of the Pixmap GTK engine. Modified to use
 * ~/.qt/qtrc to get colour information to recolour widgets.
 *
 * Changes are (C) Craig Drummond 2002 - Craig.Drummond@lycos.co.uk
 *
 */

#include <math.h>
#include <stdlib.h>
#include "qtpixmap_theme.h"

#include "common-cache.c"

#if 0
static void display_widget(GtkWidget *widget, int level)
{
    if(level>=0)
    {
        printf("%s(%s) ", widget ? gtk_type_name(GTK_WIDGET_TYPE(widget)) : "NULL", widget && widget->name ? widget->name : "NULL");
        if(widget && widget->parent)
            display_widget(widget->parent, --level);
        else
            printf("\n");
    }
    else
        printf("\n");
}
#endif

extern GtkStyleClass qtpixmap_default_class;

/* internal functions */
static void         draw_hline(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x1,
			       gint x2,
			       gint y);
static void         draw_vline(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint y1,
			       gint y2,
			       gint x);
static void         draw_shadow(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);

static void         draw_polygon(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 GdkPoint * point,
				 gint npoints,
				 gint fill);
static void         draw_arrow(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       GtkArrowType arrow_type,
			       gint fill,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_diamond(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 gint x,
				 gint y,
				 gint width,
				 gint height);
static void         draw_oval(GtkStyle * style,
			      GdkWindow * window,
			      GtkStateType state_type,
			      GtkShadowType shadow_type,
			      GdkRectangle * area,
			      GtkWidget * widget,
			      gchar * detail,
			      gint x,
			      gint y,
			      gint width,
			      gint height);
static void         draw_string(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				const gchar * string);
static void         draw_box(GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void         draw_flat_box(GtkStyle * style,
				  GdkWindow * window,
				  GtkStateType state_type,
				  GtkShadowType shadow_type,
				  GdkRectangle * area,
				  GtkWidget * widget,
				  gchar * detail,
				  gint x,
				  gint y,
				  gint width,
				  gint height);
static void         draw_check(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_option(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);
static void         draw_cross(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_ramp(GtkStyle * style,
			      GdkWindow * window,
			      GtkStateType state_type,
			      GtkShadowType shadow_type,
			      GdkRectangle * area,
			      GtkWidget * widget,
			      gchar * detail,
			      GtkArrowType arrow_type,
			      gint x,
			      gint y,
			      gint width,
			      gint height);
static void         draw_tab(GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void         draw_shadow_gap(GtkStyle * style,
				    GdkWindow * window,
				    GtkStateType state_type,
				    GtkShadowType shadow_type,
				    GdkRectangle * area,
				    GtkWidget * widget,
				    gchar * detail,
				    gint x,
				    gint y,
				    gint width,
				    gint height,
				    GtkPositionType gap_side,
				    gint gap_x,
				    gint gap_width);
static void         draw_box_gap(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 gint x,
				 gint y,
				 gint width,
				 gint height,
				 GtkPositionType gap_side,
				 gint gap_x,
				 gint gap_width);
static void         draw_extension(GtkStyle * style,
				   GdkWindow * window,
				   GtkStateType state_type,
				   GtkShadowType shadow_type,
				   GdkRectangle * area,
				   GtkWidget * widget,
				   gchar * detail,
				   gint x,
				   gint y,
				   gint width,
				   gint height,
				   GtkPositionType gap_side);
static void         draw_focus(GtkStyle * style,
			       GdkWindow * window,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_slider(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height,
				GtkOrientation orientation);
static void        draw_handle(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height,
				GtkOrientation orientation);

/* internal data structs */

GtkStyleClass       qtpixmap_default_class =
{
  2,
  2,
  draw_hline,
  draw_vline,
  draw_shadow,
  draw_polygon,
  draw_arrow,
  draw_diamond,
  draw_oval,
  draw_string,
  draw_box,
  draw_flat_box,
  draw_check,
  draw_option,
  draw_cross,
  draw_ramp,
  draw_tab,
  draw_shadow_gap,
  draw_box_gap,
  draw_extension,
  draw_focus,
  draw_slider,
  draw_handle
};

GdkImlibImage *
pixbuf_cache_value_new (struct image_file *file)
{
  GdkImlibImage *img = gdk_imlib_load_image(file->name);

  if (!img)
      g_warning ("Pixbuf theme: Cannot load pixmap file %s\n", file->name);
  else
      if(COLOR_APP==file->color || (file->color>=0 && file->color<COLOR_NUMCOLORS))
      {
          GdkImlibColorModifier redMod,
                                greenMod,
                                blueMod;

          gdk_imlib_get_image_red_modifier(img, &redMod);
          gdk_imlib_get_image_green_modifier(img, &greenMod);
          gdk_imlib_get_image_blue_modifier(img, &blueMod);

          if(COLOR_APP==file->color)
          {
              redMod.brightness=file->c.rgb.r;
              greenMod.brightness=file->c.rgb.g;
              blueMod.brightness=file->c.rgb.b;
          }
          else
          {
              redMod.brightness=qtpixmap_engine_data.colors[file->color].r+file->c.mod;
              greenMod.brightness=qtpixmap_engine_data.colors[file->color].g+file->c.mod;
              blueMod.brightness=qtpixmap_engine_data.colors[file->color].b+file->c.mod;
          }

          gdk_imlib_set_image_red_modifier(img, &redMod);
          gdk_imlib_set_image_green_modifier(img, &greenMod);
          gdk_imlib_set_image_blue_modifier(img, &blueMod);
      }

  return img;
}

static GdkImlibImage *
load_image(struct image_data *file, GdkColor *widget_color)
{
    if(NULL!=widget_color || NULL==file->img)
    {
        if (!pixbuf_cache)
            pixbuf_cache = g_cache_new ((GCacheNewFunc)pixbuf_cache_value_new,
                                      (GCacheDestroyFunc)gdk_imlib_destroy_image,
                                      (GCacheDupFunc)pixbuf_cache_dup_key,
                                      (GCacheDestroyFunc)pixbuf_cache_dest_key,
                                      pixbuf_cache_hash_key, g_direct_hash, pixbuf_cache_key_equal);

        if(widget_color)
        {
            struct image_file newfile;

            memcpy(&newfile, file, sizeof(struct image_file));
            newfile.c.rgb.r=widget_color->red>>8;
            newfile.c.rgb.g=widget_color->green>>8;
            newfile.c.rgb.b=widget_color->blue>>8;
            newfile.color=COLOR_APP;
            return g_cache_insert (pixbuf_cache, &newfile);
        }
        else
            file->img=g_cache_insert (pixbuf_cache, file);
    }

    return file->img;
}

static struct theme_image *
match_theme_image(GtkStyle * style, 
		  GtkStateType state, 
		  GtkShadowType shadow_type, 
		  GtkWidget *widget, 
		  char *detail, 
		  GtkArrowType arrow_type, 
		  GtkOrientation orientation, 
		  GtkPositionType gap_side, 
		  guint function)
{
  GList              *l;
  struct theme_image *i;
  char               *str = "";

  l = ((ThemeData *)(style->engine_data))->img_list;
  if (!detail) 
    detail = str;
  
  while (l)
    {
      i = (struct theme_image *)l->data;
      if ( (i) &&
	  (function == i->function) &&
	  
	  (((i->__state) &&
	    (state == i->state))
	   || (!(i->__state))) &&
	  
	  (((i->__shadow) && 
	    (shadow_type == i->shadow))
	   || (!(i->__shadow))) &&
	  
	  (((i->__arrow_direction) && 
	    (arrow_type == i->arrow_direction))
	   || (!(i->__arrow_direction))) &&
	  
	  (((i->__orientation) && 
	    (orientation == i->orientation))
	   || (!(i->__orientation))) &&

	  (((i->__gap_side) && 
	    (gap_side == i->gap_side))
	   || (!(i->__gap_side))) &&
	  
	  (((i->__state) && 
	    (state == i->state))
	   || (!(i->__state))) &&
	  
	  (((i->detail) && 
	    (!strcmp(detail, i->detail)))
	   || (!(i->detail))))
	return i;
      
      l = l->next;
    }
  return NULL;
}

static void
apply_theme_image(GdkWindow *window, struct theme_image *img, gchar setbg, 
		  GdkGC *gc, GdkRectangle *area, gint x, gint y, gint width, 
		  gint height, GdkColor *widget_color)
{
  GdkImlibImage      *im=NULL;
  GdkPixmap          *p=NULL, *m=NULL;
  GdkRectangle       rect0, rect;
  gchar              haverect = 1;

  if (gdk_window_get_type (window) == GDK_WINDOW_PIXMAP)
    setbg = FALSE;
  
  if (img->file.file.name)
    {
      im = load_image(&(img->file), widget_color);
      if (im)
	{  
	  gdk_imlib_set_image_border(im, &(img->border));
	  if (STRETCH_NONE!=img->stretch)
	    gdk_imlib_render(im, STRETCH_HEIGHT==img->stretch ? im->rgb_width : width,
                                 STRETCH_WIDTH==img->stretch ? im->rgb_height : height);
	  else
	    gdk_imlib_render(im, im->rgb_width, im->rgb_height);
	  p = gdk_imlib_move_image(im);
	  m = gdk_imlib_move_mask(im);
	  
	  if (area)
	    {
	      rect0.x = x;
	      rect0.y = y;
	      rect0.width = width;
	      rect0.height = height;
	      haverect = gdk_rectangle_intersect(&rect0, area, &rect);
	    }
	  else
	    {
	      rect.x = x;
	      rect.y = y;
	      rect.width = width;
	      rect.height = height;
	    }
	  if ((haverect) && (p))
	    {
	      if (setbg)
		{
		  gdk_window_set_back_pixmap(window, p, 0);
		  if (area)
		    gdk_window_clear_area(window, rect.x, rect.y,
					  rect.width, rect.height);
		  else
		    gdk_window_clear(window);
		  if (m)
		    gdk_window_shape_combine_mask(window, m, 0, 0);
		}
	      else
		{
		  if (m)
		    {
		      gdk_gc_set_clip_mask(gc, m);
		      gdk_gc_set_clip_origin(gc, x, y);
		    }
		  gdk_draw_pixmap(window, gc, p, rect.x - x, rect.y - y, 
				  rect.x, rect.y, rect.width, rect.height);
		  if (m)
		    {
		      gdk_gc_set_clip_mask(gc, NULL);
		      gdk_gc_set_clip_origin(gc, 0, 0);
		    }
		}
	    }
           if(p)
               gdk_imlib_free_pixmap(p);
	  /* CPD gdk_imlib_destroy_image(im); */
	}
    }
  
  if (!img->overlay_file.file.name ||
      (img->overlay_min_size.width && width < img->overlay_min_size.width) ||
      (img->overlay_min_size.height && height < img->overlay_min_size.height) )
    {
      if (area)
	gdk_gc_set_clip_rectangle(gc, NULL);
      return;
    }
  im = load_image(&(img->overlay_file), NULL);
  if (!im)
    {
      if (area)
	gdk_gc_set_clip_rectangle(gc, NULL);
      return;
    }
    
  gdk_imlib_set_image_border(im, &(img->overlay_border));
  switch(img->overlay_stretch)
  {
      default:
      case STRETCH_NONE:
      {
        x += (width - im->rgb_width) / 2;
        y += (height - im->rgb_height) / 2;
        width = im->rgb_width;
        height = im->rgb_height;
        gdk_imlib_render(im, im->rgb_width, im->rgb_height);
        break;
     }
     case STRETCH_BOTH:
        gdk_imlib_render(im, width, height);
        break;
     case STRETCH_HEIGHT:
     {
        x += (width - im->rgb_width) / 2;
        width = im->rgb_width;
        gdk_imlib_render(im, im->rgb_width, height);
        break;
     }
     case STRETCH_WIDTH:
     {
        y += (height - im->rgb_height) / 2;
        height = im->rgb_height;
        gdk_imlib_render(im, width, im->rgb_height);
        break;
     }
  }

  p = gdk_imlib_move_image(im);
  m = gdk_imlib_move_mask(im);
  haverect = 1;
  
  if (area)
    {
      rect0.x = x;
      rect0.y = y;
      rect0.width = width;
      rect0.height = height;
      haverect = gdk_rectangle_intersect(&rect0, area, &rect);
    }
  else
    {
      rect.x = x;
      rect.y = y;
      rect.width = width;
      rect.height = height;
    }
  if ((haverect) && (p))
    {
      if (m)
	{
	  gdk_gc_set_clip_mask(gc, m);
	  gdk_gc_set_clip_origin(gc, x, y);
	}
      gdk_draw_pixmap(window, gc, p, rect.x - x, rect.y - y, 
		      rect.x, rect.y, rect.width, rect.height);
      if (m)
	{
	  gdk_gc_set_clip_mask(gc, NULL);
	  gdk_gc_set_clip_origin(gc, 0, 0);
	}
    }

  if(p)
      gdk_imlib_free_pixmap(p);
 /* CPD gdk_imlib_destroy_image(im); */
}

static void
apply_theme_image_border(GdkWindow *window, struct theme_image *img, gchar setbg, 
			 GdkGC *gc, GdkRectangle *area, gint x, gint y, gint width, 
			 gint height)
{
  GdkImlibImage      *im=NULL;
  GdkPixmap          *p=NULL, *m=NULL;
  GdkRectangle       rect0, rect1, rect;
  gchar              haverect = 1;
  
  if (img->file.file.name)
    {
      im = load_image(&(img->file), NULL);
      if (im)
	{  
	  gdk_imlib_set_image_border(im, &(img->border));
	  gdk_imlib_render(im, width, height);
	  p = gdk_imlib_move_image(im);
	  m = gdk_imlib_move_mask(im);
	  
	  if (area)
	    {
	      rect0.x = x;
	      rect0.y = y;
	      rect0.width = width;
	      rect0.height = height;
	      haverect = gdk_rectangle_intersect(&rect0, area, &rect);
	    }
	  rect.x = x;
	  rect.y = y;
	  rect.width = width;
	  rect.height = height;
	  if ((haverect) && (p))
	    {
	      if (m)
		{
		  gdk_gc_set_clip_mask(gc, m);
		  gdk_gc_set_clip_origin(gc, x, y);
		}
	      rect0.x = rect.x;
	      rect0.y = rect.y;
	      rect0.width = width;
	      rect0.height = im->border.top;
	      if (area)
		{
		  if (gdk_rectangle_intersect(&rect0, area, &rect1))
		    gdk_draw_pixmap(window, gc, p,
				    rect1.x - rect0.x, rect1.y - rect0.y,
				    rect1.x, rect1.y,
				    rect1.width, rect1.height);
		}
	      else
		gdk_draw_pixmap(window, gc, p,
				0, 0,
				rect0.x, rect0.y,
				rect0.width, rect0.height);

	      rect0.x = rect.x;
	      rect0.y = rect.y + height - im->border.bottom;
	      rect0.width = width;
	      rect0.height = im->border.bottom;
	      if (area)
		{ 
		  if (gdk_rectangle_intersect(&rect0, area, &rect1))
		    gdk_draw_pixmap(window, gc, p,
				    rect1.x - rect0.x, rect1.y - rect0.y +
				    height - im->border.bottom,
				    rect1.x, rect1.y,
				    rect1.width, rect1.height);
		}
	      else
		gdk_draw_pixmap(window, gc, p,
				0, height - im->border.bottom,
				rect0.x, rect0.y,
				rect0.width, rect0.height);

	      rect0.x = rect.x;
	      rect0.y = rect.y + im->border.top;
	      rect0.width = im->border.left;
	      rect0.height = height - (im->border.top + im->border.bottom);
	      if (area)
		{
		  if (gdk_rectangle_intersect(&rect0, area, &rect1))
		    gdk_draw_pixmap(window, gc, p,
				    rect1.x - rect0.x, rect1.y - rect0.y +
				    im->border.top,
				    rect1.x, rect1.y,
				    rect1.width, rect1.height);
		}
	      else
		gdk_draw_pixmap(window, gc, p,
				0, im->border.top,
				rect0.x, rect0.y,
				rect0.width, rect0.height);

	      rect0.x = rect.x + width - im->border.right;
	      rect0.y = rect.y + im->border.top;
	      rect0.width = im->border.right;
	      rect0.height = height - (im->border.top + im->border.bottom);
	      if (area)
		{
		  if (gdk_rectangle_intersect(&rect0, area, &rect1))
		    gdk_draw_pixmap(window, gc, p,
				    rect1.x - rect0.x + width - im->border.right, 
				    rect1.y - rect0.y + im->border.top,
				    rect1.x, rect1.y,
				    rect1.width, rect1.height);
		}
	      else
		gdk_draw_pixmap(window, gc, p,
				width - im->border.right, im->border.top,
				rect0.x, rect0.y,
				rect0.width, rect0.height);

	      if (m)
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	    }
          if(p)
              gdk_imlib_free_pixmap(p);
	  /* CPD gdk_imlib_destroy_image(im); */
	}
    }
}

static void
apply_theme_image_shadow_gap(GdkWindow *window, 
			     struct theme_image *img, 
			     gchar setbg, 
			     GdkGC *gc, 
			     GdkRectangle *area, 
			     gint x, gint y, gint width, gint height, 
			     GtkPositionType gap_side, 
			     gint gap_x, 
			     gint gap_width, 
			     GtkStyle *style)
{
  GdkImlibImage     *im=NULL, *im1=NULL, *im2=NULL;
  GdkPixmap         *p=NULL, *m=NULL, *p1=NULL, *m1=NULL, *p2=NULL, *m2=NULL;
  GdkRectangle       r1, r2;
  GdkRectangle       rect0, rect1, rect;
  gchar              haverect = 1;

  switch (gap_side)
    {
    case GTK_POS_TOP:
      r1.x      = x;
      r1.y      = y;
      r1.width  = gap_x;
      r1.height = style->klass->ythickness;
      r2.x      = x + gap_x + gap_width;
      r2.y      = y;
      r2.width  = width - (gap_x + gap_width);
      r2.height = style->klass->ythickness;
      break;
    case GTK_POS_BOTTOM:
      r1.x      = x;
      r1.y      = y + height - style->klass->ythickness;
      r1.width  = gap_x;
      r1.height = style->klass->ythickness;
      r2.x      = x + gap_x + gap_width;
      r2.y      = y + height - style->klass->ythickness;
      r2.width  = width - (gap_x + gap_width);
      r2.height = style->klass->ythickness;
      break;
    case GTK_POS_LEFT:
      r1.x      = x;
      r1.y      = y;
      r1.width  = style->klass->xthickness;
      r1.height = gap_x;
      r2.x      = x;
      r2.y      = y + gap_x + gap_width;
      r2.width  = style->klass->xthickness;
      r2.height = height - (gap_x + gap_width);
      break;
    case GTK_POS_RIGHT:
      r1.x      = x + width - style->klass->xthickness;
      r1.y      = y;
      r1.width  = style->klass->xthickness;
      r1.height = gap_x;
      r2.x      = x + width - style->klass->xthickness;
      r2.y      = y + gap_x + gap_width;
      r2.width  = style->klass->xthickness;
      r2.height = height - (gap_x + gap_width);
      break;
    }

  if ((img->file.file.name) && (img->gap_start_file.file.name) && (img->gap_end_file.file.name))
    {
      im = load_image(&(img->file), NULL);
      im1 = load_image(&(img->gap_start_file), NULL);
      im2 = load_image(&(img->gap_end_file), NULL);
      if ((im) && (im1) && (im2))
	{  
	  gdk_imlib_set_image_border(im, &(img->border));
	  gdk_imlib_set_image_border(im1, &(img->gap_start_border));
	  gdk_imlib_set_image_border(im2, &(img->gap_end_border));
	  gdk_imlib_render(im, width, height);
	  p = gdk_imlib_move_image(im);
	  m = gdk_imlib_move_mask(im);
	  gdk_imlib_render(im1, r1.width, r1.height);
	  p1 = gdk_imlib_move_image(im1);
	  m1 = gdk_imlib_move_mask(im1);
	  gdk_imlib_render(im2, r2.width, r2.height);
	  p2 = gdk_imlib_move_image(im2);
	  m2 = gdk_imlib_move_mask(im2);
	  
	  if (area)
	    {
	      rect0.x = x;
	      rect0.y = y;
	      rect0.width = width;
	      rect0.height = height;
	      haverect = gdk_rectangle_intersect(&rect0, area, &rect);
	    }
	  rect.x = x;
	  rect.y = y;
	  rect.width = width;
	  rect.height = height;

	  if (p)
	    {
	      if (m1)
		{
		  gdk_gc_set_clip_mask(gc, m1);
		  gdk_gc_set_clip_origin(gc, r1.x, r1.y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}

	      if (p1)
		{
		  rect0.x = r1.x;
		  rect0.y = r1.y;
		  rect0.width = r1.width;
		  rect0.height = r1.height;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p1,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p1,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		}

	      if (m2)
		{
		  gdk_gc_set_clip_mask(gc, m2);
		  gdk_gc_set_clip_origin(gc, r2.x, r2.y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}

	      if (p2)
		{
		  rect0.x = r2.x;
		  rect0.y = r2.y;
		  rect0.width = r2.width;
		  rect0.height = r2.height;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p2,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p2,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		}

	      if (m)
		{
		  gdk_gc_set_clip_mask(gc, m);
		  gdk_gc_set_clip_origin(gc, x, y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	      
	      switch (gap_side)
		{
		case GTK_POS_TOP:
		  rect0.x = rect.x;
		  rect0.y = rect.y + height - im->border.bottom;
		  rect0.width = width;
		  rect0.height = im->border.bottom;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x, rect1.y - rect0.y +
					height - im->border.bottom,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, height - im->border.bottom,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		  
		  rect0.x = rect.x;
		  rect0.y = rect.y + im->border.top;
		  rect0.width = im->border.left;
		  rect0.height = height - (im->border.top + im->border.bottom);
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x,
					rect1.y - rect0.y + im->border.top,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, im->border.top,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		  break;
		case GTK_POS_BOTTOM:
		  rect0.x = rect.x;
		  rect0.y = rect.y;
		  rect0.width = width;
		  rect0.height = im->border.top;

		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);

		  rect0.x = rect.x;
		  rect0.y = rect.y + im->border.top;
		  rect0.width = im->border.left;
		  rect0.height = height - (im->border.top + im->border.bottom);
		  if (area) 
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x,
					rect1.y - rect0.y + im->border.top,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, im->border.top,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		  
		  break;
		case GTK_POS_LEFT:
		  
		  rect0.x = rect.x;
		  rect0.y = rect.y;
		  rect0.width = width;
		  rect0.height = im->border.top;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);

		  rect0.x = rect.x;
		  rect0.y = rect.y + height - im->border.bottom;
		  rect0.width = width;
		  rect0.height = im->border.bottom;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x, rect1.y - rect0.y +
					height - im->border.bottom,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, height - im->border.bottom,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		  break;

		case GTK_POS_RIGHT:
		  rect0.x = rect.x;
		  rect0.y = rect.y;
		  rect0.width = width;
		  rect0.height = im->border.top;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		  
		  rect0.x = rect.x;
		  rect0.y = rect.y + height - im->border.bottom;
		  rect0.width = width;
		  rect0.height = im->border.bottom;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p,
					rect1.x - rect0.x, rect1.y - rect0.y +
					height - im->border.bottom,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p,
				    0, height - im->border.bottom,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		  break;
		}
	      
	      rect0.x = rect.x + width - im->border.right;
	      rect0.y = rect.y + im->border.top;
	      rect0.width = im->border.right;
	      rect0.height = height - (im->border.top + im->border.bottom);
	      if (area)
		{
		  if (gdk_rectangle_intersect(&rect0, area, &rect1))
		    gdk_draw_pixmap(window, gc, p,
				    rect1.x - rect0.x + width - im->border.right, 
				    rect1.y - rect0.y + im->border.top,
				    rect1.x, rect1.y,
				    rect1.width, rect1.height);
		}
	      else
		gdk_draw_pixmap(window, gc, p,
				width - im->border.right, 
				im->border.top,
				rect0.x, rect0.y,
				rect0.width, rect0.height);
	      
	      if (m)
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	    }
          if(p)
              gdk_imlib_free_pixmap(p);
          if(p1)
              gdk_imlib_free_pixmap(p1);
          if(p2)
              gdk_imlib_free_pixmap(p2);

	  /* CPD gdk_imlib_destroy_image(im);
	  gdk_imlib_destroy_image(im1);
	  gdk_imlib_destroy_image(im2); */
	}
    }
}

static void
apply_theme_image_box_gap(GdkWindow *window, struct theme_image *img, gchar setbg, 
			  GdkGC *gc, GdkRectangle *area, gint x, gint y, gint width, 
			  gint height, GtkPositionType gap_side, gint gap_x, gint gap_width, 
			  GtkStyle *style)
{
  GdkImlibImage      *im=NULL, *im1=NULL, *im2=NULL, *im3=NULL;
  GdkPixmap          *p=NULL, *m=NULL, *p1=NULL, *m1=NULL, *p2=NULL, *m2=NULL, *p3=NULL, *m3=NULL;
  GdkRectangle        r1, r2, r3;
  GdkRectangle       rect0, rect1, rect;
  gchar              haverect = 1;

  switch (gap_side)
    {
    case GTK_POS_TOP:
      r1.x      = x;
      r1.y      = y;
      r1.width  = gap_x;
      r1.height = style->klass->ythickness;
      r2.x      = x + gap_x + gap_width;
      r2.y      = y;
      r2.width  = width - (gap_x + gap_width);
      r2.height = style->klass->ythickness;
      r3.x      = x + gap_x;
      r3.y      = y;
      r3.width  = gap_width;
      r3.height = style->klass->ythickness;
      break;
    case GTK_POS_BOTTOM:
      r1.x      = x;
      r1.y      = y + height - style->klass->ythickness;
      r1.width  = gap_x;
      r1.height = style->klass->ythickness;
      r2.x      = x + gap_x + gap_width;
      r2.y      = y + height - style->klass->ythickness;
      r2.width  = width - (gap_x + gap_width);
      r2.height = style->klass->ythickness;
      r3.x      = x + gap_x;
      r3.y      = y + height - style->klass->ythickness;
      r3.width  = gap_width;
      r3.height = style->klass->ythickness;
      break;
    case GTK_POS_LEFT:
      r1.x      = x;
      r1.y      = y;
      r1.width  = style->klass->xthickness;
      r1.height = gap_x;
      r2.x      = x;
      r2.y      = y + gap_x + gap_width;
      r2.width  = style->klass->xthickness;
      r2.height = height - (gap_x + gap_width);
      r3.x      = x;
      r3.y      = y + gap_x;
      r3.width  = style->klass->xthickness;
      r3.height = gap_width;
      break;
    case GTK_POS_RIGHT:
      r1.x      = x + width - style->klass->xthickness;
      r1.y      = y;
      r1.width  = style->klass->xthickness;
      r1.height = gap_x;
      r2.x      = x + width - style->klass->xthickness;
      r2.y      = y + gap_x + gap_width;
      r2.width  = style->klass->xthickness;
      r2.height = height - (gap_x + gap_width);
      r3.x      = x + width - style->klass->xthickness;
      r3.y      = y + gap_x;
      r3.width  = style->klass->xthickness;
      r3.height = gap_width;
      break;
    }

  if ((img->file.file.name) && (img->gap_start_file.file.name) && (img->gap_end_file.file.name) &&
      (img->gap_file.file.name))
    {
      im = load_image(&(img->file), NULL);
      im1 = load_image(&(img->gap_start_file), NULL);
      im2 = load_image(&(img->gap_end_file), NULL);
      im3 = load_image(&(img->gap_file), NULL);
      if ((im) && (im1) && (im2) && (im3))
	{  
	  gdk_imlib_set_image_border(im, &(img->border));
	  gdk_imlib_set_image_border(im1, &(img->gap_start_border));
	  gdk_imlib_set_image_border(im2, &(img->gap_end_border));
	  gdk_imlib_set_image_border(im3, &(img->gap_border));
	  gdk_imlib_render(im, width, height);
	  p = gdk_imlib_move_image(im);
	  m = gdk_imlib_move_mask(im);
	  gdk_imlib_render(im1, r1.width, r1.height);
	  p1 = gdk_imlib_move_image(im1);
	  m1 = gdk_imlib_move_mask(im1);
	  gdk_imlib_render(im2, r2.width, r2.height);
	  p2 = gdk_imlib_move_image(im2);
	  m2 = gdk_imlib_move_mask(im2);
	  gdk_imlib_render(im3, r3.width, r3.height);
	  p3 = gdk_imlib_move_image(im3);
	  m3 = gdk_imlib_move_mask(im3);
	  
	  if (area)
	    {
	      rect0.x = x;
	      rect0.y = y;
	      rect0.width = width;
	      rect0.height = height;
	      haverect = gdk_rectangle_intersect(&rect0, area, &rect);
	    }
	  rect.x = x;
	  rect.y = y;
	  rect.width = width;
	  rect.height = height;
	  if ((p) && (haverect))
	    {
	      if (m)
		{
		  gdk_gc_set_clip_mask(gc, m);
		  gdk_gc_set_clip_origin(gc, x, y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	      rect0.x = rect.x;
	      rect0.y = rect.y;
	      rect0.width = width;
	      rect0.height = height;
	      if (area)
		{
		  if (gdk_rectangle_intersect(&rect0, area, &rect1))
		    gdk_draw_pixmap(window, gc, p,
				    rect1.x - rect0.x, rect1.y - rect0.y,
				    rect1.x, rect1.y,
				    rect1.width, rect1.height);
		}
	      else
		gdk_draw_pixmap(window, gc, p,
				0, 0,
				rect0.x, rect0.y,
				rect0.width, rect0.height);
	      if (m1)
		{
		  gdk_gc_set_clip_mask(gc, m1);
		  gdk_gc_set_clip_origin(gc, r1.x, r1.y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	      if (p1)
		{
		  rect0.x = r1.x;
		  rect0.y = r1.y;
		  rect0.width = r1.width;
		  rect0.height = r1.height;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p1,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p1,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		}
	      if (m2)
		{
		  gdk_gc_set_clip_mask(gc, m2);
		  gdk_gc_set_clip_origin(gc, r2.x, r2.y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	      if (p2)
		{
		  rect0.x = r2.x;
		  rect0.y = r2.y;
		  rect0.width = r2.width;
		  rect0.height = r2.height;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p2,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p2,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		}
	      if (m3)
		{
		  gdk_gc_set_clip_mask(gc, m3);
		  gdk_gc_set_clip_origin(gc, r3.x, r3.y);
		}
	      else
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	      if (p3)
		{
		  rect0.x = r3.x;
		  rect0.y = r3.y;
		  rect0.width = r3.width;
		  rect0.height = r3.height;
		  if (area)
		    {
		      if (gdk_rectangle_intersect(&rect0, area, &rect1))
			gdk_draw_pixmap(window, gc, p3,
					rect1.x - rect0.x, rect1.y - rect0.y,
					rect1.x, rect1.y,
					rect1.width, rect1.height);
		    }
		  else
		    gdk_draw_pixmap(window, gc, p3,
				    0, 0,
				    rect0.x, rect0.y,
				    rect0.width, rect0.height);
		}
	      if (m3)
		{
		  gdk_gc_set_clip_mask(gc, NULL);
		  gdk_gc_set_clip_origin(gc, 0, 0);
		}
	    }
          if(p)
              gdk_imlib_free_pixmap(p);
          if(p1)
              gdk_imlib_free_pixmap(p1);
          if(p2)
              gdk_imlib_free_pixmap(p2);
          if(p3)
              gdk_imlib_free_pixmap(p3);
	  /* CPD gdk_imlib_destroy_image(im);
	  gdk_imlib_destroy_image(im1);
	  gdk_imlib_destroy_image(im2);
	  gdk_imlib_destroy_image(im3); */
	}
    }
}

static void
draw_hline(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x1,
	   gint x2,
	   gint y)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  
  img = match_theme_image(style,
			  state_type,
			  GTK_SHADOW_IN,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_HLINE);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x1, y, (x2 - x1) + 1, 2, NULL);
    }
}

static void
draw_vline(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint y1,
	   gint y2,
	   gint x)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  GTK_SHADOW_IN,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_VLINE);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y1, 2, (y2 - y1) + 1, NULL);
    }
}

static void
draw_shadow(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  GTK_ORIENTATION_HORIZONTAL,
			  0,
			  TOKEN_D_SHADOW);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image_border(window, img, setbg, gc, area, x, y, width, height);
    }
}

static void
draw_polygon(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     GdkPoint * points,
	     gint npoints,
	     gint fill)
{
#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif /* M_PI */
#ifndef M_PI_4
#define M_PI_4  0.78539816339744830962
#endif /* M_PI_4 */

  static const gdouble pi_over_4 = M_PI_4;
  static const gdouble pi_3_over_4 = M_PI_4 * 3;

  GdkGC              *gc3;
  GdkGC              *gc4;
  gdouble             angle;
  gint                i;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
  g_return_if_fail(points != NULL);

  switch (shadow_type)
    {
    case GTK_SHADOW_IN:
      gc3 = style->light_gc[state_type];
      gc4 = style->black_gc;
      break;
    case GTK_SHADOW_OUT:
      gc3 = style->black_gc;
      gc4 = style->light_gc[state_type];
      break;
    default:
      return;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc3, area);
      gdk_gc_set_clip_rectangle(gc4, area);
    }
  if (fill)
    gdk_draw_polygon(window, style->bg_gc[state_type], TRUE, points, npoints);

  npoints--;

  for (i = 0; i < npoints; i++)
    {
      if ((points[i].x == points[i + 1].x) &&
	  (points[i].y == points[i + 1].y))
	angle = 0;
      else
	angle = atan2(points[i + 1].y - points[i].y,
		      points[i + 1].x - points[i].x);

      if ((angle > -pi_3_over_4) && (angle < pi_over_4))
	gdk_draw_line(window, gc3,
		      points[i].x, points[i].y,
		      points[i + 1].x, points[i + 1].y);
      else
	gdk_draw_line(window, gc4,
		      points[i].x, points[i].y,
		      points[i + 1].x, points[i + 1].y);
    }
  if (area)
    {
      gdk_gc_set_clip_rectangle(gc3, NULL);
      gdk_gc_set_clip_rectangle(gc4, NULL);
    }
}

static void
draw_arrow(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   GtkArrowType arrow_type,
	   gint fill,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

#if 0
printf("Draw arrow %d %d %d %s  ", state_type, shadow_type, orientation, detail ? detail : "NULL");
display_widget(widget, 4);
#endif

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  arrow_type,
			  orientation,
			  0,
			  TOKEN_D_ARROW);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
#if 0
else
printf("Could not find a suitable image %d %d %s %d %d\n", state_type, shadow_type, detail ? detail : "NULL", arrow_type, orientation);
#endif
}

static void
draw_diamond(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y,
	     gint width,
	     gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_DIAMOND);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_oval(GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail,
	  gint x,
	  gint y,
	  gint width,
	  gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_OVAL);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static int is_on_a_button(GtkWidget *w, int level)
{
    if(w)
        if(GTK_IS_BUTTON(w) &&
          (!(GTK_IS_RADIO_BUTTON(w) || GTK_IS_CHECK_BUTTON(w) || GTK_IS_OPTION_MENU(w))))
            return 1;
        else
            if (level<3)
                return is_on_a_button(w->parent, level++);
    else
        return 0;
}

static int is_on_a_toolbar(GtkWidget *w, int level)
{
    if(w)
        if(GTK_IS_TOOLBAR(w))
            return 1;
        else
            if (level<3)
                return is_on_a_toolbar(w->parent, level++);
    else
        return 0;
}

static void
draw_string(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    const gchar * string)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

#if 0
printf("Draw string %d %s \"%s\"  ", state_type, detail ? detail : "NULL", string ? string : "NULL");
display_widget(widget, 3);
#endif

  if(is_on_a_button(widget, 0))
  {
    int tbar=is_on_a_toolbar(widget, 0),
        x_offset=GTK_STATE_ACTIVE==state_type ? tbar ? qtpixmap_engine_data.tb_button_x_offset : qtpixmap_engine_data.button_x_offset : 0,
        y_offset=GTK_STATE_ACTIVE==state_type ? tbar ? qtpixmap_engine_data.tb_button_y_offset : qtpixmap_engine_data.button_y_offset : 0;

    GdkGC *gc=GTK_STATE_PRELIGHT==state_type || GTK_STATE_ACTIVE==state_type
                      ? style->text_gc[GTK_STATE_NORMAL]
                      : style->text_gc[state_type];

    if (area)
    {
      gdk_gc_set_clip_rectangle(style->white_gc, area);
      gdk_gc_set_clip_rectangle(gc, area);
    }
    if (state_type == GTK_STATE_INSENSITIVE)
      gdk_draw_string(window, style->font, style->white_gc,
                      x + 1 + x_offset,
                      y + 1 + y_offset, string);
    gdk_draw_string(window, style->font, gc,
                      x + x_offset,
                      y + y_offset, string);
    if (area)
    {
      gdk_gc_set_clip_rectangle(style->white_gc, NULL);
      gdk_gc_set_clip_rectangle(gc, NULL);
    }
  }
  else
  {
    int mbar=QTP_IS_ON_MENU_BAR(widget),
        menu=QTP_IS_ON_MENU(widget);

    if((mbar || menu) && state_type==GTK_STATE_PRELIGHT)
        state_type=(qtpixmap_engine_data.use_selected_for_menu_item && menu) ||
                   (qtpixmap_engine_data.use_selected_for_menu && mbar) ? GTK_STATE_SELECTED : GTK_STATE_NORMAL;

    qtpixmap_engine_data.parent_class->draw_string (style, window, state_type, area, widget, detail, x, y, string);
  }
}

static void
draw_box(GtkStyle * style,
	 GdkWindow * window,
	 GtkStateType state_type,
	 GtkShadowType shadow_type,
	 GdkRectangle * area,
	 GtkWidget * widget,
	 gchar * detail,
	 gint x,
	 gint y,
	 gint width,
	 gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  GdkColor           *widget_color=NULL;

#if 0
printf("Draw box %d %d %s  ", state_type, shadow_type, detail ? detail : "NULL");
display_widget(widget, 4);
#endif

  if(QTP_SET_COLOR_BUTTON(detail, style))
      widget_color=&(style->bg[GTK_STATE_NORMAL]);

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;

  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_BOX);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, widget_color);
    }
}

static void
draw_flat_box(GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail,
	      gint x,
	      gint y,
	      gint width,
	      gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;

#if 0 
printf("Draw flat box %d %d %d %s  ", state_type, shadow_type, orientation, detail ? detail : "NULL");
display_widget(widget, 4);
#endif
 
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if(qtpixmap_engine_data.no_radio_check_highlight && (GTK_IS_RADIO_BUTTON(widget) || GTK_IS_CHECK_BUTTON(widget)))
      return;

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_FLAT_BOX);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
  else
    gtk_style_apply_default_background (style, window, setbg, state_type,
					area, x, y, width, height);
}

static void
draw_check(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_CHECK);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_option(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_OPTION);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_cross(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_CROSS);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_ramp(GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail,
	  GtkArrowType arrow_type,
	  gint x,
	  gint y,
	  gint width,
	  gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  arrow_type,
			  orientation,
			  0,
			  TOKEN_D_RAMP);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_tab(GtkStyle * style,
	 GdkWindow * window,
	 GtkStateType state_type,
	 GtkShadowType shadow_type,
	 GdkRectangle * area,
	 GtkWidget * widget,
	 gchar * detail,
	 gint x,
	 gint y,
	 gint width,
	 gint height)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_TAB);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_shadow_gap(GtkStyle * style,
		GdkWindow * window,
		GtkStateType state_type,
		GtkShadowType shadow_type,
		GdkRectangle * area,
		GtkWidget * widget,
		gchar * detail,
		gint x,
		gint y,
		gint width,
		gint height,
		GtkPositionType gap_side,
		gint gap_x,
		gint gap_width)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  gap_side,
			  TOKEN_D_SHADOW_GAP);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image_shadow_gap(window, img, setbg, gc, area, x, y, width, 
				   height, gap_side, gap_x, gap_width, style);
    }
}

static void
draw_box_gap(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y,
	     gint width,
	     gint height,
	     GtkPositionType gap_side,
	     gint gap_x,
	     gint gap_width)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      setbg = 1;
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  gap_side,
			  TOKEN_D_BOX_GAP);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image_box_gap(window, img, setbg, gc, area, x, y, width, 
				height, gap_side, gap_x, gap_width, style);
    }
}

static void
draw_extension(GtkStyle * style,
	       GdkWindow * window,
	       GtkStateType state_type,
	       GtkShadowType shadow_type,
	       GdkRectangle * area,
	       GtkWidget * widget,
	       gchar * detail,
	       gint x,
	       gint y,
	       gint width,
	       gint height,
	       GtkPositionType gap_side)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  GtkOrientation      orientation;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if (width >=0)
    width++;
  if (height >=0)
    height++;
  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);
  
  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;
  
  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  gap_side,
			  TOKEN_D_EXTENSION);
  if (img)
    {
      gc = style->bg_gc[GTK_STATE_NORMAL];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
sanitize_size (GdkWindow *window,
               gint      *width,
               gint      *height)
{
  if ((*width == -1) && (*height == -1))
    gdk_window_get_size (window, width, height);
  else if (*width == -1)
    gdk_window_get_size (window, width, NULL);
  else if (*height == -1)
    gdk_window_get_size (window, NULL, height);
}

static void
draw_focus(GtkStyle * style,
	   GdkWindow * window,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  GdkPoint points[5];
  GdkGC    *gc;
  gboolean free_dash_list = FALSE;
  gint line_width = 1;
  gchar *dash_list = "\1\1";
  gint dash_len;

#if 0
printf("Draw focus %s  ", detail ? detail : "NULL");
display_widget(widget, 4);
#endif

  if(GTK_IS_EDITABLE(widget))
  {
      gdk_draw_rectangle(window, style->black_gc, FALSE, x, y, width, height);
      return;
  }

  gc = style->black_gc;

  if(GTK_IS_OPTION_MENU(widget) || GTK_IS_BUTTON(widget) || GTK_IS_RADIO_BUTTON(widget) || GTK_IS_CHECK_BUTTON(widget))
      if(GTK_IS_OPTION_MENU(widget))
      {
          x+=x+qtpixmap_engine_data.combo_focus_mod.x >=0 ? qtpixmap_engine_data.combo_focus_mod.x : 0;
          y+=y+qtpixmap_engine_data.combo_focus_mod.y >=0 ? qtpixmap_engine_data.combo_focus_mod.y : 0;
          width+=width+qtpixmap_engine_data.combo_focus_mod.w >=0 ? qtpixmap_engine_data.combo_focus_mod.w : 0;
          height+=height+qtpixmap_engine_data.combo_focus_mod.h >=0 ? qtpixmap_engine_data.combo_focus_mod.h : 0;
      }
      else if (!(detail && 0==strcmp(detail, "togglebutton")) && (GTK_IS_RADIO_BUTTON(widget) || GTK_IS_CHECK_BUTTON(widget)))
      {
          x+=x+qtpixmap_engine_data.radio_check_focus_mod.x >=0 ? qtpixmap_engine_data.radio_check_focus_mod.x : 0;
          y+=y+qtpixmap_engine_data.radio_check_focus_mod.y >=0 ? qtpixmap_engine_data.radio_check_focus_mod.y : 0;
          width+=width+qtpixmap_engine_data.radio_check_focus_mod.w >=0 ? qtpixmap_engine_data.radio_check_focus_mod.w : 0;
          height+=height+qtpixmap_engine_data.radio_check_focus_mod.h >=0 ? qtpixmap_engine_data.radio_check_focus_mod.h : 0;
      }
      else
      {
          x+=x+qtpixmap_engine_data.button_focus_mod.x >=0 ? qtpixmap_engine_data.button_focus_mod.x : 0;
          y+=y+qtpixmap_engine_data.button_focus_mod.y >=0 ? qtpixmap_engine_data.button_focus_mod.y : 0;
          width+=width+qtpixmap_engine_data.button_focus_mod.w >=0 ? qtpixmap_engine_data.button_focus_mod.w : 0;
          height+=height+qtpixmap_engine_data.button_focus_mod.h >=0 ? qtpixmap_engine_data.button_focus_mod.h : 0;
      }

  sanitize_size (window, &width, &height);

  if (area)
    gdk_gc_set_clip_rectangle (gc, area);

  gdk_gc_set_line_attributes (gc, line_width,
                              dash_list[0] ? GDK_LINE_ON_OFF_DASH : GDK_LINE_SOLID,
                              GDK_CAP_BUTT, GDK_JOIN_MITER);


  if (detail && !strcmp (detail, "add-mode"))
    {
      if (free_dash_list)
        g_free (dash_list);

      dash_list = "\4\4";
      free_dash_list = FALSE;
    }

  points[0].x = x + line_width / 2;
  points[0].y = y + line_width / 2;
  points[1].x = x + width - line_width + line_width / 2;
  points[1].y = y + line_width / 2;
  points[2].x = x + width - line_width + line_width / 2;
  points[2].y = y + height - line_width + line_width / 2;
  points[3].x = x + line_width / 2;
  points[3].y = y + height - line_width + line_width / 2;
  points[4] = points[0];

  if (!dash_list[0])
    {
      gdk_draw_lines (window, gc, points, 5);
    }
  else
    {
      dash_len = strlen (dash_list);

      if (dash_list[0])
        gdk_gc_set_dashes (gc, 0, dash_list, dash_len);

      gdk_draw_lines (window, gc, points, 3);

      points[2].x += 1;

      if (dash_list[0])
        {
          gint dash_pixels = 0;
          gint i;

          for (i = 0; i < dash_len; i++)
            dash_pixels += dash_list[i];

          if (dash_len % 2 == 1)
            dash_pixels *= 2;

          gdk_gc_set_dashes (gc, dash_pixels - (width + height - 2 * line_width) % dash_pixels, dash_list, dash_len);
        }

      gdk_draw_lines (window, gc, points + 2, 3);
    }

  gdk_gc_set_line_attributes (gc, 0, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);

  if (area)
    gdk_gc_set_clip_rectangle (gc, NULL);

  if (free_dash_list)
    g_free (dash_list);
}

static void
draw_slider(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height,
	    GtkOrientation orientation)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;
  
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      setbg = 1;
      gdk_window_get_size(window, &width, &height);
    }
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_SLIDER);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}

static void
draw_handle(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height,
	    GtkOrientation orientation)
{
  struct theme_image *img;
  GdkGC              *gc;
  gchar               setbg = 0;

#if 0
printf("Draw handle %d %d %d %s  ", state_type, shadow_type, orientation, detail ? detail : "NULL");
display_widget(widget, 4);
#endif

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  orientation = height<width ? GTK_ORIENTATION_HORIZONTAL : GTK_ORIENTATION_VERTICAL;

  img = match_theme_image(style,
			  state_type,
			  shadow_type,
			  widget,
			  detail,
			  GTK_ARROW_UP,
			  orientation,
			  0,
			  TOKEN_D_HANDLE);
  if (img)
    {
      gc = style->bg_gc[state_type];
      apply_theme_image(window, img, setbg, gc, area, x, y, width, height, NULL);
    }
}
