/* 
 * VControl Library, Copyright (c) 2003, Michael Martin
 *
 * VControl is distributed under the terms of the BSD license, and as
 * such has NO WARRANTY.  See the LICENSE file for details.
 */

#include <SDL.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "vcontrol.h"

/* If we're in Windows, we don't have strcasecmp */
#ifdef WIN32
#define strcasecmp stricmp
#endif

/* How many binding slots are allocated at once. */
#define POOL_CHUNK_SIZE 64

typedef struct _vcontrol_keybinding {
	int *target;
	struct _vcontrol_keypool *parent;
	struct _vcontrol_keybinding *next;
} keybinding;

typedef struct _vcontrol_keypool {
	keybinding pool[POOL_CHUNK_SIZE];
	int remaining;
	struct _vcontrol_keypool *next;
} keypool;

typedef struct _vcontrol_joystick_axis {
	keybinding *neg, *pos;
	int polarity;
} axis;

typedef struct _vcontrol_joystick_hat {
	keybinding *left, *right, *up, *down;
	Uint8 last;
} hat;

typedef struct _vcontrol_joystick {
	SDL_Joystick *stick;
	int numaxes, numbuttons, numhats;
	int threshold;
	axis *axes;
	keybinding **buttons;
	hat *hats;
} joystick;

static keybinding *bindings[SDLK_LAST];
static joystick *joysticks;
static int joycount;

static keypool *pool;
static VControl_NameBinding *nametable;

static keypool *
_allocate_key_chunk (void)
{
	keypool *x = (keypool *) malloc (sizeof (keypool));
	if (x)
	{
		int i;
		x->remaining = POOL_CHUNK_SIZE;
		x->next = NULL;
		for (i = 0; i < POOL_CHUNK_SIZE; i++)
		{
			x->pool[i].target = NULL;
			x->pool[i].next = NULL;
			x->pool[i].parent = x;
		}
	}
	return x;
}

static void
_free_key_pool (keypool *x)
{
	if (x)
	{
		_free_key_pool (x->next);
		free (x);
	}
}

static void
_create_joystick (int index)
{
	SDL_Joystick *stick;
	int axes, buttons, hats;
	if (index >= joycount)
	{
		fprintf (stderr, "VControl warning: Tried to open a non-existent joystick!");
		return;
	}
	if (joysticks[index].stick)
	{
		// Joystick is already created.  Return.
		return;
	}
	stick = SDL_JoystickOpen (index);
	if (stick)
	{
		joystick *x = &joysticks[index];
		int j;
		fprintf (stderr, "VControl opened joystick: %s\n", SDL_JoystickName (index));
		axes = SDL_JoystickNumAxes (stick);
		buttons = SDL_JoystickNumButtons (stick);
		hats = SDL_JoystickNumHats (stick);
		fprintf (stderr, "%d axes, %d buttons, %d hats.\n", axes, buttons, hats);
		x->numaxes = axes;
		x->numbuttons = buttons;
		x->numhats = hats;
		x->axes = (axis *) malloc (sizeof (axis) * axes);
		x->buttons = (keybinding **) malloc (sizeof (keybinding *) * buttons);
		x->hats = (hat *) malloc (sizeof (hat) * hats);
		for (j = 0; j < axes; j++)
		{
			x->axes[j].neg = x->axes[j].pos = NULL;
		}
		for (j = 0; j < hats; j++)
		{
			x->hats[j].left = x->hats[j].right = NULL;
			x->hats[j].up = x->hats[j].down = NULL;
			x->hats[j].last = SDL_HAT_CENTERED;
		}
		for (j = 0; j < buttons; j++)
		{
			x->buttons[j] = NULL;
		}
		x->stick = stick;
	}
	else
	{
		fprintf (stderr, "VControl: Could not initialize joystick #%d\n", index);
	}
}
			
static void
_destroy_joystick (int index)
{
	SDL_Joystick *stick = joysticks[index].stick;
	if (stick)
	{
		SDL_JoystickClose (stick);
		joysticks[index].stick = NULL;
		free (joysticks[index].axes);
		free (joysticks[index].buttons);
		free (joysticks[index].hats);
		joysticks[index].numaxes = joysticks[index].numbuttons = 0;
		joysticks[index].axes = NULL;
		joysticks[index].buttons = NULL;
		joysticks[index].hats = NULL;
	}
}

static void
_key_init (void)
{
	int i;
	pool = _allocate_key_chunk ();
	for (i = 0; i < SDLK_LAST; i++)
		bindings[i] = NULL;
	/* Prepare for possible joystick controls.  We don't actually
	   GRAB joysticks unless we're asked to make a joystick
	   binding, though. */
	joycount = SDL_NumJoysticks ();
	if (joycount)
	{
		joysticks = (joystick *) malloc (sizeof (joystick) * joycount);
		for (i = 0; i < joycount; i++)
		{
			joysticks[i].stick = NULL;	
			joysticks[i].numaxes = joysticks[i].numbuttons = 0;
			joysticks[i].axes = NULL;
			joysticks[i].buttons = NULL;
		}
	}
	else
	{
		joysticks = NULL;
	}
}

static void
_key_uninit (void)
{
	int i;
	_free_key_pool (pool);
	for (i = 0; i < SDLK_LAST; i++)
		bindings[i] = NULL;
	pool = NULL;
	for (i = 0; i < joycount; i++)
		_destroy_joystick (i);
	free (joysticks);
}

static void
_name_init (void)
{
	nametable = NULL;
}

static void
_name_uninit (void)
{
	nametable = NULL;
}

void
VControl_Init (void)
{
	_key_init ();
	_name_init ();
}

void
VControl_Uninit (void)
{
	_key_uninit ();
	_name_uninit ();
}

int
VControl_SetJoyThreshold (int port, int threshold)
{
	if (port >= 0 && port < joycount)
	{
		joysticks[port].threshold = threshold;
	}
	else
	{
		fprintf (stderr, "VControl_SetJoyThreshold passed illegal port %d\n", port);
		return -1;
	}
	return 0;
}


static void
_add_binding (keybinding **newptr, int *target)
{
	keybinding *newbinding;
	keypool *searchbase;
	int i;

	/* Acquire a pointer to the keybinding * that we'll be
	 * overwriting.  Along the way, ensure we haven't already
	 * bound this symbol to this target.  If we have, return.*/
	while (*newptr != NULL)
	{
		if ((*newptr)->target == target)
		{
			return;
		}
		newptr = &((*newptr)->next);
	}

	/* Now hunt through the binding pool for a free binding. */

	/* First, find a chunk with free spots in it */

	searchbase = pool;
	while (searchbase->remaining == 0)
	{
		/* If we're completely full, allocate a new chunk */
		if (searchbase->next == NULL)
		{
			searchbase->next = _allocate_key_chunk ();
		}
		searchbase = searchbase->next;
	}

	/* Now find a free binding within it */

	newbinding = NULL;
	for (i = 0; i < POOL_CHUNK_SIZE; i++)
	{
		if (searchbase->pool[i].target == NULL)
		{
			newbinding = &searchbase->pool[i];
			break;
		}
	}

	/* Sanity check. */
	if (!newbinding)
	{
		fprintf (stderr, "VControl_AddKeyBinding failed to find a free binding slot!\n");
		return;
	}

	newbinding->target = target;
	newbinding->next = NULL;
	*newptr = newbinding;
	searchbase->remaining--;
}

static void
_remove_binding (keybinding **ptr, int *target)
{
	if (!(*ptr))
	{
		/* Nothing bound to symbol; return. */
		return;
	}
	else if ((*ptr)->target == target)
	{
		keybinding *todel = *ptr;
		*ptr = todel->next;
		todel->target = NULL;
		todel->next = NULL;
		todel->parent->remaining++;
	}
	else
	{
		keybinding *prev = *ptr;
		while (prev->next != NULL)
		{
			if (prev->next->target == target)
			{
				keybinding *todel = prev->next;
				prev->next = todel->next;
				todel->target = NULL;
				todel->next = NULL;
				todel->parent->remaining++;
			}
		}
	}
}

static void
_activate (keybinding *i)
{
	while (i != NULL)
	{
		*(i->target) = *(i->target)+1;
		i = i->next;
	}
}

static void
_deactivate (keybinding *i)
{
	while (i != NULL)
	{
		if (*(i->target) > 0)
		{
			*(i->target) = *(i->target)-1;
		}
		i = i->next;
	}
}

int
VControl_AddBinding (SDL_Event *e, int *target)
{
	int result;
	switch (e->type)
	{
	case SDL_KEYDOWN:
		result = VControl_AddKeyBinding (e->key.keysym.sym, target);
		break;
	case SDL_JOYAXISMOTION:
		result = VControl_AddJoyAxisBinding (e->jaxis.which, e->jaxis.axis, (e->jaxis.value < 0) ? -1 : 1, target);
		break;
	case SDL_JOYHATMOTION:
		result = VControl_AddJoyHatBinding (e->jhat.which, e->jhat.hat, e->jhat.value, target);
		break;
	case SDL_JOYBUTTONDOWN:
		result = VControl_AddJoyButtonBinding (e->jbutton.which, e->jbutton.button, target);
		break;
	default:
		fprintf (stderr, "VControl_AddBinding didn't understand argument event\n");
		result = -1;
		break;
	}
	return result;
}

void
VControl_RemoveBinding (SDL_Event *e, int *target)
{
	switch (e->type)
	{
	case SDL_KEYDOWN:
		VControl_RemoveKeyBinding (e->key.keysym.sym, target);
		break;
	case SDL_JOYAXISMOTION:
		VControl_RemoveJoyAxisBinding (e->jaxis.which, e->jaxis.axis, (e->jaxis.value < 0) ? -1 : 1, target);
		break;
	case SDL_JOYHATMOTION:
		VControl_RemoveJoyHatBinding (e->jhat.which, e->jhat.hat, e->jhat.value, target);
		break;
	case SDL_JOYBUTTONDOWN:
		VControl_RemoveJoyButtonBinding (e->jbutton.which, e->jbutton.button, target);
		break;
	default:
		fprintf (stderr, "VControl_RemoveBinding didn't understand argument event\n");
		break;
	}
}

int
VControl_AddKeyBinding (SDLKey symbol, int *target)
{
	if ((symbol < 0) || (symbol >= SDLK_LAST)) {
		fprintf (stderr, "VControl: Illegal key index %d\n", symbol);
		return -1;
	}
	_add_binding(&bindings[symbol], target);
	return 0;
}

void
VControl_RemoveKeyBinding (SDLKey symbol, int *target)
{
	if ((symbol < 0) || (symbol >= SDLK_LAST)) {
		fprintf (stderr, "VControl: Illegal key index %d\n", symbol);
		return;
	}
	_remove_binding (&bindings[symbol], target);
}

int
VControl_AddJoyAxisBinding (int port, int axis, int polarity, int *target)
{
	if (port >= 0 && port < joycount)
	{
		joystick *j = &joysticks[port];
		if (!(j->stick))
			_create_joystick (port);
		if ((axis >= 0) && (axis < j->numaxes))
		{
			if (polarity < 0)
			{
				_add_binding(&joysticks[port].axes[axis].neg, target);
			}
			else if (polarity > 0)
			{
				_add_binding(&joysticks[port].axes[axis].pos, target);
			}
			else
			{
				fprintf (stderr, "VControl: Attempted to bind to polarity zero\n");
				return -1;
			}
		}
		else
		{
			fprintf (stderr, "VControl: Attempted to bind to illegal axis %d\n", axis);
			return -1;
		}
	}
	else
	{
		fprintf (stderr, "VControl: Attempted to bind to illegal port %d\n", port);
		return -1;
	}
	return 0;
}

void
VControl_RemoveJoyAxisBinding (int port, int axis, int polarity, int *target)
{
	if (port >= 0 && port < joycount)
	{
		joystick *j = &joysticks[port];
		if (!(j->stick))
			_create_joystick (port);
		if ((axis >= 0) && (axis < j->numaxes))
		{
			if (polarity < 0)
			{
				_remove_binding(&joysticks[port].axes[axis].neg, target);
			}
			else if (polarity > 0)
			{
				_remove_binding(&joysticks[port].axes[axis].pos, target);
			}
			else
			{
				fprintf (stderr, "VControl: Attempted to unbind from polarity zero\n");
			}
		}
		else
		{
			fprintf (stderr, "VControl: Attempted to unbind from illegal axis %d\n", axis);
		}
	}
	else
	{
		fprintf (stderr, "VControl: Attempted to unbind from illegal port %d\n", port);
	}
}

int
VControl_AddJoyButtonBinding (int port, int button, int *target)
{
	if (port >= 0 && port < joycount)
	{
		joystick *j = &joysticks[port];
		if (!(j->stick))
			_create_joystick (port);
		if ((button >= 0) && (button < j->numbuttons))
		{
			_add_binding(&joysticks[port].buttons[button], target);
		}
		else
		{
			fprintf (stderr, "VControl: Attempted to bind to illegal button %d\n", button);
			return -1;
		}
	}
	else
	{
		fprintf (stderr, "VControl: Attempted to bind to illegal port %d\n", port);
		return -1;
	}
	return 0;
}

void
VControl_RemoveJoyButtonBinding (int port, int button, int *target)
{
	if (port >= 0 && port < joycount)
	{
		joystick *j = &joysticks[port];
		if (!(j->stick))
			_create_joystick (port);
		if ((button >= 0) && (button < j->numbuttons))
		{
			_remove_binding (&joysticks[port].buttons[button], target);
		}
		else
		{
			fprintf (stderr, "VControl: Attempted to unbind from illegal button %d\n", button);
		}
	}
	else
	{
		fprintf (stderr, "VControl: Attempted to unbind from illegal port %d\n", port);
	}
}

int
VControl_AddJoyHatBinding (int port, int which, Uint8 dir, int *target)
{
	if (port >= 0 && port < joycount)
	{
		joystick *j = &joysticks[port];
		if (!(j->stick))
			_create_joystick (port);
		if ((which >= 0) && (which < j->numhats))
		{
			if (dir == SDL_HAT_LEFT)
			{
				_add_binding(&joysticks[port].hats[which].left, target);
			}
			else if (dir == SDL_HAT_RIGHT)
			{
				_add_binding(&joysticks[port].hats[which].right, target);
			}
			else if (dir == SDL_HAT_UP)
			{
				_add_binding(&joysticks[port].hats[which].up, target);
			}
			else if (dir == SDL_HAT_DOWN)
			{
				_add_binding(&joysticks[port].hats[which].down, target);
			}
			else
			{
				fprintf (stderr, "VControl: Attempted to bind to illegal direction\n");
				return -1;
			}
		}
		else
		{
			fprintf (stderr, "VControl: Attempted to bind to illegal hat %d\n", which);
			return -1;
		}
	}
	else
	{
		fprintf (stderr, "VControl: Attempted to bind to illegal port %d\n", port);
		return -1;
	}
	return 0;
}

void
VControl_RemoveJoyHatBinding (int port, int which, Uint8 dir, int *target)
{
	if (port >= 0 && port < joycount)
	{
		joystick *j = &joysticks[port];
		if (!(j->stick))
			_create_joystick (port);
		if ((which >= 0) && (which < j->numhats))
		{
			if (dir == SDL_HAT_LEFT)
			{
				_remove_binding(&joysticks[port].hats[which].left, target);
			}
			else if (dir == SDL_HAT_RIGHT)
			{
				_remove_binding(&joysticks[port].hats[which].right, target);
			}
			else if (dir == SDL_HAT_UP)
			{
				_remove_binding(&joysticks[port].hats[which].up, target);
			}
			else if (dir == SDL_HAT_DOWN)
			{
				_remove_binding(&joysticks[port].hats[which].down, target);
			}
			else
			{
				fprintf (stderr, "VControl: Attempted to unbind from illegal direction\n");
			}
		}
		else
		{
			fprintf (stderr, "VControl: Attempted to unbind from illegal hat %d\n", which);
		}
	}
	else
	{
		fprintf (stderr, "VControl: Attempted to unbind from illegal port %d\n", port);
	}
}

void
VControl_RemoveAllBindings ()
{
	_key_uninit ();
	_key_init ();
}

void
VControl_ProcessKeyDown (SDLKey symbol)
{
	_activate (bindings[symbol]);
}

void
VControl_ProcessKeyUp (SDLKey symbol)
{
	_deactivate (bindings[symbol]);
}

void
VControl_ProcessJoyButtonDown (int port, int button)
{
	if (!joysticks[port].stick)
		return;
	_activate (joysticks[port].buttons[button]);
}

void
VControl_ProcessJoyButtonUp (int port, int button)
{
	if (!joysticks[port].stick)
		return;
	_deactivate (joysticks[port].buttons[button]);
}

void
VControl_ProcessJoyAxis (int port, int axis, int value)
{
	int t;
	if (!joysticks[port].stick)
		return;
	t = joysticks[port].threshold;
	if (value > t)
	{
		if (joysticks[port].axes[axis].polarity != 1)
		{
			if (joysticks[port].axes[axis].polarity == -1)
			{
				_deactivate (joysticks[port].axes[axis].neg);
			}
			joysticks[port].axes[axis].polarity = 1;
			_activate (joysticks[port].axes[axis].pos);
		}
	}
	else if (value < -t)
	{
		if (joysticks[port].axes[axis].polarity != -1)
		{
			if (joysticks[port].axes[axis].polarity == 1)
			{
				_deactivate (joysticks[port].axes[axis].pos);
			}
			joysticks[port].axes[axis].polarity = -1;
			_activate (joysticks[port].axes[axis].neg);
		}
	}
	else
	{
		if (joysticks[port].axes[axis].polarity == -1)
		{
			_deactivate (joysticks[port].axes[axis].neg);
		}
		else if (joysticks[port].axes[axis].polarity == 1)
		{
			_deactivate (joysticks[port].axes[axis].pos);
		}
		joysticks[port].axes[axis].polarity = 0;
	}
}

void
VControl_ProcessJoyHat (int port, int which, Uint8 value)
{
	Uint8 old;
	if (!joysticks[port].stick)
		return;
	old = joysticks[port].hats[which].last;
	if (!(old & SDL_HAT_LEFT) && (value & SDL_HAT_LEFT))
		_activate (joysticks[port].hats[which].left);
	if (!(old & SDL_HAT_RIGHT) && (value & SDL_HAT_RIGHT))
		_activate (joysticks[port].hats[which].right);
	if (!(old & SDL_HAT_UP) && (value & SDL_HAT_UP))
		_activate (joysticks[port].hats[which].up);
	if (!(old & SDL_HAT_DOWN) && (value & SDL_HAT_DOWN))
		_activate (joysticks[port].hats[which].down);
	if ((old & SDL_HAT_LEFT) && !(value & SDL_HAT_LEFT))
		_deactivate (joysticks[port].hats[which].left);
	if ((old & SDL_HAT_RIGHT) && !(value & SDL_HAT_RIGHT))
		_deactivate (joysticks[port].hats[which].right);
	if ((old & SDL_HAT_UP) && !(value & SDL_HAT_UP))
		_deactivate (joysticks[port].hats[which].up);
	if ((old & SDL_HAT_DOWN) && !(value & SDL_HAT_DOWN))
		_deactivate (joysticks[port].hats[which].down);
	joysticks[port].hats[which].last = value;
}

void
VControl_ResetInput ()
{
	/* Step through every valid entry in the binding pool and zero
	 * them out.  This will probably zero entries multiple times;
	 * oh well, no harm done. */

	keypool *base = pool;
	while (base != NULL)
	{
		int i;
		for (i = 0; i < POOL_CHUNK_SIZE; i++)
		{
			if(base->pool[i].target)
			{
				*(base->pool[i].target) = 0;
			}
		}
		base = base->next;
	}
}

void
VControl_HandleEvent (SDL_Event *e)
{
	switch (e->type)
	{
		case SDL_KEYDOWN:
			VControl_ProcessKeyDown (e->key.keysym.sym);
			break;
		case SDL_KEYUP:
			VControl_ProcessKeyUp (e->key.keysym.sym);
			break;
		case SDL_JOYAXISMOTION:
			VControl_ProcessJoyAxis (e->jaxis.which, e->jaxis.axis, e->jaxis.value);
			break;
		case SDL_JOYHATMOTION:
			VControl_ProcessJoyHat (e->jhat.which, e->jhat.hat, e->jhat.value);
			break;
		case SDL_JOYBUTTONDOWN:
			VControl_ProcessJoyButtonDown (e->jbutton.which, e->jbutton.button);
			break;
		case SDL_JOYBUTTONUP:
			VControl_ProcessJoyButtonUp (e->jbutton.which, e->jbutton.button);
			break;
		default:
			break;
	}
}

void
VControl_RegisterNameTable (VControl_NameBinding *table)
{
	nametable = table;
}
