// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Tools/BinnedHistogram.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/DecayedParticles.hh"

namespace Rivet {


  /// @brief tau -> pi- pi0 nu_tau
  class ARGUS_1994_I375417 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ARGUS_1994_I375417);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      UnstableParticles ufs = UnstableParticles(Cuts::abspid==15);
      declare(ufs, "UFS");
      DecayedParticles TAU(ufs);
      TAU.addStable(310);
      TAU.addStable(111);
      TAU.addStable(221);
      declare(TAU, "TAU");
      // histos
      vector<double> bins = {0.5,.69,.776,.85,1.1};
      for(unsigned int ix=0;ix<4;++ix) {
	Histo1DPtr tmp;
	book(tmp,1,1,1+ix);
	_b.add(bins[ix],bins[ix+1],tmp);
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      static const map<PdgId,unsigned int> & mode   = { { 111,1},{-211,1},{ 16,1}};
      static const map<PdgId,unsigned int> & modeCC = { { 111,1},{ 211,1},{-16,1}};
      DecayedParticles TAU = apply<DecayedParticles>(event, "TAU");
      // loop over particles
      for(unsigned int ix=0;ix<TAU.decaying().size();++ix) {
      	int sign = TAU.decaying()[ix].pid()>0 ? 1 : -1;
      	if(!(TAU.modeMatches(ix,3,mode  ) ||
	     TAU.modeMatches(ix,3,modeCC))) continue;
	const Particle & pi0 = TAU.decayProducts()[ix].at( 111     )[0];
	const Particle & pim = TAU.decayProducts()[ix].at(-211*sign)[0];
	FourMomentum phad = pim.momentum()+pi0.momentum();
	LorentzTransform boost =
	  LorentzTransform::mkFrameTransformFromBeta(TAU.decaying()[ix].momentum().betaVec());
	phad = boost.transform(phad);
	LorentzTransform boost2 = LorentzTransform::mkFrameTransformFromBeta(phad.betaVec());
	FourMomentum ppi = boost2.transform(boost.transform(pim.momentum()));
	double cAlpha = ppi.p3().unit().dot(phad.p3().unit());
	_b.fill(phad.mass(),cAlpha);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for(const Histo1DPtr & h : _b.histos())
	normalize(h,1.,false);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistogram _b;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ARGUS_1994_I375417);

}
