/* $Id: file.c,v 1.1 2011/06/01 02:42:25 ve3wwg Exp $
 * Warren W. Gay VE3WWG		Sun Feb 16 20:43:59 1997
 *
 * WAV FILE OPERATIONS:
 *
 * 	X LessTif WAV Play :
 * 
 * 	Copyright (C) 1997  Warren W. Gay VE3WWG
 * 
 * This  program is free software; you can redistribute it and/or modify it
 * under the  terms  of  the GNU General Public License as published by the
 * Free Software Foundation.
 * 
 * This  program  is  distributed  in  the hope that it will be useful, but
 * WITHOUT   ANY   WARRANTY;   without   even  the   implied   warranty  of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details (licensed by file COPYING or GPLv*).
 */
static const char rcsid[] = "$Id: file.c,v 1.1 2011/06/01 02:42:25 ve3wwg Exp $";

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#ifdef HAVE_OS_BSD
#include <malloc.h>
#endif

#include <string.h>
#include <memory.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/ioctl.h>
#include <assert.h>

#ifdef HAVE_OS_LINUX
#include <linux/soundcard.h>
#endif
#ifdef HAVE_OS_BSD
#include <machine/soundcard.h>
#endif

#include "wavplay.h"

extern int clntIPC;    /* Needed for message passing in PlayDSP */

/* Number of updates sent to the client per second during playback and */
/* record.  For high sample rates in stereo, too many per second floods the */
/* client.  3 is a compromise between update speed and smoothness; 4 tends */
/* to cause "jumpy" updates.  (The cause of the jumps probably lie in */
/* LessTif and/or X since the IPC messages are sent regularly). */
#define RECPLAY_UPDATES_PER_SEC                3

static ErrFunc v_erf;				/* This module's error reporting function */
static char emsg[2048];

/*
 * Error reporting function for this source module:
 */
static void
err(const char *format,...) {
	va_list ap;

	if ( v_erf == NULL )
		return;				/* Only report error if we have function */
	va_start(ap,format);
	v_erf(format,ap);			/* Use caller's supplied function */
	va_end(ap);
}

static void
_v_erf(const char *format,va_list ap) {
	vsprintf(emsg,format,ap);		/* Capture message into emsg[] */
}

/*
 * Internal routine to allocate WAVFILE structure:
 */
static WAVFILE *
wavfile_alloc(const char *Pathname) {
	WAVFILE *wfile = (WAVFILE *) malloc(sizeof (WAVFILE));

	if ( wfile == NULL ) {
		err("%s: Allocating WAVFILE structure",strerror(ENOMEM));
		return NULL;
	}

	memset(wfile,0,sizeof *wfile);

	if ( (wfile->Pathname = strdup(Pathname)) == NULL ) {
		free(wfile);
		err("%s: Allocating storage for WAVFILE.Pathname",strerror(ENOMEM));
		return NULL;
	}

	wfile->fd = -1;				/* Initialize fd as not open */
	wfile->wavinfo.Channels = Mono;
	wfile->wavinfo.DataBits = 8;

	return wfile;
}

/*
 * Internal routine to release WAVFILE structure:
 * No errors reported.
 */
static void
wavfile_free(WAVFILE *wfile) {
	if ( wfile->Pathname != NULL )
		free(wfile->Pathname);
	free(wfile);
}

/*
 * Open a WAV file for reading: returns (WAVFILE *)
 *
 * The opened file is positioned at the first byte of WAV file data, or
 * NULL is returned if the open is unsuccessful.
 */
WAVFILE *
WavOpenForRead(const char *Pathname,ErrFunc erf) {
	WAVFILE *wfile = wavfile_alloc(Pathname);
	int e;					/* Saved errno value */
	UInt32 offset;				/* File offset */
	Byte ubuf[4];				/* 4 byte buffer */
	UInt32 dbytes;				/* Data byte count */
						/* wavfile.c values : */
	int channels;				/* Channels recorded in this wav file */
	uint32_t samplerate;			/* Sampling rate */
	int sample_bits;			/* data bit size (8/12/16) */
	uint32_t samples;				/* The number of samples in this file */
	uint32_t datastart;			/* The offset to the wav data */

	v_erf = erf;				/* Set error reporting function */

	if ( wfile == NULL )
		return NULL;			/* Insufficient memory (class B msg) */

	/*
	 * Open the file for reading:
	 */
	if ( (wfile->fd = open(wfile->Pathname,O_RDONLY)) < 0 ) {
		err("%s:\nOpening WAV file %s",
			strerror(errno),
			wfile->Pathname);
		goto errxit;
	}

	if ( lseek(wfile->fd,0L,SEEK_SET) != 0L ) {
		err("%s:\nRewinding WAV file %s",
			strerror(errno),
			wfile->Pathname);
		goto errxit;		/* Wav file must be seekable device */
	}

	if ( (e = WaveReadHeader(wfile->fd,&channels,&samplerate,&sample_bits,&samples,&datastart,_v_erf)) != 0 ) {
		err("%s:\nReading WAV header from %s",
			emsg,
			wfile->Pathname);
		goto errxit;
	}

	/*
	 * Copy WAV data over to WAVFILE struct:
	 */
	if ( channels == 2 )
		wfile->wavinfo.Channels = Stereo;
	else	wfile->wavinfo.Channels = Mono;

	wfile->wavinfo.SamplingRate = (UInt32) samplerate;
	wfile->wavinfo.Samples = (UInt32) samples;
	wfile->wavinfo.DataBits = (UInt16) sample_bits;
	wfile->wavinfo.DataStart = (UInt32) datastart;
        wfile->num_samples = wfile->wavinfo.Samples;
	wfile->rw = 'R';			/* Read mode */

	offset = wfile->wavinfo.DataStart - 4;

	/*
	 * Seek to byte count and read dbytes:
	 */
	if ( lseek(wfile->fd,offset,SEEK_SET) != offset ) {
		err("%s:\nSeeking to WAV data in %s",strerror(errno),wfile->Pathname);
		goto errxit;			/* Seek failure */
	}

	if ( read(wfile->fd,ubuf,4) != 4 ) {
		err("%s:\nReading dbytes from %s",strerror(errno),wfile->Pathname);
		goto errxit;
	}

	/*
	 * Put little endian value into 32 bit value:
	 */
	dbytes = ubuf[3];
	dbytes = (dbytes << 8) | ubuf[2];
	dbytes = (dbytes << 8) | ubuf[1];
	dbytes = (dbytes << 8) | ubuf[0];

	wfile->wavinfo.DataBytes = dbytes;

	/*
	 * Open succeeded:
	 */
	return wfile;				/* Return open descriptor */

	/*
	 * Return error after failed open:
	 */
errxit:	e = errno;				/* Save errno */
	free(wfile->Pathname);			/* Dispose of copied pathname */
	free(wfile);				/* Dispose of WAVFILE struct */
	errno = e;				/* Restore error number */
	return NULL;				/* Return error indication */
}

/*
 * Apply command line option overrides to the interpretation of the input
 * wav file:
 *
 */
void
WavReadOverrides(WAVFILE *wfile,WavPlayOpts *wavopts) {
	UInt32 num_samples;

	/*
	 * Override sampling rate: -s sampling_rate
	 */
	if ( wavopts->SamplingRate.optChar != 0 ) {
		wfile->wavinfo.SamplingRate = wavopts->SamplingRate.optValue;
		wfile->wavinfo.bOvrSampling = 1;
	}

	/*
	 * Override mono/stereo mode: -S / -M
	 */
	if ( wavopts->Channels.optChar != 0 ) {
		wfile->wavinfo.Channels = wavopts->Channels.optValue;
		wfile->wavinfo.bOvrMode = 1;
	}
	
	/*
	 * Override the sample size in bits: -b bits
	 */
	if ( wavopts->DataBits.optChar != 0 ) {
		wfile->wavinfo.DataBits = wavopts->DataBits.optValue;
		wfile->wavinfo.bOvrBits = 1;
	}

        /*
         * Set the first sample:
         */
        wfile->StartSample = 0;
        num_samples = wfile->wavinfo.Samples = wfile->num_samples;
        if ( wavopts->StartSample != 0 ) {
                wfile->StartSample = wavopts->StartSample;
                wfile->wavinfo.Samples -= wfile->StartSample;
        }
 
        /*
         * Override # of samples if -t seconds option given:
         */
        if ( wavopts->Seconds != 0 ) {
                wfile->wavinfo.Samples = wavopts->Seconds * wfile->wavinfo.SamplingRate;
                if (wfile->StartSample+wfile->wavinfo.Samples > num_samples)
                        wfile->wavinfo.Samples = num_samples-1;
        }
}

/*
 * Close a WAVFILE
 */
int
WavClose(WAVFILE *wfile,ErrFunc erf) {
	int e = 0;				/* Returned error code */
	int channels;				/* Channels recorded in this wav file */
	uint32_t samplerate;			/* Sampling rate */
	int sample_bits;			/* data bit size (8/12/16) */
	uint32_t samples;				/* The number of samples in this file */
	uint32_t datastart;			/* The offset to the wav data */
	long fpos;				/* File position in bytes */

	v_erf = erf;				/* Set error reporting function */

	if ( wfile == NULL ) {
		err("%s: WAVFILE pointer is NULL!",strerror(EINVAL));
		errno = EINVAL;
		return -1;
	}

	/*
	 * If the wav file was open for write, update the actual number
	 * of samples written to the file:
	 */
	if ( wfile->rw == 'W' ) {
		fpos = lseek(wfile->fd,0L,SEEK_CUR);	/* Get out file position */
		if ( (e = WaveReadHeader(wfile->fd,&channels,&samplerate,&sample_bits,&samples,&datastart,_v_erf)) != 0 )
			err("%s:\nReading WAV header from %s",emsg,wfile->Pathname);
		else if ( lseek(wfile->fd,(long)(datastart-4),SEEK_SET) != (long)(datastart-4) )
			err("%s:\nSeeking in WAV header file %s",strerror(errno),wfile->Pathname);
		else if ( write(wfile->fd,&wfile->wavinfo.Samples,sizeof wfile->wavinfo.Samples) != sizeof wfile->wavinfo.Samples )
			err("%s:\nWriting in WAV header file %s",strerror(errno),wfile->Pathname);
		else	{
			/*
			 * 'data' chunk was updated OK: Now we have to update the RIFF block
			 * count. Someday very soon, a real RIFF module is going to replace
			 * this fudging.
			 */
			if ( ftruncate(wfile->fd,(size_t)fpos) )
				err("%s:\nTruncating file %s to correct size",
					strerror(errno),
					wfile->Pathname);
			else if ( lseek(wfile->fd,4L,SEEK_SET) < 0L )
				err("%s:\nSeek 4 for RIFF block update of %s",
					strerror(errno),
					wfile->Pathname);
			else	{
				fpos -= 8;		/* Byte count for RIFF block */
				if ( write(wfile->fd,&fpos,sizeof fpos) != sizeof fpos )
					err("%s:\nUpdate of RIFF block count in %s failed",
						strerror(errno),
						wfile->Pathname);
			}
		}
	}

	if ( close(wfile->fd) < 0 ) {
		err("%s:\nClosing WAV file",strerror(errno));
		e = errno;			/* Save errno value to return */
	}

	wavfile_free(wfile);			/* Release WAVFILE structure */

	if ( (errno = e) != 0 )
		return -1;			/* Failed exit */
	return 0;				/* Successful exit */
}

/*
 * Open a WAV file for writing:
 */
WAVFILE *
WavOpenForWrite(const char *Pathname,Chan chmode,UInt32 sample_rate,UInt16 bits,UInt32 samples,ErrFunc erf) {
	WAVFILE *wfile = wavfile_alloc(Pathname);
	int e;					/* Saved errno value */

	v_erf = erf;				/* Set error reporting function */

	if ( wfile == NULL )
		return NULL;			/* ENOMEM (class_b msg) */

	wfile->rw = 'W';			/* Mark as for writing */
	wfile->wavinfo.SamplingRate = sample_rate;
	wfile->wavinfo.Channels = chmode;	/* Mono / Stereo */
	wfile->wavinfo.Samples = samples;
	wfile->wavinfo.DataBits = bits;

	/*
	 * Open/create the file for writing:
	 */
	if ( (wfile->fd = open(wfile->Pathname,O_RDWR|O_TRUNC|O_CREAT,0666)) < 0 ) {
		err("%s:\nOpening %s for WAV writing",
			strerror(errno),
			wfile->Pathname);
		return NULL;			/* Open error */
	}

	/*
	 * Write out a WAV file header:
	 */
	e = WaveWriteHeader(wfile->fd,
		wfile->wavinfo.Channels == Mono ? 1 : 2,
		wfile->wavinfo.SamplingRate,
		wfile->wavinfo.DataBits,
		wfile->wavinfo.Samples,
		_v_erf);			/* Capture error messages to emsg[] */

	if ( e != 0 ) {
		wavfile_free(wfile);
		err("%s:\nWriting WAV header to %s",
			emsg,
			wfile->Pathname);
		return NULL;
	}

	/*
	 * Return successfuly opened file:
	 */
	return wfile;
}

/*
 * Open /dev/dsp for reading or writing:
 */
DSPFILE *
OpenDSP(WAVFILE *wfile,int omode,ErrFunc erf) {
	int e;					/* Saved errno value */
	int t;					/* Work int */
	unsigned long ul;			/* Work unsigned long */
	DSPFILE *dfile = (DSPFILE *) malloc(sizeof (DSPFILE));

	v_erf = erf;				/* Set error reporting function */

	if ( dfile == NULL ) {
		err("%s: Opening DSP device",strerror(errno=ENOMEM));
		return NULL;
	}

	memset(dfile,0,sizeof *dfile);
	dfile->dspbuf = NULL;

	/*
	 * Open the device driver:
	 */
	if ( (dfile->fd = open(env_AUDIODEV,omode,0)) < 0 ) {
		err("%s:\nOpening audio device %s",
			strerror(errno),
			env_AUDIODEV);
		goto errxit;
	}

        /*
         * Determine the audio device's block size.  Should be done after
         * setting sampling rate etc.
         */
        if ( ioctl(dfile->fd,SNDCTL_DSP_GETBLKSIZE,&dfile->dspblksiz) < 0 ) {
                err("%s: Optaining DSP's block size",strerror(errno));
                goto errxit;
        }
 
        /*
         * Check the range on the buffer sizes:
         */
        /* Minimum was 4096 but es1370 returns 1024 for 44.1kHz, 16 bit */
        /* and 64 for 8130Hz, 8 bit */
        if ( dfile->dspblksiz < 32 || dfile->dspblksiz > 65536 ) {
                err("%s: Audio block size (%d bytes)",
                        strerror(errno=EINVAL),
                        (int)dfile->dspblksiz);
                goto errxit;
        }
 
        /*
         * Allocate a buffer to do the I/O through:
         */
        if ( (dfile->dspbuf = (char *) malloc(dfile->dspblksiz)) == NULL ) {
                err("%s: For DSP I/O buffer",strerror(errno));
                goto errxit;
        }

	/*
	 * Set the data bit size:
	 */
	t = wfile->wavinfo.DataBits;
        if ( ioctl(dfile->fd,SNDCTL_DSP_SAMPLESIZE,&t) < 0 ) {
		err("%s: Setting DSP to %u bits",strerror(errno),(unsigned)t);
		goto errxit;
	}

	/*
	 * Set the mode to be Stereo or Mono:
	 */
	t = wfile->wavinfo.Channels == Stereo ? 1 : 0;
	if ( ioctl(dfile->fd,SNDCTL_DSP_STEREO,&t) < 0 ) {
		err("%s: Unable to set DSP to %s mode",
			strerror(errno),
			t?"Stereo":"Mono");
		goto errxit;
	}		
      
	/*
	 * Set the sampling rate:
	 */
	ul = wfile->wavinfo.SamplingRate;
	if ( ioctl(dfile->fd,SNDCTL_DSP_SPEED,&ul) < 0 ) {
		err("Unable to set audio sampling rate",strerror(errno));
		goto errxit;
	}

	/*
	 * Return successfully opened device:
	 */
	return dfile;				/* Return file descriptor */

	/*
	 * Failed to open/initialize properly:
	 */
errxit:	e = errno;				/* Save the errno value */
	if ( dfile->fd >= 0 )
		close(dfile->fd);		/* Close device */
	if ( dfile->dspbuf != NULL )
		free(dfile->dspbuf);
	free(dfile);
	errno = e;				/* Restore error code */
	return NULL;				/* Return error indication */
}

/*
 * Close the DSP device:
 */
int
CloseDSP(DSPFILE *dfile,ErrFunc erf) {
	int fd;

	v_erf = erf;				/* Set error reporting function */

	if ( dfile == NULL ) {
		err("%s: DSPFILE is not open",strerror(errno=EINVAL));
		return -1;
	}

	fd = dfile->fd;
	if ( dfile->dspbuf != NULL )
		free(dfile->dspbuf);
	free(dfile);
	
	if ( close(fd) ) {
		err("%s: Closing DSP fd %d",strerror(errno),fd);
		return -1;
	}

	return 0;
}

/*
 * Play DSP from WAV file:
 */
int
PlayDSP(DSPFILE *dfile,WAVFILE *wfile,DSPPROC work_proc,ErrFunc erf) {
	UInt32 byte_count = (UInt32) wfile->wavinfo.Samples;
	int bytes;
	int n;
	int byte_modulo;
        int total_bytes, update_bytes;
        SVRMSG msg;

	v_erf = erf;				/* Set error reporting function */

	/*
	 * Check that the WAVFILE is open for reading:
	 */
	if ( wfile->rw != 'R' ) {
		err("%s: WAVFILE must be open for reading",strerror(errno=EINVAL));
		return -1;
	}

	/*
	 * First determine how many bytes are required for each channel's sample:
	 */
	switch ( wfile->wavinfo.DataBits ) {
	case 8 :
		byte_count = 1;
		break;
	case 16 :
		byte_count = 2;
		break;
	default :
		err("%s: Cannot process %u bit samples",
			strerror(errno=EINVAL),
			(unsigned)wfile->wavinfo.DataBits);
		return -1;
	}

	/*
	 * Allow for Mono/Stereo difference:
	 */
	if ( wfile->wavinfo.Channels == Stereo )
		byte_count *= 2;		/* Twice as many bytes for stereo */
	else if ( wfile->wavinfo.Channels != Mono ) {
		err("%s: DSPFILE control block is corrupted (chan_mode)",
			strerror(errno=EINVAL));
		return -1;
	}		

	byte_modulo = byte_count;				/* This many bytes per sample */
	byte_count  = wfile->wavinfo.Samples * byte_modulo;	/* Total bytes to process */
        total_bytes = byte_count;
 
        /* Number of bytes to write between client updates.  Must be */
        /* a multiple of dspblksiz.  */
        update_bytes = ((wfile->wavinfo.SamplingRate*byte_modulo) / (RECPLAY_UPDATES_PER_SEC*dfile->dspblksiz)) * dfile->dspblksiz;

	if ( ioctl(dfile->fd,SNDCTL_DSP_SYNC,0) != 0 )
		err("%s: ioctl(%d,SNDCTL_DSP_SYNC,0)",strerror(errno));

        /* Seek to requested start sample */
        lseek(wfile->fd,wfile->StartSample*byte_modulo,SEEK_CUR);
 
	for ( ; byte_count > 0 && wfile->wavinfo.DataBytes > 0; byte_count -= (UInt32) n ) {

		bytes = (int) ( byte_count > dfile->dspblksiz ? dfile->dspblksiz : byte_count );

		if ( bytes > wfile->wavinfo.DataBytes )	/* Size bigger than data chunk? */
			bytes = wfile->wavinfo.DataBytes;	/* Data chunk only has this much left */

		if ( (n = read(wfile->fd,dfile->dspbuf,bytes)) != bytes ) {
			if ( n >= 0 )
				err("Unexpected EOF reading samples from WAV file",strerror(errno=EIO));
			else	err("Reading samples from WAV file",strerror(errno));
			goto errxit;
		}

                if ((clntIPC >= 0) && !((total_bytes-byte_count) % update_bytes)) {
                        msg.msg_type = ToClnt_PlayState;
                        msg.bytes = sizeof(msg.u.toclnt_playstate);
                        msg.u.toclnt_playstate.SamplesLeft = byte_count / byte_modulo;
                        msg.u.toclnt_playstate.CurrentSample =
                          wfile->num_samples - msg.u.toclnt_playstate.SamplesLeft;
                        MsgToClient(clntIPC,&msg,0);    /* Tell client playback status */
                }

		if ( write(dfile->fd,dfile->dspbuf,n) != n ) {
			err("Writing samples to audio device",strerror(errno));
			goto errxit;
		}

		wfile->wavinfo.DataBytes -= (UInt32) bytes;	/* We have fewer bytes left to read */

		/*
		 * The work procedure function is called when operating
		 * in server mode to check for more server messages:
		 */
		if ( work_proc != NULL && work_proc(dfile) )	/* Did work_proc() return TRUE? */
			break;					/* Yes, quit playing */
	}
			
#if 0	/* I think this is doing a destructive flush: disabled */
	if ( ioctl(dfile->fd,SNDCTL_DSP_SYNC,0) != 0 )
		err("%s: ioctl(%d,SNDCTL_DSP_SYNC,0)",strerror(errno));
#endif
        /* Update client time display at end of sucessful play */
        if (clntIPC >= 0) {
                msg.msg_type = ToClnt_PlayState;
                msg.bytes = sizeof(msg.u.toclnt_playstate);
                msg.u.toclnt_playstate.SamplesLeft = byte_count / byte_modulo;
                msg.u.toclnt_playstate.CurrentSample =
                  wfile->num_samples - msg.u.toclnt_playstate.SamplesLeft;
                MsgToClient(clntIPC,&msg,0);    /* Tell client playback status */
        }
	return 0;	/* All samples played successfully */

errxit:	return -1;	/* Indicate error return */
}

/*
 * Record DSP to WAV file: If samples == 0UL, then record continues until
 * a bRecordStopPosted becomes true (via SIGINT).
 */
int
RecordDSP(DSPFILE *dfile,WAVFILE *wfile,UInt32 samples,DSPPROC work_proc,ErrFunc erf) {
	UInt32 byte_count = (UInt32) wfile->wavinfo.Samples;
	UInt32 chunk;
	int bytes;
	int n;
	UInt32 bytes_per_sample = 0;
	UInt32 bytes_written = 0;
        int update_bytes;
        SVRMSG msg;

	v_erf = erf;				/* Set error reporting function */

	/*
	 * Check that the WAVFILE is open for writing:
	 */
	if ( wfile->rw != 'W' ) {
		err("WAVFILE must be open for writing",strerror(errno=EINVAL));
		return -1;
	}

	/*
	 * First determine how many bytes are required for each channel's sample:
	 */
	switch ( wfile->wavinfo.DataBits ) {
	case 8 :
		byte_count = 1;
		break;
	case 12 :
	case 16 :
		byte_count = 2;
		break;
	default :
		err("Cannot process %u bit samples",
			strerror(errno=EINVAL),
			(unsigned)wfile->wavinfo.DataBits);
		return -1;
	}

	/*
	 * Allow for Mono/Stereo difference:
	 */
	if ( wfile->wavinfo.Channels == Stereo )
		byte_count *= 2;		/* Twice as many bytes for stereo */
	else if ( wfile->wavinfo.Channels != Mono ) {
		err("DSPFILE control block is corrupted (chan_mode)",strerror(errno=EINVAL));
		return -1;
	}		

	bytes_per_sample = byte_count;		/* Save for close later */

	if ( samples > 0 )
		byte_count *= wfile->wavinfo.Samples;	/* Total number of bytes to collect */
	else	{
		/* Compute a smallish sized chunk of the correct modulo bytes */
		chunk = dfile->dspblksiz > 4096 ? 4096 : dfile->dspblksiz;
		byte_count = (chunk + (byte_count-1)) & ~(byte_count-1);
	}

        /* Number of bytes to write between client updates.  Must be */
        /* a multiple of dspblksiz. */
        update_bytes = ((wfile->wavinfo.SamplingRate*bytes_per_sample)/
            (RECPLAY_UPDATES_PER_SEC*dfile->dspblksiz))*dfile->dspblksiz;
 
	while ( 1 ) {
		/*
		 * Determine how many samples to read:
		 */
		if ( samples > 0 )
			bytes = byte_count > dfile->dspblksiz ? dfile->dspblksiz : byte_count;
		else	bytes = byte_count;	/* Record until interrupted */

		/*
		 * Read a block of samples:
		 */
		if ( (n = read(dfile->fd,dfile->dspbuf,bytes)) < 0 ) {
			err("Reading DSP device",strerror(errno));
			goto errxit;
		} else if ( n == 0 )
			break;
			
		/*
		 * Write a block of samples to the file:
		 */
		if ( (bytes = write(wfile->fd,dfile->dspbuf,n)) < 0 ) {
			err("Writing WAV samples to %s",strerror(errno),wfile->Pathname);
			goto errxit;
		} else if ( bytes != n ) {
			if ( bytes > 0 )
				bytes_written += bytes;
			err("Did not write all WAV successfully",strerror(errno=EIO));
			goto errxit;
		}

		bytes_written += bytes;

                if ((clntIPC >= 0) && !(bytes_written % update_bytes)) {
                        msg.msg_type = ToClnt_RecState;
                        msg.bytes = sizeof(msg.u.toclnt_recstate);
                        msg.u.toclnt_recstate.bytes_written = bytes_written;
                        msg.u.toclnt_recstate.num_samples = bytes_written / bytes_per_sample;
                        MsgToClient(clntIPC,&msg,0);    /* Tell client playback status */
                }

		if ( samples > 0 )
			if ( (byte_count -= (unsigned) n) < 1 )
				break;

		/*
		 * The work procedure function is called when operating
		 * in server mode to check for more server messages:
		 */
		if ( work_proc != NULL && work_proc(dfile) )	/* Did work_proc() return TRUE? */
			break;					/* Yes, quit recording */
	}

	wfile->wavinfo.Samples = bytes_written & ~(bytes_per_sample-1);
	return 0;	/* All samples played successfully */

errxit:	wfile->wavinfo.Samples = bytes_written & ~(bytes_per_sample-1);
	return -1;	/* Indicate error return */
}

/* $Source: /cvsroot/wavplay/code/src/file.c,v $ */
