/*
 * thermocouple.c
 *
 *  Created on: 6 Apr 2012
 *      Author: sjm
 */

#include <stdlib.h>
#include <math.h>

#include "thermocouple.h"

//#define MAXVTCOEFF 11
#define MAXTVCOEFF 15
#define MAXEXPCOEFF 3
#define NVTCOEFFSET 23
#define NTVCOEFFSET 18

//typedef struct {
//  int    type; // Thermocouple type
//  double minV; // Minimum Voltage for this set
//  double maxV; // Maximum Voltage for this set
//  double coeff[MAXVTCOEFF]; // Coefficient array
//  int    Ncoeff; // No. coefficients used
//} vtcoeff_t ;
//
typedef struct {
  int    type; // Thermocouple type
  double min; // Minimum x for this set
  double max; // Maximum x for this set
  double coeff[MAXTVCOEFF]; // Coefficient array
  double expc[MAXEXPCOEFF]; // Coefficient array - exp term
  int    Nexp; // No. of exp coefficients used
  int    Ncoeff; // No. coefficients used
} coeff_t ;

#define NOEXP {0.0, 0.0, 0.0}

/*
 * Types are:
 *
 * B -> 0
 * E -> 1
 * J -> 2
 * K -> 3
 * N -> 4
 * R -> 5
 * S -> 6
 * T -> 7
 */

coeff_t vtcoeff_set[] = {
/*
 *
 *
 * Inverse coefficients for type B:

Temperature   250.           700.
  Range:      700.          1820.

  Voltage    0.291          2.431
  Range:     2.431         13.820

         9.8423321E+01  2.1315071E+02
         6.9971500E+02  2.8510504E+02
        -8.4765304E+02 -5.2742887E+01
         1.0052644E+03  9.9160804E+00
        -8.3345952E+02 -1.2965303E+00
         4.5508542E+02  1.1195870E-01
        -1.5523037E+02 -6.0625199E-03
         2.9886750E+01  1.8661696E-04
        -2.4742860E+00 -2.4878585E-06

  Error      -0.02          -0.01
  Range:      0.03           0.02
 */
	{0, 0.291, 2.431, { 9.8423321E+01,  6.9971500E+02, -8.4765304E+02,  1.0052644E+03, -8.3345952E+02,  4.5508542E+02, -1.5523037E+02,  2.9886750E+01, -2.4742860E+00}, NOEXP, 0, 9},
	{0, 2.431, 13.820, { 2.1315071E+02,  2.8510504E+02, -5.2742887E+01,  9.9160804E+00, -1.2965303E+00,  1.1195870E-01, -6.0625199E-03,  1.8661696E-04, -2.4878585E-06}, NOEXP, 0, 9},
 /*
 * Inverse coefficients for type E:

Temperature  -200.             0.
  Range:        0.          1000.

  Voltage   -8.825          0.000
  Range:     0.000         76.373

         0.0000000E+00  0.0000000E+00
         1.6977288E+01  1.7057035E+01
        -4.3514970E-01 -2.3301759E-01
        -1.5859697E-01  6.5435585E-03
        -9.2502871E-02 -7.3562749E-05
        -2.6084314E-02 -1.7896001E-06
        -4.1360199E-03  8.4036165E-08
        -3.4034030E-04 -1.3735879E-09
        -1.1564890E-05  1.0629823E-11
         0.0000000E+00 -3.2447087E-14

  Error      -0.01          -0.02
  Range:      0.03           0.02
 */
	{1, -8.825, 0.000, { 0.0000000E+00,  1.6977288E+01, -4.3514970E-01, -1.5859697E-01, -9.2502871E-02, -2.6084314E-02, -4.1360199E-03, -3.4034030E-04, -1.1564890E-05,  0.0000000E+00}, NOEXP, 0, 9},
	{1, 0.000, 76.373, { 0.0000000E+00,  1.7057035E+01, -2.3301759E-01,  6.5435585E-03, -7.3562749E-05, -1.7896001E-06,  8.4036165E-08, -1.3735879E-09,  1.0629823E-11, -3.2447087E-14}, NOEXP, 0, 10},

 /*
 * Inverse coefficients for type J:

Temperature  -210.             0.           760.
  Range:        0.           760.          1200.

  Voltage   -8.095          0.000         42.919
  Range:     0.000         42.919         69.553

         0.0000000E+00  0.000000E+00 -3.11358187E+03
         1.9528268E+01  1.978425E+01  3.00543684E+02
        -1.2286185E+00 -2.001204E-01 -9.94773230E+00
        -1.0752178E+00  1.036969E-02  1.70276630E-01
        -5.9086933E-01 -2.549687E-04 -1.43033468E-03
        -1.7256713E-01  3.585153E-06  4.73886084E-06
        -2.8131513E-02 -5.344285E-08  0.00000000E+00
        -2.3963370E-03  5.099890E-10  0.00000000E+00
        -8.3823321E-05  0.000000E+00  0.00000000E+00

  Error      -0.05          -0.04          -0.04
  Range:      0.03           0.04           0.03
 */
	{2, -8.095, 0.000, { 0.0000000E+00,  1.9528268E+01, -1.2286185E+00, -1.0752178E+00, -5.9086933E-01, -1.7256713E-01, -2.8131513E-02, -2.3963370E-03, -8.3823321E-05}, NOEXP, 0, 9},
	{2, 0.000, 42.919,{ 0.000000E+00,  1.978425E+01, -2.001204E-01,  1.036969E-02, -2.549687E-04,  3.585153E-06, -5.344285E-08,  5.099890E-10}, NOEXP, 0, 8},
	{2, 42.919, 69.553, {-3.11358187E+03,  3.00543684E+02, -9.94773230E+00,  1.70276630E-01, -1.43033468E-03,  4.73886084E-06}, NOEXP, 0, 6},

/*
 * Inverse coefficients for type K:

Temperature  -200.             0.           500.
  Range:        0.           500.          1372.

  Voltage   -5.891          0.000         20.644
  Range:     0.000         20.644         54.886

         0.0000000E+00  0.000000E+00 -1.318058E+02
         2.5173462E+01  2.508355E+01  4.830222E+01
        -1.1662878E+00  7.860106E-02 -1.646031E+00
        -1.0833638E+00 -2.503131E-01  5.464731E-02
        -8.9773540E-01  8.315270E-02 -9.650715E-04
        -3.7342377E-01 -1.228034E-02  8.802193E-06
        -8.6632643E-02  9.804036E-04 -3.110810E-08
        -1.0450598E-02 -4.413030E-05  0.000000E+00
        -5.1920577E-04  1.057734E-06  0.000000E+00
         0.0000000E+00 -1.052755E-08  0.000000E+00

  Error      -0.02          -0.05          -0.05
  Range:      0.04           0.04           0.06
 */
	{3, -5.891, 0.000, { 0.0000000E+00,  2.5173462E+01, -1.1662878E+00, -1.0833638E+00, -8.9773540E-01, -3.7342377E-01, -8.6632643E-02, -1.0450598E-02, -5.1920577E-04}, NOEXP, 0, 9},
	{3, 0.000, 20.644, { 0.000000E+00,  2.508355E+01,  7.860106E-02, -2.503131E-01,  8.315270E-02, -1.228034E-02,  9.804036E-04, -4.413030E-05,  1.057734E-06, -1.052755E-08}, NOEXP, 0, 10},
	{3, 20.644, 54.886, {-1.318058E+02,  4.830222E+01, -1.646031E+00,  5.464731E-02, -9.650715E-04,  8.802193E-06, -3.110810E-08}, NOEXP, 0, 7},
 /*
 * Inverse coefficients for type N:

Temperature  -200.             0.           600.
  Range:        0.           600.          1300.

  Voltage   -3.990          0.000         20.613
  Range:     0.000         20.613         47.513

         0.0000000E+00  0.00000E+00  1.972485E+01
         3.8436847E+01  3.86896E+01  3.300943E+01
         1.1010485E+00 -1.08267E+00 -3.915159E-01
         5.2229312E+00  4.70205E-02  9.855391E-03
         7.2060525E+00 -2.12169E-06 -1.274371E-04
         5.8488586E+00 -1.17272E-04  7.767022E-07
         2.7754916E+00  5.39280E-06  0.000000E+00
         7.7075166E-01 -7.98156E-08  0.000000E+00
         1.1582665E-01  0.00000E+00  0.000000E+00
         7.3138868E-03  0.00000E+00  0.000000E+00

  Error      -0.02          -0.02          -0.04
  Range:      0.03           0.03           0.02
 */
	{4, -3.990, 0.000, {0.0000000E+00, 3.8436847E+01, 1.1010485E+00, 5.2229312E+00, 7.2060525E+00, 5.8488586E+00, 2.7754916E+00, 7.7075166E-01, 1.1582665E-01, 7.3138868E-03}, NOEXP, 0, 10},
	{4, 0.000, 20.613, {0.00000E+00, 3.86896E+01, -1.08267E+00, 4.70205E-02, -2.12169E-06, -1.17272E-04, 5.39280E-06, -7.98156E-08}, NOEXP, 0, 8},
	{4, 20.613, 47.513, {1.972485E+01, 3.300943E+01, -3.915159E-01, 9.855391E-03, -1.274371E-04, 7.767022E-07}, NOEXP, 0, 6},

 /*
 * Inverse coefficients for type R:

Temperature   -50.           250.          1064.          1664.5
  Range:      250.          1200.          1664.5         1768.1

  Voltage   -0.226          1.923         11.361         19.739
  Range:     1.923         13.228         19.739         21.103

       0.0000000E+00  1.334584505E+01 -8.199599416E+01  3.406177836E+04
       1.8891380E+02  1.472644573E+02  1.553962042E+02 -7.023729171E+03
      -9.3835290E+01 -1.844024844E+01 -8.342197663E+00  5.582903813E+02
       1.3068619E+02  4.031129726E+00  4.279433549E-01 -1.952394635E+01
      -2.2703580E+02 -6.249428360E-01 -1.191577910E-02  2.560740231E-01
       3.5145659E+02  6.468412046E-02  1.492290091E-04  0.000000000E+00
      -3.8953900E+02 -4.458750426E-03  0.000000000E+00  0.000000000E+00
       2.8239471E+02  1.994710149E-04  0.000000000E+00  0.000000000E+00
      -1.2607281E+02 -5.313401790E-06  0.000000000E+00  0.000000000E+00
       3.1353611E+01  6.481976217E-08  0.000000000E+00  0.000000000E+00
      -3.3187769E+00  0.000000000E+00  0.000000000E+00  0.000000000E+00

  Error      -0.02         -0.005         -0.0005        -0.001
  Range:      0.02          0.005          0.001          0.002
 */
	{5, -0.226, 1.923, {0.0000000E+00, 1.8891380E+02, -9.3835290E+01, 1.3068619E+02, -2.2703580E+02, 3.5145659E+02, -3.8953900E+02, 2.8239471E+02, -1.2607281E+02, 3.1353611E+01, -3.3187769E+00}, NOEXP, 0, 11},
	{5, 1.923, 13.228, {1.334584505E+01, 1.472644573E+02, -1.844024844E+01, 4.031129726E+00, -6.249428360E-01, 6.468412046E-02, -4.458750426E-03, 1.994710149E-04, -5.313401790E-06, 6.481976217E-08}, NOEXP, 0, 10},
	{5, 11.361, 19.739, {-8.199599416E+01,  1.553962042E+02, -8.342197663E+00, 4.279433549E-01, -1.191577910E-02, 1.492290091E-04}, NOEXP, 0, 6},
	{5, 19.739, 21.103, { 3.406177836E+04, -7.023729171E+03, 5.582903813E+02, -1.952394635E+01,  2.560740231E-01}, NOEXP, 0, 5},
 /*
 * Inverse coefficients for type S:

Temperature   -50.           250.          1064.          1664.5
  Range:      250.          1200.          1664.5         1768.1

  Voltage   -0.235          1.874         10.332         17.536
  Range:     1.874         11.950         17.536         18.693

       0.00000000E+00  1.291507177E+01 -8.087801117E+01  5.333875126E+04
       1.84949460E+02  1.466298863E+02  1.621573104E+02 -1.235892298E+04
      -8.00504062E+01 -1.534713402E+01 -8.536869453E+00  1.092657613E+03
       1.02237430E+02  3.145945973E+00  4.719686976E-01 -4.265693686E+01
      -1.52248592E+02 -4.163257839E-01 -1.441693666E-02  6.247205420E-01
       1.88821343E+02  3.187963771E-02  2.081618890E-04  0.000000000E+00
      -1.59085941E+02 -1.291637500E-03  0.000000000E+00  0.000000000E+00
       8.23027880E+01  2.183475087E-05  0.000000000E+00  0.000000000E+00
      -2.34181944E+01 -1.447379511E-07  0.000000000E+00  0.000000000E+00
       2.79786260E+00  8.211272125E-09  0.000000000E+00  0.000000000E+00

  Error      -0.02          -0.01         -0.0002        -0.002
  Range:      0.02           0.01          0.0002         0.002
 */
	{6, -0.235, 1.874, {0.00000000E+00, 1.84949460E+02, -8.00504062E+01, 1.02237430E+02, -1.52248592E+02, 1.88821343E+02, -1.59085941E+02, 8.23027880E+01, -2.34181944E+01, 2.79786260E+00}, NOEXP, 0, 10},
	{6,  1.874, 11.950, {1.291507177E+01, 1.466298863E+02, -1.534713402E+01, 3.145945973E+00, -4.163257839E-01, 3.187963771E-02, -1.291637500E-03, 2.183475087E-05, -1.447379511E-07, 8.211272125E-09}, NOEXP, 0, 10},
	{6, 10.332, 17.536, {-8.087801117E+01,  1.621573104E+02, -8.536869453E+00,  4.719686976E-01, -1.441693666E-02,  2.081618890E-04}, NOEXP, 0, 6},
	{6, 17.536, 18.693, { 5.333875126E+04, -1.235892298E+04,  1.092657613E+03, -4.265693686E+01,  6.247205420E-01}, NOEXP, 0, 5},
 /*
 * Inverse coefficients for type T:

Temperature  -200.             0.
  Range:        0.           400.

  Voltage   -5.603          0.000
  Range:     0.000         20.872

         0.0000000E+00  0.000000E+00
         2.5949192E+01  2.592800E+01
        -2.1316967E-01 -7.602961E-01
         7.9018692E-01  4.637791E-02
         4.2527777E-01 -2.165394E-03
         1.3304473E-01  6.048144E-05
         2.0241446E-02 -7.293422E-07
         1.2668171E-03  0.000000E+00

  Error      -0.02          -0.03
  Range:      0.04           0.03
 *
 *
 */
	{7, -5.603, 0.000, { 0.0000000E+00,  2.5949192E+01, -2.1316967E-01,  7.9018692E-01,  4.2527777E-01,  1.3304473E-01,  2.0241446E-02,  1.2668171E-03}, NOEXP, 0, 8},
	{7, 0.000, 20.872, { 0.000000E+00,  2.592800E+01, -7.602961E-01,  4.637791E-02, -2.165394E-03,  6.048144E-05, -7.293422E-07}, NOEXP, 0, 7}
};

coeff_t tvcoeff_set[] = {

/*
 * Coefficients for C to mV conversion
 * 
name: reference function on ITS-90
type: B
temperature units: °C
emf units: mV
range: 0.000, 630.615, 6
  0.000000000000E+00
 -0.246508183460E-03
  0.590404211710E-05
 -0.132579316360E-08
  0.156682919010E-11
 -0.169445292400E-14
  0.629903470940E-18
range: 630.615, 1820.000, 8
 -0.389381686210E+01
  0.285717474700E-01
 -0.848851047850E-04
  0.157852801640E-06
 -0.168353448640E-09
  0.111097940130E-12
 -0.445154310330E-16
  0.989756408210E-20
 -0.937913302890E-24
 */

 {0, 0.000, 630.615, {
   0.000000000000E+00, -0.246508183460E-03,  0.590404211710E-05, -0.132579316360E-08,  0.156682919010E-11, -0.169445292400E-14,  0.629903470940E-18}, NOEXP, 0, 7},
 {1, 630.615, 1820.000, {
   -0.389381686210E+01,   0.285717474700E-01,  -0.848851047850E-04,   0.157852801640E-06,  -0.168353448640E-09,   0.111097940130E-12,  -0.445154310330E-16,   0.989756408210E-20,  -0.937913302890E-24}, NOEXP, 0, 9},
/*
 name: reference function on ITS-90
type: E
temperature units: °C
emf units: mV
range: -270.000, 0.000, 13
  0.000000000000E+00
  0.586655087080E-01
  0.454109771240E-04
 -0.779980486860E-06
 -0.258001608430E-07
 -0.594525830570E-09
 -0.932140586670E-11
 -0.102876055340E-12
 -0.803701236210E-15
 -0.439794973910E-17
 -0.164147763550E-19
 -0.396736195160E-22
 -0.558273287210E-25
 -0.346578420130E-28
range: 0.000, 1000.000, 10
  0.000000000000E+00
  0.586655087100E-01
  0.450322755820E-04
  0.289084072120E-07
 -0.330568966520E-09
  0.650244032700E-12
 -0.191974955040E-15
 -0.125366004970E-17
  0.214892175690E-20
 -0.143880417820E-23
  0.359608994810E-27
  */

 {1, -270.000, 0.000, {
  0.000000000000E+00, 0.586655087080E-01, 0.454109771240E-04, -0.779980486860E-06, -0.258001608430E-07, -0.594525830570E-09, -0.932140586670E-11, -0.102876055340E-12, -0.803701236210E-15, -0.439794973910E-17, -0.164147763550E-19, -0.396736195160E-22, -0.558273287210E-25, -0.346578420130E-28
 }, NOEXP, 0, 14},
 {1, 0.000, 1000.000, {
  0.000000000000E+00, 0.586655087100E-01, 0.450322755820E-04, 0.289084072120E-07, -0.330568966520E-09, 0.650244032700E-12, -0.191974955040E-15, -0.125366004970E-17, 0.214892175690E-20, -0.143880417820E-23, 0.359608994810E-27
 }, NOEXP, 0, 11},

 /*

name: reference function on ITS-90
type: J
temperature units: °C
emf units: mV
range: -210.000, 760.000, 8
  0.000000000000E+00
  0.503811878150E-01
  0.304758369300E-04
 -0.856810657200E-07
  0.132281952950E-09
 -0.170529583370E-12
  0.209480906970E-15
 -0.125383953360E-18
  0.156317256970E-22
range: 760.000, 1200.000, 5
  0.296456256810E+03
 -0.149761277860E+01
  0.317871039240E-02
 -0.318476867010E-05
  0.157208190040E-08
 -0.306913690560E-12
*/

 {2, 210.000, 760.000, {
	  0.000000000000E+00, 0.503811878150E-01, 0.304758369300E-04, -0.856810657200E-07, 0.132281952950E-09, -0.170529583370E-12, 0.209480906970E-15, -0.125383953360E-18, 0.156317256970E-22
 }, NOEXP, 0, 9},
 {2, 760.000, 1200.000, {
	  0.296456256810E+03, -0.149761277860E+01, 0.317871039240E-02, -0.318476867010E-05, 0.157208190040E-08, -0.306913690560E-12}, NOEXP, 0, 6},
 /*

************************************
* This section contains coefficients for type K thermocouples for
* the two subranges of temperature listed below.  The coefficients 
* are in units of °C and mV and are listed in the order of constant 
* term up to the highest order.  The equation below 0 °C is of the form 
* E = sum(i=0 to n) c_i t^i.
*
* The equation above 0 °C is of the form 
* E = sum(i=0 to n) c_i t^i + a0 exp(a1 (t - a2)^2).
*
*     Temperature Range (°C)
*        -270.000 to 0.000 
*         0.000 to 1372.000
************************************
name: reference function on ITS-90
type: K
temperature units: °C
emf units: mV
range: -270.000, 0.000, 10
  0.000000000000E+00
  0.394501280250E-01
  0.236223735980E-04
 -0.328589067840E-06
 -0.499048287770E-08
 -0.675090591730E-10
 -0.574103274280E-12
 -0.310888728940E-14
 -0.104516093650E-16
 -0.198892668780E-19
 -0.163226974860E-22
 */

 {3, -270.000, 0.000, {
	  0.000000000000E+00,   0.394501280250E-01,   0.236223735980E-04,  -0.328589067840E-06,  -0.499048287770E-08,  -0.675090591730E-10,  -0.574103274280E-12,  -0.310888728940E-14,  -0.104516093650E-16,  -0.198892668780E-19,  -0.163226974860E-22
 }, NOEXP, 0, 11},

 /*
range: 0.000, 1372.000, 9
 -0.176004136860E-01
  0.389212049750E-01
  0.185587700320E-04
 -0.994575928740E-07
  0.318409457190E-09
 -0.560728448890E-12
  0.560750590590E-15
 -0.320207200030E-18
  0.971511471520E-22
 -0.121047212750E-25
exponential:
 a0 =  0.118597600000E+00
 a1 = -0.118343200000E-03
 a2 =  0.126968600000E+03

*/

 {3, 0.000, 1372.000, {
	 -0.176004136860E-01,   0.389212049750E-01,   0.185587700320E-04,  -0.994575928740E-07,   0.318409457190E-09,  -0.560728448890E-12,   0.560750590590E-15,  -0.320207200030E-18,   0.971511471520E-22,  -0.121047212750E-25
 }, {
	  0.118597600000E+00,  -0.118343200000E-03,   0.126968600000E+03
 }, 3, 10},
 
 /*

 name: reference function on ITS-90
type: N
temperature units: °C
emf units: mV
range: -270.000, 0.000, 8
  0.000000000000E+00
  0.261591059620E-01
  0.109574842280E-04
 -0.938411115540E-07
 -0.464120397590E-10
 -0.263033577160E-11
 -0.226534380030E-13
 -0.760893007910E-16
 -0.934196678350E-19
 */

 {4, -270.000, 0.000, {
	  0.000000000000E+00,   0.261591059620E-01,   0.109574842280E-04,  -0.938411115540E-07,  -0.464120397590E-10,  -0.263033577160E-11,  -0.226534380030E-13,  -0.760893007910E-16,  -0.934196678350E-19
 }, NOEXP, 0, 9},

 /*
range: 0., 1300., 10
  0.000000000000E+00
  0.259293946010E-01
  0.157101418800E-04
  0.438256272370E-07
 -0.252611697940E-09
  0.643118193390E-12
 -0.100634715190E-14
  0.997453389920E-18
 -0.608632456070E-21
  0.208492293390E-24
 -0.306821961510E-28
*/

 {4, 0., 1300.0, {
	  0.000000000000E+00,   0.259293946010E-01,   0.157101418800E-04,   0.438256272370E-07,  -0.252611697940E-09,   0.643118193390E-12,  -0.100634715190E-14,   0.997453389920E-18,  -0.608632456070E-21,   0.208492293390E-24,  -0.306821961510E-28
 }, NOEXP, 0, 11},

/*
name: reference function on ITS-90
type: R
temperature units: °C
emf units: mV
range: -50.000, 1064.180, 9
  0.000000000000E+00
  0.528961729765E-02
  0.139166589782E-04
 -0.238855693017E-07
  0.356916001063E-10
 -0.462347666298E-13
  0.500777441034E-16
 -0.373105886191E-19
  0.157716482367E-22
 -0.281038625251E-26
 */

 {5, -50.000, 1064.180, {
	  0.000000000000E+00,   0.528961729765E-02,   0.139166589782E-04,  -0.238855693017E-07,   0.356916001063E-10,  -0.462347666298E-13,   0.500777441034E-16,  -0.373105886191E-19,   0.157716482367E-22,  -0.281038625251E-26
 }, NOEXP, 0, 10},

 /*
range: 1064.180, 1664.500, 5
  0.295157925316E+01
 -0.252061251332E-02
  0.159564501865E-04
 -0.764085947576E-08
  0.205305291024E-11
 -0.293359668173E-15
 */

 { 5, 1064.180, 1664.500, {
	  0.295157925316E+01,  -0.252061251332E-02,   0.159564501865E-04,  -0.764085947576E-08,   0.205305291024E-11,  -0.293359668173E-15
 }, NOEXP, 0, 6},

 /*
range: 1664.5, 1768.1, 4
  0.152232118209E+03
 -0.268819888545E+00
  0.171280280471E-03
 -0.345895706453E-07
 -0.934633971046E-14
*/

 {5, 1664.5, 1768.1, {
	  0.152232118209E+03,  -0.268819888545E+00,   0.171280280471E-03,  -0.345895706453E-07,  -0.934633971046E-14
 }, NOEXP, 0, 5},

 /*
 
name: reference function on ITS-90
type: S
temperature units: °C
emf units: mV
range: -50.000, 1064.180, 8
  0.000000000000E+00
  0.540313308631E-02
  0.125934289740E-04
 -0.232477968689E-07
  0.322028823036E-10
 -0.331465196389E-13
  0.255744251786E-16
 -0.125068871393E-19
  0.271443176145E-23
  */

 {6, -50.000, 1064.180, {
	  0.000000000000E+00,   0.540313308631E-02,   0.125934289740E-04,  -0.232477968689E-07,   0.322028823036E-10,  -0.331465196389E-13,   0.255744251786E-16,  -0.125068871393E-19,   0.271443176145E-23
 }, NOEXP, 0, 9},

 /*
range: 1064.180, 1664.500, 4
  0.132900444085E+01
  0.334509311344E-02
  0.654805192818E-05
 -0.164856259209E-08
  0.129989605174E-13
  */

 {6, 1064.180, 1664.500, {
	  0.132900444085E+01,   0.334509311344E-02,   0.654805192818E-05,  -0.164856259209E-08,   0.129989605174E-13
 }, NOEXP, 0, 5},

 /*
range: 1664.5, 1768.1, 4
  0.146628232636E+03
 -0.258430516752E+00
  0.163693574641E-03
 -0.330439046987E-07
 -0.943223690612E-14
*/

 {6, 1664.5, 1768.1, {
	  0.146628232636E+03,  -0.258430516752E+00,   0.163693574641E-03,  -0.330439046987E-07,  -0.943223690612E-14
 }, NOEXP, 0, 5},

 /*

 name: reference function on ITS-90
type: T
temperature units: °C
emf units: mV
range: -270.000, 0.000, 14
  0.000000000000E+00
  0.387481063640E-01
  0.441944343470E-04
  0.118443231050E-06
  0.200329735540E-07
  0.901380195590E-09
  0.226511565930E-10
  0.360711542050E-12
  0.384939398830E-14
  0.282135219250E-16
  0.142515947790E-18
  0.487686622860E-21
  0.107955392700E-23
  0.139450270620E-26
  0.797951539270E-30
  */

 {7, -270.000, 0.000, {
	  0.000000000000E+00,   0.387481063640E-01,   0.441944343470E-04,   0.118443231050E-06,   0.200329735540E-07,   0.901380195590E-09,   0.226511565930E-10,   0.360711542050E-12,   0.384939398830E-14,   0.282135219250E-16,   0.142515947790E-18,   0.487686622860E-21,   0.107955392700E-23,   0.139450270620E-26,   0.797951539270E-30
 }, NOEXP, 0, 15},

 /*
range: 0.000, 400.000, 8
  0.000000000000E+00
  0.387481063640E-01
  0.332922278800E-04
  0.206182434040E-06
 -0.218822568460E-08
  0.109968809280E-10
 -0.308157587720E-13
  0.454791352900E-16
 -0.275129016730E-19
 * */

 {7, 0.000, 400.000, {
	  0.000000000000E+00,   0.387481063640E-01,   0.332922278800E-04,   0.206182434040E-06,  -0.218822568460E-08,   0.109968809280E-10,  -0.308157587720E-13,   0.454791352900E-16,  -0.275129016730E-19
 }, NOEXP, 0, 9}
};

coeff_t *find_coeff_set(int type, double V, coeff_t *set, int setLength)
{
  int i;

  for (i=0; i<setLength; ++i)
  {
	if (set[i].type != type) continue;
	if (V < set[i].min) continue;
	if (V > set[i].max) continue;

	return &(set[i]);
  }

  return NULL; // Not found
}

double thrmcpPoly(coeff_t *set, double x)
{
  int i;
  double m = 1.0;
  double sum = 0;

  for (i=0; i< (set->Ncoeff); ++i)
  {
	sum += m * set->coeff[i];
	m *= x;
  }

  if (set->Nexp > 0)
  {
	// We have the exp term : a0 exp(a1 (t - a2)^2)
	double dt, dt2;

	dt = x - set->expc[2];
	dt2 = dt * dt;

	sum += set->expc[0] * exp(set->expc[1] * dt2);
  }

  return sum;
}

int getThermocoupleT(tc_struct *tc)
{
//  int i,j;
//  for(i=0;i<10;++i){
//	printf("%d %d %f:%f %d:",i,tccoeff_set[i].type, tccoeff_set[i].minV, tccoeff_set[i].maxV, tccoeff_set[i].Ncoeff);
//	for(j=0;j<tccoeff_set[i].Ncoeff;++j)
//	  printf(" %g", tccoeff_set[i].coeff[j]);
//	printf("\n");
//  }

  coeff_t *vtset, *tvset;
  double Vcj;

  /* Find coefficients to perform generate cold junction Voltage */
  tvset = find_coeff_set(tc->type, tc->Tcj, tvcoeff_set, NTVCOEFFSET);
  if (tvset==NULL) return -1;

  Vcj = thrmcpPoly(tvset, tc->Tcj);

  /* Find coefficients to convert Voltage to temperature */
  vtset = find_coeff_set(tc->type, Vcj + tc->V, vtcoeff_set, NVTCOEFFSET);
  if (vtset==NULL) return -1;

  tc->T = thrmcpPoly(vtset, Vcj + tc->V);

//  printf("Tcj = %f => V = %f + %f = %f => T = %f\n", tc->Tcj, tc->V, Vcj, Vcj + tc->V, tc->T);

  return 0; // Ok
}

