/*
 * Copyright 2016-2017 Józef Kucia for CodeWeavers
 * Copyright 2020-2021 Philip Rebohle for Valve Corporation
 * Copyright 2020-2021 Joshua Ashton for Valve Corporation
 * Copyright 2020-2021 Hans-Kristian Arntzen for Valve Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#define VKD3D_DBG_CHANNEL VKD3D_DBG_CHANNEL_API
#include "d3d12_crosstest.h"

struct triangle
{
    struct vec4 v[3];
};

#define check_triangles(a, b, c, d, e) check_triangles_(__LINE__, a, b, c, d, e)
static void check_triangles_(unsigned int line, ID3D12Resource *buffer,
        ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        const struct triangle *triangles, unsigned int triangle_count)
{
    const struct triangle *current, *expected;
    struct resource_readback rb;
    unsigned int i, j, offset;
    bool all_match = true;

    get_buffer_readback_with_command_list(buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

    for (i = 0; i < triangle_count; ++i)
    {
        current = get_readback_data(&rb, i, 0, 0, sizeof(*current));
        expected = &triangles[i];

        offset = ~0u;
        for (j = 0; j < ARRAY_SIZE(expected->v); ++j)
        {
            if (compare_vec4(&current->v[0], &expected->v[j], 0))
            {
                offset = j;
                break;
            }
        }

        if (offset == ~0u)
        {
            all_match = false;
            break;
        }

        for (j = 0; j < ARRAY_SIZE(expected->v); ++j)
        {
            if (!compare_vec4(&current->v[j], &expected->v[(j + offset) % 3], 0))
            {
                all_match = false;
                break;
            }
        }
        if (!all_match)
            break;
    }

    ok_(line)(all_match, "Triangle %u vertices {%.8e, %.8e, %.8e, %.8e}, "
            "{%.8e, %.8e, %.8e, %.8e}, {%.8e, %.8e, %.8e, %.8e} "
            "do not match {%.8e, %.8e, %.8e, %.8e}, {%.8e, %.8e, %.8e, %.8e}, "
            "{%.8e, %.8e, %.8e, %.8e}.\n", i,
            current->v[0].x, current->v[0].y, current->v[0].z, current->v[0].w,
            current->v[1].x, current->v[1].y, current->v[1].z, current->v[1].w,
            current->v[2].x, current->v[2].y, current->v[2].z, current->v[2].w,
            expected->v[0].x, expected->v[0].y, expected->v[0].z, expected->v[0].w,
            expected->v[1].x, expected->v[1].y, expected->v[1].z, expected->v[1].w,
            expected->v[2].x, expected->v[2].y, expected->v[2].z, expected->v[2].w);

    release_resource_readback(&rb);
}

void test_nop_tessellation_shaders(void)
{
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    struct test_context_desc desc;
    struct test_context context;
    ID3D12CommandQueue *queue;
    struct vec4 tess_factors;
    unsigned int i;
    HRESULT hr;

    static const DWORD hs_cb_code[] =
    {
#if 0
        float4 tess_factor;

        struct data
        {
            float4 position : SV_Position;
        };

        struct patch_constant_data
        {
            float edges[3] : SV_TessFactor;
            float inside : SV_InsideTessFactor;
        };

        void patch_constant(InputPatch<data, 3> input, out patch_constant_data output)
        {
            output.edges[0] = tess_factor.x;
            output.edges[1] = tess_factor.y;
            output.edges[2] = tess_factor.z;
            output.inside = tess_factor.w;
        }

        [domain("tri")]
        [outputcontrolpoints(3)]
        [partitioning("integer")]
        [outputtopology("triangle_cw")]
        [patchconstantfunc("patch_constant")]
        data hs_main(InputPatch<data, 3> input, uint i : SV_OutputControlPointID)
        {
            return input[i];
        }
#endif
        0x43425844, 0x7e698b53, 0x830de202, 0x4287601f, 0x4315faa4, 0x00000001, 0x00000228, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000008c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000e01,
        0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x00000068, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469,
        0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x000000f4, 0x00030050, 0x0000003d, 0x01000071,
        0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01001897, 0x0100086a, 0x04000059, 0x00208e46,
        0x00000000, 0x00000001, 0x01000073, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x06000036,
        0x00102012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000001, 0x00000012, 0x06000036, 0x00102012, 0x00000001, 0x0020801a, 0x00000000,
        0x00000000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000002, 0x00000013, 0x06000036,
        0x00102012, 0x00000002, 0x0020802a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000003, 0x00000014, 0x06000036, 0x00102012, 0x00000003, 0x0020803a, 0x00000000,
        0x00000000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs_cb = {hs_cb_code, sizeof(hs_cb_code)};
#if 0
    struct data
    {
        float4 position : SV_Position;
    };

    struct patch_constant_data
    {
        float edges[3] : SV_TessFactor;
        float inside : SV_InsideTessFactor;
    };

    void patch_constant(InputPatch<data, 3> input, out patch_constant_data output)
    {
        output.edges[0] = output.edges[1] = output.edges[2] = 1.0f;
        output.inside = 1.0f;
    }

    [domain("tri")]
    [outputcontrolpoints(3)]
    [partitioning("integer")]
    [outputtopology("triangle_cw")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 3> input, uint i : SV_OutputControlPointID)
    {
        return input[i];
    }

    [domain("tri")]
    void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<data, 3> patch,
            out data output)
    {
        output.position = tess_coord.x * patch[0].position
                + tess_coord.y * patch[1].position
                + tess_coord.z * patch[2].position;
    }
#endif
    static const DWORD hs_code[] =
    {
        0x43425844, 0x0e9a8861, 0x39351e76, 0x0e10883f, 0x6054b5a1, 0x00000001, 0x0000020c, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000008c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000e01,
        0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x00000068, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469,
        0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x000000d8, 0x00030050, 0x00000036, 0x01000071,
        0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01001897, 0x0100086a, 0x01000073, 0x02000099,
        0x00000003, 0x0200005f, 0x00017000, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x04000067,
        0x00102012, 0x00000001, 0x00000012, 0x04000067, 0x00102012, 0x00000002, 0x00000013, 0x02000068,
        0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000003, 0x04000036, 0x00100012, 0x00000000,
        0x0001700a, 0x06000036, 0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
        0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x00000014, 0x05000036, 0x00102012, 0x00000003,
        0x00004001, 0x3f800000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs = {hs_code, sizeof(hs_code)};
    static const DWORD ds_code[] =
    {
        0x43425844, 0x8ed11021, 0x414dff74, 0x426849eb, 0x312f4860, 0x00000001, 0x000001e0, 0x00000004,
        0x00000030, 0x00000064, 0x000000f8, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d,
        0x00000003, 0x00000000, 0x00000001, 0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001,
        0x00000001, 0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002, 0x00000001, 0x00000076,
        0x00000000, 0x0000000e, 0x00000003, 0x00000003, 0x00000001, 0x545f5653, 0x46737365, 0x6f746361,
        0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x4e47534f, 0x0000002c,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x000000ac, 0x00040050, 0x0000002b, 0x01001893,
        0x01001095, 0x0100086a, 0x0200005f, 0x0001c072, 0x0400005f, 0x002190f2, 0x00000003, 0x00000000,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x02000068, 0x00000001, 0x07000038, 0x001000f2,
        0x00000000, 0x0001c556, 0x00219e46, 0x00000001, 0x00000000, 0x09000032, 0x001000f2, 0x00000000,
        0x0001c006, 0x00219e46, 0x00000000, 0x00000000, 0x00100e46, 0x00000000, 0x09000032, 0x001020f2,
        0x00000000, 0x0001caa6, 0x00219e46, 0x00000002, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ds = {ds_code, sizeof(ds_code)};
    static const DWORD hs_index_range_code[] =
    {
#if 0
        float4 tess_factor;

        struct data
        {
            float4 position : SV_Position;
        };

        struct patch_constant_data
        {
            float edges[3] : SV_TessFactor;
            float inside : SV_InsideTessFactor;
        };

        void patch_constant(InputPatch<data, 3> input, out patch_constant_data output)
        {
            output.edges[0] = tess_factor.x;
            output.edges[1] = 1.0f;
            output.edges[2] = 1.0f;
            output.inside = tess_factor.y;
        }

        [domain("tri")]
        [outputcontrolpoints(3)]
        [partitioning("integer")]
        [outputtopology("triangle_cw")]
        [patchconstantfunc("patch_constant")]
        data hs_main(InputPatch<data, 3> input, uint i : SV_OutputControlPointID)
        {
            return input[i];
        }
#endif
        0x43425844, 0xf9d52cfc, 0xb299036a, 0x66bf56b7, 0x6161e921, 0x00000001, 0x00000244, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000008c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000e01,
        0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x00000068, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469,
        0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x00000110, 0x00030050, 0x00000044, 0x01000071,
        0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01001897, 0x0100086a, 0x04000059, 0x00208e46,
        0x00000000, 0x00000001, 0x01000073, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x06000036,
        0x00102012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x02000099,
        0x00000002, 0x0200005f, 0x00017000, 0x04000067, 0x00102012, 0x00000001, 0x00000012, 0x04000067,
        0x00102012, 0x00000002, 0x00000013, 0x02000068, 0x00000001, 0x0400005b, 0x00102012, 0x00000001,
        0x00000002, 0x04000036, 0x00100012, 0x00000000, 0x0001700a, 0x07000036, 0x00d02012, 0x00000001,
        0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012,
        0x00000003, 0x00000014, 0x06000036, 0x00102012, 0x00000003, 0x0020801a, 0x00000000, 0x00000000,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs_index_range = {hs_index_range_code, sizeof(hs_index_range_code)};
#if 0
        float4 tess_factor;

        struct hs_input
        {
            float4 position : SV_Position;
        };

        struct hs_output
        {
            float4 position : SV_Position;
            float  clip     : SV_ClipDistance;
        };

        struct patch_constant_data
        {
            float edges[3] : SV_TessFactor;
            float inside : SV_InsideTessFactor;
        };

        void patch_constant(InputPatch<hs_input, 3> input, out patch_constant_data output)
        {
            output.edges[0] = tess_factor.x;
            output.edges[1] = 1.0f;
            output.edges[2] = 1.0f;
            output.inside = tess_factor.y;
        }

        [domain("tri")]
        [outputcontrolpoints(3)]
        [partitioning("integer")]
        [outputtopology("triangle_cw")]
        [patchconstantfunc("patch_constant")]
        hs_output hs_main(InputPatch<hs_input, 3> input, uint i : SV_OutputControlPointID)
        {
            hs_output output;
            output.position = input[i].position;
            output.clip     = 1024.0;
            return output;
        }

        [domain("tri")]
        void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<hs_output, 3> patch,
            out hs_output output)
        {
            output.position = tess_coord.x * patch[0].position * patch[0].clip
                    + tess_coord.y * patch[1].position * patch[1].clip
                    + tess_coord.z * patch[2].position * patch[2].clip;
            output.clip = 1.0;
        }
#endif
    static const DWORD hs_clip_distance_code[] =
    {
        0x43425844, 0x5f7dacae, 0xb950946c, 0x46ce2f56, 0x08cd9f72, 0x00000001, 0x00000300, 0x00000004,
        0x00000030, 0x00000064, 0x000000c0, 0x00000154, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x00000054, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000002, 0x00000003, 0x00000001,
        0x00000e01, 0x505f5653, 0x7469736f, 0x006e6f69, 0x435f5653, 0x4470696c, 0x61747369, 0x0065636e,
        0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003,
        0x00000000, 0x00000e01, 0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01,
        0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000,
        0x0000000e, 0x00000003, 0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072,
        0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x000001a4, 0x00030051,
        0x00000069, 0x01000071, 0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01001897, 0x0100086a,
        0x07000059, 0x00308e46, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x01000072,
        0x0200005f, 0x00016000, 0x0400005f, 0x002010f2, 0x00000003, 0x00000000, 0x03000065, 0x001020f2,
        0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000001, 0x04000036, 0x00100012,
        0x00000000, 0x00016001, 0x07000036, 0x001020f2, 0x00000000, 0x00a01e46, 0x0010000a, 0x00000000,
        0x00000000, 0x05000036, 0x00102012, 0x00000001, 0x00004001, 0x44800000, 0x0100003e, 0x01000073,
        0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x07000036, 0x00102012, 0x00000000, 0x0030800a,
        0x00000000, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x02000099, 0x00000002, 0x0200005f,
        0x00017000, 0x04000067, 0x00102012, 0x00000001, 0x00000012, 0x04000067, 0x00102012, 0x00000002,
        0x00000013, 0x02000068, 0x00000001, 0x0400005b, 0x00102012, 0x00000001, 0x00000002, 0x04000036,
        0x00100012, 0x00000000, 0x0001700a, 0x07000036, 0x00d02012, 0x00000001, 0x0010000a, 0x00000000,
        0x00004001, 0x3f800000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x00000014,
        0x07000036, 0x00102012, 0x00000003, 0x0030801a, 0x00000000, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const DWORD ds_clip_distance_code[] =
    {
        0x43425844, 0xe2161d6b, 0x7cc8473f, 0xbc603fc3, 0x2fd75c33, 0x00000001, 0x000002c4, 0x00000004,
        0x00000030, 0x0000008c, 0x00000120, 0x0000017c, 0x4e475349, 0x00000054, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x00000044, 0x00000000,
        0x00000002, 0x00000003, 0x00000001, 0x00000101, 0x505f5653, 0x7469736f, 0x006e6f69, 0x435f5653,
        0x4470696c, 0x61747369, 0x0065636e, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000001, 0x00000068, 0x00000001, 0x0000000d,
        0x00000003, 0x00000001, 0x00000001, 0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002,
        0x00000001, 0x00000076, 0x00000000, 0x0000000e, 0x00000003, 0x00000003, 0x00000001, 0x545f5653,
        0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072,
        0x4e47534f, 0x00000054, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000002, 0x00000003, 0x00000001, 0x00000e01,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x435f5653, 0x4470696c, 0x61747369, 0x0065636e, 0x58454853,
        0x00000140, 0x00040050, 0x00000050, 0x01001893, 0x01001095, 0x0100086a, 0x0200005f, 0x0001c072,
        0x0400005f, 0x002190f2, 0x00000003, 0x00000000, 0x0400005f, 0x00219012, 0x00000003, 0x00000001,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x04000067, 0x00102012, 0x00000001, 0x00000002,
        0x02000068, 0x00000002, 0x07000038, 0x001000f2, 0x00000000, 0x0001c556, 0x00219e46, 0x00000001,
        0x00000000, 0x08000038, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00219006, 0x00000001,
        0x00000001, 0x07000038, 0x001000f2, 0x00000001, 0x0001c006, 0x00219e46, 0x00000000, 0x00000000,
        0x0a000032, 0x001000f2, 0x00000000, 0x00100e46, 0x00000001, 0x00219006, 0x00000000, 0x00000001,
        0x00100e46, 0x00000000, 0x07000038, 0x001000f2, 0x00000001, 0x0001caa6, 0x00219e46, 0x00000002,
        0x00000000, 0x0a000032, 0x001020f2, 0x00000000, 0x00100e46, 0x00000001, 0x00219006, 0x00000002,
        0x00000001, 0x00100e46, 0x00000000, 0x05000036, 0x00102012, 0x00000001, 0x00004001, 0x3f800000,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs_clip_distance = {hs_clip_distance_code, sizeof(hs_clip_distance_code)};
    static const D3D12_SHADER_BYTECODE ds_clip_distance = {ds_clip_distance_code, sizeof(ds_clip_distance_code)};
    static const D3D12_SHADER_BYTECODE *hull_shaders[] = {&hs_cb, &hs, &hs_index_range, &hs_clip_distance};
    static const D3D12_SHADER_BYTECODE *domain_shaders[] = {&ds, &ds, &ds, &ds_clip_distance};

    memset(&desc, 0, sizeof(desc));
    desc.no_root_signature = true;
    if (!init_test_context(&context, &desc))
        return;
    command_list = context.list;
    queue = context.queue;

    context.root_signature = create_32bit_constants_root_signature(context.device,
            0, 4, D3D12_SHADER_VISIBILITY_HULL);

    init_pipeline_state_desc(&pso_desc, context.root_signature,
            context.render_target_desc.Format, NULL, NULL, NULL);
    pso_desc.HS = hs_cb;
    pso_desc.DS = ds;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#x.\n", hr);
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);
    ID3D12PipelineState_Release(context.pipeline_state);

    for (i = 0; i < ARRAY_SIZE(hull_shaders); ++i)
    {
        vkd3d_test_set_context("Test %u", i);

        pso_desc.HS = *hull_shaders[i];
        pso_desc.DS = *domain_shaders[i];
        hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
                &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
        ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

        ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

        ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
        ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
        ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_3_CONTROL_POINT_PATCHLIST);
        ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
        ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
        tess_factors.x = tess_factors.y = tess_factors.z = tess_factors.w = 1.0f;
        ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(command_list, 0, 4, &tess_factors.x, 0);
        ID3D12GraphicsCommandList_DrawInstanced(command_list, 3, 1, 0, 0);

        transition_resource_state(command_list, context.render_target,
                D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);

        check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff00ff00, 0);

        reset_command_list(command_list, context.allocator);
        transition_resource_state(command_list, context.render_target,
                D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_RENDER_TARGET);

        ID3D12PipelineState_Release(context.pipeline_state);
        context.pipeline_state = NULL;
    }
    vkd3d_test_set_context(NULL);

    destroy_test_context(&context);
}

static void test_quad_tessellation(bool use_dxil)
{
#if 0
    void main(float4 in_position : POSITION, out float4 out_position : SV_POSITION)
    {
        out_position = in_position;
    }
#endif
    static const DWORD vs_code_dxbc[] =
    {
        0x43425844, 0xa7a2f22d, 0x83ff2560, 0xe61638bd, 0x87e3ce90, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
    static const BYTE vs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xfe, 0xec, 0x6d, 0x32, 0xe6, 0xa7, 0x2d, 0x4b, 0x03, 0x74, 0xd7, 0xdb, 0x3f, 0x8c, 0xb9, 0x51, 0x01, 0x00, 0x00, 0x00, 0xf1, 0x05, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0x39, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x31, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x4f, 0x53, 0x47,
        0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00,
        0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
        0x00, 0x01, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xb0, 0x04, 0x00,
        0x00, 0x60, 0x00, 0x01, 0x00, 0x2c, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x98, 0x04, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
        0x00, 0x23, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84,
        0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14,
        0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00,
        0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00,
        0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4,
        0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x30, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30, 0x98, 0x23, 0x40,
        0x8a, 0x31, 0x44, 0x54, 0x44, 0x56, 0x0c, 0x20, 0xa2, 0x1a, 0xc2, 0x81, 0x80, 0x34, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72,
        0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
        0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc8, 0x02, 0x01, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x18, 0x01, 0x28, 0x84, 0x62, 0xa0,
        0x2a, 0x89, 0x11, 0x80, 0x42, 0x28, 0x03, 0xda, 0xb1, 0x0c, 0x82, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
        0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x88, 0x61, 0x82, 0x40, 0x10, 0x1b, 0x84, 0x81, 0x98,
        0x20, 0x10, 0xc5, 0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10, 0x08, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58,
        0x9c, 0x09, 0xc2, 0xb2, 0x6c, 0x08, 0x20, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x44, 0xa0, 0x9e, 0xa6, 0x92, 0xa8, 0x92, 0x9e, 0x9c, 0x26, 0x08, 0xc5, 0x31, 0x41, 0x28, 0x90, 0x0d, 0x01, 0x31, 0x41,
        0x28, 0x92, 0x0d, 0x0b, 0x31, 0x51, 0x95, 0x55, 0x0d, 0x17, 0x51, 0x01, 0x1b, 0x02, 0x8c, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0xa1, 0x50, 0x36, 0x2c,
        0x84, 0x46, 0x6d, 0xd6, 0x35, 0x5c, 0x44, 0x05, 0x6c, 0x08, 0xb8, 0x0d, 0x43, 0xd6, 0x01, 0x1b, 0x8a, 0x45, 0xf2, 0x00, 0xa0, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd,
        0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78,
        0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x82, 0xa4, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b,
        0xdb, 0xdc, 0x94, 0xc0, 0xa9, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56, 0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0x80, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9, 0xe5, 0x41,
        0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0x3c, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
        0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
        0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70,
        0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
        0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
        0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80,
        0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
        0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
        0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x0b, 0x00, 0x00,
        0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4,
        0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
        0x00, 0x44, 0x33, 0x00, 0xa5, 0x40, 0x55, 0x02, 0x45, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x3c, 0x0b, 0xc3, 0x20, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18,
        0x10, 0xd3, 0x34, 0x43, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0xd4, 0x38, 0xce, 0x60, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x21, 0x39, 0xcf, 0x93, 0x1c, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x80, 0x48, 0x07, 0x04, 0x31, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xd2, 0x01, 0x41, 0xc6, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x88, 0x74, 0x40, 0x50, 0x21, 0x8c,
        0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x22, 0x1d, 0x10, 0xa4, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE vs = {
        use_dxil ? (const void*)vs_code_dxil : (const void*)vs_code_dxbc,
        use_dxil ? sizeof(vs_code_dxil) : sizeof(vs_code_dxbc)
    };
#if 0
    struct point_data
    {
        float4 position : SV_POSITION;
    };

    struct patch_constant_data
    {
        float edges[4] : SV_TessFactor;
        float inside[2] : SV_InsideTessFactor;
    };

    float4 tess_factors;
    float2 inside_tess_factors;

    patch_constant_data patch_constant(InputPatch<point_data, 4> input)
    {
        patch_constant_data output;

        output.edges[0] = tess_factors.x;
        output.edges[1] = tess_factors.y;
        output.edges[2] = tess_factors.z;
        output.edges[3] = tess_factors.w;
        output.inside[0] = inside_tess_factors.x;
        output.inside[1] = inside_tess_factors.y;

        return output;
    }

    [domain("quad")]
    [outputcontrolpoints(4)]
    [outputtopology("triangle_ccw")]
    [partitioning("integer")]
    [patchconstantfunc("patch_constant")]
    point_data hs_main(InputPatch<point_data, 4> input,
            uint i : SV_OutputControlPointID)
    {
        return input[i];
    }

    [domain("quad")]
    point_data ds_main(patch_constant_data input,
            float2 tess_coord : SV_DomainLocation,
            const OutputPatch<point_data, 4> patch)
    {
        point_data output;

        float4 a = lerp(patch[0].position, patch[1].position, tess_coord.x);
        float4 b = lerp(patch[2].position, patch[3].position, tess_coord.x);
        output.position = lerp(a, b, tess_coord.y);

        return output;
    }
#endif
    static const DWORD hs_quad_ccw_code_dxbc[] =
    {
        0x43425844, 0xdf8df700, 0x58b08fb1, 0xbd23d2c3, 0xcf884094, 0x00000001, 0x000002b8, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000015c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x47534350, 0x000000bc,
        0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x0000000b, 0x00000003, 0x00000000, 0x00000e01,
        0x00000098, 0x00000001, 0x0000000b, 0x00000003, 0x00000001, 0x00000e01, 0x00000098, 0x00000002,
        0x0000000b, 0x00000003, 0x00000002, 0x00000e01, 0x00000098, 0x00000003, 0x0000000b, 0x00000003,
        0x00000003, 0x00000e01, 0x000000a6, 0x00000000, 0x0000000c, 0x00000003, 0x00000004, 0x00000e01,
        0x000000a6, 0x00000001, 0x0000000c, 0x00000003, 0x00000005, 0x00000e01, 0x545f5653, 0x46737365,
        0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x58454853,
        0x00000154, 0x00030050, 0x00000055, 0x01000071, 0x01002093, 0x01002094, 0x01001895, 0x01000896,
        0x01002097, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002, 0x01000073, 0x04000067,
        0x00102012, 0x00000000, 0x0000000b, 0x06000036, 0x00102012, 0x00000000, 0x0020800a, 0x00000000,
        0x00000000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000001, 0x0000000c, 0x06000036,
        0x00102012, 0x00000001, 0x0020801a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000002, 0x0000000d, 0x06000036, 0x00102012, 0x00000002, 0x0020802a, 0x00000000,
        0x00000000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x0000000e, 0x06000036,
        0x00102012, 0x00000003, 0x0020803a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000004, 0x0000000f, 0x06000036, 0x00102012, 0x00000004, 0x0020800a, 0x00000000,
        0x00000001, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000005, 0x00000010, 0x06000036,
        0x00102012, 0x00000005, 0x0020801a, 0x00000000, 0x00000001, 0x0100003e,
    };
    static const BYTE hs_quad_ccw_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x29, 0x85, 0x29, 0xbb, 0xe8, 0xc1, 0xa9, 0x93, 0xec, 0x8a, 0x91, 0xb5, 0x5b, 0xaf, 0x79, 0xa1, 0x01, 0x00, 0x00, 0x00, 0x1a, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0xb2, 0x01, 0x00, 0x00, 0x7e, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x50, 0x53, 0x47, 0x31, 0xea, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x53, 0x56,
        0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x50, 0x53, 0x56, 0x30, 0xc4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x06, 0x00, 0x01, 0x01, 0x02, 0x01, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x04, 0x71, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58,
        0x49, 0x4c, 0x94, 0x07, 0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0xe5, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x7c, 0x07, 0x00, 0x00, 0x42, 0x43,
        0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xdc, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10,
        0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32,
        0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29,
        0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30,
        0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20,
        0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84,
        0xa4, 0x4c, 0x10, 0x68, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x08, 0x63, 0x10, 0x99, 0x01, 0x98, 0x23, 0x40, 0x8a, 0x61, 0xc6, 0x18, 0x67, 0x10, 0x2a, 0x06, 0x18, 0x63, 0x1c,
        0x43, 0xea, 0xa8, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x44, 0x18, 0x63, 0x0c, 0x9d, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10,
        0x68, 0x86, 0x85, 0x40, 0x41, 0x2b, 0x04, 0x1b, 0x6e, 0xd0, 0x9b, 0x23, 0x08, 0x8a, 0xe1, 0xc6, 0x19, 0x23, 0x92, 0x1c, 0x08, 0x18, 0x46, 0x20, 0x8c, 0x61, 0x04, 0xc1, 0x98, 0x09, 0x21, 0x83,
        0x53, 0x60, 0x87, 0x77, 0x10, 0x87, 0x70, 0x60, 0x87, 0x79, 0x40, 0x42, 0xb5, 0x74, 0x73, 0x61, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03,
        0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x08, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x18, 0xf2, 0x20, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x51, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xa3,
        0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xe7, 0x01, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x04, 0x04, 0xc0, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30,
        0x02, 0x50, 0x08, 0xc5, 0x50, 0x30, 0x45, 0x50, 0x06, 0x74, 0x8a, 0xa6, 0x24, 0xca, 0xa0, 0x10, 0x46, 0x00, 0x4a, 0xa6, 0x04, 0xc8, 0x8c, 0x11, 0x80, 0x20, 0x08, 0xa2, 0x61, 0x40, 0x78, 0x06,
        0x80, 0xf2, 0x58, 0x0b, 0x41, 0x04, 0x02, 0x81, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c,
        0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x64, 0x82, 0x30, 0x24, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xca,
        0x06, 0x61, 0x30, 0x28, 0xa0, 0xcd, 0x4d, 0x10, 0x86, 0x65, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x01, 0xbb, 0x08, 0x4c, 0x10, 0x06, 0x66, 0x03, 0x42, 0x2c, 0x0c, 0x41, 0x0c, 0x0d, 0xb0, 0x21, 0x70,
        0x36, 0x10, 0x00, 0xf0, 0x00, 0x13, 0x04, 0x01, 0xd8, 0x00, 0x6c, 0x18, 0x08, 0x49, 0xda, 0x10, 0x4c, 0x13, 0x04, 0x21, 0xd8, 0x50, 0x0c, 0x11, 0x55, 0x51, 0x13, 0x84, 0x0c, 0xdb, 0x10, 0x5c,
        0x3c, 0xd0, 0xe6, 0xbe, 0xda, 0xc2, 0xd2, 0xdc, 0xb8, 0x4c, 0x59, 0x7d, 0x41, 0xbd, 0xcd, 0xa5, 0xd1, 0xa5, 0xbd, 0xb9, 0x4d, 0x10, 0x0e, 0x68, 0x82, 0x70, 0x44, 0x1b, 0x02, 0x62, 0x82, 0x70,
        0x48, 0x13, 0x84, 0x63, 0xda, 0xb0, 0x10, 0xda, 0xc6, 0x75, 0xde, 0xe0, 0x11, 0x1f, 0xb0, 0x21, 0x00, 0x03, 0x36, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72,
        0x13, 0x84, 0x83, 0x9a, 0x20, 0x0c, 0xcd, 0x04, 0x61, 0x70, 0x36, 0x10, 0xc4, 0x40, 0x06, 0x65, 0x30, 0x41, 0x38, 0xaa, 0x0d, 0x0b, 0x21, 0x06, 0xdb, 0x18, 0x98, 0xc1, 0x67, 0x9c, 0x01, 0xc1,
        0x01, 0x9c, 0xa6, 0xac, 0xbe, 0x92, 0xdc, 0xe6, 0xd2, 0xc8, 0xca, 0xa8, 0xca, 0xe6, 0xe6, 0x8c, 0xc2, 0xc6, 0xe8, 0xde, 0xe4, 0x26, 0x08, 0xc7, 0xb3, 0x41, 0x20, 0x86, 0x0d, 0xcb, 0x90, 0x06,
        0x9b, 0x1a, 0xac, 0xc1, 0x47, 0x06, 0x67, 0x60, 0x70, 0xc0, 0x06, 0x01, 0x0d, 0xd8, 0x60, 0xc3, 0x10, 0x06, 0x61, 0xd0, 0x06, 0x13, 0x04, 0xc3, 0xda, 0x70, 0x44, 0x86, 0x51, 0x06, 0x83, 0xf1,
        0x06, 0x1b, 0x84, 0x32, 0x80, 0x83, 0x0d, 0x45, 0x95, 0xb9, 0x01, 0x14, 0x07, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2, 0xc8, 0xca, 0xdc, 0xe8, 0xa6, 0x04, 0x41, 0x15, 0x32, 0x3c, 0x17, 0xbb,
        0x32, 0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a, 0x60, 0xd4, 0x21, 0xc3, 0x73, 0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93, 0x6b,
        0x7a, 0x23, 0x2b, 0x63, 0x9b, 0x12, 0x24, 0x65, 0xc8, 0xf0, 0x5c, 0xe4, 0xca, 0xe6, 0xde, 0xea, 0xe4, 0xc6, 0xca, 0xe6, 0xa6, 0x04, 0x4f, 0x25, 0x32, 0x3c, 0x17, 0xba, 0x3c, 0xb8, 0xb2, 0x20,
        0x37, 0xb7, 0x37, 0xba, 0x30, 0xba, 0xb4, 0x37, 0xb7, 0xb9, 0x29, 0x41, 0x55, 0x87, 0x0c, 0xcf, 0xc5, 0x2e, 0xad, 0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c, 0x4a, 0x70, 0xd5, 0x21,
        0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83, 0x7a, 0x4b, 0x73, 0xa3, 0x9b, 0x9b, 0x12, 0xc4, 0x01, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1,
        0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c,
        0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70,
        0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f,
        0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c,
        0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28,
        0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e,
        0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3,
        0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74,
        0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x05, 0xe0, 0x0a, 0x81, 0x1f, 0x3c, 0x4c, 0x1f, 0xa3, 0xbf, 0xe3, 0xb7, 0x7b, 0x4e,
        0x0f, 0xbb, 0xe9, 0x40, 0x60, 0x36, 0xf8, 0x0b, 0x56, 0xe1, 0x61, 0xfa, 0x18, 0xfd, 0x1d, 0xbf, 0xdd, 0x73, 0x7a, 0xd8, 0x4d, 0xff, 0x92, 0xc3, 0xf4, 0x30, 0x10, 0x68, 0xfd, 0x91, 0x92, 0x6e,
        0x78, 0x9d, 0x0e, 0x0d, 0xd3, 0xc7, 0x68, 0x60, 0x15, 0xfe, 0x4e, 0xbb, 0xe9, 0x5f, 0x72, 0x98, 0x1e, 0x06, 0x02, 0x49, 0xb0, 0x19, 0x10, 0x08, 0xd4, 0x9a, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8,
        0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x5f, 0xdc, 0xb6, 0x1d, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0x61,
        0x03, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x26, 0xe0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x43, 0x4d, 0x0f, 0x35, 0x71, 0x4e, 0x33, 0x45, 0xce,
        0x92, 0x3a, 0x48, 0x33, 0x89, 0x9d, 0x8f, 0xdc, 0xb6, 0x11, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x15, 0x70, 0xc3, 0xe5, 0x3b,
        0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa4, 0xc0, 0x24, 0x1c, 0x9c, 0xd3, 0x48, 0x13, 0xd0, 0x4c, 0x7e, 0x71, 0xdb, 0x16, 0x70, 0x1c, 0xd2, 0xcf, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x33, 0x00,
        0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x34, 0xca, 0xae, 0xa8, 0x03, 0x4a, 0x8e, 0x4e, 0x11, 0x50, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x20, 0x5d, 0x83, 0x50, 0x55, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x3c, 0x99, 0x11, 0x58, 0xa3, 0x09, 0x01, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x77,
        0x60, 0x98, 0x13, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0xb3, 0x21, 0x03, 0x36, 0x9a, 0x10, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x78, 0x89, 0xb6, 0x41, 0xc1, 0x88, 0xc1, 0x01,
        0x80, 0x20, 0x18, 0x3c, 0x9d, 0x52, 0x68, 0xa3, 0x09, 0x81, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x18, 0x2c, 0x1c, 0x26, 0x05, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xf0, 0x7c, 0xcc,
        0xc1, 0x8d, 0x26, 0x04, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x62, 0xd0, 0x78, 0x19, 0x15, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0x13, 0x06, 0x4e, 0xf2, 0x8d, 0x26, 0x04, 0xc0,
        0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x64, 0xf0, 0x84, 0x01, 0x18, 0x58, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x3c, 0x63, 0x00, 0x2d, 0x61, 0x30, 0x9a, 0x10, 0x04, 0x23, 0x06, 0x09,
        0x00, 0x82, 0x60, 0xa0, 0x98, 0x41, 0x34, 0x06, 0x63, 0x80, 0x05, 0x08, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x34, 0x4a, 0xa1, 0xac, 0x03, 0xe8, 0x14, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x05, 0x00, 0x82, 0x60, 0x40, 0x68, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xda, 0x44,
        0x51, 0x4a, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x46, 0x55, 0x55, 0x22, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x55, 0x96, 0x45, 0x0c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x80, 0x74, 0xd6, 0x75, 0x3d, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x1d, 0x82, 0x61, 0x0e, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x4a, 0x87, 0x60, 0x18, 0x33, 0x8c, 0x18, 0x24,
        0x00, 0x08, 0x82, 0x81, 0xd2, 0x21, 0x18, 0x66, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x74, 0x08, 0x86, 0x41, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE hs_quad_ccw = {
        use_dxil ? (const void*)hs_quad_ccw_code_dxil : (const void*)hs_quad_ccw_code_dxbc,
        use_dxil ? sizeof(hs_quad_ccw_code_dxil) : sizeof(hs_quad_ccw_code_dxbc)
    };
    static const DWORD ds_code_dxbc[] =
    {
        0x43425844, 0xeb6b7631, 0x07f5469e, 0xed0cbf4a, 0x7158b3a6, 0x00000001, 0x00000284, 0x00000004,
        0x00000030, 0x00000064, 0x00000128, 0x0000015c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x47534350, 0x000000bc, 0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x0000000b,
        0x00000003, 0x00000000, 0x00000001, 0x00000098, 0x00000001, 0x0000000b, 0x00000003, 0x00000001,
        0x00000001, 0x00000098, 0x00000002, 0x0000000b, 0x00000003, 0x00000002, 0x00000001, 0x00000098,
        0x00000003, 0x0000000b, 0x00000003, 0x00000003, 0x00000001, 0x000000a6, 0x00000000, 0x0000000c,
        0x00000003, 0x00000004, 0x00000001, 0x000000a6, 0x00000001, 0x0000000c, 0x00000003, 0x00000005,
        0x00000001, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365,
        0x6f746361, 0xabab0072, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000,
        0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x58454853,
        0x00000120, 0x00040050, 0x00000048, 0x01002093, 0x01001895, 0x0100086a, 0x0200005f, 0x0001c032,
        0x0400005f, 0x002190f2, 0x00000004, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x02000068, 0x00000002, 0x0a000000, 0x001000f2, 0x00000000, 0x80219e46, 0x00000041, 0x00000002,
        0x00000000, 0x00219e46, 0x00000003, 0x00000000, 0x09000032, 0x001000f2, 0x00000000, 0x0001c006,
        0x00100e46, 0x00000000, 0x00219e46, 0x00000002, 0x00000000, 0x0a000000, 0x001000f2, 0x00000001,
        0x80219e46, 0x00000041, 0x00000000, 0x00000000, 0x00219e46, 0x00000001, 0x00000000, 0x09000032,
        0x001000f2, 0x00000001, 0x0001c006, 0x00100e46, 0x00000001, 0x00219e46, 0x00000000, 0x00000000,
        0x08000000, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x80100e46, 0x00000041, 0x00000001,
        0x08000032, 0x001020f2, 0x00000000, 0x0001c556, 0x00100e46, 0x00000000, 0x00100e46, 0x00000001,
        0x0100003e,
    };
    static const BYTE ds_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x1a, 0xac, 0x5e, 0xb9, 0x0b, 0x74, 0xe7, 0xdf, 0xb5, 0x5c, 0x48, 0x37, 0x06, 0xb0, 0x84, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x7e, 0x09, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0xb2, 0x01, 0x00, 0x00, 0xba, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x50, 0x53, 0x47, 0x31, 0xea, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x53, 0x56,
        0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x50, 0x53, 0x56, 0x30, 0x00, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x06, 0x00, 0x01, 0x01, 0x02, 0x01, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x02, 0x04, 0x71, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xbc, 0x06,
        0x00, 0x00, 0x60, 0x00, 0x04, 0x00, 0xaf, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xa4, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c,
        0x00, 0x00, 0xa6, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01,
        0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90,
        0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18,
        0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22,
        0xa4, 0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x38, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30, 0x98, 0x23,
        0x40, 0xca, 0x30, 0x54, 0x64, 0xc5, 0x18, 0x22, 0x2a, 0x22, 0x2c, 0x06, 0x10, 0x51, 0x0d, 0xe9, 0x40, 0x40, 0x7e, 0x10, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68,
        0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
        0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07,
        0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x02,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x02, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x32, 0x1e,
        0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x18, 0x01, 0x28, 0x84, 0x62, 0x28, 0x82, 0x32, 0xa0, 0x2a, 0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x92,
        0x29, 0x81, 0xa2, 0xa1, 0x1e, 0xeb, 0x23, 0x88, 0x40, 0x20, 0x10, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6,
        0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x08, 0x62, 0x82, 0x40, 0x14, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x10, 0xc6, 0x06, 0x61, 0x30, 0x28, 0x90, 0xcd, 0x4d, 0x10, 0x88, 0x63, 0xc3, 0x80,
        0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58, 0x9c, 0x09, 0x42, 0x13, 0x6d, 0x08, 0x20, 0x1e, 0x64, 0x73, 0x5f, 0x6d, 0x61, 0x69, 0x6e,
        0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x85, 0x32, 0x41, 0x28, 0x96, 0x0d, 0x01, 0x31, 0x41, 0x28, 0x98, 0x09, 0x42, 0xd1, 0x6c, 0x58, 0x88, 0x89,
        0xaa, 0xac, 0x6b, 0xb8, 0x08, 0x0c, 0xd8, 0x10, 0x64, 0x6c, 0xa6, 0xac, 0xbe, 0xa8, 0xca, 0xe6, 0xe6, 0x8c, 0xc2, 0xc6, 0xe8, 0xde, 0xe4, 0x26, 0x08, 0x85, 0x33, 0x41, 0x20, 0x90, 0x09, 0x02,
        0x91, 0x6c, 0x20, 0x88, 0xa1, 0xf3, 0x26, 0x08, 0xc5, 0xb3, 0x61, 0x21, 0x36, 0x8a, 0xfb, 0x30, 0x03, 0x0c, 0x88, 0x0a, 0xe0, 0x34, 0x65, 0xf5, 0x95, 0xe4, 0x36, 0x97, 0x46, 0x56, 0x46, 0x55,
        0x36, 0x37, 0x67, 0x14, 0x36, 0x46, 0xf7, 0x26, 0x37, 0x41, 0x28, 0xa0, 0x0d, 0x02, 0x31, 0x6c, 0x58, 0x06, 0x31, 0xa0, 0xc6, 0x80, 0x0c, 0xb0, 0x0e, 0x0c, 0x8c, 0x0a, 0xd8, 0x20, 0x84, 0x41,
        0x19, 0x6c, 0x18, 0x34, 0xcd, 0x0c, 0x36, 0x08, 0x9e, 0xb1, 0x41, 0xe8, 0xd0, 0x60, 0x43, 0xb1, 0x48, 0x67, 0x00, 0xa4, 0x41, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba,
        0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x18, 0x75, 0xc8, 0xf0,
        0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x49, 0x25, 0x32, 0x3c, 0x17, 0xba, 0x3c, 0xb8, 0xb2, 0x20, 0x37, 0xb7, 0x37, 0xba, 0x30, 0xba, 0xb4, 0x37,
        0xb7, 0xb9, 0x29, 0x81, 0x53, 0x87, 0x0c, 0xcf, 0xc5, 0x2e, 0xad, 0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c, 0x4a, 0x00, 0xd5, 0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83,
        0x7a, 0x4b, 0x73, 0xa3, 0x9b, 0x9b, 0x12, 0xa4, 0x01, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38,
        0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c,
        0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70,
        0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10,
        0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0,
        0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77,
        0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74,
        0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00,
        0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x06, 0x70, 0x0c, 0xd2, 0xcf, 0x00, 0x48, 0x63, 0x01, 0xd8, 0x70, 0xf9, 0xce, 0xe3, 0x0f, 0x0e, 0x03, 0x20, 0x4d, 0xe9, 0x08, 0xc0,
        0x84, 0x38, 0x8d, 0x5f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0xdb, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44,
        0x04, 0x4a, 0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x03, 0x00, 0x61, 0x20, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x44, 0xa5,
        0x50, 0xd2, 0x01, 0x54, 0x45, 0x00, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x60, 0x50, 0x02, 0x32, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x46, 0x35, 0x1c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80,
        0x54, 0x91, 0x24, 0x29, 0xd3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x96, 0x34, 0x4d, 0x09, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x35, 0x51, 0x54, 0x51, 0x8d, 0x18, 0x24, 0x00,
        0x08, 0x82, 0x01, 0x82, 0x51, 0x55, 0xf5, 0x58, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x64, 0x95, 0x65, 0x39, 0xd6, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x9a, 0x75, 0x5d, 0xcd, 0x35,
        0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x76, 0x61, 0x58, 0x82, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x61, 0x59, 0x36, 0x65, 0x86, 0x10, 0xf1, 0x31, 0x84, 0x88, 0x8f, 0x21, 0x44,
        0x7c, 0x0c, 0x21, 0xe2, 0x63, 0x84, 0x23, 0x1f, 0x23, 0x1e, 0xf9, 0x18, 0x01, 0xc9, 0xc7, 0x88, 0x48, 0x3e, 0x46, 0x30, 0xf0, 0x31, 0x82, 0x81, 0x8f, 0x11, 0x0c, 0x7c, 0x8c, 0x60, 0xe0, 0x33,
        0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x1a, 0x94, 0x81, 0x19, 0x98, 0x81, 0x27, 0x06, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xa8, 0x81, 0x19, 0x9c, 0xc1, 0x19, 0x74, 0x63, 0x30, 0x62,
        0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x1a, 0x9c, 0x01, 0x1a, 0xa0, 0x41, 0x46, 0x06, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb0, 0x01, 0x1a, 0xa4, 0x41, 0x1a, 0x8c, 0x41, 0x19, 0x8c, 0x18,
        0x24, 0x00, 0x08, 0x82, 0x01, 0xd2, 0x06, 0x69, 0xa0, 0x06, 0x6a, 0x20, 0x06, 0x67, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x88, 0x1b, 0xa8, 0xc1, 0x1a, 0xac, 0x41, 0x18, 0xa0, 0xc1, 0x88,
        0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x6f, 0xb0, 0x06, 0x6c, 0xc0, 0x06, 0x5d, 0x1a, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x02, 0x07, 0x6c, 0xd0, 0x06, 0x6d, 0x70, 0x06, 0x6a, 0x60, 0x08,
        0x11, 0x1f, 0x43, 0x88, 0xf8, 0x18, 0x42, 0xc4, 0xc7, 0x10, 0x22, 0x3e, 0x46, 0x88, 0x81, 0x7c, 0x8c, 0x18, 0x03, 0xf9, 0x18, 0x41, 0x06, 0xf2, 0x31, 0xa2, 0x0c, 0xe4, 0x63, 0x0c, 0x15, 0x1f,
        0x0b, 0x0a, 0xf8, 0x58, 0x53, 0xc5, 0xc7, 0x02, 0x03, 0x3e, 0xe6, 0x58, 0xf1, 0xb1, 0xe0, 0x80, 0x8f, 0x3d, 0x57, 0x7c, 0x2c, 0x40, 0xe0, 0x63, 0x47, 0x1b, 0xc8, 0xc7, 0x0c, 0x37, 0x90, 0x8f,
        0x15, 0x6f, 0x20, 0x1f, 0x23, 0xe0, 0x40, 0x3e, 0x46, 0x80, 0x01, 0x7c, 0x8c, 0x00, 0x03, 0xf8, 0x18, 0x01, 0x06, 0xf0, 0x31, 0x02, 0x0c, 0xe0, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a,
        0x2a, 0xe4, 0x81, 0x29, 0x98, 0x82, 0x1f, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0xa0, 0x42, 0x1e, 0x98, 0x82, 0x29, 0xf0, 0xc1, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x2a,
        0xe4, 0x81, 0x29, 0x98, 0xc2, 0x1d, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0xa0, 0x42, 0x1e, 0x98, 0x82, 0x29, 0x80, 0x42, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE ds = {
        use_dxil ? (const void*)ds_code_dxil : (const void*)ds_code_dxbc,
        use_dxil ? sizeof(ds_code_dxil) : sizeof(ds_code_dxbc)
    };
#if 0
    ...
    [outputtopology("triangle_cw")]
    ...
#endif
    static const DWORD hs_quad_cw_code_dxbc[] =
    {
        0x43425844, 0x1ab30cc8, 0x94174771, 0x61f4cdd0, 0xa287f62c, 0x00000001, 0x000002b8, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000015c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x47534350, 0x000000bc,
        0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x0000000b, 0x00000003, 0x00000000, 0x00000e01,
        0x00000098, 0x00000001, 0x0000000b, 0x00000003, 0x00000001, 0x00000e01, 0x00000098, 0x00000002,
        0x0000000b, 0x00000003, 0x00000002, 0x00000e01, 0x00000098, 0x00000003, 0x0000000b, 0x00000003,
        0x00000003, 0x00000e01, 0x000000a6, 0x00000000, 0x0000000c, 0x00000003, 0x00000004, 0x00000e01,
        0x000000a6, 0x00000001, 0x0000000c, 0x00000003, 0x00000005, 0x00000e01, 0x545f5653, 0x46737365,
        0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x58454853,
        0x00000154, 0x00030050, 0x00000055, 0x01000071, 0x01002093, 0x01002094, 0x01001895, 0x01000896,
        0x01001897, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002, 0x01000073, 0x04000067,
        0x00102012, 0x00000000, 0x0000000b, 0x06000036, 0x00102012, 0x00000000, 0x0020800a, 0x00000000,
        0x00000000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000001, 0x0000000c, 0x06000036,
        0x00102012, 0x00000001, 0x0020801a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000002, 0x0000000d, 0x06000036, 0x00102012, 0x00000002, 0x0020802a, 0x00000000,
        0x00000000, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x0000000e, 0x06000036,
        0x00102012, 0x00000003, 0x0020803a, 0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000004, 0x0000000f, 0x06000036, 0x00102012, 0x00000004, 0x0020800a, 0x00000000,
        0x00000001, 0x0100003e, 0x01000073, 0x04000067, 0x00102012, 0x00000005, 0x00000010, 0x06000036,
        0x00102012, 0x00000005, 0x0020801a, 0x00000000, 0x00000001, 0x0100003e,
    };
    static const BYTE hs_quad_cw_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x3d, 0x10, 0x45, 0x5f, 0x37, 0xd9, 0xce, 0xce, 0x71, 0x8f, 0xdb, 0x35, 0x25, 0xf0, 0x4c, 0x25, 0x01, 0x00, 0x00, 0x00, 0x1a, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0xb2, 0x01, 0x00, 0x00, 0x7e, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x50, 0x53, 0x47, 0x31, 0xea, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x53, 0x56,
        0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x50, 0x53, 0x56, 0x30, 0xc4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x06, 0x00, 0x01, 0x01, 0x02, 0x01, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x04, 0x71, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58,
        0x49, 0x4c, 0x94, 0x07, 0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0xe5, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x7c, 0x07, 0x00, 0x00, 0x42, 0x43,
        0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xdc, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10,
        0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32,
        0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29,
        0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30,
        0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20,
        0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84,
        0xa4, 0x4c, 0x10, 0x68, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x08, 0x63, 0x10, 0x99, 0x01, 0x98, 0x23, 0x40, 0x8a, 0x61, 0xc6, 0x18, 0x67, 0x10, 0x2a, 0x06, 0x18, 0x63, 0x1c,
        0x43, 0xea, 0xa8, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x44, 0x18, 0x63, 0x0c, 0x9d, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10,
        0x68, 0x86, 0x85, 0x40, 0x41, 0x2b, 0x04, 0x1b, 0x6e, 0xd0, 0x9b, 0x23, 0x08, 0x8a, 0xe1, 0xc6, 0x19, 0x23, 0x92, 0x1c, 0x08, 0x18, 0x46, 0x20, 0x8c, 0x61, 0x04, 0xc1, 0x98, 0x09, 0x21, 0x83,
        0x53, 0x60, 0x87, 0x77, 0x10, 0x87, 0x70, 0x60, 0x87, 0x79, 0x40, 0x42, 0xb5, 0x74, 0x73, 0x61, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03,
        0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x08, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x18, 0xf2, 0x20, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x51, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xa3,
        0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xe7, 0x01, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x04, 0x04, 0xc0, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30,
        0x02, 0x50, 0x08, 0xc5, 0x50, 0x30, 0x45, 0x50, 0x06, 0x74, 0x8a, 0xa6, 0x24, 0xca, 0xa0, 0x10, 0x46, 0x00, 0x4a, 0xa6, 0x04, 0xc8, 0x8c, 0x11, 0x80, 0x20, 0x08, 0xa2, 0x61, 0x40, 0x78, 0x06,
        0x80, 0xf2, 0x58, 0x0b, 0x41, 0x04, 0x02, 0x81, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c,
        0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x64, 0x82, 0x30, 0x24, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xca,
        0x06, 0x61, 0x30, 0x28, 0xa0, 0xcd, 0x4d, 0x10, 0x86, 0x65, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x01, 0xbb, 0x08, 0x4c, 0x10, 0x06, 0x66, 0x03, 0x42, 0x2c, 0x0c, 0x41, 0x0c, 0x0d, 0xb0, 0x21, 0x70,
        0x36, 0x10, 0x00, 0xf0, 0x00, 0x13, 0x04, 0x01, 0xd8, 0x00, 0x6c, 0x18, 0x08, 0x49, 0xda, 0x10, 0x4c, 0x13, 0x04, 0x21, 0xd8, 0x50, 0x0c, 0x11, 0x55, 0x51, 0x13, 0x84, 0x0c, 0xdb, 0x10, 0x5c,
        0x3c, 0xd0, 0xe6, 0xbe, 0xda, 0xc2, 0xd2, 0xdc, 0xb8, 0x4c, 0x59, 0x7d, 0x41, 0xbd, 0xcd, 0xa5, 0xd1, 0xa5, 0xbd, 0xb9, 0x4d, 0x10, 0x0e, 0x68, 0x82, 0x70, 0x44, 0x1b, 0x02, 0x62, 0x82, 0x70,
        0x48, 0x13, 0x84, 0x63, 0xda, 0xb0, 0x10, 0xda, 0xc6, 0x75, 0xde, 0xe0, 0x11, 0x1f, 0xb0, 0x21, 0x00, 0x03, 0x36, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72,
        0x13, 0x84, 0x83, 0x9a, 0x20, 0x0c, 0xcd, 0x04, 0x61, 0x70, 0x36, 0x10, 0xc4, 0x40, 0x06, 0x65, 0x30, 0x41, 0x38, 0xaa, 0x0d, 0x0b, 0x21, 0x06, 0xdb, 0x18, 0x98, 0xc1, 0x67, 0x9c, 0x01, 0xc1,
        0x01, 0x9c, 0xa6, 0xac, 0xbe, 0x92, 0xdc, 0xe6, 0xd2, 0xc8, 0xca, 0xa8, 0xca, 0xe6, 0xe6, 0x8c, 0xc2, 0xc6, 0xe8, 0xde, 0xe4, 0x26, 0x08, 0xc7, 0xb3, 0x41, 0x20, 0x86, 0x0d, 0xcb, 0x90, 0x06,
        0x9b, 0x1a, 0xac, 0xc1, 0x47, 0x06, 0x67, 0x60, 0x70, 0xc0, 0x06, 0x01, 0x0d, 0xd8, 0x60, 0xc3, 0x10, 0x06, 0x61, 0xd0, 0x06, 0x13, 0x04, 0xc3, 0xda, 0x70, 0x44, 0x86, 0x51, 0x06, 0x43, 0x19,
        0xbc, 0xc1, 0x06, 0xa1, 0x0c, 0xe0, 0x60, 0x43, 0x51, 0x65, 0x6e, 0x00, 0xc5, 0x41, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5,
        0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x18, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4,
        0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x49, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0xc1, 0x53, 0x89, 0x0c, 0xcf, 0x85, 0x2e, 0x0f, 0xae, 0x2c,
        0xc8, 0xcd, 0xed, 0x8d, 0x2e, 0x8c, 0x2e, 0xed, 0xcd, 0x6d, 0x6e, 0x4a, 0x50, 0xd5, 0x21, 0xc3, 0x73, 0xb1, 0x4b, 0x2b, 0xbb, 0x4b, 0x22, 0x9b, 0xa2, 0x0b, 0xa3, 0x2b, 0x9b, 0x12, 0x5c, 0x75,
        0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x71, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1,
        0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c,
        0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70,
        0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f,
        0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c,
        0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28,
        0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e,
        0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3,
        0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74,
        0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x05, 0xe0, 0x0a, 0x81, 0x1f, 0x3c, 0x4c, 0x1f, 0xa3, 0xbf, 0xe3, 0xb7, 0x7b, 0x4e,
        0x0f, 0xbb, 0xe9, 0x40, 0x60, 0x36, 0xf8, 0x0b, 0x56, 0xe1, 0x61, 0xfa, 0x18, 0xfd, 0x1d, 0xbf, 0xdd, 0x73, 0x7a, 0xd8, 0x4d, 0xff, 0x92, 0xc3, 0xf4, 0x30, 0x10, 0x68, 0xfd, 0x91, 0x92, 0x6e,
        0x78, 0x9d, 0x0e, 0x0d, 0xd3, 0xc7, 0x68, 0x60, 0x15, 0xfe, 0x4e, 0xbb, 0xe9, 0x5f, 0x72, 0x98, 0x1e, 0x06, 0x02, 0x49, 0xb0, 0x19, 0x10, 0x08, 0xd4, 0x9a, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8,
        0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x5f, 0xdc, 0xb6, 0x1d, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0x61,
        0x03, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x26, 0xe0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x43, 0x4d, 0x0f, 0x35, 0x71, 0x4e, 0x33, 0x45, 0xce,
        0x92, 0x3a, 0x48, 0x33, 0x89, 0x9d, 0x8f, 0xdc, 0xb6, 0x11, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x15, 0x70, 0xc3, 0xe5, 0x3b,
        0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa4, 0xc0, 0x24, 0x1c, 0x9c, 0xd3, 0x48, 0x13, 0xd0, 0x4c, 0x7e, 0x71, 0xdb, 0x16, 0x70, 0x1c, 0xd2, 0xcf, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x33, 0x00,
        0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x34, 0xca, 0xae, 0xa8, 0x03, 0x4a, 0x8e, 0x4e, 0x11, 0x50, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x20, 0x5d, 0x83, 0x50, 0x55, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x3c, 0x99, 0x11, 0x58, 0xa3, 0x09, 0x01, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x77,
        0x60, 0x98, 0x13, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0xb3, 0x21, 0x03, 0x36, 0x9a, 0x10, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x78, 0x89, 0xb6, 0x41, 0xc1, 0x88, 0xc1, 0x01,
        0x80, 0x20, 0x18, 0x3c, 0x9d, 0x52, 0x68, 0xa3, 0x09, 0x81, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x18, 0x2c, 0x1c, 0x26, 0x05, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xf0, 0x7c, 0xcc,
        0xc1, 0x8d, 0x26, 0x04, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x62, 0xd0, 0x78, 0x19, 0x15, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0x13, 0x06, 0x4e, 0xf2, 0x8d, 0x26, 0x04, 0xc0,
        0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x64, 0xf0, 0x84, 0x01, 0x18, 0x58, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x3c, 0x63, 0x00, 0x2d, 0x61, 0x30, 0x9a, 0x10, 0x04, 0x23, 0x06, 0x09,
        0x00, 0x82, 0x60, 0xa0, 0x98, 0x41, 0x34, 0x06, 0x63, 0x80, 0x05, 0x08, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x34, 0x4a, 0xa1, 0xac, 0x03, 0xe8, 0x14, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x05, 0x00, 0x82, 0x60, 0x40, 0x68, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xda, 0x44,
        0x51, 0x4a, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x46, 0x55, 0x55, 0x22, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x55, 0x96, 0x45, 0x0c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x80, 0x74, 0xd6, 0x75, 0x3d, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x1d, 0x82, 0x61, 0x0e, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x4a, 0x87, 0x60, 0x18, 0x33, 0x8c, 0x18, 0x24,
        0x00, 0x08, 0x82, 0x81, 0xd2, 0x21, 0x18, 0x66, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x74, 0x08, 0x86, 0x41, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE hs_quad_cw = {
        use_dxil ? (const void*)hs_quad_cw_code_dxil : (const void*)hs_quad_cw_code_dxbc,
        use_dxil ? sizeof(hs_quad_cw_code_dxil) : sizeof(hs_quad_cw_code_dxbc)
    };
    static const struct vec4 quad[] =
    {
        {-1.0f, -1.0f, 0.0f, 1.0f},
        {-1.0f,  1.0f, 0.0f, 1.0f},
        { 1.0f, -1.0f, 0.0f, 1.0f},
        { 1.0f,  1.0f, 0.0f, 1.0f},
    };
    static const D3D12_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 0, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
    };
    static const D3D12_SO_DECLARATION_ENTRY so_declaration[] =
    {
        {0, "SV_POSITION", 0, 0, 4, 0},
    };
    unsigned int strides[] = {16};
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    static const BYTE zero_data[2048];
    static const struct triangle expected_quad_ccw[] =
    {
        {{{-1.0f, -1.0f, 0.0f, 1.0f},
          { 1.0f, -1.0f, 0.0f, 1.0f},
          {-1.0f,  1.0f, 0.0f, 1.0f}}},
        {{{-1.0f,  1.0f, 0.0f, 1.0f},
          { 1.0f, -1.0f, 0.0f, 1.0f},
          { 1.0f,  1.0f, 0.0f, 1.0f}}},
        {{{ 0.0f,  0.0f, 0.0f, 0.0f},
          { 0.0f,  0.0f, 0.0f, 0.0f},
          { 0.0f,  0.0f, 0.0f, 0.0f}}},
    };
    static const struct triangle expected_quad_cw[] =
    {
        {{{-1.0f, -1.0f, 0.0f, 1.0f},
          {-1.0f,  1.0f, 0.0f, 1.0f},
          { 1.0f, -1.0f, 0.0f, 1.0f}}},
        {{{-1.0f,  1.0f, 0.0f, 1.0f},
          { 1.0f,  1.0f, 0.0f, 1.0f},
          { 1.0f, -1.0f, 0.0f, 1.0f}}},
        {{{ 0.0f,  0.0f, 0.0f, 0.0f},
          { 0.0f,  0.0f, 0.0f, 0.0f},
          { 0.0f,  0.0f, 0.0f, 0.0f}}},
    };
    struct
    {
        float tess_factors[4];
        float inside_tess_factors[2];
        uint32_t padding[2];
    } constant;

    ID3D12Resource *vb, *so_buffer, *upload_buffer, *readback_buffer;
    D3D12_QUERY_DATA_SO_STATISTICS *so_statistics;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    D3D12_QUERY_HEAP_DESC query_heap_desc;
    D3D12_STREAM_OUTPUT_BUFFER_VIEW sobv;
    D3D12_INPUT_LAYOUT_DESC input_layout;
    struct test_context_desc desc;
    D3D12_VERTEX_BUFFER_VIEW vbv;
    struct resource_readback rb;
    ID3D12QueryHeap *query_heap;
    struct test_context context;
    ID3D12CommandQueue *queue;
    ID3D12Device *device;
    unsigned int i;
    HRESULT hr;

    memset(&desc, 0, sizeof(desc));
    desc.no_root_signature = true;
    if (!init_test_context(&context, &desc))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    device = context.device;
    command_list = context.list;
    queue = context.queue;

    query_heap_desc.Type = D3D12_QUERY_HEAP_TYPE_SO_STATISTICS;
    query_heap_desc.Count = 2;
    query_heap_desc.NodeMask = 0;
    hr = ID3D12Device_CreateQueryHeap(device, &query_heap_desc, &IID_ID3D12QueryHeap, (void **)&query_heap);
    if (hr == E_NOTIMPL)
    {
        skip("Stream output is not supported.\n");
        destroy_test_context(&context);
        return;
    }
    ok(hr == S_OK, "Failed to create query heap, hr %#x.\n", hr);

    context.root_signature = create_32bit_constants_root_signature_(__LINE__,
            device, 0, 6, D3D12_SHADER_VISIBILITY_HULL,
            D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT
            | D3D12_ROOT_SIGNATURE_FLAG_ALLOW_STREAM_OUTPUT);

    input_layout.pInputElementDescs = layout_desc;
    input_layout.NumElements = ARRAY_SIZE(layout_desc);

    if (use_dxil)
    {
        init_pipeline_state_desc_dxil(&pso_desc, context.root_signature,
                context.render_target_desc.Format, NULL, NULL, &input_layout);
    }
    else
    {
        init_pipeline_state_desc(&pso_desc, context.root_signature,
                context.render_target_desc.Format, NULL, NULL, &input_layout);
    }
    pso_desc.VS = vs;
    pso_desc.HS = hs_quad_cw;
    pso_desc.DS = ds;
    pso_desc.StreamOutput.NumEntries = ARRAY_SIZE(so_declaration);
    pso_desc.StreamOutput.pSODeclaration = so_declaration;
    pso_desc.StreamOutput.pBufferStrides = strides;
    pso_desc.StreamOutput.NumStrides = ARRAY_SIZE(strides);
    pso_desc.StreamOutput.RasterizedStream = 0;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;

    vb = create_upload_buffer(device, sizeof(quad), quad);

    vbv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(vb);
    vbv.StrideInBytes = sizeof(*quad);
    vbv.SizeInBytes = sizeof(quad);

    upload_buffer = create_upload_buffer(device, sizeof(zero_data), &zero_data);

    so_buffer = create_default_buffer(device, sizeof(zero_data),
            D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);

    ID3D12GraphicsCommandList_CopyBufferRegion(command_list, so_buffer, 0,
            upload_buffer, 0, sizeof(zero_data));
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_STREAM_OUT);

    sobv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(so_buffer);
    sobv.SizeInBytes = 1024;
    sobv.BufferFilledSizeLocation = sobv.BufferLocation + sobv.SizeInBytes;

    for (i = 0; i < ARRAY_SIZE(constant.tess_factors); ++i)
        constant.tess_factors[i] = 1.0f;
    for (i = 0; i < ARRAY_SIZE(constant.inside_tess_factors); ++i)
        constant.inside_tess_factors[i] = 1.0f;

    pso_desc.HS = hs_quad_ccw;
    hr = ID3D12Device_CreateGraphicsPipelineState(device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_4_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(command_list, 0, 6, &constant, 0);
    ID3D12GraphicsCommandList_IASetVertexBuffers(command_list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_SOSetTargets(command_list, 0, 1, &sobv);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xffffffff, 0);

    reset_command_list(command_list, context.allocator);
    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_RENDER_TARGET);

    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_STREAM_OUT, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_triangles(so_buffer, queue, command_list, expected_quad_ccw, ARRAY_SIZE(expected_quad_ccw));

    reset_command_list(command_list, context.allocator);
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_COPY_DEST);
    ID3D12GraphicsCommandList_CopyBufferRegion(command_list, so_buffer, 0,
            upload_buffer, 0, sizeof(zero_data));
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_STREAM_OUT);

    ID3D12PipelineState_Release(context.pipeline_state);
    pso_desc.HS = hs_quad_cw;
    hr = ID3D12Device_CreateGraphicsPipelineState(device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_4_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(command_list, 0, 6, &constant, 0);
    ID3D12GraphicsCommandList_IASetVertexBuffers(command_list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_SOSetTargets(command_list, 0, 1, &sobv);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff00ff00, 0);

    reset_command_list(command_list, context.allocator);
    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_RENDER_TARGET);

    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_STREAM_OUT, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_triangles(so_buffer, queue, command_list, expected_quad_cw, ARRAY_SIZE(expected_quad_cw));

    reset_command_list(command_list, context.allocator);
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_COPY_DEST);
    ID3D12GraphicsCommandList_CopyBufferRegion(command_list, so_buffer, 0,
            upload_buffer, 0, sizeof(zero_data));
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_STREAM_OUT);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_4_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_IASetVertexBuffers(command_list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_SOSetTargets(command_list, 0, 1, &sobv);

    ID3D12GraphicsCommandList_BeginQuery(command_list, query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 0);

    for (i = 0; i < ARRAY_SIZE(constant.tess_factors); ++i)
        constant.tess_factors[i] = 2.0f;
    ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(command_list, 0, 6, &constant, 0);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    constant.tess_factors[0] = 0.0f; /* A patch is discarded. */
    ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(command_list, 0, 6, &constant, 0);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    ID3D12GraphicsCommandList_EndQuery(command_list, query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 0);

    ID3D12GraphicsCommandList_BeginQuery(command_list, query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 1);

    constant.tess_factors[0] = 5.0f;
    ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(command_list, 0, 6, &constant, 0);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    ID3D12GraphicsCommandList_EndQuery(command_list, query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 1);

    readback_buffer = create_readback_buffer(device, 2 * sizeof(*so_statistics));
    ID3D12GraphicsCommandList_ResolveQueryData(command_list,
            query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 0, 2, readback_buffer, 0);

    get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);
    so_statistics = get_readback_data(&rb, 0, 0, 0, sizeof(*so_statistics));
    ok(so_statistics[0].NumPrimitivesWritten == 8, "Got unexpected primitives written %u.\n",
            (unsigned int)so_statistics[0].NumPrimitivesWritten);
    ok(so_statistics[0].PrimitivesStorageNeeded == 8, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)so_statistics[0].PrimitivesStorageNeeded);
    ok(so_statistics[1].NumPrimitivesWritten == 11, "Got unexpected primitives written %u.\n",
            (unsigned int)so_statistics[1].NumPrimitivesWritten);
    ok(so_statistics[1].PrimitivesStorageNeeded == 11, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)so_statistics[1].PrimitivesStorageNeeded);
    release_resource_readback(&rb);

    ID3D12Resource_Release(readback_buffer);
    ID3D12Resource_Release(so_buffer);
    ID3D12Resource_Release(upload_buffer);
    ID3D12Resource_Release(vb);
    ID3D12QueryHeap_Release(query_heap);
    destroy_test_context(&context);
}

void test_quad_tessellation_dxbc(void)
{
    test_quad_tessellation(false);
}

void test_quad_tessellation_dxil(void)
{
    test_quad_tessellation(true);
}

void test_tessellation_dcl_index_range(void)
{
    static const DWORD vs_code[] =
    {
#if 0
        void main(float4 in_position : POSITION, out float4 out_position : SV_POSITION)
        {
            out_position = in_position;
        }
#endif
        0x43425844, 0xa7a2f22d, 0x83ff2560, 0xe61638bd, 0x87e3ce90, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE vs = {vs_code, sizeof(vs_code)};
#if 0
    struct point_data
    {
        float4 position : SV_POSITION;
    };

    struct patch_constant_data
    {
        float edges[4] : SV_TessFactor;
        float inside[2] : SV_InsideTessFactor;
    };

    patch_constant_data patch_constant(InputPatch<point_data, 4> input)
    {
        patch_constant_data output;

        output.edges[0] = 1.0f;
        output.edges[1] = 1.0f;
        output.edges[2] = 1.0f;
        output.edges[3] = 1.0f;
        output.inside[0] = 1.0f;
        output.inside[1] = 1.0f;

        return output;
    }

    [domain("quad")]
    [outputcontrolpoints(4)]
    [outputtopology("triangle_cw")]
    [partitioning("integer")]
    [patchconstantfunc("patch_constant")]
    point_data hs_main(InputPatch<point_data, 4> input,
            uint i : SV_OutputControlPointID)
    {
        return input[i];
    }

    [domain("quad")]
    point_data ds_main(patch_constant_data input,
            float2 tess_coord : SV_DomainLocation,
            const OutputPatch<point_data, 4> patch)
    {
        point_data output;

        float4 a = lerp(patch[0].position, patch[1].position, tess_coord.x);
        float4 b = lerp(patch[2].position, patch[3].position, tess_coord.x);
        output.position = lerp(a, b, tess_coord.y);

        return output;
    }
#endif
    static const DWORD hs_code[] =
    {
        0x43425844, 0x0a619042, 0x424471f9, 0x9f0f4ff1, 0x065efacc, 0x00000001, 0x0000029c, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000015c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x47534350, 0x000000bc,
        0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x0000000b, 0x00000003, 0x00000000, 0x00000e01,
        0x00000098, 0x00000001, 0x0000000b, 0x00000003, 0x00000001, 0x00000e01, 0x00000098, 0x00000002,
        0x0000000b, 0x00000003, 0x00000002, 0x00000e01, 0x00000098, 0x00000003, 0x0000000b, 0x00000003,
        0x00000003, 0x00000e01, 0x000000a6, 0x00000000, 0x0000000c, 0x00000003, 0x00000004, 0x00000e01,
        0x000000a6, 0x00000001, 0x0000000c, 0x00000003, 0x00000005, 0x00000e01, 0x545f5653, 0x46737365,
        0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x58454853,
        0x00000138, 0x00030050, 0x0000004e, 0x01000071, 0x01002093, 0x01002094, 0x01001895, 0x01000896,
        0x01001897, 0x0100086a, 0x01000073, 0x02000099, 0x00000004, 0x0200005f, 0x00017000, 0x04000067,
        0x00102012, 0x00000000, 0x0000000b, 0x04000067, 0x00102012, 0x00000001, 0x0000000c, 0x04000067,
        0x00102012, 0x00000002, 0x0000000d, 0x04000067, 0x00102012, 0x00000003, 0x0000000e, 0x02000068,
        0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000004, 0x04000036, 0x00100012, 0x00000000,
        0x0001700a, 0x06000036, 0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
        0x01000073, 0x02000099, 0x00000002, 0x0200005f, 0x00017000, 0x04000067, 0x00102012, 0x00000004,
        0x0000000f, 0x04000067, 0x00102012, 0x00000005, 0x00000010, 0x02000068, 0x00000001, 0x0400005b,
        0x00102012, 0x00000004, 0x00000002, 0x04000036, 0x00100012, 0x00000000, 0x0001700a, 0x07000036,
        0x00d02012, 0x00000004, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs = {hs_code, sizeof(hs_code)};
    static const DWORD ds_code[] =
    {
        0x43425844, 0x4f187d50, 0x6743fe93, 0x10dfbe63, 0xf8cfd202, 0x00000001, 0x00000284, 0x00000004,
        0x00000030, 0x00000064, 0x00000128, 0x0000015c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x47534350, 0x000000bc, 0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x0000000b,
        0x00000003, 0x00000000, 0x00000001, 0x00000098, 0x00000001, 0x0000000b, 0x00000003, 0x00000001,
        0x00000001, 0x00000098, 0x00000002, 0x0000000b, 0x00000003, 0x00000002, 0x00000001, 0x00000098,
        0x00000003, 0x0000000b, 0x00000003, 0x00000003, 0x00000001, 0x000000a6, 0x00000000, 0x0000000c,
        0x00000003, 0x00000004, 0x00000001, 0x000000a6, 0x00000001, 0x0000000c, 0x00000003, 0x00000005,
        0x00000001, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365,
        0x6f746361, 0xabab0072, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000,
        0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x58454853,
        0x00000120, 0x00040050, 0x00000048, 0x01002093, 0x01001895, 0x0100086a, 0x0200005f, 0x0001c032,
        0x0400005f, 0x002190f2, 0x00000004, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x02000068, 0x00000002, 0x0a000000, 0x001000f2, 0x00000000, 0x80219e46, 0x00000041, 0x00000002,
        0x00000000, 0x00219e46, 0x00000003, 0x00000000, 0x09000032, 0x001000f2, 0x00000000, 0x0001c006,
        0x00100e46, 0x00000000, 0x00219e46, 0x00000002, 0x00000000, 0x0a000000, 0x001000f2, 0x00000001,
        0x80219e46, 0x00000041, 0x00000000, 0x00000000, 0x00219e46, 0x00000001, 0x00000000, 0x09000032,
        0x001000f2, 0x00000001, 0x0001c006, 0x00100e46, 0x00000001, 0x00219e46, 0x00000000, 0x00000000,
        0x08000000, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x80100e46, 0x00000041, 0x00000001,
        0x08000032, 0x001020f2, 0x00000000, 0x0001c556, 0x00100e46, 0x00000000, 0x00100e46, 0x00000001,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ds = {ds_code, sizeof(ds_code)};
    static const struct vec4 quad[] =
    {
        {-1.0f, -1.0f, 0.0f, 1.0f},
        {-1.0f,  1.0f, 0.0f, 1.0f},
        { 1.0f, -1.0f, 0.0f, 1.0f},
        { 1.0f,  1.0f, 0.0f, 1.0f},
    };
    static const D3D12_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 0, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
    };
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};

    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    D3D12_INPUT_LAYOUT_DESC input_layout;
    struct test_context_desc desc;
    D3D12_VERTEX_BUFFER_VIEW vbv;
    struct test_context context;
    ID3D12CommandQueue *queue;
    ID3D12Device *device;
    ID3D12Resource *vb;
    HRESULT hr;

    memset(&desc, 0, sizeof(desc));
    desc.root_signature_flags = D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT;
    desc.no_pipeline = true;
    if (!init_test_context(&context, &desc))
        return;
    device = context.device;
    command_list = context.list;
    queue = context.queue;

    input_layout.pInputElementDescs = layout_desc;
    input_layout.NumElements = ARRAY_SIZE(layout_desc);

    init_pipeline_state_desc(&pso_desc, context.root_signature,
            context.render_target_desc.Format, NULL, NULL, &input_layout);
    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    hr = ID3D12Device_CreateGraphicsPipelineState(device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    vb = create_upload_buffer(device, sizeof(quad), quad);

    vbv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(vb);
    vbv.StrideInBytes = sizeof(*quad);
    vbv.SizeInBytes = sizeof(quad);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_4_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_IASetVertexBuffers(command_list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff00ff00, 0);

    ID3D12Resource_Release(vb);
    destroy_test_context(&context);
}

static void test_hull_shader_control_point_phase(bool use_dxil)
{
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    struct test_context_desc desc;
    struct test_context context;
    ID3D12CommandQueue *queue;
    HRESULT hr;

#if 0
    void main()
    {
    }
#endif
    static const DWORD vs_code_dxbc[] =
    {
        0x43425844, 0x590b08ae, 0x11d28adb, 0x825a5628, 0x34c0c208, 0x00000001, 0x00000064, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000010, 0x00010050, 0x00000004, 0x0100086a,
        0x0100003e,
    };
    static const BYTE vs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x67, 0x4b, 0x48, 0x78, 0x42, 0x07, 0x4a, 0xf2, 0x68, 0xc3, 0xd1, 0xf0, 0x88, 0xec, 0x01, 0xef, 0x01, 0x00, 0x00, 0x00, 0x04, 0x04, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xa4, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x38, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x58, 0x03, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x40, 0x03, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x0c, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x64, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x32, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x21, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x19, 0x46, 0x06, 0x89, 0x20, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x22, 0xc8, 0x08, 0x20, 0x64, 0x85, 0x04, 0x93, 0x21, 0xa4, 0x84, 0x04, 0x93, 0x21, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x86, 0x8c, 0x0b, 0x84, 0x64, 0x4c, 0x10, 0x14, 0x23, 0x00, 0x25, 0x00, 0x65, 0x20, 0x60, 0x8e, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0,
        0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x0c,
        0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x62, 0x09, 0x8c, 0x00, 0x14, 0x42, 0x31, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x88, 0x60, 0x82, 0x40, 0x08, 0x1b,
        0x84, 0x81, 0x98, 0x20, 0x10, 0xc3, 0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10, 0x08, 0x62, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x86, 0x81, 0x61, 0x36, 0x04, 0xcd,
        0x86, 0x61, 0x58, 0x1c, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x1b, 0x8a, 0x05, 0x02, 0x00, 0xa0, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7,
        0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65,
        0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x82, 0xa4, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x94, 0xc0, 0xa9, 0x43,
        0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0x80, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c,
        0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e,
        0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c,
        0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e,
        0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4,
        0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07,
        0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5,
        0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b,
        0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x13, 0x04, 0x01, 0x02,
        0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE vs = {
        use_dxil ? (const void*)vs_code_dxil : (const void*)vs_code_dxbc,
        use_dxil ? sizeof(vs_code_dxil) : sizeof(vs_code_dxbc)
    };
#if 0
    struct data
    {
        float4 position : SV_Position;
    };

    struct patch_constant_data
    {
        float edges[3] : SV_TessFactor;
        float inside : SV_InsideTessFactor;
    };

    void patch_constant(out patch_constant_data output)
    {
        output.edges[0] = output.edges[1] = output.edges[2] = 1.0f;
        output.inside = 1.0f;
    }

    [domain("tri")]
    [outputcontrolpoints(3)]
    [partitioning("integer")]
    [outputtopology("triangle_cw")]
    [patchconstantfunc("patch_constant")]
    data hs_main(uint i : SV_OutputControlPointID)
    {
        data output;

        if (i == 0)
            output.position = float4(-1, 1, 0, 1);
        else if (i == 1)
            output.position = float4(3, 1, 0, 1);
        else
            output.position = float4(-1, -3, 0, 1);

        return output;
    }

    [domain("tri")]
    void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<data, 3> patch,
            out data output)
    {
        uint index = uint(tess_coord.y + 2 * tess_coord.z);
        output.position = patch[index].position;
    }
#endif
    static const DWORD hs_code_dxbc[] =
    {
        0x43425844, 0x4204890e, 0x43f4f8e0, 0xbff7edfd, 0x0a48b715, 0x00000001, 0x0000028c, 0x00000004,
        0x00000030, 0x00000040, 0x00000074, 0x00000108, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000008c, 0x00000004,
        0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000e01, 0x00000068,
        0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x00000068, 0x00000002, 0x0000000d,
        0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000, 0x0000000e, 0x00000003, 0x00000003,
        0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469, 0x46737365,
        0x6f746361, 0xabab0072, 0x58454853, 0x0000017c, 0x00030050, 0x0000005f, 0x01000071, 0x01000893,
        0x01001894, 0x01001095, 0x01000896, 0x01001897, 0x0100086a, 0x01000072, 0x0200005f, 0x00016000,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x06000020, 0x00100012, 0x00000000,
        0x00016001, 0x00004001, 0x00000001, 0x0f000037, 0x001000f2, 0x00000000, 0x00100006, 0x00000000,
        0x00004002, 0x40400000, 0x3f800000, 0x00000000, 0x3f800000, 0x00004002, 0xbf800000, 0xc0400000,
        0x00000000, 0x3f800000, 0x0b000037, 0x001020f2, 0x00000000, 0x00016001, 0x00100e46, 0x00000000,
        0x00004002, 0xbf800000, 0x3f800000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000073, 0x02000099,
        0x00000003, 0x0200005f, 0x00017000, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x04000067,
        0x00102012, 0x00000001, 0x00000012, 0x04000067, 0x00102012, 0x00000002, 0x00000013, 0x02000068,
        0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000003, 0x04000036, 0x00100012, 0x00000000,
        0x0001700a, 0x06000036, 0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
        0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x00000014, 0x05000036, 0x00102012, 0x00000003,
        0x00004001, 0x3f800000, 0x0100003e,
    };
    static const BYTE hs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x26, 0xbc, 0x0a, 0x30, 0x27, 0x56, 0x0c, 0xe8, 0x7c, 0x19, 0x19, 0xe9, 0x53, 0x9d, 0x90, 0xab, 0x01, 0x00, 0x00, 0x00, 0x16, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x46, 0x01, 0x00, 0x00, 0xca, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x50, 0x53, 0x47, 0x31, 0xaa, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x53, 0x56, 0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46,
        0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x50, 0x53, 0x56, 0x30, 0x7c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x04, 0x00, 0x00, 0x01, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x03, 0x41, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x44, 0x06, 0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0x91, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x2c, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x88, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00,
        0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92,
        0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22,
        0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02,
        0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4, 0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84,
        0x84, 0x4c, 0x10, 0x38, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x39, 0x02, 0x30, 0x28, 0xc2, 0x0c, 0xd1, 0x1c, 0x01, 0x32, 0x03, 0x50, 0x0c, 0x98, 0x19, 0x3b, 0xc2, 0x81, 0x80, 0x1c, 0x98, 0x39, 0x82,
        0x60, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50,
        0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0,
        0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x08, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x20, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x41, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11,
        0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x9a, 0x12, 0x18, 0x01, 0x28, 0x84, 0x62, 0x28, 0x82, 0x32, 0x20, 0x2b, 0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x92, 0x29, 0x81, 0xa2, 0xa1,
        0x1b, 0x23, 0x00, 0x41, 0x10, 0x44, 0xc3, 0x80, 0x76, 0xac, 0x01, 0x20, 0x1a, 0x00, 0x79, 0x18, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c,
        0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc7,
        0x06, 0x61, 0x30, 0x28, 0xa0, 0xcd, 0x4d, 0x10, 0x06, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x04, 0x41, 0x08, 0x36, 0x14, 0xc3,
        0xe2, 0x3c, 0xce, 0x04, 0x61, 0x99, 0x36, 0x04, 0x11, 0x0f, 0xb4, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x04, 0x63,
        0x99, 0x20, 0x18, 0xcc, 0x86, 0x80, 0x98, 0x20, 0x18, 0xcd, 0x04, 0xc1, 0x70, 0x36, 0x2c, 0x04, 0x55, 0x59, 0x17, 0x36, 0x60, 0x44, 0x06, 0x6c, 0x08, 0x34, 0x36, 0x53, 0x56, 0x5f, 0x54, 0x65,
        0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x13, 0x04, 0xe3, 0x99, 0x20, 0x0c, 0xc9, 0x86, 0x81, 0x18, 0xbc, 0x09, 0xc2, 0xa0, 0x4c, 0x10, 0x0c, 0x68, 0xc3, 0x42, 0x70, 0x55, 0xf7, 0x65,
        0x60, 0x10, 0x06, 0x84, 0x05, 0x70, 0x9a, 0xb2, 0xfa, 0x4a, 0x72, 0x9b, 0x4b, 0x23, 0x2b, 0xa3, 0x2a, 0x9b, 0x9b, 0x33, 0x0a, 0x1b, 0xa3, 0x7b, 0x93, 0x9b, 0x20, 0x18, 0xd1, 0x86, 0x65, 0x18,
        0x83, 0x8a, 0x0c, 0xae, 0x6c, 0x08, 0x03, 0x30, 0xc8, 0x80, 0x0d, 0x82, 0x18, 0x94, 0xc1, 0x86, 0x01, 0xd8, 0xcc, 0x60, 0x82, 0x70, 0x48, 0x1b, 0x8e, 0x65, 0x00, 0x03, 0x6f, 0x00, 0x03, 0x34,
        0xd8, 0x20, 0x80, 0x41, 0x1a, 0x6c, 0x28, 0x9e, 0xe9, 0x0c, 0x00, 0x35, 0xa8, 0xc2, 0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9, 0xd8, 0x95,
        0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53, 0x02, 0xa3, 0x0e, 0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3,
        0x1b, 0x59, 0x19, 0xdb, 0x94, 0x20, 0xa9, 0x44, 0x86, 0xe7, 0x42, 0x97, 0x07, 0x57, 0x16, 0xe4, 0xe6, 0xf6, 0x46, 0x17, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x37, 0x25, 0x78, 0xea, 0x90, 0xe1, 0xb9,
        0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d, 0x09, 0xa2, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x53, 0x02, 0x35,
        0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78,
        0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84,
        0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87,
        0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d,
        0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37,
        0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87,
        0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c,
        0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88,
        0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x05, 0xd0, 0x06, 0x81, 0x1f, 0x3c,
        0x4c, 0x1f, 0xa3, 0xbf, 0xe3, 0xb7, 0x7b, 0x4e, 0x0f, 0xbb, 0xe9, 0x40, 0x60, 0x36, 0x88, 0xad, 0xc2, 0xc3, 0xf4, 0x31, 0xfa, 0x3b, 0x7e, 0xbb, 0xe7, 0xf4, 0xb0, 0x9b, 0xfe, 0x25, 0x87, 0xe9,
        0x61, 0x20, 0x10, 0xa8, 0x35, 0x01, 0x6f, 0xb8, 0x7c, 0xe7, 0xf1, 0x1d, 0x6a, 0x7a, 0xa8, 0x89, 0x73, 0x9a, 0x29, 0x72, 0x96, 0xd4, 0x41, 0x9a, 0x49, 0xec, 0x7c, 0xe4, 0xb6, 0x6d, 0xa0, 0x1a,
        0x2e, 0xdf, 0x79, 0x7c, 0x69, 0x72, 0x22, 0x02, 0xa5, 0xa6, 0x87, 0x9a, 0xfc, 0xe2, 0xb6, 0x8d, 0x80, 0x1b, 0x2e, 0xdf, 0x79, 0x7c, 0x69, 0x72, 0x22, 0x22, 0x05, 0x26, 0xe1, 0xe0, 0x9c, 0x46,
        0x9a, 0x80, 0x66, 0xf2, 0x8b, 0xdb, 0xb6, 0x80, 0xe3, 0x90, 0x7e, 0x06, 0x40, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x34, 0x45, 0x1d, 0x40, 0x37, 0x46, 0x00, 0x82, 0x20, 0x88, 0x7f, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x48, 0x02, 0xd4, 0x20, 0xc1, 0x88, 0x41,
        0x02, 0x80, 0x20, 0x18, 0x20, 0x92, 0x00, 0x45, 0x48, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x88, 0x24, 0x40, 0x10, 0x12, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x22, 0x09, 0x11, 0x84,
        0x04, 0x08, 0x61, 0x20, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x46, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x34, 0xa5, 0x50, 0xd6, 0x01, 0x64, 0x45, 0x40, 0x37, 0x46,
        0x00, 0x82, 0x20, 0x88, 0x7f, 0x23, 0x00, 0x63, 0x04, 0x20, 0x08, 0x82, 0xf8, 0x2f, 0x8c, 0x11, 0x80, 0x20, 0x08, 0x92, 0x60, 0x30, 0x46, 0x00, 0x82, 0x20, 0x48, 0x82, 0x03, 0x00, 0x23, 0x06,
        0x05, 0x00, 0x82, 0x60, 0x40, 0x68, 0xc7, 0x70, 0x43, 0x70, 0x81, 0xc1, 0x2c, 0x43, 0x20, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x6c, 0x0a, 0x86, 0x41, 0xc5, 0x88, 0x41, 0x02, 0x80,
        0x20, 0x18, 0x20, 0x9b, 0x82, 0x61, 0xce, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0xa6, 0x60, 0x18, 0x62, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb2, 0x29, 0x18, 0x26, 0x1d, 0xb3,
        0x04, 0xc5, 0x70, 0x83, 0x90, 0x81, 0xc1, 0x2c, 0xc3, 0x40, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x70, 0x4b, 0x96, 0x45, 0xc5, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xdc, 0x92,
        0x65, 0x0f, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0xb7, 0x64, 0x59, 0x72, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x2d, 0x59, 0x36, 0x21, 0xb3, 0x04, 0xc5, 0x88, 0x41, 0x02,
        0x80, 0x20, 0x18, 0x20, 0xdc, 0x92, 0x65, 0x91, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0xb7, 0x64, 0xd9, 0x43, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x2d, 0x59, 0x96, 0x1c,
        0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x70, 0x4b, 0x96, 0x4d, 0xc8, 0x2c, 0x41, 0x81, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE hs = {
        use_dxil ? (const void*)hs_code_dxil : (const void*)hs_code_dxbc,
        use_dxil ? sizeof(hs_code_dxil) : sizeof(hs_code_dxbc)
    };
    static const DWORD ds_code_dxbc[] =
    {
        0x43425844, 0x33c6120a, 0x2d46da82, 0x2c17dddf, 0x252ae7e0, 0x00000001, 0x000001c8, 0x00000004,
        0x00000030, 0x00000064, 0x000000f8, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d,
        0x00000003, 0x00000000, 0x00000001, 0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001,
        0x00000001, 0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002, 0x00000001, 0x00000076,
        0x00000000, 0x0000000e, 0x00000003, 0x00000003, 0x00000001, 0x545f5653, 0x46737365, 0x6f746361,
        0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x4e47534f, 0x0000002c,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x00000094, 0x00040050, 0x00000025, 0x01001893,
        0x01001095, 0x0100086a, 0x0200005f, 0x0001c062, 0x0400005f, 0x002190f2, 0x00000003, 0x00000000,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x02000068, 0x00000001, 0x07000032, 0x00100012,
        0x00000000, 0x0001c02a, 0x00004001, 0x40000000, 0x0001c01a, 0x0500001c, 0x00100012, 0x00000000,
        0x0010000a, 0x00000000, 0x07000036, 0x001020f2, 0x00000000, 0x00a19e46, 0x0010000a, 0x00000000,
        0x00000000, 0x0100003e,
    };
    static const BYTE ds_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x72, 0xd0, 0x50, 0xec, 0x20, 0xc5, 0xd7, 0xfc, 0xfd, 0x9d, 0x4b, 0x3a, 0x48, 0x5f, 0x29, 0x25, 0x01, 0x00, 0x00, 0x00, 0xb6, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0x56, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x50, 0x53, 0x47, 0x31, 0xaa, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x53, 0x56,
        0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x50, 0x53, 0x56, 0x30, 0xdc, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x04, 0x00, 0x01, 0x01, 0x02, 0x01, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03,
        0x41, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x58, 0x05, 0x00, 0x00, 0x60, 0x00,
        0x04, 0x00, 0x56, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x40, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x4d, 0x01,
        0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05,
        0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46,
        0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00,
        0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4, 0x84, 0x04, 0x13,
        0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x38, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30, 0x98, 0x23, 0x40, 0xca, 0x30, 0x54,
        0x64, 0xc5, 0x18, 0x22, 0x2a, 0x22, 0x2c, 0x06, 0x10, 0x51, 0x0d, 0xe9, 0x40, 0x40, 0x52, 0x10, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03,
        0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11,
        0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x18, 0x01, 0x28, 0x84, 0x62, 0x28, 0x82, 0x32, 0xa0, 0x2a, 0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x92, 0x29, 0x81, 0xa2, 0xa1,
        0x1e, 0x8b, 0x22, 0x88, 0x40, 0x20, 0x90, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02,
        0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x08, 0x62, 0x82, 0x40, 0x14, 0x1b, 0x84, 0x81,
        0x98, 0x20, 0x10, 0xc6, 0x06, 0x61, 0x30, 0x28, 0x90, 0xcd, 0x4d, 0x10, 0x88, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61,
        0x58, 0x9c, 0x09, 0x42, 0x13, 0x6d, 0x08, 0x20, 0x1e, 0x64, 0x73, 0x5f, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x85, 0x32,
        0x41, 0x28, 0x96, 0x0d, 0x01, 0x31, 0x41, 0x28, 0x98, 0x09, 0x42, 0xd1, 0x6c, 0x58, 0x88, 0x89, 0xaa, 0xac, 0x6b, 0xb8, 0x08, 0x0c, 0xd8, 0x10, 0x64, 0x6c, 0xa6, 0xac, 0xbe, 0xa8, 0xca, 0xe6,
        0xe6, 0x8c, 0xc2, 0xc6, 0xe8, 0xde, 0xe4, 0x26, 0x08, 0x85, 0x33, 0x41, 0x20, 0x90, 0x0d, 0x03, 0x31, 0x74, 0x13, 0x04, 0x22, 0x99, 0x20, 0x14, 0xcf, 0x86, 0x85, 0xd8, 0x28, 0xce, 0xc3, 0x3e,
        0x30, 0x20, 0x2a, 0x80, 0xd3, 0x94, 0xd5, 0x57, 0x92, 0xdb, 0x5c, 0x1a, 0x59, 0x19, 0x55, 0xd9, 0xdc, 0x9c, 0x51, 0xd8, 0x18, 0xdd, 0x9b, 0xdc, 0x04, 0xa1, 0x80, 0x36, 0x2c, 0x83, 0x18, 0x50,
        0x63, 0x60, 0x61, 0x03, 0x18, 0x7c, 0x18, 0xb0, 0x41, 0x08, 0x03, 0x32, 0xd8, 0x30, 0x68, 0x5a, 0x19, 0x6c, 0x10, 0xba, 0x6f, 0x83, 0xd0, 0x9d, 0xc1, 0x86, 0x62, 0x91, 0xcc, 0x00, 0x40, 0x83,
        0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62,
        0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x4a, 0x64, 0x78, 0x2e, 0x74, 0x79,
        0x70, 0x65, 0x41, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x02, 0xa7, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94,
        0x00, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0x40, 0x03, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08,
        0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4,
        0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc,
        0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30,
        0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43,
        0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70,
        0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0,
        0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06,
        0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4,
        0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x06, 0x70, 0x0c, 0xd2, 0xcf, 0x00, 0x48, 0x63, 0x01, 0xd8, 0x70, 0xf9, 0xce, 0xe3, 0x0f, 0x0e, 0x03, 0x20,
        0x4d, 0xe9, 0x08, 0xc0, 0x84, 0x38, 0x8d, 0x5f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0xdb, 0x40, 0x35, 0x5c, 0xbe, 0xf3,
        0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a, 0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x03, 0x00, 0x61, 0x20, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00,
        0x00, 0x00, 0x54, 0x45, 0x40, 0x54, 0x0a, 0x25, 0x1d, 0x40, 0x33, 0x46, 0x00, 0x82, 0x20, 0x08, 0x82, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x60, 0x54, 0xc2, 0x31,
        0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x86, 0x35, 0x14, 0x16, 0x0c, 0xf2, 0xb1, 0x60, 0x80, 0xcf, 0x05, 0x34, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x64, 0x95, 0x65, 0x39, 0xc1, 0x88, 0x41,
        0x02, 0x80, 0x20, 0x18, 0x20, 0x9a, 0x75, 0x5d, 0x8d, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x76, 0x61, 0xd8, 0x32, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x61, 0x59, 0x36,
        0x11, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x70, 0x8c, 0xa6, 0x49, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x1c, 0xa3, 0x69, 0xd0, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a,
        0xc7, 0x68, 0x5a, 0x23, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xc2, 0x31, 0x9a, 0x46, 0x05, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE ds = {
        use_dxil ? (const void*)ds_code_dxil : (const void*)ds_code_dxbc,
        use_dxil ? sizeof(ds_code_dxil) : sizeof(ds_code_dxbc)
    };

    memset(&desc, 0, sizeof(desc));
    desc.no_pipeline = true;
    if (!init_test_context(&context, &desc))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    if (use_dxil)
    {
        init_pipeline_state_desc_dxil(&pso_desc, context.root_signature,
                context.render_target_desc.Format, NULL, NULL, NULL);
    }
    else
    {
        init_pipeline_state_desc(&pso_desc, context.root_signature,
            context.render_target_desc.Format, NULL, NULL, NULL);
    }

    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_3_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 3, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);

    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff00ff00, 0);

    destroy_test_context(&context);
}

void test_hull_shader_control_point_phase_dxbc(void)
{
    test_hull_shader_control_point_phase(false);
}

void test_hull_shader_control_point_phase_dxil(void)
{
    test_hull_shader_control_point_phase(true);
}

void test_hull_shader_vertex_input_patch_constant_phase(void)
{
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    struct test_context_desc desc;
    struct test_context context;
    ID3D12CommandQueue *queue;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        struct data
        {
            float4 position : SV_Position;
            float4 a : TEXCOORD0;
            float4 b : TEXCOORD1;
            float4 c : TEXCOORD2;
        };
        data main(uint vid : SV_VertexID)
        {
            data d;
            d.a = (100.0 / 255.0).xxxx;
            d.b = (200.0 / 255.0).xxxx;
            d.c = (155.0 / 255.0).xxxx;

            if (vid == 0)
                d.position = float4(-1, -1, 0, 1);
            else if (vid == 1)
                d.position = float4(-1, 3, 0, 1);
            else
                d.position = float4(3, -1, 0, 1);

            return d;
        }
#endif
        0x43425844, 0x64268a24, 0xcd8d91b3, 0x70514911, 0x95555eb8, 0x00000001, 0x00000234, 0x00000003,
        0x0000002c, 0x00000060, 0x000000e8, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000006, 0x00000001, 0x00000000, 0x00000101, 0x565f5653, 0x65747265, 0x00444978,
        0x4e47534f, 0x00000080, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000074, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x00000074, 0x00000001, 0x00000000, 0x00000003, 0x00000002, 0x0000000f, 0x00000074, 0x00000002,
        0x00000000, 0x00000003, 0x00000003, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554,
        0x44524f4f, 0xababab00, 0x58454853, 0x00000144, 0x00010050, 0x00000051, 0x0100086a, 0x04000060,
        0x00101012, 0x00000000, 0x00000006, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x03000065,
        0x001020f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000002, 0x03000065, 0x001020f2, 0x00000003,
        0x02000068, 0x00000001, 0x07000020, 0x00100012, 0x00000000, 0x0010100a, 0x00000000, 0x00004001,
        0x00000001, 0x0f000037, 0x001000f2, 0x00000000, 0x00100006, 0x00000000, 0x00004002, 0xbf800000,
        0x40400000, 0x00000000, 0x3f800000, 0x00004002, 0x40400000, 0xbf800000, 0x00000000, 0x3f800000,
        0x0c000037, 0x001020f2, 0x00000000, 0x00101006, 0x00000000, 0x00100e46, 0x00000000, 0x00004002,
        0xbf800000, 0xbf800000, 0x00000000, 0x3f800000, 0x08000036, 0x001020f2, 0x00000001, 0x00004002,
        0x3ec8c8c9, 0x3ec8c8c9, 0x3ec8c8c9, 0x3ec8c8c9, 0x08000036, 0x001020f2, 0x00000002, 0x00004002,
        0x3f48c8c9, 0x3f48c8c9, 0x3f48c8c9, 0x3f48c8c9, 0x08000036, 0x001020f2, 0x00000003, 0x00004002,
        0x3f1b9b9c, 0x3f1b9b9c, 0x3f1b9b9c, 0x3f1b9b9c, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE vs = {vs_code, sizeof(vs_code)};
#if 0
    struct data
    {
        float4 position : SV_Position;
        float4 a : TEXCOORD0;
        float4 b : TEXCOORD1;
        float4 c : TEXCOORD2;
    };

    struct patch_constant_data
    {
        float edges[3] : SV_TessFactor;
        float inside[1] : SV_InsideTessFactor;
    };

    void patch_constant(OutputPatch<data, 3> control_points,
            InputPatch<data, 3> vertices,
            out patch_constant_data output)
    {
        float4 should_be_one = vertices[0].a + vertices[2].c;
        output.edges[0] = should_be_one.x;
        output.edges[1] = should_be_one.y;
        output.edges[2] = should_be_one.z;
        output.inside[0] = should_be_one.w;
    }

    [domain("tri")]
    [outputcontrolpoints(3)]
    [partitioning("integer")]
    [outputtopology("triangle_cw")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 3> input, uint i : SV_OutputControlPointID)
    {
        data o = (data)0;
        o.position = input[i].position;
        o.a = input[i].a;
        o.b = input[i].b;
        o.c = input[i].c;
        return o;
    }

    [domain("tri")]
    void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<data, 3> patch,
            out float4 position : SV_Position,
            out float4 a : TEXCOORD0,
            out float4 b : TEXCOORD1,
            out float4 c : TEXCOORD2)
    {
        position = patch[0].position * tess_coord.x +
            patch[1].position * tess_coord.y +
            patch[2].position * tess_coord.z;

        a = patch[0].a;
        b = patch[1].b;
        c = patch[2].c;
    }
#endif
    static const DWORD hs_code[] =
    {
        0x43425844, 0xa41c1952, 0x06171644, 0x3e67c076, 0x54c6869d, 0x00000001, 0x00000370, 0x00000004,
        0x00000030, 0x000000b8, 0x00000140, 0x000001d4, 0x4e475349, 0x00000080, 0x00000004, 0x00000008,
        0x00000068, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x00000074, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000f0f, 0x00000074, 0x00000001, 0x00000000, 0x00000003,
        0x00000002, 0x00000f0f, 0x00000074, 0x00000002, 0x00000000, 0x00000003, 0x00000003, 0x00000f0f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f, 0xababab00, 0x4e47534f, 0x00000080,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f,
        0x00000074, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f, 0x00000074, 0x00000001,
        0x00000000, 0x00000003, 0x00000002, 0x0000000f, 0x00000074, 0x00000002, 0x00000000, 0x00000003,
        0x00000003, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f, 0xababab00,
        0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003,
        0x00000000, 0x00000e01, 0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01,
        0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000,
        0x0000000e, 0x00000003, 0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072,
        0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x00000194, 0x00030050,
        0x00000065, 0x01000071, 0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01001897, 0x0100086a,
        0x01000073, 0x0400005f, 0x00219012, 0x00000003, 0x00000001, 0x0400005f, 0x00219012, 0x00000003,
        0x00000003, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x09000000, 0x00102012, 0x00000000,
        0x0021900a, 0x00000000, 0x00000001, 0x0021900a, 0x00000002, 0x00000003, 0x0100003e, 0x01000073,
        0x0400005f, 0x00219022, 0x00000003, 0x00000001, 0x0400005f, 0x00219022, 0x00000003, 0x00000003,
        0x04000067, 0x00102012, 0x00000001, 0x00000012, 0x09000000, 0x00102012, 0x00000001, 0x0021901a,
        0x00000000, 0x00000001, 0x0021901a, 0x00000002, 0x00000003, 0x0100003e, 0x01000073, 0x0400005f,
        0x00219042, 0x00000003, 0x00000001, 0x0400005f, 0x00219042, 0x00000003, 0x00000003, 0x04000067,
        0x00102012, 0x00000002, 0x00000013, 0x09000000, 0x00102012, 0x00000002, 0x0021902a, 0x00000000,
        0x00000001, 0x0021902a, 0x00000002, 0x00000003, 0x0100003e, 0x01000073, 0x0400005f, 0x00219082,
        0x00000003, 0x00000001, 0x0400005f, 0x00219082, 0x00000003, 0x00000003, 0x04000067, 0x00102012,
        0x00000003, 0x00000014, 0x09000000, 0x00102012, 0x00000003, 0x0021903a, 0x00000000, 0x00000001,
        0x0021903a, 0x00000002, 0x00000003, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs = {hs_code, sizeof(hs_code)};
    static const DWORD ds_code[] =
    {
        0x43425844, 0x6123c00c, 0x864b0c7d, 0xd72187ab, 0x455c2254, 0x00000001, 0x00000324, 0x00000004,
        0x00000030, 0x000000b8, 0x0000014c, 0x000001d4, 0x4e475349, 0x00000080, 0x00000004, 0x00000008,
        0x00000068, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x00000074, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000f0f, 0x00000074, 0x00000001, 0x00000000, 0x00000003,
        0x00000002, 0x00000f0f, 0x00000074, 0x00000002, 0x00000000, 0x00000003, 0x00000003, 0x00000f0f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f, 0xababab00, 0x47534350, 0x0000008c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000001,
        0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000001, 0x00000068, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000001, 0x00000076, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000001, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469,
        0x46737365, 0x6f746361, 0xabab0072, 0x4e47534f, 0x00000080, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000074, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x0000000f, 0x00000074, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x0000000f, 0x00000074, 0x00000002, 0x00000000, 0x00000003, 0x00000003, 0x0000000f, 0x505f5653,
        0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f, 0xababab00, 0x58454853, 0x00000148, 0x00040050,
        0x00000052, 0x01001893, 0x01001095, 0x0100086a, 0x0200005f, 0x0001c072, 0x0400005f, 0x002190f2,
        0x00000003, 0x00000000, 0x0400005f, 0x002190f2, 0x00000003, 0x00000001, 0x0400005f, 0x002190f2,
        0x00000003, 0x00000002, 0x0400005f, 0x002190f2, 0x00000003, 0x00000003, 0x04000067, 0x001020f2,
        0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000002,
        0x03000065, 0x001020f2, 0x00000003, 0x02000068, 0x00000001, 0x07000038, 0x001000f2, 0x00000000,
        0x0001c556, 0x00219e46, 0x00000001, 0x00000000, 0x09000032, 0x001000f2, 0x00000000, 0x00219e46,
        0x00000000, 0x00000000, 0x0001c006, 0x00100e46, 0x00000000, 0x09000032, 0x001020f2, 0x00000000,
        0x00219e46, 0x00000002, 0x00000000, 0x0001caa6, 0x00100e46, 0x00000000, 0x06000036, 0x001020f2,
        0x00000001, 0x00219e46, 0x00000000, 0x00000001, 0x06000036, 0x001020f2, 0x00000002, 0x00219e46,
        0x00000001, 0x00000002, 0x06000036, 0x001020f2, 0x00000003, 0x00219e46, 0x00000002, 0x00000003,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ds = {ds_code, sizeof(ds_code)};
    static const DWORD ps_code[] =
    {
#if 0
        float4 main(in float4 p : SV_Position, in float4 a : TEXCOORD0, in float4 b : TEXCOORD1, in float4 c : TEXCOORD2) : SV_Target
        {
            return float4(a.x, b.y, c.z, 1.0);
        }
#endif
        0x43425844, 0x005c9297, 0xc1901d09, 0x4eebb7e6, 0x6117a0e8, 0x00000001, 0x00000180, 0x00000003,
        0x0000002c, 0x000000b4, 0x000000e8, 0x4e475349, 0x00000080, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000074, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x0000010f, 0x00000074, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x0000020f, 0x00000074, 0x00000002, 0x00000000, 0x00000003, 0x00000003, 0x0000040f, 0x505f5653,
        0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f, 0xababab00, 0x4e47534f, 0x0000002c, 0x00000001,
        0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653,
        0x65677261, 0xabab0074, 0x58454853, 0x00000090, 0x00000050, 0x00000024, 0x0100086a, 0x03001062,
        0x00101012, 0x00000001, 0x03001062, 0x00101022, 0x00000002, 0x03001062, 0x00101042, 0x00000003,
        0x03000065, 0x001020f2, 0x00000000, 0x05000036, 0x00102012, 0x00000000, 0x0010100a, 0x00000001,
        0x05000036, 0x00102022, 0x00000000, 0x0010101a, 0x00000002, 0x05000036, 0x00102042, 0x00000000,
        0x0010102a, 0x00000003, 0x05000036, 0x00102082, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ps = {ps_code, sizeof(ps_code)};

    memset(&desc, 0, sizeof(desc));
    desc.no_pipeline = true;
    if (!init_test_context(&context, &desc))
        return;
    command_list = context.list;
    queue = context.queue;

    init_pipeline_state_desc(&pso_desc, context.root_signature,
            context.render_target_desc.Format, NULL, NULL, NULL);
    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    pso_desc.PS = ps;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_3_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 3, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff9bc864, 0);

    destroy_test_context(&context);
}

static void test_hull_shader_fork_phase(bool use_dxil)
{
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    struct test_context_desc desc;
    struct test_context context;
    ID3D12CommandQueue *queue;
    HRESULT hr;
#if 0
    float4 main() : SV_Position
    {
        return float4(-1, -1, 0, 0);
    }
#endif
    static const DWORD vs_code_dxbc[] =
    {
        0x43425844, 0x9f5f70e7, 0x57507df4, 0xbf1a4a34, 0xdb2531df, 0x00000001, 0x000000b8, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000,
        0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x00000040, 0x00010050, 0x00000010,
        0x0100086a, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0xbf800000, 0xbf800000, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const BYTE vs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xa5, 0x25, 0x1d, 0x59, 0xb5, 0x9d, 0x3e, 0x85, 0x9c, 0x32, 0x1f, 0x9e, 0x23, 0x9f, 0xd9, 0xda, 0x01, 0x00, 0x00, 0x00, 0x2c, 0x05, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x50, 0x53, 0x56, 0x30, 0x50, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x3c, 0x04, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x0f, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x24, 0x04, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x1b, 0x88, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0x00, 0x00,
        0x49, 0x18, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x13, 0x82, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4, 0x84,
        0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x28, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x39, 0x02, 0x30, 0x98, 0x23, 0x40, 0x66, 0x00, 0x8a,
        0x01, 0x33, 0x43, 0x45, 0x36, 0x10, 0x90, 0x02, 0x03, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50,
        0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0,
        0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x81, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90,
        0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x9a, 0x12, 0x18, 0x01, 0x28, 0x84, 0x62, 0x20, 0x2a, 0x89, 0x32, 0x28, 0x84, 0x11, 0x00, 0xca, 0xb1, 0x04, 0x80, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10,
        0x4c, 0x10, 0x06, 0x61, 0x82, 0x30, 0x0c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc4, 0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10, 0x86, 0x62, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36,
        0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58, 0x9c, 0x09, 0x42, 0xa2, 0x6c, 0x08, 0x20, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8,
        0xd2, 0xde, 0xdc, 0x26, 0x08, 0x84, 0x31, 0x41, 0x20, 0x8e, 0x0d, 0x01, 0x31, 0x41, 0x20, 0x90, 0x09, 0x02, 0x91, 0x6c, 0x58, 0x88, 0x89, 0xaa, 0xac, 0x6b, 0xb8, 0x08, 0x0c, 0xd8, 0x10, 0x64,
        0x1b, 0x06, 0x40, 0x03, 0x36, 0x14, 0x8b, 0xb4, 0x01, 0x40, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e,
        0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x18, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca,
        0xd8, 0xa6, 0x04, 0x49, 0x25, 0x32, 0x3c, 0x17, 0xba, 0x3c, 0xb8, 0xb2, 0x20, 0x37, 0xb7, 0x37, 0xba, 0x30, 0xba, 0xb4, 0x37, 0xb7, 0xb9, 0x29, 0x81, 0x53, 0x87, 0x0c, 0xcf, 0xc5, 0x2e, 0xad,
        0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c, 0x4a, 0x00, 0xd5, 0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83, 0x7a, 0x4b, 0x73, 0xa3, 0x9b, 0x9b, 0x12, 0x6c, 0x00, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73,
        0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b,
        0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20,
        0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61,
        0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87,
        0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98,
        0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61,
        0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b,
        0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x16, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34,
        0x39, 0x11, 0x81, 0x52, 0xd3, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x61, 0x20, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x34, 0xa5, 0x40, 0x54, 0x02, 0x45, 0x40, 0x35, 0x46, 0x00, 0x82, 0x20, 0x88, 0xff, 0xc2, 0x08, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x3c,
        0x45, 0xd3, 0x1c, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x4f, 0xd1, 0x34, 0x84, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x53, 0x34, 0xcd, 0x10, 0x8c, 0x18, 0x24, 0x00, 0x08,
        0x82, 0x81, 0xf1, 0x14, 0x4d, 0x93, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE vs = {
        use_dxil ? (const void*)vs_code_dxil : (const void*)vs_code_dxbc,
        use_dxil ? sizeof(vs_code_dxil) : sizeof(vs_code_dxbc)
    };
#if 0
    struct data
    {
        float4 position : SV_Position;
    };

    struct patch_constant_data
    {
        float edges[4] : SV_TessFactor;
        float inside[2] : SV_InsideTessFactor;
        float3 center : CENTER;
    };

    void patch_constant(OutputPatch<data, 4> control_points,
            out patch_constant_data output)
    {
        uint i;

        output = (patch_constant_data)0;

        for (i = 0; i < 4; ++i)
        {
            output.edges[i] = 1.0;
        }

        for (i = 0; i < 2; ++i)
        {
            output.inside[i] = 1.0;
        }

        for (i = 0; i < 4; ++i)
        {
            output.center += control_points[i].position.xyz;
        }
        output.center /= 4;
    }

    [domain("quad")]
    [outputcontrolpoints(4)]
    [partitioning("integer")]
    [outputtopology("triangle_cw")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 1> input,
            uint i : SV_OutputControlPointID)
    {
        data o = (data)0;
        const float4 vertices[] =
        {
            float4(0, 0, 0, 1),
            float4(0, 2, 0, 1),
            float4(2, 0, 0, 1),
            float4(2, 2, 0, 1),
        };

        o.position = input[0].position + vertices[i];

        return o;
    }

    [domain("quad")]
    void ds_main(patch_constant_data input,
            float2 tess_coord : SV_DomainLocation,
            const OutputPatch<data, 4> patch,
            out float4 position : SV_Position,
            out float4 color : COLOR)
    {
        float4 a = lerp(patch[0].position, patch[1].position, tess_coord.x);
        float4 b = lerp(patch[2].position, patch[3].position, tess_coord.x);
        position = lerp(a, b, tess_coord.y);

        color = float4(input.center, 1.0);
    }
#endif
    static const DWORD hs_code_dxbc[] =
    {
        0x43425844, 0xcecdcc7a, 0x1e0454ae, 0x8c490947, 0x7e33bb11, 0x00000001, 0x000005a4, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000017c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x000000dc,
        0x00000007, 0x00000008, 0x000000b0, 0x00000000, 0x0000000b, 0x00000003, 0x00000000, 0x00000e01,
        0x000000be, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000010e, 0x000000b0, 0x00000001,
        0x0000000b, 0x00000003, 0x00000001, 0x00000e01, 0x000000b0, 0x00000002, 0x0000000b, 0x00000003,
        0x00000002, 0x00000e01, 0x000000b0, 0x00000003, 0x0000000b, 0x00000003, 0x00000003, 0x00000e01,
        0x000000c5, 0x00000000, 0x0000000c, 0x00000003, 0x00000004, 0x00000e01, 0x000000c5, 0x00000001,
        0x0000000c, 0x00000003, 0x00000005, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x45430072,
        0x5245544e, 0x5f565300, 0x69736e49, 0x65546564, 0x61467373, 0x726f7463, 0xababab00, 0x58454853,
        0x00000420, 0x00030050, 0x00000108, 0x01000071, 0x01000893, 0x01002094, 0x01001895, 0x01000896,
        0x01001897, 0x0100086a, 0x00001835, 0x00000012, 0x3f800000, 0x00000000, 0x00000000, 0x00000000,
        0x3f800000, 0x00000000, 0x40000000, 0x00000000, 0x3f800000, 0x40000000, 0x00000000, 0x00000000,
        0x3f800000, 0x40000000, 0x40000000, 0x00000000, 0x01000072, 0x0200005f, 0x00016000, 0x0400005f,
        0x002010f2, 0x00000001, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001,
        0x04000036, 0x00100012, 0x00000000, 0x00016001, 0x06000036, 0x00100032, 0x00000000, 0x00909596,
        0x0010000a, 0x00000000, 0x08000036, 0x001000c2, 0x00000000, 0x00004002, 0x00000000, 0x00000000,
        0x00000000, 0x3f800000, 0x08000000, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00201e46,
        0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x02000099, 0x00000004, 0x0200005f, 0x00017000,
        0x04000067, 0x00102012, 0x00000000, 0x0000000b, 0x04000067, 0x00102012, 0x00000001, 0x0000000c,
        0x04000067, 0x00102012, 0x00000002, 0x0000000d, 0x04000067, 0x00102012, 0x00000003, 0x0000000e,
        0x02000068, 0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000004, 0x04000036, 0x00100012,
        0x00000000, 0x0001700a, 0x06000036, 0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000,
        0x0100003e, 0x01000073, 0x02000099, 0x00000002, 0x0200005f, 0x00017000, 0x04000067, 0x00102012,
        0x00000004, 0x0000000f, 0x04000067, 0x00102012, 0x00000005, 0x00000010, 0x02000068, 0x00000001,
        0x0400005b, 0x00102012, 0x00000004, 0x00000002, 0x04000036, 0x00100012, 0x00000000, 0x0001700a,
        0x07000036, 0x00d02012, 0x00000004, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
        0x01000073, 0x0400005f, 0x0021a012, 0x00000004, 0x00000000, 0x03000065, 0x00102022, 0x00000000,
        0x02000068, 0x00000001, 0x09000000, 0x00100012, 0x00000000, 0x0021a00a, 0x00000001, 0x00000000,
        0x0021a00a, 0x00000000, 0x00000000, 0x08000000, 0x00100012, 0x00000000, 0x0010000a, 0x00000000,
        0x0021a00a, 0x00000002, 0x00000000, 0x08000000, 0x00100012, 0x00000000, 0x0010000a, 0x00000000,
        0x0021a00a, 0x00000003, 0x00000000, 0x07000038, 0x00102022, 0x00000000, 0x0010000a, 0x00000000,
        0x00004001, 0x3e800000, 0x0100003e, 0x01000073, 0x0400005f, 0x0021a022, 0x00000004, 0x00000000,
        0x03000065, 0x00102042, 0x00000000, 0x02000068, 0x00000001, 0x09000000, 0x00100012, 0x00000000,
        0x0021a01a, 0x00000001, 0x00000000, 0x0021a01a, 0x00000000, 0x00000000, 0x08000000, 0x00100012,
        0x00000000, 0x0010000a, 0x00000000, 0x0021a01a, 0x00000002, 0x00000000, 0x08000000, 0x00100012,
        0x00000000, 0x0010000a, 0x00000000, 0x0021a01a, 0x00000003, 0x00000000, 0x07000038, 0x00102042,
        0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x3e800000, 0x0100003e, 0x01000073, 0x0400005f,
        0x0021a042, 0x00000004, 0x00000000, 0x03000065, 0x00102082, 0x00000000, 0x02000068, 0x00000001,
        0x09000000, 0x00100012, 0x00000000, 0x0021a02a, 0x00000001, 0x00000000, 0x0021a02a, 0x00000000,
        0x00000000, 0x08000000, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x0021a02a, 0x00000002,
        0x00000000, 0x08000000, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x0021a02a, 0x00000003,
        0x00000000, 0x07000038, 0x00102082, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x3e800000,
        0x0100003e,
    };
    static const BYTE hs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xa3, 0xba, 0x27, 0x38, 0x0c, 0xfb, 0xbe, 0x0c, 0x6a, 0xac, 0x40, 0x4c, 0x11, 0x2f, 0x73, 0xcd, 0x01, 0x00, 0x00, 0x00, 0x75, 0x0c, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0xd9, 0x01, 0x00, 0x00, 0xa5, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x50, 0x53, 0x47, 0x31, 0x11, 0x01, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x53, 0x56,
        0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x43, 0x45, 0x4e, 0x54, 0x45, 0x52, 0x00, 0x50, 0x53, 0x56, 0x30, 0xc4, 0x00, 0x00,
        0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x06,
        0x00, 0x01, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x43, 0x45, 0x4e, 0x54, 0x45, 0x52, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44,
        0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x71,
        0x19, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x04, 0x71, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x43,
        0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xc8, 0x09, 0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0x72, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
        0x00, 0xb0, 0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x69, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23,
        0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
        0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
        0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff,
        0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x12, 0x00, 0x00,
        0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x44, 0x33,
        0x00, 0x49, 0x00, 0x28, 0x8c, 0x00, 0x94, 0x60, 0x10, 0x99, 0x23, 0x00, 0x83, 0x22, 0x18, 0x43, 0x67, 0x8e, 0x00, 0x29, 0x06, 0x30, 0xc6, 0x24, 0x43, 0xaa, 0x18, 0xc3, 0x18, 0x93, 0x00, 0xb1,
        0x81, 0x80, 0x5c, 0x30, 0x04, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
        0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07,
        0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
        0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x24, 0x90, 0x21, 0x23, 0x25, 0x40, 0x80, 0x1d, 0x4c, 0x30, 0x70,
        0x03, 0x00, 0x00, 0x40, 0x00, 0x00, 0xc0, 0x0e, 0x26, 0x18, 0xb6, 0x01, 0x00, 0x00, 0x20, 0x00, 0x00, 0x60, 0x07, 0x13, 0x0c, 0xda, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0xb0, 0x83, 0x09, 0x86,
        0x6c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x18, 0xf2, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x60, 0xc8, 0x73, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x47, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x03, 0x04,
        0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x06, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x0a, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x64, 0x81, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x4a, 0x45, 0x53, 0x02,
        0x25, 0x33, 0x02, 0x50, 0x08, 0x64, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a, 0xa1, 0x08, 0xca, 0x80, 0x52, 0x49, 0x94, 0x01, 0x85, 0xb1, 0x08, 0x10, 0x04, 0x41, 0xfc, 0x03, 0x41, 0x10, 0xc4, 0x3f,
        0x10, 0x04, 0x41, 0xfc, 0x03, 0x41, 0x10, 0xc4, 0xbf, 0x11, 0x80, 0xb1, 0x08, 0x00, 0x04, 0x41, 0x10, 0x04, 0x03, 0x00, 0x04, 0x41, 0x10, 0x04, 0x83, 0xb1, 0x08, 0x00, 0x00, 0x41, 0x10, 0x04,
        0xc1, 0x00, 0x04, 0x41, 0x10, 0x04, 0x03, 0x02, 0x63, 0x04, 0x20, 0x08, 0x82, 0x68, 0x18, 0xd0, 0x1b, 0x6b, 0x21, 0x88, 0x40, 0x20, 0x10, 0x2c, 0x10, 0x08, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00,
        0x00, 0x61, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43,
        0x10, 0x4c, 0x10, 0x0c, 0x68, 0x82, 0x60, 0x44, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x18, 0xd2, 0x06, 0x61, 0x30, 0x28, 0xa0, 0xcd, 0x4d, 0x10, 0x8c, 0x69, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xa1, 0x20,
        0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x04, 0xa1, 0x28, 0x36, 0x14, 0xc3, 0xe2, 0x3c, 0xce, 0x04, 0xe1, 0xe9, 0x36, 0x04, 0x11, 0x0f, 0xb4, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0x37,
        0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x84, 0xc4, 0x9a, 0x20, 0x24, 0xd7, 0x86, 0x80, 0x98, 0x20, 0x24, 0xcf, 0x04, 0x21, 0x71, 0x36, 0x2c, 0x04, 0x55,
        0x59, 0x17, 0x36, 0x60, 0x44, 0x06, 0x6c, 0x08, 0x34, 0x36, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x13, 0x84, 0xa4, 0x99, 0x20, 0x18, 0xd4, 0x04, 0xc1,
        0xa8, 0x36, 0x10, 0xc4, 0xe0, 0x7d, 0x13, 0x84, 0x84, 0xd9, 0xb0, 0x10, 0x5c, 0xd5, 0x81, 0x41, 0x66, 0x84, 0x01, 0x61, 0x01, 0x9c, 0xa6, 0xac, 0xbe, 0x92, 0xdc, 0xe6, 0xd2, 0xc8, 0xca, 0xa8,
        0xca, 0xe6, 0xe6, 0x8c, 0xc2, 0xc6, 0xe8, 0xde, 0xe4, 0x26, 0x08, 0xc9, 0xb2, 0x41, 0x20, 0x86, 0x0d, 0xcb, 0x30, 0x06, 0x15, 0x19, 0x94, 0x41, 0xe6, 0x85, 0x81, 0x61, 0x01, 0x34, 0x86, 0x8a,
        0x9c, 0xa8, 0x8a, 0xa4, 0x36, 0x2c, 0xde, 0x19, 0x54, 0xd9, 0x95, 0x0d, 0x16, 0x91, 0x01, 0x1b, 0x06, 0x31, 0x30, 0x03, 0x34, 0xd8, 0x30, 0x6c, 0x5b, 0x1a, 0x4c, 0x10, 0x00, 0x6e, 0xc3, 0xb1,
        0x0c, 0xc6, 0x37, 0x7c, 0x6b, 0xb0, 0x41, 0xf8, 0xd8, 0x60, 0x43, 0xf1, 0x4c, 0x6a, 0x00, 0xb4, 0x41, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85,
        0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x18, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2,
        0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x49, 0x25, 0x32, 0x3c, 0x17, 0xba, 0x3c, 0xb8, 0xb2, 0x20, 0x37, 0xb7, 0x37, 0xba, 0x30, 0xba, 0xb4, 0x37, 0xb7, 0xb9, 0x29, 0xc1,
        0x53, 0x87, 0x0c, 0xcf, 0xc5, 0x2e, 0xad, 0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c, 0x4a, 0x10, 0xd5, 0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83, 0x7a, 0x4b, 0x73, 0xa3,
        0x9b, 0x9b, 0x12, 0xb4, 0x01, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
        0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
        0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70,
        0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
        0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
        0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80,
        0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
        0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
        0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x7d, 0x00, 0x00,
        0x00, 0x45, 0x80, 0x0a, 0x81, 0x1f, 0x3c, 0x4c, 0x1f, 0xa3, 0xbf, 0xe3, 0xb7, 0x7b, 0x4e, 0x0f, 0xbb, 0xe9, 0x40, 0x60, 0x36, 0x88, 0xb5, 0xfe, 0x48, 0xcf, 0x3a, 0x1d, 0x5e, 0xa7, 0x43, 0xc3,
        0xf4, 0x31, 0x1a, 0x58, 0x25, 0x87, 0xe9, 0x61, 0x20, 0x90, 0x04, 0x9b, 0x01, 0x81, 0x55, 0x78, 0x98, 0x3e, 0x46, 0x7f, 0xc7, 0x6f, 0xf7, 0x9c, 0x1e, 0x76, 0xd3, 0xbf, 0xe4, 0x30, 0x3d, 0x0c,
        0x04, 0x02, 0xb5, 0x05, 0x70, 0x0b, 0x81, 0x9f, 0xbd, 0x2c, 0xa7, 0xa7, 0xc7, 0xe5, 0x39, 0xf0, 0x17, 0xfb, 0xfd, 0xd0, 0xf3, 0x6f, 0x3b, 0x9c, 0x76, 0x03, 0x81, 0xd9, 0xe0, 0x2f, 0x58, 0x25,
        0x87, 0xe9, 0x61, 0x20, 0xd0, 0xfa, 0x23, 0x25, 0xdd, 0xf0, 0x3a, 0x1d, 0x1a, 0xa6, 0x8f, 0xd1, 0xc0, 0x2a, 0x39, 0x4c, 0x0f, 0x03, 0x81, 0x24, 0x18, 0x0c, 0x08, 0x4c, 0x02, 0xb5, 0xc0, 0x59,
        0x54, 0x68, 0xfd, 0x91, 0xec, 0xe5, 0x31, 0xfd, 0x2d, 0x07, 0x36, 0x49, 0xb0, 0x19, 0x10, 0x28, 0x74, 0xc1, 0x2a, 0xe0, 0x16, 0x02, 0x3f, 0x7b, 0x59, 0x4e, 0x4f, 0x8f, 0xcb, 0x73, 0xe0, 0x2f,
        0xf6, 0xfb, 0xa1, 0xe7, 0xdf, 0x76, 0x38, 0xed, 0x06, 0x02, 0xb3, 0xc1, 0x5f, 0xb0, 0x4a, 0x0e, 0xd3, 0xc3, 0x40, 0xa0, 0xf5, 0x47, 0x4a, 0xba, 0xe1, 0x75, 0x3a, 0x34, 0x4c, 0x1f, 0xa3, 0x81,
        0x55, 0x72, 0x98, 0x1e, 0x06, 0x02, 0x49, 0x30, 0x18, 0x10, 0x98, 0x04, 0x6a, 0x81, 0xb3, 0xa8, 0xd0, 0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x50, 0xe8,
        0x8a, 0x95, 0xc0, 0x2d, 0x04, 0x7e, 0xf6, 0xb2, 0x9c, 0x9e, 0x1e, 0x97, 0xe7, 0xc0, 0x5f, 0xec, 0xf7, 0x43, 0xcf, 0xbf, 0xed, 0x70, 0xda, 0x0d, 0x04, 0x66, 0x83, 0xbf, 0x60, 0x95, 0x1c, 0xa6,
        0x87, 0x81, 0x40, 0xeb, 0x8f, 0x94, 0x74, 0xc3, 0xeb, 0x74, 0x68, 0x98, 0x3e, 0x46, 0x03, 0xab, 0xe4, 0x30, 0x3d, 0x0c, 0x04, 0x92, 0x60, 0x30, 0x20, 0x30, 0x09, 0xd4, 0x02, 0x67, 0x51, 0xa1,
        0xf5, 0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40, 0xa0, 0xd0, 0x25, 0xab, 0x81, 0x5b, 0x08, 0xfc, 0xec, 0x65, 0x39, 0x3d, 0x3d, 0x2e, 0xcf, 0x81, 0xbf, 0xd8, 0xef,
        0x87, 0x9e, 0x7f, 0xdb, 0xe1, 0xb4, 0x1b, 0x08, 0xcc, 0x06, 0x7f, 0xc1, 0x2a, 0x39, 0x4c, 0x0f, 0x03, 0x81, 0xd6, 0x1f, 0x29, 0xe9, 0x86, 0xd7, 0xe9, 0xd0, 0x30, 0x7d, 0x8c, 0x06, 0x56, 0xc9,
        0x61, 0x7a, 0x18, 0x08, 0x24, 0xc1, 0x60, 0x40, 0x60, 0x12, 0xa8, 0x05, 0xce, 0xa2, 0x42, 0xeb, 0x8f, 0x64, 0x2f, 0x8f, 0xe9, 0x6f, 0x39, 0xb0, 0x49, 0x82, 0xcd, 0x80, 0x40, 0xa1, 0x6b, 0x66,
        0x07, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0xa6, 0x00, 0x06, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x80, 0x52, 0xd3, 0x43, 0x4d, 0x9c,
        0xd3, 0x4c, 0x91, 0xb3, 0xa4, 0x0e, 0xd2, 0x4c, 0x7e, 0x71, 0xdb, 0x66, 0xe0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x43, 0x4d, 0x0f, 0x35, 0x71, 0x4e, 0x33, 0x45, 0xce, 0x92, 0x3a, 0x48, 0x33, 0x89,
        0x9d, 0x8f, 0xdc, 0xb6, 0x21, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x25, 0x70, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44,
        0xa4, 0xc0, 0x24, 0x1c, 0x9c, 0xd3, 0x48, 0x13, 0xd0, 0x4c, 0x7e, 0x71, 0xdb, 0x56, 0x70, 0x1c, 0xd2, 0xcf, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
        0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x94, 0x8a, 0x80, 0x4c, 0x39, 0x07, 0x14, 0x75, 0x00, 0x81, 0x31, 0x02, 0x10, 0x04, 0x41, 0xf4, 0x1b, 0x23, 0x00, 0x41, 0x10, 0xc4, 0xbf,
        0x11, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0x6c, 0xc4, 0x34, 0x59, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x1b, 0x31, 0x51, 0x56, 0x30, 0x62, 0x90, 0x00,
        0x20, 0x08, 0x06, 0xcc, 0x46, 0x4c, 0x90, 0x15, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb3, 0x11, 0xd3, 0x63, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0x6c, 0x04, 0x35, 0x59, 0xc1,
        0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x1b, 0x41, 0x51, 0x56, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xcc, 0x46, 0x4c, 0x93, 0x25, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb3, 0x11,
        0x13, 0x65, 0x09, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0x6c, 0xc4, 0x04, 0x59, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x1b, 0x31, 0x3d, 0x96, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08,
        0x06, 0xcc, 0x46, 0x50, 0x93, 0x25, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb3, 0x11, 0x14, 0x65, 0x09, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x68, 0xc5, 0x34, 0x59, 0xd3, 0x88, 0x41,
        0x02, 0x80, 0x20, 0x18, 0x28, 0x9b, 0x41, 0x51, 0x19, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x77, 0x54, 0x15, 0x52, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xd2, 0x21, 0x96, 0x95,
        0x5d, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x78, 0xc9, 0x75, 0x71, 0xd8, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x9f, 0x82, 0x61, 0x4b, 0x66, 0x83, 0x01, 0x1f, 0x1b, 0x0c, 0xf8, 0xd8,
        0x60, 0xc0, 0x67, 0xc4, 0x20, 0x01, 0x40, 0x10, 0x0c, 0x94, 0x31, 0x70, 0x38, 0xee, 0xcb, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0x40, 0x21, 0x83, 0xa7, 0xeb, 0xc4, 0x40, 0x1b, 0x31, 0x48, 0x00,
        0x10, 0x04, 0x03, 0xa5, 0x0c, 0x20, 0xcf, 0x8b, 0x36, 0x33, 0x06, 0xf8, 0x98, 0x31, 0xc0, 0xc7, 0x8c, 0x01, 0x3e, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0xa4, 0x01, 0x25, 0x06, 0x62, 0x50,
        0x06, 0xde, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x6a, 0x50, 0x8d, 0xc1, 0x18, 0xa0, 0xc1, 0x37, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xca, 0x1a, 0x58, 0x64, 0x40, 0x06, 0x17, 0x18, 0x98,
        0x31, 0xc0, 0xc7, 0x8c, 0x01, 0x3e, 0x66, 0x0c, 0xf0, 0xb1, 0x01, 0x93, 0x8f, 0x0d, 0x99, 0x7c, 0x6c, 0xd0, 0xe4, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xcc, 0x1c, 0x70, 0x68, 0xb0, 0x06,
        0x6e, 0x30, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x33, 0x07, 0x1c, 0x1a, 0xac, 0x01, 0x1c, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0xcc, 0x01, 0x87, 0x06, 0x6b, 0xe0, 0x05, 0x08,
        0x00, 0x61, 0x20, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x94, 0x8a, 0x80, 0x4c, 0x29, 0x94, 0x75, 0x00, 0x81, 0x11, 0x00,
        0x00, 0x23, 0x06, 0x05, 0x00, 0x82, 0x60, 0x70, 0x70, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x1b, 0x21, 0x49, 0x95, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xcc, 0x46, 0x48, 0xd2,
        0x25, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb3, 0x11, 0x92, 0x54, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0x6c, 0x84, 0x24, 0x31, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28,
        0x5c, 0x24, 0x49, 0x95, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x4a, 0x27, 0x4d, 0x13, 0x36, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xe2, 0x4d, 0x14, 0x75, 0x50, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0xa0, 0x7c, 0x54, 0x55, 0x3d, 0xd5, 0x0a, 0xc3, 0x19, 0x58, 0xc5, 0x0a, 0xc3, 0x19, 0x5c, 0xc6, 0x0a, 0xc3, 0x19, 0x60, 0xc7, 0x0a, 0xc3, 0x19, 0x64, 0x08, 0x11, 0x60, 0x10, 0x01,
        0x06, 0x11, 0x60, 0x10, 0x01, 0x86, 0x11, 0x0c, 0x7c, 0x8c, 0x60, 0xe0, 0x63, 0x04, 0x03, 0x1f, 0x23, 0x18, 0xf8, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb3, 0x06, 0x94, 0x18, 0x88, 0x41,
        0x19, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0xac, 0x01, 0x25, 0x06, 0x62, 0x70, 0x06, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x6b, 0x40, 0x89, 0x81, 0x18, 0x54, 0xc2, 0x88,
        0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x6b, 0x40, 0x89, 0x81, 0x18, 0x70, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE hs = {
        use_dxil ? (const void*)hs_code_dxil : (const void*)hs_code_dxbc,
        use_dxil ? sizeof(hs_code_dxil) : sizeof(hs_code_dxbc)
    };
    static const DWORD ds_code_dxbc[] =
    {
        0x43425844, 0x6a721eb5, 0xa3ba6439, 0x3f31e765, 0x07e272c6, 0x00000001, 0x00000304, 0x00000004,
        0x00000030, 0x00000064, 0x00000148, 0x0000019c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x47534350, 0x000000dc, 0x00000007, 0x00000008, 0x000000b0, 0x00000000, 0x0000000b,
        0x00000003, 0x00000000, 0x00000001, 0x000000be, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x00000e0e, 0x000000b0, 0x00000001, 0x0000000b, 0x00000003, 0x00000001, 0x00000001, 0x000000b0,
        0x00000002, 0x0000000b, 0x00000003, 0x00000002, 0x00000001, 0x000000b0, 0x00000003, 0x0000000b,
        0x00000003, 0x00000003, 0x00000001, 0x000000c5, 0x00000000, 0x0000000c, 0x00000003, 0x00000004,
        0x00000001, 0x000000c5, 0x00000001, 0x0000000c, 0x00000003, 0x00000005, 0x00000001, 0x545f5653,
        0x46737365, 0x6f746361, 0x45430072, 0x5245544e, 0x5f565300, 0x69736e49, 0x65546564, 0x61467373,
        0x726f7463, 0xababab00, 0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x58454853,
        0x00000160, 0x00040050, 0x00000058, 0x01002093, 0x01001895, 0x0100086a, 0x0300005f, 0x0011b0e2,
        0x00000000, 0x0200005f, 0x0001c032, 0x0400005f, 0x002190f2, 0x00000004, 0x00000000, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x02000068, 0x00000002,
        0x0a000000, 0x001000f2, 0x00000000, 0x80219e46, 0x00000041, 0x00000002, 0x00000000, 0x00219e46,
        0x00000003, 0x00000000, 0x09000032, 0x001000f2, 0x00000000, 0x0001c006, 0x00100e46, 0x00000000,
        0x00219e46, 0x00000002, 0x00000000, 0x0a000000, 0x001000f2, 0x00000001, 0x80219e46, 0x00000041,
        0x00000000, 0x00000000, 0x00219e46, 0x00000001, 0x00000000, 0x09000032, 0x001000f2, 0x00000001,
        0x0001c006, 0x00100e46, 0x00000001, 0x00219e46, 0x00000000, 0x00000000, 0x08000000, 0x001000f2,
        0x00000000, 0x00100e46, 0x00000000, 0x80100e46, 0x00000041, 0x00000001, 0x08000032, 0x001020f2,
        0x00000000, 0x0001c556, 0x00100e46, 0x00000000, 0x00100e46, 0x00000001, 0x05000036, 0x00102072,
        0x00000001, 0x0011b796, 0x00000000, 0x05000036, 0x00102082, 0x00000001, 0x00004001, 0x3f800000,
        0x0100003e,
    };
    static const BYTE ds_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x99, 0x42, 0x1c, 0xe3, 0xdd, 0x1b, 0x39, 0x93, 0x0b, 0xc5, 0x18, 0x45, 0x55, 0x20, 0x2c, 0xc8, 0x01, 0x00, 0x00, 0x00, 0x9b, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xe6, 0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x00, 0x33, 0x03, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x5a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x53, 0x47, 0x31, 0x11, 0x01, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63,
        0x74, 0x6f, 0x72, 0x00, 0x53, 0x56, 0x5f, 0x49, 0x6e, 0x73, 0x69, 0x64, 0x65, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x43, 0x45, 0x4e, 0x54, 0x45, 0x52, 0x00, 0x50,
        0x53, 0x56, 0x30, 0x2c, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
        0xff, 0xff, 0xff, 0x04, 0x00, 0x06, 0x00, 0x01, 0x02, 0x03, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x43, 0x45,
        0x4e, 0x54, 0x45, 0x52, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03,
        0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x71, 0x19, 0x03,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x04, 0x71, 0x1a, 0x03, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x43, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x60, 0x07, 0x00, 0x00, 0x60, 0x00, 0x04, 0x00, 0xd8,
        0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x48, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xcf, 0x01, 0x00, 0x00, 0x0b,
        0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e,
        0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00,
        0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b,
        0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13,
        0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84,
        0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x40, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc3, 0x20, 0x85, 0x4c, 0x31,
        0x06, 0x21, 0xa4, 0x10, 0x42, 0xc5, 0x00, 0x84, 0x90, 0x32, 0x48, 0x95, 0x62, 0x10, 0x42, 0x0a, 0xb1, 0x81, 0x80, 0xfc, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36,
        0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07,
        0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40,
        0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x61, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x08,
        0xc5, 0x50, 0x04, 0x65, 0x40, 0xa5, 0x24, 0xca, 0xa0, 0x10, 0x46, 0x00, 0x8a, 0xa0, 0x64, 0x4a, 0xa0, 0x68, 0xe8, 0x8d, 0xf5, 0x11, 0x48, 0x20, 0x10, 0x08, 0x06, 0x81, 0x01, 0x28, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88,
        0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x88, 0x62, 0x82, 0x40, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x10,
        0xc7, 0x06, 0x61, 0x30, 0x28, 0x90, 0xcd, 0x4d, 0x10, 0x08, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58, 0x9c, 0x09,
        0xc2, 0x33, 0x6d, 0x08, 0x20, 0x1e, 0x64, 0x73, 0x5f, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0xc5, 0x32, 0x41, 0x28, 0x98,
        0x0d, 0x01, 0x31, 0x41, 0x28, 0x9a, 0x09, 0x42, 0xe1, 0x6c, 0x58, 0x88, 0x89, 0xaa, 0xac, 0x6b, 0xb8, 0x08, 0x0c, 0xd8, 0x10, 0x64, 0x2c, 0x86, 0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0xc5, 0xb3,
        0x61, 0x19, 0x36, 0x0a, 0xb3, 0xb8, 0xe1, 0x1a, 0x30, 0x60, 0x83, 0x90, 0x75, 0x6c, 0xa6, 0xac, 0xbe, 0xa8, 0xca, 0xe6, 0xe6, 0x8c, 0xc2, 0xc6, 0xe8, 0xde, 0xe4, 0x26, 0x08, 0x05, 0x34, 0x41,
        0x20, 0x92, 0x09, 0x02, 0xa1, 0x6c, 0x20, 0x88, 0x21, 0x0c, 0xc4, 0x60, 0x82, 0x50, 0x44, 0x1b, 0x16, 0xe2, 0xa3, 0xc0, 0x60, 0x0c, 0x30, 0x83, 0x0c, 0x88, 0x0a, 0xe0, 0x34, 0x65, 0xf5, 0x95,
        0xe4, 0x36, 0x97, 0x46, 0x56, 0x46, 0x55, 0x36, 0x37, 0x67, 0x14, 0x36, 0x46, 0xf7, 0x26, 0x37, 0x41, 0x28, 0xa4, 0x0d, 0x02, 0x31, 0x6c, 0x58, 0x06, 0x33, 0xa0, 0xce, 0x00, 0x0d, 0xb0, 0x30,
        0x20, 0x03, 0xa3, 0x02, 0x68, 0x0c, 0x15, 0x39, 0x51, 0x15, 0x49, 0x6d, 0x58, 0xc2, 0x40, 0x0d, 0x28, 0xcc, 0xc2, 0x86, 0x8a, 0xc0, 0x80, 0x0d, 0x43, 0x19, 0xa4, 0xc1, 0x1a, 0x6c, 0x18, 0x34,
        0x8f, 0x0d, 0x36, 0x08, 0x62, 0x60, 0x6c, 0x10, 0xc2, 0xc0, 0x0d, 0x36, 0x14, 0x8b, 0xd4, 0x06, 0xc0, 0x1b, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04,
        0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e,
        0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x90, 0x54, 0x22, 0xc3, 0x73, 0xa1, 0xcb, 0x83, 0x2b, 0x0b, 0x72, 0x73, 0x7b, 0xa3, 0x0b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b,
        0x12, 0x38, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x50, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34,
        0x37, 0xba, 0xb9, 0x29, 0xc1, 0x1b, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c,
        0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a,
        0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde,
        0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60,
        0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f,
        0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc,
        0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3,
        0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x06,
        0x70, 0x0c, 0xd2, 0xcf, 0x00, 0x48, 0x63, 0x01, 0xd8, 0x70, 0xf9, 0xce, 0xe3, 0x0f, 0x0e, 0x03, 0x20, 0x4d, 0xe9, 0x08, 0xc0, 0x84, 0x38, 0x8d, 0x5f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3, 0xe5, 0x3b,
        0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x29, 0x30, 0x09, 0x07, 0xe7, 0x34, 0xd2, 0x04, 0x34, 0x93, 0x5f,
        0xdc, 0xb6, 0x0d, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0x36, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x8e, 0x00, 0x00, 0x00, 0x13,
        0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x44, 0x4a, 0x3a, 0xa0, 0x14, 0x0a, 0x3a, 0x80, 0xc6, 0x18, 0x01, 0x08, 0x82, 0x20, 0xfe, 0x01, 0x23, 0x06, 0x08, 0x00, 0x82,
        0x60, 0xc0, 0x50, 0xc2, 0x23, 0x29, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xc0, 0x54, 0x03, 0x34, 0x21, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xc0, 0x58, 0x44, 0x44, 0x2d, 0x23, 0x06, 0x06, 0x00,
        0x82, 0x60, 0x60, 0x68, 0x47, 0x33, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x86, 0x2c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x6c, 0xd6, 0x75, 0x3d, 0xd8, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x20, 0xdc, 0x85, 0x61, 0x4d, 0x36, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x87, 0x65, 0x19, 0xa4, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xe2, 0x65, 0x9a, 0x46, 0x6d, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x80, 0x7c, 0xda, 0xb6, 0x4d, 0xdb, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x60, 0xb0, 0x71, 0x5c, 0xc4, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x12, 0x06, 0x5c,
        0xd7, 0x51, 0xdd, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x62, 0xd0, 0x79, 0x1e, 0xe6, 0x19, 0x42, 0xc4, 0xc7, 0x10, 0x22, 0x3e, 0x86, 0x10, 0xf1, 0x31, 0x84, 0x88, 0x8f, 0x11, 0x8e, 0x7c,
        0x8c, 0x78, 0xe4, 0x63, 0x04, 0x24, 0x1f, 0x23, 0x22, 0xf9, 0x18, 0xc1, 0xc0, 0xc7, 0x08, 0x06, 0x3e, 0x46, 0x30, 0xf0, 0x31, 0x82, 0x81, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x6f,
        0xa0, 0x06, 0x6b, 0xb0, 0x06, 0x63, 0x70, 0x06, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xc0, 0xc1, 0x1a, 0xb0, 0x01, 0x1b, 0x84, 0x01, 0x1a, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x12,
        0x07, 0x6c, 0xd0, 0x06, 0x6d, 0x40, 0x06, 0x69, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x88, 0x1c, 0xb4, 0x81, 0x1b, 0xb8, 0x01, 0x1a, 0xa8, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
        0x73, 0xe0, 0x06, 0x6f, 0xf0, 0x06, 0x67, 0xc0, 0x06, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xd0, 0xc1, 0x1b, 0xc0, 0x01, 0x1c, 0x94, 0x41, 0x1b, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01,
        0x52, 0x07, 0x70, 0x10, 0x07, 0x71, 0x80, 0x06, 0x6e, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x88, 0x1d, 0xc4, 0x81, 0x1c, 0xc8, 0x01, 0x1b, 0xbc, 0x81, 0x21, 0x44, 0x7c, 0x0c, 0x21, 0xe2,
        0x63, 0x08, 0x11, 0x1f, 0x43, 0x88, 0xf8, 0x18, 0x21, 0x06, 0xf2, 0x31, 0x62, 0x0c, 0xe4, 0x63, 0x04, 0x19, 0xc8, 0xc7, 0x88, 0x32, 0x90, 0x8f, 0x31, 0x54, 0x7c, 0x2c, 0x28, 0xe0, 0x63, 0x4d,
        0x15, 0x1f, 0x0b, 0x0c, 0xf8, 0x98, 0x63, 0xc5, 0xc7, 0x82, 0x03, 0x3e, 0xf6, 0x5c, 0xf1, 0xb1, 0x00, 0x81, 0x8f, 0x1d, 0x6d, 0x20, 0x1f, 0x33, 0xdc, 0x40, 0x3e, 0x56, 0xbc, 0x81, 0x7c, 0x8c,
        0x80, 0x03, 0xf9, 0x18, 0x01, 0x06, 0xf0, 0x31, 0x02, 0x0c, 0xe0, 0x63, 0x04, 0x18, 0xc0, 0xc7, 0x08, 0x30, 0x80, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0xae, 0xc0, 0x07, 0xab, 0xb0,
        0x0a, 0xa3, 0x40, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xe2, 0x0a, 0x7c, 0xb0, 0x0a, 0xab, 0x00, 0x0a, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0xae, 0xc0, 0x07, 0xab, 0xb0, 0x0a,
        0xa2, 0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xe2, 0x0a, 0x7c, 0xb0, 0x0a, 0xab, 0x50, 0x0a, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0xae, 0xc0, 0x07, 0xac, 0xb0, 0x0a, 0xa3,
        0x90, 0x07, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0xb8, 0x02, 0x1f, 0xb0, 0xc2, 0x2a, 0x80, 0x02, 0x1e, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xe2, 0x0a, 0x7c, 0xc0, 0x0a, 0xab, 0x20,
        0x0a, 0x77, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x8a, 0x2b, 0xf0, 0x01, 0x2b, 0xac, 0x42, 0x29, 0xe8, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE ds = {
        use_dxil ? (const void*)ds_code_dxil : (const void*)ds_code_dxbc,
        use_dxil ? sizeof(ds_code_dxil) : sizeof(ds_code_dxbc)
    };
#if 0
    float4 main(in float4 p : SV_Position, in float4 color : COLOR) : SV_Target
    {
        return color;
    }
#endif
    static const DWORD ps_code_dxbc[] =
    {
        0x43425844, 0xbd83f517, 0x8974e87a, 0xaf402223, 0xaec7f351, 0x00000001, 0x000000f8, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000003c, 0x00000050,
        0x0000000f, 0x0100086a, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const BYTE ps_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x63, 0x87, 0x0a, 0x9e, 0xcc, 0x88, 0x70, 0xdd, 0x61, 0x01, 0x91, 0xf4, 0x4b, 0x44, 0xef, 0x22, 0x01, 0x00, 0x00, 0x00, 0x50, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xa6, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00, 0x7c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x5a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00,
        0x50, 0x53, 0x56, 0x30, 0x94, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c,
        0xcc, 0x04, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x33, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xb4, 0x04, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0x2a, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00,
        0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04,
        0x13, 0x22, 0xa4, 0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x30, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30,
        0x98, 0x23, 0x40, 0x8a, 0x31, 0x44, 0x54, 0x44, 0x56, 0x0c, 0x20, 0xa2, 0x1a, 0xc2, 0x81, 0x80, 0x54, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x18, 0x01, 0x28,
        0x84, 0x62, 0xa0, 0x2a, 0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x22, 0x28, 0x10, 0xda, 0xb1, 0x14, 0x84, 0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x01, 0x79, 0x18, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x88, 0x61,
        0x82, 0x40, 0x10, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x10, 0xc5, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x08, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82,
        0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58, 0x9c, 0x09, 0xc2, 0xd2, 0x6c, 0x08, 0x20, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26,
        0x08, 0xc5, 0x31, 0x41, 0x28, 0x90, 0x0d, 0x01, 0x31, 0x41, 0x28, 0x92, 0x09, 0x42, 0xa1, 0x6c, 0x58, 0x88, 0x89, 0xaa, 0xac, 0x6b, 0xb8, 0x08, 0x0c, 0x60, 0x31, 0xf4, 0xc4, 0xf4, 0x24, 0x35,
        0x41, 0x28, 0x96, 0x0d, 0xcb, 0xa0, 0x51, 0x98, 0xb5, 0x0d, 0xd7, 0x80, 0x01, 0x1b, 0x84, 0x8c, 0x63, 0x32, 0x65, 0xf5, 0x45, 0x15, 0x26, 0x77, 0x56, 0x46, 0x37, 0x41, 0x28, 0x98, 0x0d, 0x0b,
        0xe1, 0x51, 0x9f, 0x85, 0x0d, 0x17, 0x81, 0x01, 0x1b, 0x02, 0x30, 0xd8, 0x30, 0x74, 0x61, 0x00, 0x6c, 0x28, 0x16, 0x49, 0x0c, 0x00, 0xa0, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99,
        0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a,
        0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x82, 0xa4, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d,
        0xda, 0x9b, 0xdb, 0xdc, 0x94, 0xc0, 0xa9, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56, 0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0x80, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9,
        0xe5, 0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0xc4, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28,
        0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x44, 0x33, 0x00, 0xa5, 0x40, 0x55, 0x02, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x40, 0x8c, 0xd3, 0x1c, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x51,
        0xf3, 0x38, 0x02, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0xe4, 0x40, 0x0f, 0x52, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x31, 0x3d, 0x11, 0xc4, 0x18, 0x23, 0x06, 0x09, 0x00, 0x82,
        0x60, 0x80, 0x4c, 0x46, 0x14, 0x2d, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x93, 0x11, 0x45, 0xc5, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x64, 0x44, 0x91, 0x22, 0x8c, 0x18,
        0x24, 0x00, 0x08, 0x82, 0x01, 0x32, 0x19, 0x51, 0xd4, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE ps = {
        use_dxil ? (const void*)ps_code_dxil : (const void*)ps_code_dxbc,
        use_dxil ? sizeof(ps_code_dxil) : sizeof(ps_code_dxbc)
    };

    memset(&desc, 0, sizeof(desc));
    desc.no_pipeline = true;
    if (!init_test_context(&context, &desc))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    init_pipeline_state_desc(&pso_desc, context.root_signature,
            context.render_target_desc.Format, NULL, NULL, NULL);
    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    pso_desc.PS = ps;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_1_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 1, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff000000, 0);

    destroy_test_context(&context);
}

void test_hull_shader_fork_phase_dxbc(void)
{
    test_hull_shader_fork_phase(false);
}

void test_hull_shader_fork_phase_dxil(void)
{
    test_hull_shader_fork_phase(true);
}

void test_tessellation_read_tesslevel(void)
{
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    D3D12_STREAM_OUTPUT_BUFFER_VIEW sobv;
    D3D12_INPUT_LAYOUT_DESC input_layout;
    struct test_context_desc desc;
    struct resource_readback rb;
    struct test_context context;
    ID3D12Resource *so_buffer;
    ID3D12CommandQueue *queue;
    unsigned int i, j;
    HRESULT hr;

#if 0
    struct data
    {
        float4 position : SV_Position;
    };
    data vs_main()
    {
        data d;
        d.position = float4(1.0, 2.0, 3.0, 4.0);
        return d;
    }

    struct patch_constant_data
    {
        float tess_factor[2] : SV_TessFactor;
        float3 a : A;
        float3 b : B;
    };

    void patch_constant(OutputPatch<data, 1> control_points,
            uint prim_id : SV_PrimitiveID,
            out patch_constant_data output)
    {
        output.tess_factor[0] = 2.0;
        output.tess_factor[1] = 1.0;
        output.a = float3(2, 4, 10);
        output.b = float3(3, 3, 12);
    }

    [domain("isoline")]
    [outputcontrolpoints(1)]
    [partitioning("integer")]
    [outputtopology("line")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 1> input)
    {
        return input[0];
    }

    [domain("isoline")]
    void ds_main(patch_constant_data input,
            float tess_factor[2] : SV_TessFactor,
            float2 tess_coord : SV_DomainLocation,
            float3 a : A,
            float3 b : B,
            const OutputPatch<data, 1> patch,
            out float4 position : SV_Position,
            out float4 out_color : COLOR)
    {
        position = float4(patch[0].position.xy, tess_coord);
        out_color = float4(0.5 * (a.xy + b.yz), tess_factor[0], tess_factor[1]);
    }
#endif
    static const DWORD vs_code[] =
    {
        0x43425844, 0xcb24c840, 0x8a22891b, 0xcf0dbe1d, 0x7b0a4fdb, 0x00000001, 0x000000b8, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000,
        0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x00000040, 0x00010050, 0x00000010,
        0x0100086a, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0x3f800000, 0x40000000, 0x40400000, 0x40800000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE vs = {vs_code, sizeof(vs_code)};
    static const DWORD hs_code[] =
    {
        0x43425844, 0x67db9c59, 0xff2c4d5e, 0xc2a4d15b, 0x0f71495f, 0x00000001, 0x00000384, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000011c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000007c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x00000010, 0x00000003, 0x00000000, 0x00000e01,
        0x00000076, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000010e, 0x00000068, 0x00000001,
        0x0000000f, 0x00000003, 0x00000001, 0x00000e01, 0x00000078, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x0000010e, 0x545f5653, 0x46737365, 0x6f746361, 0x00410072, 0xabab0042, 0x58454853,
        0x00000260, 0x00030050, 0x00000098, 0x01000071, 0x01000893, 0x01000894, 0x01000895, 0x01000896,
        0x01001097, 0x0100086a, 0x00001835, 0x0000000a, 0x41200000, 0x40800000, 0x40000000, 0x40000000,
        0x41400000, 0x40400000, 0x40400000, 0x3f800000, 0x01000072, 0x0400005f, 0x002010f2, 0x00000001,
        0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x06000036, 0x001020f2, 0x00000000, 0x00201e46,
        0x00000000, 0x00000000, 0x0100003e, 0x01000073, 0x02000099, 0x00000002, 0x0200005f, 0x00017000,
        0x04000067, 0x00102012, 0x00000000, 0x00000016, 0x04000067, 0x00102012, 0x00000001, 0x00000015,
        0x02000068, 0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000002, 0x04000036, 0x00100012,
        0x00000000, 0x0001700a, 0x07000036, 0x00902012, 0x0010000a, 0x00000000, 0x0090903a, 0x0010000a,
        0x00000000, 0x0100003e, 0x01000073, 0x02000099, 0x00000002, 0x0200005f, 0x00017000, 0x03000065,
        0x00102022, 0x00000000, 0x03000065, 0x00102022, 0x00000001, 0x02000068, 0x00000001, 0x0400005b,
        0x00102022, 0x00000000, 0x00000002, 0x04000036, 0x00100012, 0x00000000, 0x0001700a, 0x07000036,
        0x00902022, 0x0010000a, 0x00000000, 0x0090902a, 0x0010000a, 0x00000000, 0x0100003e, 0x01000073,
        0x02000099, 0x00000002, 0x0200005f, 0x00017000, 0x03000065, 0x00102042, 0x00000000, 0x03000065,
        0x00102042, 0x00000001, 0x02000068, 0x00000001, 0x0400005b, 0x00102042, 0x00000000, 0x00000002,
        0x04000036, 0x00100012, 0x00000000, 0x0001700a, 0x07000036, 0x00902042, 0x0010000a, 0x00000000,
        0x0090901a, 0x0010000a, 0x00000000, 0x0100003e, 0x01000073, 0x02000099, 0x00000002, 0x0200005f,
        0x00017000, 0x03000065, 0x00102082, 0x00000000, 0x03000065, 0x00102082, 0x00000001, 0x02000068,
        0x00000001, 0x0400005b, 0x00102082, 0x00000000, 0x00000002, 0x04000036, 0x00100012, 0x00000000,
        0x0001700a, 0x07000036, 0x00902082, 0x0010000a, 0x00000000, 0x0090900a, 0x0010000a, 0x00000000,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs = {hs_code, sizeof(hs_code)};
    static const DWORD ds_code[] =
    {
        0x43425844, 0xef67f8be, 0x1be137b6, 0x0e4a6b01, 0xf0cd92d8, 0x00000001, 0x00000264, 0x00000004,
        0x00000030, 0x00000064, 0x000000e8, 0x0000013c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x47534350, 0x0000007c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x00000010,
        0x00000003, 0x00000000, 0x00000101, 0x00000076, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000060e, 0x00000068, 0x00000001, 0x0000000f, 0x00000003, 0x00000001, 0x00000101, 0x00000078,
        0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x00000c0e, 0x545f5653, 0x46737365, 0x6f746361,
        0x00410072, 0xabab0042, 0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x58454853,
        0x00000120, 0x00040050, 0x00000048, 0x01000893, 0x01000895, 0x0100086a, 0x04000061, 0x0011b012,
        0x00000000, 0x00000016, 0x0300005f, 0x0011b062, 0x00000000, 0x04000061, 0x0011b012, 0x00000001,
        0x00000015, 0x0300005f, 0x0011b0c2, 0x00000001, 0x0200005f, 0x0001c032, 0x0400005f, 0x00219032,
        0x00000001, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2,
        0x00000001, 0x02000068, 0x00000001, 0x06000036, 0x00102032, 0x00000000, 0x00219046, 0x00000000,
        0x00000000, 0x04000036, 0x001020c2, 0x00000000, 0x0001c406, 0x07000000, 0x00100032, 0x00000000,
        0x0011b596, 0x00000000, 0x0011bae6, 0x00000001, 0x0a000038, 0x00102032, 0x00000001, 0x00100046,
        0x00000000, 0x00004002, 0x3f000000, 0x3f000000, 0x00000000, 0x00000000, 0x05000036, 0x00102042,
        0x00000001, 0x0011b00a, 0x00000000, 0x05000036, 0x00102082, 0x00000001, 0x0011b00a, 0x00000001,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ds = {ds_code, sizeof(ds_code)};
    static const D3D12_SO_DECLARATION_ENTRY so_declaration[] =
    {
        {0, "SV_POSITION",  0, 0, 4, 0},
        {0, "COLOR",        0, 0, 4, 0},
    };
    unsigned int stride = 32;
    static const float reference[4][8] = {
        { 1.0f, 2.0f, 0.0f, 0.0f, 2.5f, 8.0f, 2.0f, 1.0f },
        { 1.0f, 2.0f, 1.0f, 0.0f, 2.5f, 8.0f, 2.0f, 1.0f },
        { 1.0f, 2.0f, 0.0f, 0.5f, 2.5f, 8.0f, 2.0f, 1.0f },
        { 1.0f, 2.0f, 1.0f, 0.5f, 2.5f, 8.0f, 2.0f, 1.0f },
    };
    memset(&desc, 0, sizeof(desc));
    desc.root_signature_flags = D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT
            | D3D12_ROOT_SIGNATURE_FLAG_ALLOW_STREAM_OUTPUT;
    desc.no_pipeline = true;
    if (!init_test_context(&context, &desc))
        return;
    command_list = context.list;
    queue = context.queue;

    memset(&input_layout, 0, sizeof(input_layout));

    init_pipeline_state_desc(&pso_desc, context.root_signature,
            DXGI_FORMAT_UNKNOWN, NULL, NULL, &input_layout);
    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    memset(&pso_desc.PS, 0, sizeof(pso_desc.PS));
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    pso_desc.StreamOutput.NumEntries = ARRAY_SIZE(so_declaration);
    pso_desc.StreamOutput.pSODeclaration = so_declaration;
    pso_desc.StreamOutput.pBufferStrides = &stride;
    pso_desc.StreamOutput.NumStrides = 1;
    pso_desc.StreamOutput.RasterizedStream = D3D12_SO_NO_RASTERIZED_STREAM;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    so_buffer = create_default_buffer(context.device, 4096,
            D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_STREAM_OUT);
    sobv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(so_buffer);
    sobv.SizeInBytes = 1024;
    sobv.BufferFilledSizeLocation = sobv.BufferLocation + sobv.SizeInBytes;

    ID3D12GraphicsCommandList_SOSetTargets(command_list, 0, 1, &sobv);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_1_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    //reset_command_list(command_list, context.allocator);
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_STREAM_OUT, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(so_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

    for (i = 0; i < 4; i++)
    {
        float *elems = get_readback_data(&rb, i, 0, 0, stride);
        for (j = 0; j < 8; j++)
        {
            ok(compare_float(reference[i][j], elems[j], 0),
                    "Got unexpected value %f for [%u][%u], expected %f.\n",
                    elems[j], i, j, reference[i][j]);
        }
    }

    release_resource_readback(&rb);
    ID3D12Resource_Release(so_buffer);
    destroy_test_context(&context);
}

static void test_line_tessellation(bool use_dxil)
{
    ID3D12Resource *vb, *so_buffer, *readback_buffer;
    D3D12_QUERY_DATA_SO_STATISTICS *so_statistics;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12GraphicsCommandList *command_list;
    D3D12_QUERY_HEAP_DESC query_heap_desc;
    D3D12_STREAM_OUTPUT_BUFFER_VIEW sobv;
    D3D12_INPUT_LAYOUT_DESC input_layout;
    struct test_context_desc desc;
    D3D12_VERTEX_BUFFER_VIEW vbv;
    struct resource_readback rb;
    ID3D12QueryHeap *query_heap;
    struct test_context context;
    const struct vec4 *expected;
    ID3D12CommandQueue *queue;
    struct vec4 *data;
    bool broken_warp;
    unsigned int i;
    HRESULT hr;

#if 0
    struct data
    {
        float4 position : SV_Position;
        float3 color : COLOR;
        float line_density : LINE_DENSITY;
        float line_detail : LINE_DETAIL;
    };

    data vs_main(data input)
    {
        return input;
    }

    struct patch_constant_data
    {
        float tess_factor[2] : SV_TessFactor;
        float3 color : COLOR;
        uint prim_id : PRIMITIVE_ID;
    };

    void patch_constant(OutputPatch<data, 1> control_points,
            uint prim_id : SV_PrimitiveID,
            out patch_constant_data output)
    {
        output.tess_factor[0] = control_points[0].line_density;
        output.tess_factor[1] = control_points[0].line_detail;
        output.color = control_points[0].color;
        output.prim_id = prim_id;
    }

    [domain("isoline")]
    [outputcontrolpoints(1)]
    [partitioning("integer")]
    [outputtopology("line")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 1> input)
    {
        return input[0];
    }

    [domain("isoline")]
    void ds_main(patch_constant_data input,
            // float tess_factor[2] : SV_TessFactor, DXC refused to compile, register overlap
            float2 tess_coord : SV_DomainLocation,
            //float3 color : COLOR, DXC refused to compile, register overlap
            //uint prim_id : PRIMITIVE_ID, DXC refused to compile, register overlap
            const OutputPatch<data, 1> patch,
            out float4 position : SV_Position,
            out float4 out_color : COLOR,
            out float4 out_prim_id : PRIMITIVE_ID)
    {
        position = patch[0].position;
        out_color = float4(color, 1.0); // input.color on DXC
        out_prim_id = prim_id; // input.prim_id on DXC
    }
#endif
    static const DWORD vs_code_dxbc[] =
    {
        0x43425844, 0x43d2f821, 0xc4bcdf60, 0xadbf5ed0, 0xe55b715d, 0x00000001, 0x00000230, 0x00000003,
        0x0000002c, 0x000000c8, 0x00000164, 0x4e475349, 0x00000094, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000074, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000707, 0x0000007a, 0x00000000, 0x00000000, 0x00000003, 0x00000002,
        0x00000101, 0x00000087, 0x00000000, 0x00000000, 0x00000003, 0x00000003, 0x00000101, 0x505f5653,
        0x7469736f, 0x006e6f69, 0x4f4c4f43, 0x494c0052, 0x445f454e, 0x49534e45, 0x4c005954, 0x5f454e49,
        0x41544544, 0xab004c49, 0x4e47534f, 0x00000094, 0x00000004, 0x00000008, 0x00000068, 0x00000000,
        0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000074, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x00000807, 0x0000007a, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x00000708,
        0x00000087, 0x00000000, 0x00000000, 0x00000003, 0x00000002, 0x00000e01, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4f4c4f43, 0x494c0052, 0x445f454e, 0x49534e45, 0x4c005954, 0x5f454e49, 0x41544544,
        0xab004c49, 0x58454853, 0x000000c4, 0x00010050, 0x00000031, 0x0100086a, 0x0300005f, 0x001010f2,
        0x00000000, 0x0300005f, 0x00101072, 0x00000001, 0x0300005f, 0x00101012, 0x00000002, 0x0300005f,
        0x00101012, 0x00000003, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x00102072,
        0x00000001, 0x03000065, 0x00102082, 0x00000001, 0x03000065, 0x00102012, 0x00000002, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x05000036, 0x00102072, 0x00000001, 0x00101246,
        0x00000001, 0x05000036, 0x00102082, 0x00000001, 0x0010100a, 0x00000002, 0x05000036, 0x00102012,
        0x00000002, 0x0010100a, 0x00000003, 0x0100003e,
    };
    static const BYTE vs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xb7, 0xf9, 0xd0, 0x58, 0x2b, 0x88, 0x39, 0xb7, 0x5e, 0x4d, 0x22, 0x99, 0x3f, 0x8a, 0x53, 0x6e, 0x01, 0x00, 0x00, 0x00, 0xba, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xba, 0x01, 0x00, 0x00, 0x0a, 0x03, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0xb3, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x4f,
        0x53, 0x47, 0x31, 0xb3, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52,
        0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x50, 0x53, 0x56, 0x30, 0x48, 0x01,
        0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00,
        0x00, 0x00, 0x04, 0x04, 0x00, 0x04, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x43,
        0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x43, 0x4f,
        0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x01, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x41, 0x00, 0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x03, 0x41, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x01, 0x43, 0x00, 0x03, 0x02, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x71, 0x00, 0x03, 0x02, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x02, 0x41, 0x00, 0x03, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00,
        0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xa8, 0x05, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x6a, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x90, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x61, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00,
        0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92,
        0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22,
        0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02,
        0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20,
        0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4, 0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84,
        0x84, 0x4c, 0x10, 0x30, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30, 0x98, 0x23, 0x40, 0x8a, 0x31, 0x44, 0x54, 0x44, 0x56, 0x0c, 0x20, 0xa2, 0x1a, 0xc2, 0x81, 0x80, 0x7c, 0x20,
        0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f,
        0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x0e, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11,
        0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x18, 0x01, 0x28, 0x84, 0x62, 0x28, 0x82, 0x32, 0xa0, 0x2a, 0x89, 0x11, 0x80, 0x42, 0x28, 0x83, 0x12, 0x28, 0x02, 0xda, 0xb1,
        0x9e, 0x26, 0x09, 0x04, 0x02, 0x81, 0xc0, 0x00, 0x14, 0x00, 0x90, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02,
        0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x88, 0x61, 0x82, 0x40, 0x10, 0x1b, 0x84, 0x81,
        0x98, 0x20, 0x10, 0xc5, 0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10, 0x08, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61,
        0x58, 0x9c, 0x09, 0xc2, 0xf2, 0x6c, 0x08, 0x20, 0x1e, 0x76, 0x73, 0x5f, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x45, 0x32,
        0x41, 0x28, 0x94, 0x0d, 0x01, 0x31, 0x41, 0x28, 0x96, 0x0d, 0x0b, 0x31, 0x51, 0x95, 0x55, 0x0d, 0x17, 0x51, 0x01, 0x2c, 0x86, 0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0x05, 0xb3, 0x61, 0x19, 0x32,
        0xaa, 0xb2, 0xaa, 0x41, 0x1b, 0x2a, 0x60, 0x82, 0x40, 0x1c, 0x64, 0x98, 0x92, 0x9c, 0x8a, 0xbe, 0x88, 0x8a, 0x9c, 0xa6, 0x92, 0xa8, 0xb2, 0x26, 0x08, 0x45, 0xb3, 0x61, 0xe1, 0x3a, 0xaa, 0xb2,
        0xaa, 0xc1, 0xe3, 0x2a, 0x60, 0x82, 0x40, 0x20, 0x5c, 0x98, 0x92, 0x9c, 0x8a, 0xbe, 0x88, 0x8a, 0xa8, 0x82, 0x92, 0x98, 0x36, 0x2c, 0x60, 0x10, 0x06, 0x54, 0x65, 0x55, 0x83, 0x07, 0x06, 0x15,
        0xb0, 0x81, 0xc0, 0xb6, 0x4f, 0x0c, 0x36, 0x2c, 0xc4, 0x44, 0x69, 0xd6, 0x35, 0x5c, 0x44, 0x05, 0x4c, 0x10, 0x0a, 0x67, 0xc3, 0x32, 0x64, 0x54, 0x65, 0x95, 0xc1, 0xa0, 0x0d, 0x15, 0xb0, 0x61,
        0xe1, 0x3a, 0xaa, 0xb2, 0xca, 0x60, 0xf0, 0x06, 0x0d, 0xd8, 0xb0, 0x80, 0x41, 0x18, 0x50, 0x95, 0x55, 0x06, 0x83, 0xc7, 0x55, 0xc0, 0x06, 0x82, 0x0c, 0xcc, 0xe0, 0x0c, 0xd0, 0x60, 0xc3, 0x30,
        0x06, 0x69, 0x00, 0x6c, 0x28, 0x16, 0x49, 0x0d, 0x00, 0xa0, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97,
        0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65,
        0x6c, 0x53, 0x82, 0xa4, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x94, 0xc0, 0xa9, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56,
        0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0x80, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9, 0xe5, 0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0xd4, 0x00, 0x79, 0x18,
        0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71,
        0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03,
        0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19,
        0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66,
        0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68,
        0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79,
        0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca,
        0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3,
        0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83,
        0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x47, 0x25, 0xfd, 0x0c, 0x80, 0x34,
        0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x44, 0x33, 0x00, 0xa5, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x60, 0x44, 0x8d, 0xe2, 0x20, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x92, 0xc3, 0x3c, 0xc9, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0xf4, 0x44, 0x90,
        0x42, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x41, 0x41, 0x52, 0x84, 0x14, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x54, 0xd1, 0x24, 0x21, 0xc6, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18,
        0x96, 0x34, 0x4d, 0xcd, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x35, 0x51, 0xd4, 0x82, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x81, 0x51, 0x55, 0xb5, 0x24, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x60, 0x64, 0x95, 0x65, 0x39, 0xca, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x99, 0x72, 0x5d, 0x11, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0xa6, 0x5c, 0x97, 0x33, 0x8c,
        0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x92, 0x29, 0xd7, 0xd5, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x64, 0xca, 0x75, 0x3d, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x99, 0x82,
        0x5d, 0xd1, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0xa6, 0x60, 0x97, 0x63, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x92, 0x29, 0xd8, 0xd5, 0x14, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x80, 0x64, 0x0a, 0x75, 0x45, 0xc8, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x99, 0x32, 0x5d, 0x51, 0x82, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE vs = {
        use_dxil ? (const void*)vs_code_dxil : (const void*)vs_code_dxbc,
        use_dxil ? sizeof(vs_code_dxil) : sizeof(vs_code_dxbc)
    };
    static const DWORD hs_code_dxbc[] =
    {
        0x43425844, 0x130f1b8c, 0x02c24095, 0x18ab42ad, 0xbb861b50, 0x00000001, 0x00000448, 0x00000004,
        0x00000030, 0x000000cc, 0x00000168, 0x000001fc, 0x4e475349, 0x00000094, 0x00000004, 0x00000008,
        0x00000068, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x00000074, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000707, 0x0000007a, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x00000808, 0x00000087, 0x00000000, 0x00000000, 0x00000003, 0x00000002, 0x00000101,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0x494c0052, 0x445f454e, 0x49534e45, 0x4c005954,
        0x5f454e49, 0x41544544, 0xab004c49, 0x4e47534f, 0x00000094, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000074, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000807, 0x0000007a, 0x00000000, 0x00000000, 0x00000003, 0x00000001,
        0x00000708, 0x00000087, 0x00000000, 0x00000000, 0x00000003, 0x00000002, 0x00000e01, 0x505f5653,
        0x7469736f, 0x006e6f69, 0x4f4c4f43, 0x494c0052, 0x445f454e, 0x49534e45, 0x4c005954, 0x5f454e49,
        0x41544544, 0xab004c49, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000,
        0x00000010, 0x00000003, 0x00000000, 0x00000e01, 0x00000076, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000010e, 0x00000068, 0x00000001, 0x0000000f, 0x00000003, 0x00000001, 0x00000e01,
        0x0000007c, 0x00000000, 0x00000000, 0x00000001, 0x00000001, 0x00000d02, 0x545f5653, 0x46737365,
        0x6f746361, 0x4f430072, 0x00524f4c, 0x4d495250, 0x56495449, 0x44495f45, 0xababab00, 0x58454853,
        0x00000244, 0x00030050, 0x00000091, 0x01000071, 0x01000893, 0x01000894, 0x01000895, 0x01000896,
        0x01001097, 0x0100086a, 0x01000072, 0x0400005f, 0x002010f2, 0x00000001, 0x00000000, 0x0400005f,
        0x00201072, 0x00000001, 0x00000001, 0x0400005f, 0x00201082, 0x00000001, 0x00000001, 0x0400005f,
        0x00201012, 0x00000001, 0x00000002, 0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102072,
        0x00000001, 0x03000065, 0x00102082, 0x00000001, 0x03000065, 0x00102012, 0x00000002, 0x06000036,
        0x001020f2, 0x00000000, 0x00201e46, 0x00000000, 0x00000000, 0x06000036, 0x001020f2, 0x00000001,
        0x00201e46, 0x00000000, 0x00000001, 0x06000036, 0x00102012, 0x00000002, 0x0020100a, 0x00000000,
        0x00000002, 0x0100003e, 0x01000073, 0x0400005f, 0x0021a082, 0x00000001, 0x00000001, 0x04000067,
        0x00102012, 0x00000000, 0x00000016, 0x06000036, 0x00102012, 0x00000000, 0x0021a03a, 0x00000000,
        0x00000001, 0x0100003e, 0x01000073, 0x0400005f, 0x0021a012, 0x00000001, 0x00000002, 0x04000067,
        0x00102012, 0x00000001, 0x00000015, 0x06000036, 0x00102012, 0x00000001, 0x0021a00a, 0x00000000,
        0x00000002, 0x0100003e, 0x01000073, 0x0400005f, 0x0021a012, 0x00000001, 0x00000001, 0x03000065,
        0x00102022, 0x00000000, 0x06000036, 0x00102022, 0x00000000, 0x0021a00a, 0x00000000, 0x00000001,
        0x0100003e, 0x01000073, 0x0400005f, 0x0021a022, 0x00000001, 0x00000001, 0x03000065, 0x00102042,
        0x00000000, 0x06000036, 0x00102042, 0x00000000, 0x0021a01a, 0x00000000, 0x00000001, 0x0100003e,
        0x01000073, 0x0400005f, 0x0021a042, 0x00000001, 0x00000001, 0x03000065, 0x00102082, 0x00000000,
        0x06000036, 0x00102082, 0x00000000, 0x0021a02a, 0x00000000, 0x00000001, 0x0100003e, 0x01000073,
        0x0200005f, 0x0000b000, 0x03000065, 0x00102022, 0x00000001, 0x04000036, 0x00102022, 0x00000001,
        0x0000b001, 0x0100003e,
    };
    static const BYTE hs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x78, 0x7d, 0x99, 0x2c, 0x3c, 0x83, 0xe3, 0xde, 0x3c, 0x35, 0x13, 0x58, 0x3d, 0x33, 0x94, 0xe0, 0x01, 0x00, 0x00, 0x00, 0xab, 0x0b, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x00, 0xbe, 0x01, 0x00, 0x00, 0x6f, 0x02, 0x00, 0x00, 0x1f, 0x04, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0xb3, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41,
        0x49, 0x4c, 0x00, 0x4f, 0x53, 0x47, 0x31, 0xb3, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x43,
        0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x50, 0x53,
        0x47, 0x31, 0xa9, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x43, 0x4f, 0x4c, 0x4f,
        0x52, 0x00, 0x50, 0x52, 0x49, 0x4d, 0x49, 0x54, 0x49, 0x56, 0x45, 0x5f, 0x49, 0x44, 0x00, 0x50, 0x53, 0x56, 0x30, 0xa8, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x02, 0x00, 0x04, 0x04, 0x03, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f,
        0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44,
        0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x52, 0x49, 0x4d, 0x49, 0x54, 0x49, 0x56, 0x45, 0x5f, 0x49, 0x44, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x43, 0x00, 0x03, 0x02, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x71, 0x00, 0x03, 0x02, 0x00, 0x00, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x41, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x20,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x43, 0x00, 0x03, 0x02, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x71, 0x00, 0x03, 0x02, 0x00, 0x00, 0x33,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x41, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x3f,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44,
        0x58, 0x49, 0x4c, 0x84, 0x07, 0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0xe1, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x6c, 0x07, 0x00, 0x00, 0x42,
        0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xd8, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06,
        0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a,
        0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04,
        0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01,
        0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20,
        0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x40, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08,
        0xc0, 0xa0, 0x08, 0x63, 0x10, 0x99, 0x01, 0x98, 0x23, 0x40, 0x8a, 0x61, 0xc6, 0x18, 0x67, 0x10, 0x2a, 0x06, 0x18, 0x63, 0x1c, 0x43, 0xaa, 0x18, 0x60, 0x8c, 0x71, 0x06, 0xb1, 0x81, 0x80, 0xac,
        0x18, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00,
        0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
        0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07,
        0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0,
        0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x08, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x20, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x30, 0xe4, 0x51, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xa3, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x87, 0x01, 0x02,
        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x02, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x32,
        0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x04, 0x65, 0x40, 0xa7, 0x14, 0x4a, 0xa2, 0x0c, 0x0a, 0x61,
        0x04, 0xa0, 0x08, 0x4a, 0xa0, 0x64, 0xc8, 0x8c, 0x11, 0x80, 0x20, 0x08, 0xa2, 0x61, 0x40, 0x6f, 0xac, 0x2a, 0x49, 0x02, 0x81, 0x40, 0x20, 0x30, 0x00, 0x05, 0x90, 0x00, 0x11, 0x04, 0x00, 0x00,
        0x20, 0x10, 0x08, 0x04, 0x24, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x6e, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06,
        0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x64, 0x82, 0x30, 0x24, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xca, 0x06, 0x61, 0x30, 0x28, 0xa0, 0xcd, 0x4d,
        0x10, 0x86, 0x65, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x04, 0x41, 0x08, 0x36, 0x14, 0xc3, 0xe2, 0x3c, 0xce, 0x04, 0xe1, 0xb9, 0x36,
        0x04, 0x11, 0x0f, 0xb4, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x84, 0xe3, 0x99, 0x20, 0x1c, 0xd0, 0x86, 0x80, 0x98,
        0x20, 0x1c, 0xd1, 0x04, 0xe1, 0x90, 0x36, 0x2c, 0x04, 0x55, 0x59, 0x17, 0x36, 0x60, 0x44, 0x06, 0xb0, 0x18, 0x7a, 0x62, 0x7a, 0x92, 0x9a, 0x20, 0x1c, 0xd3, 0x86, 0x65, 0xd8, 0xaa, 0xec, 0xe2,
        0x06, 0x6b, 0xc8, 0x80, 0x09, 0xc2, 0xc0, 0x90, 0x61, 0x4a, 0x72, 0x2a, 0xfa, 0x22, 0x2a, 0x72, 0x9a, 0x4a, 0xa2, 0xca, 0x9a, 0x20, 0x1c, 0xd4, 0x86, 0xc5, 0xfb, 0xaa, 0xec, 0xe2, 0x06, 0x30,
        0x18, 0x2c, 0x60, 0x82, 0x30, 0x34, 0x5c, 0x98, 0x92, 0x9c, 0x8a, 0xbe, 0x88, 0x8a, 0xa8, 0x82, 0x92, 0x98, 0x36, 0x2c, 0x62, 0x30, 0x06, 0x55, 0x76, 0x71, 0x03, 0x18, 0x78, 0x19, 0xb0, 0x81,
        0xd0, 0xba, 0x30, 0x20, 0x03, 0x36, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x13, 0x84, 0xa3, 0xda, 0x20, 0x10, 0xc3, 0x86, 0x85, 0x30, 0x83, 0xea, 0x0c,
        0xd0, 0x20, 0xf3, 0xc0, 0x80, 0xb0, 0x80, 0x0d, 0xcb, 0xb0, 0x55, 0xd9, 0x95, 0x0d, 0x16, 0x91, 0x01, 0x64, 0xa0, 0xa4, 0x92, 0x9a, 0x92, 0xa8, 0x92, 0xac, 0x8a, 0xbe, 0x92, 0x88, 0x26, 0x08,
        0x87, 0xb3, 0x61, 0xf1, 0xd6, 0x80, 0x0d, 0xb2, 0x2b, 0x1b, 0xc0, 0x60, 0xc8, 0x80, 0x0d, 0x43, 0x1a, 0xa8, 0x41, 0x1b, 0x6c, 0x18, 0xca, 0xa0, 0x0c, 0xdc, 0x60, 0x82, 0x60, 0x58, 0x1b, 0x8e,
        0x65, 0x18, 0x86, 0xc1, 0x83, 0x83, 0x0d, 0x82, 0x18, 0xc4, 0xc1, 0x86, 0xe2, 0x99, 0xde, 0x00, 0x90, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0,
        0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1,
        0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x4a, 0x64, 0x78, 0x2e, 0x74, 0x79, 0x70, 0x65, 0x41, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x53,
        0x82, 0xa7, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94, 0x20, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6,
        0x46, 0x37, 0x37, 0x25, 0x90, 0x03, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c,
        0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a,
        0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde,
        0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60,
        0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f,
        0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc,
        0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3,
        0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x05,
        0x90, 0x0a, 0x81, 0x1f, 0x3c, 0x4c, 0x1f, 0xa3, 0xbf, 0xe3, 0xb7, 0x7b, 0x4e, 0x0f, 0xbb, 0xe9, 0x40, 0x60, 0x36, 0x88, 0xb5, 0xfe, 0x48, 0xcf, 0x3a, 0x1d, 0x5e, 0xa7, 0x43, 0xc3, 0xf4, 0x31,
        0x1a, 0x58, 0x25, 0x87, 0xe9, 0x61, 0x20, 0x90, 0x04, 0x83, 0x01, 0x81, 0xc9, 0x2a, 0x3c, 0x4c, 0x1f, 0xa3, 0xbf, 0xe3, 0xb7, 0x7b, 0x4e, 0x0f, 0xbb, 0xe9, 0x5f, 0x72, 0x98, 0x1e, 0x06, 0x02,
        0x81, 0x5a, 0x1b, 0x98, 0x86, 0xcb, 0x77, 0x1e, 0x7f, 0x71, 0x80, 0x41, 0x6c, 0x1e, 0x6a, 0xf2, 0x8b, 0xdb, 0xb6, 0x03, 0x30, 0x18, 0x2e, 0xdf, 0x79, 0xfc, 0xc5, 0x01, 0x06, 0x94, 0x9a, 0x1e,
        0x6a, 0xe2, 0x9c, 0x66, 0x8a, 0x9c, 0x25, 0x75, 0x90, 0x66, 0xf2, 0x8b, 0xdb, 0x36, 0x81, 0x6a, 0xb8, 0x7c, 0xe7, 0xf1, 0x9f, 0x08, 0x61, 0x90, 0x09, 0xa9, 0x08, 0xb1, 0xf3, 0x91, 0xdb, 0x36,
        0x82, 0x6a, 0xb8, 0x7c, 0xe7, 0xf1, 0xa5, 0xc9, 0x89, 0x08, 0x94, 0x9a, 0x1e, 0x6a, 0xf2, 0x8b, 0xdb, 0xb6, 0x02, 0x6e, 0xb8, 0x7c, 0xe7, 0xf1, 0xa5, 0xc9, 0x89, 0x88, 0x14, 0x98, 0x84, 0x83,
        0x73, 0x1a, 0x69, 0x02, 0x9a, 0xc9, 0x2f, 0x6e, 0xdb, 0x0c, 0xb8, 0xe1, 0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x52, 0x60, 0x12, 0x0e, 0xce, 0x69, 0xa4, 0x09, 0x68, 0x26, 0x1f, 0xb9, 0x6d,
        0x0b, 0x38, 0x0e, 0xe9, 0x67, 0x00, 0xa4, 0x01, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x34,
        0x8a, 0x3a, 0xa0, 0x9c, 0x03, 0x0a, 0x3b, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x05, 0x00, 0x82, 0x60, 0x40, 0x64, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xd5, 0x00, 0x4d, 0xca, 0x34,
        0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x56, 0x50, 0xd4, 0x12, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x62, 0x11, 0x10, 0xb5, 0x50, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x64, 0x46,
        0x65, 0x31, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x57, 0x61, 0x55, 0x4c, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x66, 0x5c, 0xd6, 0x62, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82,
        0x01, 0x92, 0x1d, 0xd8, 0xd5, 0x5c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x70, 0x49, 0x86, 0x3d, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x5c, 0x92, 0x61, 0x8d, 0x30, 0x62, 0x90,
        0x00, 0x20, 0x08, 0x06, 0x0a, 0x97, 0x64, 0x98, 0x13, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xb3, 0x25, 0x15, 0xf6, 0x18, 0x08, 0x00, 0x61, 0x20, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x13,
        0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x34, 0x4a, 0x01, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x58, 0x0f, 0x04, 0x1d, 0xd0, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x20, 0x17, 0x14, 0x45, 0x46, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x16, 0x49, 0x12, 0x22, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x92, 0x49, 0xd3, 0xc4, 0x4c, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0xa0, 0x64, 0x05, 0x45, 0x2d, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x59, 0x41, 0x51, 0xc9, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x4a, 0x56, 0x50, 0x94,
        0x22, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x92, 0x15, 0x14, 0xd5, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x68, 0x53, 0x45, 0x2d, 0xd4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
        0x1b, 0x65, 0x55, 0x4a, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x57, 0x5d, 0x16, 0x63, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xc2, 0x21, 0xd8, 0xe5, 0x0c, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0xa0, 0x70, 0x08, 0x76, 0x31, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x1c, 0x82, 0x5d, 0x4d, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x67, 0x51, 0x97, 0x73, 0x8d,
        0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xd2, 0x25, 0x15, 0xf6, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x78, 0x17, 0x85, 0x3d, 0xd8, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x9e, 0x52,
        0x65, 0x50, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE hs = {
        use_dxil ? (const void*)hs_code_dxil : (const void*)hs_code_dxbc,
        use_dxil ? sizeof(hs_code_dxil) : sizeof(hs_code_dxbc)
    };
    static const DWORD ds_code_dxbc[] =
    {
        0x43425844, 0xc78d05dd, 0x4b270467, 0xb480a2fb, 0x02f0edc7, 0x00000001, 0x0000029c, 0x00000004,
        0x00000030, 0x000000cc, 0x00000160, 0x000001d8, 0x4e475349, 0x00000094, 0x00000004, 0x00000008,
        0x00000068, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x00000074, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000007, 0x0000007a, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x00000008, 0x00000087, 0x00000000, 0x00000000, 0x00000003, 0x00000002, 0x00000001,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0x494c0052, 0x445f454e, 0x49534e45, 0x4c005954,
        0x5f454e49, 0x41544544, 0xab004c49, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000010, 0x00000003, 0x00000000, 0x00000001, 0x00000076, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x00000e0e, 0x00000068, 0x00000001, 0x0000000f, 0x00000003, 0x00000001,
        0x00000001, 0x0000007c, 0x00000000, 0x00000000, 0x00000001, 0x00000001, 0x00000202, 0x545f5653,
        0x46737365, 0x6f746361, 0x4f430072, 0x00524f4c, 0x4d495250, 0x56495449, 0x44495f45, 0xababab00,
        0x4e47534f, 0x00000070, 0x00000003, 0x00000008, 0x00000050, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x0000005c, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x00000062, 0x00000000, 0x00000000, 0x00000003, 0x00000002, 0x0000000f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4f4c4f43, 0x52500052, 0x54494d49, 0x5f455649, 0xab004449, 0x58454853, 0x000000bc,
        0x00040050, 0x0000002f, 0x01000893, 0x01000895, 0x0100086a, 0x0300005f, 0x0011b0e2, 0x00000000,
        0x0300005f, 0x0011b022, 0x00000001, 0x0400005f, 0x002190f2, 0x00000001, 0x00000000, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x03000065, 0x001020f2,
        0x00000002, 0x06000036, 0x001020f2, 0x00000000, 0x00219e46, 0x00000000, 0x00000000, 0x05000036,
        0x00102072, 0x00000001, 0x0011b796, 0x00000000, 0x05000036, 0x00102082, 0x00000001, 0x00004001,
        0x3f800000, 0x05000056, 0x001020f2, 0x00000002, 0x0011b556, 0x00000001, 0x0100003e,
    };
    static const BYTE ds_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xe6, 0xbd, 0xa9, 0xf2, 0x6c, 0xc0, 0xd1, 0xdb, 0x5c, 0x78, 0x77, 0x1a, 0x0c, 0x2a, 0xca, 0xa5, 0x01, 0x00, 0x00, 0x00, 0x2f, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x00, 0x92, 0x01, 0x00, 0x00, 0x43, 0x02, 0x00, 0x00, 0xc7, 0x03, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0xb3, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41,
        0x49, 0x4c, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x87, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x43,
        0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x52, 0x49, 0x4d, 0x49, 0x54, 0x49, 0x56, 0x45, 0x5f, 0x49, 0x44, 0x00, 0x50, 0x53, 0x47, 0x31, 0xa9, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x65, 0x73, 0x73, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x52, 0x49, 0x4d, 0x49, 0x54, 0x49, 0x56, 0x45, 0x5f,
        0x49, 0x44, 0x00, 0x50, 0x53, 0x56, 0x30, 0x7c, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x02, 0x00, 0x04, 0x03, 0x03, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x43, 0x4f, 0x4c, 0x4f,
        0x52, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x4e, 0x53, 0x49, 0x54, 0x59, 0x00, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52,
        0x00, 0x50, 0x52, 0x49, 0x4d, 0x49, 0x54, 0x49, 0x56, 0x45, 0x5f, 0x49, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x52, 0x49, 0x4d, 0x49, 0x54, 0x49, 0x56, 0x45, 0x5f, 0x49, 0x44,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x43, 0x00, 0x03, 0x02, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x01, 0x71, 0x00, 0x03, 0x02, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x41, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x02, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x71, 0x19, 0x03, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04,
        0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x60, 0x06, 0x00, 0x00, 0x60, 0x00, 0x04, 0x00, 0x98, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x48, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x8f, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07,
        0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4,
        0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41,
        0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84,
        0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x12,
        0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10,
        0x40, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0x52, 0x0c, 0x42, 0x48, 0x21,
        0x55, 0x0a, 0x42, 0x08, 0x29, 0xc4, 0x06, 0x02, 0x92, 0x82, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e,
        0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a,
        0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07,
        0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x28, 0x40,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x61, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32,
        0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x04, 0x65, 0x40, 0xa5, 0x24, 0xca, 0xa0, 0x10, 0x46, 0x00,
        0x8a, 0xa0, 0x04, 0x4a, 0xa6, 0x14, 0xe8, 0x8d, 0x45, 0x25, 0x4c, 0x20, 0x10, 0x08, 0x00, 0x00, 0x00, 0x80, 0x40, 0x60, 0x00, 0x0a, 0x00, 0x88, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79,
        0x18, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06,
        0x66, 0x43, 0x10, 0x4c, 0x10, 0x88, 0x62, 0x82, 0x40, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x30, 0x28, 0x90, 0xcd, 0x4d, 0x10, 0x08, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04,
        0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58, 0x9c, 0x09, 0xc2, 0x33, 0x6d, 0x08, 0x20, 0x1e, 0x64, 0x73, 0x5f, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac,
        0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0xc5, 0x32, 0x41, 0x28, 0x98, 0x0d, 0x01, 0x31, 0x41, 0x28, 0x9a, 0x09, 0x42, 0xe1, 0x6c, 0x58, 0x88, 0x89, 0xaa, 0xac, 0x6b,
        0xb8, 0x08, 0x0c, 0x60, 0x31, 0xf4, 0xc4, 0xf4, 0x24, 0x35, 0x41, 0x28, 0x9e, 0x0d, 0xcb, 0xa0, 0x51, 0x98, 0xb5, 0x0d, 0xd5, 0x80, 0x01, 0x13, 0x04, 0x22, 0x21, 0xc3, 0x94, 0xe4, 0x54, 0xf4,
        0x45, 0x54, 0xe4, 0x34, 0x95, 0x44, 0x95, 0x35, 0x41, 0x28, 0xa0, 0x0d, 0x4b, 0xe7, 0x51, 0x98, 0xb5, 0x0d, 0xdf, 0x50, 0x01, 0x13, 0x04, 0x42, 0xe1, 0xc2, 0x94, 0xe4, 0x54, 0xf4, 0x45, 0x54,
        0x44, 0x15, 0x94, 0xc4, 0xb4, 0x61, 0x09, 0x03, 0x31, 0xa0, 0x30, 0x6b, 0x1b, 0xbe, 0x0e, 0x03, 0x36, 0x10, 0x19, 0x07, 0x06, 0x63, 0xb0, 0x61, 0x19, 0x34, 0x0a, 0xb3, 0xb6, 0xe1, 0x1a, 0x30,
        0x80, 0x0c, 0x94, 0x54, 0x52, 0x53, 0x12, 0x55, 0x92, 0x55, 0xd1, 0x57, 0x12, 0xd1, 0x86, 0xa5, 0x33, 0x03, 0x0a, 0xb3, 0xb6, 0xe1, 0xea, 0x30, 0x60, 0xc3, 0x90, 0x95, 0xc1, 0x19, 0xb0, 0x99,
        0xb2, 0xfa, 0xa2, 0x2a, 0x9b, 0x9b, 0x33, 0x0a, 0x1b, 0xa3, 0x7b, 0x93, 0x9b, 0x20, 0x14, 0xd1, 0x06, 0x81, 0x18, 0x36, 0x2c, 0x44, 0x1a, 0x50, 0x6a, 0xb0, 0x06, 0x58, 0xf7, 0x11, 0x15, 0xb0,
        0x61, 0x19, 0x34, 0x0a, 0xb3, 0xb0, 0xa1, 0x22, 0x30, 0x60, 0x82, 0x50, 0x48, 0x1b, 0x96, 0xce, 0x0c, 0xdc, 0x00, 0xb3, 0xb0, 0xe1, 0x1b, 0x30, 0x60, 0xc3, 0xc0, 0x06, 0x6d, 0xf0, 0x06, 0x1b,
        0x06, 0x32, 0x40, 0x03, 0x38, 0xd8, 0x20, 0x0c, 0xc3, 0x06, 0xa1, 0x93, 0x83, 0x0d, 0xc5, 0x22, 0xc5, 0x01, 0x30, 0x07, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2, 0xc8, 0xca, 0xdc, 0xe8, 0xa6,
        0x04, 0x41, 0x15, 0x32, 0x3c, 0x17, 0xbb, 0x32, 0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a, 0x60, 0xd4, 0x21, 0xc3, 0x73,
        0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93, 0x6b, 0x7a, 0x23, 0x2b, 0x63, 0x9b, 0x12, 0x24, 0x95, 0xc8, 0xf0, 0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8, 0xd2, 0xde, 0xdc,
        0xe6, 0xa6, 0x04, 0x4e, 0x1d, 0x32, 0x3c, 0x17, 0xbb, 0xb4, 0xb2, 0xbb, 0x24, 0xb2, 0x29, 0xba, 0x30, 0xba, 0xb2, 0x29, 0x01, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea,
        0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x30, 0x07, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1,
        0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87,
        0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33,
        0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc,
        0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87,
        0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c,
        0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8,
        0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x16,
        0x00, 0x00, 0x00, 0x06, 0x70, 0x0c, 0xd2, 0xcf, 0x00, 0x48, 0x63, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0xb0, 0x0d, 0x97, 0xef,
        0x3c, 0xfe, 0xe2, 0x00, 0x43, 0x0a, 0x4c, 0xc2, 0xc1, 0x39, 0x8d, 0x34, 0x01, 0xcd, 0xe4, 0x17, 0xb7, 0x6d, 0x04, 0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0xa4, 0xc0, 0x24, 0x1c, 0x9c,
        0xd3, 0x48, 0x13, 0xd0, 0x4c, 0x3e, 0x72, 0xdb, 0x26, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52, 0xd3, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x00, 0x61, 0x20, 0x00, 0x00, 0x42,
        0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x44, 0x0a, 0x3a, 0xa0, 0x14, 0x68, 0x8c, 0x11, 0x80, 0x20, 0x08, 0xe2, 0x1f, 0x00, 0x00, 0x00, 0x23,
        0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0x50, 0x83, 0x14, 0x25, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0x54, 0xc4, 0x24, 0x21, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0x58, 0x05, 0x35, 0x29,
        0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xc0, 0x58, 0x46, 0x44, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x68, 0x54, 0x55, 0x35, 0xd5, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x5b, 0x65,
        0x59, 0x8c, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x06, 0x67, 0x5d, 0x97, 0x73, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xd1, 0x5d, 0x18, 0x26, 0x61, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x80, 0x74, 0x4a, 0x96, 0x45, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x9d, 0x92, 0x65, 0xcf, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0xa7, 0x64, 0x19, 0x24, 0x8c, 0x18, 0x24,
        0x00, 0x08, 0x82, 0x01, 0xd2, 0x29, 0x59, 0x36, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x74, 0x8a, 0x96, 0x45, 0xc8, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x9d, 0xa2, 0x65, 0xcf,
        0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0xa7, 0x68, 0x19, 0x64, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xd2, 0x29, 0x5a, 0x36, 0x25, 0x57, 0x8c, 0x1a, 0x31, 0x48, 0x00, 0x10, 0x04,
        0x03, 0xc4, 0x5b, 0x2e, 0x4d, 0x0a, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0x00, 0xf1, 0x96, 0x4b, 0x83, 0x82, 0x11, 0x83, 0x04, 0x00, 0x41, 0x30, 0x40, 0xbc, 0xe5, 0xd2, 0xa2, 0x60, 0xc4, 0x20,
        0x01, 0x40, 0x10, 0x0c, 0x10, 0x6f, 0xb9, 0x34, 0x2a, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE ds = {
        use_dxil ? (const void*)ds_code_dxil : (const void*)ds_code_dxbc,
        use_dxil ? sizeof(ds_code_dxil) : sizeof(ds_code_dxbc)
    };
    static const D3D12_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"SV_POSITION",  0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,  0, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
        {"COLOR",        0, DXGI_FORMAT_R32G32B32_FLOAT,    0, 16, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
        {"LINE_DENSITY", 0, DXGI_FORMAT_R32_FLOAT,          0, 32, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
        {"LINE_DETAIL",  0, DXGI_FORMAT_R32_FLOAT,          0, 36, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
    };
    static const D3D12_SO_DECLARATION_ENTRY so_declaration[] =
    {
        {0, "SV_POSITION",  0, 0, 4, 0},
        {0, "COLOR",        0, 0, 4, 0},
        {0, "PRIMITIVE_ID", 0, 0, 4, 0},
    };
    unsigned int strides[] = {48};
    static const struct
    {
        struct vec4 position;
        struct vec4 color;
        float line_density;
        float line_detail;
    }
    vertices[] =
    {
        {{0.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 1.0f, 0.0f}, 1.0f, 1.0f},
        {{1.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 0.5f, 0.0f}, 2.0f, 1.0f},
        {{2.0f, 0.0f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f}, 1.0f, 2.0f},
        {{3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f}, 2.0f, 2.0f},
    };
    static const struct vec4 expected_data[] =
    {
        {0.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 1.0f, 0.0f, 1.0f}, {0.0f, 0.0f, 0.0f, 0.0f},
        {0.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 1.0f, 0.0f, 1.0f}, {0.0f, 0.0f, 0.0f, 0.0f},

        {1.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 0.5f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f},
        {1.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 0.5f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f},
        {1.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 0.5f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f},
        {1.0f, 0.0f, 0.0f, 1.0f}, {0.0f, 0.5f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f},

        {2.0f, 0.0f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f}, {2.0f, 2.0f, 2.0f, 2.0f},
        {2.0f, 0.0f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f}, {2.0f, 2.0f, 2.0f, 2.0f},
        {2.0f, 0.0f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f}, {2.0f, 2.0f, 2.0f, 2.0f},
        {2.0f, 0.0f, 0.0f, 1.0f}, {1.0f, 1.0f, 1.0f, 1.0f}, {2.0f, 2.0f, 2.0f, 2.0f},

        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
        {3.0f, 0.0f, 0.0f, 1.0f}, {0.5f, 0.5f, 0.5f, 1.0f}, {3.0f, 3.0f, 3.0f, 3.0f},
    };

    memset(&desc, 0, sizeof(desc));
    desc.root_signature_flags = D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT
            | D3D12_ROOT_SIGNATURE_FLAG_ALLOW_STREAM_OUTPUT;
    desc.no_pipeline = true;
    if (!init_test_context(&context, &desc))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    query_heap_desc.Type = D3D12_QUERY_HEAP_TYPE_SO_STATISTICS;
    query_heap_desc.Count = 2;
    query_heap_desc.NodeMask = 0;
    hr = ID3D12Device_CreateQueryHeap(context.device, &query_heap_desc, &IID_ID3D12QueryHeap, (void **)&query_heap);
    if (hr == E_NOTIMPL)
    {
        skip("Stream output is not supported.\n");
        destroy_test_context(&context);
        return;
    }
    ok(hr == S_OK, "Failed to create query heap, hr %#x.\n", hr);

    input_layout.pInputElementDescs = layout_desc;
    input_layout.NumElements = ARRAY_SIZE(layout_desc);

    if (use_dxil)
    {
        init_pipeline_state_desc_dxil(&pso_desc, context.root_signature,
                DXGI_FORMAT_UNKNOWN, NULL, NULL, &input_layout);
    }
    else
    {
        init_pipeline_state_desc(&pso_desc, context.root_signature,
                DXGI_FORMAT_UNKNOWN, NULL, NULL, &input_layout);
    }
    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    memset(&pso_desc.PS, 0, sizeof(pso_desc.PS));
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    pso_desc.StreamOutput.NumEntries = ARRAY_SIZE(so_declaration);
    pso_desc.StreamOutput.pSODeclaration = so_declaration;
    pso_desc.StreamOutput.pBufferStrides = strides;
    pso_desc.StreamOutput.NumStrides = ARRAY_SIZE(strides);
    pso_desc.StreamOutput.RasterizedStream = D3D12_SO_NO_RASTERIZED_STREAM;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    vb = create_upload_buffer(context.device, sizeof(vertices), vertices);
    vbv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(vb);
    vbv.StrideInBytes = sizeof(*vertices);
    vbv.SizeInBytes = sizeof(vertices);

    so_buffer = create_default_buffer(context.device, 4096,
            D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_STREAM_OUT);
    sobv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(so_buffer);
    sobv.SizeInBytes = 1024;
    sobv.BufferFilledSizeLocation = sobv.BufferLocation + sobv.SizeInBytes;

    ID3D12GraphicsCommandList_BeginQuery(command_list, query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 0);

    ID3D12GraphicsCommandList_SOSetTargets(command_list, 0, 1, &sobv);

    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_1_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_IASetVertexBuffers(command_list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 4, 1, 0, 0);

    ID3D12GraphicsCommandList_EndQuery(command_list, query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 0);

    readback_buffer = create_readback_buffer(context.device, sizeof(*so_statistics));
    ID3D12GraphicsCommandList_ResolveQueryData(command_list,
            query_heap, D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0, 0, 1, readback_buffer, 0);

    get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);
    so_statistics = get_readback_data(&rb, 0, 0, 0, sizeof(*so_statistics));
    broken_warp = broken_on_warp(so_statistics[0].NumPrimitivesWritten != 9);
    ok(so_statistics[0].NumPrimitivesWritten == 9 || broken_warp, "Got unexpected primitives written %u.\n",
            (unsigned int)so_statistics[0].NumPrimitivesWritten);
    ok(so_statistics[0].PrimitivesStorageNeeded == 9 || broken_warp, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)so_statistics[0].PrimitivesStorageNeeded);
    release_resource_readback(&rb);

    if (broken_warp)
    {
        skip("Broken on WARP.\n");
        goto done;
    }

    reset_command_list(command_list, context.allocator);
    transition_resource_state(command_list, so_buffer,
            D3D12_RESOURCE_STATE_STREAM_OUT, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(so_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);
    for (i = 0; i < ARRAY_SIZE(expected_data) / 3 ; ++i)
    {
        data = get_readback_data(&rb, i, 0, 0, 3 * sizeof(*data));
        expected = &expected_data[3 * i + 0];
        ok(compare_vec4(data, expected, 1),
                "Got position {%.8e, %.8e, %.8e, %.8e}, expected {%.8e, %.8e, %.8e, %.8e} at %u.\n",
                data->x, data->y, data->z, data->w, expected->x, expected->y, expected->z, expected->w, i);
        ++data;
        expected = &expected_data[3 * i + 1];
        ok(compare_vec4(data, expected, 1),
                "Got color {%.8e, %.8e, %.8e, %.8e}, expected {%.8e, %.8e, %.8e, %.8e} at %u.\n",
                data->x, data->y, data->z, data->w, expected->x, expected->y, expected->z, expected->w, i);
        ++data;
        expected = &expected_data[3 * i + 2];
        ok(compare_vec4(data, expected, 1),
                "Got primitive ID {%.8e, %.8e, %.8e, %.8e}, expected {%.8e, %.8e, %.8e, %.8e} at %u.\n",
                data->x, data->y, data->z, data->w, expected->x, expected->y, expected->z, expected->w, i);
    }
    release_resource_readback(&rb);

done:
    ID3D12QueryHeap_Release(query_heap);
    ID3D12Resource_Release(readback_buffer);
    ID3D12Resource_Release(so_buffer);
    ID3D12Resource_Release(vb);
    destroy_test_context(&context);
}

void test_line_tessellation_dxbc(void)
{
    test_line_tessellation(false);
}

void test_line_tessellation_dxil(void)
{
    test_line_tessellation(true);
}

void test_tessellation_primitive_id(void)
{
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    D3D12_ROOT_PARAMETER root_parameters[1];
    ID3D12GraphicsCommandList *command_list;
    D3D12_INPUT_LAYOUT_DESC input_layout;
    struct test_context_desc desc;
    D3D12_VERTEX_BUFFER_VIEW vbv;
    struct test_context context;
    ID3D12Resource *raw_buffer;
    ID3D12CommandQueue *queue;
    ID3D12Resource *vb;
    HRESULT hr;

#if 0
    ByteAddressBuffer b;

    struct data
    {
        float4 position : POSITION;
        float ref_buffer_data : REF_BUFFER_DATA;
    };

    struct ds_data
    {
        float4 position : POSITION;
        float ref_buffer_data : REF_BUFFER_DATA;
        uint primitive_id : PRIM_ID;
        uint invocation_id : CP_ID;
    };

    struct ps_data
    {
        float4 position : SV_POSITION;
        float4 color : COLOR;
    };

    struct patch_constant_data
    {
        float edges[3] : SV_TessFactor;
        float inside : SV_InsideTessFactor;
        float buffer_data : BUFFER_DATA;
        uint primitive_id : PATCH_PRIM_ID;
    };

    data vs_main(in data input)
    {
        return input;
    }

    void patch_constant(uint prim_id : SV_PrimitiveID, out patch_constant_data output)
    {
        output.edges[0] = output.edges[1] = output.edges[2] = 4.0f;
        output.inside = 4.0f;
        output.buffer_data = b.Load(4 * prim_id);
        output.primitive_id = prim_id;
    }

    [domain("tri")]
    [outputcontrolpoints(3)]
    [partitioning("integer")]
    [outputtopology("triangle_cw")]
    [patchconstantfunc("patch_constant")]
    ds_data hs_main(const InputPatch<data, 3> input,
            uint prim_id : SV_PrimitiveID, uint i : SV_OutputControlPointID)
    {
        ds_data output;
        output.position = input[i].position;
        output.ref_buffer_data = input[i].ref_buffer_data;
        output.primitive_id = prim_id;
        output.invocation_id = i;
        return output;
    }

    [domain("tri")]
    void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<ds_data, 3> patch,
            out ps_data output)
    {
        uint i;

        output.position = tess_coord.x * patch[0].position
                + tess_coord.y * patch[1].position
                + tess_coord.z * patch[2].position;

        for (i = 0; i < 3; ++i)
        {
            if (patch[i].ref_buffer_data != input.buffer_data)
            {
                output.color = float4(1, patch[i].ref_buffer_data / 255.0f, input.buffer_data / 255.0f, 0);
                return;
            }
        }

        for (i = 0; i < 3; ++i)
        {
            if (patch[i].primitive_id != input.primitive_id)
            {
                output.color = float4(1, 0, 1, 1);
                return;
            }
        }

        if (patch[0].invocation_id != 0 || patch[1].invocation_id != 1 || patch[2].invocation_id != 2)
        {
            output.color = float4(1, 1, 0, 1);
            return;
        }

        output.color = float4(0, 1, 0, 1);
    }
#endif
    static const DWORD vs_code[] =
    {
        0x43425844, 0x1cf34a89, 0x09f0ca02, 0x1d9d7e25, 0x4161cddd, 0x00000001, 0x00000154, 0x00000003,
        0x0000002c, 0x00000088, 0x000000e4, 0x4e475349, 0x00000054, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000101, 0x49534f50, 0x4e4f4954, 0x46455200, 0x4655425f, 0x5f524546,
        0x41544144, 0xababab00, 0x4e47534f, 0x00000054, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000041, 0x00000000, 0x00000000, 0x00000003,
        0x00000001, 0x00000e01, 0x49534f50, 0x4e4f4954, 0x46455200, 0x4655425f, 0x5f524546, 0x41544144,
        0xababab00, 0x58454853, 0x00000068, 0x00010050, 0x0000001a, 0x0100086a, 0x0300005f, 0x001010f2,
        0x00000000, 0x0300005f, 0x00101012, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x03000065,
        0x00102012, 0x00000001, 0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x05000036,
        0x00102012, 0x00000001, 0x0010100a, 0x00000001, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE vs = {vs_code, sizeof(vs_code)};
    static const DWORD hs_code[] =
    {
        0x43425844, 0x23a919a7, 0xf5fdd1b4, 0x4f5a835f, 0xca389c71, 0x00000001, 0x00000464, 0x00000004,
        0x00000030, 0x0000008c, 0x00000124, 0x00000200, 0x4e475349, 0x00000054, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000101, 0x49534f50, 0x4e4f4954, 0x46455200, 0x4655425f,
        0x5f524546, 0x41544144, 0xababab00, 0x4e47534f, 0x00000090, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000071, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000e01, 0x00000081, 0x00000000, 0x00000000, 0x00000001, 0x00000002,
        0x00000e01, 0x00000089, 0x00000000, 0x00000000, 0x00000001, 0x00000002, 0x00000d02, 0x49534f50,
        0x4e4f4954, 0x46455200, 0x4655425f, 0x5f524546, 0x41544144, 0x49525000, 0x44495f4d, 0x5f504300,
        0xab004449, 0x47534350, 0x000000d4, 0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x0000000d,
        0x00000003, 0x00000000, 0x00000e01, 0x000000a6, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x00000d02, 0x00000098, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x000000b2,
        0x00000000, 0x00000000, 0x00000001, 0x00000001, 0x00000d02, 0x00000098, 0x00000002, 0x0000000d,
        0x00000003, 0x00000002, 0x00000e01, 0x000000c0, 0x00000000, 0x0000000e, 0x00000003, 0x00000003,
        0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x55420072, 0x52454646, 0x5441445f, 0x41500041,
        0x5f484354, 0x4d495250, 0x0044495f, 0x495f5653, 0x6469736e, 0x73655465, 0x63614673, 0x00726f74,
        0x58454853, 0x0000025c, 0x00030050, 0x00000097, 0x01000071, 0x01001893, 0x01001894, 0x01001095,
        0x01000896, 0x01001897, 0x0100086a, 0x030000a1, 0x00107000, 0x00000000, 0x01000072, 0x0200005f,
        0x00016000, 0x0200005f, 0x0000b000, 0x0400005f, 0x002010f2, 0x00000003, 0x00000000, 0x0400005f,
        0x00201012, 0x00000003, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012,
        0x00000001, 0x03000065, 0x00102012, 0x00000002, 0x03000065, 0x00102022, 0x00000002, 0x02000068,
        0x00000001, 0x04000036, 0x00100012, 0x00000000, 0x00016001, 0x07000036, 0x001020f2, 0x00000000,
        0x00a01e46, 0x0010000a, 0x00000000, 0x00000000, 0x07000036, 0x00102012, 0x00000001, 0x00a0100a,
        0x0010000a, 0x00000000, 0x00000001, 0x04000036, 0x00102012, 0x00000002, 0x0000b001, 0x04000036,
        0x00102022, 0x00000002, 0x00016001, 0x0100003e, 0x01000073, 0x02000099, 0x00000003, 0x0200005f,
        0x00017000, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x04000067, 0x00102012, 0x00000001,
        0x00000012, 0x04000067, 0x00102012, 0x00000002, 0x00000013, 0x02000068, 0x00000001, 0x0400005b,
        0x00102012, 0x00000000, 0x00000003, 0x04000036, 0x00100012, 0x00000000, 0x0001700a, 0x06000036,
        0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x40800000, 0x0100003e, 0x01000073, 0x04000067,
        0x00102012, 0x00000003, 0x00000014, 0x05000036, 0x00102012, 0x00000003, 0x00004001, 0x40800000,
        0x0100003e, 0x01000073, 0x0200005f, 0x0000b000, 0x03000065, 0x00102022, 0x00000000, 0x02000068,
        0x00000001, 0x06000029, 0x00100012, 0x00000000, 0x0000b001, 0x00004001, 0x00000002, 0x890000a5,
        0x800002c2, 0x00199983, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x00107006, 0x00000000,
        0x05000056, 0x00102022, 0x00000000, 0x0010000a, 0x00000000, 0x0100003e, 0x01000073, 0x0200005f,
        0x0000b000, 0x03000065, 0x00102022, 0x00000001, 0x04000036, 0x00102022, 0x00000001, 0x0000b001,
        0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE hs = {hs_code, sizeof(hs_code)};
    static const DWORD ds_code[] =
    {
        0x43425844, 0x4b659bb4, 0x3eb61f16, 0xaa8397b6, 0xd60e8c43, 0x00000001, 0x000005ec, 0x00000004,
        0x00000030, 0x000000c8, 0x000001a4, 0x000001f8, 0x4e475349, 0x00000090, 0x00000004, 0x00000008,
        0x00000068, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000071, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000101, 0x00000081, 0x00000000, 0x00000000, 0x00000001,
        0x00000002, 0x00000101, 0x00000089, 0x00000000, 0x00000000, 0x00000001, 0x00000002, 0x00000202,
        0x49534f50, 0x4e4f4954, 0x46455200, 0x4655425f, 0x5f524546, 0x41544144, 0x49525000, 0x44495f4d,
        0x5f504300, 0xab004449, 0x47534350, 0x000000d4, 0x00000006, 0x00000008, 0x00000098, 0x00000000,
        0x0000000d, 0x00000003, 0x00000000, 0x00000001, 0x000000a6, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x00000202, 0x00000098, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000001,
        0x000000b2, 0x00000000, 0x00000000, 0x00000001, 0x00000001, 0x00000202, 0x00000098, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000001, 0x000000c0, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000001, 0x545f5653, 0x46737365, 0x6f746361, 0x55420072, 0x52454646, 0x5441445f,
        0x41500041, 0x5f484354, 0x4d495250, 0x0044495f, 0x495f5653, 0x6469736e, 0x73655465, 0x63614673,
        0x00726f74, 0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001,
        0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052, 0x58454853, 0x000003ec,
        0x00040050, 0x000000fb, 0x01001893, 0x01001095, 0x0100086a, 0x0300005f, 0x0011b022, 0x00000000,
        0x0300005f, 0x0011b022, 0x00000001, 0x0200005f, 0x0001c072, 0x0400005f, 0x002190f2, 0x00000003,
        0x00000000, 0x0400005f, 0x00219012, 0x00000003, 0x00000001, 0x0400005f, 0x00219012, 0x00000003,
        0x00000002, 0x0400005f, 0x00219022, 0x00000003, 0x00000002, 0x04000067, 0x001020f2, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x02000068, 0x00000002, 0x07000038, 0x001000f2,
        0x00000000, 0x0001c556, 0x00219e46, 0x00000001, 0x00000000, 0x09000032, 0x001000f2, 0x00000000,
        0x0001c006, 0x00219e46, 0x00000000, 0x00000000, 0x00100e46, 0x00000000, 0x09000032, 0x001000f2,
        0x00000000, 0x0001caa6, 0x00219e46, 0x00000002, 0x00000000, 0x00100e46, 0x00000000, 0x05000036,
        0x00100012, 0x00000001, 0x00004001, 0x00000000, 0x01000030, 0x07000050, 0x00100022, 0x00000001,
        0x0010000a, 0x00000001, 0x00004001, 0x00000003, 0x03040003, 0x0010001a, 0x00000001, 0x09000039,
        0x00100022, 0x00000001, 0x0011b01a, 0x00000000, 0x00a1900a, 0x0010000a, 0x00000001, 0x00000001,
        0x0304001f, 0x0010001a, 0x00000001, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000,
        0x08000036, 0x00102092, 0x00000001, 0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x00000000,
        0x09000038, 0x00102022, 0x00000001, 0x00004001, 0x3b808081, 0x00a1900a, 0x0010000a, 0x00000001,
        0x00000001, 0x07000038, 0x00102042, 0x00000001, 0x0011b01a, 0x00000000, 0x00004001, 0x3b808081,
        0x0100003e, 0x01000015, 0x0700001e, 0x00100012, 0x00000001, 0x0010000a, 0x00000001, 0x00004001,
        0x00000001, 0x01000016, 0x05000036, 0x00100012, 0x00000001, 0x00004001, 0x00000000, 0x01000030,
        0x07000050, 0x00100022, 0x00000001, 0x0010000a, 0x00000001, 0x00004001, 0x00000003, 0x03040003,
        0x0010001a, 0x00000001, 0x09000027, 0x00100022, 0x00000001, 0x0011b01a, 0x00000001, 0x00a1900a,
        0x0010000a, 0x00000001, 0x00000002, 0x0304001f, 0x0010001a, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00100e46, 0x00000000, 0x08000036, 0x001020f2, 0x00000001, 0x00004002, 0x3f800000,
        0x00000000, 0x3f800000, 0x3f800000, 0x0100003e, 0x01000015, 0x0700001e, 0x00100012, 0x00000001,
        0x0010000a, 0x00000001, 0x00004001, 0x00000001, 0x01000016, 0x08000027, 0x00100012, 0x00000001,
        0x00004001, 0x00000000, 0x0021901a, 0x00000000, 0x00000002, 0x08000027, 0x00100022, 0x00000001,
        0x00004001, 0x00000001, 0x0021901a, 0x00000001, 0x00000002, 0x0700003c, 0x00100012, 0x00000001,
        0x0010001a, 0x00000001, 0x0010000a, 0x00000001, 0x08000027, 0x00100022, 0x00000001, 0x00004001,
        0x00000002, 0x0021901a, 0x00000002, 0x00000002, 0x0700003c, 0x00100012, 0x00000001, 0x0010001a,
        0x00000001, 0x0010000a, 0x00000001, 0x0304001f, 0x0010000a, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00100e46, 0x00000000, 0x08000036, 0x001020f2, 0x00000001, 0x00004002, 0x3f800000,
        0x3f800000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000015, 0x05000036, 0x001020f2, 0x00000000,
        0x00100e46, 0x00000000, 0x08000036, 0x001020f2, 0x00000001, 0x00004002, 0x00000000, 0x3f800000,
        0x00000000, 0x3f800000, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ds = {ds_code, sizeof(ds_code)};
    static const DWORD ps_code[] =
    {
#if 0
        float4 main(in float4 p : SV_Position, in float4 color : COLOR) : SV_Target
        {
            return color;
        }
#endif
        0x43425844, 0xbd83f517, 0x8974e87a, 0xaf402223, 0xaec7f351, 0x00000001, 0x000000f8, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000003c, 0x00000050,
        0x0000000f, 0x0100086a, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const D3D12_SHADER_BYTECODE ps = {ps_code, sizeof(ps_code)};
    static const D3D12_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION",        0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,  0, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
        {"REF_BUFFER_DATA", 0, DXGI_FORMAT_R32_FLOAT,          0, 16, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0},
    };
    static const struct
    {
        struct vec4 position;
        float ref_buffer_data;
    }
    vertices[] =
    {
        {{-1.0f, -1.0f, 0.0f, 1.0f}, 1.0f},
        {{-1.0f,  1.0f, 0.0f, 1.0f}, 1.0f},
        {{ 1.0f, -1.0f, 0.0f, 1.0f}, 1.0f},

        {{-1.0f,  1.0f, 0.0f, 1.0f}, 2.0f},
        {{ 1.0f,  1.0f, 0.0f, 1.0f}, 2.0f},
        {{ 1.0f, -1.0f, 0.0f, 1.0f}, 2.0f},
    };
    static const uint32_t buffer_data[] = {1, 2};

    memset(&desc, 0, sizeof(desc));
    desc.no_root_signature = true;
    if (!init_test_context(&context, &desc))
        return;
    command_list = context.list;
    queue = context.queue;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].Descriptor.ShaderRegister = 0;
    root_parameters[0].Descriptor.RegisterSpace = 0;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_HULL;
    memset(&root_signature_desc, 0, sizeof(root_signature_desc));
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    root_signature_desc.pParameters = root_parameters;
    root_signature_desc.Flags = D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT;
    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(hr == S_OK, "Failed to create root signature, hr %#x.\n", hr);

    input_layout.pInputElementDescs = layout_desc;
    input_layout.NumElements = ARRAY_SIZE(layout_desc);
    init_pipeline_state_desc(&pso_desc, context.root_signature,
            context.render_target_desc.Format, NULL, NULL, &input_layout);
    pso_desc.VS = vs;
    pso_desc.HS = hs;
    pso_desc.DS = ds;
    pso_desc.PS = ps;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH;
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    vb = create_upload_buffer(context.device, sizeof(vertices), vertices);
    vbv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(vb);
    vbv.StrideInBytes = sizeof(*vertices);
    vbv.SizeInBytes = sizeof(vertices);

    raw_buffer = create_upload_buffer(context.device, sizeof(buffer_data), buffer_data);

    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, white, 0, NULL);

    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_3_CONTROL_POINT_PATCHLIST);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_IASetVertexBuffers(command_list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_SetGraphicsRootShaderResourceView(command_list,
            0, ID3D12Resource_GetGPUVirtualAddress(raw_buffer));
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 6, 1, 0, 0);

    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    check_sub_resource_uint(context.render_target, 0, queue, command_list, 0xff00ff00, 0);

    ID3D12Resource_Release(vb);
    ID3D12Resource_Release(raw_buffer);
    destroy_test_context(&context);
}

