#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "uostr.h"
#include "uoio.h"
#include "uogetopt.h"
#include "smtp.h"
#include "smtptools.h"
#include "attribs.h"
#include "dns.h"

extern void _exit P__((int)) UO_ATTRIB_NORET;

int logmsgfd=2;

char *myhostname=0;
uoio_t *reply_log_chan=NULL;

/* user changeable options */
char *greeting=NULL;
unsigned long smtpport=25;
unsigned long connect_timeout=90;
unsigned long read_timeout=1200;
unsigned long write_timeout=1200;
static char *from_address;
static char *to_address;
static const char *relayhost=NULL; /* "localhost"; */
static const char *inputfname=NULL;
static long reply_log_fd;
static const char *reply_log_file;

static struct uogetopt myopts[]={
	{'C',"connect-timeout", UOGO_ULONG,&connect_timeout,0, 
		"timeout for SMTP connect", "TIMEOUT"},
	{'f',"from", UOGO_STRING,&from_address,0, "sender address\nmandatory, but may be empty", "SENDER"},
	{'g',"greeting", UOGO_STRING,&greeting,0, "name to use for HELO","MYHOSTNAME"},
	{'i',"input", UOGO_STRING,&inputfname,0, "file to read message from (instead of stdin)","FILENAME"},
	{'p',"port", UOGO_ULONG,&smtpport,0, "TCP port of the SMTP server","PORT"},
	{'R',"read-timeout", UOGO_ULONG,&read_timeout,0, 
		"timeout for reading from remote", "TIMEOUT"},
	{'r',"relayhost", UOGO_STRING,&relayhost,0, 
		"host name of SMTP server to use\nmake sure you are allowed to use it!","RELAYHOST"},
	{  0,"reply-log-file", UOGO_STRING,&reply_log_file,0, "file to log SMTP replies into","SMTPLOGFILE"},
	{  0,"reply-log-fd", UOGO_LONG,&reply_log_fd,0, "file descriptor to log SMTP replies into","SMTPLOGFD"},
	{'t',"to",   UOGO_STRING,&to_address,0,
		"recipient address\nmandatory, should be a full qualified address\n","RECIPIENT"},
	{'W',"write-timeout", UOGO_ULONG,&write_timeout,0, 
		"timeout for writing to remote", "TIMEOUT"},
	{0,0}
};

static void exitmsguo P__((int exitcode,uostr_t *s)) UO_ATTRIB_NORET;
static void exitmsg0 P__((int exitcode,const char *s)) UO_ATTRIB_NORET;

static int 
translate_exitcode(int exitcode)
{
	switch(exitcode) {
	case SMTP_SUCCESS: return (0);
	case SMTP_DEFER: return (111);
	case SMTP_FATAL: return (100);
	default: return (111); /* should never arrive here */
	}
}

static void
exitmsguo(int exitcode,uostr_t *s)
{
	logmsguo(exitcode,s);
	_exit(translate_exitcode(exitcode));
}

static void
exitmsg0(int exitcode,const char *s)
{
	uostr_t us;
	union { const char *co; char *nco; } u;
	u.co=s;
	us.data=u.nco;
	us.len=strlen(s);
	exitmsguo(exitcode,&us);
}

int 
main(int argc, char **argv)
{
	char buf[128];
	int status;
	uoio_t rop,wop;
	int ifd;
	const char *argv0;
	uostr_t logstr;
	logstr.data=0;
	argv0=strrchr(argv[0],'/'); if (!argv0) argv0=argv[0];

	if (gethostname(buf,sizeof(buf)-1)==-1) {
		exitmsg0(1,"hostname too long"); /* unsuited for mail, anyway */
	} else {
		buf[sizeof(buf)-1]=0;
		myhostname=malloc(strlen(buf)+1);
		if (!myhostname) exitmsg0(1,"out of memory");
		memcpy(myhostname,buf,strlen(buf)+1);
	}
	greeting=myhostname;

	uogetopt("smtpblast",PACKAGE,VERSION,&argc,argv,uogetopt_out,0,myopts,0);

	if (reply_log_file || reply_log_fd) {
		struct stat st;
		if (reply_log_file) {
			reply_log_fd=open(reply_log_file,O_WRONLY|O_CREAT|O_APPEND,0600);
			if (reply_log_fd==-1) {
				logmsg5(SMTP_DEFER,argv0,": cannot open ",reply_log_file, ": ",strerror(errno));
				_exit(translate_exitcode(SMTP_DEFER));
			}
		} else if (-1==fstat(reply_log_fd,&st)) {
			logmsg3(SMTP_FATAL,argv0,": reply-log-fd: ",strerror(errno));
			_exit(translate_exitcode(SMTP_DEFER));
		} 
		reply_log_chan=malloc(sizeof(uoio_t));
		if (!reply_log_chan) exitmsg0(SMTP_DEFER,"out of memory");
		uoio_assign_w(reply_log_chan,reply_log_fd,write,0);
	}
	if (smtpport>65535) exitmsg0(SMTP_FATAL,"SMTP port too large\n");
	/* need target host, from and to */
	if (relayhost && !*relayhost) exitmsg0(SMTP_FATAL,"relay host is empty\n");
	if (!from_address) exitmsg0(SMTP_FATAL,"need from address\n");
	if (!to_address) exitmsg0(SMTP_FATAL,"need to address\n");
	if (!*to_address) exitmsg0(SMTP_FATAL,"target address is empty\n");
	if (!inputfname)
		ifd=0;
	else {
		ifd=open(inputfname,O_RDONLY);
		if (ifd==-1) {
			logmsg4(SMTP_FATAL,"cannot open ",inputfname,":",strerror(errno));
			_exit(translate_exitcode(SMTP_FATAL));
		}
	}
	if (relayhost) {
	  no_mx:
		status=smtp_open("@",relayhost,smtpport,greeting,&rop,&wop,&logstr);
		if (status!=SMTP_SUCCESS) {
			logmsguo(status,&logstr);
			_exit(translate_exitcode(SMTP_DEFER)); /* relay, not recipient! */
		}
	} else {
		const char *d;
		dns_t *mx=NULL;
		int ret;
		d=strchr(to_address,'@');
		if (!d||!d[1]) exitmsg0(SMTP_FATAL,"target address has no domain part\n");
		d++;
		ret=dns_mx(d,&mx);
		if (ret==DNS_HARD) {relayhost=d; goto no_mx;}
		if (ret==DNS_SOFT) {logmsg2(SMTP_DEFER,d,": soft DNS error");_exit(translate_exitcode(SMTP_DEFER));}
		if (!mx) { relayhost=d; goto no_mx; /* should never need this! */ }
		while (mx) {
			status=smtp_open("@",mx->name,smtpport,greeting,&rop,&wop,&logstr);
			if (status==SMTP_SUCCESS) break;
			logmsguo(SMTP_DEFER,&logstr);
			mx=mx->next;
		}
		if (status!=SMTP_SUCCESS) _exit(translate_exitcode(status));
	}
	status=smtp_send("@",relayhost,&rop,&wop,from_address,to_address,ifd,&logstr);
	if (!(status & SMTP_CLOSED)) {
		smtp_close("@",relayhost,&rop,&wop);
	} else {
		status &= ~(SMTP_CLOSED);
	}
	if (status==SMTP_SUCCESS) {
		if (inputfname) {
			if (-1==unlink(inputfname)) {
				/* gna ... */
				uostr_xadd_cstrmulti(&logstr,", but cannot unlink ",inputfname,": ",strerror(errno),NULL);
				/* we delivered it ... */
			}
		}
	}
	logmsguo(status,&logstr);
	_exit(translate_exitcode(status));
}
