/*
filter_subtitler.c

Copyright (C) Jan Panteltje  2002

Font reading etc from Linux mplayer

subtitler-yuv is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
  
subtitler-yuv is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
  
You should have received a copy of the GNU General Public License
along with GNU Make; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
*/

#include "subtitler.h"

/* for YUV to RGB in X11 */
#define LIMIT(x) ((((x)>0xffffff)?0xff0000:(((x)<=0xffff)?0:(x)&0xff0000))>>16)
int write_ppm_flag;

int dcontrast, brightness;
double dsaturation;
double dhue, dhue_line_drift;
int u_shift, v_shift;
int slice_level;

int add_objects_flag;

int width, height;
unsigned char buffer[720*576*4];
int reverse_uv_flag;


int main(int argc, char **argv)
{
int a, x;
double da, db;
char *running;
char *token;
static int y, b;
uint8_t *py, *pu, *pv;
static int cr, cg, cb, cy, cu, cv;
static int have_bottom_margin_flag;
int frame_rate;
char interlace_mode;
double aspect_ratio = 0;
int length, ylength, ulength, vlength;
FILE *outfile;
char *bptr;
size_t sb;

/* identify */
fprintf(stderr, "\nPanteltje (c) movie composer%s (alias subtitler-yuv)\n\n",\
SUBTITLER_VERSION);

/* get home directory */  
userinfo = getpwuid(getuid() );
home_dir = strsave(userinfo -> pw_dir);
user_name = strsave(userinfo -> pw_name);

/* set some defaults */

/* for stdin to stdout */
outfile = stdout;

/* for subtitles */

/* a frame in transcode is 40 ms */
frame_offset = 0;

/* this selects some other symbols it seems */
default_font = 0;	// 1 = strange symbols like stars etc..

/* this sets the font outline */
default_font_factor = 10.75;	// outline, was .75

/* this sets the horizontal space for the subtitles */
subtitle_h_factor = SUBTITLE_H_FACTOR;

/* this sets how far the subtitles start from the bottom */
subtitle_v_factor = SUBTITLE_V_FACTOR;

/*
the mplayer font seems to overlap getween 'l' and 't', added some
extra space between characters
*/
extra_character_space = EXTRA_CHAR_SPACE;
subtitle_extra_character_space = EXTRA_CHAR_SPACE;

/* vertical space betweem lines in multiline subtitles */
subtitle_vespace = 0;

/* for picture adjust */
brightness = 0;			// steps
dcontrast = 100.0;		// percent
dsaturation = 100.0;	// percent
u_shift = 0;			// steps
v_shift = 0;			// steps

/* for color correction */
dhue = 0.0;
dhue_line_drift = 0.0;	// The rotation in degrees at the start and end
						// of a line.
						// This is for correcting color errors in NTSC
						// tapes.
						// Use in combination with dhue in color
						// correction.


/* for show output in X11 */ 
window_open_flag = 0;
color_depth = 0; /* get from X */

/* module settings */
add_objects_flag = 1;
show_output_flag = 0;
center_flag = 1;

/* for chroma key, better do this not each pixel .. */
dmax_vector = sqrt( (127.0 * 127.0) + (127.0 * 127.0) );

/*
for rotate and shear, the level we cut the remaining parts.
Note that yuv_to_ppm() also uses this, to write a modified .ppm
that does NOT have this luminance in it, for use by mogrify.
This ONLY happens if rotate or shear present.
*/

default_border_luminance = LUMINANCE_MASK;

/* seems mjpegtools writes v first */
reverse_uv_flag = 1;

test_mode = 0;	
/* end defaults */

/* get command line options, these override the defaults */
while(1)
	{
	a = getopt(argc, argv, "c:dho:p:rt");
	if(a == -1) break;
	switch(a)
		{
		case 'c':
			color_depth = atoi(optarg);
			break;
        case 'd':
			debug_flag = 1;
			break;
		case 'h':
			fprintf(stderr,\
"Usage:\n\
subtitler-yuv [-c colordepth] [-d] [-h] [-o frame_offset] \n\
-p filename.ppml [-r] [-t]\n\
\n\
\t-c  X windows colordepth (bits per pixel), used in show output, try 32!.\n\
\t-d  print debug messages (o.a. function calls and arguments).\n\
\t-h  print command line options (this help), and exit.\n\
\t-o  frame offset.\n\
\t-p  filename.ppml the file with the subtitles and commands.\n\
\t-r  reverses u and v, use when colors are reversed, default is vu.\n\
\t-t  test mode, does a fast run first to check for syntax errors in the ppml file, then exits.\n\
");
			exit(0);	
			break;
		case 'o':
			frame_offset = atoi(optarg);
			break;
		case 'p':
			subtitle_file = strsave(optarg);
			if(! subtitle_file)
				{
				fprintf(stderr, "malloc failed ,aborting\n");
				exit(0);
				}			
			break;
		case 'r':
			reverse_uv_flag = 1 - reverse_uv_flag;
			break;
		case 't':
			test_mode = 1;
			break;
		case '?':
			if (isprint(optopt) )
 				{
 				fprintf(stderr, "Unknown option `-%c'.\n", optopt);
 				}
			else
				{
				fprintf(stderr,\
				"Unknown option character `\\x%x', try -h\n.",\
				optopt);
				}
			exit(1);
			break;			
		default:
			break;
		}/* end switch a */
	}/* end while getopt() */

if(debug_flag)
	{		
	fprintf(stderr,\
	"subtitler(): PARSER RESULT\n\
	color_depth=%d frame_offset=%d subtitle_file=%s\n",\
	color_depth, frame_offset, subtitle_file);
	}

if(frame_offset)
	{
	fprintf(stderr,\
	"subtitler-yuv(): using frame_offset=%d\n", frame_offset);
	}

if(add_objects_flag)
	{
	/* load ppml file */
	if(! load_ppml_file(subtitle_file) )
		{
		fprintf(stderr, "subtitler(): could not load ppml file %s\n",\
		subtitle_file);

		/* return init error */
		exit(1);
		}

	} /* end if add_objects_flag */

/* read yuv stream from stdin */
/*
From mjpegtools:
Description of the (new!, forever?) YUV4MPEG2 stream format:

STREAM consists of
 o one '\n' terminated STREAM-HEADER
 o unlimited number of FRAMEs

FRAME consists of
 o one '\n' terminated FRAME-HEADER
 o "length" octets of planar YCrCb 4:2:0 image data
   (if frame is interlaced, then the two fields are interleaved)

STREAM-HEADER consists of
 o string "YUV4MPEG2 "  (note the space after the '2')
 o unlimited number of ' ' separated TAGGED-FIELDs
 o '\n' line terminator

FRAME-HEADER consists of
 o string "FRAME "  (note the space after the 'E')
 o unlimited number of ' ' separated TAGGED-FIELDs
 o '\n' line terminator

TAGGED-FIELD consists of
 o single ascii character tag
 o VALUE (which does not contain whitespace)

VALUE consists of
 o integer (base 10 ascii representation)
 or o RATIO
 or o single ascii character
 or o generic ascii string

RATIO consists of
 o numerator (integer)
 o ':' (a colon)
 o denominator (integer)

The currently supported tags for the STREAM-HEADER:
	 W - [integer] frame width, pixels, should be > 0
 H - [integer] frame height, pixels, should be > 0
 I - [char] interlacing:  p - progressive (none)
                          t - top-field-first
                          b - bottom-field-first
                          ? - unknown
 F - [ratio] frame-rate, 0:0 == unknown
 A - [ratio] sample (pixel) aspect ratio, 0:0 == unknown
 X - [character string] 'metadata' (unparsed, but passed around)

The currently supported tags for the FRAME-HEADER:
 X - character string 'metadata' (unparsed, but passed around)
*/

/* stream stream header to get width and height and interlace */

/*
YUV4MPEG2 W720 H576 F25:1 I? A0:0 XM2AR002
FRAME
data
*/

/* create a line buffer */
bptr = malloc(READSIZE);
if(! bptr)
	{
	fprintf(stderr, "subtitler-yuv(): malloc bptr failed\n");

	exit(1);
	}

/*
From libc.info:
ssize_t getline (char **LINEPTR, size_t *N, FILE *STREAM)
This function is a GNU extension, but it is the recommended way to
read lines from a stream.  The alternative standard functions are 
unreliable.
*/

b = READSIZE;
a = getline(&bptr, &sb, stdin);
if(a == -1)
	{
	fprintf(stderr,\
	"subtitler-yuv(): EOF in read stream header, stop.\n");

	exit(0);
	}

if(strncmp(bptr, "YUV4MPEG2", 9) != 0)
	{
	fprintf(stderr,\
	"subtitler-yuv(): YUV4MPEG2 not found in stream header, aborting.\n");

	exit(1);
	}

/* parse arguments */
running = strsave(bptr);
if(! running)
	{
	fprintf(stderr,\
	"subtitler-yuv(): strsave(options) failed, aborting.\n");

	exit(1);
	}

while(1)
	{
	token = strsep (&running, " ");
	if(token == NULL) break;

	/* avoid empty string */
	if(token[0] == 0) continue;
			
	if(strncmp(token, "W", 1) == 0)
	sscanf(token, "W%d", &width);
	sscanf(token, "H%d", &height);
	sscanf(token, "F%d:%d", &frame_rate, &a);
	sscanf(token, "I%c", &interlace_mode);
	} /* end while parse tags */

free(running);

if(debug_flag)
	{
	fprintf(stderr, "width=%d height=%d\n\
	frame_rate=%d a=%d\n\
	interlace_mode=%c\n\
	aspect_ratio=%.2f\n",
	width, height,\
	frame_rate, a,\
	interlace_mode,\
	aspect_ratio);
	}

if(test_mode)
	{
	fprintf(stderr,\
	"ppml_first_frame=%d ppml_last_frame=%d\n",\
	ppml_first_frame, ppml_last_frame);

	/* test if ppml file can be executed */
	for(current_frame_nr = ppml_first_frame;\
			current_frame_nr < ppml_last_frame; current_frame_nr++)
		{	
		/*
		calculate where to put and how to reformat the subtitles.
		These are globals.
		*/
		ImageData = buffer;
		image_width = width;
		image_height = height;
		if(! have_bottom_margin_flag)
			{
			window_bottom = image_height - window_bottom;
			have_bottom_margin_flag = 1;
			}

		line_h_start = subtitle_h_factor * (double)image_width;
		line_h_end = (double)image_width - (double)line_h_start;
		window_bottom = \
		image_height - (subtitle_v_factor * (double)image_height);

		/*
		collect any objects from database for this frame
		and add to object list.
		*/
		process_frame_number(current_frame_nr);

		/* add objects in object list to display, and update params */
		add_objects(current_frame_nr);
		}
	have_bottom_margin_flag = 0;
	delete_all_objects();
	delete_all_frames();

	fprintf(stderr, "TEST COMPLETE, NO ERRORS FOUND\n");

	exit(1);

	/* cancel testmode, go for real */
	test_mode = 0;

	/* re-load ppml file */
	if(! load_ppml_file(subtitle_file) )
		{
		fprintf(stderr, "subtitler(): could not re-load ppml file %s\n",\
		subtitle_file);

		/* return init error */
		exit(1);
		}

	} /* end if test_mode */

/* echo the stream header */
fprintf(outfile, "%s", bptr);

/* read frames in YUV format from stdin */
current_frame_nr = 1;
while(1)
	{
	/* read a frame */
	b = READSIZE;
	a = getline(&bptr, &sb, stdin);
	if(a == -1)
		{
		fprintf(stderr,\
		"subtitler-yuv(): unexpected EOF in frame header, aborting.\n");
		
		exit(1);
		}
	if(strncmp(bptr, "FRAME", 5) != 0)
		{
		fprintf(stderr,\
		"subtitler-yuv(): No FRAME found in frame, aborting.\n");

		exit(1);
		}

	/* 
	This is how it is written:
	Write Y', Cb, and Cr planes
	for (i = 0; i < 3; i++)
		{
		unsigned char *srctop = upper_field[i];
		unsigned char *srcbot = lower_field[i];
		// alternately write one line from each
		for (y = 0; y < height; y += 2)
			{
			if (y4m_write(fd, srctop, width)) return Y4M_ERR_SYSTEM;
			srctop += width;
			if (y4m_write(fd, srcbot, width)) return Y4M_ERR_SYSTEM;
			srcbot += width;
			}
		// for chroma, width/height are half as big
		if (i == 0)
			{
			width /= 2;
			height /= 2;
			}
		}
	*/

	/* calculate length */
	ylength = width * height;
	ulength = (width / 2) * (height / 2);
	vlength = (width / 2) * (height / 2);
	length =  ylength + ulength + vlength;

	if(debug_flag)
		{
		fprintf(stderr, "main(): calculated length=%d\n", length);
		}

	/* read in data */
	a = fread(buffer, sizeof(char), length, stdin);
	if(a != length)
		{
		fprintf(stderr, "subtitler-yuv(): EOF in frame data, stop.\n");

		/* this is not an error ? */
		exit(0);
		}

	if(debug_flag)
		{
		fprintf(stderr, "main(): have frame=%d\n", current_frame_nr);
		}

	ImageData = buffer;
	image_width = width;
	image_height = height;
	if(! have_bottom_margin_flag)
		{
		window_bottom = image_height - window_bottom;
		have_bottom_margin_flag = 1;
		}

	if(debug_flag)
		{
		fprintf(stderr, \
		"frame_nr=%d\n\
		ImageData=%p image_width=%d image_height=%d\n",\
		current_frame_nr,\
		ImageData, image_width, image_height);
		}

	/*
	calculate where to put and how to reformat the subtitles.
	These are globals.
	*/
	line_h_start = subtitle_h_factor * (double)image_width;
	line_h_end = (double)image_width - (double)line_h_start;
	window_bottom = image_height - (subtitle_v_factor * (double)image_height);

	if\
	(\
	(dcontrast != 100.0) ||\
	(dsaturation != 100.0) ||\
	(u_shift) ||\
	(v_shift)\
	)
		{	
		/*
		brightness, contrast, saturation, U zero shift, V zero shift.
		*/
		ucptrs = ImageData;
		/* set pointers */
	    py = ImageData;

		if(! reverse_uv_flag)
			{
		    pu = ImageData + image_width * image_height;
		    pv = ImageData + (image_width * image_height * 5) / 4;
			}
		else
			{
		    pv = ImageData + image_width * image_height;
		    pu = ImageData + (image_width * image_height * 5) / 4;
			}

		for(y = 0; y < height; y++)
			{
			for(x = 0; x < width; x++)
				{
				/* brightness */
				if( (brightness + *py) > 255) *py = 255;
				else if ( (brightness + *py) < 0) *py = 0;
				else *py += brightness;

				/* contrast */
				da = *py;
				da *= dcontrast / 100.0;
				*py = (int)da;

				/* saturation */
				a = (int)*pu - 128;
				b = (int)*pv - 128;

				a *= dsaturation / 100.0;
				b *= dsaturation / 100.0;

				*pu = (uint8_t)a + 128;
				*pv = (uint8_t)b + 128;
						
				/* u_shift */
				*pu += u_shift;
				*pu &= 0xff;

				/* v_shift */
				*pv += v_shift;
				*pv &= 255;
			
				/* increment Y pointer */
				py++;

				/* increment U and V vector pointers */
				if(x % 2)
					{
					pu++;
					pv++;
					}
				} /* end for all x */

			if( (y + 1) % 2)
				{
				pu -= width / 2;
				pv -= width / 2;
				}

			} /* end for y */
		} /* end if contrast, saturation, u_shift, v_shift */

	if( dhue || dhue_line_drift)
		{
		/*
		UV vector rotation.
		Dynamic UV vector rotation (NTSC line phase error correction).
		*/
		/* set pointers */
	    py = ImageData;

		if(! reverse_uv_flag)
			{
		    pu = ImageData + image_width * image_height;
		    pv = ImageData + (image_width * image_height * 5) / 4;
			}
		else
			{
		    pv = ImageData + image_width * image_height;
		    pu = ImageData + (image_width * image_height * 5) / 4;
			}

		for(y = 0; y < height; y++)
			{
			for(x = 0; x < width; x++)
				{
				/*
				NTSC color correction at start and end of line
				Assuming middle to be correct, most users would have
				adjusted on face color somewhere in the middle.
				*/

				/* the phase drift over one horizontal line */
				da = (double)x / (double)width; // 0 to 1
				
				/* go for middle, now -.5 to +.5 */
				da -= .5;
						
				/* multiply by specified dynamic correction factor */
				db = dhue_line_drift * da;

				/* add the static hue correction specified */
				db += (double)dhue;

				/* hue and saturation*/
				a = (int)*pu - 128;
				b = (int)*pv - 128;
				adjust_color(&a, &b, db, dsaturation);
				*pu = (uint8_t)a + 128;
				*pv = (uint8_t)b + 128;
						
				/* increment Y pointer */
				py++;

				/* increment U and V vector pointers */
				if(x % 2)
					{
					pu++;
					pv++;
					}
				} /* end for all x */
					
			/* 
			2 x 2 color pixels on screen for each Y value,
			repeat each line twice.					

			Orientation on screen Y (*) and U V (o)
			* o
			o o 
			drop shadow :-) color less area below char looks better.
			sink a line.
			*/
			if( (y + 1) % 2)
				{
				pu -= width / 2;
				pv -= width / 2;
				}

			} /* end for y */

		} /* end if some sort of hue */

	if(add_objects_flag)
		{
		/*
		collect any objects from database for this frame
		and add to object list.
		*/
		process_frame_number(current_frame_nr);

		/* add objects in object list to display, and update params */
		add_objects(current_frame_nr);

		} /* end if add_objects_flag */

	if(show_output_flag)
		{
		/* create an xwindows display */
		if(! window_open_flag)
			{
			if(debug_flag)
				{
				fprintf(stderr, "subtitler(): opening window\n");
				}

//			openwin(argc, argv, width, height);
			openwin(0, NULL, width, height);
	
			window_size = width * height;
			window_open_flag = 1;
	
			if(color_depth == 0) color_depth = get_x11_bpp();

			} /* end if ! window_open_flag */

		/* get X11 buffer */
		ucptrd = (unsigned char *)getbuf();

		/* copy data to X11 buffer */
		ucptrs = ImageData;

		/* set pointers */
	    py = ImageData;
		
//		if(! reverse_uv_flag)
//			{
// ?????????
		    pu = ImageData + image_width * image_height;
		    pv = ImageData + (image_width * image_height * 5) / 4;
//			}
//		else
//			{
// ??????????
//		    pu = ImageData + image_width * image_height;
//		    pv = ImageData + (image_width * image_height * 5) / 4;
//			}

		/* ucptrd is pointer to xbuffer BGR */

		for(y = 0; y < height; y++)
			{
			/* get a line from buffer start, copy to xbuffer BGR */
			for(x = 0; x < width; x++)
				{
				cy = ( (0xff & *py) - 16);
				cy  *= 76310;

				cu = (0xff & *pu) - 128;
				cv = (0xff & *pv) - 128;

				cr = 104635 * cv;
				cg = -25690 * cu + -53294 * cv;
				cb = 132278 * cu;

				if(color_depth == 32) // 4 bytes per pixel							
					{
					*ucptrd++ = LIMIT(cb + cy); // B
					*ucptrd++ = LIMIT(cg + cy); // G
					*ucptrd++ = LIMIT(cr + cy); // R

					/* one more byte */
					*ucptrd++ = 0; // last byte is empty.
					} /* end if color depth 32 */

				/* 24 bpp not tested */
				else if(color_depth == 24) // 3 bytes per pixel
					{
					*ucptrd++ = LIMIT(cb + cy); // B
					*ucptrd++ = LIMIT(cg + cy); // G
					*ucptrd++ = LIMIT(cr + cy); // R
					}
						
				/* increment Y pointer */
				py++;

				/* increment U and V vector pointers */
				if(x % 2)
					{
					pu++;
					pv++;
					}
				} /* end for all x */
					
			/* 
			2 x 2 color pixels on screen for each Y value,
			repeat each line twice.					

			Orientation on screen Y (*) and U V (o)
			* o
			o o 
			drop shadow :-) color less area below char looks better.
			sink a line.
			*/
			if( (y + 1) % 2)
				{
				pu -= width / 2;
				pv -= width / 2;
				}

			} /* end for y (all lines) */

		/* show X11 buffer */
		putimage(width, height);
		} /* end if show_output_flag */

	if(! skip_flag)
		{
		/* echo the frame header */
		fprintf(outfile, "%s", bptr);

		/*
		write the frame to the output
		Note:
		we have only added data to the buffer, not changed its shape or size.
		*/
		a = fwrite(buffer, sizeof(char), length, outfile);
		if(a != length)
			{
			fprintf(stderr,\
			"subtitler-yuv(): fwrite(): failed,\n\
			tried to write %d bytes, only could write %d bytes, aborting.\n",\
			length, a);

			exit(1);
			}

		} /* end if ! skip_flag */

	current_frame_nr++;
	} /* end while frames */

/* exit OK */
exit(0);
} /* end function main */


int add_text(\
int x, int y,\
char *text, int u, int v,\
double contrast, double transparency, font_desc_t *pfd, int espace)
{
int a;
char *ptr;

if(debug_flag)
	{
	fprintf(stderr, "subtitler(): add_text(): x=%d y=%d text=%s\n\
	u=%d v=%d contrast=%.2f transparency=%.2f\n\
	font_desc_t=%p espace=%d\n",\
	x, y, text, u, v, contrast, transparency, pfd, espace);
	} 

if(! pfd)
	{
	fprintf(stderr,
	"subtitler-yuv(): add_text(): frame=%d no font specified, aborting\n",\
	current_frame_nr);

	exit(1);
	}

ptr = text;
while(*ptr)
	{
	/* convert to signed */
	a = *ptr;
	if(*ptr < 0) a += 256;

	if(a == ' ')
		{
		}
	else
		{
		print_char(x, y, a, u, v, contrast, transparency, pfd);
		}

	x += pfd->width[a] + pfd->charspace;

	x += espace; //extra_character_space;

	ptr++;
	}

return 1;
} /* end function add_text */


int test_char_set(int frame)
{
int a, i, j, x, y;
char temp[1024];

if(debug_flag)
	{
	fprintf(stderr, "subtitler(): test_char_set(): arg frame=%d\n", frame);
	} 

a = 128;
y = 100;
for(i = 0; i < 16; i++)
	{
	if(a > 256) return 1;

	sprintf(temp, "pos=%d", a);
	add_text(0, y, temp, 0, 0, 0.0, 0.0, NULL, extra_character_space);

	x = 200;
	for(j = 0; j < 16; j++)
		{
		print_char(x, y, a, 0, 0, 0.0, 0.0, NULL);

		x += vo_font->width[a] + vo_font->charspace;

		x += extra_character_space;

		a++;
		} /* end for j */

	y += 33;
	} /* end for i */

return 1;
} /* end function test_char_set */


int print_char(\
int x, int y, int c,\
int u, int v,\
double contrast, double transparency, font_desc_t *pfd)
{
if(debug_flag)
	{
	fprintf(stderr, "subtiter(): print_char(): arg\n\
	x=%d y=%d c=%d u=%d v=%d contrast=%.2f transparency=%.2f\n\
	pfd=%p\n",\
	x, y, c, u, v, contrast, transparency, pfd);
	}

if(! pfd)
	{
	fprintf(stderr,
	"subtitler-yuv(): print_char(): frame=%d no font specified, aborting\n",\
	current_frame_nr);

	exit(1);
	}

if(test_mode) return 1;

draw_alpha(\
	x,\
	y,\
	pfd->width[c],\
	pfd->pic_a[default_font]->h,\
	pfd->pic_b[default_font]->bmp + pfd->start[c],\
	pfd->pic_a[default_font]->bmp + pfd->start[c],\
	pfd->pic_a[default_font]->w,\
	u, v, contrast, transparency);

return 1;
} /* end function print_char */


void draw_alpha(\
	int x0, int y0,\
	int w, int h,\
	uint8_t *src, uint8_t *srca, int stride,\
	int u, int v, double contrast, double transparency)
{
int a, b, c, x, y, sx, cd;
uint8_t *py, *pu, *pv;
uint8_t *sc, *sa;
double dm, di, dci;
uint8_t uy, ua, uc;
int mu, mv, iu, iv;
uint8_t *psy, *psu, *psv, *pse;

if(debug_flag)
	{
	fprintf(stderr, \
	"subtitler(): draw_alpha(): x0=%d y0=%d w=%d h=%d\n\
	src=%p srca=%p stride=%d u=%d v=%d\n\
	contrast=%.2f transparency=%.2f\n",\
	x0, y0, w, h,\
	src, srca, stride, u, v,\
	contrast, transparency);

	fprintf(stderr, "image_width=%d image_height=%d\n", image_width, image_height);
	fprintf(stderr, "ImageData=%p\n", ImageData);
	}

/* calculate multiplier for transparency ouside loops */
dm = transparency / 100.0; // main, 1 for 100 % transparent
di = 1.0 - dm; // insert (text here) 0 for 100% transparent

/*
do not multiply color (saturation) with contrast,
saturation could be done in adjust color, but done here for speed
*/
dci = di;

di *= (contrast / 100.0); // adjust contrast insert (text)

/* 
We seem to be in this format I420:
   y = dest;
   v = dest + width * height;
   u = dest + width * height * 5 / 4;

Orientation of Y (*) relative to chroma U and V (o)
* o
o o
So, an array of 2x2 chroma pixels exists for each luminance pixel
The consequence of this is that there will be a color-less area
of one line on the right and on the bottom of each character.
Dropshadow :-)
*/

py = ImageData;
pu = ImageData + image_width * image_height;
pv = ImageData + (image_width * image_height * 5) / 4;

sc = src;
sa = srca;

a = y0 * image_width;
b = image_width / 4;
c = image_width / 2;

py += x0 + a;
a /= 4;

pu += (x0 / 2) + a;
pv += (x0 / 2) + a;

/* on odd lines, need to go a quarter of a 'line' back */
if(y0 % 2)
	{
	pu -= b;
	pv -= b;				
	}

psy = py;
psu = pu;
psv = pv;
pse = pv + (image_width * image_height * 5) / 4;

for(y = 0; y < h; y++)
	{
	for(x = 0; x < w; x++)
		{
		/* clip right scroll */
		if( (x + x0) >= image_width) continue;

		/* clip left scroll */
		if( (x + x0 ) < 0) continue;

		/* clip top scroll */
		if( (y + y0) >= image_height) continue;
	
		/* clip bottom scroll */
		if( (y + y0) < 0) continue;

		if(py < psy) continue;
		if(pu < psu) continue;
		if(pv < psv) continue;
		
		if(py >= psu) continue;
		if(pu >= psv) continue;
		if(pv >= pse) continue;

		if(sa[x])
			{
			/* some temp vars */
			uy = py[x];
			ua = sa[x];
			uc = sc[x];

			/* get decision factor before we change anything */
			cd = ( (py[x] * sa[x]) >> 8) < 5;

			/* calculate value insert (character) */
			uy = ( (uy * ua) >> 8) + uc;

			/* attenuate insert (character) the opposite way */
			uy *= di; // di is 0 for 100 % transparent

			/* attenuate main */
			py[x] *= dm; // dm is 1 for 100% transp
							
			/* add what is left of the insert (character) */
			py[x] += uy;
				
			sx = 1;
			if( (x + x0) % 2) sx = 0; 

			/* trailing shadow no */
//			if(x  < (w - 4) ) sx = 1; // hack, looks better :-)
//			else sx = 0;

			if(cd)
				{
				/* some temp vars, integer so we can multiply */
				mu = pu[x / 2 + sx] - 128;
				mv = pv[x / 2 + sx] - 128;

				/* adjust main color saturation */
				mu *= dm;
				mv *= dm;

				/* adjust insert (character) color saturation */
				iu = u * dci;
				iv = v * dci;

				if(sc[x]) /* white part of char */
					{
					/* set U vector */
					pu[x / 2 + sx] = 128 + mu + iu;

					/* set V vector */
					pv[x / 2 + sx] = 128 + mv + iv;
					}
				else /* shadow around char, no color */
					{
					/* set U vector */
					pu[ (x / 2) + sx] = 128 + mu;

					/* set V vector */
					pv[ (x / 2) + sx] = 128 + mv;
					}
				} /* end if cd */
			} /* end if sa[x] */
		} /* end for all x */

	sc += stride;
	sa += stride;

	py += image_width;

	if( (y + y0) % 2)
		{
		pu += c;
		pv += c;
		}

	} /* end for all y */

} /* end function draw_alpha */


int add_picture(struct object *pa)
{
/*
reads yuyv in pa -> data into the YUV 420 ImageData buffer.
*/
uint8_t *py, *pu, *pv;
int a, b, c, x, y;
char *ps;
char ca;
int u_time;
int in_range;
double dc, dd, dm, ds;
int ck_flag = 0;
int odd_line;
uint8_t *psy, *psu, *psv, *pse;

if(debug_flag)
	{
	fprintf(stderr, "subtitler(): add_picture(): arg pa=%p\n\
	pa->xsize=%.2f pa->ysize=%.2f pa->ck_color=%.2f\n",\
	pa,\
	pa -> xsize, pa -> ysize,\
	pa -> chroma_key_color);
	}

/* argument check */
if(! ImageData) return 0;
if(! pa) return 0;
if( (int)pa -> xsize == 0) return 1;
if( (int)pa -> ysize == 0) return 1;

if(test_mode) return 1;

/* calculate multiplier for transparency ouside loops */
dm = (100.0 - pa -> transparency) / 100.0;
dd = 1.0 - dm;

dc = dm * (pa -> contrast / 100.0);
ds = (pa -> saturation / 100.0);

/* saturation could be done in adjust color, but done here for speed */
//ds = 1.0;

b = image_width / 4;
c = image_width / 2;

py = ImageData;
if(! reverse_uv_flag)
	{
	pu = ImageData + (image_width * image_height * 5) / 4;
	pv = ImageData + (image_width * image_height);
	}
else
	{
	pv = ImageData + (image_width * image_height * 5) / 4;
	pu = ImageData + (image_width * image_height);
	}

a = (int)pa -> ypos * image_width;

py += (int)pa -> xpos + a;
a /= 4;
pu += ( (int)pa -> xpos / 2) + a;
pv += ( (int)pa -> xpos / 2) + a;

ps = pa -> data;

if( (int)pa -> ypos % 2 )
	{
	pu -= b;
	pv -= b;
	}

psy = py;
psu = pu;
psv = pv;
pse = pv + (image_width * image_height * 5) / 4;

// reading sequence is YUYV, so U is first.
u_time = 1;
for(y = 0; y < (int)pa -> ysize; y++)
	{
	odd_line = (y + (int)pa -> ypos) % 2;

	for(x = 0; x < (int)pa -> xsize; x++)
		{
		/* find out if OK to display */
		if(py < psy) continue;
		if(pu < psu) continue;
		if(pv < psv) continue;
		
		if(py >= psu) continue;
		if(pu >= psv) continue;
		if(pv >= pse) continue;

		in_range = 1;
		/* clip right scroll */
		if( (x + (int)pa -> xpos) >= image_width) in_range = 0;

		/* clip left scroll */
		if( (x + (int)pa -> xpos ) < 0) in_range = 0;

		/* clip top scroll */
		if( (y + (int)pa -> ypos) >= image_height) in_range = 0;

		/* clip bottom scroll */
		if( (y + (int)pa -> ypos) < 0) in_range = 0;

		/* slice level */
		a = *ps;
		if(a < 0) a += 256;
		if( a < ( (int)pa -> slice_level) ) in_range = 0;

		if(\
		(pa -> zrotation != 0) ||\
		(pa -> xshear != 0) || (pa -> yshear != 0)\
		)
			{
			/*
			for rotate and shear, the luminance value of the border
			to cut away.
			Since this would remove picture data, for this not to
			happen, we add 1 step to the luminance if it happens to
			be the same as border_luminanc in yuv_to_ppm().
			With this trick it is guaranteed border_luminance never happens
			in the .ppm file that mogrify processes.
			*/
			if(pa -> mask_level)
				{
				if(a == pa -> mask_level) in_range = 0;
				}
			else
				{
				if(a == default_border_luminance) in_range = 0;
				}
			} /* end if rotate or shear */

		/* test for chroma key match if color specified */
		if(pa -> chroma_key_saturation)
			{
			if(u_time)
				{
				if(! odd_line)
					{
					a = (int)pu[x / 2] - 128;
					b = (int)pv[x / 2] - 128;
					ck_flag =\
					chroma_key(\
					a, b,\
					pa -> chroma_key_color,\
					pa -> chroma_key_window,\
					pa -> chroma_key_saturation);
					} /* end if even line */
				else
					{
					a = (int)pu[(x / 2) + c] - 128;
					b = (int)pv[(x / 2) + c] - 128;
					ck_flag =\
					chroma_key(\
					a, b,\
					pa -> chroma_key_color,\
					pa -> chroma_key_window,\
					pa -> chroma_key_saturation);
					} /* end if odd line */
				} /* end if u_time */	

			/* transport to next time here ! */
			if(! ck_flag) in_range = 0;
			} /* end if chroma key */

		if(in_range)
			{
			py[x] *= dd;
			py[x] += dc * (uint8_t)*ps;
			} /* end if in_range */

		ps++;

		if(in_range)
			{
			if(u_time)
				{
				ca = *ps;
				ca = 128 + ( ( (uint8_t)*ps - 128 ) * ds);	
				
				pu[x / 2] *= dd;
				pu[x / 2] += dm * (uint8_t)ca; 
				}
			else
				{
				ca = *ps;
				ca = 128 + ( ( (uint8_t)*ps - 128 ) * ds);

				pv[x / 2] *= dd;
				pv[x / 2] += dm * (uint8_t)ca; 
				}

			/* apply hue correction if both U and V set */

//				if(! u_time)
				{
				if(pa -> hue)
					{
					/*
					hue,
					saturation done outside adjust_color() for speed
					*/

					a = (int)pu[x / 2] - 128;
					b = (int)pv[x / 2] - 128;
					adjust_color(&a, &b, pa -> hue, 100.0);
					pu[x / 2] = (uint8_t)a + 128;
					pv[x / 2] = (uint8_t)b + 128;

					} /* end if hue */

				} /* end if ! u_time */
			} /* end if in range */

		ps++;
		u_time = 1 - u_time;

		} /* end for all x */

	if( (int) pa -> xsize % 2) u_time = 1 - u_time;

	py += image_width;

	if(odd_line)
		{
		pu += c;
		pv += c;
		}

	} /* end for all y */

return 1;
}/* end function add_picture */ 


int set_main_movie_properties(struct object *pa)
{
if(debug_flag)
	{
	fprintf(stderr, "set_main_movie_properties(): arg pa=%p\n", pa);
	}

if(! pa) return 0;

dcontrast = pa -> contrast;
brightness = (int)pa -> brightness;
dsaturation = pa -> saturation;
dhue = pa -> hue;
dhue_line_drift = pa -> hue_line_drift;
u_shift = (int)pa -> u_shift;
v_shift = (int)pa -> v_shift;
show_output_flag = (int)pa -> show_output;
if(pa -> status & OBJECT_STATUS_NO_YUV_OUTPUT) skip_flag = 1;
else skip_flag = 0;

return 1;
} /* end function set_main_movie_properties */

