; A tool to make an Axe Rewriter for a given application
;
; Copyright (C) 2008-2011 Eric Smith and Stanford University
; Copyright (C) 2013-2024 Kestrel Institute
; Copyright (C) 2016-2020 Kestrel Technology, LLC
;
; License: A 3-clause BSD license. See the file books/3BSD-mod.txt.
;
; Author: Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

;; This book contains the tool make-rewriter-simple, which creates custom
;; rewriters.  When calling it, the user indicates an evaluator to use for
;; ground terms, an evaluator to use for syntaxp functions, and an evaluator to
;; use for bind-free functions.

;; The generated rewriter is a simple variant of the Axe Rewriter (no
;; "work-hard", no calling the Axe Prover or STP, no special xor
;; simplification).

;; Rewriters generated by this tool are defined in :logic mode and are guard-verified.

;; See example in rewriter-basic.lisp.

;; See justification theorems in rewriter-justification.lisp.

;; If you have rules that contain work-hard hyps, consider passing in work-hard
;; as a named rule, to expand it.

;; Consider doing:
;;   (set-evisc-tuple t :iprint nil :sites :gag-mode)
;  when working with
;; calls to make-rewriter-simple, to prevent printing of enormous induction
;; schemes.

;; TODO: Add checks (or guards?) that interpreted-function-alists are complete.

;; TODO: Consider making a separate version for when we are not memoizing.

;; TODO: Consider adding special handling for bv-array-if.

;; TODO: Add more common parameters to the rewrite-stobj instead of passing them around.

(include-book "rewriter-common")
(include-book "supporting-nodes") ; for drop-non-supporters-array
(include-book "node-replacement-array3")
(include-book "refined-assumption-alists2")
(include-book "tries")
(include-book "kestrel/utilities/print-levels" :dir :system)
(include-book "rule-limits")
;(include-book "rule-alists") ; for get-rules-for-fn
(include-book "make-substitution-code-simple")
;; (include-book "make-instantiation-code-simple")
(include-book "make-instantiation-code-simple-free-vars")
;(include-book "make-instantiation-code-simple-no-free-vars")
(include-book "make-instantiation-code-simple-no-free-vars2")
(include-book "make-dag-indices")
;(include-book "dag-array-builders")
(include-book "add-and-normalize-expr")
(include-book "memoization")
(include-book "dag-array-printing2")
(include-book "unify-tree-and-dag")
(include-book "unify-term-and-dag-fast")
(include-book "hit-counts")
(include-book "dag-to-term")
(include-book "kestrel/utilities/defconst-computed" :dir :system) ;not strictly needed
;(include-book "def-dag-builder-theorems")
;(include-book "kestrel/utilities/all-vars-in-term-bound-in-alistp" :dir :system)
(include-book "kestrel/utilities/make-event-quiet" :dir :system)
;(include-book "kestrel/typed-lists-light/pseudo-term-listp" :dir :system) ;drop?
(include-book "kestrel/alists-light/strip-cdrs" :dir :system) ;need strip-cdrs-of-append for the generated proofs
(include-book "kestrel/utilities/with-local-stobjs" :dir :system)
(include-book "kestrel/utilities/redundancy" :dir :system)
(include-book "kestrel/utilities/strip-stars-from-name" :dir :system) ; for starts-and-ends-with-starsp
(include-book "renumbering-stobj")
(include-book "rewrite-stobj")
(include-book "cars-increasing-by-1")
(local (include-book "kestrel/lists-light/nth" :dir :system))
(local (include-book "kestrel/lists-light/len" :dir :system))

;; TODO: Consider putting these support rules into a book that gets locally included by the encapsulate.

;; Would something like this be helpful?
;; (defthmd quotep-compound-recognizer
;;   (implies (quotep x)
;;            (consp x))
;;   :rule-classes :compound-recognizer)

(defthm trees-to-memoizep-of-cons-if-not-equal-car
  (equal (trees-to-memoizep (cons-if-not-equal-car tree trees))
         (and (tree-to-memoizep tree)
              (trees-to-memoizep trees)))
  :hints (("Goal" :in-theory (enable cons-if-not-equal-car))))

(defthmd <-of-+-of-1-when-natps
  (implies (and (syntaxp (not (quotep y)))
                (natp x)
                (natp y))
           (equal (< x (+ 1 y)) (<= x y))))

;; Axe version, to be kept disabled except in certain Axe proofs
(defthmd <-of-if-arg2-axe
  (equal (< x (if test y z))
         (if test
             (< x y)
           (< x z))))

;; Can't be local, unless we move it into the macro
(defun simplify-trees-and-add-to-dag-induct (trees count)
  (declare (xargs :measure (len trees)))
  (if (or (not (mbt (natp count)))
          (= 0 count))
      (list trees count)
    (if (atom trees)
        (list trees count)
      (simplify-trees-and-add-to-dag-induct (rest trees) (+ -1 count)))))

;; Used in generated proofs. Kept disabled so as not to interfere with anything else.
(defthmd symbol-listp-of-append-disabled
  (equal (symbol-listp (append x y))
         (and (symbol-listp (true-list-fix x))
              (symbol-listp y)))
  :hints (("Goal" :in-theory (enable append symbol-listp true-list-fix))))

;gen
(defthm pseudo-dag-arrayp-of-+-of-1-and-largest-non-quotep-of-car
  (implies (and (bounded-darg-list-listp assumption-arg-lists dag-len)
                (consp assumption-arg-lists)
                (pseudo-dag-arrayp dag-array-name dag-array dag-len))
           (pseudo-dag-arrayp dag-array-name dag-array (+ 1 (largest-non-quotep (car assumption-arg-lists))))))

;; used in generated proofs, to be kept disabled
(defthmd symbolp-when-member-equal-disabled
  (implies (and (member-equal x free)
                (symbol-listp free))
           (symbolp x))
  :hints (("Goal" :in-theory (enable symbol-listp member-equal))))

;; used in generated proofs, to be kept disabled
(defthmd not-equal-when-member-equal-disabled
  (implies (and (syntaxp (quotep y))
                (member-equal x vals)
                (syntaxp (quotep vals))
                (not (member-equal y vals)))
           (not (equal x y))))

;; used in generated proofs, to be kept disabled
(defthmd not-equal-when-member-equal-alt-disabled
  (implies (and (syntaxp (quotep y))
                (member-equal x vals)
                (syntaxp (quotep vals))
                (not (member-equal y vals)))
           (not (equal y x))))

;why is this needed?  the hyp is not being rewritten right during backchaining
(defthmd max-key-hack
  (equal (if (consp alist) x (< y (max-key alist z)))
         (if (consp alist) x (< y z))))

(defthmd max-key-hack-2
  (equal (if (consp alist) x (< (max-key alist z) y))
         (if (consp alist) x (< z y))))

;; used in generated proofs, to be kept disabled
(defthmd integerp-when-natp-disabled
  (implies (natp x)
           (integerp x)))

;; used in generated proofs, to be kept disabled
(defthmd strip-cars-of-append-disabled
  (equal (strip-cars (append x y))
         (append (strip-cars x)
                 (strip-cars y)))
  :hints (("Goal" :in-theory (enable strip-cars))))

;; used in generated proofs, to be kept disabled
(defthmd pseudo-termp-of-car-when-pseudo-term-listp-cheap-disabled
  (implies (pseudo-term-listp terms)
           (pseudo-termp (car terms)))
  :rule-classes ((:rewrite :backchain-limit-lst (0)))
  :hints (("Goal" :in-theory (enable pseudo-term-listp))))

;; used in generated proofs, to be kept disabled
(defthmd pseudo-term-listp-of-cdr-when-pseudo-term-listp-cheap-disabled
  (implies (pseudo-term-listp terms)
           (pseudo-term-listp (cdr terms)))
  :rule-classes ((:rewrite :backchain-limit-lst (0)))
  :hints (("Goal" :in-theory (enable pseudo-term-listp))))

(defthmd axe-treep-when-dag-exprp
  (implies (dag-exprp expr)
           (axe-treep expr))
  :hints (("Goal" :in-theory (enable axe-treep dag-exprp))))

;; used in generated proofs, to be kept disabled
(defthmd not-<-of-0-when-natp-disabled
  (implies (natp x)
           (not (< x 0))))

;; used in generated proofs, to be kept disabled
(defthmd natp-of-+-of--1-when-natp-disabled
  (implies (natp x)
           (equal (natp (+ -1 x))
                  (< 0 x))))

(defthm bounded-axe-treep-when-bounded-dag-exprp
  (implies (and (bounded-dag-exprp bound2 tree)
                (natp bound2)
                (<= bound2 bound))
           (bounded-axe-treep tree bound))
  :hints (("Goal" :in-theory (enable bounded-axe-treep
                                     bounded-dag-exprp
                                     dag-exprp))))

(defthm bounded-axe-treep-of-cdar-when-weak-dagp-aux
  (implies (and (weak-dagp-aux dag)
                (<= (car (car dag)) bound))
           (bounded-axe-treep (cdr (car dag)) bound))
  :hints (("Goal" :in-theory (enable weak-dagp-aux))))

(defthm consp-of-cdr-forward-to-consp
  (implies (consp (cdr x))
           (consp x))
  :rule-classes :forward-chaining)

(defthmd member-equal-when-member-equal-and-subsetp-equal
  (implies (and (member-equal a x2)
                (subsetp-equal x2 x))
           (member-equal a x)))

;; ;loops with LEN-WHEN-DARGP-LESS-THAN?
;; (defthmd consp-to-len-bound-for-make-rewriter-simple
;;   (equal (consp x) (< 0 (len x)))
;;   :hints (("Goal" :in-theory (e/d (len) (len-of-cdr)))))

;loops with LEN-WHEN-DARGP-LESS-THAN
;; (defthmd len-of-cdr-better-for-make-rewriter-simple
;;   (equal (len (cdr x))
;;          (if (equal 0 (len x))
;;              0 (+ -1 (len x)))))

;; How we use the refined-assumption-alist:
;; - To bind free vars in a hyp (calling lookup-in-refined-assumption-alist on the fn and unifying the hyp's args against each arglist).
;; - When rewriting, a THEN or ELSE branch, if not memoizing, temporarily add info by calling extend-refined-assumption-alist-assuming-node
;;   or extend-refined-assumption-alist-assuming-negation-of-node.

;; How we use the node-replacement-array:
;; - To check if a rewritten hyp is known to be non-nil (calling known-true-in-node-replacement-arrayp on its nodenum).
;; - To handle a simplified IF/MYIF/BOOLIF/BVIF test that is known to be non-nil (calling apply-node-replacement-array-bool-to-darg on it).
;; - To handle an argument to NOT that is known to be non-nil (calling apply-node-replacement-array-bool-to-darg on it).
;; - To replace a var (calling apply-node-replacement-array on its nodenum after we add the node to the dag).
;; - To replace a simplified function call (calling apply-node-replacement-array on its nodenum after we add the node to the dag).
;; - When rewriting, a THEN or ELSE branch, if not memoizing, temporarily add info by calling update-node-replacement-array-for-assuming-node or
;;   update-node-replacement-array-for-assuming-negation-of-node and later calling undo-writes-to-node-replacement-array to undo.
;; Advantages: Lookup is very fast
;; Disadvantages: The thing being looked up must already be in the dag.
;; TODO: Consider whether to support chains of replacements, by looking up repeatedly (might loop).
;; TODO: Consider whether to look up unsimplified assumptions.
;; TODO: Consider whether to simplify the RHSes of assumptions (at start, or when used).

;; OLD:
;; How we use the equality-assumption-alist:
;; 1. To replace a term that is a var (calling replace-var-using-equality-assumption-alist).  This may be rare.
;;  TODO: Could this be handled using node-replacement-alist instead, letting us eliminate the :var case?
;; 2. To replace a (simplified) term that is a function call (calling replace-fun-call-using-equality-assumption-alist).

(defun make-rewriter-simple-fn (suffix ;; gets added to generated names
                                evaluator-base-name
                                syntaxp-evaluator-suffix
                                bind-free-evaluator-suffix)
  (declare (xargs :guard (and (symbolp suffix)
                              (symbolp evaluator-base-name)
                              (symbolp syntaxp-evaluator-suffix)
                              (symbolp bind-free-evaluator-suffix))))
  (let* ((eval-axe-syntaxp-expr-fn (pack$ 'eval-axe-syntaxp-expr- syntaxp-evaluator-suffix)) ; keep in sync with make-axe-syntaxp-evaluator.lisp
         (eval-axe-bind-free-function-application-fn (pack$ 'eval-axe-bind-free-function-application- bind-free-evaluator-suffix)) ; keep in sync with make-axe-bind-free-evaluator.lisp
         (apply-axe-evaluator-to-quoted-args-name (pack$ 'apply- evaluator-base-name '-to-quoted-args))
         (sublis-var-and-eval-name (pack$ 'sublis-var-and-eval- suffix)) ; keep in sync with the name generated by make-substitution-code-simple
         ;; (instantiate-hyp-name (pack$ 'instantiate-hyp- suffix)) ; keep in sync with the name generated by make-substitution-code-simple
         (instantiate-hyp-free-vars-name (pack$ 'instantiate-hyp- suffix '-free-vars)) ; keep in sync with the name generated by make-substitution-code-simple-free-vars
         (instantiate-hyp-no-free-vars2-name (pack$ 'instantiate-hyp- suffix '-no-free-vars2)) ; keep in sync with the name generated by make-substitution-code-simple-no-free-vars
         ;; functions in the main mutual-recursion:
         (relieve-free-var-hyp-and-all-others-name (pack$ 'relieve-free-var-hyp-and-all-others- suffix))
         (relieve-rule-hyps-name (pack$ 'relieve-rule-hyps- suffix))
         (try-to-apply-rules-name (pack$ 'try-to-apply-rules- suffix))
         (simplify-fun-call-and-add-to-dag-name (pack$ 'simplify-fun-call-and-add-to-dag- suffix))
         (simplify-if/myif/boolif-tree-and-add-to-dag3-name (pack$ 'simplify-if/myif/boolif-tree-and-add-to-dag3- suffix))
         (simplify-if/myif/boolif-tree-and-add-to-dag2-name (pack$ 'simplify-if/myif/boolif-tree-and-add-to-dag2- suffix))
         (simplify-if/myif-tree-and-add-to-dag-name (pack$ 'simplify-if/myif-tree-and-add-to-dag- suffix))
         (simplify-boolif-tree-and-add-to-dag-name (pack$ 'simplify-boolif-tree-and-add-to-dag- suffix))
         (simplify-bvif-tree-and-add-to-dag3-name (pack$ 'simplify-bvif-tree-and-add-to-dag3- suffix))
         (simplify-bvif-tree-and-add-to-dag2-name (pack$ 'simplify-bvif-tree-and-add-to-dag2- suffix))
         (simplify-bvif-tree-and-add-to-dag1-name (pack$ 'simplify-bvif-tree-and-add-to-dag1- suffix))
         (simplify-bvif-tree-and-add-to-dag-name (pack$ 'simplify-bvif-tree-and-add-to-dag- suffix))
         (simplify-not-tree-and-add-to-dag-name (pack$ 'simplify-not-tree-and-add-to-dag- suffix))
         (simplify-tree-and-add-to-dag-name (pack$ 'simplify-tree-and-add-to-dag- suffix))
         (simplify-trees-and-add-to-dag-name (pack$ 'simplify-trees-and-add-to-dag- suffix))
         ;; functions after the main mutual-recursion:
         (simplify-dag-expr-name (pack$ 'simplify-dag-expr- suffix))
         (simplify-dag-aux-name (pack$ 'simplify-dag-aux- suffix))
         (simplify-dag-name (pack$ 'simplify-dag- suffix))
         (simplify-term-name (pack$ 'simplify-term- suffix)) ; produces a DAG
         (simp-term-name (pack$ 'simp-term- suffix))         ; produces a term
         (simp-terms-name (pack$ 'simp-terms- suffix)) ; produces a list of terms
         (def-simplified-dag-fn-name (pack$ 'def-simplified-dag-fn- suffix))
         (def-simplified-dag-name (pack$ 'def-simplified-dag- suffix))

         ;; Keep these in sync with the formals of each function:

         (call-of-relieve-free-var-hyp-and-all-others `(,relieve-free-var-hyp-and-all-others-name
                                                        assumption-arg-lists
                                                        hyp-args
                                                        hyp-num
                                                        other-hyps
                                                        alist
                                                        rule-symbol
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                        rewrite-stobj count))
         (call-of-relieve-rule-hyps `(,relieve-rule-hyps-name
                                      hyps hyp-num alist rule-symbol
                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                      node-replacement-array node-replacement-count refined-assumption-alist
                                      rewrite-stobj count))
         (call-of-try-to-apply-rules `(,try-to-apply-rules-name
                                       stored-rules
                                       args-to-match
                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                       node-replacement-array node-replacement-count refined-assumption-alist
                                       rewrite-stobj count))
         (call-of-simplify-fun-call-and-add-to-dag `(,simplify-fun-call-and-add-to-dag-name
                                                     fn
                                                     args
                                                     trees-equal-to-tree
                                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                     node-replacement-array node-replacement-count refined-assumption-alist
                                                     rewrite-stobj count))
         (call-of-simplify-if/myif/boolif-tree-and-add-to-dag3 `(,simplify-if/myif/boolif-tree-and-add-to-dag3-name
                                                                 fn
                                                                 simplified-test
                                                                 simplified-then-branch
                                                                 else-branch
                                                                 tree
                                                                 trees-equal-to-tree
                                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                 node-replacement-array node-replacement-count refined-assumption-alist
                                                                 rewrite-stobj count))
         (call-of-simplify-if/myif/boolif-tree-and-add-to-dag2 `(,simplify-if/myif/boolif-tree-and-add-to-dag2-name
                                                                 fn
                                                                 simplified-test
                                                                 then-branch
                                                                 else-branch
                                                                 tree
                                                                 trees-equal-to-tree
                                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                 node-replacement-array node-replacement-count refined-assumption-alist
                                                                 rewrite-stobj count))
         (call-of-simplify-if/myif-tree-and-add-to-dag `(,simplify-if/myif-tree-and-add-to-dag-name
                                                         tree
                                                         trees-equal-to-tree
                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                         rewrite-stobj count))
         (call-of-simplify-boolif-tree-and-add-to-dag `(,simplify-boolif-tree-and-add-to-dag-name
                                                         tree
                                                         trees-equal-to-tree
                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                         rewrite-stobj count))
         (call-of-simplify-bvif-tree-and-add-to-dag3 `(,simplify-bvif-tree-and-add-to-dag3-name
                                                       simplified-size
                                                       simplified-test
                                                       simplified-then-branch
                                                       else-branch
                                                       tree
                                                       trees-equal-to-tree
                                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                       node-replacement-array node-replacement-count refined-assumption-alist
                                                       rewrite-stobj count))
         (call-of-simplify-bvif-tree-and-add-to-dag2 `(,simplify-bvif-tree-and-add-to-dag2-name
                                                       simplified-size
                                                       simplified-test ; a nodenum
                                                       then-arg ; unsimplified
                                                       else-arg
                                                       tree
                                                       trees-equal-to-tree
                                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                       node-replacement-array node-replacement-count refined-assumption-alist
                                                       rewrite-stobj count))
         (call-of-simplify-bvif-tree-and-add-to-dag1 `(,simplify-bvif-tree-and-add-to-dag1-name
                                                       size-arg ; unsimplified
                                                       simplified-test ; a nodenum
                                                       then-arg ; unsimplified
                                                       else-arg
                                                       tree
                                                       trees-equal-to-tree
                                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                       node-replacement-array node-replacement-count refined-assumption-alist
                                                       rewrite-stobj count))

         (call-of-simplify-bvif-tree-and-add-to-dag `(,simplify-bvif-tree-and-add-to-dag-name
                                                       tree
                                                       trees-equal-to-tree
                                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                       node-replacement-array node-replacement-count refined-assumption-alist
                                                       rewrite-stobj count))
         (call-of-simplify-not-tree-and-add-to-dag `(,simplify-not-tree-and-add-to-dag-name
                                                     tree
                                                     trees-equal-to-tree
                                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                     node-replacement-array node-replacement-count refined-assumption-alist
                                                     rewrite-stobj count))
         (call-of-simplify-tree-and-add-to-dag `(,simplify-tree-and-add-to-dag-name
                                                 tree
                                                 trees-equal-to-tree
                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                 node-replacement-array node-replacement-count refined-assumption-alist
                                                 rewrite-stobj count))
         (call-of-simplify-trees-and-add-to-dag `(,simplify-trees-and-add-to-dag-name
                                                  trees
                                                  dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                  node-replacement-array node-replacement-count refined-assumption-alist
                                                  rewrite-stobj count))
         )
    `(encapsulate ()

       (local (include-book "kestrel/lists-light/cdr" :dir :system))
       (local (include-book "kestrel/lists-light/len" :dir :system))
       (local (include-book "kestrel/lists-light/nth" :dir :system))
       (local (include-book "kestrel/lists-light/true-list-fix" :dir :system))
       (local (include-book "kestrel/lists-light/last" :dir :system))
       (local (include-book "kestrel/lists-light/take" :dir :system))
       (local (include-book "kestrel/arithmetic-light/plus" :dir :system))
       (local (include-book "kestrel/arithmetic-light/natp" :dir :system))
       (local (include-book "kestrel/arithmetic-light/less-than" :dir :system))
       (local (include-book "kestrel/arithmetic-light/less-than-or-equal" :dir :system))
       (local (include-book "kestrel/arithmetic-light/types" :dir :system))
       (local (include-book "kestrel/utilities/if-rules" :dir :system))
       (local (include-book "kestrel/typed-lists-light/pseudo-term-listp" :dir :system)) ;reduce?
       (local (include-book "kestrel/axe/rewriter-support" :dir :system))

       (local (in-theory (disable wf-dagp wf-dagp-expander
                                  default-car
                                  default-cdr
                                  axe-treep
                                  axe-treep-when-pseudo-termp
                                  ;; member-of-cons
                                  ;; SUBSETP-CAR-MEMBER
                                  ;;SUBSETP-CONS-2
                                  AXE-TREE-LISTP PSEUDO-TERM-LISTP
                                  DEFAULT-+-1
                                  DEFAULT-+-2
                                  SYMBOLP-OF-CAR-OF-CAR-WHEN-SYMBOL-TERM-ALISTP
                                  symbol-listp
                                  symbol-alistp axe-rule-hyp-listp
                                  EQLABLE-ALISTP ;prevent inductions
                                  member-equal   ; prevent case splitting
                                  strip-cars
                                  nat-listp ; !
                                  weak-dagp-aux
                                  myquotep
                                  dargp-less-than)))

       (local (in-theory (enable ;;consp-of-assoc-equal-when-node-replacement-alistp
                          ;;dargp-of-cdr-of-assoc-equal-when-node-replacement-alistp
                          ;;dargp-less-than-of-cdr-of-assoc-equal-when-node-replacement-alistp
                          ;;myquotep-of-cdr-of-assoc-equal-when-node-replacement-alistp
                          ;;natp-of-cdr-of-assoc-equal-when-node-replacement-alistp
                          strip-cars-of-append-disabled
                          pseudo-termp-of-car-when-pseudo-term-listp-cheap-disabled
                          pseudo-term-listp-of-cdr-when-pseudo-term-listp-cheap-disabled
                          ;;consp-to-len-bound-for-make-rewriter-simple
                          ;;len-of-cdr-better-for-make-rewriter-simple
                          myquotep-when-dag-exprp-and-quote
                          rationalp-of-sub-tries)))

       ;; Make a version of sublis-var-and-eval:
       (make-substitution-code-simple ,suffix ,evaluator-base-name)

       ;; Make versions of instantiate-hyp, etc.
       ;; (make-instantiation-code-simple ,suffix ,evaluator-base-name)
       (make-instantiation-code-simple-free-vars ,suffix ,evaluator-base-name)
       (make-instantiation-code-simple-no-free-vars2 ,suffix ,evaluator-base-name)

       ;;
       ;; The main mutual-recursion:
       ;;

       ;; TODO: Is the stuff in the dag assumed to be simplified, or not?  Some of those nodes may come from assumptions or even context.

       ;; TODO: It may be possible to avoid checking the count in some functions by using a more complicated measure.

       (mutual-recursion

        ;; Returns (mv erp hyps-relievedp extended-alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array), where extended-alist is irrelevant if hyps-relievedp is nil
        ;; keeps trying ASSUMPTION-ARG-LISTS until it finds a match for HYP-ARGS (thus binding some free vars) for which it can relieve all the OTHER-HYPS (using those variable bindings)
        ;; we need to keep the whole alist in addition to walking down the entry for the current fn
        (defund ,relieve-free-var-hyp-and-all-others-name (assumption-arg-lists ;these are lists of nodenums/quoteps for calls of fn that we can assume (where fn is the top function symbol of the hyp)
                                                           hyp-args ;partially instantiated; any vars that remain must match the assumption
                                                           hyp-num ; only used for printing, we could drop this for speed?
                                                           other-hyps
                                                           alist
                                                           rule-symbol
                                                           dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                           node-replacement-array node-replacement-count refined-assumption-alist
                                                           rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (bounded-darg-list-listp assumption-arg-lists dag-len)
                                      (axe-tree-listp hyp-args) ; todo replace this and the next one with axe-tree-listp?
                                      (posp hyp-num)
                                      (axe-rule-hyp-listp other-hyps)
                                      (symbol-alistp alist)
                                      (bounded-darg-listp (strip-cdrs alist) dag-len)
                                      (alist-suitable-for-hyp-args-and-hypsp alist hyp-args other-hyps)
                                      (symbolp rule-symbol)
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (info-worldp info)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :measure (nfix count)
                          :split-types t
                          :verify-guards nil ; done below
                          )
                   (type (unsigned-byte 60) count)
                   (type (integer 1 *) hyp-num) ;; restrict to a fixnum?
                   (type symbol rule-symbol))
          (if (or (not (mbt (natp count))) (= 0 count))
              (mv :count-exceeded nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
            (if (endp assumption-arg-lists)
                ;; failed to relieve the hyp:
                (prog2$ (and (member-eq rule-symbol (get-monitored-symbols rewrite-stobj))
                             (cw "(Failed to relieve free vars in hyp ~x0 of rule ~x1.)~%" hyp-num rule-symbol))
                        (mv (erp-nil) nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
              (b* ((arg-list (first assumption-arg-lists)) ;; args of the assumption we are currently checking for a match
                   (fail-or-extended-alist (unify-trees-with-dag-nodes hyp-args arg-list dag-array alist)))
                (if (eq :fail fail-or-extended-alist)
                    ;; this assumption didn't match, so proceed to the next assumption:
                    (,relieve-free-var-hyp-and-all-others-name (rest assumption-arg-lists)
                                                               hyp-args hyp-num other-hyps
                                                               alist rule-symbol
                                                               dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                               node-replacement-array node-replacement-count refined-assumption-alist
                                                               rewrite-stobj (+ -1 count))
                  ;; this assumption matched, so try to relieve the rest of the hyps using the resulting extension of ALIST:
                  (b* (((mv erp other-hyps-relievedp extended-alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                        (,relieve-rule-hyps-name other-hyps (+ 1 hyp-num)
                                                 fail-or-extended-alist ; matching with the ASSUMPTION caused free vars to be bound here
                                                 rule-symbol
                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                 node-replacement-array node-replacement-count refined-assumption-alist rewrite-stobj (+ -1 count)))
                       ((when erp) (mv erp nil nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                    (if other-hyps-relievedp
                        (mv (erp-nil) t extended-alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                      ;; this assumption matched, but we couldn't relieve the rest of the hyps, so discard the extension of the alist and proceed to the next assumption:
                      (,relieve-free-var-hyp-and-all-others-name (rest assumption-arg-lists)
                                                                 hyp-args hyp-num other-hyps
                                                                 alist ;the original alist
                                                                 rule-symbol
                                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                 node-replacement-array node-replacement-count refined-assumption-alist
                                                                 rewrite-stobj (+ -1 count)))))))))

        ;; ALIST is the substitution alist so far (it maps vars in the rule to nodenums and quoteps). If alist doesn't bind all the variables in the
        ;; HYP, we'll search for free variable matches in REFINED-ASSUMPTION-ALIST.
        ;; Relieving the hyp through rewriting may cause more nodes to be added to the DAG and more things to be added to memoization, info, and tries.
        ;; BOZO precompute the list of vars in the hyp?  or maybe just the ones that need to be bound in the alist?
        ;; Returns (mv erp hyps-relievedp alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array), where alist is irrelevant if hyps-relievedp is nil.
        ;; Otherwise, the alist returned may have been extended by the binding of free vars.
        (defund ,relieve-rule-hyps-name (hyps hyp-num alist rule-symbol
                                              dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                              node-replacement-array node-replacement-count refined-assumption-alist
                                              rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (axe-rule-hyp-listp hyps)
                                      (posp hyp-num)
                                      (symbol-alistp alist)
                                      (bounded-darg-listp (strip-cdrs alist) dag-len)
                                      (alist-suitable-for-hypsp alist hyps)
                                      (symbolp rule-symbol)
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (info-worldp info)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                                                            (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :measure (nfix count)
                          :split-types t)
                   (type (unsigned-byte 60) count)
                   (type (integer 1 *) hyp-num) ;; restrict to a fixnum?
                   )
          (if (or (not (mbt (natp count)))
                  (= 0 count))
              (mv :count-exceeded t alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                  node-replacement-array)
            (if (endp hyps)
                ;; all hyps relieved:
                (mv (erp-nil) t alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
              (b* ((hyp (first hyps))
                   (print (get-print rewrite-stobj))
                   (fn (ffn-symb hyp)) ;; all hyps are conses
                   (- (and (eq :verbose! print)
                           (cw "Relieving hyp: ~x0 with alist ~x1.~%" hyp alist))))
                ;; todo: consider using CASE here:
                (if (eq :axe-syntaxp fn)
                    (let* ((syntaxp-expr (cdr hyp)) ;; strip off the :axe-syntaxp
                           (result (and
                                        (,eval-axe-syntaxp-expr-fn syntaxp-expr alist dag-array) ;could make a version without dag-array (may be very common?).. TODO: use :dag-array?
                                        )))
                      (if result
                          ;;this hyp counts as relieved
                          (,relieve-rule-hyps-name (rest hyps) (+ 1 hyp-num) alist rule-symbol
                                                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                   node-replacement-array node-replacement-count refined-assumption-alist
                                                   rewrite-stobj (+ -1 count))
                        (prog2$ (and (member-eq rule-symbol (get-monitored-symbols rewrite-stobj))
                                     ;;is it worth printing in this case?
                                     (progn$ (cw "(Failed to relieve axe-syntaxp hyp: ~x0 for ~x1.)~%" hyp rule-symbol)
                                             ;; (cw "(Alist: ~x0)~%" alist)
                                             ;; (cw "(DAG:~%")
                                             ;; (print-array2 'dag-array dag-array dag-len)
                                             ;; (cw ")~%")
                                             ))
                                (mv (erp-nil) nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))))
                  (if (eq :axe-bind-free fn)
                      ;; To evaluate the axe-bind-free hyp, we use alist, which binds vars to their nodenums or quoteps.
                      ;;The soundness of Axe should not depend on what an axe-bind-free function does; thus we cannot pass alist to such a function and trust it to faithfully extend it.  Nor can we trust it to extend the dag without changing any existing nodes. TODO: What if the axe-bind-free function gives back a result that is not even well-formed?
                      ;;TODO: It might be nice to be able to pass in the assumptions to the axe-bind-free-function? e.g., for finding usbp facts.
                      (let* ((bind-free-expr (cadr hyp)) ;; strip off the :axe-bind-free
                             (result (and ; (all-vars-in-terms-bound-in-alistp (fargs bind-free-expr) alist) ; TODO: remove this check, since it should be guaranteed statically!  need a better guards in the alist wrt future hyps
                                          (,eval-axe-bind-free-function-application-fn (ffn-symb bind-free-expr) (fargs bind-free-expr) alist dag-array) ;could make a version without dag-array (may be very common?).. TODO: use :dag-array?
                                          )))
                        (if result ;; nil to indicate failure, or an alist whose keys should be exactly (cddr hyp)
                            (let ((vars-to-bind (cddr hyp)))
                              (if (not (axe-bind-free-result-okayp result vars-to-bind dag-len))
                                  (mv (erp-t)
                                      (er hard? ',relieve-rule-hyps-name "Bind free hyp ~x0 for rule ~x1 returned ~x2, but this is not a well-formed alist that binds ~x3." hyp rule-symbol result vars-to-bind)
                                      alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                      node-replacement-array)
                                ;; this hyp counts as relieved:
                                (,relieve-rule-hyps-name (rest hyps) (+ 1 hyp-num)
                                                         (append result alist) ;; guaranteed to be disjoint given the analysis done when the rule was made and the call of axe-bind-free-result-okayp above
                                                         rule-symbol
                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                         rewrite-stobj (+ -1 count))))
                          ;; failed to relieve the axe-bind-free hyp:
                          (prog2$ (and (member-eq rule-symbol (get-monitored-symbols rewrite-stobj))
                                       (cw "(Failed to relieve axe-bind-free hyp: ~x0 for ~x1.)~%" hyp rule-symbol))
                                  (mv (erp-nil) nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))))
                    (if (eq :free-vars fn) ;can't be a work-hard since there are free vars
                        (b* (;; Partially instantiate the hyp (the free vars will remain free):
                             ;; TODO: Could we just do the matching wrt the alist, and skip this instantiation step?:
                             (partially-instantiated-hyp (,instantiate-hyp-free-vars-name (cdr hyp) ;strip the :free-vars
                                                                                alist (get-interpreted-function-alist rewrite-stobj)))
                             ((when (eq 'quote (ffn-symb partially-instantiated-hyp))) ;todo: this should not happen since there are free vars (unless perhaps we give special treatment to IFs)
                              (er hard? ',relieve-rule-hyps-name "ERROR: Instantiating a hyp with free vars produced a constant.")
                              (mv :error-instantiating nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                          ;; Some free vars remain in the partially-instantiated-hyp, so we search the REFINED-ASSUMPTION-ALIST for matches to bind them:
                          ;; If the NODE-REPLACEMENT-ARRAY ever contains more information than the REFINED-ASSUMPTION-ALIST, we might need to search it too.
                          ;; The refined-assumptions have been refined so that (equal (pred x) t) becomes (pred x) for better matching.
                          ;; TODO: Should we simplify the terms to which the free vars were bound (in case the assumptions are not simplified)?
                          (,relieve-free-var-hyp-and-all-others-name (lookup-in-refined-assumption-alist (ffn-symb partially-instantiated-hyp) refined-assumption-alist)
                                                                     (fargs partially-instantiated-hyp)
                                                                     hyp-num
                                                                     (rest hyps)
                                                                     alist rule-symbol
                                                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                     node-replacement-array node-replacement-count refined-assumption-alist
                                                                     rewrite-stobj (+ -1 count)))
                      ;; HYP is not a call to axe-syntaxp or axe-bind-free or :free-vars:
                      ;; First, we substitute in for all the vars in HYP (this also evaluates what it can):
                      (b* ((instantiated-hyp (,instantiate-hyp-no-free-vars2-name hyp alist (get-interpreted-function-alist rewrite-stobj))))
                        ;; instantiated-hyp is now fully instantiated.  It is an axe-tree with leaves that are quoteps and nodenums (from vars already bound):
                        (if (fquotep instantiated-hyp) ;; we know the instantiated-hyp is a cons, because hyp is
                            ;; The instantiated-hyp is a quoted constant:
                            (if (unquote instantiated-hyp)
                                ;; The instantiated-hyp is a non-nil constant, so it counts as relieved:
                                (,relieve-rule-hyps-name (rest hyps)
                                                         (+ 1 hyp-num)
                                                         alist
                                                         rule-symbol
                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                         rewrite-stobj (+ -1 count))
                              ;; The instantiated-hyp is 'nil, so it failed to be relieved:
                              (progn$ (and (member-eq rule-symbol (get-monitored-symbols rewrite-stobj))
                                           ;; We don't print much here, because a hyp that turns out to be nil (as opposed to some term for which we need a rewrite rule) is not very interesting.
                                           (cw "(Failed to relieve hyp ~x0 for ~x1.~% Reason: Instantiated to nil.~%" hyp rule-symbol))
                                      (mv (erp-nil) nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                          ;; There are no free vars, so we try to relieve the (fully-instantiated) hyp by simplifying it:
                          (b* ((old-try-count tries)
                               ((mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                    node-replacement-array)
                                ;; TODO: This tests whether atoms in the instiantiated-hyp are vars, but that seems wasteful because the hyp is fully instantiated):
                                ;; bozo do we really want to add stupid natp hyps, etc. to the memoization? what about ground terms (most of them will have been evaluated above)?
                                ;; TODO: Consider not instantiating the hyp but rather simplifying it wrt an alist:
                                (,simplify-tree-and-add-to-dag-name instantiated-hyp ; todo: is this known to be a non-var?  if so, take advantage of that fact
                                                                    nil ;nothing is yet known to be equal to instantiated-hyp
                                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                                    rewrite-stobj (+ -1 count)))
                               ((when erp) (mv erp nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                            (if (consp new-nodenum-or-quotep) ;tests for quotep
                                (if (unquote new-nodenum-or-quotep) ;the unquoted value is non-nil:
                                    (prog2$ (and old-try-count
                                                 print
                                                 (print-level-at-least-verbosep print)
                                                 (< 100 (sub-tries tries old-try-count))
                                                 (cw " (~x1 tries used ~x0:~x2 (rewrote to true).)~%" rule-symbol (sub-tries tries old-try-count) hyp-num))
                                            ;;hyp rewrote to a non-nil constant and so counts as relieved:
                                            (,relieve-rule-hyps-name (rest hyps)
                                                                     (+ 1 hyp-num)
                                                                     alist
                                                                     rule-symbol
                                                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                     node-replacement-array node-replacement-count refined-assumption-alist
                                                                     rewrite-stobj (+ -1 count)))
                                  ;;hyp rewrote to 'nil
                                  (progn$ (and old-try-count
                                               print
                                               (print-level-at-least-verbosep print)
                                               (< 100 (sub-tries tries old-try-count))
                                               (cw "(~x1 tries wasted ~x0:~x2 (rewrote to NIL))~%" rule-symbol (sub-tries tries old-try-count) hyp-num))
                                          (and (member-eq rule-symbol (get-monitored-symbols rewrite-stobj))
                                               ;; We don't print much here, because a hyp that turns out to be nil (as opposed to some term for which we need a rewrite rule) is not very interesting.
                                               (cw "(Failed to relieve hyp ~x0 for ~x1.~% Reason: Rewrote to nil.~%" hyp rule-symbol))
                                          (mv (erp-nil) nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                              ;;hyp didn't rewrite to a constant (new-nodenum-or-quotep is a node number):
                              ;; Check whether the rewritten hyp is one of the known assumptions (todo: would be better to rewrite it using IFF).  TODO: Do the other versions of the rewriter/prover do something like this?  Or enable the node-replacement-array to include the special symbol :non-nil and use it here:
                              (if ;;(nodenum-equal-to-refined-assumptionp new-nodenum-or-quotep refined-assumption-alist dag-array) ;todo: only do this if the hyp is not a known-boolean?
                                  (known-true-in-node-replacement-arrayp new-nodenum-or-quotep node-replacement-array node-replacement-count)
                                  (prog2$ (and old-try-count
                                               print
                                               (print-level-at-least-verbosep print)
                                               (< 100 (sub-tries tries old-try-count))
                                               (cw " (~x1 tries used ~x0:~x2 (rewrote to true).)~%" rule-symbol (sub-tries tries old-try-count) hyp-num))
                                          ;;hyp rewrote to a known assumption and so counts as relieved:
                                          (,relieve-rule-hyps-name (rest hyps)
                                                                   (+ 1 hyp-num)
                                                                   alist
                                                                   rule-symbol
                                                                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                   node-replacement-array node-replacement-count refined-assumption-alist
                                                                   rewrite-stobj (+ -1 count)))
                                ;; Failed to relieve the hyp:
                                (progn$ (and old-try-count
                                             print
                                             (print-level-at-least-verbosep print)
                                             (< 100 (sub-tries tries old-try-count))
                                             (cw "(~x1 tries wasted: ~x0:~x2 (non-constant result))~%" rule-symbol (sub-tries tries old-try-count) hyp-num))
                                        (and (member-eq rule-symbol (get-monitored-symbols rewrite-stobj))
                                             (progn$ (cw "(Failed to relieve hyp ~x0 of rule ~x1.~%" hyp rule-symbol)
                                                     (cw "Reason: Rewrote to:~%")
                                                     (print-dag-node-nicely new-nodenum-or-quotep 'dag-array dag-array dag-len 200)
                                                     (cw "(Alist: ~x0)~%(Refined assumption alist: ~x1)~%" alist refined-assumption-alist )
                                                     ;;print these better?:
                                                     ;;(cw "(node-replacement-array: ~x0)~%" node-replacement-array)
                                                     (cw "(DAG:~%")
                                                     (print-array2 'dag-array dag-array dag-len)
                                                     (cw "))~%")))
                                        (mv (erp-nil) nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))))))))))))))

        ;; Returns (mv erp instantiated-rhs-or-nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        (defund ,try-to-apply-rules-name (stored-rules ;the list of rules for the fn in question
                                          args-to-match
                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                          node-replacement-array node-replacement-count refined-assumption-alist
                                          rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (stored-axe-rule-listp stored-rules)
                                                                            (bounded-darg-listp args-to-match dag-len) ;todo: combine with the next one
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (info-worldp info)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (if (or (not (mbt (natp count)))
                  (= 0 count))
              (mv :count-exceeded nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                  node-replacement-array)
            (if (endp stored-rules) ;no rule fired
                (mv (erp-nil) nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
              (b* ((stored-rule (first stored-rules))
                   (print (get-print rewrite-stobj))
                   ((when (and limits (limit-reachedp stored-rule limits print)))
                    ;; the limit for this rule is reached, so try the next rule:
                    (,try-to-apply-rules-name
                     (rest stored-rules) args-to-match
                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                     node-replacement-array node-replacement-count refined-assumption-alist
                     rewrite-stobj
                     (+ -1 count)))
                   (tries (and tries (increment-tries tries)))
                   ;; Try to match the args-to-match with the args of the LHS of the rule:
                   (alist-or-fail (unify-terms-and-dag-items-fast (stored-rule-lhs-args stored-rule) args-to-match dag-array dag-len)))
                (if (eq :fail alist-or-fail)
                    ;; the rule didn't match, so try the next rule:
                    (,try-to-apply-rules-name
                     (rest stored-rules) args-to-match
                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                     node-replacement-array node-replacement-count refined-assumption-alist
                     rewrite-stobj
                     (+ -1 count))
                  ;; The rule matched, so try to relieve its hyps:
                  (b* ((- (and (eq print :verbose!)
                               (cw "(Trying to apply ~x0.~%" (stored-rule-symbol stored-rule))))
                       (hyps (stored-rule-hyps stored-rule))
                       ((mv erp hyps-relievedp alist ; may get extended by the binding of free vars
                            dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                            node-replacement-array)
                        (if hyps
                            (let ((rule-symbol (stored-rule-symbol stored-rule))) ;delay extracting this? not always needed?
                              (,relieve-rule-hyps-name hyps
                                                       1 ;initial hyp number
                                                       alist-or-fail
                                                       rule-symbol
                                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                       node-replacement-array node-replacement-count refined-assumption-alist
                                                       rewrite-stobj (+ -1 count)))
                          ;;if there are no hyps, don't even bother: BOZO inefficient?:
                          (mv (erp-nil) t alist-or-fail dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                       ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                    (if hyps-relievedp
                        ;; the hyps were relieved, so instantiate the RHS:
                        (prog2$ (and (eq print :verbose!)
                                     (cw "Rewriting with ~x0.)~%" (stored-rule-symbol stored-rule)))
                                (mv (erp-nil)
                                    ;; could use a faster version where we know there are no free vars:
                                    (,sublis-var-and-eval-name alist (stored-rule-rhs stored-rule) (get-interpreted-function-alist rewrite-stobj))
                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                    memoization
                                    ;;no need to assemble the info if we are not going to print it
                                    (and info (increment-hit-count-in-info-world (stored-rule-symbol stored-rule) info))
                                    tries
                                    (and limits (decrement-rule-limit stored-rule limits))
                                    node-replacement-array))
                      ;;failed to relieve the hyps, so try the next rule:
                      (prog2$ (and (eq print :verbose!)
                                   (cw "Failed to apply rule ~x0.)~%" (stored-rule-symbol stored-rule)))
                              (,try-to-apply-rules-name
                               (cdr stored-rules)
                               args-to-match
                               dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                               node-replacement-array node-replacement-count refined-assumption-alist
                               rewrite-stobj (+ -1 count))))))))))

        ;; Simplifies the application of FN to the ARGS where FN is a symbol and the ARGS are simplified.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; No special handling if FN is an IF (of any type).  No evaluation of ground terms.
        (defund ,simplify-fun-call-and-add-to-dag-name (fn ;;a function symbol
                                                        args ;these are simplified (so these are nodenums or quoteps)
                                                        trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to FN applied to ARGS (to be added to the memoization) ;todo: rename
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                        rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (symbolp fn)
                                      (not (equal 'quote fn))
                                      (bounded-darg-listp args dag-len)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (info-worldp info)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                                                            (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count)))
                (mv :count-exceeded nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               (expr (cons fn args)) ;todo: save this cons, or use below?
               ;;Try looking it up in the memoization (note that the args are now simplified):
               (memo-match (and memoization (lookup-in-memoization expr memoization))) ; todo: use a more specialized version of lookup-in-memoization, since we know the shape of expr (also avoid the cons for expr here)?
               ((when memo-match)
                (mv (erp-nil) memo-match
                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                    (add-pairs-to-memoization trees-equal-to-tree
                                              memo-match ;the nodenum or quotep they are all equal to
                                              memoization)
                    info tries limits
                    node-replacement-array))
               ;; Next, try to apply rules:
               ((mv erp rhs-or-nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,try-to-apply-rules-name (get-rules-for-fn fn (get-rule-alist rewrite-stobj))
                                          args
                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                          node-replacement-array node-replacement-count refined-assumption-alist
                                          rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
            (if rhs-or-nil
                ;;A rule fired, so simplify the instantiated right-hand-side:
                ;; This is a tail call, which allows long chains of rewrites:
                (,simplify-tree-and-add-to-dag-name rhs-or-nil
                                                    ;;in the common case in which simplifying the args had no effect, the car of trees-equal-to-tree will be the same as (cons fn args), so don't add it twice
                                                    (and memoization
                                                         (cons-if-not-equal-car expr ;could save this and similar conses in the function
                                                                                trees-equal-to-tree))
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count))
              ;; No rule fired, so no simplification can be done.  Add the expression to the dag, but perhaps normalize nests of certain functions:
              (b* (((mv erp nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
                    (if (get-normalize-xors rewrite-stobj)
                        (add-and-normalize-expr fn args ; can we often save consing FN onto ARGS in this?
                                                dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
                      (mv-let (erp nodenum dag-array dag-len dag-parent-array dag-constant-alist)
                        (add-function-call-expr-to-dag-array fn args ;(if any-arg-was-simplifiedp (cons fn args) tree) ;could put back the any-arg-was-simplifiedp trick to save this cons
                                                             dag-array dag-len dag-parent-array dag-constant-alist)
                        (mv erp nodenum dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist))))
                   ((when erp) (mv erp nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                   ;; See if the nodenum returned is equated to anything:
                   ;; Result is not rewritten (we could rewrite all such items (that replacements can introduce) outside the main clique)
                   (new-nodenum-or-quotep (if (consp nodenum-or-quotep) ; check for constant (e.g., if all xors cancelled)
                                              nodenum-or-quotep
                                            (apply-node-replacement-array nodenum-or-quotep node-replacement-array node-replacement-count))))
                (mv (erp-nil)
                    new-nodenum-or-quotep
                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                    (and memoization ; we could save this cons:
                         (add-pairs-to-memoization (cons-if-not-equal-car expr trees-equal-to-tree) ; might be the same as tree if the args aren't simplified?) well, each arg should be simplified and memoed.
                                                   new-nodenum-or-quotep ;the nodenum-or-quotep they are all equal to
                                                   memoization))
                    info tries limits node-replacement-array)))))

        ;; Helper function for rewriting a tree that is an IF or MYIF or BOOLIF (used for both if/myif and boolif).  This is separate just to keep the caller small.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; Note that this function does not return node-replacement-count since no nodes have been assumed that are relevant to the caller.
        ;; TODO: Are all elements in the array beyond node-replacement-count nil?
        (defund ,simplify-if/myif/boolif-tree-and-add-to-dag3-name (fn
                                                                    simplified-test ; a nodenum
                                                                    simplified-then-branch
                                                                    else-branch ; unsimplified
                                                                    tree
                                                                    trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                                    rewrite-stobj count)
          (declare (xargs :guard (and (member-eq fn '(if myif boolif))
                                      (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (natp simplified-test)
                                      (< simplified-test dag-len)
                                      (dargp-less-than simplified-then-branch dag-len)
                                      (axe-treep else-branch)
                                      (bounded-axe-treep else-branch dag-len)
                                      (tree-to-memoizep tree)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ;; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Now we rewrite the else-branch.
               ;; Assume the test false (if not memoizing), step 1:
               ((mv node-replacement-array node-replacement-count undo-pairs)
                (if memoization ; can't use context if we are memoizing:
                    (mv node-replacement-array node-replacement-count nil)
                  (update-node-replacement-array-for-assuming-negation-of-node simplified-test node-replacement-array node-replacement-count dag-array dag-len (get-known-booleans rewrite-stobj))))
               ;; Assume the test false (if not memoizing), step 2:
               (refined-assumption-alist-for-else-branch
                 (if memoization ; can't use context if we are memoizing:
                     refined-assumption-alist
                   (extend-refined-assumption-alist-assuming-negation-of-node refined-assumption-alist simplified-test dag-array dag-len)))
               ;; Rewrite the else-branch:
               ((mv erp simplified-else-branch dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name else-branch
                                                    nil ;no trees are yet known equal to the else branch
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist-for-else-branch
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Special treatment for BOOLIF: Handle a simplified-else-branch that is known to be :non-nil as if it were 't:
               (simplified-else-branch (if (eq 'boolif fn)
                                           (apply-node-replacement-array-bool-to-darg simplified-else-branch node-replacement-array node-replacement-count)
                                         simplified-else-branch))
               ;; Clear the test assumption. node-replacement-array should then be
               ;; like it was before we set it (except perhaps longer):
               ;; If memoizing, undo-pairs will be nil:
               (node-replacement-array (undo-writes-to-node-replacement-array undo-pairs node-replacement-array node-replacement-count dag-len)))
            ;; Now simplify the call of IF/MYIF/BOOLIF (this function takes simplified args and does not handle ifs specially, or else things might loop):
            ;; (We know we don't have a ground term, because simplified-test is not a constant.)
            (,simplify-fun-call-and-add-to-dag-name fn (list simplified-test simplified-then-branch simplified-else-branch)
                                                    (and memoization (cons tree trees-equal-to-tree)) ;the call of FN we are rewriting here is equal to tree
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count))))

        ;; Helper function for rewriting a tree that is an IF or MYIF or BOOLIF (used for both if/myif and boolif).
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; Note that this function does not return node-replacement-count since no changes have been made that are relevant to the caller.
        (defund ,simplify-if/myif/boolif-tree-and-add-to-dag2-name (fn ; if or myif or boolif
                                                                    simplified-test ; a nodenum
                                                                    then-branch ; to be simplified
                                                                    else-branch ; to be simplified
                                                                    tree ; original tree, to be added to the memoization
                                                                    trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                                    rewrite-stobj count)
          (declare (xargs :guard (and (member-eq fn '(if myif boolif))
                                      (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (natp simplified-test)
                                      (< simplified-test dag-len)
                                      (axe-treep then-branch)
                                      (bounded-axe-treep then-branch dag-len)
                                      (axe-treep else-branch)
                                      (bounded-axe-treep else-branch dag-len)
                                      (tree-to-memoizep tree)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ;; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; First, we rewrite the then-branch.
               ;; Assume the test true (if not memoizing), step 1:
               ((mv node-replacement-array node-replacement-count undo-pairs)
                (if memoization ; can't use context if we are memoizing:
                    (mv node-replacement-array node-replacement-count nil)
                  (update-node-replacement-array-for-assuming-node simplified-test node-replacement-array node-replacement-count dag-array dag-len (get-known-booleans rewrite-stobj))))
               ;; Assume the test true (if not memoizing), step 2:
               (refined-assumption-alist-for-then-branch
                 (if memoization ; can't use context if we are memoizing:
                     refined-assumption-alist
                   (extend-refined-assumption-alist-assuming-node refined-assumption-alist simplified-test dag-array dag-len)))
               ;; Rewrite the then-branch:
               ((mv erp simplified-then-branch dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name then-branch
                                                    nil ;no trees are yet known equal to the then-branch
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist-for-then-branch
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Special treatment for BOOLIF: Handle a simplified-then-branch that is known to be :non-nil as if it were 't:
               (simplified-then-branch (if (eq 'boolif fn)
                                           (apply-node-replacement-array-bool-to-darg simplified-then-branch node-replacement-array node-replacement-count)
                                         simplified-then-branch))
               ;; Undo the assumption of the test being true.  node-replacement-array should then be
               ;; like it was before we assumed the test (except perhaps longer):
               ;; If memoizing, undo-pairs will be nil:
               (node-replacement-array (undo-writes-to-node-replacement-array undo-pairs node-replacement-array node-replacement-count dag-len)))
            ;; Continue rewriting the IF/MYIF/BOOLIF:
            (,simplify-if/myif/boolif-tree-and-add-to-dag3-name fn
                                                                simplified-test ; a nodenum
                                                                simplified-then-branch ; a nodenum or quotep
                                                                else-branch ; unsimplified
                                                                tree
                                                                trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                                dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                node-replacement-array node-replacement-count refined-assumption-alist
                                                                rewrite-stobj (+ -1 count))))

        ;; Rewrite a tree that is an IF or MYIF.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; This is separate just to keep the main function small.
        (defund ,simplify-if/myif-tree-and-add-to-dag-name (tree ; a call of IF or MYIF
                                                            trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                            dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                            node-replacement-array node-replacement-count refined-assumption-alist
                                                            rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (axe-treep tree)
                                      (bounded-axe-treep tree dag-len)
                                      (consp tree)
                                      (member-eq (ffn-symb tree) '(if myif))
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len) ;todo: name these 3?
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ;; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               (args (fargs tree))
               ((when (not (consp (rest (rest args))))) ;; for guards
                (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; First, simplify the test (TODO: would like to do this in an iff context):
               ((mv erp simplified-test dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name (first args) ; the test of the IF or MYIF
                                                    nil ;no trees are yet known equal to the test
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Special treatment for IF/MYIF: Handle a simplified-test that is known to be :non-nil as if it were 't:
               (simplified-test (apply-node-replacement-array-bool-to-darg simplified-test node-replacement-array node-replacement-count)))
            (if (consp simplified-test) ; tests for quotep (that is, checks whether we resolved the test)
                ;; Rewrite either the "then" branch or the "else" branch, according to whether the test simplified to nil:
                (,simplify-tree-and-add-to-dag-name (if (unquote simplified-test) (second args) (third args))
                                                    (and memoization (cons tree trees-equal-to-tree)) ;the thing we are rewriting here is equal to tree
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count))
              ;; Failed to resolve the test:
              (progn$
               ;; If this gets printed too often for known predicates, we can preprocess such things:
               ;; (and (equal test (second args)) (cw "Unresolved IF test with test same as then-branch (from an OR?): ~x0.~%" test))
               (,simplify-if/myif/boolif-tree-and-add-to-dag2-name (ffn-symb tree) ; if or myif
                                                                   simplified-test
                                                                   (second args) ; "then" branch
                                                                   (third args) ; "else" branch
                                                                   tree trees-equal-to-tree ; could cons these and pass them together (they will be consed later)
                                                                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                   node-replacement-array node-replacement-count refined-assumption-alist
                                                                   rewrite-stobj (+ -1 count))))))

        ;; Rewrite a tree that is a BOOLIF.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; This is separate just to keep the main function small
        (defund ,simplify-boolif-tree-and-add-to-dag-name (tree ; a call of BOOLIF
                                                           trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                           dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                           node-replacement-array node-replacement-count refined-assumption-alist
                                                           rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (axe-treep tree)
                                      (bounded-axe-treep tree dag-len)
                                      (consp tree)
                                      (equal 'boolif (ffn-symb tree))
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count)))
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               (args (fargs tree))
               ((when (not (consp (rest (rest args))))) ;; for guards
                (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; First, simplify the test (TODO: would like to do this in an iff context):
               ((mv erp simplified-test dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name (first args) ; the test of the BOOLIF
                                                    nil ;no trees are yet known equal to the test
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Special treatment for BOOLIF: Handle a simplified-test that is known to be :non-nil as if it were 't:
               ;; skip this if the test is a known boolean??
               (simplified-test (apply-node-replacement-array-bool-to-darg simplified-test node-replacement-array node-replacement-count)))
            (if (consp simplified-test) ; tests for quotep (that is, checks whether we resolved the test)
                ;; Rewrite either the "then" branch or the "else" branch, according to whether the test simplified to nil, wrapping the result in bool-fix:
                ;; TODO: Consider dropping the bool-fix if we have a known boolean:
                (,simplify-tree-and-add-to-dag-name `(bool-fix$inline ,(if (unquote simplified-test) (second args) (third args))) ;the "then" branch or the "else" branch
                                                    (and memoization (cons tree trees-equal-to-tree)) ;the bool-fix$inline tree we are rewriting here is equal to TREE
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count))
              ;; Failed to resolve the test (from here on, the process is mostly the same as for IF/MYIF except we pass BOOLIF as the FN):
              (,simplify-if/myif/boolif-tree-and-add-to-dag2-name 'boolif
                                                                  simplified-test
                                                                  (second args)
                                                                  (third args)
                                                                  tree trees-equal-to-tree ; could cons these and pass them together (they will be consed later)
                                                                  dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                  node-replacement-array node-replacement-count refined-assumption-alist
                                                                  rewrite-stobj (+ -1 count)))))

        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; This is separate just to keep the proofs tractable (avoid too many sequential rewriter calls in one function).
        (defund ,simplify-bvif-tree-and-add-to-dag3-name (simplified-size
                                                          simplified-test ; a nodenum
                                                          simplified-then-branch
                                                          else-branch ; to be simplified
                                                          tree ; original BVIF tree
                                                          trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (dargp-less-than simplified-size dag-len)
                                      (natp simplified-test)
                                      (< simplified-test dag-len)
                                      (dargp-less-than simplified-then-branch dag-len)
                                      (axe-treep else-branch)
                                      (bounded-axe-treep else-branch dag-len)
                                      (tree-to-memoizep tree)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Now we rewrite the else-branch.
               ;; Assume the test false (if not memoizing), step 1:
               ((mv node-replacement-array node-replacement-count undo-pairs)
                (if memoization ; can't use context if we are memoizing:
                    (mv node-replacement-array node-replacement-count nil)
                  (update-node-replacement-array-for-assuming-negation-of-node simplified-test node-replacement-array node-replacement-count dag-array dag-len (get-known-booleans rewrite-stobj))))
               ;; Assume the test false (if not memoizing), step 2:
               (refined-assumption-alist-for-else-branch
                 (if memoization ; can't use context if we are memoizing:
                     refined-assumption-alist
                   (extend-refined-assumption-alist-assuming-negation-of-node refined-assumption-alist simplified-test dag-array dag-len)))
               ;; Simplify the else-branch
               ((mv erp simplified-else-branch dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name else-branch
                                                    nil ;no trees are yet known equal to the else branch
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist-for-else-branch
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Clear the test assumption. node-replacement-array should then be
               ;; like it was before we set it (except perhaps longer):
               ;; If memoizing, undo-pairs will be nil:
               (node-replacement-array (undo-writes-to-node-replacement-array undo-pairs node-replacement-array node-replacement-count dag-len)))
            ;; Try to apply rules to the call of BVIF on the simplified args:
            (,simplify-fun-call-and-add-to-dag-name 'bvif (list simplified-size simplified-test simplified-then-branch simplified-else-branch)
                                                    (and memoization (cons tree trees-equal-to-tree)) ; the BVIF call we are rewriting here is equal to TREE
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count))))

        ;; Continue rewriting a call of BVIF.  This is for the case where we cannot resolve the test.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; This is separate just to keep the proofs tractable (avoid too many sequential rewriter calls in one function).
        (defund ,simplify-bvif-tree-and-add-to-dag2-name (simplified-size
                                                          simplified-test ; a nodenum
                                                          then-arg ; unsimplified
                                                          else-arg ; unsimplified
                                                          tree ; original BVIF tree
                                                          trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (dargp-less-than simplified-size dag-len)
                                      (natp simplified-test)
                                      (< simplified-test dag-len)
                                      (axe-treep then-arg)
                                      (bounded-axe-treep then-arg dag-len)
                                      (axe-treep else-arg)
                                      (bounded-axe-treep else-arg dag-len)
                                      (tree-to-memoizep tree)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Next, we rewrite the then-branch.
               ;; Assume the test true (if not memoizing), step 1:
               ((mv node-replacement-array node-replacement-count undo-pairs)
                (if memoization ; can't use context if we are memoizing:
                    (mv node-replacement-array node-replacement-count nil)
                  (update-node-replacement-array-for-assuming-node simplified-test node-replacement-array node-replacement-count dag-array dag-len (get-known-booleans rewrite-stobj))))
               ;; Assume the test true (if not memoizing), step 2:
               (refined-assumption-alist-for-then-branch
                 (if memoization ; can't use context if we are memoizing:
                     refined-assumption-alist
                   (extend-refined-assumption-alist-assuming-node refined-assumption-alist simplified-test dag-array dag-len)))
               ;; Simplify the then-branch:
               ((mv erp simplified-then-branch dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name then-arg
                                                    nil ;no trees are yet known equal to the then-branch
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist-for-then-branch
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Undo the assumption of the test being true.  node-replacement-array should then be
               ;; like it was before we assumed the test (except perhaps longer):
               ;; If memoizing, undo-pairs will be nil:
               (node-replacement-array (undo-writes-to-node-replacement-array undo-pairs node-replacement-array node-replacement-count dag-len)))
            (,simplify-bvif-tree-and-add-to-dag3-name simplified-size
                                                      simplified-test
                                                      simplified-then-branch
                                                      else-arg ; unsimplified
                                                      tree ; original bvif tree (bvif applied to the unsimplified args)
                                                      trees-equal-to-tree
                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                      rewrite-stobj (+ -1 count))))

        ;; Continue rewriting a call of BVIF.  This is for the case where we cannot resolve the test.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; This is separate just to keep the proofs tractable (avoid too many sequential rewriter calls in one function).
        ;; todo: renumber functions so this one is number 2
        (defund ,simplify-bvif-tree-and-add-to-dag1-name (size-arg ; unsimplified
                                                          simplified-test ; a nodenum
                                                          then-arg ; unsimplified
                                                          else-arg ; unsimplified
                                                          tree ; original BVIF tree
                                                          trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (axe-treep size-arg)
                                      (bounded-axe-treep size-arg dag-len)
                                      (natp simplified-test)
                                      (< simplified-test dag-len)
                                      (axe-treep then-arg)
                                      (bounded-axe-treep then-arg dag-len)
                                      (axe-treep else-arg)
                                      (bounded-axe-treep else-arg dag-len)
                                      (tree-to-memoizep tree)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Simplify the size param:
               ((mv erp simplified-size dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name size-arg
                                                    nil ;no trees are yet known equal to the the size param
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
            (,simplify-bvif-tree-and-add-to-dag2-name simplified-size
                                                      simplified-test
                                                      then-arg ; unsimplified
                                                      else-arg ; unsimplified
                                                      tree ; original BVIF tree
                                                      trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                      rewrite-stobj (+ -1 count))))

        ;; Rewrites TREE, which must be of the form (bvif <size> <test> <then> <else>).
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; This is separate just to keep the main function small.
        (defund ,simplify-bvif-tree-and-add-to-dag-name (tree ; a call of BVIF
                                                         trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                         rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (axe-treep tree)
                                      (bounded-axe-treep tree dag-len)
                                      (consp tree)
                                      (equal 'bvif (ffn-symb tree))
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               (args (fargs tree))
               ((when (not (consp (rest (rest (rest args)))))) ;; check arity, for guards
                (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; First, simplify the test of the BVIF:
               ((mv erp simplified-test dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name (second args) ;the test of the BVIF
                                                    nil ;no trees are yet known equal to the test
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Special treatment for BVIF: Handle a simplified-test that is known to be :non-nil as if it were 't:
               (simplified-test (apply-node-replacement-array-bool-to-darg simplified-test node-replacement-array node-replacement-count)))
            (if (consp simplified-test) ; tests for quotep
                ;; test was resolved (todo: do better here, if already a bv??):
                (,simplify-tree-and-add-to-dag-name `(bvchop
                                                       ,(first args) ; size arg
                                                       ,(if (unquote simplified-test)
                                                            ;; test rewrote to non-nil:
                                                            (third args) ; "then" branch
                                                          ;; test rewrote to nil:
                                                          (fourth args) ; "else" branch
                                                          ))
                                                    (and memoization (cons tree trees-equal-to-tree)) ;the thing we are rewriting here is equal to tree
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count))
              ;;couldn't resolve the test:
              (,simplify-bvif-tree-and-add-to-dag1-name (first args) ; size param
                                                        simplified-test
                                                        (third args) ; then-branch
                                                        (fourth args) ; else-branch
                                                        tree trees-equal-to-tree
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                        rewrite-stobj (+ -1 count)))))

        ;; Rewrites a tree that is a NOT.
        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        (defund ,simplify-not-tree-and-add-to-dag-name (tree ; a call of NOT
                                                        trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                        rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (axe-treep tree)
                                      (bounded-axe-treep tree dag-len)
                                      (consp tree)
                                      (equal 'not (ffn-symb tree))
                                      (tree-to-memoizep tree)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (b* (((when (or (not (mbt (natp count))) (= 0 count))) ; for termination
                (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               (args (fargs tree))
               ((when (not (consp args))) ; for guards
                (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; First, rewrite the arg:
               ((mv erp simplified-arg dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                (,simplify-tree-and-add-to-dag-name (first args) ;the single arg
                                                    nil ;no trees are yet known equal to the test
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj (+ -1 count)))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Special treatment for NOT: Treat any non-nil arg like t:
               (simplified-arg (apply-node-replacement-array-bool-to-darg simplified-arg node-replacement-array node-replacement-count)))
            (if (consp simplified-arg) ; tests for quotep
                (mv (erp-nil)
                    (if (unquote simplified-arg) *nil* *t*) ; negate since we are rewriting a NOT
                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                    node-replacement-array)
              ;; Arg did not rewrite to a constant, so try to apply rules to the call of NOT on the simplified arg:
              (,simplify-fun-call-and-add-to-dag-name 'not (list simplified-arg)
                                                      (and memoization (cons tree trees-equal-to-tree)) ;the thing we are rewriting here is equal to tree
                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                      rewrite-stobj (+ -1 count)))))

        ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;; Rewrite TREE repeatedly using RULE-ALIST (from rewrite-stobj) and REFINED-ASSUMPTION-ALIST and add the result to the DAG-ARRAY (if not a quote), returning a nodenum or a quotep.
        ;; TREE is an axe-tree (with nodenums and quoteps and variables at the leaves).
        ;; TREES-EQUAL-TO-TREE is a list of terms (not vars) equal to TREE (at the bottom is the original term we are rewriting) - when we get the final result, all these terms will be equal to it - TODO: option to turn this off?

        ;; All nodenums mentioned in the MEMOIZATION must be part of the DAG.
        ;; If we are memoizing, we don't use information from if tests when rewriting then-branches and else-branches (potentially unsound).

        ;; This does not return node-replacement-count, because although some node replacement entries may be set (when processing IFs), they should all be cleared again.

        ;;leaves nodes below dag-len untouched..
        ;;TODO: could put in simple loop checking; check whether TREE is already present in TREES-EQUAL-TO-TREE (maybe only check the first few elements), but TREES-EQUAL-TO-TREE may only be valid if we are memoizing.
        (defund ,simplify-tree-and-add-to-dag-name (tree
                                                    trees-equal-to-tree ;a list of the successive RHSes, all of which are equivalent to tree (to be added to the memoization)
                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                    rewrite-stobj count)
          (declare (xargs :guard (and (axe-treep tree)
                                      (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (bounded-axe-treep tree dag-len)
                                      (trees-to-memoizep trees-equal-to-tree)
                                      (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (info-worldp info)
                                                                            (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :split-types t
                          :measure (nfix count))
                   (type (unsigned-byte 60) count))
          (if (or (not (mbt (natp count))) (= 0 count))
              (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
            (if (atom tree)
                (if (symbolp tree)
                    ;; It's a variable (this case may be very rare; can we eliminate it by pre-handling vars in the initial term?):
                    (b* ( ;; Add it to the DAG:
                         ((mv erp nodenum dag-array dag-len dag-parent-array dag-variable-alist)
                          (add-variable-to-dag-array tree dag-array dag-len dag-parent-array dag-variable-alist))
                         ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                         ;; See if the resulting node is known to be equal to something:
                         (new-nodenum-or-quotep (apply-node-replacement-array nodenum node-replacement-array node-replacement-count)))
                      (mv (erp-nil)
                          new-nodenum-or-quotep
                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                          (and memoization
                               (add-pairs-to-memoization trees-equal-to-tree ;the items (TODO: Can this be non-empty?) ; We cannot add a memoization entry for TREE itself, because it is not a function call.
                                                         new-nodenum-or-quotep ;the nodenum-or-quotep they are all equal to
                                                         memoization))
                          info tries limits
                          node-replacement-array))
                  ;; TREE is a nodenum (because it's an atom but not a symbol): TODO: use equalities?
                  ;; TODO: replacement works best if the nodes in the dag are already rewritten.  but what if this nodenum came from a node-equality assumption? in that case, it may not be rewritten! should we simplify the entries in the node-replacement-array once at the beginning?
                  ;; First, see if the nodenum is mapped to anything in the node-replacement-count:
                  (let* ((replacement-match nil ;(assoc-in-node-replacement-count tree node-replacement-count)
                                            )
                         (tree (if (and replacement-match
                                        ;; This is new (8/10/15), so to be safe we only do it if the result is a constant (I am worried about loops):
                                        (quotep (cdr replacement-match)))
                                   (cdr replacement-match)
                                 ;; No match found in the node-replacement-alist, so just use tree:
                                 tree)))
                    (mv (erp-nil)
                        tree
                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                        (if (and memoization
                                 ;; todo: drop this check?:
                                 trees-equal-to-tree ; could check just this, but then it *must* always be nil if we are not memoizing
                                 )
                            (add-pairs-to-memoization trees-equal-to-tree ; We cannot add a memoization entry for TREE itself, because it is not a function call.
                                                      tree ; the nodenum to which all the TREES-EQUAL-TO-TREE rewrote
                                                      memoization)
                          memoization)
                        info tries limits
                        node-replacement-array)))
              ;; TREE is a cons:
              (let ((fn (ffn-symb tree)))
                (if (eq 'quote fn)
                    ;; TREE is a quoted constant:
                    (mv (erp-nil)
                        tree ; return the quoted constant
                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                        (if (and memoization trees-equal-to-tree)
                            (add-pairs-to-memoization trees-equal-to-tree ; We cannot add a memoization entry for TREE itself, because it is not a function call.
                                                      tree ; the constant to which all the TREES-EQUAL-TO-TREE rewrote
                                                      memoization)
                          memoization)
                        info tries limits
                        node-replacement-array)
                  ;; TREE is a function call:
                  ;; Try looking it up in memoization (note that the args are not yet simplified):
                  (let ((memo-match (and memoization (lookup-in-memoization tree memoization))))
                    (if memo-match
                        (mv (erp-nil)
                            memo-match ; a darg
                            dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                            (add-pairs-to-memoization trees-equal-to-tree ; We don't add a memoization entry for TREE itself, because it already has one.
                                                      memo-match ;the nodenum or quotep to which all the TREES-EQUAL-TO-TREE rewrote
                                                      memoization)
                            info tries limits
                            node-replacement-array)
                      ;; Handle NOT and the various kinds of IF:
                      (case fn
                        (not
                         (,simplify-not-tree-and-add-to-dag-name tree trees-equal-to-tree
                                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                 node-replacement-array node-replacement-count refined-assumption-alist
                                                                 rewrite-stobj
                                                                 (+ -1 count) ;could perhaps be avoided with a more complex measure
                                                                 ))
                        ((if myif)
                         (,simplify-if/myif-tree-and-add-to-dag-name tree trees-equal-to-tree
                                                                 dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                 node-replacement-array node-replacement-count refined-assumption-alist
                                                                 rewrite-stobj
                                                                 (+ -1 count) ;could perhaps be avoided with a more complex measure
                                                                 ))
                        (boolif (,simplify-boolif-tree-and-add-to-dag-name tree trees-equal-to-tree
                                                                            dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                            node-replacement-array node-replacement-count refined-assumption-alist
                                                                            rewrite-stobj
                                                                            (+ -1 count) ;could perhaps be avoided with a more complex measure
                                                                            ))
                        (bvif (,simplify-bvif-tree-and-add-to-dag-name tree trees-equal-to-tree
                                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                                        rewrite-stobj
                                                                        (+ -1 count) ;could perhaps be avoided with a more complex measure
                                                                        ))
                        (t
                         ;; It wasn't a NOT or any kind of IF:
                         (b* (;; We are simplifying a call of FN on args that are axe-trees.
                              ;; First we simplify the args:
                              ;; TODO: Should we simplify the args earlier? e.g., before looking the term up in the memoization?
                              ;; TODO: might it be possible to not check for ground-terms because we never build them? -- think about where terms might come from other than sublis-var-simple, which we could change to not build ground terms (of functions we know about)
                              ;; TODO: maybe we should try to apply rules here (maybe outside-in rules) instead of rewriting the args
                              ;; TODO: could pass in a flag for the common case where the args are known to be already simplified (b/c the tree is a dag node?)
                              ;; (- (and (eq :verbose! print) (cw "(Rewriting args of ~x0:~%" fn)))
                              ((mv erp simplified-args dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                   node-replacement-array)
                               (,simplify-trees-and-add-to-dag-name (fargs tree)
                                                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                    node-replacement-array node-replacement-count refined-assumption-alist
                                                                    rewrite-stobj (+ -1 count)))
                              ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                              ;; (- (and (eq :verbose! print) (cw "Done rewriting args.)~%")))
                              )
                           ;; SIMPLIFIED-ARGS is now a list of dargs (nodenums and quoteps). Now we simplify FN applied to SIMPLIFIED-ARGS:
                           (if (consp fn) ;;tests for lambda
                               ;; It's a lambda, so we beta-reduce and simplify the result:
                               ;; note that we don't look up lambdas in the assumptions (this is consistent with simplifying first)
                               ;; TODO: It's possible that we wasted time above simplifying lambda args (some args may be unneeded if the lambda body is a resolvable if), but
                               ;; I'm not sure how to prevent that (we want to simplify args first because they may be mentioned multiple times in the lambda-body).
                               (let* ((formals (lambda-formals fn))
                                      (body (lambda-body fn))
                                      ;; TTODO: could optimize this pattern: (,sublis-var-and-eval-name (pairlis$-fast formals simplified-args) body ...) -- see subcor-var.
                                      (new-expr (,sublis-var-and-eval-name (pairlis$-fast formals simplified-args) body (get-interpreted-function-alist rewrite-stobj))))
                                 ;;simplify the result of beta-reducing:
                                 (,simplify-tree-and-add-to-dag-name new-expr
                                                                     (and memoization (cons tree trees-equal-to-tree)) ;we memoize the lambda
                                                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                     node-replacement-array node-replacement-count refined-assumption-alist
                                                                     rewrite-stobj (+ -1 count)))
                             ;; not a lambda:
                             (b* (;; handle possible ground term by evaluating (since ,simplify-fun-call-and-add-to-dag-name doesn't handle ground terms):
                                  ;; todo: this code is essentially duplicated in ,simplify-dag-aux-name:
                                  ((mv erp evaluatedp val)
                                   (if (not (all-consp simplified-args)) ;; test for all quoted constants
                                       ;; not a ground term:
                                       (mv (erp-nil) nil nil)
                                     ;; ground term, so try to evaluate (may fail, but we may have a constant opener rule to apply later):
                                     (b* (((mv erp val)
                                           (,apply-axe-evaluator-to-quoted-args-name fn simplified-args (get-interpreted-function-alist rewrite-stobj))))
                                       (if erp
                                           (if (call-of :unknown-function erp) ; I suppose this could also be a bad arity?
                                               (mv (erp-nil) nil nil) ;no error, but it didn't produce a value (todo: print a warning?)
                                             ;; anything else non-nil is a true error:
                                             (mv erp nil nil))
                                         ;; normal case:
                                         (mv (erp-nil) t val)))))
                                  ;; I suppose we could suppress any evaluator error here if we choose to (might be a bit faster)?
                                  ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                               (if evaluatedp
                                   (let ((quoted-val (enquote val)))
                                     (mv (erp-nil)
                                         quoted-val
                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                         (and memoization
                                              (add-pairs-to-memoization (cons ;todo: we could avoid this cons
                                                                          tree ;; the original tree, with unsimplified args (might be ground, but that might be rare)
                                                                          ;; should we include here fn applied to simplified-args? could use a separate memoization for ground terms
                                                                          trees-equal-to-tree)
                                                                        quoted-val ;the quoted constant to which TREE and all the TREES-EQUAL-TO-TREE rewrote
                                                                        memoization))
                                         info tries limits
                                         node-replacement-array))
                                 ;; Otherwise, simplify the non-lambda FN applied to the simplified args:
                                 ;; TODO: Perhaps pass in the original tree for use by cons-with-hint:
                                 (,simplify-fun-call-and-add-to-dag-name fn simplified-args
                                                                         (and memoization (cons tree trees-equal-to-tree)) ;the thing we are rewriting is equal to tree
                                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                                         rewrite-stobj (+ -1 count)))))))))))))))

        ;; Simplifies all the trees in TREES.
        ;; Returns (mv erp nodenums-or-quoteps dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
        ;;if the items in trees are already all nodenums or quoted constants this doesn't re-cons-up the list
        (defund ,simplify-trees-and-add-to-dag-name (trees
                                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                     node-replacement-array node-replacement-count refined-assumption-alist
                                                     rewrite-stobj count)
          (declare (xargs :guard (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                      (bounded-axe-tree-listp trees dag-len)
                                      (maybe-bounded-memoizationp memoization dag-len)
                                      (info-worldp info)
                                      (triesp tries)
                                      (rule-limitsp limits)
                                      (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                      (natp node-replacement-count)
                                      (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                                                            (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                      (unsigned-byte-p 60 count))
                          :stobjs rewrite-stobj
                          :measure (nfix count)
                          :split-types t)
                   (type (unsigned-byte 60) count))
          (if (or (not (mbt (natp count)))
                  (= 0 count))
              (mv :count-exceeded trees dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
            (if (atom trees)
                (mv (erp-nil) trees dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
              (b* ((first-tree (first trees))
                   (rest (rest trees))
                   ;; why do we handle the rest first?
                   ((mv erp rest-result dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                        node-replacement-array)
                    (,simplify-trees-and-add-to-dag-name rest
                                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                         node-replacement-array node-replacement-count refined-assumption-alist
                                                         rewrite-stobj (+ -1 count)))
                   ((when erp) (mv erp trees dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                   ((mv erp first-tree-result dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                    (,simplify-tree-and-add-to-dag-name first-tree
                                                        nil ;; nothing is yet known equal to first-tree
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                        rewrite-stobj (+ -1 count)))
                   ((when erp) (mv erp trees dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                (mv (erp-nil)
                    (cons-with-hint first-tree-result rest-result trees)
                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))))

        ) ;end mutual-recursion

       ;; Theorems about when the count reaches 0:
       (progn
         (defthm ,(pack$ 'relieve-free-var-hyp-and-all-others- suffix '-of-0)
           (equal ,(subst 0 'count call-of-relieve-free-var-hyp-and-all-others)
                  (mv :count-exceeded nil alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-relieve-free-var-hyp-and-all-others)))))

         (defthm ,(pack$ 'relieve-rule-hyps- suffix '-of-0)
           (equal ,(subst 0 'count call-of-relieve-rule-hyps)
                  (mv :count-exceeded t alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-relieve-rule-hyps)))))

         (defthm ,(pack$ 'try-to-apply-rules- suffix '-of-0)
           (equal ,(subst 0 'count call-of-try-to-apply-rules)
                  (mv :count-exceeded nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-try-to-apply-rules)))))

         (defthm ,(pack$ 'simplify-trees-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-trees-and-add-to-dag)
                  (mv :count-exceeded trees dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-trees-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-fun-call-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-fun-call-and-add-to-dag)
                  (mv :count-exceeded nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-fun-call-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-tree-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-tree-and-add-to-dag)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-tree-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-if/myif-tree-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-if/myif-tree-and-add-to-dag)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-if/myif-tree-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-not-tree-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-not-tree-and-add-to-dag)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-not-tree-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-if/myif/boolif-tree-and-add-to-dag2- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))))

         (defthm ,(pack$ 'simplify-if/myif/boolif-tree-and-add-to-dag3- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))))

         (defthm ,(pack$ 'simplify-boolif-tree-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-boolif-tree-and-add-to-dag)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-boolif-tree-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-bvif-tree-and-add-to-dag- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-bvif-tree-and-add-to-dag)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-bvif-tree-and-add-to-dag)))))

         (defthm ,(pack$ 'simplify-bvif-tree-and-add-to-dag1- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-bvif-tree-and-add-to-dag1)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-bvif-tree-and-add-to-dag1)))))

         (defthm ,(pack$ 'simplify-bvif-tree-and-add-to-dag2- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-bvif-tree-and-add-to-dag2)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-bvif-tree-and-add-to-dag2)))))

         (defthm ,(pack$ 'simplify-bvif-tree-and-add-to-dag3- suffix '-of-0)
           (equal ,(subst 0 'count call-of-simplify-bvif-tree-and-add-to-dag3)
                  (mv :count-exceeded dag-len dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                      node-replacement-array))
           :hints (("Goal" :expand ((:free (count) ,call-of-simplify-bvif-tree-and-add-to-dag3)))))
         ) ;end progn

       ;; This may speed things up a bit:
       ;; (defun quick-and-dirty-srs-off (cl1 ac)
       ;;   (declare (ignore cl1 ac)
       ;;            (xargs :mode :logic :guard t))
       ;;   nil)
       ;; (defattach-system quick-and-dirty-srs quick-and-dirty-srs-off)

       (make-flag ,simplify-tree-and-add-to-dag-name)

       (defthm ,(pack$ 'len-of-mv-nth-1-of-simplify-trees-and-add-to-dag- suffix)
         (equal (len (mv-nth 1 ,call-of-simplify-trees-and-add-to-dag))
                (len trees))
         :hints (("Goal" :expand ((:free (count) ,call-of-simplify-trees-and-add-to-dag))
                  :induct (simplify-trees-and-add-to-dag-induct trees count)
                  :in-theory (enable (:i len)))))

       ;; Everything returns an info-world:
       (,(pack$ 'defthm-flag-simplify-tree-and-add-to-dag- suffix)
        (defthm ,(pack$ 'info-worldp-of-mv-nth-9-of-relieve-free-var-hyp-and-all-others- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 9 ,call-of-relieve-free-var-hyp-and-all-others)))

          :flag ,relieve-free-var-hyp-and-all-others-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-9-of-relieve-rule-hyps- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 9 ,call-of-relieve-rule-hyps)))
          :flag ,relieve-rule-hyps-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-try-to-apply-rules- suffix)
          (implies (and (stored-axe-rule-listp stored-rules)
                        (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-try-to-apply-rules)))
          :flag ,try-to-apply-rules-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-trees-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-trees-and-add-to-dag)))
          :flag ,simplify-trees-and-add-to-dag-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-if/myif-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
          :flag ,simplify-if/myif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-not-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-not-tree-and-add-to-dag)))
          :flag ,simplify-not-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-boolif-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-boolif-tree-and-add-to-dag)))
          :flag ,simplify-boolif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-bvif-tree-and-add-to-dag3- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
          :flag ,simplify-bvif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-bvif-tree-and-add-to-dag2- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
          :flag ,simplify-bvif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-bvif-tree-and-add-to-dag1- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
          :flag ,simplify-bvif-tree-and-add-to-dag1-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-bvif-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-bvif-tree-and-add-to-dag)))
          :flag ,simplify-bvif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-tree-and-add-to-dag)))
          :flag ,simplify-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'info-worldp-of-mv-nth-8-of-simplify-fun-call-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (info-worldp info))
                   (info-worldp (mv-nth 8 ,call-of-simplify-fun-call-and-add-to-dag)))
          :flag ,simplify-fun-call-and-add-to-dag-name)

        :hints (("Goal" :do-not '(generalize eliminate-destructors)
                 :in-theory (e/d ( ;TAKE-WHEN-<=-OF-LEN
                                  len-of-cadar-when-axe-treep
                                  pseudo-termp-of-cadddr-when-axe-treep
                                  axe-bind-free-result-okayp-rewrite
                                  symbol-alistp-when-alistp
                                  true-listp-of-cdr)
                                 (dargp-less-than
                                  natp
                                  quotep
                                  myquotep))
                 :expand ((:free (memoization count other-hyps alist)
                                 ,call-of-relieve-free-var-hyp-and-all-others)
                          (:free (memoization count)
                                 ,call-of-relieve-rule-hyps)
                          (:free (memoization)
                                 (,relieve-rule-hyps-name nil hyp-num alist rule-symbol
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count))
                          (:free (memoization count)
                                 (,simplify-trees-and-add-to-dag-name nil
                                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                                      rewrite-stobj count))
                          (:free (memoization count)
                                 ,call-of-simplify-trees-and-add-to-dag)
                          (:free (memoization limits info tries count)
                                 ,call-of-try-to-apply-rules)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                          (:free (memoization fn count)
                                 ,call-of-simplify-not-tree-and-add-to-dag)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag)
                          (:free (memoization count)
                                 ,call-of-simplify-boolif-tree-and-add-to-dag)
                          (:free (MEMOIZATION count TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-tree-and-add-to-dag)
                          (:free (MEMOIZATION count TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-fun-call-and-add-to-dag)
                          (:free (MEMOIZATION count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                          (:free (MEMOIZATION count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                          (:free (MEMOIZATION count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                          (axe-rule-hyp-listp hyps)))))

       ;; The rule-limits returned are ok:
       (,(pack$ 'defthm-flag-simplify-tree-and-add-to-dag- suffix)
        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-11-of-relieve-free-var-hyp-and-all-others- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 11 ,call-of-relieve-free-var-hyp-and-all-others)))
          :flag ,relieve-free-var-hyp-and-all-others-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-11-of-relieve-rule-hyps- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 11 ,call-of-relieve-rule-hyps)))
          :flag ,relieve-rule-hyps-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-try-to-apply-rules- suffix)
          (implies (and (stored-axe-rule-listp stored-rules)
                        (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-try-to-apply-rules)))
          :flag ,try-to-apply-rules-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-trees-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-trees-and-add-to-dag)))
          :flag ,simplify-trees-and-add-to-dag-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-if/myif-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
          :flag ,simplify-if/myif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-not-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-not-tree-and-add-to-dag)))
          :flag ,simplify-not-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-boolif-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-boolif-tree-and-add-to-dag)))
          :flag ,simplify-boolif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-bvif-tree-and-add-to-dag3- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
          :flag ,simplify-bvif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-bvif-tree-and-add-to-dag2- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
          :flag ,simplify-bvif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-bvif-tree-and-add-to-dag1- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
          :flag ,simplify-bvif-tree-and-add-to-dag1-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-bvif-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-bvif-tree-and-add-to-dag)))
          :flag ,simplify-bvif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-tree-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-tree-and-add-to-dag)))
          :flag ,simplify-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'rule-limitsp-of-mv-nth-10-of-simplify-fun-call-and-add-to-dag- suffix)
          (implies (and (rewrite-stobjp rewrite-stobj)
                        (rule-limitsp limits))
                   (rule-limitsp (mv-nth 10 ,call-of-simplify-fun-call-and-add-to-dag)))
          :flag ,simplify-fun-call-and-add-to-dag-name)

        :hints (("Goal" :do-not '(generalize eliminate-destructors)
                 :in-theory (e/d ( ;TAKE-WHEN-<=-OF-LEN
                                  len-of-cadar-when-axe-treep
                                  pseudo-termp-of-cadddr-when-axe-treep
                                  axe-bind-free-result-okayp-rewrite
                                  symbol-alistp-when-alistp
                                  true-listp-of-cdr)
                                 (dargp-less-than
                                  natp
                                  quotep
                                  myquotep
                                  ))
                 :expand ((:free (memoization count other-hyps alist)
                                 ,call-of-relieve-free-var-hyp-and-all-others)
                          (:free (memoization count)
                                 ,call-of-relieve-rule-hyps)
                          (:free (memoization)
                                 (,relieve-rule-hyps-name nil hyp-num alist rule-symbol
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count))
                          (:free (memoization count)
                                 (,simplify-trees-and-add-to-dag-name nil
                                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                                      rewrite-stobj count))
                          (:free (memoization count)
                                 ,call-of-simplify-trees-and-add-to-dag)
                          (:free (memoization limits info tries count)
                                 ,call-of-try-to-apply-rules)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                          (:free (memoization fn count)
                                 ,call-of-simplify-not-tree-and-add-to-dag)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag)
                          (:free (memoization count)
                                 ,call-of-simplify-boolif-tree-and-add-to-dag)
                          (:free (memoization count TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-tree-and-add-to-dag)
                          (:free (memoization count TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-fun-call-and-add-to-dag)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                          (axe-rule-hyp-listp hyps)))))

       ;; The tries returned are ok:
       (,(pack$ 'defthm-flag-simplify-tree-and-add-to-dag- suffix)
        (defthm ,(pack$ 'triesp-of-mv-nth-10-of-relieve-free-var-hyp-and-all-others- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 10 ,call-of-relieve-free-var-hyp-and-all-others)))
          :flag ,relieve-free-var-hyp-and-all-others-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-10-of-relieve-rule-hyps- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 10 ,call-of-relieve-rule-hyps)))
          :flag ,relieve-rule-hyps-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-try-to-apply-rules- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-try-to-apply-rules)))
          :flag ,try-to-apply-rules-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-trees-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-trees-and-add-to-dag)))
          :flag ,simplify-trees-and-add-to-dag-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-if/myif-tree-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
          :flag ,simplify-if/myif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-not-tree-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-not-tree-and-add-to-dag)))
          :flag ,simplify-not-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-boolif-tree-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-boolif-tree-and-add-to-dag)))
          :flag ,simplify-boolif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-bvif-tree-and-add-to-dag3- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
          :flag ,simplify-bvif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-bvif-tree-and-add-to-dag2- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
          :flag ,simplify-bvif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-bvif-tree-and-add-to-dag1- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
          :flag ,simplify-bvif-tree-and-add-to-dag1-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-bvif-tree-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-bvif-tree-and-add-to-dag)))
          :flag ,simplify-bvif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-tree-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-tree-and-add-to-dag)))
          :flag ,simplify-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'triesp-of-mv-nth-9-of-simplify-fun-call-and-add-to-dag- suffix)
          (implies (triesp tries)
                   (triesp (mv-nth 9 ,call-of-simplify-fun-call-and-add-to-dag)))
          :flag ,simplify-fun-call-and-add-to-dag-name)

        :hints (("Goal" :do-not '(generalize eliminate-destructors)
                 :in-theory (e/d ( ;TAKE-WHEN-<=-OF-LEN
                                  len-of-cadar-when-axe-treep
                                  pseudo-termp-of-cadddr-when-axe-treep
                                  axe-bind-free-result-okayp-rewrite
                                  symbol-alistp-when-alistp
                                  true-listp-of-cdr)
                                 (dargp-less-than
                                  natp
                                  quotep
                                  myquotep
                                  ))
                 :expand ((:free (memoization count other-hyps alist)
                                 ,call-of-relieve-free-var-hyp-and-all-others)
                          (:free (memoization count)
                                 ,call-of-relieve-rule-hyps)
                          (:free (memoization)
                                 (,relieve-rule-hyps-name nil hyp-num alist rule-symbol
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count))
                          (:free (memoization count)
                                 (,simplify-trees-and-add-to-dag-name nil
                                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                                      rewrite-stobj count))
                          (:free (memoization count)
                                 ,call-of-simplify-trees-and-add-to-dag)
                          (:free (memoization limits info tries count)
                                 ,call-of-try-to-apply-rules)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                          (:free (memoization fn count)
                                 ,call-of-simplify-not-tree-and-add-to-dag)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                          (:free (memoization fn count)
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag)
                          (:free (memoization count)
                                 ,call-of-simplify-boolif-tree-and-add-to-dag)
                          (:free (memoization count TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-tree-and-add-to-dag)
                          (:free (memoization count TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-fun-call-and-add-to-dag)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                          (:free (memoization count)
                                 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                          (AXE-RULE-HYP-LISTP HYPS)))))

       ;; main return-type theorems about functions in the mutual-recursion:
       (,(pack$ 'defthm-flag-simplify-tree-and-add-to-dag- suffix)
        (defthm ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-relieve-free-var-hyp-and-all-others))
                        (axe-tree-listp hyp-args)
                        (alist-suitable-for-hyp-args-and-hypsp alist hyp-args other-hyps)
                        (axe-rule-hyp-listp other-hyps)
                        (maybe-bounded-memoizationp memoization dag-len)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                        (symbol-alistp alist)
                        (bounded-darg-listp (strip-cdrs alist) dag-len)
                        (bounded-darg-list-listp assumption-arg-lists dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 3 ,call-of-relieve-free-var-hyp-and-all-others)
                                 (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others)
                                 'dag-parent-array
                                 (mv-nth 5 ,call-of-relieve-free-var-hyp-and-all-others)
                                 (mv-nth 6 ,call-of-relieve-free-var-hyp-and-all-others)
                                 (mv-nth 7 ,call-of-relieve-free-var-hyp-and-all-others))
                        (<= dag-len (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others))
                        (maybe-bounded-memoizationp (mv-nth 8 ,call-of-relieve-free-var-hyp-and-all-others)
                                                    (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others))
                        (iff (mv-nth 8 ,call-of-relieve-free-var-hyp-and-all-others) memoization)
                        (symbol-alistp (mv-nth 2 ,call-of-relieve-free-var-hyp-and-all-others))
                        (bounded-darg-listp (strip-cdrs (mv-nth 2 ,call-of-relieve-free-var-hyp-and-all-others))
                                             (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others))
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 12 ,call-of-relieve-free-var-hyp-and-all-others)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 12 ,call-of-relieve-free-var-hyp-and-all-others)
                                                         (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others))))
          :flag ,relieve-free-var-hyp-and-all-others-name)

        (defthm ,(pack$ 'theorem-for-relieve-rule-hyps- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (axe-rule-hyp-listp hyps)
                        (alist-suitable-for-hypsp alist hyps)
                        (not (mv-nth 0 ,call-of-relieve-rule-hyps))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                        (symbol-alistp alist)
                        (bounded-darg-listp (strip-cdrs alist) dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 3 ,call-of-relieve-rule-hyps)
                                 (mv-nth 4 ,call-of-relieve-rule-hyps)
                                 'dag-parent-array
                                 (mv-nth 5 ,call-of-relieve-rule-hyps)
                                 (mv-nth 6 ,call-of-relieve-rule-hyps)
                                 (mv-nth 7 ,call-of-relieve-rule-hyps))
                        (<= dag-len (mv-nth 4 ,call-of-relieve-rule-hyps))
                        (maybe-bounded-memoizationp (mv-nth 8 ,call-of-relieve-rule-hyps)
                                                    (mv-nth 4 ,call-of-relieve-rule-hyps))
                        (iff (mv-nth 8 ,call-of-relieve-rule-hyps) memoization)
                        (symbol-alistp (mv-nth 2 ,call-of-relieve-rule-hyps))
                        (bounded-darg-listp (strip-cdrs (mv-nth 2 ,call-of-relieve-rule-hyps))
                                             (mv-nth 4 ,call-of-relieve-rule-hyps))
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 12 ,call-of-relieve-rule-hyps)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 12 ,call-of-relieve-rule-hyps)
                                                         (mv-nth 4 ,call-of-relieve-rule-hyps))))
          :flag ,relieve-rule-hyps-name)

        (defthm ,(pack$ 'theorem-for-try-to-apply-rules- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (bounded-darg-listp args-to-match dag-len)
                        (not (mv-nth 0 ,call-of-try-to-apply-rules))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (stored-axe-rule-listp stored-rules)
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (axe-treep (mv-nth 1 ,call-of-try-to-apply-rules))
                        (bounded-axe-treep (mv-nth 1 ,call-of-try-to-apply-rules)
                                           (mv-nth 3 ,call-of-try-to-apply-rules))
                        (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-try-to-apply-rules)
                                 (mv-nth 3 ,call-of-try-to-apply-rules)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-try-to-apply-rules)
                                 (mv-nth 5 ,call-of-try-to-apply-rules)
                                 (mv-nth 6 ,call-of-try-to-apply-rules))
                        (<= dag-len (mv-nth 3 ,call-of-try-to-apply-rules))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-try-to-apply-rules)
                                                    (mv-nth 3 ,call-of-try-to-apply-rules))
                        (iff (mv-nth 7 ,call-of-try-to-apply-rules) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-try-to-apply-rules)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-try-to-apply-rules)
                                                         (mv-nth 3 ,call-of-try-to-apply-rules))))
          :flag ,try-to-apply-rules-name)

        (defthm ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (bounded-axe-tree-listp trees dag-len)
                        (not (mv-nth 0 ,call-of-simplify-trees-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-trees-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-trees-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-trees-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-trees-and-add-to-dag))
                        (true-listp (mv-nth 1 ,call-of-simplify-trees-and-add-to-dag))
                        (bounded-darg-listp (mv-nth 1 ,call-of-simplify-trees-and-add-to-dag)
                                             (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag))
                        ;; implied by the above
                        (all-dargp (mv-nth 1 ,call-of-simplify-trees-and-add-to-dag))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-trees-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-trees-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-trees-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-trees-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag))))
          :flag ,simplify-trees-and-add-to-dag-name)

        (defthm ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
          (implies (and (member-eq fn '(if myif boolif))
                        (natp simplified-test)
                        (< simplified-test dag-len)
                        (dargp-less-than simplified-then-branch dag-len)
                        (axe-treep else-branch)
                        (bounded-axe-treep else-branch dag-len)
                        (tree-to-memoizep tree)
                        ;; (bounded-axe-treep tree dag-len)
                        (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        ;; (bounded-axe-tree-listp trees-equal-to-tree dag-len)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                 (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                 (mv-nth 5 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                 (mv-nth 6 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                         (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                                    (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                        (iff (mv-nth 7 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                                                         (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
          (implies (and (member-eq fn '(if myif boolif))
                        (natp simplified-test)
                        (< simplified-test dag-len)
                        (axe-treep then-branch)
                        (bounded-axe-treep then-branch dag-len)
                        (axe-treep else-branch)
                        (bounded-axe-treep else-branch dag-len)
                        (tree-to-memoizep tree)
                        ;; (bounded-axe-treep tree dag-len)
                        (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        ;; (bounded-axe-tree-listp trees-equal-to-tree dag-len)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                 (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                 (mv-nth 5 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                 (mv-nth 6 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                         (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                                    (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                        (iff (mv-nth 7 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                                                         (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))))
          :flag ,simplify-if/myif/boolif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix)
          (implies (and (axe-treep tree)
                        (bounded-axe-treep tree dag-len)
                        (consp tree) ; not logically necessary but speeds up the proof
                        (member-eq (ffn-symb tree) '(if myif))
                        ;; (equal 3 (len (fargs tree)))
                        (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                         (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-if/myif-tree-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag))))
          :flag ,simplify-if/myif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix)
          (implies (and (axe-treep tree)
                        (bounded-axe-treep tree dag-len)
                        (consp tree) ; not logically necessary but speeds up the proof
                        (eq (ffn-symb tree) 'not)
                        (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-simplify-not-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-not-tree-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-not-tree-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-not-tree-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-not-tree-and-add-to-dag))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-not-tree-and-add-to-dag)
                                         (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-not-tree-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-not-tree-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-not-tree-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-not-tree-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag))))
          :flag ,simplify-not-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix)
          (implies (and (axe-treep tree)
                        (bounded-axe-treep tree dag-len)
                        (consp tree)
                        (equal 'boolif (ffn-symb tree))
                        (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-simplify-boolif-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-boolif-tree-and-add-to-dag))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                         (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag))
                        (<= dag-len (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-boolif-tree-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-boolif-tree-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-boolif-tree-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag))))
          :flag ,simplify-boolif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (dargp-less-than simplified-size dag-len)
                        (natp simplified-test)
                        (< simplified-test dag-len)
                        (dargp-less-than simplified-then-branch dag-len)
                        (axe-treep else-branch)
                        (bounded-axe-treep else-branch dag-len)
                        (tree-to-memoizep tree)
                        ;; (bounded-axe-treep tree dag-len)
                        (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                 (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                 (mv-nth 5 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                 (mv-nth 6 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                        (<= dag-len (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                                    (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                        (iff (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag3) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3))))
          :flag ,simplify-bvif-tree-and-add-to-dag3-name)

        (defthm ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (dargp-less-than simplified-size dag-len)
                        (natp simplified-test)
                        (< simplified-test dag-len)
                        (axe-treep then-arg)
                        (bounded-axe-treep then-arg dag-len)
                        (axe-treep else-arg)
                        (bounded-axe-treep else-arg dag-len)
                        (tree-to-memoizep tree)
                        ;; (bounded-axe-treep tree dag-len)
                        (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                 (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                 (mv-nth 5 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                 (mv-nth 6 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                        (<= dag-len (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                                    (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                        (iff (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag2) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2))))
          :flag ,simplify-bvif-tree-and-add-to-dag2-name)

        (defthm ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (axe-treep size-arg)
                        (bounded-axe-treep size-arg dag-len)
                        (natp simplified-test)
                        (< simplified-test dag-len)
                        (axe-treep then-arg)
                        (bounded-axe-treep then-arg dag-len)
                        (axe-treep else-arg)
                        (bounded-axe-treep else-arg dag-len)
                        (tree-to-memoizep tree)
                        ;; (bounded-axe-treep tree dag-len)
                        (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                 (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                 (mv-nth 5 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                 (mv-nth 6 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                        (<= dag-len (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                                    (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                        (iff (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag1) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1))))
          :flag ,simplify-bvif-tree-and-add-to-dag1-name)

        (defthm ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (axe-treep tree)
                        (bounded-axe-treep tree dag-len)
                        (consp tree)
                        (equal 'bvif (ffn-symb tree))
                        (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-bvif-tree-and-add-to-dag))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-bvif-tree-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag))))
          :flag ,simplify-bvif-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)
          (implies (and (axe-treep tree)
                        (bounded-axe-treep tree dag-len)
                        (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-tree-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-tree-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-tree-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-tree-and-add-to-dag))
                        (dargp-less-than (mv-nth 1 ,call-of-simplify-tree-and-add-to-dag)
                                         (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag))
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-tree-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-tree-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-tree-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-tree-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag))))
          :flag ,simplify-tree-and-add-to-dag-name)

        (defthm ,(pack$ 'theorem-for-simplify-fun-call-and-add-to-dag- suffix)
          (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                        (symbolp fn)
                        (not (equal 'quote fn))
                        (bounded-darg-listp args dag-len)
                        (not (mv-nth 0 ,call-of-simplify-fun-call-and-add-to-dag))
                        (maybe-bounded-memoizationp memoization dag-len)
                        (trees-to-memoizep trees-equal-to-tree)
                        (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                        (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                        (rewrite-stobjp rewrite-stobj)
                        (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                   (and (wf-dagp 'dag-array
                                 (mv-nth 2 ,call-of-simplify-fun-call-and-add-to-dag)
                                 (mv-nth 3 ,call-of-simplify-fun-call-and-add-to-dag)
                                 'dag-parent-array
                                 (mv-nth 4 ,call-of-simplify-fun-call-and-add-to-dag)
                                 (mv-nth 5 ,call-of-simplify-fun-call-and-add-to-dag)
                                 (mv-nth 6 ,call-of-simplify-fun-call-and-add-to-dag))
                        (DARGP-LESS-THAN (mv-nth 1 ,call-of-simplify-fun-call-and-add-to-dag)
                                         (mv-nth 3 ,call-of-simplify-fun-call-and-add-to-dag))
                        (DARGP (mv-nth 1 ,call-of-simplify-fun-call-and-add-to-dag)) ; follows from the above but no free vars
                        (<= dag-len
                            (mv-nth 3 ,call-of-simplify-fun-call-and-add-to-dag))
                        (maybe-bounded-memoizationp (mv-nth 7 ,call-of-simplify-fun-call-and-add-to-dag)
                                                    (mv-nth 3 ,call-of-simplify-fun-call-and-add-to-dag))
                        (iff (mv-nth 7 ,call-of-simplify-fun-call-and-add-to-dag) memoization)
                        (<= (alen1 'node-replacement-array node-replacement-array)
                            (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-fun-call-and-add-to-dag)))
                        (bounded-node-replacement-arrayp 'node-replacement-array
                                                         (mv-nth 11 ,call-of-simplify-fun-call-and-add-to-dag)
                                                         (mv-nth 3 ,call-of-simplify-fun-call-and-add-to-dag))))
          :flag ,simplify-fun-call-and-add-to-dag-name)

        :hints (("Goal" :do-not '(generalize eliminate-destructors)
                 :in-theory (e/d ( ;TAKE-WHEN-<=-OF-LEN
                                  len-of-cadar-when-axe-treep
                                  pseudo-termp-of-cadddr-when-axe-treep
                                  axe-bind-free-result-okayp-rewrite
                                  symbol-alistp-when-alistp
                                  true-listp-of-cdr
                                  symbolp-when-member-equal-disabled
                                  not-equal-when-member-equal-disabled
                                  not-equal-when-member-equal-alt-disabled
                                  <=-transitive-1
                                  <=-transitive-2
                                  symbol-listp-of-append-disabled
                                  member-equal-when-member-equal-and-subsetp-equal)
                                 (dargp-less-than
                                  natp
                                  quotep
                                  myquotep))
                 :expand ((:free (memoization ;count
                                  other-hyps alist)
                                 ,call-of-relieve-free-var-hyp-and-all-others)
                          (:free (memoization ;count
                                  )
                                 ,call-of-relieve-rule-hyps)
                          (:free (memoization)
                                 (,relieve-rule-hyps-name nil hyp-num alist rule-symbol
                                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                          node-replacement-array node-replacement-count refined-assumption-alist
                                                          rewrite-stobj count))
                          (:free (memoization ;count
                                  )
                                 (,simplify-trees-and-add-to-dag-name nil
                                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                                      rewrite-stobj count))
                          (:free (memoization ;count
                                  )
                                 ,call-of-simplify-trees-and-add-to-dag)
                          (:free (memoization limits info tries ;count
                                              )
                                 ,call-of-try-to-apply-rules)
                          (:free (memoization fn ;count
                                              )
                                 ,call-of-simplify-if/myif-tree-and-add-to-dag)
                          (:free (memoization fn ;count
                                              )
                                 ,call-of-simplify-not-tree-and-add-to-dag)
                          (:free (memoization fn ;count
                                              )
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                          (:free (memoization fn ;count
                                              )
                                 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                          (:free (memoization ;count
                                  )
                                 ,call-of-simplify-bvif-tree-and-add-to-dag)
                          (:free (memoization ;count
                                  )
                                 ,call-of-simplify-boolif-tree-and-add-to-dag)
                          (:free (memoization ;count
                                  TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-tree-and-add-to-dag)
                          (:free (memoization ;count
                                  TREES-EQUAL-TO-TREE)
                                 ,call-of-simplify-fun-call-and-add-to-dag)
                          (:free (memoization ;count
                                  )
                                 ,call-of-simplify-bvif-tree-and-add-to-dag1)
                          (:free (memoization ;count
                                  )
                                 ,call-of-simplify-bvif-tree-and-add-to-dag2)
                          (:free (memoization ;count
                                  )
                                 ,call-of-simplify-bvif-tree-and-add-to-dag3)
                          (axe-rule-hyp-listp hyps)))))

       ;; Now we prove some facts that follow from the main theorem proved about the
       ;; mutual-recursion (e.g., weaker claims that are sometimes needed, or claims
       ;; phrased in a better way for rewriting):

        ;; (defthm ,(pack$ 'theorem-for-try-to-apply-rules- suffix '-corollary)
        ;;   (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
        ;;                 (bounded-darg-listp args-to-match dag-len)
        ;;                 (not (mv-nth 0 ,call-of-try-to-apply-rules))
        ;;                 (maybe-bounded-memoizationp memoization dag-len)
        ;;                 (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
        ;;                 (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
        ;;                 (stored-axe-rule-listp stored-rules)
        ;;                 (rewrite-stobjp rewrite-stobj)
        ;;                 (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
        ;;            (dag-constant-alistp (mv-nth 5 ,call-of-try-to-apply-rules)))
        ;;   :hints (("Goal" :use ,(pack$ 'theorem-for-try-to-apply-rules- suffix '-corollary)
        ;;            :in-theory (disable ,(pack$ 'theorem-for-try-to-apply-rules- suffix '-corollary)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-fun-call-and-add-to-dag- suffix)
         (implies (and (<= bound (alen1 'node-replacement-array node-replacement-array))
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (symbolp fn)
                       (not (equal 'quote fn))
                       (bounded-darg-listp args dag-len)
                       (not (mv-nth 0 ,call-of-simplify-fun-call-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (<= bound
                      (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-fun-call-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-fun-call-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-fun-call-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-relieve-free-var-hyp-and-all-others- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-relieve-free-var-hyp-and-all-others))
                       (axe-tree-listp hyp-args)
                       (alist-suitable-for-hyp-args-and-hypsp alist hyp-args other-hyps)
                       (axe-rule-hyp-listp other-hyps)
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (symbol-alistp alist)
                       (bounded-darg-listp (strip-cdrs alist) dag-len)
                       (bounded-darg-list-listp assumption-arg-lists dag-len)
                       (<= x dag-len))
                  (<= x (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-relieve-rule-hyps- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-rule-hyp-listp hyps)
                       (alist-suitable-for-hypsp alist hyps)
                       (not (mv-nth 0 ,call-of-relieve-rule-hyps))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (symbol-alistp alist)
                       (bounded-darg-listp (strip-cdrs alist) dag-len)
                       (<= x dag-len))
                  (and
                   (<= x (mv-nth 4 ,call-of-relieve-rule-hyps))
                   (all-dargp (strip-cdrs (mv-nth 2 ,call-of-relieve-rule-hyps)))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-rule-hyps- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-relieve-rule-hyps- suffix)))))

       (defthm ,(pack$ 'corollary-theorem-for-relieve-rule-hyps- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-rule-hyp-listp hyps)
                       (alist-suitable-for-hypsp alist hyps)
                       (not (mv-nth 0 ,call-of-relieve-rule-hyps))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (symbol-alistp alist)
                       (bounded-darg-listp (strip-cdrs alist) dag-len))
                  (all-dargp (strip-cdrs (mv-nth 2 ,call-of-relieve-rule-hyps))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-rule-hyps- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-relieve-rule-hyps- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-try-to-apply-rules- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-darg-listp args-to-match dag-len)
                       (not (mv-nth 0 ,call-of-try-to-apply-rules))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (stored-axe-rule-listp stored-rules)
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x (mv-nth 3 ,call-of-try-to-apply-rules)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-try-to-apply-rules- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-try-to-apply-rules- suffix)))))

       (defthm ,(pack$ 'corollary-theorem-for-try-to-apply-rules- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-darg-listp args-to-match dag-len)
                       (not (mv-nth 0 ,call-of-try-to-apply-rules))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (stored-axe-rule-listp stored-rules)
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (and (maybe-memoizationp (mv-nth 7 ,call-of-try-to-apply-rules))
                       (equal (alen1 'dag-parent-array (mv-nth 4 ,call-of-try-to-apply-rules))
                              (alen1 'dag-array (mv-nth 2 ,call-of-try-to-apply-rules)))
                       (dag-constant-alistp (mv-nth 5 ,call-of-try-to-apply-rules))
                       (bounded-dag-parent-arrayp 'dag-parent-array
                                                  (mv-nth 4 ,call-of-try-to-apply-rules)
                                                  (mv-nth 3 ,call-of-try-to-apply-rules))
                       (dag-parent-arrayp 'dag-parent-array (mv-nth 4 ,call-of-try-to-apply-rules))
                       (pseudo-dag-arrayp 'dag-array
                                          (mv-nth 2 ,call-of-try-to-apply-rules)
                                          (mv-nth 3 ,call-of-try-to-apply-rules))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-try-to-apply-rules- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-try-to-apply-rules- suffix)))))

       (defthm ,(pack$ 'pseudo-dag-arrayp-of-mv-nth-2-of-try-to-apply-rules- suffix)
         (implies (and (<= n (mv-nth 3 ,call-of-try-to-apply-rules))
                       (natp n)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-darg-listp args-to-match dag-len)
                       (not (mv-nth 0 ,call-of-try-to-apply-rules))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (stored-axe-rule-listp stored-rules)
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (PSEUDO-DAG-ARRAYP 'DAG-ARRAY
                                     (MV-NTH 2
                                             ,call-of-try-to-apply-rules)
                                     n))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-try-to-apply-rules- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-try-to-apply-rules- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-trees-and-add-to-dag- suffix)
         (implies (and (<= x dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-axe-tree-listp trees dag-len)
                       (not (mv-nth 0 ,call-of-simplify-trees-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       )
                  (<= x
                      (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'corollary-theorem-for-simplify-trees-and-add-to-dag- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-axe-tree-listp trees dag-len)
                       (not (mv-nth 0 ,call-of-simplify-trees-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (maybe-memoizationp (mv-nth 7 ,call-of-simplify-trees-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
         (implies (and (member-eq fn '(if myif boolif))
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (dargp-less-than simplified-then-branch dag-len)
                       (axe-treep else-branch)
                       (bounded-axe-treep else-branch dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       ;; (bounded-axe-tree-listp trees-equal-to-tree dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
         (implies (and (member-eq fn '(if myif boolif))
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (axe-treep then-branch)
                       (bounded-axe-treep then-branch dag-len)
                       (axe-treep else-branch)
                       (bounded-axe-treep else-branch dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       ;; (bounded-axe-tree-listp trees-equal-to-tree dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree) ; not logically necessary, but included above
                       (member-eq (ffn-symb tree) '(if myif))
                       ;; (equal 3 (len (fargs tree)))
                       (rewrite-stobjp rewrite-stobj)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)

                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x
                      (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix))
                  :in-theory (e/d (member-equal ; split into 2 cases
                                   )
                                  (,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix))))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-not-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree) ; not logically necessary, but included above
                       (eq (ffn-symb tree) 'not)
                       (rewrite-stobjp rewrite-stobj)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-not-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x
                      (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix))
                  :in-theory (e/d (;member-equal ; split into 2 cases
                                   )
                                  (,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix))))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-boolif-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree)
                       (equal 'boolif (ffn-symb tree))
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-boolif-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (dargp-less-than simplified-size dag-len)
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (dargp-less-than simplified-then-branch dag-len)
                       (axe-treep else-branch)
                       (bounded-axe-treep else-branch dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (dargp-less-than simplified-size dag-len)
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (axe-treep then-arg)
                       (bounded-axe-treep then-arg dag-len)
                       (axe-treep else-arg)
                       (bounded-axe-treep else-arg dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x
                      (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-treep size-arg)
                       (bounded-axe-treep size-arg dag-len)
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (axe-treep then-arg)
                       (bounded-axe-treep then-arg dag-len)
                       (axe-treep else-arg)
                       (bounded-axe-treep else-arg dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x
                      (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix)))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-bvif-tree-and-add-to-dag- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree)
                       (equal 'bvif (ffn-symb tree))
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x
                      (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'corollary-theorem-for-simplify-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (and (pseudo-dag-arrayp 'dag-array (mv-nth '2 ,call-of-simplify-tree-and-add-to-dag) (mv-nth '3 ,call-of-simplify-tree-and-add-to-dag))
                       (array1p 'dag-array (mv-nth 2 ,call-of-simplify-tree-and-add-to-dag))
                       (integerp (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag))
                       (natp (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag))
                       (<= (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag)
                           (alen1 'dag-array (mv-nth 2 ,call-of-simplify-tree-and-add-to-dag)))
                       (maybe-memoizationp (mv-nth 7 ,call-of-simplify-tree-and-add-to-dag))
                       (dargp (mv-nth 1 ,call-of-simplify-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'pseudo-dag-arrayp-of-mv-nth-2-of- simplify-tree-and-add-to-dag-name)
         (implies (and (natp len)
                       (<= len
                           (MV-NTH '3
                                   ,call-of-simplify-tree-and-add-to-dag))
                       (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (PSEUDO-DAG-ARRAYP
                   'DAG-ARRAY
                   (MV-NTH '2
                           ,call-of-simplify-tree-and-add-to-dag)
                   len))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ '<-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
         (implies (and ;(natp bound)
                   (<= (MV-NTH '3
                               ,call-of-simplify-tree-and-add-to-dag)
                       bound)
                   (not (consp (MV-NTH '1
                                       ,call-of-simplify-tree-and-add-to-dag)))
                   (axe-treep tree)
                   (bounded-axe-treep tree dag-len)
                   (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                   (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                   (maybe-bounded-memoizationp memoization dag-len)
                   (trees-to-memoizep trees-equal-to-tree)
                   (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                   (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                   (rewrite-stobjp rewrite-stobj)
                   (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (< (MV-NTH '1 ,call-of-simplify-tree-and-add-to-dag)
                     bound))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'dargp-less-than-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
         (implies (and (<= (MV-NTH '3
                                   ,call-of-simplify-tree-and-add-to-dag)
                           bound)
                       (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (DARGP-LESS-THAN (MV-NTH '1
                                           ,call-of-simplify-tree-and-add-to-dag)
                                   bound))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'dargp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (dargp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)
                                      ,(pack$ 'dargp-less-than-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)))))

       ;; Use consp as the normal form
       (defthm ,(pack$ 'natp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (equal (natp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag))
                         (not (consp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag)))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)
                                      ,(pack$ 'dargp-less-than-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ,(pack$ 'dargp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)))))

       ;; Use consp as the normal form
       (defthm ,(pack$ 'integerp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (equal (integerp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag))
                         (not (consp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag)))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)
                                      ,(pack$ 'dargp-less-than-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ,(pack$ 'dargp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)))))

       ;; Use consp as the normal form
       (defthm ,(pack$ 'myquotep-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (equal (myquotep (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag))
                         (consp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)
                                      ,(pack$ 'dargp-less-than-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ,(pack$ 'dargp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ))))

       ;; Use consp as the normal form
       (defthm ,(pack$ 'equal-of-car-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name '-and-quote)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (equal (equal (car (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag)) 'quote)
                         (consp (mv-nth '1 ,call-of-simplify-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)
                                      ,(pack$ 'dargp-less-than-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ,(pack$ 'dargp-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ,(pack$ 'myquotep-of-mv-nth-1-of- simplify-tree-and-add-to-dag-name)
                                      ))))

       (defthm ,(pack$ 'bound-theorem-for-simplify-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x dag-len))
                  (<= x
                      (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'maybe-memoizationp-of-mv-nth-7-of- simplify-fun-call-and-add-to-dag-name)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (symbolp fn)
                       (not (equal 'quote fn))
                       (bounded-darg-listp args dag-len)
                       (not (mv-nth 0 ,call-of-simplify-fun-call-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                  (maybe-memoizationp (mv-nth 7 ,call-of-simplify-fun-call-and-add-to-dag)))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-fun-call-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-fun-call-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-relieve-free-var-hyp-and-all-others- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-relieve-free-var-hyp-and-all-others))
                       (axe-tree-listp hyp-args)
                       (alist-suitable-for-hyp-args-and-hypsp alist hyp-args other-hyps)
                       (axe-rule-hyp-listp other-hyps)
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (symbol-alistp alist)
                       (bounded-darg-listp (strip-cdrs alist) dag-len)
                       (bounded-darg-list-listp assumption-arg-lists dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 12 ,call-of-relieve-free-var-hyp-and-all-others))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-relieve-rule-hyps- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-rule-hyp-listp hyps)
                       (alist-suitable-for-hypsp alist hyps)
                       (not (mv-nth 0 ,call-of-relieve-rule-hyps))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (symbol-alistp alist)
                       (bounded-darg-listp (strip-cdrs alist) dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 12 ,call-of-relieve-rule-hyps))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-rule-hyps- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-relieve-rule-hyps- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-try-to-apply-rules- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-darg-listp args-to-match dag-len)
                       (not (mv-nth 0 ,call-of-try-to-apply-rules))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (stored-axe-rule-listp stored-rules)
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-try-to-apply-rules))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-try-to-apply-rules- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-try-to-apply-rules- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-trees-and-add-to-dag- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (bounded-axe-tree-listp trees dag-len)
                       (not (mv-nth 0 ,call-of-simplify-trees-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-trees-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
         (implies (and (or (equal fn 'if)
                           (equal fn 'myif))
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (dargp-less-than simplified-then-branch dag-len)
                       (axe-treep else-branch)
                       (bounded-axe-treep else-branch dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
         (implies (and (or (equal fn 'if)
                           (equal fn 'myif))
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (axe-treep then-branch)
                       (bounded-axe-treep then-branch dag-len)
                       (axe-treep else-branch)
                       (bounded-axe-treep else-branch dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree) ; not logically necessary but speeds up the proof
                       (or (eq (ffn-symb tree) 'if)
                           (eq (ffn-symb tree) 'myif))
                       ;; (equal 3 (len (fargs tree)))
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-not-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree) ; not logically necessary but speeds up the proof
                       (eq (ffn-symb tree) 'not)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-not-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-not-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-boolif-tree-and-add-to-dag- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree)
                       (equal 'boolif (ffn-symb tree))
                       (not (mv-nth 0 ,call-of-simplify-boolif-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-boolif-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (dargp-less-than simplified-size dag-len)
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (dargp-less-than simplified-then-branch dag-len)
                       (axe-treep else-branch)
                       (bounded-axe-treep else-branch dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag3))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (dargp-less-than simplified-size dag-len)
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (axe-treep then-arg)
                       (bounded-axe-treep then-arg dag-len)
                       (axe-treep else-arg)
                       (bounded-axe-treep else-arg dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag2))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-treep size-arg)
                       (bounded-axe-treep size-arg dag-len)
                       (natp simplified-test)
                       (< simplified-test dag-len)
                       (axe-treep then-arg)
                       (bounded-axe-treep then-arg dag-len)
                       (axe-treep else-arg)
                       (bounded-axe-treep else-arg dag-len)
                       (tree-to-memoizep tree)
                       ;; (bounded-axe-treep tree dag-len)
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag1))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-bvif-tree-and-add-to-dag- suffix)
         (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (consp tree)
                       (equal 'bvif (ffn-symb tree))
                       (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix)))))

       (defthm ,(pack$ 'node-replacement-array-bound-theorem-for-simplify-tree-and-add-to-dag- suffix)
         (implies (and (axe-treep tree)
                       (bounded-axe-treep tree dag-len)
                       (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                       (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                       (maybe-bounded-memoizationp memoization dag-len)
                       (trees-to-memoizep trees-equal-to-tree)
                       (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                       (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                       (rewrite-stobjp rewrite-stobj)
                       (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                       (<= x (alen1 'node-replacement-array node-replacement-array)))
                  (<= x (alen1 'node-replacement-array (mv-nth 11 ,call-of-simplify-tree-and-add-to-dag))))
         :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                  :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))

       ;; This next batch shows that node-replacement-arrayp always holds (we known from above that bounded-node-replacement-arrayp holds):
       (progn
         (defthm ,(pack$ 'node-replacement-arrayp-of-relieve-free-var-hyp-and-all-others- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (not (mv-nth 0 ,call-of-relieve-free-var-hyp-and-all-others))
                         (axe-tree-listp hyp-args)
                         (alist-suitable-for-hyp-args-and-hypsp alist hyp-args other-hyps)
                         (axe-rule-hyp-listp other-hyps)
                         (maybe-bounded-memoizationp memoization dag-len)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                         (symbol-alistp alist)
                         (bounded-darg-listp (strip-cdrs alist) dag-len)
                         (bounded-darg-list-listp assumption-arg-lists dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 12 ,call-of-relieve-free-var-hyp-and-all-others)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-relieve-free-var-hyp-and-all-others- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-relieve-rule-hyps- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (axe-rule-hyp-listp hyps)
                         (alist-suitable-for-hypsp alist hyps)
                         (not (mv-nth 0 ,call-of-relieve-rule-hyps))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                         (symbol-alistp alist)
                         (bounded-darg-listp (strip-cdrs alist) dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 12 ,call-of-relieve-rule-hyps)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-relieve-rule-hyps- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-relieve-rule-hyps- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-try-to-apply-rules- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (bounded-darg-listp args-to-match dag-len)
                         (not (mv-nth 0 ,call-of-try-to-apply-rules))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (stored-axe-rule-listp stored-rules)
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-try-to-apply-rules)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-try-to-apply-rules- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-try-to-apply-rules- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-trees-and-add-to-dag- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (bounded-axe-tree-listp trees dag-len)
                         (not (mv-nth 0 ,call-of-simplify-trees-and-add-to-dag))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-trees-and-add-to-dag)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-trees-and-add-to-dag- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
           (implies (and (or (equal fn 'if)
                             (equal fn 'myif))
                         (natp simplified-test)
                         (< simplified-test dag-len)
                         (dargp-less-than simplified-then-branch dag-len)
                         (axe-treep else-branch)
                         (bounded-axe-treep else-branch dag-len)
                         (tree-to-memoizep tree)
                         ;; (bounded-axe-treep tree dag-len)
                         (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
           (implies (and (or (equal fn 'if)
                             (equal fn 'myif))
                         (natp simplified-test)
                         (< simplified-test dag-len)
                         (axe-treep then-branch)
                         (bounded-axe-treep then-branch dag-len)
                         (axe-treep else-branch)
                         (bounded-axe-treep else-branch dag-len)
                         (tree-to-memoizep tree)
                         ;; (bounded-axe-treep tree dag-len)
                         (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (not (mv-nth 0 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-if/myif-tree-and-add-to-dag- suffix)
           (implies (and (axe-treep tree)
                         (bounded-axe-treep tree dag-len)
                         (consp tree) ; not logically necessary but speeds up the proof
                         (or (eq (ffn-symb tree) 'if)
                             (eq (ffn-symb tree) 'myif))
                         ;; (equal 3 (len (fargs tree)))
                         (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (not (mv-nth 0 ,call-of-simplify-if/myif-tree-and-add-to-dag))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-if/myif-tree-and-add-to-dag- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-not-tree-and-add-to-dag- suffix)
           (implies (and (axe-treep tree)
                         (bounded-axe-treep tree dag-len)
                         (consp tree) ; not logically necessary but speeds up the proof
                         (eq (ffn-symb tree) 'not)
                         (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (not (mv-nth 0 ,call-of-simplify-not-tree-and-add-to-dag))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-not-tree-and-add-to-dag)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-not-tree-and-add-to-dag- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-boolif-tree-and-add-to-dag- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (axe-treep tree)
                         (bounded-axe-treep tree dag-len)
                         (consp tree)
                         (equal 'boolif (ffn-symb tree))
                         (not (mv-nth 0 ,call-of-simplify-boolif-tree-and-add-to-dag))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-boolif-tree-and-add-to-dag)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-boolif-tree-and-add-to-dag- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-bvif-tree-and-add-to-dag3- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (dargp-less-than simplified-size dag-len)
                         (natp simplified-test)
                         (< simplified-test dag-len)
                         (dargp-less-than simplified-then-branch dag-len)
                         (axe-treep else-branch)
                         (bounded-axe-treep else-branch dag-len)
                         (tree-to-memoizep tree)
                         ;; (bounded-axe-treep tree dag-len)
                         (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag3))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag3- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-bvif-tree-and-add-to-dag2- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (dargp-less-than simplified-size dag-len)
                         (natp simplified-test)
                         (< simplified-test dag-len)
                         (axe-treep then-arg)
                         (bounded-axe-treep then-arg dag-len)
                         (axe-treep else-arg)
                         (bounded-axe-treep else-arg dag-len)
                         (tree-to-memoizep tree)
                         ;; (bounded-axe-treep tree dag-len)
                         (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag2))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag2- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-bvif-tree-and-add-to-dag1- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (axe-treep size-arg)
                         (bounded-axe-treep size-arg dag-len)
                         (natp simplified-test)
                         (< simplified-test dag-len)
                         (axe-treep then-arg)
                         (bounded-axe-treep then-arg dag-len)
                         (axe-treep else-arg)
                         (bounded-axe-treep else-arg dag-len)
                         (tree-to-memoizep tree)
                         ;; (bounded-axe-treep tree dag-len)
                         (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag1))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag1- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-bvif-tree-and-add-to-dag- suffix)
           (implies (and (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (axe-treep tree)
                         (bounded-axe-treep tree dag-len)
                         (consp tree)
                         (equal 'bvif (ffn-symb tree))
                         (not (mv-nth 0 ,call-of-simplify-bvif-tree-and-add-to-dag))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-bvif-tree-and-add-to-dag)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-bvif-tree-and-add-to-dag- suffix)))))

         (defthm ,(pack$ 'node-replacement-arrayp-of-simplify-tree-and-add-to-dag- suffix)
           (implies (and (axe-treep tree)
                         (bounded-axe-treep tree dag-len)
                         (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                         (not (mv-nth 0 ,call-of-simplify-tree-and-add-to-dag))
                         (maybe-bounded-memoizationp memoization dag-len)
                         (trees-to-memoizep trees-equal-to-tree)
                         (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                         (natp node-replacement-count) (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                         (rewrite-stobjp rewrite-stobj)
                         (bounded-refined-assumption-alistp refined-assumption-alist dag-len))
                    (node-replacement-arrayp 'node-replacement-array (mv-nth 11 ,call-of-simplify-tree-and-add-to-dag)))
           :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
                    :in-theory (disable ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))
         )


       ;; ;todo: more like this.  or just add to the corollary theorems above?
       ;; (def-dag-builder-theorems
       ;;   (,simplify-tree-and-add-to-dag-name tree trees-equal-to-tree dag-array dag-len dag-parent-array
       ;;                                 dag-constant-alist dag-variable-alist memoization info tries limits
       ;;                                 rule-alist
       ;;                                 refined-assumption-alist
       ;;
       ;;                                 node-replacement-count
       ;;                                 print
       ;;                                 interpreted-function-alist
       ;;                                 monitored-symbols count)
       ;;   (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
    ;;   :hyps ((axe-treep tree)
    ;;          (bounded-axe-treep tree dag-len)
    ;;          (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
    ;;          (not (mv-nth 0 (,simplify-tree-and-add-to-dag-name
    ;;                          tree trees-equal-to-tree dag-array dag-len dag-parent-array
    ;;                          dag-constant-alist dag-variable-alist memoization info tries limits
    ;;                          rule-alist
    ;;                          refined-assumption-alist
    ;;
    ;;                          node-replacement-count
    ;;                          print
    ;;                          interpreted-function-alist
    ;;                          monitored-symbols count)))
    ;;
    ;;          (maybe-bounded-memoizationp memoization dag-len)
    ;;          (trees-to-memoizep trees-equal-to-tree)
    ;;
    ;;          (alistp node-replacement-count)
    ;;          (bounded-darg-listp (strip-cdrs node-replacement-count) dag-len)
    ;;          (rewrite-stobjp rewrite-stobj)
    ;;          (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
    ;;          )
    ;;   :hyps-everywhere t
    ;;   :dag-array-name 'dag-array
    ;;   :dag-parent-array-name 'dag-parent-array
    ;;   :hints (("Goal" :use (:instance ,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix))
    ;;            :in-theory (e/d (WF-DAGP) (,(pack$ 'theorem-for-simplify-tree-and-add-to-dag- suffix)))))
    ;;   )

    ;; Show that the dag-len is always a natp:
    (,(pack$ 'defthm-flag-simplify-tree-and-add-to-dag- suffix)
     (defthm ,(pack$ 'natp-of-mv-nth-4-of-relieve-free-var-hyp-and-all-others- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 4 ,call-of-relieve-free-var-hyp-and-all-others)))
       :rule-classes (:rewrite :type-prescription) :flag ,relieve-free-var-hyp-and-all-others-name)

     (defthm ,(pack$ 'natp-of-mv-nth-4-of-relieve-rule-hyps- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 4 ,call-of-relieve-rule-hyps)))
       :rule-classes (:rewrite :type-prescription) :flag ,relieve-rule-hyps-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-try-to-apply-rules- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-try-to-apply-rules)))
       :rule-classes (:rewrite :type-prescription) :flag ,try-to-apply-rules-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-trees-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-trees-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-trees-and-add-to-dag-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-if/myif/boolif-tree-and-add-to-dag3- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-if/myif/boolif-tree-and-add-to-dag3-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-if/myif/boolif-tree-and-add-to-dag2- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-if/myif/boolif-tree-and-add-to-dag2-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-if/myif-tree-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-if/myif-tree-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-if/myif-tree-and-add-to-dag-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-not-tree-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-not-tree-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-not-tree-and-add-to-dag-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-boolif-tree-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-boolif-tree-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-boolif-tree-and-add-to-dag-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-bvif-tree-and-add-to-dag3- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag3)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-bvif-tree-and-add-to-dag3-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-bvif-tree-and-add-to-dag2- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag2)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-bvif-tree-and-add-to-dag2-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-bvif-tree-and-add-to-dag1- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag1)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-bvif-tree-and-add-to-dag1-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-bvif-tree-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-bvif-tree-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-bvif-tree-and-add-to-dag-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-tree-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-tree-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-tree-and-add-to-dag-name)

     (defthm ,(pack$ 'natp-of-mv-nth-3-of-simplify-fun-call-and-add-to-dag- suffix)
       (implies (natp dag-len)
                (natp (mv-nth 3 ,call-of-simplify-fun-call-and-add-to-dag)))
       :rule-classes (:rewrite :type-prescription) :flag ,simplify-fun-call-and-add-to-dag-name)

     :hints (("Goal" :do-not '(generalize eliminate-destructors)
              :in-theory (e/d ( ;TAKE-WHEN-<=-OF-LEN
                               len-of-cadar-when-axe-treep
                               pseudo-termp-of-cadddr-when-axe-treep
                               axe-bind-free-result-okayp-rewrite
                               symbol-alistp-when-alistp
                               true-listp-of-cdr)
                              (dargp-less-than
                               natp
                               quotep
                               myquotep))
              :expand ((:free (memoization count other-hyps alist)
                              ,call-of-relieve-free-var-hyp-and-all-others)
                       (:free (memoization count)
                              ,call-of-relieve-rule-hyps)
                       (:free (memoization)
                              (,relieve-rule-hyps-name nil hyp-num alist rule-symbol
                                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                       node-replacement-array node-replacement-count refined-assumption-alist
                                                       rewrite-stobj count))
                       (:free (memoization count)
                              (,simplify-trees-and-add-to-dag-name nil
                                                                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                   node-replacement-array node-replacement-count refined-assumption-alist
                                                                   rewrite-stobj count))
                       (:free (memoization count)
                              ,call-of-simplify-trees-and-add-to-dag)
                       (:free (memoization limits info tries count)
                              ,call-of-try-to-apply-rules)
                       (:free (memoization fn count)
                              ,call-of-simplify-if/myif-tree-and-add-to-dag)
                       (:free (memoization fn count)
                              ,call-of-simplify-not-tree-and-add-to-dag)
                       (:free (memoization fn count)
                              ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag2)
                       (:free (memoization fn count)
                              ,call-of-simplify-if/myif/boolif-tree-and-add-to-dag3)
                       (:free (memoization count)
                              ,call-of-simplify-bvif-tree-and-add-to-dag)
                       (:free (memoization count)
                              ,call-of-simplify-boolif-tree-and-add-to-dag)
                       (:free (memoization count TREES-EQUAL-TO-TREE)
                              ,call-of-simplify-tree-and-add-to-dag)
                       (:free (memoization count TREES-EQUAL-TO-TREE)
                              ,call-of-simplify-fun-call-and-add-to-dag)
                       (:free (memoization count)
                              ,call-of-simplify-bvif-tree-and-add-to-dag1)
                       (:free (memoization count)
                              ,call-of-simplify-bvif-tree-and-add-to-dag2)
                       (:free (memoization count)
                              ,call-of-simplify-bvif-tree-and-add-to-dag3)
                       (axe-rule-hyp-listp hyps)))))

    (verify-guards ,simplify-fun-call-and-add-to-dag-name
      :hints (("Goal" :do-not '(generalize eliminate-destructors)
               :expand ((axe-bind-free-function-applicationp (nth 1 (car hyps)))
                        (axe-rule-hyp-listp hyps)
                        ;; (axe-treep tree)
                        (myquotep tree)
                        (quotep tree))
               :in-theory (e/d (true-list-of-car-when-bounded-darg-list-listp
                                all-dargp-of-car-when-bounded-darg-list-listp
                                all-myquotep-when-all-dargp
                                axe-bind-free-result-okayp-rewrite
                                axe-rule-hypp
                                integerp-when-dargp
                                <=-of-0-when-dargp
                                <-when-dargp-less-than
                                len-when-dargp
                                natp-when-dargp
                                quotep-when-dargp
                                <-of--1-when-dargp
                                integerp-of-if
                                <-of--0-when-dargp
                                natp-of-+-of-1
                                <-of-+-of-1-when-integers
                                ;;symbol-alistp-when-alistp
                                cadr-becomes-nth-of-1
                                memoizationp-when-maybe-memoizationp
                                tree-to-memoizep ;todo
                                <=-transitive-1  ;drop?
                                symbolp-when-member-equal-disabled
                                not-equal-when-member-equal-alt-disabled
                                not-equal-when-member-equal-disabled
                                pseudo-termp-of-lambda-body-when-axe-treep
                                true-listp-of-lambda-formals-when-axe-treep
                                symbol-listp-of-lambda-formals-when-axe-treep
                                len-of-lambda-formals-when-axe-treep
                                len-when-equal-of-car-and-quote-and-axe-treep
                                consp-of-cdr-when-equal-of-car-and-quote-and-axe-treep
                                len-of-car-when-axe-treep
                                member-equal-when-member-equal-and-subsetp-equal)
                               (dargp
                                dargp-less-than
                                natp
                                quotep
                                myquotep
                                nth-of-cdr
                                cadr-becomes-nth-of-1)))
              ;;(and stable-under-simplificationp '(:cases (memoizep)))
              ))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; Returns (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array).
    (defund ,simplify-dag-expr-name (expr
                                     old-nodenum ; just for guards?
                                     dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                     memoization ; this is over the NEW nodenums (the ones in dag-array)
                                     info tries limits
                                     node-replacement-array node-replacement-count ; this is over nodes in the NEW dag
                                     refined-assumption-alist
                                     rewrite-stobj
                                     ;; maps nodenums in rev-dag to the dargs (nodenums or quoteps) they rewrote to in dag-array
                                     ;; we could renumber outside this function, but not all nodes may be needed:
                                     renumbering-stobj
                                     )
      (declare (xargs :guard (and (natp old-nodenum)
                                  (bounded-dag-exprp old-nodenum expr)
                                  (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                  (maybe-bounded-memoizationp memoization dag-len)
                                  (info-worldp info)
                                  (triesp tries)
                                  (rule-limitsp limits)
                                  (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                  (natp node-replacement-count)
                                  (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                  (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                  (< old-nodenum (renumbering-length renumbering-stobj))
                                  (bounded-good-renumbering-stobj (+ -1 old-nodenum) dag-len renumbering-stobj))
                      :stobjs (rewrite-stobj renumbering-stobj)
                      :guard-hints (("Goal" :in-theory (e/d (;; todo: simplify this hint
                                                             integerp-when-dargp
                                                             rationalp-when-integerp
                                                             symbolp-of-car-when-dag-exprp
                                                             axe-treep-when-dag-exprp
                                                             car-of-cadr-when-cars-increasing-by-1
                                                             all-myquotep-when-all-dargp
                                                             consp-of-cdr-when-dargp
                                                             consp-of-cdr-when-dag-exprp-and-quote
                                                             not-cddr-when-dag-exprp-and-quotep
                                                             natp-of-renumber-darg-with-stobj
                                                             consp-of-dargs-when-dag-exprp-iff
                                                             true-listp-of-renumber-darg-with-stobj
                                                             <-of-renumber-darg-with-stobj
                                                             <-of-if-arg2-axe ; todo: -axe suffix here is confusing (not an axe rule)
                                                             )
                                                            (natp dargp dargp-less-than-when-not-consp-cheap dargp-less-than-when-consp-cheap)))))
               (ignore old-nodenum))
      (if (atom expr)
          ;; EXPR is a variable:
          (b* (;; Add it to the DAG:
               ((mv erp new-nodenum dag-array dag-len dag-parent-array dag-variable-alist)
                (add-variable-to-dag-array expr dag-array dag-len dag-parent-array dag-variable-alist))
               ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
               ;; Maybe apply a replacement:
               (new-nodenum-or-quotep (apply-node-replacement-array new-nodenum node-replacement-array node-replacement-count)))
            (mv (erp-nil) new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
        (let ((fn (ffn-symb expr)))
          (case fn
            (quote ; EXPR is a quoted constant (rare):
              (mv (erp-nil) expr dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
            ;; (if/myif <test> <then> <else>):
            ;; Functions like simplify-bvif-tree-and-add-to-dag-name are not appropriate here because they take entire trees.
            ((if myif)
             (b* ((dargs (dargs expr))
                  ((when (not (consp (rest (rest dargs))))) ; for guards
                   (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                  ;; Renumber the test (then-branch and else-branch get renumbered below only if needed):
                  (renumbered-test-darg (renumber-darg-with-stobj (first dargs) renumbering-stobj))
                  ;; Special treatment for IF (can rewrite the test in an IFF context):
                  (renumbered-test-darg (apply-node-replacement-array-bool-to-darg renumbered-test-darg node-replacement-array node-replacement-count))
                  ;; TODO: Consult the memoization?
                  )
               (if (consp renumbered-test-darg) ; test for quotep
                   ;; The test was resolved, so the whole node is replaced by one branch:
                   ;; I suppose we could update the memoization here if we wanted to (but remember that it deals in the new nodenums).
                   (mv (erp-nil)
                       (if (unquote renumbered-test-darg)
                           (renumber-darg-with-stobj (second dargs) renumbering-stobj)
                         (renumber-darg-with-stobj (third dargs) renumbering-stobj))
                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                 ;; The test was not resolved, so just try to apply rules:
                 (,simplify-fun-call-and-add-to-dag-name ;; TODO: Perhaps pass in the original expr for use by cons-with-hint?
                  fn (list renumbered-test-darg
                           ;; TODO: Try to apply node replacements, including using info from the test?
                           (renumber-darg-with-stobj (second dargs) renumbering-stobj)
                           (renumber-darg-with-stobj (third dargs) renumbering-stobj))
                  nil ; Can't memoize anything about EXPR because its nodenums are in the old dag (but we could cons the new expr?)
                  dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                  node-replacement-array node-replacement-count refined-assumption-alist
                  rewrite-stobj 1000000000))))
            ;; (not <expr>):
            (not
             (b* ((dargs (dargs expr))
                  ((when (not (consp dargs))) ; for guards
                   (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                  ;; Renumber the expr:
                  (renumbered-expr (renumber-darg-with-stobj (first dargs) renumbering-stobj))
                  ;; Special treatment for NOT (can rewrite the arg in an IFF context):
                  (renumbered-expr (apply-node-replacement-array-bool-to-darg renumbered-expr node-replacement-array node-replacement-count))
                  ;; TODO: Consult the memoization?
                  )
               (if (consp renumbered-expr) ; test for quotep
                   ;; Ground term:
                   ;; I suppose we could update the memoization here if we wanted to (but remember that it deals in the new nodenums).
                   (mv (erp-nil)
                       (if (unquote renumbered-expr) *nil* *t*) ; negate the expr
                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                 ;; The expr is not constant, so just try to apply rules:
                 (,simplify-fun-call-and-add-to-dag-name ;; TODO: Perhaps pass in the original expr for use by cons-with-hint?
                  fn (list renumbered-expr)
                  nil ; Can't memoize anything about EXPR because its nodenums are in the old dag (but we could cons the new expr?)
                  dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                  node-replacement-array node-replacement-count refined-assumption-alist
                  rewrite-stobj 1000000000))))
            ;; (boolif <test> <then> <else>):
            (boolif
              (b* ((dargs (dargs expr))
                   ((when (not (consp (rest (rest dargs))))) ; for guards
                    (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                   ;; Renumber the test (then-branch and else-branch get renumbered below only if needed):
                   (renumbered-test-darg (renumber-darg-with-stobj (first dargs) renumbering-stobj))
                   ;; Special treatment for BOOLIF (can rewrite the test in an IFF context):
                   (renumbered-test-darg (apply-node-replacement-array-bool-to-darg renumbered-test-darg node-replacement-array node-replacement-count))
                   ;; TODO: Consult the memoization?
                   )
                (if (consp renumbered-test-darg) ; test for quotep
                    ;; The test was resolved, so the whole node is replaced by (the bool-fix of) one branch:
                    ;; TODO: Do something faster, with no bool-fix, if the selected branch is a known-boolean?
                    (let ((selected-branch
                            ;; Since this is a boolif, we can rewrite the selected branch in an IFF context:
                            (apply-node-replacement-array-bool-to-darg
                              (renumber-darg-with-stobj
                                (if (unquote renumbered-test-darg)
                                    (second dargs)
                                  (third dargs))
                                renumbering-stobj)
                              node-replacement-array node-replacement-count)))
                      (if (consp selected-branch) ; tests for quotep
                          (mv (erp-nil)
                              (if (unquote selected-branch) *t* *nil*) ; bool-fixes the constant
                              dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                        (,simplify-fun-call-and-add-to-dag-name 'bool-fix$inline
                                                                (list selected-branch)
                                                                nil ; should we pass the tree here (remember that the memoization deals in the new nodenums)?
                                                                dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                node-replacement-array node-replacement-count refined-assumption-alist
                                                                rewrite-stobj 1000000000)))
                  ;; The test was not resolved, so just try to apply rules (can't be a ground term because the test is not a constant):
                  (,simplify-fun-call-and-add-to-dag-name ;; TODO: Perhaps pass in the original expr for use by cons-with-hint?
                   fn (list renumbered-test-darg
                            ;; Since this is a boolif, we can rewrite the branches in an IFF context:
                            ;; TODO: Try to apply node replacements using info from the test as well
                            (apply-node-replacement-array-bool-to-darg (renumber-darg-with-stobj (second dargs) renumbering-stobj) node-replacement-array node-replacement-count)
                            (apply-node-replacement-array-bool-to-darg (renumber-darg-with-stobj (third dargs) renumbering-stobj) node-replacement-array node-replacement-count))
                   nil ; Can't memoize anything about EXPR because its nodenums are in the old dag (but we could cons the new expr?)
                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                   node-replacement-array node-replacement-count refined-assumption-alist
                   rewrite-stobj 1000000000))))
            ;; (bvif <size> <test> <then> <else>):
            (bvif
              (b* ((dargs (dargs expr))
                   ((when (not (consp (rest (rest (rest dargs)))))) ; for guards
                    (mv :bad-arity nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array))
                   ;; Renumber the size and test (then-branch and else-branch get renumbered below only if needed):
                   (renumbered-size-darg (renumber-darg-with-stobj (first dargs) renumbering-stobj))
                   (renumbered-test-darg (renumber-darg-with-stobj (second dargs) renumbering-stobj))
                   ;; Special treatment for BVIF (can rewrite the test in an IFF context):
                   (renumbered-test-darg (apply-node-replacement-array-bool-to-darg renumbered-test-darg node-replacement-array node-replacement-count))
                   ;; TODO: Consult the memoization?
                   )
                (if (consp renumbered-test-darg) ; test for quotep
                    ;; The test was resolved, so the whole node is replaced by (the bvchop of) one branch:
                    ;; TODO: Do something faster, with no bvchop, if the selected branch is already a BV
                    (let ((selected-branch
                            (renumber-darg-with-stobj
                              (if (unquote renumbered-test-darg)
                                  (third dargs)
                                (fourth dargs))
                              renumbering-stobj)))
                      (if (and (consp selected-branch) ; tests for quotep
                               (consp renumbered-size-darg))
                          ;; Ground term:
                          (mv (erp-nil)
                              (enquote (bvchop-unguarded (unquote renumbered-size-darg) (unquote selected-branch))) ; todo: optimize
                              dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                        (,simplify-fun-call-and-add-to-dag-name 'bvchop
                                                                (list renumbered-size-darg selected-branch)
                                                                nil ; should we pass the tree here (remember that the memoization deals in the new nodenums)?
                                                                dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                                                node-replacement-array node-replacement-count refined-assumption-alist
                                                                rewrite-stobj 1000000000)))
                  ;; The test was not resolved, so just try to apply rules (can't be a ground term because the test is not a constant):
                  (,simplify-fun-call-and-add-to-dag-name ;; TODO: Perhaps pass in the original expr for use by cons-with-hint?
                   fn (list renumbered-size-darg
                            renumbered-test-darg
                            ;; TODO: Try to apply node replacements using info from the test as well
                            (renumber-darg-with-stobj (third dargs) renumbering-stobj)
                            (renumber-darg-with-stobj (fourth dargs) renumbering-stobj))
                   nil ; Can't memoize anything about EXPR because its nodenums are in the old dag (but we could cons the new expr?)
                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                   node-replacement-array node-replacement-count refined-assumption-alist
                   rewrite-stobj 1000000000))))
            (t ;; EXPR is some other function call (can't be a lambda application since it is a dag-expr):
              (b* (;; Renumber the args:
                   (new-dargs (renumber-dargs-with-stobj (dargs expr) renumbering-stobj)) ; todo: have the renumbering function return a groundp flag
                   ;; TODO: Try to apply node replacements?
                   ;; TODO: Consider consulting the memoization here, now that the nodenums have been renumbered
                   ;; Handle possible ground term by evaluating (since ,simplify-fun-call-and-add-to-dag-name doesn't handle ground terms):
                   ((mv erp evaluatedp val)
                    (if (not (all-consp new-dargs)) ;; test for args being quoted constants
                        ;; not a ground term:
                        (mv (erp-nil) nil nil)
                      ;; ground term, so try to evaluate (may fail, but we may have a constant opener rule to apply later):
                      (b* (((mv erp val)
                            (,apply-axe-evaluator-to-quoted-args-name fn new-dargs (get-interpreted-function-alist rewrite-stobj))))
                        (if erp
                            (if (call-of :unknown-function erp)
                                (mv (erp-nil) nil nil) ;no error, but it didn't produce a value (todo: print a warning?)
                              ;; anything else non-nil is a true error:
                              (mv erp nil nil))
                          ;; normal case (evaluated to VAL):
                          (mv (erp-nil) t val)))))
                   ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)))
                (if evaluatedp
                    ;; I suppose we could update the memoization here if we wanted to (but remember that it deals in the new nodenums).
                    (mv (erp-nil) (enquote val) dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                  ;; Not a ground term we could evaluate, so rewrite the non-lambda FN applied to the simplified args:
                  (,simplify-fun-call-and-add-to-dag-name ;; TODO: Perhaps pass in the original expr for use by cons-with-hint?
                   fn new-dargs
                   nil ; Can't memoize anything about EXPR because its nodenums are in the old dag
                   dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                   node-replacement-array node-replacement-count refined-assumption-alist
                   rewrite-stobj 1000000000))))))))

    (defthm ,(pack$ simplify-dag-expr-name '-return-type)
      (implies (and (natp old-nodenum)
                    (bounded-dag-exprp old-nodenum expr)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (< old-nodenum (renumbering-length renumbering-stobj))
                    (bounded-good-renumbering-stobj (+ -1 old-nodenum) dag-len renumbering-stobj))
               (mv-let (erp new-nodenum-or-quotep new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info new-tries new-limits
                            new-node-replacement-array ; no real change
                            )
                 (,simplify-dag-expr-name expr
                                          old-nodenum
                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                          node-replacement-array node-replacement-count refined-assumption-alist
                                          rewrite-stobj
                                          renumbering-stobj)
                 (implies (not erp)
                          (and (wf-dagp 'dag-array new-dag-array new-dag-len 'dag-parent-array new-dag-parent-array new-dag-constant-alist new-dag-variable-alist)
                               (dargp-less-than new-nodenum-or-quotep new-dag-len)
                               (dargp new-nodenum-or-quotep) ; implied by dargp-less-than (just above) but matches better
                               (<= dag-len new-dag-len)
                               (info-worldp new-info)
                               (maybe-memoizationp new-memoization)
                               (maybe-bounded-memoizationp new-memoization new-dag-len)
                               (iff new-memoization memoization) ; whether we are memoizing doesn't change
                               (triesp new-tries)
                               (rule-limitsp new-limits)
                               (bounded-node-replacement-arrayp 'node-replacement-array new-node-replacement-array new-dag-len)
                               (<= node-replacement-count (alen1 'node-replacement-array new-node-replacement-array))))))
      :hints (("Goal" :in-theory (e/d (,simplify-dag-expr-name
                                       ;; todo: simplify this hint
                                       integerp-when-dargp
                                       rationalp-when-integerp
                                       symbolp-of-car-when-dag-exprp
                                       axe-treep-when-dag-exprp
                                       car-of-cadr-when-cars-increasing-by-1
                                       all-myquotep-when-all-dargp
                                       consp-of-cdr-when-dargp
                                       consp-of-cdr-when-dag-exprp-and-quote
                                       not-cddr-when-dag-exprp-and-quotep
                                       natp-of-renumber-darg-with-stobj
                                       consp-of-dargs-when-dag-exprp-iff
                                       true-listp-of-renumber-darg-with-stobj
                                       <-of-renumber-darg-with-stobj
                                       <-of-if-arg2-axe ; todo: -axe suffix here is confusing (not an axe rule)
                                       not-<-of-nth-of-dargs)
                                      (natp dargp dargp-less-than-when-not-consp-cheap dargp-less-than-when-consp-cheap)))))

    (defthm ,(pack$ simplify-dag-expr-name '-return-type-corollary)
      (implies (and (natp old-nodenum)
                    (bounded-dag-exprp old-nodenum expr)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (< old-nodenum (renumbering-length renumbering-stobj))
                    (bounded-good-renumbering-stobj (+ -1 old-nodenum) dag-len renumbering-stobj))
               (mv-let (erp new-nodenum-or-quotep new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info new-tries new-limits
                            new-node-replacement-array ; no real change
                            )
                 (,simplify-dag-expr-name expr
                                          old-nodenum
                                          dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits
                                          node-replacement-array node-replacement-count refined-assumption-alist
                                          rewrite-stobj
                                          renumbering-stobj)
                 (declare (ignore new-nodenum-or-quotep new-dag-array new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-info new-tries new-limits))
                 (implies (not erp)
                          (and (natp new-dag-len)
                               (node-replacement-arrayp 'node-replacement-array new-node-replacement-array)
                               (implies (equal memoization nil) (equal new-memoization nil))))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-expr-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-expr-name '-return-type)))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; For each node in REV-DAG, fix up its args (if any) according to the renumbering-stobj, then add its simplified form to the dag-array and add its new nodenum or quotep to the renumbering-stobj.
    ;; Returns (mv erp dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj).
    ;; TODO: Add support for assumptions that come in array form?
    ;; TODO: Add support for rewriting nodes in (an approximation of) their contexts -- but disallow memoization in that case!
    (defund ,simplify-dag-aux-name (rev-dag ; the old dag, low nodes come first
                                    dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist ; the new DAG
                                    maybe-internal-context-array
                                    memoization ; this is over the NEW nodenums (the ones in dag-array)
                                    info tries limits
                                    node-replacement-array node-replacement-count ; this is over nodes in the NEW dag
                                    refined-assumption-alist ; these are over node in the NEW dag
                                    rewrite-stobj
                                    renumbering-stobj ; maps nodenums in rev-dag to the dargs (nodenums or quoteps) they rewrote to in dag-array
                                    )
      (declare (xargs :guard (and (weak-dagp-aux rev-dag)
                                  (cars-increasing-by-1 rev-dag)
                                  (if (consp rev-dag)
                                      (<= (car (car (last rev-dag))) 2147483646)
                                    t)
                                  (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                                  (if (consp rev-dag)
                                      (or (null maybe-internal-context-array)
                                          (context-arrayp 'context-array maybe-internal-context-array (+ 1 (car (car (last rev-dag))))))
                                    t)
                                  (maybe-bounded-memoizationp memoization dag-len)
                                  (not (and memoization maybe-internal-context-array)) ; would be unsound to have both
                                  (info-worldp info)
                                  (triesp tries)
                                  (rule-limitsp limits)
                                  (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                                  (natp node-replacement-count)
                                  (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                                  (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                                  (if (consp rev-dag)
                                      (equal (renumbering-length renumbering-stobj)
                                             (+ 1 (car (car (last rev-dag))))) ; the highest nodenum
                                    t)
                                  (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                                      (+ -1 (car (first rev-dag)))
                                                                    -1)
                                                                  dag-len renumbering-stobj))
                      :stobjs (rewrite-stobj renumbering-stobj)
                      :guard-hints (("Goal" ;:expand (WEAK-DAGP-AUX REV-DAG)
                                     :in-theory (e/d (car-of-car-of-last-when-cars-increasing-by-1-linear ; todo: simplify this hint
                                                      ;maybe-dargp
                                                      integerp-when-dargp
                                                      rationalp-when-integerp
                                                      symbolp-of-car-when-dag-exprp
                                                      ;tree-to-memoizep
                                                      axe-treep-when-dag-exprp
                                                      car-of-cadr-when-cars-increasing-by-1
                                                      all-myquotep-when-all-dargp
                                                      consp-of-cdr-when-dargp
                                                      consp-of-cdr-when-dag-exprp-and-quote
                                                      not-cddr-when-dag-exprp-and-quotep
                                                      consp-of-car-of-last-when-weak-dagp-aux
                                                      acl2-numberp-of-car-of-car-of-last-when-weak-dagp-aux
                                                      consp-of-dargs-when-dag-exprp-iff
                                                      )
                                                     (natp dargp DARGP-LESS-THAN-WHEN-NOT-CONSP-CHEAP DARGP-LESS-THAN-WHEN-CONSP-CHEAP))
                                     :do-not '(generalize eliminate-destructors)))))
      (if (endp rev-dag)
          ;; Done rewriting nodes.  The caller can use the renumbering-stobj to lookup what the old top node rewrote to:
          (mv (erp-nil) dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj)
        (b* ((entry (first rev-dag))
             (nodenum (the (integer 0 2147483646) (car entry))) ; or, since they are consecutive, we could track this numerically.
             (print (get-print rewrite-stobj))
             (- (and print (= 0 (mod nodenum 1000)) (cw "Simplifying node ~x0.~%" nodenum)))
             (context-for-this-node (if maybe-internal-context-array (aref1 'context-array maybe-internal-context-array nodenum) (true-context)))
             (context-for-this-node (if (false-contextp context-for-this-node)
                                        (cw "WARNING: False context for node ~x0.~%" nodenum)
                                      (true-context) ; safe
                                      ))
             ;; Temporarily push context info into the node-replacement-array:
             ((mv node-replacement-array node-replacement-count-for-this-node undo-pairs)
              (update-node-replacement-array-for-assuming-possibly-negated-nodenums context-for-this-node
                                                                                    node-replacement-array node-replacement-count
                                                                                    dag-array dag-len
                                                                                    (get-known-booleans rewrite-stobj)
                                                                                    nil))
             ;; Also include the context information in the assumptions used for free var matching:
             (context-exprs-for-this-node (context-to-exprs context-for-this-node dag-array dag-len))
             (refined-assumption-alist-for-this-node (extend-refined-assumption-alist context-exprs-for-this-node refined-assumption-alist))
             ((mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
              (,simplify-dag-expr-name (cdr entry) ; the expr
                                       nodenum
                                       dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                       memoization ; this is over the NEW nodenums (the ones in dag-array)
                                       info tries limits
                                       node-replacement-array node-replacement-count ; this is over nodes in the NEW dag
                                       refined-assumption-alist-for-this-node
                                       rewrite-stobj
                                       renumbering-stobj))
             ((when erp) (mv erp dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj))
             ;; Pop the context (if not using contexts, undo-pairs will be nil):
             (node-replacement-array (undo-writes-to-node-replacement-array undo-pairs node-replacement-array node-replacement-count-for-this-node dag-len))
             ;; Record the fact that NODENUM rewrote to NEW-NODENUM-OR-QUOTEP:
             (renumbering-stobj (update-renumbering nodenum new-nodenum-or-quotep renumbering-stobj)))
          (,simplify-dag-aux-name (rest rev-dag)
                                  dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                  maybe-internal-context-array
                                  memoization info tries limits
                                  node-replacement-array
                                  node-replacement-count-for-this-node ; or just use node-replacement-count, as the relevant entires have been undone?
                                  refined-assumption-alist
                                  rewrite-stobj
                                  renumbering-stobj))))

    (defthm ,(pack$ simplify-dag-aux-name '-return-type)
      (implies (and (weak-dagp-aux rev-dag)
                    (cars-increasing-by-1 rev-dag)
                    (if (consp rev-dag)
                        (<= (car (car (last rev-dag))) 2147483646)
                      t)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (if (consp rev-dag)
                        (or (null maybe-internal-context-array)
                            (context-arrayp 'context-array maybe-internal-context-array (+ 1 (car (car (last rev-dag))))))
                      t)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (if (consp rev-dag)
                        (equal (renumbering-length renumbering-stobj)
                               (+ 1 (car (car (last rev-dag))))) ; the highest nodenum
                      t)
                    (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                     (+ -1 (car (first rev-dag)))
                                                   -1)
                                                 dag-len renumbering-stobj)
                    (renumbering-stobjp renumbering-stobj))
               (mv-let (erp new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj)
                 (,simplify-dag-aux-name rev-dag
                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                         maybe-internal-context-array memoization info tries limits
                                         node-replacement-array node-replacement-count refined-assumption-alist
                                         rewrite-stobj
                                         renumbering-stobj)
                 (declare (ignore tries limits node-replacement-array))
                 (implies (not erp)
                          (and (wf-dagp 'dag-array new-dag-array new-dag-len 'dag-parent-array new-dag-parent-array new-dag-constant-alist new-dag-variable-alist)
                               (info-worldp new-info)
                               (maybe-memoizationp new-memoization)
                               (iff new-memoization memoization)
                               (renumbering-stobjp new-renumbering-stobj)
                               (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                                (car (car (last rev-dag)))
                                                              -1)
                                                            new-dag-len
                                                            new-renumbering-stobj)
                               (equal (renumbering-length new-renumbering-stobj) (renumbering-length renumbering-stobj))))))
      :hints (("Goal" :induct (,simplify-dag-aux-name rev-dag
                                                      dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                                      maybe-internal-context-array memoization info tries limits
                                                      node-replacement-array node-replacement-count refined-assumption-alist
                                                      rewrite-stobj
                                                      renumbering-stobj)
               :in-theory (enable ,simplify-dag-aux-name
                                  car-of-car-of-last-when-cars-increasing-by-1-linear
                                  ;maybe-dargp
                                  integerp-when-dargp
                                  rationalp-when-integerp
                                  symbolp-of-car-when-dag-exprp
                                  tree-to-memoizep
                                  axe-treep-when-dag-exprp
                                  car-of-cadr-when-cars-increasing-by-1
                                  all-myquotep-when-all-dargp
                                  consp-of-cdr-when-dargp
                                  consp-of-cdr-when-dag-exprp-and-quote
                                  not-cddr-when-dag-exprp-and-quotep
                                  consp-of-car-of-last-when-weak-dagp-aux
                                  acl2-numberp-of-car-of-car-of-last-when-weak-dagp-aux
                                  natp-of-car-of-car-when-weak-dagp-aux)
               :do-not '(generalize eliminate-destructors))))

    ;; A simple consequence of the return type theorem
    (defthm ,(pack$ simplify-dag-aux-name '-return-type-corollary0)
      (implies (and (weak-dagp-aux rev-dag)
                    (cars-increasing-by-1 rev-dag)
                    (if (consp rev-dag)
                        (<= (car (car (last rev-dag))) 2147483646)
                      t)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (if (consp rev-dag)
                        (or (null maybe-internal-context-array)
                            (context-arrayp 'context-array maybe-internal-context-array (+ 1 (car (car (last rev-dag))))))
                      t)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (if (consp rev-dag)
                        (equal (renumbering-length renumbering-stobj)
                               (+ 1 (car (car (last rev-dag))))) ; the highest nodenum
                      t)
                    (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                     (+ -1 (car (first rev-dag)))
                                                   -1)
                                                 dag-len renumbering-stobj)
                    (renumbering-stobjp renumbering-stobj))
               (mv-let (erp new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj)
                 (,simplify-dag-aux-name rev-dag
                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                         maybe-internal-context-array memoization info tries limits
                                         node-replacement-array node-replacement-count refined-assumption-alist
                                         rewrite-stobj
                                         renumbering-stobj)
                 (declare (ignore new-dag-array new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj))
                 (implies (not erp)
                          (and (natp new-dag-len)
                               (integerp new-dag-len)
                               ;; (not (< 2147483646 new-dag-len)) ; todo
                               ))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-aux-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-aux-name '-return-type)))))

    ;; A simple consequence of the return type theorem
    (defthm ,(pack$ simplify-dag-aux-name '-return-type-corollary)
      (implies (and (weak-dagp-aux rev-dag)
                    (cars-increasing-by-1 rev-dag)
                    (if (consp rev-dag)
                        (<= (car (car (last rev-dag))) 2147483646)
                      t)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (if (consp rev-dag)
                        (or (null maybe-internal-context-array)
                            (context-arrayp 'context-array maybe-internal-context-array (+ 1 (car (car (last rev-dag))))))
                      t)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (if (consp rev-dag)
                        (equal (renumbering-length renumbering-stobj)
                               (+ 1 (car (car (last rev-dag))))) ; the highest nodenum
                      t)
                    (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                     (+ -1 (car (first rev-dag)))
                                                   -1)
                                                 dag-len renumbering-stobj)
                    (renumbering-stobjp renumbering-stobj))
               (mv-let (erp new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj)
                 (,simplify-dag-aux-name rev-dag
                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist maybe-internal-context-array
                                         memoization info tries limits
                                         node-replacement-array node-replacement-count refined-assumption-alist
                                         rewrite-stobj
                                         renumbering-stobj)
                 (declare (ignore new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj))
                 (implies (and (not erp)
                               (<= bound new-dag-len) ; note this
                               (natp bound))
                          (pseudo-dag-arrayp 'dag-array new-dag-array bound))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-aux-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-aux-name '-return-type)))))


    ;; A simple consequence of the return type theorem
    (defthm ,(pack$ simplify-dag-aux-name '-return-type-corollary2)
      (implies (and (weak-dagp-aux rev-dag)
                    (cars-increasing-by-1 rev-dag)
                    (if (consp rev-dag)
                        (<= (car (car (last rev-dag))) 2147483646)
                      t)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (if (consp rev-dag)
                        (or (null maybe-internal-context-array)
                            (context-arrayp 'context-array maybe-internal-context-array (+ 1 (car (car (last rev-dag))))))
                      t)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (if (consp rev-dag)
                        (equal (renumbering-length renumbering-stobj)
                               (+ 1 (car (car (last rev-dag))))) ; the highest nodenum
                      t)
                    (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                     (+ -1 (car (first rev-dag)))
                                                   -1)
                                                 dag-len renumbering-stobj)
                    (renumbering-stobjp renumbering-stobj)
                    (consp rev-dag) ; note this
                    )
               (mv-let (erp new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj)
                 (,simplify-dag-aux-name rev-dag
                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                         maybe-internal-context-array
                                         memoization info tries limits
                                         node-replacement-array node-replacement-count refined-assumption-alist
                                         rewrite-stobj
                                         renumbering-stobj)
                 (declare (ignore new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array))
                 (implies (and (not erp)
                               (natp bound)
                               (<= bound (car (car (last rev-dag)))))
                          (good-renumbering-stobj bound new-renumbering-stobj))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-aux-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-aux-name '-return-type)))))

    ;; A simple consequence of the return type theorem
    (defthm ,(pack$ simplify-dag-aux-name '-return-type-corollary3)
      (implies (and (weak-dagp-aux rev-dag)
                    (cars-increasing-by-1 rev-dag)
                    (if (consp rev-dag)
                        (<= (car (car (last rev-dag))) 2147483646)
                      t)
                    (wf-dagp 'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist)
                    (if (consp rev-dag)
                        (or (null maybe-internal-context-array)
                            (context-arrayp 'context-array maybe-internal-context-array (+ 1 (car (car (last rev-dag))))))
                      t)
                    (maybe-bounded-memoizationp memoization dag-len)
                    (info-worldp info)
                    (triesp tries)
                    (rule-limitsp limits)
                    (bounded-node-replacement-arrayp 'node-replacement-array node-replacement-array dag-len)
                    (natp node-replacement-count)
                    (<= node-replacement-count (alen1 'node-replacement-array node-replacement-array))
                    (rewrite-stobjp rewrite-stobj)
                    (bounded-refined-assumption-alistp refined-assumption-alist dag-len)
                    (if (consp rev-dag)
                        (equal (renumbering-length renumbering-stobj)
                               (+ 1 (car (car (last rev-dag))))) ; the highest nodenum
                      t)
                    (bounded-good-renumbering-stobj (if (consp rev-dag)
                                                     (+ -1 (car (first rev-dag)))
                                                   -1)
                                                 dag-len renumbering-stobj)
                    (renumbering-stobjp renumbering-stobj)
                    (consp rev-dag) ; note this
                    )
               (mv-let (erp new-dag-array new-dag-len new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array new-renumbering-stobj)
                 (,simplify-dag-aux-name rev-dag
                                         dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                         maybe-internal-context-array memoization info tries limits
                                         node-replacement-array node-replacement-count refined-assumption-alist
                                         rewrite-stobj
                                         renumbering-stobj)
                 (declare (ignore new-dag-array new-dag-parent-array new-dag-constant-alist new-dag-variable-alist new-memoization new-info tries limits node-replacement-array))
                 (implies (and (not erp)
                               (natp bound)
                               (<= bound (car (car (last rev-dag))))
                               (<= new-dag-len bound2)
                               (natp bound2)
                               )
                          (bounded-good-renumbering-stobj bound bound2 new-renumbering-stobj))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-aux-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-aux-name '-return-type)))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; Returns (mv erp dag-or-quotep).
    ;; TODO: Make a version that returns an array (call crunch-dag instead of drop-non-supporters-array-with-name)?
    ;; TODO: Add support for rewriting nodes in their (approximate) contexts.
    (defund ,simplify-dag-name (dag
                                assumptions
                                interpreted-function-alist
                                limits
                                rule-alist
                                count-hits
                                print
                                known-booleans
                                monitored-symbols
                                normalize-xors
                                memoize)
      (declare (xargs :guard (and (pseudo-dagp dag)
                                  (< (top-nodenum dag) 2147483646)
                                  (pseudo-term-listp assumptions)
                                  (rule-limitsp limits)
                                  (rule-alistp rule-alist)
                                  (booleanp count-hits)
                                  (print-levelp print)
                                  (interpreted-function-alistp interpreted-function-alist)
                                  (symbol-listp known-booleans)
                                  (symbol-listp monitored-symbols)
                                  (booleanp normalize-xors)
                                  (booleanp memoize))
                      :guard-hints (("Goal" :do-not '(generalize eliminate-destructors)
                                     :in-theory (e/d (not-<-of-0-when-natp-disabled
                                                      acl2-numberp-when-natp
                                                      natp-of-+-of--1-when-natp-disabled
                                                      ;; natp-when-dargp ; too strong?
                                                      <-of-+-of-1-when-integers
                                                      natp-of-+-of-1
                                                      integerp-of-renumberingi
                                                      natp-of-renumberingi
                                                      len-when-pseudo-dagp
                                                      car-of-nth-when-pseudo-dagp)
                                                     (natp))))))
      (b* ((old-top-nodenum (top-nodenum dag))
           (old-len (+ 1 old-top-nodenum))
           (- (and print (cw "(Simplifying DAG (~x0 nodes, ~x1 assumptions):~%" old-len (len assumptions))))
           (use-internal-contextsp (and (not memoize) ; unsound to use contexts if memoizing
                                        (dag-has-internal-contextsp dag)))
           (initial-array-size (if use-internal-contextsp (min 2147483646 (* 2 old-len)) old-len)) ; could make this adjustable
           ;; Start with either an empty dag-array, or an array with all the nodes (if using contexts):
           ((mv dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
            (if use-internal-contextsp
                (b* ((dag-array (make-into-array-with-len 'dag-array dag initial-array-size))
                     ;; Make the auxiliary data structures for the DAG:
                     ((mv dag-parent-array dag-constant-alist dag-variable-alist)
                      (make-dag-indices 'dag-array dag-array 'dag-parent-array old-len)))
                  (mv dag-array old-len dag-parent-array dag-constant-alist dag-variable-alist))
              (empty-dag-array initial-array-size)))
           (maybe-internal-context-array (if use-internal-contextsp
                                             (make-full-context-array-with-parents 'dag-array dag-array dag-len dag-parent-array)
                                           nil))
           ;; Create the refined-assumption-alist and add relevant nodes to the DAG:
           ((mv erp refined-assumption-alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
            ;; TODO: Make a version specialized to these array names:
            (refine-assumptions-and-add-to-dag-array assumptions
                                                     'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist
                                                     known-booleans))
           ((when erp) (mv erp nil))
           ;; Create the node-replacement-array and add relevant nodes to the DAG:
           ;; TODO: Consider combining this with the above, in a single pass through the assumptions):
           ((mv erp node-replacement-array node-replacement-count dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
            (make-node-replacement-array-and-extend-dag assumptions
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                                        known-booleans))
           ((when erp) (mv erp nil)))
        (with-local-stobjs
         (renumbering-stobj rewrite-stobj)
           (mv-let (erp new-top-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj rewrite-stobj)
             (b* ((rewrite-stobj (put-monitored-symbols monitored-symbols rewrite-stobj))
                  (rewrite-stobj (put-known-booleans known-booleans rewrite-stobj))
                  (rewrite-stobj (put-normalize-xors normalize-xors rewrite-stobj))
                  (rewrite-stobj (put-interpreted-function-alist interpreted-function-alist rewrite-stobj))
                  (rewrite-stobj (put-rule-alist rule-alist rewrite-stobj))
                  (rewrite-stobj (put-print print rewrite-stobj))
                  (renumbering-stobj (resize-renumbering old-len renumbering-stobj))
                  ((mv erp dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj)
                   (,simplify-dag-aux-name (reverse-list dag) ;;we'll simplify nodes from the bottom-up
                                           dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                           maybe-internal-context-array
                                           (and memoize (empty-memoization)) ; todo: add an option to make this bigger?
                                           ;; TODO: If print is :brief, maybe-print-hit-counts below will only print the total number of hits, so
                                           ;; tracking hit counts for each rule is overkill:
                                           (and count-hits print (empty-info-world)) ;used to track the number of rule hits
                                           (and print (zero-tries)) ;todo: think about this
                                           limits
                                           node-replacement-array node-replacement-count refined-assumption-alist
                                           rewrite-stobj
                                           renumbering-stobj))
                  ((when erp) (mv erp nil dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj rewrite-stobj))
                  ;; See what the top node of the old dag became (after this point, we won't have access to renumbering-stobj anymore, due to with-local-stobj):
                  (new-top-nodenum-or-quotep (renumberingi old-top-nodenum renumbering-stobj)))
               (mv (erp-nil) new-top-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array renumbering-stobj rewrite-stobj))
             ;; Cannot refer to the stobjs:
             (declare (ignore dag-len dag-parent-array dag-constant-alist dag-variable-alist limits node-replacement-array)) ; print some stats from these?
             (b* (((when erp) (mv erp nil))
                  ;; todo: do we support both brief hit counting (just the total) and totals per-rule?:
                  (- (and count-hits print (maybe-print-hit-counts print info)))
                  (- (and print tries (cw "(~x0 tries.)" tries))) ;print these after dropping non supps?
                  (- (and (print-level-at-least-tp print) memoization (print-memo-stats memoization)))
                  ;; todo: print the new len?
                  (- (and print (cw ")~%"))) ; balances "(Simplifying DAG"
                  )
               (if (quotep new-top-nodenum-or-quotep)
                   (mv (erp-nil) new-top-nodenum-or-quotep)
                 (mv (erp-nil)
                     (drop-non-supporters-array-with-name 'dag-array dag-array new-top-nodenum-or-quotep nil))))))))

    (defthm ,(pack$ simplify-dag-name '-return-type)
      (implies (and (not (myquotep (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize))))
                    (not (mv-nth 0 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize))) ; no error
                    (pseudo-dagp dag)
                    (< (top-nodenum dag) 2147483646)
                    (pseudo-term-listp assumptions)
                    (rule-limitsp limits)
                    (rule-alistp rule-alist)
                    (booleanp count-hits)
                    (print-levelp print)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp known-booleans)
                    (symbol-listp monitored-symbols)
                    (booleanp normalize-xors)
                    (booleanp memoize))
               (and (pseudo-dagp (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize)))
                    ;; (< (len (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize)))
                    ;;    2147483647) ;; todo
                    ))
      :hints (("Goal" :do-not '(generalize eliminate-destructors)
               :in-theory (e/d (,simplify-dag-name
                                       natp-of-renumberingi
                                       integerp-of-renumberingi
                                       <-of-+-of-1-when-integers
                                       len-when-pseudo-dagp
                                       car-of-nth-when-pseudo-dagp)
                                      (myquotep
                                       mv-nth ; why?
                                       natp)))))

    ;; ;; It's a consp either way
    ;; (defthm ,(pack$ simplify-dag-name '-return-type-corollary-1)
    ;;   (implies (and (not (mv-nth 0 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize))) ; no error
    ;;                 (pseudo-dagp dag)
    ;;                 (< (top-nodenum dag) 2147483646)
    ;;                 (pseudo-term-listp assumptions)
    ;;                 (rule-limitsp limits)
    ;;                 (rule-alistp rule-alist)
    ;;                 (booleanp count-hits)
    ;;                 (print-levelp print)
    ;;                 (interpreted-function-alistp interpreted-function-alist)
    ;;                 (symbol-listp known-booleans)
    ;;                 (symbol-listp monitored-symbols)
    ;;                 (booleanp normalize-xors)
    ;;                 (booleanp memoize))
    ;;            (consp (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize)))
    ;;                   )
    ;;   :hints (("Goal" :use (:instance ,(pack$ simplify-dag-name '-return-type))
    ;;            :in-theory (disable ,(pack$ simplify-dag-name '-return-type)))))

    ;; Uses myquotep as the normal form.
    (defthm ,(pack$ simplify-dag-name '-return-type-corollary-2)
      (implies (and (not (mv-nth 0 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize))) ; no error
                    (pseudo-dagp dag)
                    (< (top-nodenum dag) 2147483646)
                    (pseudo-term-listp assumptions)
                    (rule-limitsp limits)
                    (rule-alistp rule-alist)
                    (booleanp count-hits)
                    (print-levelp print)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp known-booleans)
                    (symbol-listp monitored-symbols)
                    (booleanp normalize-xors)
                    (booleanp memoize))
               (equal (quotep (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize)))
                      (myquotep (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize)))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-name '-return-type)))))

    ;; Uses myquotep as the normal form.
    (defthm ,(pack$ simplify-dag-name '-return-type-corollary-3)
      (implies (and (not (mv-nth 0 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize))) ; no error
                    (pseudo-dagp dag)
                    (< (top-nodenum dag) 2147483646)
                    (pseudo-term-listp assumptions)
                    (rule-limitsp limits)
                    (rule-alistp rule-alist)
                    (booleanp count-hits)
                    (print-levelp print)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp known-booleans)
                    (symbol-listp monitored-symbols)
                    (booleanp normalize-xors)
                    (booleanp memoize))
               (equal (pseudo-dagp (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize)))
                      (not (myquotep (mv-nth 1 (,simplify-dag-name dag assumptions interpreted-function-alist limits rule-alist count-hits print known-booleans monitored-symbols normalize-xors memoize))))))
      :hints (("Goal" :use (:instance ,(pack$ simplify-dag-name '-return-type))
               :in-theory (disable ,(pack$ simplify-dag-name '-return-type)))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; Simplify a term and return an equivalent DAG.  Returns (mv erp dag-or-quotep).
    ;; TODO: add support for multiple rule-alists.
    ;; TODO: Factor out move of this, which returns a dag-array and its various auxiliary indices.
    (defund ,simplify-term-name (term
                                 assumptions
                                 rule-alist
                                 interpreted-function-alist
                                 monitored-symbols
                                 memoizep
                                 ;; todo: add context array and other args?
                                 count-hits
                                 print
                                 normalize-xors
                                 wrld)
      (declare (xargs :guard (and (pseudo-termp term)
                                  (pseudo-term-listp assumptions)
                                  (rule-alistp rule-alist)
                                  (interpreted-function-alistp interpreted-function-alist)
                                  (symbol-listp monitored-symbols)
                                  (booleanp memoizep)
                                  (booleanp count-hits)
                                  (print-levelp print)
                                  (booleanp normalize-xors)
                                  (plist-worldp wrld))
                      :guard-hints (("Goal" :in-theory (e/d (natp-when-dargp
                                                             natp-of-+-of-1
                                                             <-of-+-of-1-when-integers
                                                             <-OF-+-OF-1-WHEN-natps
                                                             ;; integerp-when-dargp ;caused problems when natp is known
                                                             axe-treep-when-pseudo-termp
                                                             dargp-when-natp
                                                             <-of-if-arg2-axe)
                                                            (natp
                                                             NATP-WHEN-DARGP ;caused problems when natp is known
                                                             ))))))
      (b* (;; Create an empty dag-array:
           (slack-amount 1000000) ;todo: make this adjustable
           ((mv dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
            (empty-dag-array slack-amount))

           ;; Create the refined-assumption-alist and add relevant nodes to the DAG:
           ((mv erp refined-assumption-alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
            ;; TODO: Make a version specialized to these array names:
            (refine-assumptions-and-add-to-dag-array assumptions
                                                     'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist
                                                     (known-booleans wrld)))
           ((when erp) (mv erp nil))

           ;; Create the node-replacement-array and add relevant nodes to the DAG:
           ;; TODO: Consider combining this with the above, in a single pass through the assumptions):
           ((mv erp node-replacement-array node-replacement-count dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
            (make-node-replacement-array-and-extend-dag assumptions
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                                        (known-booleans wrld)))
           ((when erp) (mv erp nil))

           ;; old:
           ;; ((mv erp node-replacement-alist dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist)
           ;;  ;; TODO: Make a version specialized to these array names:
           ;;  (make-node-replacement-alist-and-add-to-dag-array assumptions
           ;;                                                    'dag-array dag-array dag-len 'dag-parent-array dag-parent-array dag-constant-alist dag-variable-alist
           ;;                                                    wrld))
           ;; ((when erp) (mv erp nil))
           ;; ;; TODO: Combine this with the above (don't actually create the node-replacement-alist):
           ;; (node-replacement-array (make-into-array 'node-replacement-array node-replacement-alist))
           ;; (node-replacement-count (+ 1 (max-key node-replacement-alist 0))) ;todo: optimize if no assumptions?  the array len of 0 will prevent any lookup

           ;; Call the core term simplification function:
           ((mv erp new-nodenum-or-quotep
                dag-array
                & & & & ; dag-len dag-parent-array dag-constant-alist dag-variable-alist
                memoization info
                & & & ; tries limits node-replacement-array
                )
            (with-local-stobj rewrite-stobj
              (mv-let (erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array rewrite-stobj)
                (let* ((rewrite-stobj (put-monitored-symbols monitored-symbols rewrite-stobj))
                       (rewrite-stobj (put-known-booleans (known-booleans wrld) ;skip if memoizing since we can't use contexts?
                                                          rewrite-stobj))
                       (rewrite-stobj (put-normalize-xors normalize-xors rewrite-stobj))
                       (rewrite-stobj (put-interpreted-function-alist interpreted-function-alist rewrite-stobj))
                       (rewrite-stobj (put-rule-alist rule-alist rewrite-stobj))
                       (rewrite-stobj (put-print print rewrite-stobj)))
                  (mv-let (erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array)
                    ;; TODO: Consider making a version of ,simplify-tree-and-add-to-dag-name that applies only to terms, not axe-trees, and calling it here.
                    ;; TODO: Or consider handling vars separately and then dropping support for vars in ,simplify-tree-and-add-to-dag-name (and in the memoization).
                    (,simplify-tree-and-add-to-dag-name term
                                                        nil ;trees-equal-to-tree
                                                        dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist
                                                        (if memoizep
                                                            (empty-memoization)
                                                          ;; not memoizing:
                                                          nil)
                                                        (if count-hits
                                                            (empty-info-world)
                                                          nil ;means no hit counting
                                                          )
                                                        0   ; tries
                                                        nil ; limits ; todo: pass in
                                                        node-replacement-array node-replacement-count refined-assumption-alist
                                                        rewrite-stobj
                                                        1000000000 ;count
                                                        )
                    (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array rewrite-stobj)))
                (mv erp new-nodenum-or-quotep dag-array dag-len dag-parent-array dag-constant-alist dag-variable-alist memoization info tries limits node-replacement-array) ; no stobj
                )))
           ((when erp) (mv erp nil))
           (- (and count-hits
                   (maybe-print-hit-counts t info)))
           (- (and nil ;; change to t to print info on the memoization
                   memoization
                   (print-memo-stats memoization))))
        (if (consp new-nodenum-or-quotep) ;check for quotep
            (mv (erp-nil) new-nodenum-or-quotep)
          (mv (erp-nil) (drop-non-supporters-array-with-name 'dag-array dag-array new-nodenum-or-quotep nil)))))

    (defthm ,(pack$ 'type-of-mv-nth-1-of- simplify-term-name)
      (implies (and (not (mv-nth 0 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))) ; no error
                    (pseudo-termp term)
                    (pseudo-term-listp assumptions)
                    (rule-alistp rule-alist)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp monitored-symbols)
                    (booleanp memoizep)
                    (print-levelp print)
                    (booleanp normalize-xors)
                    (booleanp count-hits)
                    (plist-worldp wrld))
               (or (myquotep (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld)))
                   (pseudo-dagp (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld)))))
      :rule-classes nil
      :hints (("Goal" :in-theory (e/d (,simplify-term-name
                                       axe-treep-when-pseudo-termp
                                       natp-of-+-of-1
                                       natp-of-max-key-2
                                       <-of-if-arg1
                                       max-key-hack
                                       max-key-hack-2
                                       <-OF-+-OF-1-WHEN-INTEGERS
                                       integerp-when-natp-disabled
                                       <-of-if-arg2-axe)
                                      (natp)))))

    (defthm ,(pack$ 'consp-of-cdr-of-mv-nth-1-of- simplify-term-name '-when-quotep)
      (implies (and (equal 'quote (car (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))))
                    (not (mv-nth 0 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))) ; no error
                    (pseudo-termp term)
                    (pseudo-term-listp assumptions)
                    (rule-alistp rule-alist)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp monitored-symbols)
                    (booleanp memoizep)
                    (print-levelp print)
                    (booleanp normalize-xors)
                    (booleanp count-hits)
                    (plist-worldp wrld))
               (consp (cdr (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld)))))
      :hints (("Goal" :use (:instance ,(pack$ 'type-of-mv-nth-1-of- simplify-term-name)))))

    (defthm ,(pack$ 'pseudo-dagp-of-mv-nth-1-of- simplify-term-name)
      (implies (and (not (mv-nth 0 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))) ; no error
                    (not (quotep (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld)))) ;not a constant
                    (pseudo-termp term)
                    (pseudo-term-listp assumptions)
                    (rule-alistp rule-alist)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp monitored-symbols)
                    (booleanp memoizep)
                    (print-levelp print)
                    (booleanp normalize-xors)
                    (booleanp count-hits)
                    (plist-worldp wrld))
               (pseudo-dagp (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))))
      :hints (("Goal" :use (:instance ,(pack$ 'type-of-mv-nth-1-of- simplify-term-name)))))

    (defthm ,(pack$ 'myquotep-of-mv-nth-1-of- simplify-term-name)
      (implies (and (not (mv-nth 0 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))) ; no error
                    (not (pseudo-dagp (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld)))) ;not a dag
                    (pseudo-termp term)
                    (pseudo-term-listp assumptions)
                    (rule-alistp rule-alist)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp monitored-symbols)
                    (booleanp memoizep)
                    (print-levelp print)
                    (booleanp normalize-xors)
                    (booleanp count-hits)
                    (plist-worldp wrld))
               (myquotep (mv-nth 1 (,simplify-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))))
      :hints (("Goal" :use (:instance ,(pack$ 'type-of-mv-nth-1-of- simplify-term-name)))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; Simplify a term and return a term (not a DAG).  Returns (mv erp term).
    ;; WARNING: The term returned might be huge!
    (defund ,simp-term-name (term
                             assumptions
                             rule-alist
                             interpreted-function-alist
                             monitored-symbols
                             memoizep
                             ;; todo: add context array and other args?
                             count-hits
                             print
                             normalize-xors
                             wrld)
      (declare (xargs :guard (and (pseudo-termp term)
                                  (pseudo-term-listp assumptions)
                                  (rule-alistp rule-alist)
                                  (interpreted-function-alistp interpreted-function-alist)
                                  (symbol-listp monitored-symbols)
                                  (booleanp memoizep)
                                  (booleanp count-hits)
                                  (print-levelp print)
                                  (booleanp normalize-xors)
                                  (plist-worldp wrld))))
      (b* (((mv erp dag) (,simplify-term-name term
                                              assumptions
                                              rule-alist
                                              interpreted-function-alist
                                              monitored-symbols
                                              memoizep
                                              ;; todo: add context array and other args?
                                              count-hits
                                              print
                                              normalize-xors wrld))
           ((when erp) (mv erp nil)))
        (mv (erp-nil) (if (quotep dag)
                          dag
                        (dag-to-term dag)))))

    (defthm ,(pack$ 'pseudo-termp-of-mv-nth-1-of- simp-term-name)
      (implies (and (pseudo-termp term)
                    (pseudo-term-listp assumptions)
                    (rule-alistp rule-alist)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp monitored-symbols)
                    (booleanp memoizep)
                    (booleanp count-hits)
                    (print-levelp print)
                    (booleanp normalize-xors)
                    (plist-worldp wrld))
               (pseudo-termp (mv-nth 1 (,simp-term-name term assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))))
      :hints (("Goal" :use (:instance ,(pack$ 'type-of-mv-nth-1-of- simplify-term-name))
               :in-theory (e/d (,simp-term-name) (,(pack$ 'pseudo-dagp-of-mv-nth-1-of- simplify-term-name))))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; Simplify a list of terms, returning a list of the simplified terms (not
    ;; DAGs).  Returns (mv erp terms).
    (defun ,simp-terms-name (terms
                             assumptions
                             rule-alist
                             interpreted-function-alist
                             monitored-symbols
                             memoizep
                             ;; todo: add context array and other args?
                             count-hits
                             print
                             normalize-xors
                             wrld)
      (declare (xargs :guard (and (pseudo-term-listp terms)
                                  (pseudo-term-listp assumptions)
                                  (rule-alistp rule-alist)
                                  (interpreted-function-alistp interpreted-function-alist)
                                  (symbol-listp monitored-symbols)
                                  (booleanp memoizep)
                                  (booleanp count-hits)
                                  (print-levelp print)
                                  (booleanp normalize-xors)
                                  (plist-worldp wrld))))
      (if (endp terms)
          (mv (erp-nil) nil)
        (b* (((mv erp first-res)
              (,simp-term-name (first terms)
                               assumptions
                               rule-alist
                               nil
                               nil
                               nil
                               t
                               print
                               normalize-xors wrld))
             ((when erp) (mv erp nil))
             ((mv erp rest-res)
              (,simp-terms-name (rest terms)
                                assumptions
                                rule-alist
                                interpreted-function-alist
                                monitored-symbols
                                memoizep
                                count-hits
                                print
                                normalize-xors wrld))
             ((when erp) (mv erp nil)))
          (mv (erp-nil)
              (cons first-res rest-res)))))

    (defthm ,(pack$ 'true-listp-of-mv-nth-1-of- simp-terms-name)
      (true-listp (mv-nth 1 (,simp-terms-name terms assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld)))
      :rule-classes :type-prescription
      :hints (("Goal" :in-theory (enable ,simp-terms-name))))

    (defthm ,(pack$ 'pseudo-term-listp-of-mv-nth-1-of- simp-terms-name)
      (implies (and (pseudo-term-listp terms)
                    (pseudo-term-listp assumptions)
                    (rule-alistp rule-alist)
                    (interpreted-function-alistp interpreted-function-alist)
                    (symbol-listp monitored-symbols)
                    (booleanp memoizep)
                    (booleanp count-hits)
                    (print-levelp print)
                    (booleanp normalize-xors)
                    (plist-worldp wrld))
               (pseudo-term-listp (mv-nth 1 (,simp-terms-name terms assumptions rule-alist interpreted-function-alist monitored-symbols memoizep count-hits print normalize-xors wrld))))
      :hints (("Goal" :in-theory (enable ,simp-terms-name))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;; Returns an (mv erp event state).
    ;; todo: check if name already exists
    (defund ,def-simplified-dag-fn-name (name ; the name of the constant to create
                                         dag
                                         assumptions
                                         interpreted-function-alist
                                         limits
                                         rules
                                         count-hits
                                         print
                                         monitored-symbols
                                         normalize-xors
                                         memoize
                                         whole-form
                                         state)
      (declare (xargs :guard (and (symbolp name)
                                  (pseudo-dagp dag)
                                  (< (top-nodenum dag) 2147483646)
                                  (pseudo-term-listp assumptions)
                                  (interpreted-function-alistp interpreted-function-alist)
                                  (rule-limitsp limits)
                                  (symbol-listp rules)
                                  (booleanp count-hits)
                                  (print-levelp print)
                                  (symbol-listp monitored-symbols)
                                  (booleanp normalize-xors)
                                  (booleanp memoize)
                                  (consp whole-form)
                                  (symbolp (car whole-form))
                                  (ilks-plist-worldp (w state)))
                      :stobjs state))
      (b* (((when (command-is-redundantp whole-form state))
            (mv nil '(value-triple :invisible) state))
           ((when (not (starts-and-ends-with-starsp name))) ; todo: stricter check?
            (er hard? ',def-simplified-dag-fn-name "The name ~x0 is not a legal constant name." name)
            (mv :bad-name nil state))
           ((when (getpropc name 'const nil (w state))) ; todo: factor out
            (er hard? ',def-simplified-dag-fn-name "The name ~x0 is already in use as a constant." name)
            (mv :name-in-use nil state))
           (known-booleans (known-booleans (w state)))
           ((mv erp rule-alist) (make-rule-alist rules (w state)))
           ((when erp) (mv erp nil state))
           ((mv erp dag-or-quotep) (,simplify-dag-name dag
                                                       assumptions
                                                       interpreted-function-alist
                                                       limits
                                                       rule-alist
                                                       count-hits
                                                       print
                                                       known-booleans
                                                       monitored-symbols
                                                       normalize-xors
                                                       memoize))
           ((when erp) (mv erp nil state)))
        (mv (erp-nil)
            `(progn (defconst ,name ',dag-or-quotep)
                    (with-output :off :all (table ,',(pack$ def-simplified-dag-name '-table) ',whole-form ':fake)))
            state)))

    ;; Creates a constant named NAME, whose value is a DAG representing the simplified form of DAG.
    (defmacro ,def-simplified-dag-name (&whole whole-form
                                        name
                                        dag
                                        &key
                                        (assumptions 'nil)
                                        (interpreted-function-alist 'nil)
                                        (limits 'nil)
                                        (rules 'nil)
                                        (count-hits 'nil)
                                        (print ':brief)
                                        (monitored-symbols 'nil)
                                        (normalize-xors 'nil)
                                        (memoize 't))
      `(make-event-quiet (,',def-simplified-dag-fn-name ',name ,dag ,assumptions ,interpreted-function-alist ,limits ,rules ,count-hits ,print ,monitored-symbols ,normalize-xors ,memoize ',whole-form state)))
    ) ; end of the generated encapsulate
    ))

;; Makes a version of the (simple) Axe Rewriter, given an evaluator, a syntaxp evaluator, and an axe-bind-free evaluator.
(defmacro make-rewriter-simple (suffix ; added to the end of each generated function/macro name
                                evaluator-base-name
                                syntaxp-evaluator-suffix ;as given to make-axe-syntaxp-evaluator
                                bind-free-evaluator-suffix ;as given to make-axe-bind-free-evaluator
                                )
  (make-rewriter-simple-fn suffix
                           evaluator-base-name
                           syntaxp-evaluator-suffix
                           bind-free-evaluator-suffix))
